/*! DataTables 2.3.2
 * © SpryMedia Ltd - datatables.net/license
 */

(function( factory ) {
	"use strict";

	if ( typeof define === 'function' && define.amd ) {
		// AMD
		define( ['jquery'], function ( $ ) {
			return factory( $, window, document );
		} );
	}
	else if ( typeof exports === 'object' ) {
		// CommonJS
		// jQuery's factory checks for a global window - if it isn't present then it
		// returns a factory function that expects the window object
		var jq = require('jquery');

		if (typeof window === 'undefined') {
			module.exports = function (root, $) {
				if ( ! root ) {
					// CommonJS environments without a window global must pass a
					// root. This will give an error otherwise
					root = window;
				}

				if ( ! $ ) {
					$ = jq( root );
				}

				return factory( $, root, root.document );
			};
		}
		else {
			module.exports = factory( jq, window, window.document );
		}
	}
	else {
		// Browser
		window.DataTable = factory( jQuery, window, document );
	}
}(function( $, window, document ) {
	"use strict";

	
	var DataTable = function ( selector, options )
	{
		// Check if called with a window or jQuery object for DOM less applications
		// This is for backwards compatibility
		if (DataTable.factory(selector, options)) {
			return DataTable;
		}
	
		// When creating with `new`, create a new DataTable, returning the API instance
		if (this instanceof DataTable) {
			return $(selector).DataTable(options);
		}
		else {
			// Argument switching
			options = selector;
		}
	
		var _that = this;
		var emptyInit = options === undefined;
		var len = this.length;
	
		if ( emptyInit ) {
			options = {};
		}
	
		// Method to get DT API instance from jQuery object
		this.api = function ()
		{
			return new _Api( this );
		};
	
		this.each(function() {
			// For each initialisation we want to give it a clean initialisation
			// object that can be bashed around
			var o = {};
			var oInit = len > 1 ? // optimisation for single table case
				_fnExtend( o, options, true ) :
				options;
	
			
			var i=0, iLen;
			var sId = this.getAttribute( 'id' );
			var defaults = DataTable.defaults;
			var $this = $(this);
			
			// Sanity check
			if ( this.nodeName.toLowerCase() != 'table' )
			{
				_fnLog( null, 0, 'Non-table node initialisation ('+this.nodeName+')', 2 );
				return;
			}
			
			// Special case for options
			if (oInit.on && oInit.on.options) {
				_fnListener($this, 'options', oInit.on.options);	
			}
			
			$this.trigger( 'options.dt', oInit );
			
			/* Backwards compatibility for the defaults */
			_fnCompatOpts( defaults );
			_fnCompatCols( defaults.column );
			
			/* Convert the camel-case defaults to Hungarian */
			_fnCamelToHungarian( defaults, defaults, true );
			_fnCamelToHungarian( defaults.column, defaults.column, true );
			
			/* Setting up the initialisation object */
			_fnCamelToHungarian( defaults, $.extend( oInit, _fnEscapeObject($this.data()) ), true );
			
			
			
			/* Check to see if we are re-initialising a table */
			var allSettings = DataTable.settings;
			for ( i=0, iLen=allSettings.length ; i<iLen ; i++ )
			{
				var s = allSettings[i];
			
				/* Base check on table node */
				if (
					s.nTable == this ||
					(s.nTHead && s.nTHead.parentNode == this) ||
					(s.nTFoot && s.nTFoot.parentNode == this)
				) {
					var bRetrieve = oInit.bRetrieve !== undefined ? oInit.bRetrieve : defaults.bRetrieve;
					var bDestroy = oInit.bDestroy !== undefined ? oInit.bDestroy : defaults.bDestroy;
			
					if ( emptyInit || bRetrieve )
					{
						return s.oInstance;
					}
					else if ( bDestroy )
					{
						new DataTable.Api(s).destroy();
						break;
					}
					else
					{
						_fnLog( s, 0, 'Cannot reinitialise DataTable', 3 );
						return;
					}
				}
			
				/* If the element we are initialising has the same ID as a table which was previously
				 * initialised, but the table nodes don't match (from before) then we destroy the old
				 * instance by simply deleting it. This is under the assumption that the table has been
				 * destroyed by other methods. Anyone using non-id selectors will need to do this manually
				 */
				if ( s.sTableId == this.id )
				{
					allSettings.splice( i, 1 );
					break;
				}
			}
			
			/* Ensure the table has an ID - required for accessibility */
			if ( sId === null || sId === "" )
			{
				sId = "DataTables_Table_"+(DataTable.ext._unique++);
				this.id = sId;
			}
			
			/* Create the settings object for this table and set some of the default parameters */
			var oSettings = $.extend( true, {}, DataTable.models.oSettings, {
				"sDestroyWidth": $this[0].style.width,
				"sInstance":     sId,
				"sTableId":      sId,
				colgroup: $('<colgroup>').prependTo(this),
				fastData: function (row, column, type) {
					return _fnGetCellData(oSettings, row, column, type);
				}
			} );
			oSettings.nTable = this;
			oSettings.oInit  = oInit;
			
			allSettings.push( oSettings );
			
			// Make a single API instance available for internal handling
			oSettings.api = new _Api( oSettings );
			
			// Need to add the instance after the instance after the settings object has been added
			// to the settings array, so we can self reference the table instance if more than one
			oSettings.oInstance = (_that.length===1) ? _that : $this.dataTable();
			
			// Backwards compatibility, before we apply all the defaults
			_fnCompatOpts( oInit );
			
			// If the length menu is given, but the init display length is not, use the length menu
			if ( oInit.aLengthMenu && ! oInit.iDisplayLength )
			{
				oInit.iDisplayLength = Array.isArray(oInit.aLengthMenu[0])
					? oInit.aLengthMenu[0][0]
					: $.isPlainObject( oInit.aLengthMenu[0] )
						? oInit.aLengthMenu[0].value
						: oInit.aLengthMenu[0];
			}
			
			// Apply the defaults and init options to make a single init object will all
			// options defined from defaults and instance options.
			oInit = _fnExtend( $.extend( true, {}, defaults ), oInit );
			
			
			// Map the initialisation options onto the settings object
			_fnMap( oSettings.oFeatures, oInit, [
				"bPaginate",
				"bLengthChange",
				"bFilter",
				"bSort",
				"bSortMulti",
				"bInfo",
				"bProcessing",
				"bAutoWidth",
				"bSortClasses",
				"bServerSide",
				"bDeferRender"
			] );
			_fnMap( oSettings, oInit, [
				"ajax",
				"fnFormatNumber",
				"sServerMethod",
				"aaSorting",
				"aaSortingFixed",
				"aLengthMenu",
				"sPaginationType",
				"iStateDuration",
				"bSortCellsTop",
				"iTabIndex",
				"sDom",
				"fnStateLoadCallback",
				"fnStateSaveCallback",
				"renderer",
				"searchDelay",
				"rowId",
				"caption",
				"layout",
				"orderDescReverse",
				"orderIndicators",
				"orderHandler",
				"titleRow",
				"typeDetect",
				[ "iCookieDuration", "iStateDuration" ], // backwards compat
				[ "oSearch", "oPreviousSearch" ],
				[ "aoSearchCols", "aoPreSearchCols" ],
				[ "iDisplayLength", "_iDisplayLength" ]
			] );
			_fnMap( oSettings.oScroll, oInit, [
				[ "sScrollX", "sX" ],
				[ "sScrollXInner", "sXInner" ],
				[ "sScrollY", "sY" ],
				[ "bScrollCollapse", "bCollapse" ]
			] );
			_fnMap( oSettings.oLanguage, oInit, "fnInfoCallback" );
			
			/* Callback functions which are array driven */
			_fnCallbackReg( oSettings, 'aoDrawCallback',       oInit.fnDrawCallback );
			_fnCallbackReg( oSettings, 'aoStateSaveParams',    oInit.fnStateSaveParams );
			_fnCallbackReg( oSettings, 'aoStateLoadParams',    oInit.fnStateLoadParams );
			_fnCallbackReg( oSettings, 'aoStateLoaded',        oInit.fnStateLoaded );
			_fnCallbackReg( oSettings, 'aoRowCallback',        oInit.fnRowCallback );
			_fnCallbackReg( oSettings, 'aoRowCreatedCallback', oInit.fnCreatedRow );
			_fnCallbackReg( oSettings, 'aoHeaderCallback',     oInit.fnHeaderCallback );
			_fnCallbackReg( oSettings, 'aoFooterCallback',     oInit.fnFooterCallback );
			_fnCallbackReg( oSettings, 'aoInitComplete',       oInit.fnInitComplete );
			_fnCallbackReg( oSettings, 'aoPreDrawCallback',    oInit.fnPreDrawCallback );
			
			oSettings.rowIdFn = _fnGetObjectDataFn( oInit.rowId );
			
			// Add event listeners
			if (oInit.on) {
				Object.keys(oInit.on).forEach(function (key) {
					_fnListener($this, key, oInit.on[key]);
				});
			}
			
			/* Browser support detection */
			_fnBrowserDetect( oSettings );
			
			var oClasses = oSettings.oClasses;
			
			$.extend( oClasses, DataTable.ext.classes, oInit.oClasses );
			$this.addClass( oClasses.table );
			
			if (! oSettings.oFeatures.bPaginate) {
				oInit.iDisplayStart = 0;
			}
			
			if ( oSettings.iInitDisplayStart === undefined )
			{
				/* Display start point, taking into account the save saving */
				oSettings.iInitDisplayStart = oInit.iDisplayStart;
				oSettings._iDisplayStart = oInit.iDisplayStart;
			}
			
			var defer = oInit.iDeferLoading;
			if ( defer !== null )
			{
				oSettings.deferLoading = true;
			
				var tmp = Array.isArray(defer);
				oSettings._iRecordsDisplay = tmp ? defer[0] : defer;
				oSettings._iRecordsTotal = tmp ? defer[1] : defer;
			}
			
			/*
			 * Columns
			 * See if we should load columns automatically or use defined ones
			 */
			var columnsInit = [];
			var thead = this.getElementsByTagName('thead');
			var initHeaderLayout = _fnDetectHeader( oSettings, thead[0] );
			
			// If we don't have a columns array, then generate one with nulls
			if ( oInit.aoColumns ) {
				columnsInit = oInit.aoColumns;
			}
			else if ( initHeaderLayout.length ) {
				for ( i=0, iLen=initHeaderLayout[0].length ; i<iLen ; i++ ) {
					columnsInit.push( null );
				}
			}
			
			// Add the columns
			for ( i=0, iLen=columnsInit.length ; i<iLen ; i++ ) {
				_fnAddColumn( oSettings );
			}
			
			// Apply the column definitions
			_fnApplyColumnDefs( oSettings, oInit.aoColumnDefs, columnsInit, initHeaderLayout, function (iCol, oDef) {
				_fnColumnOptions( oSettings, iCol, oDef );
			} );
			
			/* HTML5 attribute detection - build an mData object automatically if the
			 * attributes are found
			 */
			var rowOne = $this.children('tbody').find('tr:first-child').eq(0);
			
			if ( rowOne.length ) {
				var a = function ( cell, name ) {
					return cell.getAttribute( 'data-'+name ) !== null ? name : null;
				};
			
				$( rowOne[0] ).children('th, td').each( function (i, cell) {
					var col = oSettings.aoColumns[i];
			
					if (! col) {
						_fnLog( oSettings, 0, 'Incorrect column count', 18 );
					}
			
					if ( col.mData === i ) {
						var sort = a( cell, 'sort' ) || a( cell, 'order' );
						var filter = a( cell, 'filter' ) || a( cell, 'search' );
			
						if ( sort !== null || filter !== null ) {
							col.mData = {
								_:      i+'.display',
								sort:   sort !== null   ? i+'.@data-'+sort   : undefined,
								type:   sort !== null   ? i+'.@data-'+sort   : undefined,
								filter: filter !== null ? i+'.@data-'+filter : undefined
							};
							col._isArrayHost = true;
			
							_fnColumnOptions( oSettings, i );
						}
					}
				} );
			}
			
			// Must be done after everything which can be overridden by the state saving!
			_fnCallbackReg( oSettings, 'aoDrawCallback', _fnSaveState );
			
			var features = oSettings.oFeatures;
			if ( oInit.bStateSave )
			{
				features.bStateSave = true;
			}
			
			// If aaSorting is not defined, then we use the first indicator in asSorting
			// in case that has been altered, so the default sort reflects that option
			if ( oInit.aaSorting === undefined ) {
				var sorting = oSettings.aaSorting;
				for ( i=0, iLen=sorting.length ; i<iLen ; i++ ) {
					sorting[i][1] = oSettings.aoColumns[ i ].asSorting[0];
				}
			}
			
			// Do a first pass on the sorting classes (allows any size changes to be taken into
			// account, and also will apply sorting disabled classes if disabled
			_fnSortingClasses( oSettings );
			
			_fnCallbackReg( oSettings, 'aoDrawCallback', function () {
				if ( oSettings.bSorted || _fnDataSource( oSettings ) === 'ssp' || features.bDeferRender ) {
					_fnSortingClasses( oSettings );
				}
			} );
			
			
			/*
			 * Table HTML init
			 * Cache the header, body and footer as required, creating them if needed
			 */
			var caption = $this.children('caption');
			
			if ( oSettings.caption ) {
				if ( caption.length === 0 ) {
					caption = $('<caption/>').appendTo( $this );
				}
			
				caption.html( oSettings.caption );
			}
			
			// Store the caption side, so we can remove the element from the document
			// when creating the element
			if (caption.length) {
				caption[0]._captionSide = caption.css('caption-side');
				oSettings.captionNode = caption[0];
			}
			
			if ( thead.length === 0 ) {
				thead = $('<thead/>').appendTo($this);
			}
			oSettings.nTHead = thead[0];
			
			var tbody = $this.children('tbody');
			if ( tbody.length === 0 ) {
				tbody = $('<tbody/>').insertAfter(thead);
			}
			oSettings.nTBody = tbody[0];
			
			var tfoot = $this.children('tfoot');
			if ( tfoot.length === 0 ) {
				// If we are a scrolling table, and no footer has been given, then we need to create
				// a tfoot element for the caption element to be appended to
				tfoot = $('<tfoot/>').appendTo($this);
			}
			oSettings.nTFoot = tfoot[0];
			
			// Copy the data index array
			oSettings.aiDisplay = oSettings.aiDisplayMaster.slice();
			
			// Initialisation complete - table can be drawn
			oSettings.bInitialised = true;
			
			// Language definitions
			var oLanguage = oSettings.oLanguage;
			$.extend( true, oLanguage, oInit.oLanguage );
			
			if ( oLanguage.sUrl ) {
				// Get the language definitions from a file
				$.ajax( {
					dataType: 'json',
					url: oLanguage.sUrl,
					success: function ( json ) {
						_fnCamelToHungarian( defaults.oLanguage, json );
						$.extend( true, oLanguage, json, oSettings.oInit.oLanguage );
			
						_fnCallbackFire( oSettings, null, 'i18n', [oSettings], true);
						_fnInitialise( oSettings );
					},
					error: function () {
						// Error occurred loading language file
						_fnLog( oSettings, 0, 'i18n file loading error', 21 );
			
						// Continue on as best we can
						_fnInitialise( oSettings );
					}
				} );
			}
			else {
				_fnCallbackFire( oSettings, null, 'i18n', [oSettings], true);
				_fnInitialise( oSettings );
			}
		} );
		_that = null;
		return this;
	};
	
	
	
	/**
	 * DataTables extensions
	 * 
	 * This namespace acts as a collection area for plug-ins that can be used to
	 * extend DataTables capabilities. Indeed many of the build in methods
	 * use this method to provide their own capabilities (sorting methods for
	 * example).
	 *
	 * Note that this namespace is aliased to `jQuery.fn.dataTableExt` for legacy
	 * reasons
	 *
	 *  @namespace
	 */
	DataTable.ext = _ext = {
		/**
		 * DataTables build type (expanded by the download builder)
		 *
		 *  @type string
		 */
		builder: "-source-",
	
		/**
		 * Buttons. For use with the Buttons extension for DataTables. This is
		 * defined here so other extensions can define buttons regardless of load
		 * order. It is _not_ used by DataTables core.
		 *
		 *  @type object
		 *  @default {}
		 */
		buttons: {},
	
	
		/**
		 * ColumnControl buttons and content
		 *
		 *  @type object
		 */
		ccContent: {},
	
	
		/**
		 * Element class names
		 *
		 *  @type object
		 *  @default {}
		 */
		classes: {},
	
	
		/**
		 * Error reporting.
		 * 
		 * How should DataTables report an error. Can take the value 'alert',
		 * 'throw', 'none' or a function.
		 *
		 *  @type string|function
		 *  @default alert
		 */
		errMode: "alert",
	
		/** HTML entity escaping */
		escape: {
			/** When reading data-* attributes for initialisation options */
			attributes: false
		},
	
		/**
		 * Legacy so v1 plug-ins don't throw js errors on load
		 */
		feature: [],
	
		/**
		 * Feature plug-ins.
		 * 
		 * This is an object of callbacks which provide the features for DataTables
		 * to be initialised via the `layout` option.
		 */
		features: {},
	
	
		/**
		 * Row searching.
		 * 
		 * This method of searching is complimentary to the default type based
		 * searching, and a lot more comprehensive as it allows you complete control
		 * over the searching logic. Each element in this array is a function
		 * (parameters described below) that is called for every row in the table,
		 * and your logic decides if it should be included in the searching data set
		 * or not.
		 *
		 * Searching functions have the following input parameters:
		 *
		 * 1. `{object}` DataTables settings object: see
		 *    {@link DataTable.models.oSettings}
		 * 2. `{array|object}` Data for the row to be processed (same as the
		 *    original format that was passed in as the data source, or an array
		 *    from a DOM data source
		 * 3. `{int}` Row index ({@link DataTable.models.oSettings.aoData}), which
		 *    can be useful to retrieve the `TR` element if you need DOM interaction.
		 *
		 * And the following return is expected:
		 *
		 * * {boolean} Include the row in the searched result set (true) or not
		 *   (false)
		 *
		 * Note that as with the main search ability in DataTables, technically this
		 * is "filtering", since it is subtractive. However, for consistency in
		 * naming we call it searching here.
		 *
		 *  @type array
		 *  @default []
		 *
		 *  @example
		 *    // The following example shows custom search being applied to the
		 *    // fourth column (i.e. the data[3] index) based on two input values
		 *    // from the end-user, matching the data in a certain range.
		 *    $.fn.dataTable.ext.search.push(
		 *      function( settings, data, dataIndex ) {
		 *        var min = document.getElementById('min').value * 1;
		 *        var max = document.getElementById('max').value * 1;
		 *        var version = data[3] == "-" ? 0 : data[3]*1;
		 *
		 *        if ( min == "" && max == "" ) {
		 *          return true;
		 *        }
		 *        else if ( min == "" && version < max ) {
		 *          return true;
		 *        }
		 *        else if ( min < version && "" == max ) {
		 *          return true;
		 *        }
		 *        else if ( min < version && version < max ) {
		 *          return true;
		 *        }
		 *        return false;
		 *      }
		 *    );
		 */
		search: [],
	
	
		/**
		 * Selector extensions
		 *
		 * The `selector` option can be used to extend the options available for the
		 * selector modifier options (`selector-modifier` object data type) that
		 * each of the three built in selector types offer (row, column and cell +
		 * their plural counterparts). For example the Select extension uses this
		 * mechanism to provide an option to select only rows, columns and cells
		 * that have been marked as selected by the end user (`{selected: true}`),
		 * which can be used in conjunction with the existing built in selector
		 * options.
		 *
		 * Each property is an array to which functions can be pushed. The functions
		 * take three attributes:
		 *
		 * * Settings object for the host table
		 * * Options object (`selector-modifier` object type)
		 * * Array of selected item indexes
		 *
		 * The return is an array of the resulting item indexes after the custom
		 * selector has been applied.
		 *
		 *  @type object
		 */
		selector: {
			cell: [],
			column: [],
			row: []
		},
	
	
		/**
		 * Legacy configuration options. Enable and disable legacy options that
		 * are available in DataTables.
		 *
		 *  @type object
		 */
		legacy: {
			/**
			 * Enable / disable DataTables 1.9 compatible server-side processing
			 * requests
			 *
			 *  @type boolean
			 *  @default null
			 */
			ajax: null
		},
	
	
		/**
		 * Pagination plug-in methods.
		 * 
		 * Each entry in this object is a function and defines which buttons should
		 * be shown by the pagination rendering method that is used for the table:
		 * {@link DataTable.ext.renderer.pageButton}. The renderer addresses how the
		 * buttons are displayed in the document, while the functions here tell it
		 * what buttons to display. This is done by returning an array of button
		 * descriptions (what each button will do).
		 *
		 * Pagination types (the four built in options and any additional plug-in
		 * options defined here) can be used through the `paginationType`
		 * initialisation parameter.
		 *
		 * The functions defined take two parameters:
		 *
		 * 1. `{int} page` The current page index
		 * 2. `{int} pages` The number of pages in the table
		 *
		 * Each function is expected to return an array where each element of the
		 * array can be one of:
		 *
		 * * `first` - Jump to first page when activated
		 * * `last` - Jump to last page when activated
		 * * `previous` - Show previous page when activated
		 * * `next` - Show next page when activated
		 * * `{int}` - Show page of the index given
		 * * `{array}` - A nested array containing the above elements to add a
		 *   containing 'DIV' element (might be useful for styling).
		 *
		 * Note that DataTables v1.9- used this object slightly differently whereby
		 * an object with two functions would be defined for each plug-in. That
		 * ability is still supported by DataTables 1.10+ to provide backwards
		 * compatibility, but this option of use is now decremented and no longer
		 * documented in DataTables 1.10+.
		 *
		 *  @type object
		 *  @default {}
		 *
		 *  @example
		 *    // Show previous, next and current page buttons only
		 *    $.fn.dataTableExt.oPagination.current = function ( page, pages ) {
		 *      return [ 'previous', page, 'next' ];
		 *    };
		 */
		pager: {},
	
	
		renderer: {
			pageButton: {},
			header: {}
		},
	
	
		/**
		 * Ordering plug-ins - custom data source
		 * 
		 * The extension options for ordering of data available here is complimentary
		 * to the default type based ordering that DataTables typically uses. It
		 * allows much greater control over the the data that is being used to
		 * order a column, but is necessarily therefore more complex.
		 * 
		 * This type of ordering is useful if you want to do ordering based on data
		 * live from the DOM (for example the contents of an 'input' element) rather
		 * than just the static string that DataTables knows of.
		 * 
		 * The way these plug-ins work is that you create an array of the values you
		 * wish to be ordering for the column in question and then return that
		 * array. The data in the array much be in the index order of the rows in
		 * the table (not the currently ordering order!). Which order data gathering
		 * function is run here depends on the `dt-init columns.orderDataType`
		 * parameter that is used for the column (if any).
		 *
		 * The functions defined take two parameters:
		 *
		 * 1. `{object}` DataTables settings object: see
		 *    {@link DataTable.models.oSettings}
		 * 2. `{int}` Target column index
		 *
		 * Each function is expected to return an array:
		 *
		 * * `{array}` Data for the column to be ordering upon
		 *
		 *  @type array
		 *
		 *  @example
		 *    // Ordering using `input` node values
		 *    $.fn.dataTable.ext.order['dom-text'] = function  ( settings, col )
		 *    {
		 *      return this.api().column( col, {order:'index'} ).nodes().map( function ( td, i ) {
		 *        return $('input', td).val();
		 *      } );
		 *    }
		 */
		order: {},
	
	
		/**
		 * Type based plug-ins.
		 *
		 * Each column in DataTables has a type assigned to it, either by automatic
		 * detection or by direct assignment using the `type` option for the column.
		 * The type of a column will effect how it is ordering and search (plug-ins
		 * can also make use of the column type if required).
		 *
		 * @namespace
		 */
		type: {
			/**
			 * Automatic column class assignment
			 */
			className: {},
	
			/**
			 * Type detection functions.
			 *
			 * The functions defined in this object are used to automatically detect
			 * a column's type, making initialisation of DataTables super easy, even
			 * when complex data is in the table.
			 *
			 * The functions defined take two parameters:
			 *
		     *  1. `{*}` Data from the column cell to be analysed
		     *  2. `{settings}` DataTables settings object. This can be used to
		     *     perform context specific type detection - for example detection
		     *     based on language settings such as using a comma for a decimal
		     *     place. Generally speaking the options from the settings will not
		     *     be required
			 *
			 * Each function is expected to return:
			 *
			 * * `{string|null}` Data type detected, or null if unknown (and thus
			 *   pass it on to the other type detection functions.
			 *
			 *  @type array
			 *
			 *  @example
			 *    // Currency type detection plug-in:
			 *    $.fn.dataTable.ext.type.detect.push(
			 *      function ( data, settings ) {
			 *        // Check the numeric part
			 *        if ( ! data.substring(1).match(/[0-9]/) ) {
			 *          return null;
			 *        }
			 *
			 *        // Check prefixed by currency
			 *        if ( data.charAt(0) == '$' || data.charAt(0) == '&pound;' ) {
			 *          return 'currency';
			 *        }
			 *        return null;
			 *      }
			 *    );
			 */
			detect: [],
	
			/**
			 * Automatic renderer assignment
			 */
			render: {},
	
	
			/**
			 * Type based search formatting.
			 *
			 * The type based searching functions can be used to pre-format the
			 * data to be search on. For example, it can be used to strip HTML
			 * tags or to de-format telephone numbers for numeric only searching.
			 *
			 * Note that is a search is not defined for a column of a given type,
			 * no search formatting will be performed.
			 * 
			 * Pre-processing of searching data plug-ins - When you assign the sType
			 * for a column (or have it automatically detected for you by DataTables
			 * or a type detection plug-in), you will typically be using this for
			 * custom sorting, but it can also be used to provide custom searching
			 * by allowing you to pre-processing the data and returning the data in
			 * the format that should be searched upon. This is done by adding
			 * functions this object with a parameter name which matches the sType
			 * for that target column. This is the corollary of <i>afnSortData</i>
			 * for searching data.
			 *
			 * The functions defined take a single parameter:
			 *
		     *  1. `{*}` Data from the column cell to be prepared for searching
			 *
			 * Each function is expected to return:
			 *
			 * * `{string|null}` Formatted string that will be used for the searching.
			 *
			 *  @type object
			 *  @default {}
			 *
			 *  @example
			 *    $.fn.dataTable.ext.type.search['title-numeric'] = function ( d ) {
			 *      return d.replace(/\n/g," ").replace( /<.*?>/g, "" );
			 *    }
			 */
			search: {},
	
	
			/**
			 * Type based ordering.
			 *
			 * The column type tells DataTables what ordering to apply to the table
			 * when a column is sorted upon. The order for each type that is defined,
			 * is defined by the functions available in this object.
			 *
			 * Each ordering option can be described by three properties added to
			 * this object:
			 *
			 * * `{type}-pre` - Pre-formatting function
			 * * `{type}-asc` - Ascending order function
			 * * `{type}-desc` - Descending order function
			 *
			 * All three can be used together, only `{type}-pre` or only
			 * `{type}-asc` and `{type}-desc` together. It is generally recommended
			 * that only `{type}-pre` is used, as this provides the optimal
			 * implementation in terms of speed, although the others are provided
			 * for compatibility with existing Javascript sort functions.
			 *
			 * `{type}-pre`: Functions defined take a single parameter:
			 *
		     *  1. `{*}` Data from the column cell to be prepared for ordering
			 *
			 * And return:
			 *
			 * * `{*}` Data to be sorted upon
			 *
			 * `{type}-asc` and `{type}-desc`: Functions are typical Javascript sort
			 * functions, taking two parameters:
			 *
		     *  1. `{*}` Data to compare to the second parameter
		     *  2. `{*}` Data to compare to the first parameter
			 *
			 * And returning:
			 *
			 * * `{*}` Ordering match: <0 if first parameter should be sorted lower
			 *   than the second parameter, ===0 if the two parameters are equal and
			 *   >0 if the first parameter should be sorted height than the second
			 *   parameter.
			 * 
			 *  @type object
			 *  @default {}
			 *
			 *  @example
			 *    // Numeric ordering of formatted numbers with a pre-formatter
			 *    $.extend( $.fn.dataTable.ext.type.order, {
			 *      "string-pre": function(x) {
			 *        a = (a === "-" || a === "") ? 0 : a.replace( /[^\d\-\.]/g, "" );
			 *        return parseFloat( a );
			 *      }
			 *    } );
			 *
			 *  @example
			 *    // Case-sensitive string ordering, with no pre-formatting method
			 *    $.extend( $.fn.dataTable.ext.order, {
			 *      "string-case-asc": function(x,y) {
			 *        return ((x < y) ? -1 : ((x > y) ? 1 : 0));
			 *      },
			 *      "string-case-desc": function(x,y) {
			 *        return ((x < y) ? 1 : ((x > y) ? -1 : 0));
			 *      }
			 *    } );
			 */
			order: {}
		},
	
		/**
		 * Unique DataTables instance counter
		 *
		 * @type int
		 * @private
		 */
		_unique: 0,
	
	
		//
		// Depreciated
		// The following properties are retained for backwards compatibility only.
		// The should not be used in new projects and will be removed in a future
		// version
		//
	
		/**
		 * Version check function.
		 *  @type function
		 *  @depreciated Since 1.10
		 */
		fnVersionCheck: DataTable.fnVersionCheck,
	
	
		/**
		 * Index for what 'this' index API functions should use
		 *  @type int
		 *  @deprecated Since v1.10
		 */
		iApiIndex: 0,
	
	
		/**
		 * Software version
		 *  @type string
		 *  @deprecated Since v1.10
		 */
		sVersion: DataTable.version
	};
	
	
	//
	// Backwards compatibility. Alias to pre 1.10 Hungarian notation counter parts
	//
	$.extend( _ext, {
		afnFiltering: _ext.search,
		aTypes:       _ext.type.detect,
		ofnSearch:    _ext.type.search,
		oSort:        _ext.type.order,
		afnSortData:  _ext.order,
		aoFeatures:   _ext.feature,
		oStdClasses:  _ext.classes,
		oPagination:  _ext.pager
	} );
	
	
	$.extend( DataTable.ext.classes, {
		container: 'dt-container',
		empty: {
			row: 'dt-empty'
		},
		info: {
			container: 'dt-info'
		},
		layout: {
			row: 'dt-layout-row',
			cell: 'dt-layout-cell',
			tableRow: 'dt-layout-table',
			tableCell: '',
			start: 'dt-layout-start',
			end: 'dt-layout-end',
			full: 'dt-layout-full'
		},
		length: {
			container: 'dt-length',
			select: 'dt-input'
		},
		order: {
			canAsc: 'dt-orderable-asc',
			canDesc: 'dt-orderable-desc',
			isAsc: 'dt-ordering-asc',
			isDesc: 'dt-ordering-desc',
			none: 'dt-orderable-none',
			position: 'sorting_'
		},
		processing: {
			container: 'dt-processing'
		},
		scrolling: {
			body: 'dt-scroll-body',
			container: 'dt-scroll',
			footer: {
				self: 'dt-scroll-foot',
				inner: 'dt-scroll-footInner'
			},
			header: {
				self: 'dt-scroll-head',
				inner: 'dt-scroll-headInner'
			}
		},
		search: {
			container: 'dt-search',
			input: 'dt-input'
		},
		table: 'dataTable',	
		tbody: {
			cell: '',
			row: ''
		},
		thead: {
			cell: '',
			row: ''
		},
		tfoot: {
			cell: '',
			row: ''
		},
		paging: {
			active: 'current',
			button: 'dt-paging-button',
			container: 'dt-paging',
			disabled: 'disabled',
			nav: ''
		}
	} );
	
	
	/*
	 * It is useful to have variables which are scoped locally so only the
	 * DataTables functions can access them and they don't leak into global space.
	 * At the same time these functions are often useful over multiple files in the
	 * core and API, so we list, or at least document, all variables which are used
	 * by DataTables as private variables here. This also ensures that there is no
	 * clashing of variable names and that they can easily referenced for reuse.
	 */
	
	
	// Defined else where
	//  _selector_run
	//  _selector_opts
	//  _selector_row_indexes
	
	var _ext; // DataTable.ext
	var _Api; // DataTable.Api
	var _api_register; // DataTable.Api.register
	var _api_registerPlural; // DataTable.Api.registerPlural
	
	var _re_dic = {};
	var _re_new_lines = /[\r\n\u2028]/g;
	var _re_html = /<([^>]*>)/g;
	var _max_str_len = Math.pow(2, 28);
	
	// This is not strict ISO8601 - Date.parse() is quite lax, although
	// implementations differ between browsers.
	var _re_date = /^\d{2,4}[./-]\d{1,2}[./-]\d{1,2}([T ]{1}\d{1,2}[:.]\d{2}([.:]\d{2})?)?$/;
	
	// Escape regular expression special characters
	var _re_escape_regex = new RegExp( '(\\' + [ '/', '.', '*', '+', '?', '|', '(', ')', '[', ']', '{', '}', '\\', '$', '^', '-' ].join('|\\') + ')', 'g' );
	
	// https://en.wikipedia.org/wiki/Foreign_exchange_market
	// - \u20BD - Russian ruble.
	// - \u20a9 - South Korean Won
	// - \u20BA - Turkish Lira
	// - \u20B9 - Indian Rupee
	// - R - Brazil (R$) and South Africa
	// - fr - Swiss Franc
	// - kr - Swedish krona, Norwegian krone and Danish krone
	// - \u2009 is thin space and \u202F is narrow no-break space, both used in many
	// - Ƀ - Bitcoin
	// - Ξ - Ethereum
	//   standards as thousands separators.
	var _re_formatted_numeric = /['\u00A0,$£€¥%\u2009\u202F\u20BD\u20a9\u20BArfkɃΞ]/gi;
	
	
	var _empty = function ( d ) {
		return !d || d === true || d === '-' ? true : false;
	};
	
	
	var _intVal = function ( s ) {
		var integer = parseInt( s, 10 );
		return !isNaN(integer) && isFinite(s) ? integer : null;
	};
	
	// Convert from a formatted number with characters other than `.` as the
	// decimal place, to a Javascript number
	var _numToDecimal = function ( num, decimalPoint ) {
		// Cache created regular expressions for speed as this function is called often
		if ( ! _re_dic[ decimalPoint ] ) {
			_re_dic[ decimalPoint ] = new RegExp( _fnEscapeRegex( decimalPoint ), 'g' );
		}
		return typeof num === 'string' && decimalPoint !== '.' ?
			num.replace( /\./g, '' ).replace( _re_dic[ decimalPoint ], '.' ) :
			num;
	};
	
	
	var _isNumber = function ( d, decimalPoint, formatted, allowEmpty ) {
		var type = typeof d;
		var strType = type === 'string';
	
		if ( type === 'number' || type === 'bigint') {
			return true;
		}
	
		// If empty return immediately so there must be a number if it is a
		// formatted string (this stops the string "k", or "kr", etc being detected
		// as a formatted number for currency
		if ( allowEmpty && _empty( d ) ) {
			return true;
		}
	
		if ( decimalPoint && strType ) {
			d = _numToDecimal( d, decimalPoint );
		}
	
		if ( formatted && strType ) {
			d = d.replace( _re_formatted_numeric, '' );
		}
	
		return !isNaN( parseFloat(d) ) && isFinite( d );
	};
	
	
	// A string without HTML in it can be considered to be HTML still
	var _isHtml = function ( d ) {
		return _empty( d ) || typeof d === 'string';
	};
	
	// Is a string a number surrounded by HTML?
	var _htmlNumeric = function ( d, decimalPoint, formatted, allowEmpty ) {
		if ( allowEmpty && _empty( d ) ) {
			return true;
		}
	
		// input and select strings mean that this isn't just a number
		if (typeof d === 'string' && d.match(/<(input|select)/i)) {
			return null;
		}
	
		var html = _isHtml( d );
		return ! html ?
			null :
			_isNumber( _stripHtml( d ), decimalPoint, formatted, allowEmpty ) ?
				true :
				null;
	};
	
	
	var _pluck = function ( a, prop, prop2 ) {
		var out = [];
		var i=0, ien=a.length;
	
		// Could have the test in the loop for slightly smaller code, but speed
		// is essential here
		if ( prop2 !== undefined ) {
			for ( ; i<ien ; i++ ) {
				if ( a[i] && a[i][ prop ] ) {
					out.push( a[i][ prop ][ prop2 ] );
				}
			}
		}
		else {
			for ( ; i<ien ; i++ ) {
				if ( a[i] ) {
					out.push( a[i][ prop ] );
				}
			}
		}
	
		return out;
	};
	
	
	// Basically the same as _pluck, but rather than looping over `a` we use `order`
	// as the indexes to pick from `a`
	var _pluck_order = function ( a, order, prop, prop2 )
	{
		var out = [];
		var i=0, ien=order.length;
	
		// Could have the test in the loop for slightly smaller code, but speed
		// is essential here
		if ( prop2 !== undefined ) {
			for ( ; i<ien ; i++ ) {
				if ( a[ order[i] ] && a[ order[i] ][ prop ] ) {
					out.push( a[ order[i] ][ prop ][ prop2 ] );
				}
			}
		}
		else {
			for ( ; i<ien ; i++ ) {
				if ( a[ order[i] ] ) {
					out.push( a[ order[i] ][ prop ] );
				}
			}
		}
	
		return out;
	};
	
	
	var _range = function ( len, start )
	{
		var out = [];
		var end;
	
		if ( start === undefined ) {
			start = 0;
			end = len;
		}
		else {
			end = start;
			start = len;
		}
	
		for ( var i=start ; i<end ; i++ ) {
			out.push( i );
		}
	
		return out;
	};
	
	
	var _removeEmpty = function ( a )
	{
		var out = [];
	
		for ( var i=0, ien=a.length ; i<ien ; i++ ) {
			if ( a[i] ) { // careful - will remove all falsy values!
				out.push( a[i] );
			}
		}
	
		return out;
	};
	
	// Replaceable function in api.util
	var _stripHtml = function (input) {
		if (! input || typeof input !== 'string') {
			return input;
		}
	
		// Irrelevant check to workaround CodeQL's false positive on the regex
		if (input.length > _max_str_len) {
			throw new Error('Exceeded max str len');
		}
	
		var previous;
	
		input = input.replace(_re_html, ''); // Complete tags
	
		// Safety for incomplete script tag - use do / while to ensure that
		// we get all instances
		do {
			previous = input;
			input = input.replace(/<script/i, '');
		} while (input !== previous);
	
		return previous;
	};
	
	// Replaceable function in api.util
	var _escapeHtml = function ( d ) {
		if (Array.isArray(d)) {
			d = d.join(',');
		}
	
		return typeof d === 'string' ?
			d
				.replace(/&/g, '&amp;')
				.replace(/</g, '&lt;')
				.replace(/>/g, '&gt;')
				.replace(/"/g, '&quot;') :
			d;
	};
	
	// Remove diacritics from a string by decomposing it and then removing
	// non-ascii characters
	var _normalize = function (str, both) {
		if (typeof str !== 'string') {
			return str;
		}
	
		// It is faster to just run `normalize` than it is to check if
		// we need to with a regex! (Check as it isn't available in old
		// Safari)
		var res = str.normalize
			? str.normalize("NFD")
			: str;
	
		// Equally, here we check if a regex is needed or not
		return res.length !== str.length
			? (both === true ? str + ' ' : '' ) + res.replace(/[\u0300-\u036f]/g, "")
			: res;
	}
	
	/**
	 * Determine if all values in the array are unique. This means we can short
	 * cut the _unique method at the cost of a single loop. A sorted array is used
	 * to easily check the values.
	 *
	 * @param  {array} src Source array
	 * @return {boolean} true if all unique, false otherwise
	 * @ignore
	 */
	var _areAllUnique = function ( src ) {
		if ( src.length < 2 ) {
			return true;
		}
	
		var sorted = src.slice().sort();
		var last = sorted[0];
	
		for ( var i=1, ien=sorted.length ; i<ien ; i++ ) {
			if ( sorted[i] === last ) {
				return false;
			}
	
			last = sorted[i];
		}
	
		return true;
	};
	
	
	/**
	 * Find the unique elements in a source array.
	 *
	 * @param  {array} src Source array
	 * @return {array} Array of unique items
	 * @ignore
	 */
	var _unique = function ( src )
	{
		if (Array.from && Set) {
			return Array.from(new Set(src));
		}
	
		if ( _areAllUnique( src ) ) {
			return src.slice();
		}
	
		// A faster unique method is to use object keys to identify used values,
		// but this doesn't work with arrays or objects, which we must also
		// consider. See jsperf.app/compare-array-unique-versions/4 for more
		// information.
		var
			out = [],
			val,
			i, ien=src.length,
			j, k=0;
	
		again: for ( i=0 ; i<ien ; i++ ) {
			val = src[i];
	
			for ( j=0 ; j<k ; j++ ) {
				if ( out[j] === val ) {
					continue again;
				}
			}
	
			out.push( val );
			k++;
		}
	
		return out;
	};
	
	// Surprisingly this is faster than [].concat.apply
	// https://jsperf.com/flatten-an-array-loop-vs-reduce/2
	var _flatten = function (out, val) {
		if (Array.isArray(val)) {
			for (var i=0 ; i<val.length ; i++) {
				_flatten(out, val[i]);
			}
		}
		else {
			out.push(val);
		}
	
		return out;
	}
	
	// Similar to jQuery's addClass, but use classList.add
	function _addClass(el, name) {
		if (name) {
			name.split(' ').forEach(function (n) {
				if (n) {
					// `add` does deduplication, so no need to check `contains`
					el.classList.add(n);
				}
			});
		}
	}
	
	/**
	 * DataTables utility methods
	 * 
	 * This namespace provides helper methods that DataTables uses internally to
	 * create a DataTable, but which are not exclusively used only for DataTables.
	 * These methods can be used by extension authors to save the duplication of
	 * code.
	 *
	 *  @namespace
	 */
	DataTable.util = {
		/**
		 * Return a string with diacritic characters decomposed
		 * @param {*} mixed Function or string to normalize
		 * @param {*} both Return original string and the normalized string
		 * @returns String or undefined
		 */
		diacritics: function (mixed, both) {
			var type = typeof mixed;
	
			if (type !== 'function') {
				return _normalize(mixed, both);
			}
			_normalize = mixed;
		},
	
		/**
		 * Debounce a function
		 *
		 * @param {function} fn Function to be called
		 * @param {integer} freq Call frequency in mS
		 * @return {function} Wrapped function
		 */
		debounce: function ( fn, timeout ) {
			var timer;
	
			return function () {
				var that = this;
				var args = arguments;
	
				clearTimeout(timer);
	
				timer = setTimeout( function () {
					fn.apply(that, args);
				}, timeout || 250 );
			};
		},
	
		/**
		 * Throttle the calls to a function. Arguments and context are maintained
		 * for the throttled function.
		 *
		 * @param {function} fn Function to be called
		 * @param {integer} freq Call frequency in mS
		 * @return {function} Wrapped function
		 */
		throttle: function ( fn, freq ) {
			var
				frequency = freq !== undefined ? freq : 200,
				last,
				timer;
	
			return function () {
				var
					that = this,
					now  = +new Date(),
					args = arguments;
	
				if ( last && now < last + frequency ) {
					clearTimeout( timer );
	
					timer = setTimeout( function () {
						last = undefined;
						fn.apply( that, args );
					}, frequency );
				}
				else {
					last = now;
					fn.apply( that, args );
				}
			};
		},
	
		/**
		 * Escape a string such that it can be used in a regular expression
		 *
		 *  @param {string} val string to escape
		 *  @returns {string} escaped string
		 */
		escapeRegex: function ( val ) {
			return val.replace( _re_escape_regex, '\\$1' );
		},
	
		/**
		 * Create a function that will write to a nested object or array
		 * @param {*} source JSON notation string
		 * @returns Write function
		 */
		set: function ( source ) {
			if ( $.isPlainObject( source ) ) {
				/* Unlike get, only the underscore (global) option is used for for
				 * setting data since we don't know the type here. This is why an object
				 * option is not documented for `mData` (which is read/write), but it is
				 * for `mRender` which is read only.
				 */
				return DataTable.util.set( source._ );
			}
			else if ( source === null ) {
				// Nothing to do when the data source is null
				return function () {};
			}
			else if ( typeof source === 'function' ) {
				return function (data, val, meta) {
					source( data, 'set', val, meta );
				};
			}
			else if (
				typeof source === 'string' && (source.indexOf('.') !== -1 ||
				source.indexOf('[') !== -1 || source.indexOf('(') !== -1)
			) {
				// Like the get, we need to get data from a nested object
				var setData = function (data, val, src) {
					var a = _fnSplitObjNotation( src ), b;
					var aLast = a[a.length-1];
					var arrayNotation, funcNotation, o, innerSrc;
		
					for ( var i=0, iLen=a.length-1 ; i<iLen ; i++ ) {
						// Protect against prototype pollution
						if (a[i] === '__proto__' || a[i] === 'constructor') {
							throw new Error('Cannot set prototype values');
						}
		
						// Check if we are dealing with an array notation request
						arrayNotation = a[i].match(__reArray);
						funcNotation = a[i].match(__reFn);
		
						if ( arrayNotation ) {
							a[i] = a[i].replace(__reArray, '');
							data[ a[i] ] = [];
		
							// Get the remainder of the nested object to set so we can recurse
							b = a.slice();
							b.splice( 0, i+1 );
							innerSrc = b.join('.');
		
							// Traverse each entry in the array setting the properties requested
							if ( Array.isArray( val ) ) {
								for ( var j=0, jLen=val.length ; j<jLen ; j++ ) {
									o = {};
									setData( o, val[j], innerSrc );
									data[ a[i] ].push( o );
								}
							}
							else {
								// We've been asked to save data to an array, but it
								// isn't array data to be saved. Best that can be done
								// is to just save the value.
								data[ a[i] ] = val;
							}
		
							// The inner call to setData has already traversed through the remainder
							// of the source and has set the data, thus we can exit here
							return;
						}
						else if ( funcNotation ) {
							// Function call
							a[i] = a[i].replace(__reFn, '');
							data = data[ a[i] ]( val );
						}
		
						// If the nested object doesn't currently exist - since we are
						// trying to set the value - create it
						if ( data[ a[i] ] === null || data[ a[i] ] === undefined ) {
							data[ a[i] ] = {};
						}
						data = data[ a[i] ];
					}
		
					// Last item in the input - i.e, the actual set
					if ( aLast.match(__reFn ) ) {
						// Function call
						data = data[ aLast.replace(__reFn, '') ]( val );
					}
					else {
						// If array notation is used, we just want to strip it and use the property name
						// and assign the value. If it isn't used, then we get the result we want anyway
						data[ aLast.replace(__reArray, '') ] = val;
					}
				};
		
				return function (data, val) { // meta is also passed in, but not used
					return setData( data, val, source );
				};
			}
			else {
				// Array or flat object mapping
				return function (data, val) { // meta is also passed in, but not used
					data[source] = val;
				};
			}
		},
	
		/**
		 * Create a function that will read nested objects from arrays, based on JSON notation
		 * @param {*} source JSON notation string
		 * @returns Value read
		 */
		get: function ( source ) {
			if ( $.isPlainObject( source ) ) {
				// Build an object of get functions, and wrap them in a single call
				var o = {};
				$.each( source, function (key, val) {
					if ( val ) {
						o[key] = DataTable.util.get( val );
					}
				} );
		
				return function (data, type, row, meta) {
					var t = o[type] || o._;
					return t !== undefined ?
						t(data, type, row, meta) :
						data;
				};
			}
			else if ( source === null ) {
				// Give an empty string for rendering / sorting etc
				return function (data) { // type, row and meta also passed, but not used
					return data;
				};
			}
			else if ( typeof source === 'function' ) {
				return function (data, type, row, meta) {
					return source( data, type, row, meta );
				};
			}
			else if (
				typeof source === 'string' && (source.indexOf('.') !== -1 ||
				source.indexOf('[') !== -1 || source.indexOf('(') !== -1)
			) {
				/* If there is a . in the source string then the data source is in a
				 * nested object so we loop over the data for each level to get the next
				 * level down. On each loop we test for undefined, and if found immediately
				 * return. This allows entire objects to be missing and sDefaultContent to
				 * be used if defined, rather than throwing an error
				 */
				var fetchData = function (data, type, src) {
					var arrayNotation, funcNotation, out, innerSrc;
		
					if ( src !== "" ) {
						var a = _fnSplitObjNotation( src );
		
						for ( var i=0, iLen=a.length ; i<iLen ; i++ ) {
							// Check if we are dealing with special notation
							arrayNotation = a[i].match(__reArray);
							funcNotation = a[i].match(__reFn);
		
							if ( arrayNotation ) {
								// Array notation
								a[i] = a[i].replace(__reArray, '');
		
								// Condition allows simply [] to be passed in
								if ( a[i] !== "" ) {
									data = data[ a[i] ];
								}
								out = [];
		
								// Get the remainder of the nested object to get
								a.splice( 0, i+1 );
								innerSrc = a.join('.');
		
								// Traverse each entry in the array getting the properties requested
								if ( Array.isArray( data ) ) {
									for ( var j=0, jLen=data.length ; j<jLen ; j++ ) {
										out.push( fetchData( data[j], type, innerSrc ) );
									}
								}
		
								// If a string is given in between the array notation indicators, that
								// is used to join the strings together, otherwise an array is returned
								var join = arrayNotation[0].substring(1, arrayNotation[0].length-1);
								data = (join==="") ? out : out.join(join);
		
								// The inner call to fetchData has already traversed through the remainder
								// of the source requested, so we exit from the loop
								break;
							}
							else if ( funcNotation ) {
								// Function call
								a[i] = a[i].replace(__reFn, '');
								data = data[ a[i] ]();
								continue;
							}
		
							if (data === null || data[ a[i] ] === null) {
								return null;
							}
							else if ( data === undefined || data[ a[i] ] === undefined ) {
								return undefined;
							}
	
							data = data[ a[i] ];
						}
					}
		
					return data;
				};
		
				return function (data, type) { // row and meta also passed, but not used
					return fetchData( data, type, source );
				};
			}
			else {
				// Array or flat object mapping
				return function (data) { // row and meta also passed, but not used
					return data[source];
				};
			}
		},
	
		stripHtml: function (mixed) {
			var type = typeof mixed;
	
			if (type === 'function') {
				_stripHtml = mixed;
				return;
			}
			else if (type === 'string') {
				return _stripHtml(mixed);
			}
			return mixed;
		},
	
		escapeHtml: function (mixed) {
			var type = typeof mixed;
	
			if (type === 'function') {
				_escapeHtml = mixed;
				return;
			}
			else if (type === 'string' || Array.isArray(mixed)) {
				return _escapeHtml(mixed);
			}
			return mixed;
		},
	
		unique: _unique
	};
	
	
	
	/**
	 * Create a mapping object that allows camel case parameters to be looked up
	 * for their Hungarian counterparts. The mapping is stored in a private
	 * parameter called `_hungarianMap` which can be accessed on the source object.
	 *  @param {object} o
	 *  @memberof DataTable#oApi
	 */
	function _fnHungarianMap ( o )
	{
		var
			hungarian = 'a aa ai ao as b fn i m o s ',
			match,
			newKey,
			map = {};
	
		$.each( o, function (key) {
			match = key.match(/^([^A-Z]+?)([A-Z])/);
	
			if ( match && hungarian.indexOf(match[1]+' ') !== -1 )
			{
				newKey = key.replace( match[0], match[2].toLowerCase() );
				map[ newKey ] = key;
	
				if ( match[1] === 'o' )
				{
					_fnHungarianMap( o[key] );
				}
			}
		} );
	
		o._hungarianMap = map;
	}
	
	
	/**
	 * Convert from camel case parameters to Hungarian, based on a Hungarian map
	 * created by _fnHungarianMap.
	 *  @param {object} src The model object which holds all parameters that can be
	 *    mapped.
	 *  @param {object} user The object to convert from camel case to Hungarian.
	 *  @param {boolean} force When set to `true`, properties which already have a
	 *    Hungarian value in the `user` object will be overwritten. Otherwise they
	 *    won't be.
	 *  @memberof DataTable#oApi
	 */
	function _fnCamelToHungarian ( src, user, force )
	{
		if ( ! src._hungarianMap ) {
			_fnHungarianMap( src );
		}
	
		var hungarianKey;
	
		$.each( user, function (key) {
			hungarianKey = src._hungarianMap[ key ];
	
			if ( hungarianKey !== undefined && (force || user[hungarianKey] === undefined) )
			{
				// For objects, we need to buzz down into the object to copy parameters
				if ( hungarianKey.charAt(0) === 'o' )
				{
					// Copy the camelCase options over to the hungarian
					if ( ! user[ hungarianKey ] ) {
						user[ hungarianKey ] = {};
					}
					$.extend( true, user[hungarianKey], user[key] );
	
					_fnCamelToHungarian( src[hungarianKey], user[hungarianKey], force );
				}
				else {
					user[hungarianKey] = user[ key ];
				}
			}
		} );
	}
	
	/**
	 * Map one parameter onto another
	 *  @param {object} o Object to map
	 *  @param {*} knew The new parameter name
	 *  @param {*} old The old parameter name
	 */
	var _fnCompatMap = function ( o, knew, old ) {
		if ( o[ knew ] !== undefined ) {
			o[ old ] = o[ knew ];
		}
	};
	
	
	/**
	 * Provide backwards compatibility for the main DT options. Note that the new
	 * options are mapped onto the old parameters, so this is an external interface
	 * change only.
	 *  @param {object} init Object to map
	 */
	function _fnCompatOpts ( init )
	{
		_fnCompatMap( init, 'ordering',      'bSort' );
		_fnCompatMap( init, 'orderMulti',    'bSortMulti' );
		_fnCompatMap( init, 'orderClasses',  'bSortClasses' );
		_fnCompatMap( init, 'orderCellsTop', 'bSortCellsTop' );
		_fnCompatMap( init, 'order',         'aaSorting' );
		_fnCompatMap( init, 'orderFixed',    'aaSortingFixed' );
		_fnCompatMap( init, 'paging',        'bPaginate' );
		_fnCompatMap( init, 'pagingType',    'sPaginationType' );
		_fnCompatMap( init, 'pageLength',    'iDisplayLength' );
		_fnCompatMap( init, 'searching',     'bFilter' );
	
		// Boolean initialisation of x-scrolling
		if ( typeof init.sScrollX === 'boolean' ) {
			init.sScrollX = init.sScrollX ? '100%' : '';
		}
		if ( typeof init.scrollX === 'boolean' ) {
			init.scrollX = init.scrollX ? '100%' : '';
		}
	
		// Objects for ordering
		if ( typeof init.bSort === 'object' ) {
			init.orderIndicators = init.bSort.indicators !== undefined ? init.bSort.indicators : true;
			init.orderHandler = init.bSort.handler !== undefined ? init.bSort.handler : true;
			init.bSort = true;
		}
		else if (init.bSort === false) {
			init.orderIndicators = false;
			init.orderHandler = false;
		}
		else if (init.bSort === true) {
			init.orderIndicators = true;
			init.orderHandler = true;
		}
	
		// Which cells are the title cells?
		if (typeof init.bSortCellsTop === 'boolean') {
			init.titleRow = init.bSortCellsTop;
		}
	
		// Column search objects are in an array, so it needs to be converted
		// element by element
		var searchCols = init.aoSearchCols;
	
		if ( searchCols ) {
			for ( var i=0, ien=searchCols.length ; i<ien ; i++ ) {
				if ( searchCols[i] ) {
					_fnCamelToHungarian( DataTable.models.oSearch, searchCols[i] );
				}
			}
		}
	
		// Enable search delay if server-side processing is enabled
		if (init.serverSide && ! init.searchDelay) {
			init.searchDelay = 400;
		}
	}
	
	
	/**
	 * Provide backwards compatibility for column options. Note that the new options
	 * are mapped onto the old parameters, so this is an external interface change
	 * only.
	 *  @param {object} init Object to map
	 */
	function _fnCompatCols ( init )
	{
		_fnCompatMap( init, 'orderable',     'bSortable' );
		_fnCompatMap( init, 'orderData',     'aDataSort' );
		_fnCompatMap( init, 'orderSequence', 'asSorting' );
		_fnCompatMap( init, 'orderDataType', 'sortDataType' );
	
		// orderData can be given as an integer
		var dataSort = init.aDataSort;
		if ( typeof dataSort === 'number' && ! Array.isArray( dataSort ) ) {
			init.aDataSort = [ dataSort ];
		}
	}
	
	
	/**
	 * Browser feature detection for capabilities, quirks
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnBrowserDetect( settings )
	{
		// We don't need to do this every time DataTables is constructed, the values
		// calculated are specific to the browser and OS configuration which we
		// don't expect to change between initialisations
		if ( ! DataTable.__browser ) {
			var browser = {};
			DataTable.__browser = browser;
	
			// Scrolling feature / quirks detection
			var n = $('<div/>')
				.css( {
					position: 'fixed',
					top: 0,
					left: -1 * window.pageXOffset, // allow for scrolling
					height: 1,
					width: 1,
					overflow: 'hidden'
				} )
				.append(
					$('<div/>')
						.css( {
							position: 'absolute',
							top: 1,
							left: 1,
							width: 100,
							overflow: 'scroll'
						} )
						.append(
							$('<div/>')
								.css( {
									width: '100%',
									height: 10
								} )
						)
				)
				.appendTo( 'body' );
	
			var outer = n.children();
			var inner = outer.children();
	
			// Get scrollbar width
			browser.barWidth = outer[0].offsetWidth - outer[0].clientWidth;
	
			// In rtl text layout, some browsers (most, but not all) will place the
			// scrollbar on the left, rather than the right.
			browser.bScrollbarLeft = Math.round( inner.offset().left ) !== 1;
	
			n.remove();
		}
	
		$.extend( settings.oBrowser, DataTable.__browser );
		settings.oScroll.iBarWidth = DataTable.__browser.barWidth;
	}
	
	/**
	 * Add a column to the list used for the table with default values
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnAddColumn( oSettings )
	{
		// Add column to aoColumns array
		var oDefaults = DataTable.defaults.column;
		var iCol = oSettings.aoColumns.length;
		var oCol = $.extend( {}, DataTable.models.oColumn, oDefaults, {
			"aDataSort": oDefaults.aDataSort ? oDefaults.aDataSort : [iCol],
			"mData": oDefaults.mData ? oDefaults.mData : iCol,
			idx: iCol,
			searchFixed: {},
			colEl: $('<col>').attr('data-dt-column', iCol)
		} );
		oSettings.aoColumns.push( oCol );
	
		// Add search object for column specific search. Note that the `searchCols[ iCol ]`
		// passed into extend can be undefined. This allows the user to give a default
		// with only some of the parameters defined, and also not give a default
		var searchCols = oSettings.aoPreSearchCols;
		searchCols[ iCol ] = $.extend( {}, DataTable.models.oSearch, searchCols[ iCol ] );
	}
	
	
	/**
	 * Apply options for a column
	 *  @param {object} oSettings dataTables settings object
	 *  @param {int} iCol column index to consider
	 *  @param {object} oOptions object with sType, bVisible and bSearchable etc
	 *  @memberof DataTable#oApi
	 */
	function _fnColumnOptions( oSettings, iCol, oOptions )
	{
		var oCol = oSettings.aoColumns[ iCol ];
	
		/* User specified column options */
		if ( oOptions !== undefined && oOptions !== null )
		{
			// Backwards compatibility
			_fnCompatCols( oOptions );
	
			// Map camel case parameters to their Hungarian counterparts
			_fnCamelToHungarian( DataTable.defaults.column, oOptions, true );
	
			/* Backwards compatibility for mDataProp */
			if ( oOptions.mDataProp !== undefined && !oOptions.mData )
			{
				oOptions.mData = oOptions.mDataProp;
			}
	
			if ( oOptions.sType )
			{
				oCol._sManualType = oOptions.sType;
			}
		
			// `class` is a reserved word in Javascript, so we need to provide
			// the ability to use a valid name for the camel case input
			if ( oOptions.className && ! oOptions.sClass )
			{
				oOptions.sClass = oOptions.className;
			}
	
			var origClass = oCol.sClass;
	
			$.extend( oCol, oOptions );
			_fnMap( oCol, oOptions, "sWidth", "sWidthOrig" );
	
			// Merge class from previously defined classes with this one, rather than just
			// overwriting it in the extend above
			if (origClass !== oCol.sClass) {
				oCol.sClass = origClass + ' ' + oCol.sClass;
			}
	
			/* iDataSort to be applied (backwards compatibility), but aDataSort will take
			 * priority if defined
			 */
			if ( oOptions.iDataSort !== undefined )
			{
				oCol.aDataSort = [ oOptions.iDataSort ];
			}
			_fnMap( oCol, oOptions, "aDataSort" );
		}
	
		/* Cache the data get and set functions for speed */
		var mDataSrc = oCol.mData;
		var mData = _fnGetObjectDataFn( mDataSrc );
	
		// The `render` option can be given as an array to access the helper rendering methods.
		// The first element is the rendering method to use, the rest are the parameters to pass
		if ( oCol.mRender && Array.isArray( oCol.mRender ) ) {
			var copy = oCol.mRender.slice();
			var name = copy.shift();
	
			oCol.mRender = DataTable.render[name].apply(window, copy);
		}
	
		oCol._render = oCol.mRender ? _fnGetObjectDataFn( oCol.mRender ) : null;
	
		var attrTest = function( src ) {
			return typeof src === 'string' && src.indexOf('@') !== -1;
		};
		oCol._bAttrSrc = $.isPlainObject( mDataSrc ) && (
			attrTest(mDataSrc.sort) || attrTest(mDataSrc.type) || attrTest(mDataSrc.filter)
		);
		oCol._setter = null;
	
		oCol.fnGetData = function (rowData, type, meta) {
			var innerData = mData( rowData, type, undefined, meta );
	
			return oCol._render && type ?
				oCol._render( innerData, type, rowData, meta ) :
				innerData;
		};
		oCol.fnSetData = function ( rowData, val, meta ) {
			return _fnSetObjectDataFn( mDataSrc )( rowData, val, meta );
		};
	
		// Indicate if DataTables should read DOM data as an object or array
		// Used in _fnGetRowElements
		if ( typeof mDataSrc !== 'number' && ! oCol._isArrayHost ) {
			oSettings._rowReadObject = true;
		}
	
		/* Feature sorting overrides column specific when off */
		if ( !oSettings.oFeatures.bSort )
		{
			oCol.bSortable = false;
		}
	}
	
	
	/**
	 * Adjust the table column widths for new data. Note: you would probably want to
	 * do a redraw after calling this function!
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnAdjustColumnSizing ( settings )
	{
		_fnCalculateColumnWidths( settings );
		_fnColumnSizes( settings );
	
		var scroll = settings.oScroll;
		if ( scroll.sY !== '' || scroll.sX !== '') {
			_fnScrollDraw( settings );
		}
	
		_fnCallbackFire( settings, null, 'column-sizing', [settings] );
	}
	
	/**
	 * Apply column sizes
	 *
	 * @param {*} settings DataTables settings object
	 */
	function _fnColumnSizes ( settings )
	{
		var cols = settings.aoColumns;
	
		for (var i=0 ; i<cols.length ; i++) {
			var width = _fnColumnsSumWidth(settings, [i], false, false);
	
			cols[i].colEl.css('width', width);
	
			if (settings.oScroll.sX) {
				cols[i].colEl.css('min-width', width);
			}
		}
	}
	
	
	/**
	 * Convert the index of a visible column to the index in the data array (take account
	 * of hidden columns)
	 *  @param {object} oSettings dataTables settings object
	 *  @param {int} iMatch Visible column index to lookup
	 *  @returns {int} i the data index
	 *  @memberof DataTable#oApi
	 */
	function _fnVisibleToColumnIndex( oSettings, iMatch )
	{
		var aiVis = _fnGetColumns( oSettings, 'bVisible' );
	
		return typeof aiVis[iMatch] === 'number' ?
			aiVis[iMatch] :
			null;
	}
	
	
	/**
	 * Convert the index of an index in the data array and convert it to the visible
	 *   column index (take account of hidden columns)
	 *  @param {int} iMatch Column index to lookup
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {int} i the data index
	 *  @memberof DataTable#oApi
	 */
	function _fnColumnIndexToVisible( oSettings, iMatch )
	{
		var aiVis = _fnGetColumns( oSettings, 'bVisible' );
		var iPos = aiVis.indexOf(iMatch);
	
		return iPos !== -1 ? iPos : null;
	}
	
	
	/**
	 * Get the number of visible columns
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {int} i the number of visible columns
	 *  @memberof DataTable#oApi
	 */
	function _fnVisbleColumns( settings )
	{
		var layout = settings.aoHeader;
		var columns = settings.aoColumns;
		var vis = 0;
	
		if ( layout.length ) {
			for ( var i=0, ien=layout[0].length ; i<ien ; i++ ) {
				if ( columns[i].bVisible && $(layout[0][i].cell).css('display') !== 'none' ) {
					vis++;
				}
			}
		}
	
		return vis;
	}
	
	
	/**
	 * Get an array of column indexes that match a given property
	 *  @param {object} oSettings dataTables settings object
	 *  @param {string} sParam Parameter in aoColumns to look for - typically
	 *    bVisible or bSearchable
	 *  @returns {array} Array of indexes with matched properties
	 *  @memberof DataTable#oApi
	 */
	function _fnGetColumns( oSettings, sParam )
	{
		var a = [];
	
		oSettings.aoColumns.map( function(val, i) {
			if ( val[sParam] ) {
				a.push( i );
			}
		} );
	
		return a;
	}
	
	/**
	 * Allow the result from a type detection function to be `true` while
	 * translating that into a string. Old type detection functions will
	 * return the type name if it passes. An obect store would be better,
	 * but not backwards compatible.
	 *
	 * @param {*} typeDetect Object or function for type detection
	 * @param {*} res Result from the type detection function
	 * @returns Type name or false
	 */
	function _typeResult (typeDetect, res) {
		return res === true
			? typeDetect._name
			: res;
	}
	
	/**
	 * Calculate the 'type' of a column
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnColumnTypes ( settings )
	{
		var columns = settings.aoColumns;
		var data = settings.aoData;
		var types = DataTable.ext.type.detect;
		var i, ien, j, jen, k, ken;
		var col, detectedType, cache;
	
		// For each column, spin over the data type detection functions, seeing if one matches
		for ( i=0, ien=columns.length ; i<ien ; i++ ) {
			col = columns[i];
			cache = [];
	
			if ( ! col.sType && col._sManualType ) {
				col.sType = col._sManualType;
			}
			else if ( ! col.sType ) {
				// With SSP type detection can be unreliable and error prone, so we provide a way
				// to turn it off.
				if (! settings.typeDetect) {
					return;
				}
	
				for ( j=0, jen=types.length ; j<jen ; j++ ) {
					var typeDetect = types[j];
	
					// There can be either one, or three type detection functions
					var oneOf = typeDetect.oneOf;
					var allOf = typeDetect.allOf || typeDetect;
					var init = typeDetect.init;
					var one = false;
	
					detectedType = null;
	
					// Fast detect based on column assignment
					if (init) {
						detectedType = _typeResult(typeDetect, init(settings, col, i));
	
						if (detectedType) {
							col.sType = detectedType;
							break;
						}
					}
	
					for ( k=0, ken=data.length ; k<ken ; k++ ) {
						if (! data[k]) {
							continue;
						}
	
						// Use a cache array so we only need to get the type data
						// from the formatter once (when using multiple detectors)
						if ( cache[k] === undefined ) {
							cache[k] = _fnGetCellData( settings, k, i, 'type' );
						}
	
						// Only one data point in the column needs to match this function
						if (oneOf && ! one) {
							one = _typeResult(typeDetect, oneOf( cache[k], settings ));
						}
	
						// All data points need to match this function
						detectedType = _typeResult(typeDetect, allOf( cache[k], settings ));
	
						// If null, then this type can't apply to this column, so
						// rather than testing all cells, break out. There is an
						// exception for the last type which is `html`. We need to
						// scan all rows since it is possible to mix string and HTML
						// types
						if ( ! detectedType && j !== types.length-3 ) {
							break;
						}
	
						// Only a single match is needed for html type since it is
						// bottom of the pile and very similar to string - but it
						// must not be empty
						if ( detectedType === 'html' && ! _empty(cache[k]) ) {
							break;
						}
					}
	
					// Type is valid for all data points in the column - use this
					// type
					if ( (oneOf && one && detectedType) || (!oneOf && detectedType) ) {
						col.sType = detectedType;
						break;
					}
				}
	
				// Fall back - if no type was detected, always use string
				if ( ! col.sType ) {
					col.sType = 'string';
				}
			}
	
			// Set class names for header / footer for auto type classes
			var autoClass = _ext.type.className[col.sType];
	
			if (autoClass) {
				_columnAutoClass(settings.aoHeader, i, autoClass);
				_columnAutoClass(settings.aoFooter, i, autoClass);
			}
	
			var renderer = _ext.type.render[col.sType];
	
			// This can only happen once! There is no way to remove
			// a renderer. After the first time the renderer has
			// already been set so createTr will run the renderer itself.
			if (renderer && ! col._render) {
				col._render = DataTable.util.get(renderer);
	
				_columnAutoRender(settings, i);
			}
		}
	}
	
	/**
	 * Apply an auto detected renderer to data which doesn't yet have
	 * a renderer
	 */
	function _columnAutoRender(settings, colIdx) {
		var data = settings.aoData;
	
		for (var i=0 ; i<data.length ; i++) {
			if (data[i].nTr) {
				// We have to update the display here since there is no
				// invalidation check for the data
				var display = _fnGetCellData( settings, i, colIdx, 'display' );
	
				data[i].displayData[colIdx] = display;
				_fnWriteCell(data[i].anCells[colIdx], display);
	
				// No need to update sort / filter data since it has
				// been invalidated and will be re-read with the
				// renderer now applied
			}
		}
	}
	
	/**
	 * Apply a class name to a column's header cells
	 */
	function _columnAutoClass(container, colIdx, className) {
		container.forEach(function (row) {
			if (row[colIdx] && row[colIdx].unique) {
				_addClass(row[colIdx].cell, className);
			}
		});
	}
	
	/**
	 * Take the column definitions and static columns arrays and calculate how
	 * they relate to column indexes. The callback function will then apply the
	 * definition found for a column to a suitable configuration object.
	 *  @param {object} oSettings dataTables settings object
	 *  @param {array} aoColDefs The aoColumnDefs array that is to be applied
	 *  @param {array} aoCols The aoColumns array that defines columns individually
	 *  @param {array} headerLayout Layout for header as it was loaded
	 *  @param {function} fn Callback function - takes two parameters, the calculated
	 *    column index and the definition for that column.
	 *  @memberof DataTable#oApi
	 */
	function _fnApplyColumnDefs( oSettings, aoColDefs, aoCols, headerLayout, fn )
	{
		var i, iLen, j, jLen, k, kLen, def;
		var columns = oSettings.aoColumns;
	
		if ( aoCols ) {
			for ( i=0, iLen=aoCols.length ; i<iLen ; i++ ) {
				if (aoCols[i] && aoCols[i].name) {
					columns[i].sName = aoCols[i].name;
				}
			}
		}
	
		// Column definitions with aTargets
		if ( aoColDefs )
		{
			/* Loop over the definitions array - loop in reverse so first instance has priority */
			for ( i=aoColDefs.length-1 ; i>=0 ; i-- )
			{
				def = aoColDefs[i];
	
				/* Each definition can target multiple columns, as it is an array */
				var aTargets = def.target !== undefined
					? def.target
					: def.targets !== undefined
						? def.targets
						: def.aTargets;
	
				if ( ! Array.isArray( aTargets ) )
				{
					aTargets = [ aTargets ];
				}
	
				for ( j=0, jLen=aTargets.length ; j<jLen ; j++ )
				{
					var target = aTargets[j];
	
					if ( typeof target === 'number' && target >= 0 )
					{
						/* Add columns that we don't yet know about */
						while( columns.length <= target )
						{
							_fnAddColumn( oSettings );
						}
	
						/* Integer, basic index */
						fn( target, def );
					}
					else if ( typeof target === 'number' && target < 0 )
					{
						/* Negative integer, right to left column counting */
						fn( columns.length+target, def );
					}
					else if ( typeof target === 'string' )
					{
						for ( k=0, kLen=columns.length ; k<kLen ; k++ ) {
							if (target === '_all') {
								// Apply to all columns
								fn( k, def );
							}
							else if (target.indexOf(':name') !== -1) {
								// Column selector
								if (columns[k].sName === target.replace(':name', '')) {
									fn( k, def );
								}
							}
							else {
								// Cell selector
								headerLayout.forEach(function (row) {
									if (row[k]) {
										var cell = $(row[k].cell);
	
										// Legacy support. Note that it means that we don't support
										// an element name selector only, since they are treated as
										// class names for 1.x compat.
										if (target.match(/^[a-z][\w-]*$/i)) {
											target = '.' + target;
										}
	
										if (cell.is( target )) {
											fn( k, def );
										}
									}
								});
							}
						}
					}
				}
			}
		}
	
		// Statically defined columns array
		if ( aoCols ) {
			for ( i=0, iLen=aoCols.length ; i<iLen ; i++ ) {
				fn( i, aoCols[i] );
			}
		}
	}
	
	
	/**
	 * Get the width for a given set of columns
	 *
	 * @param {*} settings DataTables settings object
	 * @param {*} targets Columns - comma separated string or array of numbers
	 * @param {*} original Use the original width (true) or calculated (false)
	 * @param {*} incVisible Include visible columns (true) or not (false)
	 * @returns Combined CSS value
	 */
	function _fnColumnsSumWidth( settings, targets, original, incVisible ) {
		if ( ! Array.isArray( targets ) ) {
			targets = _fnColumnsFromHeader( targets );
		}
	
		var sum = 0;
		var unit;
		var columns = settings.aoColumns;
		
		for ( var i=0, ien=targets.length ; i<ien ; i++ ) {
			var column = columns[ targets[i] ];
			var definedWidth = original ?
				column.sWidthOrig :
				column.sWidth;
	
			if ( ! incVisible && column.bVisible === false ) {
				continue;
			}
	
			if ( definedWidth === null || definedWidth === undefined ) {
				return null; // can't determine a defined width - browser defined
			}
			else if ( typeof definedWidth === 'number' ) {
				unit = 'px';
				sum += definedWidth;
			}
			else {
				var matched = definedWidth.match(/([\d\.]+)([^\d]*)/);
	
				if ( matched ) {
					sum += matched[1] * 1;
					unit = matched.length === 3 ?
						matched[2] :
						'px';
				}
			}
		}
	
		return sum + unit;
	}
	
	function _fnColumnsFromHeader( cell )
	{
		var attr = $(cell).closest('[data-dt-column]').attr('data-dt-column');
	
		if ( ! attr ) {
			return [];
		}
	
		return attr.split(',').map( function (val) {
			return val * 1;
		} );
	}
	/**
	 * Add a data array to the table, creating DOM node etc. This is the parallel to
	 * _fnGatherData, but for adding rows from a Javascript source, rather than a
	 * DOM source.
	 *  @param {object} settings dataTables settings object
	 *  @param {array} data data array to be added
	 *  @param {node} [tr] TR element to add to the table - optional. If not given,
	 *    DataTables will create a row automatically
	 *  @param {array} [tds] Array of TD|TH elements for the row - must be given
	 *    if nTr is.
	 *  @returns {int} >=0 if successful (index of new aoData entry), -1 if failed
	 *  @memberof DataTable#oApi
	 */
	function _fnAddData ( settings, dataIn, tr, tds )
	{
		/* Create the object for storing information about this new row */
		var rowIdx = settings.aoData.length;
		var rowModel = $.extend( true, {}, DataTable.models.oRow, {
			src: tr ? 'dom' : 'data',
			idx: rowIdx
		} );
	
		rowModel._aData = dataIn;
		settings.aoData.push( rowModel );
	
		var columns = settings.aoColumns;
	
		for ( var i=0, iLen=columns.length ; i<iLen ; i++ )
		{
			// Invalidate the column types as the new data needs to be revalidated
			columns[i].sType = null;
		}
	
		/* Add to the display array */
		settings.aiDisplayMaster.push( rowIdx );
	
		var id = settings.rowIdFn( dataIn );
		if ( id !== undefined ) {
			settings.aIds[ id ] = rowModel;
		}
	
		/* Create the DOM information, or register it if already present */
		if ( tr || ! settings.oFeatures.bDeferRender )
		{
			_fnCreateTr( settings, rowIdx, tr, tds );
		}
	
		return rowIdx;
	}
	
	
	/**
	 * Add one or more TR elements to the table. Generally we'd expect to
	 * use this for reading data from a DOM sourced table, but it could be
	 * used for an TR element. Note that if a TR is given, it is used (i.e.
	 * it is not cloned).
	 *  @param {object} settings dataTables settings object
	 *  @param {array|node|jQuery} trs The TR element(s) to add to the table
	 *  @returns {array} Array of indexes for the added rows
	 *  @memberof DataTable#oApi
	 */
	function _fnAddTr( settings, trs )
	{
		var row;
	
		// Allow an individual node to be passed in
		if ( ! (trs instanceof $) ) {
			trs = $(trs);
		}
	
		return trs.map( function (i, el) {
			row = _fnGetRowElements( settings, el );
			return _fnAddData( settings, row.data, el, row.cells );
		} );
	}
	
	
	/**
	 * Get the data for a given cell from the internal cache, taking into account data mapping
	 *  @param {object} settings dataTables settings object
	 *  @param {int} rowIdx aoData row id
	 *  @param {int} colIdx Column index
	 *  @param {string} type data get type ('display', 'type' 'filter|search' 'sort|order')
	 *  @returns {*} Cell data
	 *  @memberof DataTable#oApi
	 */
	function _fnGetCellData( settings, rowIdx, colIdx, type )
	{
		if (type === 'search') {
			type = 'filter';
		}
		else if (type === 'order') {
			type = 'sort';
		}
	
		var row = settings.aoData[rowIdx];
	
		if (! row) {
			return undefined;
		}
	
		var draw           = settings.iDraw;
		var col            = settings.aoColumns[colIdx];
		var rowData        = row._aData;
		var defaultContent = col.sDefaultContent;
		var cellData       = col.fnGetData( rowData, type, {
			settings: settings,
			row:      rowIdx,
			col:      colIdx
		} );
	
		// Allow for a node being returned for non-display types
		if (type !== 'display' && cellData && typeof cellData === 'object' && cellData.nodeName) {
			cellData = cellData.innerHTML;
		}
	
		if ( cellData === undefined ) {
			if ( settings.iDrawError != draw && defaultContent === null ) {
				_fnLog( settings, 0, "Requested unknown parameter "+
					(typeof col.mData=='function' ? '{function}' : "'"+col.mData+"'")+
					" for row "+rowIdx+", column "+colIdx, 4 );
				settings.iDrawError = draw;
			}
			return defaultContent;
		}
	
		// When the data source is null and a specific data type is requested (i.e.
		// not the original data), we can use default column data
		if ( (cellData === rowData || cellData === null) && defaultContent !== null && type !== undefined ) {
			cellData = defaultContent;
		}
		else if ( typeof cellData === 'function' ) {
			// If the data source is a function, then we run it and use the return,
			// executing in the scope of the data object (for instances)
			return cellData.call( rowData );
		}
	
		if ( cellData === null && type === 'display' ) {
			return '';
		}
	
		if ( type === 'filter' ) {
			var fomatters = DataTable.ext.type.search;
	
			if ( fomatters[ col.sType ] ) {
				cellData = fomatters[ col.sType ]( cellData );
			}
		}
	
		return cellData;
	}
	
	
	/**
	 * Set the value for a specific cell, into the internal data cache
	 *  @param {object} settings dataTables settings object
	 *  @param {int} rowIdx aoData row id
	 *  @param {int} colIdx Column index
	 *  @param {*} val Value to set
	 *  @memberof DataTable#oApi
	 */
	function _fnSetCellData( settings, rowIdx, colIdx, val )
	{
		var col     = settings.aoColumns[colIdx];
		var rowData = settings.aoData[rowIdx]._aData;
	
		col.fnSetData( rowData, val, {
			settings: settings,
			row:      rowIdx,
			col:      colIdx
		}  );
	}
	
	/**
	 * Write a value to a cell
	 * @param {*} td Cell
	 * @param {*} val Value
	 */
	function _fnWriteCell(td, val)
	{
		if (val && typeof val === 'object' && val.nodeName) {
			$(td)
				.empty()
				.append(val);
		}
		else {
			td.innerHTML = val;
		}
	}
	
	
	// Private variable that is used to match action syntax in the data property object
	var __reArray = /\[.*?\]$/;
	var __reFn = /\(\)$/;
	
	/**
	 * Split string on periods, taking into account escaped periods
	 * @param  {string} str String to split
	 * @return {array} Split string
	 */
	function _fnSplitObjNotation( str )
	{
		var parts = str.match(/(\\.|[^.])+/g) || [''];
	
		return parts.map( function ( s ) {
			return s.replace(/\\\./g, '.');
		} );
	}
	
	
	/**
	 * Return a function that can be used to get data from a source object, taking
	 * into account the ability to use nested objects as a source
	 *  @param {string|int|function} mSource The data source for the object
	 *  @returns {function} Data get function
	 *  @memberof DataTable#oApi
	 */
	var _fnGetObjectDataFn = DataTable.util.get;
	
	
	/**
	 * Return a function that can be used to set data from a source object, taking
	 * into account the ability to use nested objects as a source
	 *  @param {string|int|function} mSource The data source for the object
	 *  @returns {function} Data set function
	 *  @memberof DataTable#oApi
	 */
	var _fnSetObjectDataFn = DataTable.util.set;
	
	
	/**
	 * Return an array with the full table data
	 *  @param {object} oSettings dataTables settings object
	 *  @returns array {array} aData Master data array
	 *  @memberof DataTable#oApi
	 */
	function _fnGetDataMaster ( settings )
	{
		return _pluck( settings.aoData, '_aData' );
	}
	
	
	/**
	 * Nuke the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnClearTable( settings )
	{
		settings.aoData.length = 0;
		settings.aiDisplayMaster.length = 0;
		settings.aiDisplay.length = 0;
		settings.aIds = {};
	}
	
	
	/**
	 * Mark cached data as invalid such that a re-read of the data will occur when
	 * the cached data is next requested. Also update from the data source object.
	 *
	 * @param {object} settings DataTables settings object
	 * @param {int}    rowIdx   Row index to invalidate
	 * @param {string} [src]    Source to invalidate from: undefined, 'auto', 'dom'
	 *     or 'data'
	 * @param {int}    [colIdx] Column index to invalidate. If undefined the whole
	 *     row will be invalidated
	 * @memberof DataTable#oApi
	 *
	 * @todo For the modularisation of v1.11 this will need to become a callback, so
	 *   the sort and filter methods can subscribe to it. That will required
	 *   initialisation options for sorting, which is why it is not already baked in
	 */
	function _fnInvalidate( settings, rowIdx, src, colIdx )
	{
		var row = settings.aoData[ rowIdx ];
		var i, ien;
	
		// Remove the cached data for the row
		row._aSortData = null;
		row._aFilterData = null;
		row.displayData = null;
	
		// Are we reading last data from DOM or the data object?
		if ( src === 'dom' || ((! src || src === 'auto') && row.src === 'dom') ) {
			// Read the data from the DOM
			row._aData = _fnGetRowElements(
					settings, row, colIdx, colIdx === undefined ? undefined : row._aData
				)
				.data;
		}
		else {
			// Reading from data object, update the DOM
			var cells = row.anCells;
			var display = _fnGetRowDisplay(settings, rowIdx);
	
			if ( cells ) {
				if ( colIdx !== undefined ) {
					_fnWriteCell(cells[colIdx], display[colIdx]);
				}
				else {
					for ( i=0, ien=cells.length ; i<ien ; i++ ) {
						_fnWriteCell(cells[i], display[i]);
					}
				}
			}
		}
	
		// Column specific invalidation
		var cols = settings.aoColumns;
		if ( colIdx !== undefined ) {
			// Type - the data might have changed
			cols[ colIdx ].sType = null;
	
			// Max length string. Its a fairly cheep recalculation, so not worth
			// something more complicated
			cols[ colIdx ].maxLenString = null;
		}
		else {
			for ( i=0, ien=cols.length ; i<ien ; i++ ) {
				cols[i].sType = null;
				cols[i].maxLenString = null;
			}
	
			// Update DataTables special `DT_*` attributes for the row
			_fnRowAttributes( settings, row );
		}
	}
	
	
	/**
	 * Build a data source object from an HTML row, reading the contents of the
	 * cells that are in the row.
	 *
	 * @param {object} settings DataTables settings object
	 * @param {node|object} TR element from which to read data or existing row
	 *   object from which to re-read the data from the cells
	 * @param {int} [colIdx] Optional column index
	 * @param {array|object} [d] Data source object. If `colIdx` is given then this
	 *   parameter should also be given and will be used to write the data into.
	 *   Only the column in question will be written
	 * @returns {object} Object with two parameters: `data` the data read, in
	 *   document order, and `cells` and array of nodes (they can be useful to the
	 *   caller, so rather than needing a second traversal to get them, just return
	 *   them from here).
	 * @memberof DataTable#oApi
	 */
	function _fnGetRowElements( settings, row, colIdx, d )
	{
		var
			tds = [],
			td = row.firstChild,
			name, col, i=0, contents,
			columns = settings.aoColumns,
			objectRead = settings._rowReadObject;
	
		// Allow the data object to be passed in, or construct
		d = d !== undefined ?
			d :
			objectRead ?
				{} :
				[];
	
		var attr = function ( str, td  ) {
			if ( typeof str === 'string' ) {
				var idx = str.indexOf('@');
	
				if ( idx !== -1 ) {
					var attr = str.substring( idx+1 );
					var setter = _fnSetObjectDataFn( str );
					setter( d, td.getAttribute( attr ) );
				}
			}
		};
	
		// Read data from a cell and store into the data object
		var cellProcess = function ( cell ) {
			if ( colIdx === undefined || colIdx === i ) {
				col = columns[i];
				contents = (cell.innerHTML).trim();
	
				if ( col && col._bAttrSrc ) {
					var setter = _fnSetObjectDataFn( col.mData._ );
					setter( d, contents );
	
					attr( col.mData.sort, cell );
					attr( col.mData.type, cell );
					attr( col.mData.filter, cell );
				}
				else {
					// Depending on the `data` option for the columns the data can
					// be read to either an object or an array.
					if ( objectRead ) {
						if ( ! col._setter ) {
							// Cache the setter function
							col._setter = _fnSetObjectDataFn( col.mData );
						}
						col._setter( d, contents );
					}
					else {
						d[i] = contents;
					}
				}
			}
	
			i++;
		};
	
		if ( td ) {
			// `tr` element was passed in
			while ( td ) {
				name = td.nodeName.toUpperCase();
	
				if ( name == "TD" || name == "TH" ) {
					cellProcess( td );
					tds.push( td );
				}
	
				td = td.nextSibling;
			}
		}
		else {
			// Existing row object passed in
			tds = row.anCells;
	
			for ( var j=0, jen=tds.length ; j<jen ; j++ ) {
				cellProcess( tds[j] );
			}
		}
	
		// Read the ID from the DOM if present
		var rowNode = row.firstChild ? row : row.nTr;
	
		if ( rowNode ) {
			var id = rowNode.getAttribute( 'id' );
	
			if ( id ) {
				_fnSetObjectDataFn( settings.rowId )( d, id );
			}
		}
	
		return {
			data: d,
			cells: tds
		};
	}
	
	/**
	 * Render and cache a row's display data for the columns, if required
	 * @returns 
	 */
	function _fnGetRowDisplay (settings, rowIdx) {
		var rowModal = settings.aoData[rowIdx];
		var columns = settings.aoColumns;
	
		if (! rowModal.displayData) {
			// Need to render and cache
			rowModal.displayData = [];
		
			for ( var colIdx=0, len=columns.length ; colIdx<len ; colIdx++ ) {
				rowModal.displayData.push(
					_fnGetCellData( settings, rowIdx, colIdx, 'display' )
				);
			}
		}
	
		return rowModal.displayData;
	}
	
	/**
	 * Create a new TR element (and it's TD children) for a row
	 *  @param {object} oSettings dataTables settings object
	 *  @param {int} iRow Row to consider
	 *  @param {node} [nTrIn] TR element to add to the table - optional. If not given,
	 *    DataTables will create a row automatically
	 *  @param {array} [anTds] Array of TD|TH elements for the row - must be given
	 *    if nTr is.
	 *  @memberof DataTable#oApi
	 */
	function _fnCreateTr ( oSettings, iRow, nTrIn, anTds )
	{
		var
			row = oSettings.aoData[iRow],
			rowData = row._aData,
			cells = [],
			nTr, nTd, oCol,
			i, iLen, create,
			trClass = oSettings.oClasses.tbody.row;
	
		if ( row.nTr === null )
		{
			nTr = nTrIn || document.createElement('tr');
	
			row.nTr = nTr;
			row.anCells = cells;
	
			_addClass(nTr, trClass);
	
			/* Use a private property on the node to allow reserve mapping from the node
			 * to the aoData array for fast look up
			 */
			nTr._DT_RowIndex = iRow;
	
			/* Special parameters can be given by the data source to be used on the row */
			_fnRowAttributes( oSettings, row );
	
			/* Process each column */
			for ( i=0, iLen=oSettings.aoColumns.length ; i<iLen ; i++ )
			{
				oCol = oSettings.aoColumns[i];
				create = nTrIn && anTds[i] ? false : true;
	
				nTd = create ? document.createElement( oCol.sCellType ) : anTds[i];
	
				if (! nTd) {
					_fnLog( oSettings, 0, 'Incorrect column count', 18 );
				}
	
				nTd._DT_CellIndex = {
					row: iRow,
					column: i
				};
				
				cells.push( nTd );
				
				var display = _fnGetRowDisplay(oSettings, iRow);
	
				// Need to create the HTML if new, or if a rendering function is defined
				if (
					create ||
					(
						(oCol.mRender || oCol.mData !== i) &&
						(!$.isPlainObject(oCol.mData) || oCol.mData._ !== i+'.display')
					)
				) {
					_fnWriteCell(nTd, display[i]);
				}
	
				// column class
				_addClass(nTd, oCol.sClass);
	
				// Visibility - add or remove as required
				if ( oCol.bVisible && create )
				{
					nTr.appendChild( nTd );
				}
				else if ( ! oCol.bVisible && ! create )
				{
					nTd.parentNode.removeChild( nTd );
				}
	
				if ( oCol.fnCreatedCell )
				{
					oCol.fnCreatedCell.call( oSettings.oInstance,
						nTd, _fnGetCellData( oSettings, iRow, i ), rowData, iRow, i
					);
				}
			}
	
			_fnCallbackFire( oSettings, 'aoRowCreatedCallback', 'row-created', [nTr, rowData, iRow, cells] );
		}
		else {
			_addClass(row.nTr, trClass);
		}
	}
	
	
	/**
	 * Add attributes to a row based on the special `DT_*` parameters in a data
	 * source object.
	 *  @param {object} settings DataTables settings object
	 *  @param {object} DataTables row object for the row to be modified
	 *  @memberof DataTable#oApi
	 */
	function _fnRowAttributes( settings, row )
	{
		var tr = row.nTr;
		var data = row._aData;
	
		if ( tr ) {
			var id = settings.rowIdFn( data );
	
			if ( id ) {
				tr.id = id;
			}
	
			if ( data.DT_RowClass ) {
				// Remove any classes added by DT_RowClass before
				var a = data.DT_RowClass.split(' ');
				row.__rowc = row.__rowc ?
					_unique( row.__rowc.concat( a ) ) :
					a;
	
				$(tr)
					.removeClass( row.__rowc.join(' ') )
					.addClass( data.DT_RowClass );
			}
	
			if ( data.DT_RowAttr ) {
				$(tr).attr( data.DT_RowAttr );
			}
	
			if ( data.DT_RowData ) {
				$(tr).data( data.DT_RowData );
			}
		}
	}
	
	
	/**
	 * Create the HTML header for the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnBuildHead( settings, side )
	{
		var classes = settings.oClasses;
		var columns = settings.aoColumns;
		var i, ien, row;
		var target = side === 'header'
			? settings.nTHead
			: settings.nTFoot;
		var titleProp = side === 'header' ? 'sTitle' : side;
	
		// Footer might be defined
		if (! target) {
			return;
		}
	
		// If no cells yet and we have content for them, then create
		if (side === 'header' || _pluck(settings.aoColumns, titleProp).join('')) {
			row = $('tr', target);
	
			// Add a row if needed
			if (! row.length) {
				row = $('<tr/>').appendTo(target)
			}
	
			// Add the number of cells needed to make up to the number of columns
			if (row.length === 1) {
				var cellCount = 0;
				
				$('td, th', row).each(function () {
					cellCount += this.colSpan;
				});
	
				for ( i=cellCount, ien=columns.length ; i<ien ; i++ ) {
					$('<th/>')
						.html( columns[i][titleProp] || '' )
						.appendTo( row );
				}
			}
		}
	
		var detected = _fnDetectHeader( settings, target, true );
	
		if (side === 'header') {
			settings.aoHeader = detected;
			$('tr', target).addClass(classes.thead.row);
		}
		else {
			settings.aoFooter = detected;
			$('tr', target).addClass(classes.tfoot.row);
		}
	
		// Every cell needs to be passed through the renderer
		$(target).children('tr').children('th, td')
			.each( function () {
				_fnRenderer( settings, side )(
					settings, $(this), classes
				);
			} );
	}
	
	/**
	 * Build a layout structure for a header or footer
	 *
	 * @param {*} settings DataTables settings
	 * @param {*} source Source layout array
	 * @param {*} incColumns What columns should be included
	 * @returns Layout array in column index order
	 */
	function _fnHeaderLayout( settings, source, incColumns )
	{
		var row, column, cell;
		var local = [];
		var structure = [];
		var columns = settings.aoColumns;
		var columnCount = columns.length;
		var rowspan, colspan;
	
		if ( ! source ) {
			return;
		}
	
		// Default is to work on only visible columns
		if ( ! incColumns ) {
			incColumns = _range(columnCount)
				.filter(function (idx) {
					return columns[idx].bVisible;
				});
		}
	
		// Make a copy of the master layout array, but with only the columns we want
		for ( row=0 ; row<source.length ; row++ ) {
			// Remove any columns we haven't selected
			local[row] = source[row].slice().filter(function (cell, i) {
				return incColumns.includes(i);
			});
	
			// Prep the structure array - it needs an element for each row
			structure.push( [] );
		}
	
		for ( row=0 ; row<local.length ; row++ ) {
			for ( column=0 ; column<local[row].length ; column++ ) {
				rowspan = 1;
				colspan = 1;
	
				// Check to see if there is already a cell (row/colspan) covering our target
				// insert point. If there is, then there is nothing to do.
				if ( structure[row][column] === undefined ) {
					cell = local[row][column].cell;
	
					// Expand for rowspan
					while (
						local[row+rowspan] !== undefined &&
						local[row][column].cell == local[row+rowspan][column].cell
					) {
						structure[row+rowspan][column] = null;
						rowspan++;
					}
	
					// And for colspan
					while (
						local[row][column+colspan] !== undefined &&
						local[row][column].cell == local[row][column+colspan].cell
					) {
						// Which also needs to go over rows
						for ( var k=0 ; k<rowspan ; k++ ) {
							structure[row+k][column+colspan] = null;
						}
	
						colspan++;
					}
	
					var titleSpan = $('span.dt-column-title', cell);
	
					structure[row][column] = {
						cell: cell,
						colspan: colspan,
						rowspan: rowspan,
						title: titleSpan.length
							? titleSpan.html()
							: $(cell).html()
					};
				}
			}
		}
	
		return structure;
	}
	
	
	/**
	 * Draw the header (or footer) element based on the column visibility states.
	 *
	 *  @param object oSettings dataTables settings object
	 *  @param array aoSource Layout array from _fnDetectHeader
	 *  @memberof DataTable#oApi
	 */
	function _fnDrawHead( settings, source )
	{
		var layout = _fnHeaderLayout(settings, source);
		var tr, n;
	
		for ( var row=0 ; row<source.length ; row++ ) {
			tr = source[row].row;
	
			// All cells are going to be replaced, so empty out the row
			// Can't use $().empty() as that kills event handlers
			if (tr) {
				while( (n = tr.firstChild) ) {
					tr.removeChild( n );
				}
			}
	
			for ( var column=0 ; column<layout[row].length ; column++ ) {
				var point = layout[row][column];
	
				if (point) {
					$(point.cell)
						.appendTo(tr)
						.attr('rowspan', point.rowspan)
						.attr('colspan', point.colspan);
				}
			}
		}
	}
	
	
	/**
	 * Insert the required TR nodes into the table for display
	 *  @param {object} oSettings dataTables settings object
	 *  @param ajaxComplete true after ajax call to complete rendering
	 *  @memberof DataTable#oApi
	 */
	function _fnDraw( oSettings, ajaxComplete )
	{
		// Allow for state saving and a custom start position
		_fnStart( oSettings );
	
		/* Provide a pre-callback function which can be used to cancel the draw is false is returned */
		var aPreDraw = _fnCallbackFire( oSettings, 'aoPreDrawCallback', 'preDraw', [oSettings] );
		if ( aPreDraw.indexOf(false) !== -1 )
		{
			_fnProcessingDisplay( oSettings, false );
			return;
		}
	
		var anRows = [];
		var iRowCount = 0;
		var bServerSide = _fnDataSource( oSettings ) == 'ssp';
		var aiDisplay = oSettings.aiDisplay;
		var iDisplayStart = oSettings._iDisplayStart;
		var iDisplayEnd = oSettings.fnDisplayEnd();
		var columns = oSettings.aoColumns;
		var body = $(oSettings.nTBody);
	
		oSettings.bDrawing = true;
	
		/* Server-side processing draw intercept */
		if ( oSettings.deferLoading )
		{
			oSettings.deferLoading = false;
			oSettings.iDraw++;
			_fnProcessingDisplay( oSettings, false );
		}
		else if ( !bServerSide )
		{
			oSettings.iDraw++;
		}
		else if ( !oSettings.bDestroying && !ajaxComplete)
		{
			// Show loading message for server-side processing
			if (oSettings.iDraw === 0) {
				body.empty().append(_emptyRow(oSettings));
			}
	
			_fnAjaxUpdate( oSettings );
			return;
		}
	
		if ( aiDisplay.length !== 0 )
		{
			var iStart = bServerSide ? 0 : iDisplayStart;
			var iEnd = bServerSide ? oSettings.aoData.length : iDisplayEnd;
	
			for ( var j=iStart ; j<iEnd ; j++ )
			{
				var iDataIndex = aiDisplay[j];
				var aoData = oSettings.aoData[ iDataIndex ];
				if ( aoData.nTr === null )
				{
					_fnCreateTr( oSettings, iDataIndex );
				}
	
				var nRow = aoData.nTr;
	
				// Add various classes as needed
				for (var i=0 ; i<columns.length ; i++) {
					var col = columns[i];
					var td = aoData.anCells[i];
	
					_addClass(td, _ext.type.className[col.sType]); // auto class
					_addClass(td, oSettings.oClasses.tbody.cell); // all cells
				}
	
				// Row callback functions - might want to manipulate the row
				// iRowCount and j are not currently documented. Are they at all
				// useful?
				_fnCallbackFire( oSettings, 'aoRowCallback', null,
					[nRow, aoData._aData, iRowCount, j, iDataIndex] );
	
				anRows.push( nRow );
				iRowCount++;
			}
		}
		else
		{
			anRows[ 0 ] = _emptyRow(oSettings);
		}
	
		/* Header and footer callbacks */
		_fnCallbackFire( oSettings, 'aoHeaderCallback', 'header', [ $(oSettings.nTHead).children('tr')[0],
			_fnGetDataMaster( oSettings ), iDisplayStart, iDisplayEnd, aiDisplay ] );
	
		_fnCallbackFire( oSettings, 'aoFooterCallback', 'footer', [ $(oSettings.nTFoot).children('tr')[0],
			_fnGetDataMaster( oSettings ), iDisplayStart, iDisplayEnd, aiDisplay ] );
	
		// replaceChildren is faster, but only became widespread in 2020,
		// so a fall back in jQuery is provided for older browsers.
		if (body[0].replaceChildren) {
			body[0].replaceChildren.apply(body[0], anRows);
		}
		else {
			body.children().detach();
			body.append( $(anRows) );
		}
	
		// Empty table needs a specific class
		$(oSettings.nTableWrapper).toggleClass('dt-empty-footer', $('tr', oSettings.nTFoot).length === 0);
	
		/* Call all required callback functions for the end of a draw */
		_fnCallbackFire( oSettings, 'aoDrawCallback', 'draw', [oSettings], true );
	
		/* Draw is complete, sorting and filtering must be as well */
		oSettings.bSorted = false;
		oSettings.bFiltered = false;
		oSettings.bDrawing = false;
	}
	
	
	/**
	 * Redraw the table - taking account of the various features which are enabled
	 *  @param {object} oSettings dataTables settings object
	 *  @param {boolean} [holdPosition] Keep the current paging position. By default
	 *    the paging is reset to the first page
	 *  @memberof DataTable#oApi
	 */
	function _fnReDraw( settings, holdPosition, recompute )
	{
		var
			features = settings.oFeatures,
			sort     = features.bSort,
			filter   = features.bFilter;
	
		if (recompute === undefined || recompute === true) {
			// Resolve any column types that are unknown due to addition or invalidation
			_fnColumnTypes( settings );
	
			if ( sort ) {
				_fnSort( settings );
			}
	
			if ( filter ) {
				_fnFilterComplete( settings, settings.oPreviousSearch );
			}
			else {
				// No filtering, so we want to just use the display master
				settings.aiDisplay = settings.aiDisplayMaster.slice();
			}
		}
	
		if ( holdPosition !== true ) {
			settings._iDisplayStart = 0;
		}
	
		// Let any modules know about the draw hold position state (used by
		// scrolling internally)
		settings._drawHold = holdPosition;
	
		_fnDraw( settings );
	
		settings.api.one('draw', function () {
			settings._drawHold = false;
		});
	}
	
	
	/*
	 * Table is empty - create a row with an empty message in it
	 */
	function _emptyRow ( settings ) {
		var oLang = settings.oLanguage;
		var zero = oLang.sZeroRecords;
		var dataSrc = _fnDataSource( settings );
	
		// Make use of the fact that settings.json is only set once the initial data has
		// been loaded. Show loading when that isn't the case
		if ((dataSrc === 'ssp' || dataSrc === 'ajax') && ! settings.json) {
			zero = oLang.sLoadingRecords;
		}
		else if ( oLang.sEmptyTable && settings.fnRecordsTotal() === 0 )
		{
			zero = oLang.sEmptyTable;
		}
	
		return $( '<tr/>' )
			.append( $('<td />', {
				'colSpan': _fnVisbleColumns( settings ),
				'class':   settings.oClasses.empty.row
			} ).html( zero ) )[0];
	}
	
	
	/**
	 * Expand the layout items into an object for the rendering function
	 */
	function _layoutItems (row, align, items) {
		if ( Array.isArray(items)) {
			for (var i=0 ; i<items.length ; i++) {
				_layoutItems(row, align, items[i]);
			}
	
			return;
		}
	
		var rowCell = row[align];
	
		// If it is an object, then there can be multiple features contained in it
		if ( $.isPlainObject( items ) ) {
			// A feature plugin cannot be named "features" due to this check
			if (items.features) {
				if (items.rowId) {
					row.id = items.rowId;
				}
				if (items.rowClass) {
					row.className = items.rowClass;
				}
	
				rowCell.id = items.id;
				rowCell.className = items.className;
	
				_layoutItems(row, align, items.features);
			}
			else {
				Object.keys(items).map(function (key) {
					rowCell.contents.push( {
						feature: key,
						opts: items[key]
					});
				});
			}
		}
		else {
			rowCell.contents.push(items);
		}
	}
	
	/**
	 * Find, or create a layout row
	 */
	function _layoutGetRow(rows, rowNum, align) {
		var row;
	
		// Find existing rows
		for (var i=0; i<rows.length; i++) {
			row = rows[i];
	
			if (row.rowNum === rowNum) {
				// full is on its own, but start and end share a row
				if (
					(align === 'full' && row.full) ||
					((align === 'start' || align === 'end') && (row.start || row.end))
				) {
					if (! row[align]) {
						row[align] = {
							contents: []
						};
					}
	
					return row;
				}
			}
		}
	
		// If we get this far, then there was no match, create a new row
		row = {
			rowNum: rowNum	
		};
	
		row[align] = {
			contents: []
		};
	
		rows.push(row);
	
		return row;
	}
	
	/**
	 * Convert a `layout` object given by a user to the object structure needed
	 * for the renderer. This is done twice, once for above and once for below
	 * the table. Ordering must also be considered.
	 *
	 * @param {*} settings DataTables settings object
	 * @param {*} layout Layout object to convert
	 * @param {string} side `top` or `bottom`
	 * @returns Converted array structure - one item for each row.
	 */
	function _layoutArray ( settings, layout, side ) {
		var rows = [];
		
		// Split out into an array
		$.each( layout, function ( pos, items ) {
			if (items === null) {
				return;
			}
	
			var parts = pos.match(/^([a-z]+)([0-9]*)([A-Za-z]*)$/);
			var rowNum = parts[2]
				? parts[2] * 1
				: 0;
			var align = parts[3]
				? parts[3].toLowerCase()
				: 'full';
	
			// Filter out the side we aren't interested in
			if (parts[1] !== side) {
				return;
			}
	
			// Get or create the row we should attach to
			var row = _layoutGetRow(rows, rowNum, align);
	
			_layoutItems(row, align, items);
		});
	
		// Order by item identifier
		rows.sort( function ( a, b ) {
			var order1 = a.rowNum;
			var order2 = b.rowNum;
	
			// If both in the same row, then the row with `full` comes first
			if (order1 === order2) {
				var ret = a.full && ! b.full ? -1 : 1;
	
				return side === 'bottom'
					? ret * -1
					: ret;
			}
	
			return order2 - order1;
		} );
	
		// Invert for below the table
		if ( side === 'bottom' ) {
			rows.reverse();
		}
	
		for (var row = 0; row<rows.length; row++) {
			delete rows[row].rowNum;
	
			_layoutResolve(settings, rows[row]);
		}
	
		return rows;
	}
	
	
	/**
	 * Convert the contents of a row's layout object to nodes that can be inserted
	 * into the document by a renderer. Execute functions, look up plug-ins, etc.
	 *
	 * @param {*} settings DataTables settings object
	 * @param {*} row Layout object for this row
	 */
	function _layoutResolve( settings, row ) {
		var getFeature = function (feature, opts) {
			if ( ! _ext.features[ feature ] ) {
				_fnLog( settings, 0, 'Unknown feature: '+ feature );
			}
	
			return _ext.features[ feature ].apply( this, [settings, opts] );
		};
	
		var resolve = function ( item ) {
			if (! row[ item ]) {
				return;
			}
	
			var line = row[ item ].contents;
	
			for ( var i=0, ien=line.length ; i<ien ; i++ ) {
				if ( ! line[i] ) {
					continue;
				}
				else if ( typeof line[i] === 'string' ) {
					line[i] = getFeature( line[i], null );
				}
				else if ( $.isPlainObject(line[i]) ) {
					// If it's an object, it just has feature and opts properties from
					// the transform in _layoutArray
					line[i] = getFeature(line[i].feature, line[i].opts);
				}
				else if ( typeof line[i].node === 'function' ) {
					line[i] = line[i].node( settings );
				}
				else if ( typeof line[i] === 'function' ) {
					var inst = line[i]( settings );
	
					line[i] = typeof inst.node === 'function' ?
						inst.node() :
						inst;
				}
			}
		};
	
		resolve('start');
		resolve('end');
		resolve('full');
	}
	
	
	/**
	 * Add the options to the page HTML for the table
	 *  @param {object} settings DataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnAddOptionsHtml ( settings )
	{
		var classes = settings.oClasses;
		var table = $(settings.nTable);
	
		// Wrapper div around everything DataTables controls
		var insert = $('<div/>')
			.attr({
				id:      settings.sTableId+'_wrapper',
				'class': classes.container
			})
			.insertBefore(table);
	
		settings.nTableWrapper = insert[0];
	
		if (settings.sDom) {
			// Legacy
			_fnLayoutDom(settings, settings.sDom, insert);
		}
		else {
			var top = _layoutArray( settings, settings.layout, 'top' );
			var bottom = _layoutArray( settings, settings.layout, 'bottom' );
			var renderer = _fnRenderer( settings, 'layout' );
		
			// Everything above - the renderer will actually insert the contents into the document
			top.forEach(function (item) {
				renderer( settings, insert, item );
			});
	
			// The table - always the center of attention
			renderer( settings, insert, {
				full: {
					table: true,
					contents: [ _fnFeatureHtmlTable(settings) ]
				}
			} );
	
			// Everything below
			bottom.forEach(function (item) {
				renderer( settings, insert, item );
			});
		}
	
		// Processing floats on top, so it isn't an inserted feature
		_processingHtml( settings );
	}
	
	/**
	 * Draw the table with the legacy DOM property
	 * @param {*} settings DT settings object
	 * @param {*} dom DOM string
	 * @param {*} insert Insert point
	 */
	function _fnLayoutDom( settings, dom, insert )
	{
		var parts = dom.match(/(".*?")|('.*?')|./g);
		var featureNode, option, newNode, next, attr;
	
		for ( var i=0 ; i<parts.length ; i++ ) {
			featureNode = null;
			option = parts[i];
	
			if ( option == '<' ) {
				// New container div
				newNode = $('<div/>');
	
				// Check to see if we should append an id and/or a class name to the container
				next = parts[i+1];
	
				if ( next[0] == "'" || next[0] == '"' ) {
					attr = next.replace(/['"]/g, '');
	
					var id = '', className;
	
					/* The attribute can be in the format of "#id.class", "#id" or "class" This logic
					 * breaks the string into parts and applies them as needed
					 */
					if ( attr.indexOf('.') != -1 ) {
						var split = attr.split('.');
	
						id = split[0];
						className = split[1];
					}
					else if ( attr[0] == "#" ) {
						id = attr;
					}
					else {
						className = attr;
					}
	
					newNode
						.attr('id', id.substring(1))
						.addClass(className);
	
					i++; // Move along the position array
				}
	
				insert.append( newNode );
				insert = newNode;
			}
			else if ( option == '>' ) {
				// End container div
				insert = insert.parent();
			}
			else if ( option == 't' ) {
				// Table
				featureNode = _fnFeatureHtmlTable( settings );
			}
			else
			{
				DataTable.ext.feature.forEach(function(feature) {
					if ( option == feature.cFeature ) {
						featureNode = feature.fnInit( settings );
					}
				});
			}
	
			// Add to the display
			if ( featureNode ) {
				insert.append( featureNode );
			}
		}
	}
	
	
	/**
	 * Use the DOM source to create up an array of header cells. The idea here is to
	 * create a layout grid (array) of rows x columns, which contains a reference
	 * to the cell that that point in the grid (regardless of col/rowspan), such that
	 * any column / row could be removed and the new grid constructed
	 *  @param {node} thead The header/footer element for the table
	 *  @returns {array} Calculated layout array
	 *  @memberof DataTable#oApi
	 */
	function _fnDetectHeader ( settings, thead, write )
	{
		var columns = settings.aoColumns;
		var rows = $(thead).children('tr');
		var row, cell;
		var i, k, l, iLen, shifted, column, colspan, rowspan;
		var titleRow = settings.titleRow;
		var isHeader = thead && thead.nodeName.toLowerCase() === 'thead';
		var layout = [];
		var unique;
		var shift = function ( a, i, j ) {
			var k = a[i];
			while ( k[j] ) {
				j++;
			}
			return j;
		};
	
		// We know how many rows there are in the layout - so prep it
		for ( i=0, iLen=rows.length ; i<iLen ; i++ ) {
			layout.push( [] );
		}
	
		for ( i=0, iLen=rows.length ; i<iLen ; i++ ) {
			row = rows[i];
			column = 0;
	
			// For every cell in the row..
			cell = row.firstChild;
			while ( cell ) {
				if (
					cell.nodeName.toUpperCase() == 'TD' ||
					cell.nodeName.toUpperCase() == 'TH'
				) {
					var cols = [];
					var jqCell = $(cell);
	
					// Get the col and rowspan attributes from the DOM and sanitise them
					colspan = cell.getAttribute('colspan') * 1;
					rowspan = cell.getAttribute('rowspan') * 1;
					colspan = (!colspan || colspan===0 || colspan===1) ? 1 : colspan;
					rowspan = (!rowspan || rowspan===0 || rowspan===1) ? 1 : rowspan;
	
					// There might be colspan cells already in this row, so shift our target
					// accordingly
					shifted = shift( layout, i, column );
	
					// Cache calculation for unique columns
					unique = colspan === 1 ?
						true :
						false;
					
					// Perform header setup
					if ( write ) {
						if (unique) {
							// Allow column options to be set from HTML attributes
							_fnColumnOptions( settings, shifted, _fnEscapeObject(jqCell.data()) );
							
							// Get the width for the column. This can be defined from the
							// width attribute, style attribute or `columns.width` option
							var columnDef = columns[shifted];
							var width = cell.getAttribute('width') || null;
							var t = cell.style.width.match(/width:\s*(\d+[pxem%]+)/);
							if ( t ) {
								width = t[1];
							}
	
							columnDef.sWidthOrig = columnDef.sWidth || width;
	
							if (isHeader) {
								// Column title handling - can be user set, or read from the DOM
								// This happens before the render, so the original is still in place
								if ( columnDef.sTitle !== null && ! columnDef.autoTitle ) {
									if (
										(titleRow === true && i === 0) || // top row
										(titleRow === false && i === rows.length -1) || // bottom row
										(titleRow === i) || // specific row
										(titleRow === null)
									) {
										cell.innerHTML = columnDef.sTitle;
									}
								}
	
								if (! columnDef.sTitle && unique) {
									columnDef.sTitle = _stripHtml(cell.innerHTML);
									columnDef.autoTitle = true;
								}
							}
							else {
								// Footer specific operations
								if (columnDef.footer) {
									cell.innerHTML = columnDef.footer;
								}
							}
	
							// Fall back to the aria-label attribute on the table header if no ariaTitle is
							// provided.
							if (! columnDef.ariaTitle) {
								columnDef.ariaTitle = jqCell.attr("aria-label") || columnDef.sTitle;
							}
	
							// Column specific class names
							if ( columnDef.className ) {
								jqCell.addClass( columnDef.className );
							}
						}
	
						// Wrap the column title so we can write to it in future
						if ( $('span.dt-column-title', cell).length === 0) {
							$('<span>')
								.addClass('dt-column-title')
								.append(cell.childNodes)
								.appendTo(cell);
						}
	
						if (
							settings.orderIndicators &&
							isHeader &&
							jqCell.filter(':not([data-dt-order=disable])').length !== 0 &&
							jqCell.parent(':not([data-dt-order=disable])').length !== 0 &&
							$('span.dt-column-order', cell).length === 0
						) {
							$('<span>')
								.addClass('dt-column-order')
								.appendTo(cell);
						}
	
						// We need to wrap the elements in the header in another element to use flexbox
						// layout for those elements
						var headerFooter = isHeader ? 'header' : 'footer';
	
						if ( $('span.dt-column-' + headerFooter, cell).length === 0) {
							$('<div>')
								.addClass('dt-column-' + headerFooter)
								.append(cell.childNodes)
								.appendTo(cell);
						}
					}
	
					// If there is col / rowspan, copy the information into the layout grid
					for ( l=0 ; l<colspan ; l++ ) {
						for ( k=0 ; k<rowspan ; k++ ) {
							layout[i+k][shifted+l] = {
								cell: cell,
								unique: unique
							};
	
							layout[i+k].row = row;
						}
	
						cols.push( shifted+l );
					}
	
					// Assign an attribute so spanning cells can still be identified
					// as belonging to a column
					cell.setAttribute('data-dt-column', _unique(cols).join(','));
				}
	
				cell = cell.nextSibling;
			}
		}
	
		return layout;
	}
	
	/**
	 * Set the start position for draw
	 *  @param {object} oSettings dataTables settings object
	 */
	function _fnStart( oSettings )
	{
		var bServerSide = _fnDataSource( oSettings ) == 'ssp';
		var iInitDisplayStart = oSettings.iInitDisplayStart;
	
		// Check and see if we have an initial draw position from state saving
		if ( iInitDisplayStart !== undefined && iInitDisplayStart !== -1 )
		{
			oSettings._iDisplayStart = bServerSide ?
				iInitDisplayStart :
				iInitDisplayStart >= oSettings.fnRecordsDisplay() ?
					0 :
					iInitDisplayStart;
	
			oSettings.iInitDisplayStart = -1;
		}
	}
	
	/**
	 * Create an Ajax call based on the table's settings, taking into account that
	 * parameters can have multiple forms, and backwards compatibility.
	 *
	 * @param {object} oSettings dataTables settings object
	 * @param {array} data Data to send to the server, required by
	 *     DataTables - may be augmented by developer callbacks
	 * @param {function} fn Callback function to run when data is obtained
	 */
	function _fnBuildAjax( oSettings, data, fn )
	{
		var ajaxData;
		var ajax = oSettings.ajax;
		var instance = oSettings.oInstance;
		var callback = function ( json ) {
			var status = oSettings.jqXHR
				? oSettings.jqXHR.status
				: null;
	
			if ( json === null || (typeof status === 'number' && status == 204 ) ) {
				json = {};
				_fnAjaxDataSrc( oSettings, json, [] );
			}
	
			var error = json.error || json.sError;
			if ( error ) {
				_fnLog( oSettings, 0, error );
			}
	
			// Microsoft often wrap JSON as a string in another JSON object
			// Let's handle that automatically
			if (json.d && typeof json.d === 'string') {
				try {
					json = JSON.parse(json.d);
				}
				catch (e) {
					// noop
				}
			}
	
			oSettings.json = json;
	
			_fnCallbackFire( oSettings, null, 'xhr', [oSettings, json, oSettings.jqXHR], true );
			fn( json );
		};
	
		if ( $.isPlainObject( ajax ) && ajax.data )
		{
			ajaxData = ajax.data;
	
			var newData = typeof ajaxData === 'function' ?
				ajaxData( data, oSettings ) :  // fn can manipulate data or return
				ajaxData;                      // an object object or array to merge
	
			// If the function returned something, use that alone
			data = typeof ajaxData === 'function' && newData ?
				newData :
				$.extend( true, data, newData );
	
			// Remove the data property as we've resolved it already and don't want
			// jQuery to do it again (it is restored at the end of the function)
			delete ajax.data;
		}
	
		var baseAjax = {
			"url": typeof ajax === 'string' ?
				ajax :
				'',
			"data": data,
			"success": callback,
			"dataType": "json",
			"cache": false,
			"type": oSettings.sServerMethod,
			"error": function (xhr, error) {
				var ret = _fnCallbackFire( oSettings, null, 'xhr', [oSettings, null, oSettings.jqXHR], true );
	
				if ( ret.indexOf(true) === -1 ) {
					if ( error == "parsererror" ) {
						_fnLog( oSettings, 0, 'Invalid JSON response', 1 );
					}
					else if ( xhr.readyState === 4 ) {
						_fnLog( oSettings, 0, 'Ajax error', 7 );
					}
				}
	
				_fnProcessingDisplay( oSettings, false );
			}
		};
	
		// If `ajax` option is an object, extend and override our default base
		if ( $.isPlainObject( ajax ) ) {
			$.extend( baseAjax, ajax )
		}
	
		// Store the data submitted for the API
		oSettings.oAjaxData = data;
	
		// Allow plug-ins and external processes to modify the data
		_fnCallbackFire( oSettings, null, 'preXhr', [oSettings, data, baseAjax], true );
	
		// Custom Ajax option to submit the parameters as a JSON string
		if (baseAjax.submitAs === 'json' && typeof data === 'object') {
			baseAjax.data = JSON.stringify(data);
		}
	
		if ( typeof ajax === 'function' )
		{
			// Is a function - let the caller define what needs to be done
			oSettings.jqXHR = ajax.call( instance, data, callback, oSettings );
		}
		else if (ajax.url === '') {
			// No url, so don't load any data. Just apply an empty data array
			// to the object for the callback.
			var empty = {};
	
			_fnAjaxDataSrc(oSettings, empty, []);
			callback(empty);
		}
		else {
			// Object to extend the base settings
			oSettings.jqXHR = $.ajax( baseAjax );
		}
	
		// Restore for next time around
		if ( ajaxData ) {
			ajax.data = ajaxData;
		}
	}
	
	
	/**
	 * Update the table using an Ajax call
	 *  @param {object} settings dataTables settings object
	 *  @returns {boolean} Block the table drawing or not
	 *  @memberof DataTable#oApi
	 */
	function _fnAjaxUpdate( settings )
	{
		settings.iDraw++;
		_fnProcessingDisplay( settings, true );
	
		_fnBuildAjax(
			settings,
			_fnAjaxParameters( settings ),
			function(json) {
				_fnAjaxUpdateDraw( settings, json );
			}
		);
	}
	
	
	/**
	 * Build up the parameters in an object needed for a server-side processing
	 * request.
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {bool} block the table drawing or not
	 *  @memberof DataTable#oApi
	 */
	function _fnAjaxParameters( settings )
	{
		var
			columns = settings.aoColumns,
			features = settings.oFeatures,
			preSearch = settings.oPreviousSearch,
			preColSearch = settings.aoPreSearchCols,
			colData = function ( idx, prop ) {
				return typeof columns[idx][prop] === 'function' ?
					'function' :
					columns[idx][prop];
			};
	
		return {
			draw: settings.iDraw,
			columns: columns.map( function ( column, i ) {
				return {
					data: colData(i, 'mData'),
					name: column.sName,
					searchable: column.bSearchable,
					orderable: column.bSortable,
					search: {
						value: preColSearch[i].search,
						regex: preColSearch[i].regex,
						fixed: Object.keys(column.searchFixed).map( function(name) {
							return {
								name: name,
								term: column.searchFixed[name].toString()
							}
						})
					}
				};
			} ),
			order: _fnSortFlatten( settings ).map( function ( val ) {
				return {
					column: val.col,
					dir: val.dir,
					name: colData(val.col, 'sName')
				};
			} ),
			start: settings._iDisplayStart,
			length: features.bPaginate ?
				settings._iDisplayLength :
				-1,
			search: {
				value: preSearch.search,
				regex: preSearch.regex,
				fixed: Object.keys(settings.searchFixed).map( function(name) {
					return {
						name: name,
						term: settings.searchFixed[name].toString()
					}
				})
			}
		};
	}
	
	
	/**
	 * Data the data from the server (nuking the old) and redraw the table
	 *  @param {object} oSettings dataTables settings object
	 *  @param {object} json json data return from the server.
	 *  @param {string} json.sEcho Tracking flag for DataTables to match requests
	 *  @param {int} json.iTotalRecords Number of records in the data set, not accounting for filtering
	 *  @param {int} json.iTotalDisplayRecords Number of records in the data set, accounting for filtering
	 *  @param {array} json.aaData The data to display on this page
	 *  @param {string} [json.sColumns] Column ordering (sName, comma separated)
	 *  @memberof DataTable#oApi
	 */
	function _fnAjaxUpdateDraw ( settings, json )
	{
		var data = _fnAjaxDataSrc(settings, json);
		var draw = _fnAjaxDataSrcParam(settings, 'draw', json);
		var recordsTotal = _fnAjaxDataSrcParam(settings, 'recordsTotal', json);
		var recordsFiltered = _fnAjaxDataSrcParam(settings, 'recordsFiltered', json);
	
		if ( draw !== undefined ) {
			// Protect against out of sequence returns
			if ( draw*1 < settings.iDraw ) {
				return;
			}
			settings.iDraw = draw * 1;
		}
	
		// No data in returned object, so rather than an array, we show an empty table
		if ( ! data ) {
			data = [];
		}
	
		_fnClearTable( settings );
		settings._iRecordsTotal   = parseInt(recordsTotal, 10);
		settings._iRecordsDisplay = parseInt(recordsFiltered, 10);
	
		for ( var i=0, ien=data.length ; i<ien ; i++ ) {
			_fnAddData( settings, data[i] );
		}
		settings.aiDisplay = settings.aiDisplayMaster.slice();
	
		_fnColumnTypes(settings);
		_fnDraw( settings, true );
		_fnInitComplete( settings );
		_fnProcessingDisplay( settings, false );
	}
	
	
	/**
	 * Get the data from the JSON data source to use for drawing a table. Using
	 * `_fnGetObjectDataFn` allows the data to be sourced from a property of the
	 * source object, or from a processing function.
	 *  @param {object} settings dataTables settings object
	 *  @param  {object} json Data source object / array from the server
	 *  @return {array} Array of data to use
	 */
	function _fnAjaxDataSrc ( settings, json, write )
	{
		var dataProp = 'data';
	
		if ($.isPlainObject( settings.ajax ) && settings.ajax.dataSrc !== undefined) {
			// Could in inside a `dataSrc` object, or not!
			var dataSrc = settings.ajax.dataSrc;
	
			// string, function and object are valid types
			if (typeof dataSrc === 'string' || typeof dataSrc === 'function') {
				dataProp = dataSrc;
			}
			else if (dataSrc.data !== undefined) {
				dataProp = dataSrc.data;
			}
		}
	
		if ( ! write ) {
			if ( dataProp === 'data' ) {
				// If the default, then we still want to support the old style, and safely ignore
				// it if possible
				return json.aaData || json[dataProp];
			}
	
			return dataProp !== "" ?
				_fnGetObjectDataFn( dataProp )( json ) :
				json;
		}
		
		// set
		_fnSetObjectDataFn( dataProp )( json, write );
	}
	
	/**
	 * Very similar to _fnAjaxDataSrc, but for the other SSP properties
	 * @param {*} settings DataTables settings object
	 * @param {*} param Target parameter
	 * @param {*} json JSON data
	 * @returns Resolved value
	 */
	function _fnAjaxDataSrcParam (settings, param, json) {
		var dataSrc = $.isPlainObject( settings.ajax )
			? settings.ajax.dataSrc
			: null;
	
		if (dataSrc && dataSrc[param]) {
			// Get from custom location
			return _fnGetObjectDataFn( dataSrc[param] )( json );
		}
	
		// else - Default behaviour
		var old = '';
	
		// Legacy support
		if (param === 'draw') {
			old = 'sEcho';
		}
		else if (param === 'recordsTotal') {
			old = 'iTotalRecords';
		}
		else if (param === 'recordsFiltered') {
			old = 'iTotalDisplayRecords';
		}
	
		return json[old] !== undefined
			? json[old]
			: json[param];
	}
	
	
	/**
	 * Filter the table using both the global filter and column based filtering
	 *  @param {object} settings dataTables settings object
	 *  @param {object} input search information
	 *  @memberof DataTable#oApi
	 */
	function _fnFilterComplete ( settings, input )
	{
		var columnsSearch = settings.aoPreSearchCols;
	
		// In server-side processing all filtering is done by the server, so no point hanging around here
		if ( _fnDataSource( settings ) != 'ssp' )
		{
			// Check if any of the rows were invalidated
			_fnFilterData( settings );
	
			// Start from the full data set
			settings.aiDisplay = settings.aiDisplayMaster.slice();
	
			// Global filter first
			_fnFilter( settings.aiDisplay, settings, input.search, input );
	
			$.each(settings.searchFixed, function (name, term) {
				_fnFilter(settings.aiDisplay, settings, term, {});
			});
	
			// Then individual column filters
			for ( var i=0 ; i<columnsSearch.length ; i++ )
			{
				var col = columnsSearch[i];
	
				_fnFilter(
					settings.aiDisplay,
					settings,
					col.search,
					col,
					i
				);
	
				$.each(settings.aoColumns[i].searchFixed, function (name, term) {
					_fnFilter(settings.aiDisplay, settings, term, {}, i);
				});
			}
	
			// And finally global filtering
			_fnFilterCustom( settings );
		}
	
		// Tell the draw function we have been filtering
		settings.bFiltered = true;
	
		_fnCallbackFire( settings, null, 'search', [settings] );
	}
	
	
	/**
	 * Apply custom filtering functions
	 * 
	 * This is legacy now that we have named functions, but it is widely used
	 * from 1.x, so it is not yet deprecated.
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnFilterCustom( settings )
	{
		var filters = DataTable.ext.search;
		var displayRows = settings.aiDisplay;
		var row, rowIdx;
	
		for ( var i=0, ien=filters.length ; i<ien ; i++ ) {
			var rows = [];
	
			// Loop over each row and see if it should be included
			for ( var j=0, jen=displayRows.length ; j<jen ; j++ ) {
				rowIdx = displayRows[ j ];
				row = settings.aoData[ rowIdx ];
	
				if ( filters[i]( settings, row._aFilterData, rowIdx, row._aData, j ) ) {
					rows.push( rowIdx );
				}
			}
	
			// So the array reference doesn't break set the results into the
			// existing array
			displayRows.length = 0;
			_fnArrayApply(displayRows, rows);
		}
	}
	
	
	/**
	 * Filter the data table based on user input and draw the table
	 */
	function _fnFilter( searchRows, settings, input, options, column )
	{
		if ( input === '' ) {
			return;
		}
	
		var i = 0;
		var matched = [];
	
		// Search term can be a function, regex or string - if a string we apply our
		// smart filtering regex (assuming the options require that)
		var searchFunc = typeof input === 'function' ? input : null;
		var rpSearch = input instanceof RegExp
			? input
			: searchFunc
				? null
				: _fnFilterCreateSearch( input, options );
	
		// Then for each row, does the test pass. If not, lop the row from the array
		for (i=0 ; i<searchRows.length ; i++) {
			var row = settings.aoData[ searchRows[i] ];
			var data = column === undefined
				? row._sFilterRow
				: row._aFilterData[ column ];
	
			if ( (searchFunc && searchFunc(data, row._aData, searchRows[i], column)) || (rpSearch && rpSearch.test(data)) ) {
				matched.push(searchRows[i]);
			}
		}
	
		// Mutate the searchRows array
		searchRows.length = matched.length;
	
		for (i=0 ; i<matched.length ; i++) {
			searchRows[i] = matched[i];
		}
	}
	
	
	/**
	 * Build a regular expression object suitable for searching a table
	 *  @param {string} sSearch string to search for
	 *  @param {bool} bRegex treat as a regular expression or not
	 *  @param {bool} bSmart perform smart filtering or not
	 *  @param {bool} bCaseInsensitive Do case insensitive matching or not
	 *  @returns {RegExp} constructed object
	 *  @memberof DataTable#oApi
	 */
	function _fnFilterCreateSearch( search, inOpts )
	{
		var not = [];
		var options = $.extend({}, {
			boundary: false,
			caseInsensitive: true,
			exact: false,
			regex: false,
			smart: true
		}, inOpts);
	
		if (typeof search !== 'string') {
			search = search.toString();
		}
	
		// Remove diacritics if normalize is set up to do so
		search = _normalize(search);
	
		if (options.exact) {
			return new RegExp(
				'^'+_fnEscapeRegex(search)+'$',
				options.caseInsensitive ? 'i' : ''
			);
		}
	
		search = options.regex ?
			search :
			_fnEscapeRegex( search );
		
		if ( options.smart ) {
			/* For smart filtering we want to allow the search to work regardless of
			 * word order. We also want double quoted text to be preserved, so word
			 * order is important - a la google. And a negative look around for
			 * finding rows which don't contain a given string.
			 * 
			 * So this is the sort of thing we want to generate:
			 * 
			 * ^(?=.*?\bone\b)(?=.*?\btwo three\b)(?=.*?\bfour\b).*$
			 */
			var parts = search.match( /!?["\u201C][^"\u201D]+["\u201D]|[^ ]+/g ) || [''];
			var a = parts.map( function ( word ) {
				var negative = false;
				var m;
	
				// Determine if it is a "does not include"
				if ( word.charAt(0) === '!' ) {
					negative = true;
					word = word.substring(1);
				}
	
				// Strip the quotes from around matched phrases
				if ( word.charAt(0) === '"' ) {
					m = word.match( /^"(.*)"$/ );
					word = m ? m[1] : word;
				}
				else if ( word.charAt(0) === '\u201C' ) {
					// Smart quote match (iPhone users)
					m = word.match( /^\u201C(.*)\u201D$/ );
					word = m ? m[1] : word;
				}
	
				// For our "not" case, we need to modify the string that is
				// allowed to match at the end of the expression.
				if (negative) {
					if (word.length > 1) {
						not.push('(?!'+word+')');
					}
	
					word = '';
				}
	
				return word.replace(/"/g, '');
			} );
	
			var match = not.length
				? not.join('')
				: '';
	
			var boundary = options.boundary
				? '\\b'
				: '';
	
			search = '^(?=.*?'+boundary+a.join( ')(?=.*?'+boundary )+')('+match+'.)*$';
		}
	
		return new RegExp( search, options.caseInsensitive ? 'i' : '' );
	}
	
	
	/**
	 * Escape a string such that it can be used in a regular expression
	 *  @param {string} sVal string to escape
	 *  @returns {string} escaped string
	 *  @memberof DataTable#oApi
	 */
	var _fnEscapeRegex = DataTable.util.escapeRegex;
	
	var __filter_div = $('<div>')[0];
	var __filter_div_textContent = __filter_div.textContent !== undefined;
	
	// Update the filtering data for each row if needed (by invalidation or first run)
	function _fnFilterData ( settings )
	{
		var columns = settings.aoColumns;
		var data = settings.aoData;
		var column;
		var j, jen, filterData, cellData, row;
		var wasInvalidated = false;
	
		for ( var rowIdx=0 ; rowIdx<data.length ; rowIdx++ ) {
			if (! data[rowIdx]) {
				continue;
			}
	
			row = data[rowIdx];
	
			if ( ! row._aFilterData ) {
				filterData = [];
	
				for ( j=0, jen=columns.length ; j<jen ; j++ ) {
					column = columns[j];
	
					if ( column.bSearchable ) {
						cellData = _fnGetCellData( settings, rowIdx, j, 'filter' );
	
						// Search in DataTables is string based
						if ( cellData === null ) {
							cellData = '';
						}
	
						if ( typeof cellData !== 'string' && cellData.toString ) {
							cellData = cellData.toString();
						}
					}
					else {
						cellData = '';
					}
	
					// If it looks like there is an HTML entity in the string,
					// attempt to decode it so sorting works as expected. Note that
					// we could use a single line of jQuery to do this, but the DOM
					// method used here is much faster https://jsperf.com/html-decode
					if ( cellData.indexOf && cellData.indexOf('&') !== -1 ) {
						__filter_div.innerHTML = cellData;
						cellData = __filter_div_textContent ?
							__filter_div.textContent :
							__filter_div.innerText;
					}
	
					if ( cellData.replace ) {
						cellData = cellData.replace(/[\r\n\u2028]/g, '');
					}
	
					filterData.push( cellData );
				}
	
				row._aFilterData = filterData;
				row._sFilterRow = filterData.join('  ');
				wasInvalidated = true;
			}
		}
	
		return wasInvalidated;
	}
	
	
	/**
	 * Draw the table for the first time, adding all required features
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnInitialise ( settings )
	{
		var i;
		var init = settings.oInit;
		var deferLoading = settings.deferLoading;
		var dataSrc = _fnDataSource( settings );
	
		// Ensure that the table data is fully initialised
		if ( ! settings.bInitialised ) {
			setTimeout( function(){ _fnInitialise( settings ); }, 200 );
			return;
		}
	
		// Build the header / footer for the table
		_fnBuildHead( settings, 'header' );
		_fnBuildHead( settings, 'footer' );
	
		// Load the table's state (if needed) and then render around it and draw
		_fnLoadState( settings, init, function () {
			// Then draw the header / footer
			_fnDrawHead( settings, settings.aoHeader );
			_fnDrawHead( settings, settings.aoFooter );
	
			// Cache the paging start point, as the first redraw will reset it
			var iAjaxStart = settings.iInitDisplayStart
	
			// Local data load
			// Check if there is data passing into the constructor
			if ( init.aaData ) {
				for ( i=0 ; i<init.aaData.length ; i++ ) {
					_fnAddData( settings, init.aaData[ i ] );
				}
			}
			else if ( deferLoading || dataSrc == 'dom' ) {
				// Grab the data from the page
				_fnAddTr( settings, $(settings.nTBody).children('tr') );
			}
	
			// Filter not yet applied - copy the display master
			settings.aiDisplay = settings.aiDisplayMaster.slice();
	
			// Enable features
			_fnAddOptionsHtml( settings );
			_fnSortInit( settings );
	
			_colGroup( settings );
	
			/* Okay to show that something is going on now */
			_fnProcessingDisplay( settings, true );
	
			_fnCallbackFire( settings, null, 'preInit', [settings], true );
	
			// If there is default sorting required - let's do it. The sort function
			// will do the drawing for us. Otherwise we draw the table regardless of the
			// Ajax source - this allows the table to look initialised for Ajax sourcing
			// data (show 'loading' message possibly)
			_fnReDraw( settings );
	
			// Server-side processing init complete is done by _fnAjaxUpdateDraw
			if ( dataSrc != 'ssp' || deferLoading ) {
				// if there is an ajax source load the data
				if ( dataSrc == 'ajax' ) {
					_fnBuildAjax( settings, {}, function(json) {
						var aData = _fnAjaxDataSrc( settings, json );
	
						// Got the data - add it to the table
						for ( i=0 ; i<aData.length ; i++ ) {
							_fnAddData( settings, aData[i] );
						}
	
						// Reset the init display for cookie saving. We've already done
						// a filter, and therefore cleared it before. So we need to make
						// it appear 'fresh'
						settings.iInitDisplayStart = iAjaxStart;
	
						_fnReDraw( settings );
						_fnProcessingDisplay( settings, false );
						_fnInitComplete( settings );
					}, settings );
				}
				else {
					_fnInitComplete( settings );
					_fnProcessingDisplay( settings, false );
				}
			}
		} );
	}
	
	
	/**
	 * Draw the table for the first time, adding all required features
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnInitComplete ( settings )
	{
		if (settings._bInitComplete) {
			return;
		}
	
		var args = [settings, settings.json];
	
		settings._bInitComplete = true;
	
		// Table is fully set up and we have data, so calculate the
		// column widths
		_fnAdjustColumnSizing( settings );
	
		_fnCallbackFire( settings, null, 'plugin-init', args, true );
		_fnCallbackFire( settings, 'aoInitComplete', 'init', args, true );
	}
	
	function _fnLengthChange ( settings, val )
	{
		var len = parseInt( val, 10 );
		settings._iDisplayLength = len;
	
		_fnLengthOverflow( settings );
	
		// Fire length change event
		_fnCallbackFire( settings, null, 'length', [settings, len] );
	}
	
	/**
	 * Alter the display settings to change the page
	 *  @param {object} settings DataTables settings object
	 *  @param {string|int} action Paging action to take: "first", "previous",
	 *    "next" or "last" or page number to jump to (integer)
	 *  @param [bool] redraw Automatically draw the update or not
	 *  @returns {bool} true page has changed, false - no change
	 *  @memberof DataTable#oApi
	 */
	function _fnPageChange ( settings, action, redraw )
	{
		var
			start     = settings._iDisplayStart,
			len       = settings._iDisplayLength,
			records   = settings.fnRecordsDisplay();
	
		if ( records === 0 || len === -1 )
		{
			start = 0;
		}
		else if ( typeof action === "number" )
		{
			start = action * len;
	
			if ( start > records )
			{
				start = 0;
			}
		}
		else if ( action == "first" )
		{
			start = 0;
		}
		else if ( action == "previous" )
		{
			start = len >= 0 ?
				start - len :
				0;
	
			if ( start < 0 )
			{
				start = 0;
			}
		}
		else if ( action == "next" )
		{
			if ( start + len < records )
			{
				start += len;
			}
		}
		else if ( action == "last" )
		{
			start = Math.floor( (records-1) / len) * len;
		}
		else if ( action === 'ellipsis' )
		{
			return;
		}
		else
		{
			_fnLog( settings, 0, "Unknown paging action: "+action, 5 );
		}
	
		var changed = settings._iDisplayStart !== start;
		settings._iDisplayStart = start;
	
		_fnCallbackFire( settings, null, changed ? 'page' : 'page-nc', [settings] );
	
		if ( changed && redraw ) {
			_fnDraw( settings );
		}
	
		return changed;
	}
	
	
	/**
	 * Generate the node required for the processing node
	 *  @param {object} settings DataTables settings object
	 */
	function _processingHtml ( settings )
	{
		var table = settings.nTable;
		var scrolling = settings.oScroll.sX !== '' || settings.oScroll.sY !== '';
	
		if ( settings.oFeatures.bProcessing ) {
			var n = $('<div/>', {
					'id': settings.sTableId + '_processing',
					'class': settings.oClasses.processing.container,
					'role': 'status'
				} )
				.html( settings.oLanguage.sProcessing )
				.append('<div><div></div><div></div><div></div><div></div></div>');
	
			// Different positioning depending on if scrolling is enabled or not
			if (scrolling) {
				n.prependTo( $('div.dt-scroll', settings.nTableWrapper) );
			}
			else {
				n.insertBefore( table );
			}
	
			$(table).on( 'processing.dt.DT', function (e, s, show) {
				n.css( 'display', show ? 'block' : 'none' );
			} );
		}
	}
	
	
	/**
	 * Display or hide the processing indicator
	 *  @param {object} settings DataTables settings object
	 *  @param {bool} show Show the processing indicator (true) or not (false)
	 */
	function _fnProcessingDisplay ( settings, show )
	{
		// Ignore cases when we are still redrawing
		if (settings.bDrawing && show === false) {
			return;
		}
	
		_fnCallbackFire( settings, null, 'processing', [settings, show] );
	}
	
	/**
	 * Show the processing element if an action takes longer than a given time
	 *
	 * @param {*} settings DataTables settings object
	 * @param {*} enable Do (true) or not (false) async processing (local feature enablement)
	 * @param {*} run Function to run
	 */
	function _fnProcessingRun( settings, enable, run ) {
		if (! enable) {
			// Immediate execution, synchronous
			run();
		}
		else {
			_fnProcessingDisplay(settings, true);
			
			// Allow the processing display to show if needed
			setTimeout(function () {
				run();
	
				_fnProcessingDisplay(settings, false);
			}, 0);
		}
	}
	/**
	 * Add any control elements for the table - specifically scrolling
	 *  @param {object} settings dataTables settings object
	 *  @returns {node} Node to add to the DOM
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlTable ( settings )
	{
		var table = $(settings.nTable);
	
		// Scrolling from here on in
		var scroll = settings.oScroll;
	
		if ( scroll.sX === '' && scroll.sY === '' ) {
			return settings.nTable;
		}
	
		var scrollX = scroll.sX;
		var scrollY = scroll.sY;
		var classes = settings.oClasses.scrolling;
		var caption = settings.captionNode;
		var captionSide = caption ? caption._captionSide : null;
		var headerClone = $( table[0].cloneNode(false) );
		var footerClone = $( table[0].cloneNode(false) );
		var footer = table.children('tfoot');
		var _div = '<div/>';
		var size = function ( s ) {
			return !s ? null : _fnStringToCss( s );
		};
	
		if ( ! footer.length ) {
			footer = null;
		}
	
		/*
		 * The HTML structure that we want to generate in this function is:
		 *  div - scroller
		 *    div - scroll head
		 *      div - scroll head inner
		 *        table - scroll head table
		 *          thead - thead
		 *    div - scroll body
		 *      table - table (master table)
		 *        thead - thead clone for sizing
		 *        tbody - tbody
		 *    div - scroll foot
		 *      div - scroll foot inner
		 *        table - scroll foot table
		 *          tfoot - tfoot
		 */
		var scroller = $( _div, { 'class': classes.container } )
			.append(
				$(_div, { 'class': classes.header.self } )
					.css( {
						overflow: 'hidden',
						position: 'relative',
						border: 0,
						width: scrollX ? size(scrollX) : '100%'
					} )
					.append(
						$(_div, { 'class': classes.header.inner } )
							.css( {
								'box-sizing': 'content-box',
								width: scroll.sXInner || '100%'
							} )
							.append(
								headerClone
									.removeAttr('id')
									.css( 'margin-left', 0 )
									.append( captionSide === 'top' ? caption : null )
									.append(
										table.children('thead')
									)
							)
					)
			)
			.append(
				$(_div, { 'class': classes.body } )
					.css( {
						position: 'relative',
						overflow: 'auto',
						width: size( scrollX )
					} )
					.append( table )
			);
	
		if ( footer ) {
			scroller.append(
				$(_div, { 'class': classes.footer.self } )
					.css( {
						overflow: 'hidden',
						border: 0,
						width: scrollX ? size(scrollX) : '100%'
					} )
					.append(
						$(_div, { 'class': classes.footer.inner } )
							.append(
								footerClone
									.removeAttr('id')
									.css( 'margin-left', 0 )
									.append( captionSide === 'bottom' ? caption : null )
									.append(
										table.children('tfoot')
									)
							)
					)
			);
		}
	
		var children = scroller.children();
		var scrollHead = children[0];
		var scrollBody = children[1];
		var scrollFoot = footer ? children[2] : null;
	
		// When the body is scrolled, then we also want to scroll the headers
		$(scrollBody).on( 'scroll.DT', function () {
			var scrollLeft = this.scrollLeft;
	
			scrollHead.scrollLeft = scrollLeft;
	
			if ( footer ) {
				scrollFoot.scrollLeft = scrollLeft;
			}
		} );
	
		// When focus is put on the header cells, we might need to scroll the body
		$('th, td', scrollHead).on('focus', function () {
			var scrollLeft = scrollHead.scrollLeft;
	
			scrollBody.scrollLeft = scrollLeft;
	
			if ( footer ) {
				scrollBody.scrollLeft = scrollLeft;
			}
		});
	
		$(scrollBody).css('max-height', scrollY);
		if (! scroll.bCollapse) {
			$(scrollBody).css('height', scrollY);
		}
	
		settings.nScrollHead = scrollHead;
		settings.nScrollBody = scrollBody;
		settings.nScrollFoot = scrollFoot;
	
		// On redraw - align columns
		settings.aoDrawCallback.push(_fnScrollDraw);
	
		return scroller[0];
	}
	
	
	
	/**
	 * Update the header, footer and body tables for resizing - i.e. column
	 * alignment.
	 *
	 * Welcome to the most horrible function DataTables. The process that this
	 * function follows is basically:
	 *   1. Re-create the table inside the scrolling div
	 *   2. Correct colgroup > col values if needed
	 *   3. Copy colgroup > col over to header and footer
	 *   4. Clean up
	 *
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnScrollDraw ( settings )
	{
		// Given that this is such a monster function, a lot of variables are use
		// to try and keep the minimised size as small as possible
		var
			scroll         = settings.oScroll,
			barWidth       = scroll.iBarWidth,
			divHeader      = $(settings.nScrollHead),
			divHeaderInner = divHeader.children('div'),
			divHeaderTable = divHeaderInner.children('table'),
			divBodyEl      = settings.nScrollBody,
			divBody        = $(divBodyEl),
			divFooter      = $(settings.nScrollFoot),
			divFooterInner = divFooter.children('div'),
			divFooterTable = divFooterInner.children('table'),
			header         = $(settings.nTHead),
			table          = $(settings.nTable),
			footer         = settings.nTFoot && $('th, td', settings.nTFoot).length ? $(settings.nTFoot) : null,
			browser        = settings.oBrowser,
			headerCopy, footerCopy;
	
		// If the scrollbar visibility has changed from the last draw, we need to
		// adjust the column sizes as the table width will have changed to account
		// for the scrollbar
		var scrollBarVis = divBodyEl.scrollHeight > divBodyEl.clientHeight;
		
		if ( settings.scrollBarVis !== scrollBarVis && settings.scrollBarVis !== undefined ) {
			settings.scrollBarVis = scrollBarVis;
			_fnAdjustColumnSizing( settings );
			return; // adjust column sizing will call this function again
		}
		else {
			settings.scrollBarVis = scrollBarVis;
		}
	
		// 1. Re-create the table inside the scrolling div
		// Remove the old minimised thead and tfoot elements in the inner table
		table.children('thead, tfoot').remove();
	
		// Clone the current header and footer elements and then place it into the inner table
		headerCopy = header.clone().prependTo( table );
		headerCopy.find('th, td').removeAttr('tabindex');
		headerCopy.find('[id]').removeAttr('id');
	
		if ( footer ) {
			footerCopy = footer.clone().prependTo( table );
			footerCopy.find('[id]').removeAttr('id');
		}
	
		// 2. Correct colgroup > col values if needed
		// It is possible that the cell sizes are smaller than the content, so we need to
		// correct colgroup>col for such cases. This can happen if the auto width detection
		// uses a cell which has a longer string, but isn't the widest! For example 
		// "Chief Executive Officer (CEO)" is the longest string in the demo, but
		// "Systems Administrator" is actually the widest string since it doesn't collapse.
		// Note the use of translating into a column index to get the `col` element. This
		// is because of Responsive which might remove `col` elements, knocking the alignment
		// of the indexes out.
		if (settings.aiDisplay.length) {
			// Get the column sizes from the first row in the table. This should really be a
			// [].find, but it wasn't supported in Chrome until Sept 2015, and DT has 10 year
			// browser support
			var firstTr = null;
			var start = _fnDataSource( settings ) !== 'ssp'
				? settings._iDisplayStart
				: 0;
	
			for (i=start ; i<start + settings.aiDisplay.length ; i++) {
				var idx = settings.aiDisplay[i];
				var tr = settings.aoData[idx].nTr;
	
				if (tr) {
					firstTr = tr;
					break;
				}
			}
	
			if (firstTr) {
				var colSizes = $(firstTr).children('th, td').map(function (vis) {
					return {
						idx: _fnVisibleToColumnIndex(settings, vis),
						width: $(this).outerWidth()
					};
				});
	
				// Check against what the colgroup > col is set to and correct if needed
				for (var i=0 ; i<colSizes.length ; i++) {
					var colEl = settings.aoColumns[ colSizes[i].idx ].colEl[0];
					var colWidth = colEl.style.width.replace('px', '');
	
					if (colWidth !== colSizes[i].width) {
						colEl.style.width = colSizes[i].width + 'px';
	
						if (scroll.sX) {
							colEl.style.minWidth = colSizes[i].width + 'px';
						}
					}
				}
			}
		}
	
		// 3. Copy the colgroup over to the header and footer
		divHeaderTable
			.find('colgroup')
			.remove();
	
		divHeaderTable.append(settings.colgroup.clone());
	
		if ( footer ) {
			divFooterTable
				.find('colgroup')
				.remove();
	
			divFooterTable.append(settings.colgroup.clone());
		}
	
		// "Hide" the header and footer that we used for the sizing. We need to keep
		// the content of the cell so that the width applied to the header and body
		// both match, but we want to hide it completely.
		$('th, td', headerCopy).each(function () {
			$(this.childNodes).wrapAll('<div class="dt-scroll-sizing">');
		});
	
		if ( footer ) {
			$('th, td', footerCopy).each(function () {
				$(this.childNodes).wrapAll('<div class="dt-scroll-sizing">');
			});
		}
	
		// 4. Clean up
		// Figure out if there are scrollbar present - if so then we need a the header and footer to
		// provide a bit more space to allow "overflow" scrolling (i.e. past the scrollbar)
		var isScrolling = Math.floor(table.height()) > divBodyEl.clientHeight || divBody.css('overflow-y') == "scroll";
		var paddingSide = 'padding' + (browser.bScrollbarLeft ? 'Left' : 'Right' );
	
		// Set the width's of the header and footer tables
		var outerWidth = table.outerWidth();
	
		divHeaderTable.css('width', _fnStringToCss( outerWidth ));
		divHeaderInner
			.css('width', _fnStringToCss( outerWidth ))
			.css(paddingSide, isScrolling ? barWidth+"px" : "0px");
	
		if ( footer ) {
			divFooterTable.css('width', _fnStringToCss( outerWidth ));
			divFooterInner
				.css('width', _fnStringToCss( outerWidth ))
				.css(paddingSide, isScrolling ? barWidth+"px" : "0px");
		}
	
		// Correct DOM ordering for colgroup - comes before the thead
		table.children('colgroup').prependTo(table);
	
		// Adjust the position of the header in case we loose the y-scrollbar
		divBody.trigger('scroll');
	
		// If sorting or filtering has occurred, jump the scrolling back to the top
		// only if we aren't holding the position
		if ( (settings.bSorted || settings.bFiltered) && ! settings._drawHold ) {
			divBodyEl.scrollTop = 0;
		}
	}
	
	/**
	 * Calculate the width of columns for the table
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnCalculateColumnWidths ( settings )
	{
		// Not interested in doing column width calculation if auto-width is disabled
		if (! settings.oFeatures.bAutoWidth) {
			return;
		}
	
		var
			table = settings.nTable,
			columns = settings.aoColumns,
			scroll = settings.oScroll,
			scrollY = scroll.sY,
			scrollX = scroll.sX,
			scrollXInner = scroll.sXInner,
			visibleColumns = _fnGetColumns( settings, 'bVisible' ),
			tableWidthAttr = table.getAttribute('width'), // from DOM element
			tableContainer = table.parentNode,
			i, column, columnIdx;
			
		var styleWidth = table.style.width;
		var containerWidth = _fnWrapperWidth(settings);
	
		// Don't re-run for the same width as the last time
		if (containerWidth === settings.containerWidth) {
			return false;
		}
	
		settings.containerWidth = containerWidth;
	
		// If there is no width applied as a CSS style or as an attribute, we assume that
		// the width is intended to be 100%, which is usually is in CSS, but it is very
		// difficult to correctly parse the rules to get the final result.
		if ( ! styleWidth && ! tableWidthAttr) {
			table.style.width = '100%';
			styleWidth = '100%';
		}
	
		if ( styleWidth && styleWidth.indexOf('%') !== -1 ) {
			tableWidthAttr = styleWidth;
		}
	
		// Let plug-ins know that we are doing a recalc, in case they have changed any of the
		// visible columns their own way (e.g. Responsive uses display:none).
		_fnCallbackFire(
			settings,
			null,
			'column-calc',
			{visible: visibleColumns},
			false
		);
	
		// Construct a single row, worst case, table with the widest
		// node in the data, assign any user defined widths, then insert it into
		// the DOM and allow the browser to do all the hard work of calculating
		// table widths
		var tmpTable = $(table.cloneNode())
			.css( 'visibility', 'hidden' )
			.removeAttr( 'id' );
	
		// Clean up the table body
		tmpTable.append('<tbody>')
		var tr = $('<tr/>').appendTo( tmpTable.find('tbody') );
	
		// Clone the table header and footer - we can't use the header / footer
		// from the cloned table, since if scrolling is active, the table's
		// real header and footer are contained in different table tags
		tmpTable
			.append( $(settings.nTHead).clone() )
			.append( $(settings.nTFoot).clone() );
	
		// Remove any assigned widths from the footer (from scrolling)
		tmpTable.find('tfoot th, tfoot td').css('width', '');
	
		// Apply custom sizing to the cloned header
		tmpTable.find('thead th, thead td').each( function () {
			// Get the `width` from the header layout
			var width = _fnColumnsSumWidth( settings, this, true, false );
	
			if ( width ) {
				this.style.width = width;
	
				// For scrollX we need to force the column width otherwise the
				// browser will collapse it. If this width is smaller than the
				// width the column requires, then it will have no effect
				if ( scrollX ) {
					this.style.minWidth = width;
	
					$( this ).append( $('<div/>').css( {
						width: width,
						margin: 0,
						padding: 0,
						border: 0,
						height: 1
					} ) );
				}
			}
			else {
				this.style.width = '';
			}
		} );
	
		// Find the widest piece of data for each column and put it into the table
		for ( i=0 ; i<visibleColumns.length ; i++ ) {
			columnIdx = visibleColumns[i];
			column = columns[ columnIdx ];
	
			var longest = _fnGetMaxLenString(settings, columnIdx);
			var autoClass = _ext.type.className[column.sType];
			var text = longest + column.sContentPadding;
			var insert = longest.indexOf('<') === -1
				? document.createTextNode(text)
				: text
			
			$('<td/>')
				.addClass(autoClass)
				.addClass(column.sClass)
				.append(insert)
				.appendTo(tr);
		}
	
		// Tidy the temporary table - remove name attributes so there aren't
		// duplicated in the dom (radio elements for example)
		$('[name]', tmpTable).removeAttr('name');
	
		// Table has been built, attach to the document so we can work with it.
		// A holding element is used, positioned at the top of the container
		// with minimal height, so it has no effect on if the container scrolls
		// or not. Otherwise it might trigger scrolling when it actually isn't
		// needed
		var holder = $('<div/>').css( scrollX || scrollY ?
				{
					position: 'absolute',
					top: 0,
					left: 0,
					height: 1,
					right: 0,
					overflow: 'hidden'
				} :
				{}
			)
			.append( tmpTable )
			.appendTo( tableContainer );
	
		// When scrolling (X or Y) we want to set the width of the table as 
		// appropriate. However, when not scrolling leave the table width as it
		// is. This results in slightly different, but I think correct behaviour
		if ( scrollX && scrollXInner ) {
			tmpTable.width( scrollXInner );
		}
		else if ( scrollX ) {
			tmpTable.css( 'width', 'auto' );
			tmpTable.removeAttr('width');
	
			// If there is no width attribute or style, then allow the table to
			// collapse
			if ( tmpTable.outerWidth() < tableContainer.clientWidth && tableWidthAttr ) {
				tmpTable.outerWidth( tableContainer.clientWidth );
			}
		}
		else if ( scrollY ) {
			tmpTable.outerWidth( tableContainer.clientWidth );
		}
		else if ( tableWidthAttr ) {
			tmpTable.outerWidth( tableWidthAttr );
		}
	
		// Get the width of each column in the constructed table
		var total = 0;
		var bodyCells = tmpTable.find('tbody tr').eq(0).children();
	
		for ( i=0 ; i<visibleColumns.length ; i++ ) {
			// Use getBounding for sub-pixel accuracy, which we then want to round up!
			var bounding = bodyCells[i].getBoundingClientRect().width;
	
			// Total is tracked to remove any sub-pixel errors as the outerWidth
			// of the table might not equal the total given here
			total += bounding;
	
			// Width for each column to use
			columns[ visibleColumns[i] ].sWidth = _fnStringToCss( bounding );
		}
	
		table.style.width = _fnStringToCss( total );
	
		// Finished with the table - ditch it
		holder.remove();
	
		// If there is a width attr, we want to attach an event listener which
		// allows the table sizing to automatically adjust when the window is
		// resized. Use the width attr rather than CSS, since we can't know if the
		// CSS is a relative value or absolute - DOM read is always px.
		if ( tableWidthAttr ) {
			table.style.width = _fnStringToCss( tableWidthAttr );
		}
	
		if ( (tableWidthAttr || scrollX) && ! settings._reszEvt ) {
			var resize = DataTable.util.throttle( function () {
				var newWidth = _fnWrapperWidth(settings);
	
				// Don't do it if destroying or the container width is 0
				if (! settings.bDestroying && newWidth !== 0) {
					_fnAdjustColumnSizing( settings );
				}
			} );
	
			// For browsers that support it (~2020 onwards for wide support) we can watch for the
			// container changing width.
			if (window.ResizeObserver) {
				// This is a tricky beast - if the element is visible when `.observe()` is called,
				// then the callback is immediately run. Which we don't want. If the element isn't
				// visible, then it isn't run, but we want it to run when it is then made visible.
				// This flag allows the above to be satisfied.
				var first = $(settings.nTableWrapper).is(':visible');
	
				// Use an empty div to attach the observer so it isn't impacted by height changes
				var resizer = $('<div>')
					.css({
						width: '100%',
						height: 0
					})
					.addClass('dt-autosize')
					.appendTo(settings.nTableWrapper);
	
				settings.resizeObserver = new ResizeObserver(function (e) {
					if (first) {
						first = false;
					}
					else {
						resize();
					}
				});
	
				settings.resizeObserver.observe(resizer[0]);
			}
			else {
				// For old browsers, the best we can do is listen for a window resize
				$(window).on('resize.DT-'+settings.sInstance, resize);
			}
	
			settings._reszEvt = true;
		}
	}
	
	/**
	 * Get the width of the DataTables wrapper element
	 *
	 * @param {*} settings DataTables settings object
	 * @returns Width
	 */
	function _fnWrapperWidth(settings) {
		return $(settings.nTableWrapper).is(':visible')
			? $(settings.nTableWrapper).width()
			: 0;
	}
	
	/**
	 * Get the maximum strlen for each data column
	 *  @param {object} settings dataTables settings object
	 *  @param {int} colIdx column of interest
	 *  @returns {string} string of the max length
	 *  @memberof DataTable#oApi
	 */
	function _fnGetMaxLenString( settings, colIdx )
	{
		var column = settings.aoColumns[colIdx];
	
		if (! column.maxLenString) {
			var s, max='', maxLen = -1;
		
			for ( var i=0, ien=settings.aiDisplayMaster.length ; i<ien ; i++ ) {
				var rowIdx = settings.aiDisplayMaster[i];
				var data = _fnGetRowDisplay(settings, rowIdx)[colIdx];
	
				var cellString = data && typeof data === 'object' && data.nodeType
					? data.innerHTML
					: data+'';
	
				// Remove id / name attributes from elements so they
				// don't interfere with existing elements
				cellString = cellString
					.replace(/id=".*?"/g, '')
					.replace(/name=".*?"/g, '');
	
				s = _stripHtml(cellString)
					.replace( /&nbsp;/g, ' ' );
		
				if ( s.length > maxLen ) {
					// We want the HTML in the string, but the length that
					// is important is the stripped string
					max = cellString;
					maxLen = s.length;
				}
			}
	
			column.maxLenString = max;
		}
	
		return column.maxLenString;
	}
	
	
	/**
	 * Append a CSS unit (only if required) to a string
	 *  @param {string} value to css-ify
	 *  @returns {string} value with css unit
	 *  @memberof DataTable#oApi
	 */
	function _fnStringToCss( s )
	{
		if ( s === null ) {
			return '0px';
		}
	
		if ( typeof s == 'number' ) {
			return s < 0 ?
				'0px' :
				s+'px';
		}
	
		// Check it has a unit character already
		return s.match(/\d$/) ?
			s+'px' :
			s;
	}
	
	/**
	 * Re-insert the `col` elements for current visibility
	 *
	 * @param {*} settings DT settings
	 */
	function _colGroup( settings ) {
		var cols = settings.aoColumns;
	
		settings.colgroup.empty();
	
		for (i=0 ; i<cols.length ; i++) {
			if (cols[i].bVisible) {
				settings.colgroup.append(cols[i].colEl);
			}
		}
	}
	
	
	function _fnSortInit( settings ) {
		var target = settings.nTHead;
		var headerRows = target.querySelectorAll('tr');
		var titleRow = settings.titleRow;
		var notSelector = ':not([data-dt-order="disable"]):not([data-dt-order="icon-only"])';
		
		// Legacy support for `orderCellsTop`
		if (titleRow === true) {
			target = headerRows[0];
		}
		else if (titleRow === false) {
			target = headerRows[ headerRows.length - 1 ];
		}
		else if (titleRow !== null) {
			target = headerRows[titleRow];
		}
		// else - all rows
	
		if (settings.orderHandler) {
			_fnSortAttachListener(
				settings,
				target,
				target === settings.nTHead
					? 'tr'+notSelector+' th'+notSelector+', tr'+notSelector+' td'+notSelector
					: 'th'+notSelector+', td'+notSelector
			);
		}
	
		// Need to resolve the user input array into our internal structure
		var order = [];
		_fnSortResolve( settings, order, settings.aaSorting );
	
		settings.aaSorting = order;
	}
	
	
	function _fnSortAttachListener(settings, node, selector, column, callback) {
		_fnBindAction( node, selector, function (e) {
			var run = false;
			var columns = column === undefined
				? _fnColumnsFromHeader( e.target )
				: typeof column === 'function'
					? column()
					: Array.isArray(column)
						? column
						: [column];
	
			if ( columns.length ) {
				for ( var i=0, ien=columns.length ; i<ien ; i++ ) {
					var ret = _fnSortAdd( settings, columns[i], i, e.shiftKey );
	
					if (ret !== false) {
						run = true;
					}					
	
					// If the first entry is no sort, then subsequent
					// sort columns are ignored
					if (settings.aaSorting.length === 1 && settings.aaSorting[0][1] === '') {
						break;
					}
				}
	
				if (run) {
					_fnProcessingRun(settings, true, function () {
						_fnSort( settings );
						_fnSortDisplay( settings, settings.aiDisplay );
	
						_fnReDraw( settings, false, false );
	
						if (callback) {
							callback();
						}
					});
				}
			}
		} );
	}
	
	/**
	 * Sort the display array to match the master's order
	 * @param {*} settings
	 */
	function _fnSortDisplay(settings, display) {
		if (display.length < 2) {
			return;
		}
	
		var master = settings.aiDisplayMaster;
		var masterMap = {};
		var map = {};
		var i;
	
		// Rather than needing an `indexOf` on master array, we can create a map
		for (i=0 ; i<master.length ; i++) {
			masterMap[master[i]] = i;
		}
	
		// And then cache what would be the indexOf fom the display
		for (i=0 ; i<display.length ; i++) {
			map[display[i]] = masterMap[display[i]];
		}
	
		display.sort(function(a, b){
			// Short version of this function is simply `master.indexOf(a) - master.indexOf(b);`
			return map[a] - map[b];
		});
	}
	
	
	function _fnSortResolve (settings, nestedSort, sort) {
		var push = function ( a ) {
			if ($.isPlainObject(a)) {
				if (a.idx !== undefined) {
					// Index based ordering
					nestedSort.push([a.idx, a.dir]);
				}
				else if (a.name) {
					// Name based ordering
					var cols = _pluck( settings.aoColumns, 'sName');
					var idx = cols.indexOf(a.name);
	
					if (idx !== -1) {
						nestedSort.push([idx, a.dir]);
					}
				}
			}
			else {
				// Plain column index and direction pair
				nestedSort.push(a);
			}
		};
	
		if ( $.isPlainObject(sort) ) {
			// Object
			push(sort);
		}
		else if ( sort.length && typeof sort[0] === 'number' ) {
			// 1D array
			push(sort);
		}
		else if ( sort.length ) {
			// 2D array
			for (var z=0; z<sort.length; z++) {
				push(sort[z]); // Object or array
			}
		}
	}
	
	
	function _fnSortFlatten ( settings )
	{
		var
			i, k, kLen,
			aSort = [],
			extSort = DataTable.ext.type.order,
			aoColumns = settings.aoColumns,
			aDataSort, iCol, sType, srcCol,
			fixed = settings.aaSortingFixed,
			fixedObj = $.isPlainObject( fixed ),
			nestedSort = [];
		
		if ( ! settings.oFeatures.bSort ) {
			return aSort;
		}
	
		// Build the sort array, with pre-fix and post-fix options if they have been
		// specified
		if ( Array.isArray( fixed ) ) {
			_fnSortResolve( settings, nestedSort, fixed );
		}
	
		if ( fixedObj && fixed.pre ) {
			_fnSortResolve( settings, nestedSort, fixed.pre );
		}
	
		_fnSortResolve( settings, nestedSort, settings.aaSorting );
	
		if (fixedObj && fixed.post ) {
			_fnSortResolve( settings, nestedSort, fixed.post );
		}
	
		for ( i=0 ; i<nestedSort.length ; i++ )
		{
			srcCol = nestedSort[i][0];
	
			if ( aoColumns[ srcCol ] ) {
				aDataSort = aoColumns[ srcCol ].aDataSort;
	
				for ( k=0, kLen=aDataSort.length ; k<kLen ; k++ )
				{
					iCol = aDataSort[k];
					sType = aoColumns[ iCol ].sType || 'string';
	
					if ( nestedSort[i]._idx === undefined ) {
						nestedSort[i]._idx = aoColumns[iCol].asSorting.indexOf(nestedSort[i][1]);
					}
	
					if ( nestedSort[i][1] ) {
						aSort.push( {
							src:       srcCol,
							col:       iCol,
							dir:       nestedSort[i][1],
							index:     nestedSort[i]._idx,
							type:      sType,
							formatter: extSort[ sType+"-pre" ],
							sorter:    extSort[ sType+"-"+nestedSort[i][1] ]
						} );
					}
				}
			}
		}
	
		return aSort;
	}
	
	/**
	 * Change the order of the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnSort ( oSettings, col, dir )
	{
		var
			i, ien, iLen,
			aiOrig = [],
			extSort = DataTable.ext.type.order,
			aoData = oSettings.aoData,
			sortCol,
			displayMaster = oSettings.aiDisplayMaster,
			aSort;
	
		// Make sure the columns all have types defined
		_fnColumnTypes(oSettings);
	
		// Allow a specific column to be sorted, which will _not_ alter the display
		// master
		if (col !== undefined) {
			var srcCol = oSettings.aoColumns[col];
	
			aSort = [{
				src:       col,
				col:       col,
				dir:       dir,
				index:     0,
				type:      srcCol.sType,
				formatter: extSort[ srcCol.sType+"-pre" ],
				sorter:    extSort[ srcCol.sType+"-"+dir ]
			}];
			displayMaster = displayMaster.slice();
		}
		else {
			aSort = _fnSortFlatten( oSettings );
		}
	
		for ( i=0, ien=aSort.length ; i<ien ; i++ ) {
			sortCol = aSort[i];
	
			// Load the data needed for the sort, for each cell
			_fnSortData( oSettings, sortCol.col );
		}
	
		/* No sorting required if server-side or no sorting array */
		if ( _fnDataSource( oSettings ) != 'ssp' && aSort.length !== 0 )
		{
			// Reset the initial positions on each pass so we get a stable sort
			for ( i=0, iLen=displayMaster.length ; i<iLen ; i++ ) {
				aiOrig[ i ] = i;
			}
	
			// If the first sort is desc, then reverse the array to preserve original
			// order, just in reverse
			if (aSort.length && aSort[0].dir === 'desc' && oSettings.orderDescReverse) {
				aiOrig.reverse();
			}
	
			/* Do the sort - here we want multi-column sorting based on a given data source (column)
			 * and sorting function (from oSort) in a certain direction. It's reasonably complex to
			 * follow on it's own, but this is what we want (example two column sorting):
			 *  fnLocalSorting = function(a,b){
			 *    var test;
			 *    test = oSort['string-asc']('data11', 'data12');
			 *      if (test !== 0)
			 *        return test;
			 *    test = oSort['numeric-desc']('data21', 'data22');
			 *    if (test !== 0)
			 *      return test;
			 *    return oSort['numeric-asc']( aiOrig[a], aiOrig[b] );
			 *  }
			 * Basically we have a test for each sorting column, if the data in that column is equal,
			 * test the next column. If all columns match, then we use a numeric sort on the row
			 * positions in the original data array to provide a stable sort.
			 */
			displayMaster.sort( function ( a, b ) {
				var
					x, y, k, test, sort,
					len=aSort.length,
					dataA = aoData[a]._aSortData,
					dataB = aoData[b]._aSortData;
	
				for ( k=0 ; k<len ; k++ ) {
					sort = aSort[k];
	
					// Data, which may have already been through a `-pre` function
					x = dataA[ sort.col ];
					y = dataB[ sort.col ];
	
					if (sort.sorter) {
						// If there is a custom sorter (`-asc` or `-desc`) for this
						// data type, use it
						test = sort.sorter(x, y);
	
						if ( test !== 0 ) {
							return test;
						}
					}
					else {
						// Otherwise, use generic sorting
						test = x<y ? -1 : x>y ? 1 : 0;
	
						if ( test !== 0 ) {
							return sort.dir === 'asc' ? test : -test;
						}
					}
				}
	
				x = aiOrig[a];
				y = aiOrig[b];
	
				return x<y ? -1 : x>y ? 1 : 0;
			} );
		}
		else if ( aSort.length === 0 ) {
			// Apply index order
			displayMaster.sort(function (x, y) {
				return x<y ? -1 : x>y ? 1 : 0;
			});
		}
	
		if (col === undefined) {
			// Tell the draw function that we have sorted the data
			oSettings.bSorted = true;
			oSettings.sortDetails = aSort;
	
			_fnCallbackFire( oSettings, null, 'order', [oSettings, aSort] );
		}
	
		return displayMaster;
	}
	
	
	/**
	 * Function to run on user sort request
	 *  @param {object} settings dataTables settings object
	 *  @param {node} attachTo node to attach the handler to
	 *  @param {int} colIdx column sorting index
	 *  @param {int} addIndex Counter
	 *  @param {boolean} [shift=false] Shift click add
	 *  @param {function} [callback] callback function
	 *  @memberof DataTable#oApi
	 */
	function _fnSortAdd ( settings, colIdx, addIndex, shift )
	{
		var col = settings.aoColumns[ colIdx ];
		var sorting = settings.aaSorting;
		var asSorting = col.asSorting;
		var nextSortIdx;
		var next = function ( a, overflow ) {
			var idx = a._idx;
			if ( idx === undefined ) {
				idx = asSorting.indexOf(a[1]);
			}
	
			return idx+1 < asSorting.length ?
				idx+1 :
				overflow ?
					null :
					0;
		};
	
		if ( ! col.bSortable ) {
			return false;
		}
	
		// Convert to 2D array if needed
		if ( typeof sorting[0] === 'number' ) {
			sorting = settings.aaSorting = [ sorting ];
		}
	
		// If appending the sort then we are multi-column sorting
		if ( (shift || addIndex) && settings.oFeatures.bSortMulti ) {
			// Are we already doing some kind of sort on this column?
			var sortIdx = _pluck(sorting, '0').indexOf(colIdx);
	
			if ( sortIdx !== -1 ) {
				// Yes, modify the sort
				nextSortIdx = next( sorting[sortIdx], true );
	
				if ( nextSortIdx === null && sorting.length === 1 ) {
					nextSortIdx = 0; // can't remove sorting completely
				}
	
				if ( nextSortIdx === null ) {
					sorting.splice( sortIdx, 1 );
				}
				else {
					sorting[sortIdx][1] = asSorting[ nextSortIdx ];
					sorting[sortIdx]._idx = nextSortIdx;
				}
			}
			else if (shift) {
				// No sort on this column yet, being added by shift click
				// add it as itself
				sorting.push( [ colIdx, asSorting[0], 0 ] );
				sorting[sorting.length-1]._idx = 0;
			}
			else {
				// No sort on this column yet, being added from a colspan
				// so add with same direction as first column
				sorting.push( [ colIdx, sorting[0][1], 0 ] );
				sorting[sorting.length-1]._idx = 0;
			}
		}
		else if ( sorting.length && sorting[0][0] == colIdx ) {
			// Single column - already sorting on this column, modify the sort
			nextSortIdx = next( sorting[0] );
	
			sorting.length = 1;
			sorting[0][1] = asSorting[ nextSortIdx ];
			sorting[0]._idx = nextSortIdx;
		}
		else {
			// Single column - sort only on this column
			sorting.length = 0;
			sorting.push( [ colIdx, asSorting[0] ] );
			sorting[0]._idx = 0;
		}
	}
	
	
	/**
	 * Set the sorting classes on table's body, Note: it is safe to call this function
	 * when bSort and bSortClasses are false
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnSortingClasses( settings )
	{
		var oldSort = settings.aLastSort;
		var sortClass = settings.oClasses.order.position;
		var sort = _fnSortFlatten( settings );
		var features = settings.oFeatures;
		var i, ien, colIdx;
	
		if ( features.bSort && features.bSortClasses ) {
			// Remove old sorting classes
			for ( i=0, ien=oldSort.length ; i<ien ; i++ ) {
				colIdx = oldSort[i].src;
	
				// Remove column sorting
				$( _pluck( settings.aoData, 'anCells', colIdx ) )
					.removeClass( sortClass + (i<2 ? i+1 : 3) );
			}
	
			// Add new column sorting
			for ( i=0, ien=sort.length ; i<ien ; i++ ) {
				colIdx = sort[i].src;
	
				$( _pluck( settings.aoData, 'anCells', colIdx ) )
					.addClass( sortClass + (i<2 ? i+1 : 3) );
			}
		}
	
		settings.aLastSort = sort;
	}
	
	
	// Get the data to sort a column, be it from cache, fresh (populating the
	// cache), or from a sort formatter
	function _fnSortData( settings, colIdx )
	{
		// Custom sorting function - provided by the sort data type
		var column = settings.aoColumns[ colIdx ];
		var customSort = DataTable.ext.order[ column.sSortDataType ];
		var customData;
	
		if ( customSort ) {
			customData = customSort.call( settings.oInstance, settings, colIdx,
				_fnColumnIndexToVisible( settings, colIdx )
			);
		}
	
		// Use / populate cache
		var row, cellData;
		var formatter = DataTable.ext.type.order[ column.sType+"-pre" ];
		var data = settings.aoData;
	
		for ( var rowIdx=0 ; rowIdx<data.length ; rowIdx++ ) {
			// Sparse array
			if (! data[rowIdx]) {
				continue;
			}
	
			row = data[rowIdx];
	
			if ( ! row._aSortData ) {
				row._aSortData = [];
			}
	
			if ( ! row._aSortData[colIdx] || customSort ) {
				cellData = customSort ?
					customData[rowIdx] : // If there was a custom sort function, use data from there
					_fnGetCellData( settings, rowIdx, colIdx, 'sort' );
	
				row._aSortData[ colIdx ] = formatter ?
					formatter( cellData, settings ) :
					cellData;
			}
		}
	}
	
	
	/**
	 * State information for a table
	 *
	 * @param {*} settings
	 * @returns State object
	 */
	function _fnSaveState ( settings )
	{
		if (settings._bLoadingState) {
			return;
		}
	
		// Sort state saving uses [[idx, order]] structure.
		var sorting = [];
		_fnSortResolve(settings, sorting, settings.aaSorting );
	
		/* Store the interesting variables */
		var columns = settings.aoColumns;
		var state = {
			time:    +new Date(),
			start:   settings._iDisplayStart,
			length:  settings._iDisplayLength,
			order:   sorting.map(function (sort) {
				// If a column name is available, use it
				return columns[sort[0]] && columns[sort[0]].sName
					? [ columns[sort[0]].sName, sort[1] ]
					: sort.slice();
			} ),
			search:  $.extend({}, settings.oPreviousSearch),
			columns: settings.aoColumns.map( function ( col, i ) {
				return {
					name: col.sName,
					visible: col.bVisible,
					search: $.extend({}, settings.aoPreSearchCols[i])
				};
			} )
		};
	
		settings.oSavedState = state;
		_fnCallbackFire( settings, "aoStateSaveParams", 'stateSaveParams', [settings, state] );
		
		if ( settings.oFeatures.bStateSave && !settings.bDestroying )
		{
			settings.fnStateSaveCallback.call( settings.oInstance, settings, state );
		}	
	}
	
	
	/**
	 * Attempt to load a saved table state
	 *  @param {object} oSettings dataTables settings object
	 *  @param {object} oInit DataTables init object so we can override settings
	 *  @param {function} callback Callback to execute when the state has been loaded
	 *  @memberof DataTable#oApi
	 */
	function _fnLoadState ( settings, init, callback )
	{
		if ( ! settings.oFeatures.bStateSave ) {
			callback();
			return;
		}
	
		var loaded = function(state) {
			_fnImplementState(settings, state, callback);
		}
	
		var state = settings.fnStateLoadCallback.call( settings.oInstance, settings, loaded );
	
		if ( state !== undefined ) {
			_fnImplementState( settings, state, callback );
		}
		// otherwise, wait for the loaded callback to be executed
	
		return true;
	}
	
	function _fnImplementState ( settings, s, callback) {
		var i, ien;
		var columns = settings.aoColumns;
		var currentNames = _pluck(settings.aoColumns, 'sName');
	
		settings._bLoadingState = true;
	
		// When StateRestore was introduced the state could now be implemented at any time
		// Not just initialisation. To do this an api instance is required in some places
		var api = settings._bInitComplete ? new DataTable.Api(settings) : null;
	
		if ( ! s || ! s.time ) {
			settings._bLoadingState = false;
			callback();
			return;
		}
	
		// Reject old data
		var duration = settings.iStateDuration;
		if ( duration > 0 && s.time < +new Date() - (duration*1000) ) {
			settings._bLoadingState = false;
			callback();
			return;
		}
	
		// Allow custom and plug-in manipulation functions to alter the saved data set and
		// cancelling of loading by returning false
		var abStateLoad = _fnCallbackFire( settings, 'aoStateLoadParams', 'stateLoadParams', [settings, s] );
		if ( abStateLoad.indexOf(false) !== -1 ) {
			settings._bLoadingState = false;
			callback();
			return;
		}
	
		// Store the saved state so it might be accessed at any time
		settings.oLoadedState = $.extend( true, {}, s );
	
		// This is needed for ColReorder, which has to happen first to allow all
		// the stored indexes to be usable. It is not publicly documented.
		_fnCallbackFire( settings, null, 'stateLoadInit', [settings, s], true );
	
		// Page Length
		if ( s.length !== undefined ) {
			// If already initialised just set the value directly so that the select element is also updated
			if (api) {
				api.page.len(s.length)
			}
			else {
				settings._iDisplayLength   = s.length;
			}
		}
	
		// Restore key features
		if ( s.start !== undefined ) {
			if(api === null) {
				settings._iDisplayStart    = s.start;
				settings.iInitDisplayStart = s.start;
			}
			else {
				_fnPageChange(settings, s.start/settings._iDisplayLength);
			}
		}
	
		// Order
		if ( s.order !== undefined ) {
			settings.aaSorting = [];
			$.each( s.order, function ( i, col ) {
				var set = [ col[0], col[1] ];
	
				// A column name was stored and should be used for restore
				if (typeof col[0] === 'string') {
					// Find the name from the current list of column names
					var idx = currentNames.indexOf(col[0]);
	
					if (idx < 0) {
						// If the column was not found ignore it and continue
						return;
					}
	
					set[0] = idx;
				}
				else if (set[0] >= columns.length) {
					// If the column index is out of bounds ignore it and continue
					return;
				}
	
				settings.aaSorting.push(set);
			} );
		}
	
		// Search
		if ( s.search !== undefined ) {
			$.extend( settings.oPreviousSearch, s.search );
		}
	
		// Columns
		if ( s.columns ) {
			var set = s.columns;
			var incoming = _pluck(s.columns, 'name');
	
			// Check if it is a 2.2 style state object with a `name` property for the columns, and if
			// the name was defined. If so, then create a new array that will map the state object
			// given, to the current columns (don't bother if they are already matching tho).
			if (incoming.join('').length && incoming.join('') !== currentNames.join('')) {
				set = [];
	
				// For each column, try to find the name in the incoming array
				for (i=0 ; i<currentNames.length ; i++) {
					if (currentNames[i] != '') {
						var idx = incoming.indexOf(currentNames[i]);
	
						if (idx >= 0) {
							set.push(s.columns[idx]);
						}
						else {
							// No matching column name in the state's columns, so this might be a new
							// column and thus can't have a state already.
							set.push({});
						}
					}
					else {
						// If no name, but other columns did have a name, then there is no knowing
						// where this one came from originally so it can't be restored.
						set.push({});
					}
				}
			}
	
			// If the number of columns to restore is different from current, then all bets are off.
			if (set.length === columns.length) {
				for ( i=0, ien=set.length ; i<ien ; i++ ) {
					var col = set[i];
	
					// Visibility
					if ( col.visible !== undefined ) {
						// If the api is defined, the table has been initialised so we need to use it rather than internal settings
						if (api) {
							// Don't redraw the columns on every iteration of this loop, we will do this at the end instead
							api.column(i).visible(col.visible, false);
						}
						else {
							columns[i].bVisible = col.visible;
						}
					}
	
					// Search
					if ( col.search !== undefined ) {
						$.extend( settings.aoPreSearchCols[i], col.search );
					}
				}
	
				// If the api is defined then we need to adjust the columns once the visibility has been changed
				if (api) {
					api.columns.adjust();
				}
			}
		}
	
		settings._bLoadingState = false;
		_fnCallbackFire( settings, 'aoStateLoaded', 'stateLoaded', [settings, s] );
		callback();
	}
	
	/**
	 * Log an error message
	 *  @param {object} settings dataTables settings object
	 *  @param {int} level log error messages, or display them to the user
	 *  @param {string} msg error message
	 *  @param {int} tn Technical note id to get more information about the error.
	 *  @memberof DataTable#oApi
	 */
	function _fnLog( settings, level, msg, tn )
	{
		msg = 'DataTables warning: '+
			(settings ? 'table id='+settings.sTableId+' - ' : '')+msg;
	
		if ( tn ) {
			msg += '. For more information about this error, please see '+
			'https://datatables.net/tn/'+tn;
		}
	
		if ( ! level  ) {
			// Backwards compatibility pre 1.10
			var ext = DataTable.ext;
			var type = ext.sErrMode || ext.errMode;
	
			if ( settings ) {
				_fnCallbackFire( settings, null, 'dt-error', [ settings, tn, msg ], true );
			}
	
			if ( type == 'alert' ) {
				alert( msg );
			}
			else if ( type == 'throw' ) {
				throw new Error(msg);
			}
			else if ( typeof type == 'function' ) {
				type( settings, tn, msg );
			}
		}
		else if ( window.console && console.log ) {
			console.log( msg );
		}
	}
	
	
	/**
	 * See if a property is defined on one object, if so assign it to the other object
	 *  @param {object} ret target object
	 *  @param {object} src source object
	 *  @param {string} name property
	 *  @param {string} [mappedName] name to map too - optional, name used if not given
	 *  @memberof DataTable#oApi
	 */
	function _fnMap( ret, src, name, mappedName )
	{
		if ( Array.isArray( name ) ) {
			$.each( name, function (i, val) {
				if ( Array.isArray( val ) ) {
					_fnMap( ret, src, val[0], val[1] );
				}
				else {
					_fnMap( ret, src, val );
				}
			} );
	
			return;
		}
	
		if ( mappedName === undefined ) {
			mappedName = name;
		}
	
		if ( src[name] !== undefined ) {
			ret[mappedName] = src[name];
		}
	}
	
	
	/**
	 * Extend objects - very similar to jQuery.extend, but deep copy objects, and
	 * shallow copy arrays. The reason we need to do this, is that we don't want to
	 * deep copy array init values (such as aaSorting) since the dev wouldn't be
	 * able to override them, but we do want to deep copy arrays.
	 *  @param {object} out Object to extend
	 *  @param {object} extender Object from which the properties will be applied to
	 *      out
	 *  @param {boolean} breakRefs If true, then arrays will be sliced to take an
	 *      independent copy with the exception of the `data` or `aaData` parameters
	 *      if they are present. This is so you can pass in a collection to
	 *      DataTables and have that used as your data source without breaking the
	 *      references
	 *  @returns {object} out Reference, just for convenience - out === the return.
	 *  @memberof DataTable#oApi
	 *  @todo This doesn't take account of arrays inside the deep copied objects.
	 */
	function _fnExtend( out, extender, breakRefs )
	{
		var val;
	
		for ( var prop in extender ) {
			if ( Object.prototype.hasOwnProperty.call(extender, prop) ) {
				val = extender[prop];
	
				if ( $.isPlainObject( val ) ) {
					if ( ! $.isPlainObject( out[prop] ) ) {
						out[prop] = {};
					}
					$.extend( true, out[prop], val );
				}
				else if ( breakRefs && prop !== 'data' && prop !== 'aaData' && Array.isArray(val) ) {
					out[prop] = val.slice();
				}
				else {
					out[prop] = val;
				}
			}
		}
	
		return out;
	}
	
	
	/**
	 * Bind an event handers to allow a click or return key to activate the callback.
	 * This is good for accessibility since a return on the keyboard will have the
	 * same effect as a click, if the element has focus.
	 *  @param {element} n Element to bind the action to
	 *  @param {object|string} selector Selector (for delegated events) or data object
	 *   to pass to the triggered function
	 *  @param {function} fn Callback function for when the event is triggered
	 *  @memberof DataTable#oApi
	 */
	function _fnBindAction( n, selector, fn )
	{
		$(n)
			.on( 'click.DT', selector, function (e) {
				fn(e);
			} )
			.on( 'keypress.DT', selector, function (e){
				if ( e.which === 13 ) {
					e.preventDefault();
					fn(e);
				}
			} )
			.on( 'selectstart.DT', selector, function () {
				// Don't want a double click resulting in text selection
				return false;
			} );
	}
	
	
	/**
	 * Register a callback function. Easily allows a callback function to be added to
	 * an array store of callback functions that can then all be called together.
	 *  @param {object} settings dataTables settings object
	 *  @param {string} store Name of the array storage for the callbacks in oSettings
	 *  @param {function} fn Function to be called back
	 *  @memberof DataTable#oApi
	 */
	function _fnCallbackReg( settings, store, fn )
	{
		if ( fn ) {
			settings[store].push(fn);
		}
	}
	
	
	/**
	 * Fire callback functions and trigger events. Note that the loop over the
	 * callback array store is done backwards! Further note that you do not want to
	 * fire off triggers in time sensitive applications (for example cell creation)
	 * as its slow.
	 *  @param {object} settings dataTables settings object
	 *  @param {string} callbackArr Name of the array storage for the callbacks in
	 *      oSettings
	 *  @param {string} eventName Name of the jQuery custom event to trigger. If
	 *      null no trigger is fired
	 *  @param {array} args Array of arguments to pass to the callback function /
	 *      trigger
	 *  @param {boolean} [bubbles] True if the event should bubble
	 *  @memberof DataTable#oApi
	 */
	function _fnCallbackFire( settings, callbackArr, eventName, args, bubbles )
	{
		var ret = [];
	
		if ( callbackArr ) {
			ret = settings[callbackArr].slice().reverse().map( function (val) {
				return val.apply( settings.oInstance, args );
			} );
		}
	
		if ( eventName !== null) {
			var e = $.Event( eventName+'.dt' );
			var table = $(settings.nTable);
			
			// Expose the DataTables API on the event object for easy access
			e.dt = settings.api;
	
			table[bubbles ?  'trigger' : 'triggerHandler']( e, args );
	
			// If not yet attached to the document, trigger the event
			// on the body directly to sort of simulate the bubble
			if (bubbles && table.parents('body').length === 0) {
				$('body').trigger( e, args );
			}
	
			ret.push( e.result );
		}
	
		return ret;
	}
	
	
	function _fnLengthOverflow ( settings )
	{
		var
			start = settings._iDisplayStart,
			end = settings.fnDisplayEnd(),
			len = settings._iDisplayLength;
	
		/* If we have space to show extra rows (backing up from the end point - then do so */
		if ( start >= end )
		{
			start = end - len;
		}
	
		// Keep the start record on the current page
		start -= (start % len);
	
		if ( len === -1 || start < 0 )
		{
			start = 0;
		}
	
		settings._iDisplayStart = start;
	}
	
	
	function _fnRenderer( settings, type )
	{
		var renderer = settings.renderer;
		var host = DataTable.ext.renderer[type];
	
		if ( $.isPlainObject( renderer ) && renderer[type] ) {
			// Specific renderer for this type. If available use it, otherwise use
			// the default.
			return host[renderer[type]] || host._;
		}
		else if ( typeof renderer === 'string' ) {
			// Common renderer - if there is one available for this type use it,
			// otherwise use the default
			return host[renderer] || host._;
		}
	
		// Use the default
		return host._;
	}
	
	
	/**
	 * Detect the data source being used for the table. Used to simplify the code
	 * a little (ajax) and to make it compress a little smaller.
	 *
	 *  @param {object} settings dataTables settings object
	 *  @returns {string} Data source
	 *  @memberof DataTable#oApi
	 */
	function _fnDataSource ( settings )
	{
		if ( settings.oFeatures.bServerSide ) {
			return 'ssp';
		}
		else if ( settings.ajax ) {
			return 'ajax';
		}
		return 'dom';
	}
	
	/**
	 * Common replacement for language strings
	 *
	 * @param {*} settings DT settings object
	 * @param {*} str String with values to replace
	 * @param {*} entries Plural number for _ENTRIES_ - can be undefined
	 * @returns String
	 */
	function _fnMacros ( settings, str, entries )
	{
		// When infinite scrolling, we are always starting at 1. _iDisplayStart is
		// used only internally
		var
			formatter  = settings.fnFormatNumber,
			start      = settings._iDisplayStart+1,
			len        = settings._iDisplayLength,
			vis        = settings.fnRecordsDisplay(),
			max        = settings.fnRecordsTotal(),
			all        = len === -1;
	
		return str.
			replace(/_START_/g, formatter.call( settings, start ) ).
			replace(/_END_/g,   formatter.call( settings, settings.fnDisplayEnd() ) ).
			replace(/_MAX_/g,   formatter.call( settings, max ) ).
			replace(/_TOTAL_/g, formatter.call( settings, vis ) ).
			replace(/_PAGE_/g,  formatter.call( settings, all ? 1 : Math.ceil( start / len ) ) ).
			replace(/_PAGES_/g, formatter.call( settings, all ? 1 : Math.ceil( vis / len ) ) ).
			replace(/_ENTRIES_/g, settings.api.i18n('entries', '', entries) ).
			replace(/_ENTRIES-MAX_/g, settings.api.i18n('entries', '', max) ).
			replace(/_ENTRIES-TOTAL_/g, settings.api.i18n('entries', '', vis) );
	}
	
	/**
	 * Add elements to an array as quickly as possible, but stack stafe.
	 *
	 * @param {*} arr Array to add the data to
	 * @param {*} data Data array that is to be added
	 * @returns 
	 */
	function _fnArrayApply(arr, data) {
		if (! data) {
			return;
		}
	
		// Chrome can throw a max stack error if apply is called with
		// too large an array, but apply is faster.
		if (data.length < 10000) {
			arr.push.apply(arr, data);
		}
		else {
			for (i=0 ; i<data.length ; i++) {
				arr.push(data[i]);
			}
		}
	}
	
	/**
	 * Add one or more listeners to the table
	 *
	 * @param {*} that JQ for the table
	 * @param {*} name Event name
	 * @param {*} src Listener(s)
	 */
	function _fnListener(that, name, src) {
		if (!Array.isArray(src)) {
			src = [src];
		}
	
		for (i=0 ; i<src.length ; i++) {
			that.on(name + '.dt', src[i]);
		}
	}
	
	/**
	 * Escape HTML entities in strings, in an object
	 */
	function _fnEscapeObject(obj) {
		if (DataTable.ext.escape.attributes) {
			$.each(obj, function (key, val) {
				obj[key] = _escapeHtml(val);
			})
		}
	
		return obj;
	}
	
	
	
	/**
	 * Computed structure of the DataTables API, defined by the options passed to
	 * `DataTable.Api.register()` when building the API.
	 *
	 * The structure is built in order to speed creation and extension of the Api
	 * objects since the extensions are effectively pre-parsed.
	 *
	 * The array is an array of objects with the following structure, where this
	 * base array represents the Api prototype base:
	 *
	 *     [
	 *       {
	 *         name:      'data'                -- string   - Property name
	 *         val:       function () {},       -- function - Api method (or undefined if just an object
	 *         methodExt: [ ... ],              -- array    - Array of Api object definitions to extend the method result
	 *         propExt:   [ ... ]               -- array    - Array of Api object definitions to extend the property
	 *       },
	 *       {
	 *         name:     'row'
	 *         val:       {},
	 *         methodExt: [ ... ],
	 *         propExt:   [
	 *           {
	 *             name:      'data'
	 *             val:       function () {},
	 *             methodExt: [ ... ],
	 *             propExt:   [ ... ]
	 *           },
	 *           ...
	 *         ]
	 *       }
	 *     ]
	 *
	 * @type {Array}
	 * @ignore
	 */
	var __apiStruct = [];
	
	
	/**
	 * `Array.prototype` reference.
	 *
	 * @type object
	 * @ignore
	 */
	var __arrayProto = Array.prototype;
	
	
	/**
	 * Abstraction for `context` parameter of the `Api` constructor to allow it to
	 * take several different forms for ease of use.
	 *
	 * Each of the input parameter types will be converted to a DataTables settings
	 * object where possible.
	 *
	 * @param  {string|node|jQuery|object} mixed DataTable identifier. Can be one
	 *   of:
	 *
	 *   * `string` - jQuery selector. Any DataTables' matching the given selector
	 *     with be found and used.
	 *   * `node` - `TABLE` node which has already been formed into a DataTable.
	 *   * `jQuery` - A jQuery object of `TABLE` nodes.
	 *   * `object` - DataTables settings object
	 *   * `DataTables.Api` - API instance
	 * @return {array|null} Matching DataTables settings objects. `null` or
	 *   `undefined` is returned if no matching DataTable is found.
	 * @ignore
	 */
	var _toSettings = function ( mixed )
	{
		var idx, jq;
		var settings = DataTable.settings;
		var tables = _pluck(settings, 'nTable');
	
		if ( ! mixed ) {
			return [];
		}
		else if ( mixed.nTable && mixed.oFeatures ) {
			// DataTables settings object
			return [ mixed ];
		}
		else if ( mixed.nodeName && mixed.nodeName.toLowerCase() === 'table' ) {
			// Table node
			idx = tables.indexOf(mixed);
			return idx !== -1 ? [ settings[idx] ] : null;
		}
		else if ( mixed && typeof mixed.settings === 'function' ) {
			return mixed.settings().toArray();
		}
		else if ( typeof mixed === 'string' ) {
			// jQuery selector
			jq = $(mixed).get();
		}
		else if ( mixed instanceof $ ) {
			// jQuery object (also DataTables instance)
			jq = mixed.get();
		}
	
		if ( jq ) {
			return settings.filter(function (v, idx) {
				return jq.includes(tables[idx]);
			});
		}
	};
	
	
	/**
	 * DataTables API class - used to control and interface with  one or more
	 * DataTables enhanced tables.
	 *
	 * The API class is heavily based on jQuery, presenting a chainable interface
	 * that you can use to interact with tables. Each instance of the API class has
	 * a "context" - i.e. the tables that it will operate on. This could be a single
	 * table, all tables on a page or a sub-set thereof.
	 *
	 * Additionally the API is designed to allow you to easily work with the data in
	 * the tables, retrieving and manipulating it as required. This is done by
	 * presenting the API class as an array like interface. The contents of the
	 * array depend upon the actions requested by each method (for example
	 * `rows().nodes()` will return an array of nodes, while `rows().data()` will
	 * return an array of objects or arrays depending upon your table's
	 * configuration). The API object has a number of array like methods (`push`,
	 * `pop`, `reverse` etc) as well as additional helper methods (`each`, `pluck`,
	 * `unique` etc) to assist your working with the data held in a table.
	 *
	 * Most methods (those which return an Api instance) are chainable, which means
	 * the return from a method call also has all of the methods available that the
	 * top level object had. For example, these two calls are equivalent:
	 *
	 *     // Not chained
	 *     api.row.add( {...} );
	 *     api.draw();
	 *
	 *     // Chained
	 *     api.row.add( {...} ).draw();
	 *
	 * @class DataTable.Api
	 * @param {array|object|string|jQuery} context DataTable identifier. This is
	 *   used to define which DataTables enhanced tables this API will operate on.
	 *   Can be one of:
	 *
	 *   * `string` - jQuery selector. Any DataTables' matching the given selector
	 *     with be found and used.
	 *   * `node` - `TABLE` node which has already been formed into a DataTable.
	 *   * `jQuery` - A jQuery object of `TABLE` nodes.
	 *   * `object` - DataTables settings object
	 * @param {array} [data] Data to initialise the Api instance with.
	 *
	 * @example
	 *   // Direct initialisation during DataTables construction
	 *   var api = $('#example').DataTable();
	 *
	 * @example
	 *   // Initialisation using a DataTables jQuery object
	 *   var api = $('#example').dataTable().api();
	 *
	 * @example
	 *   // Initialisation as a constructor
	 *   var api = new DataTable.Api( 'table.dataTable' );
	 */
	_Api = function ( context, data )
	{
		if ( ! (this instanceof _Api) ) {
			return new _Api( context, data );
		}
	
		var i;
		var settings = [];
		var ctxSettings = function ( o ) {
			var a = _toSettings( o );
			if ( a ) {
				settings.push.apply( settings, a );
			}
		};
	
		if ( Array.isArray( context ) ) {
			for ( i=0 ; i<context.length ; i++ ) {
				ctxSettings( context[i] );
			}
		}
		else {
			ctxSettings( context );
		}
	
		// Remove duplicates
		this.context = settings.length > 1
			? _unique( settings )
			: settings;
	
		// Initial data
		_fnArrayApply(this, data);
	
		// selector
		this.selector = {
			rows: null,
			cols: null,
			opts: null
		};
	
		_Api.extend( this, this, __apiStruct );
	};
	
	DataTable.Api = _Api;
	
	// Don't destroy the existing prototype, just extend it. Required for jQuery 2's
	// isPlainObject.
	$.extend( _Api.prototype, {
		any: function ()
		{
			return this.count() !== 0;
		},
	
		context: [], // array of table settings objects
	
		count: function ()
		{
			return this.flatten().length;
		},
	
		each: function ( fn )
		{
			for ( var i=0, ien=this.length ; i<ien; i++ ) {
				fn.call( this, this[i], i, this );
			}
	
			return this;
		},
	
		eq: function ( idx )
		{
			var ctx = this.context;
	
			return ctx.length > idx ?
				new _Api( ctx[idx], this[idx] ) :
				null;
		},
	
		filter: function ( fn )
		{
			var a = __arrayProto.filter.call( this, fn, this );
	
			return new _Api( this.context, a );
		},
	
		flatten: function ()
		{
			var a = [];
	
			return new _Api( this.context, a.concat.apply( a, this.toArray() ) );
		},
	
		get: function ( idx )
		{
			return this[ idx ];
		},
	
		join:    __arrayProto.join,
	
		includes: function ( find ) {
			return this.indexOf( find ) === -1 ? false : true;
		},
	
		indexOf: __arrayProto.indexOf,
	
		iterator: function ( flatten, type, fn, alwaysNew ) {
			var
				a = [], ret,
				i, ien, j, jen,
				context = this.context,
				rows, items, item,
				selector = this.selector;
	
			// Argument shifting
			if ( typeof flatten === 'string' ) {
				alwaysNew = fn;
				fn = type;
				type = flatten;
				flatten = false;
			}
	
			for ( i=0, ien=context.length ; i<ien ; i++ ) {
				var apiInst = new _Api( context[i] );
	
				if ( type === 'table' ) {
					ret = fn.call( apiInst, context[i], i );
	
					if ( ret !== undefined ) {
						a.push( ret );
					}
				}
				else if ( type === 'columns' || type === 'rows' ) {
					// this has same length as context - one entry for each table
					ret = fn.call( apiInst, context[i], this[i], i );
	
					if ( ret !== undefined ) {
						a.push( ret );
					}
				}
				else if ( type === 'every' || type === 'column' || type === 'column-rows' || type === 'row' || type === 'cell' ) {
					// columns and rows share the same structure.
					// 'this' is an array of column indexes for each context
					items = this[i];
	
					if ( type === 'column-rows' ) {
						rows = _selector_row_indexes( context[i], selector.opts );
					}
	
					for ( j=0, jen=items.length ; j<jen ; j++ ) {
						item = items[j];
	
						if ( type === 'cell' ) {
							ret = fn.call( apiInst, context[i], item.row, item.column, i, j );
						}
						else {
							ret = fn.call( apiInst, context[i], item, i, j, rows );
						}
	
						if ( ret !== undefined ) {
							a.push( ret );
						}
					}
				}
			}
	
			if ( a.length || alwaysNew ) {
				var api = new _Api( context, flatten ? a.concat.apply( [], a ) : a );
				var apiSelector = api.selector;
				apiSelector.rows = selector.rows;
				apiSelector.cols = selector.cols;
				apiSelector.opts = selector.opts;
				return api;
			}
			return this;
		},
	
		lastIndexOf: __arrayProto.lastIndexOf,
	
		length:  0,
	
		map: function ( fn )
		{
			var a = __arrayProto.map.call( this, fn, this );
	
			return new _Api( this.context, a );
		},
	
		pluck: function ( prop )
		{
			var fn = DataTable.util.get(prop);
	
			return this.map( function ( el ) {
				return fn(el);
			} );
		},
	
		pop:     __arrayProto.pop,
	
		push:    __arrayProto.push,
	
		reduce: __arrayProto.reduce,
	
		reduceRight: __arrayProto.reduceRight,
	
		reverse: __arrayProto.reverse,
	
		// Object with rows, columns and opts
		selector: null,
	
		shift:   __arrayProto.shift,
	
		slice: function () {
			return new _Api( this.context, this );
		},
	
		sort:    __arrayProto.sort,
	
		splice:  __arrayProto.splice,
	
		toArray: function ()
		{
			return __arrayProto.slice.call( this );
		},
	
		to$: function ()
		{
			return $( this );
		},
	
		toJQuery: function ()
		{
			return $( this );
		},
	
		unique: function ()
		{
			return new _Api( this.context, _unique(this.toArray()) );
		},
	
		unshift: __arrayProto.unshift
	} );
	
	
	function _api_scope( scope, fn, struc ) {
		return function () {
			var ret = fn.apply( scope || this, arguments );
	
			// Method extension
			_Api.extend( ret, ret, struc.methodExt );
			return ret;
		};
	}
	
	function _api_find( src, name ) {
		for ( var i=0, ien=src.length ; i<ien ; i++ ) {
			if ( src[i].name === name ) {
				return src[i];
			}
		}
		return null;
	}
	
	window.__apiStruct = __apiStruct;
	
	_Api.extend = function ( scope, obj, ext )
	{
		// Only extend API instances and static properties of the API
		if ( ! ext.length || ! obj || ( ! (obj instanceof _Api) && ! obj.__dt_wrapper ) ) {
			return;
		}
	
		var
			i, ien,
			struct;
	
		for ( i=0, ien=ext.length ; i<ien ; i++ ) {
			struct = ext[i];
	
			if (struct.name === '__proto__') {
				continue;
			}
	
			// Value
			obj[ struct.name ] = struct.type === 'function' ?
				_api_scope( scope, struct.val, struct ) :
				struct.type === 'object' ?
					{} :
					struct.val;
	
			obj[ struct.name ].__dt_wrapper = true;
	
			// Property extension
			_Api.extend( scope, obj[ struct.name ], struct.propExt );
		}
	};
	
	//     [
	//       {
	//         name:      'data'                -- string   - Property name
	//         val:       function () {},       -- function - Api method (or undefined if just an object
	//         methodExt: [ ... ],              -- array    - Array of Api object definitions to extend the method result
	//         propExt:   [ ... ]               -- array    - Array of Api object definitions to extend the property
	//       },
	//       {
	//         name:     'row'
	//         val:       {},
	//         methodExt: [ ... ],
	//         propExt:   [
	//           {
	//             name:      'data'
	//             val:       function () {},
	//             methodExt: [ ... ],
	//             propExt:   [ ... ]
	//           },
	//           ...
	//         ]
	//       }
	//     ]
	
	
	_Api.register = _api_register = function ( name, val )
	{
		if ( Array.isArray( name ) ) {
			for ( var j=0, jen=name.length ; j<jen ; j++ ) {
				_Api.register( name[j], val );
			}
			return;
		}
	
		var
			i, ien,
			heir = name.split('.'),
			struct = __apiStruct,
			key, method;
	
		for ( i=0, ien=heir.length ; i<ien ; i++ ) {
			method = heir[i].indexOf('()') !== -1;
			key = method ?
				heir[i].replace('()', '') :
				heir[i];
	
			var src = _api_find( struct, key );
			if ( ! src ) {
				src = {
					name:      key,
					val:       {},
					methodExt: [],
					propExt:   [],
					type:      'object'
				};
				struct.push( src );
			}
	
			if ( i === ien-1 ) {
				src.val = val;
				src.type = typeof val === 'function' ?
					'function' :
					$.isPlainObject( val ) ?
						'object' :
						'other';
			}
			else {
				struct = method ?
					src.methodExt :
					src.propExt;
			}
		}
	};
	
	_Api.registerPlural = _api_registerPlural = function ( pluralName, singularName, val ) {
		_Api.register( pluralName, val );
	
		_Api.register( singularName, function () {
			var ret = val.apply( this, arguments );
	
			if ( ret === this ) {
				// Returned item is the API instance that was passed in, return it
				return this;
			}
			else if ( ret instanceof _Api ) {
				// New API instance returned, want the value from the first item
				// in the returned array for the singular result.
				return ret.length ?
					Array.isArray( ret[0] ) ?
						new _Api( ret.context, ret[0] ) : // Array results are 'enhanced'
						ret[0] :
					undefined;
			}
	
			// Non-API return - just fire it back
			return ret;
		} );
	};
	
	
	/**
	 * Selector for HTML tables. Apply the given selector to the give array of
	 * DataTables settings objects.
	 *
	 * @param {string|integer} [selector] jQuery selector string or integer
	 * @param  {array} Array of DataTables settings objects to be filtered
	 * @return {array}
	 * @ignore
	 */
	var __table_selector = function ( selector, a )
	{
		if ( Array.isArray(selector) ) {
			var result = [];
	
			selector.forEach(function (sel) {
				var inner = __table_selector(sel, a);
	
				_fnArrayApply(result, inner);
			});
	
			return result.filter( function (item) {
				return item;
			});
		}
	
		// Integer is used to pick out a table by index
		if ( typeof selector === 'number' ) {
			return [ a[ selector ] ];
		}
	
		// Perform a jQuery selector on the table nodes
		var nodes = a.map( function (el) {
			return el.nTable;
		} );
	
		return $(nodes)
			.filter( selector )
			.map( function () {
				// Need to translate back from the table node to the settings
				var idx = nodes.indexOf(this);
				return a[ idx ];
			} )
			.toArray();
	};
	
	
	
	/**
	 * Context selector for the API's context (i.e. the tables the API instance
	 * refers to.
	 *
	 * @name    DataTable.Api#tables
	 * @param {string|integer} [selector] Selector to pick which tables the iterator
	 *   should operate on. If not given, all tables in the current context are
	 *   used. This can be given as a jQuery selector (for example `':gt(0)'`) to
	 *   select multiple tables or as an integer to select a single table.
	 * @returns {DataTable.Api} Returns a new API instance if a selector is given.
	 */
	_api_register( 'tables()', function ( selector ) {
		// A new instance is created if there was a selector specified
		return selector !== undefined && selector !== null ?
			new _Api( __table_selector( selector, this.context ) ) :
			this;
	} );
	
	
	_api_register( 'table()', function ( selector ) {
		var tables = this.tables( selector );
		var ctx = tables.context;
	
		// Truncate to the first matched table
		return ctx.length ?
			new _Api( ctx[0] ) :
			tables;
	} );
	
	// Common methods, combined to reduce size
	[
		['nodes', 'node', 'nTable'],
		['body', 'body', 'nTBody'],
		['header', 'header', 'nTHead'],
		['footer', 'footer', 'nTFoot'],
	].forEach(function (item) {
		_api_registerPlural(
			'tables().' + item[0] + '()',
			'table().' + item[1] + '()' ,
			function () {
				return this.iterator( 'table', function ( ctx ) {
					return ctx[item[2]];
				}, 1 );
			}
		);
	});
	
	// Structure methods
	[
		['header', 'aoHeader'],
		['footer', 'aoFooter'],
	].forEach(function (item) {
		_api_register( 'table().' + item[0] + '.structure()' , function (selector) {
			var indexes = this.columns(selector).indexes().flatten().toArray();
			var ctx = this.context[0];
			var structure = _fnHeaderLayout(ctx, ctx[item[1]], indexes);
	
			// The structure is in column index order - but from this method we want the return to be
			// in the columns() selector API order. In order to do that we need to map from one form
			// to the other
			var orderedIndexes = indexes.slice().sort(function (a, b) {
				return a - b;
			});
	
			return structure.map(function (row) {
				return indexes.map(function (colIdx) {
					return row[orderedIndexes.indexOf(colIdx)];
				});
			});
		});
	});
	
	
	_api_registerPlural( 'tables().containers()', 'table().container()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTableWrapper;
		}, 1 );
	} );
	
	_api_register( 'tables().every()', function ( fn ) {
		var that = this;
	
		return this.iterator('table', function (s, i) {
			fn.call(that.table(i), i);
		});
	});
	
	_api_register( 'caption()', function ( value, side ) {
		var context = this.context;
	
		// Getter - return existing node's content
		if ( value === undefined ) {
			var caption = context[0].captionNode;
	
			return caption && context.length ?
				caption.innerHTML : 
				null;
		}
	
		return this.iterator( 'table', function ( ctx ) {
			var table = $(ctx.nTable);
			var caption = $(ctx.captionNode);
			var container = $(ctx.nTableWrapper);
	
			// Create the node if it doesn't exist yet
			if ( ! caption.length ) {
				caption = $('<caption/>').html( value );
				ctx.captionNode = caption[0];
	
				// If side isn't set, we need to insert into the document to let the
				// CSS decide so we can read it back, otherwise there is no way to
				// know if the CSS would put it top or bottom for scrolling
				if (! side) {
					table.prepend(caption);
	
					side = caption.css('caption-side');
				}
			}
	
			caption.html( value );
	
			if ( side ) {
				caption.css( 'caption-side', side );
				caption[0]._captionSide = side;
			}
	
			if (container.find('div.dataTables_scroll').length) {
				var selector = (side === 'top' ? 'Head' : 'Foot');
	
				container.find('div.dataTables_scroll'+ selector +' table').prepend(caption);
			}
			else {
				table.prepend(caption);
			}
		}, 1 );
	} );
	
	_api_register( 'caption.node()', function () {
		var ctx = this.context;
	
		return ctx.length ? ctx[0].captionNode : null;
	} );
	
	
	/**
	 * Redraw the tables in the current context.
	 */
	_api_register( 'draw()', function ( paging ) {
		return this.iterator( 'table', function ( settings ) {
			if ( paging === 'page' ) {
				_fnDraw( settings );
			}
			else {
				if ( typeof paging === 'string' ) {
					paging = paging === 'full-hold' ?
						false :
						true;
				}
	
				_fnReDraw( settings, paging===false );
			}
		} );
	} );
	
	
	
	/**
	 * Get the current page index.
	 *
	 * @return {integer} Current page index (zero based)
	 *//**
	 * Set the current page.
	 *
	 * Note that if you attempt to show a page which does not exist, DataTables will
	 * not throw an error, but rather reset the paging.
	 *
	 * @param {integer|string} action The paging action to take. This can be one of:
	 *  * `integer` - The page index to jump to
	 *  * `string` - An action to take:
	 *    * `first` - Jump to first page.
	 *    * `next` - Jump to the next page
	 *    * `previous` - Jump to previous page
	 *    * `last` - Jump to the last page.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'page()', function ( action ) {
		if ( action === undefined ) {
			return this.page.info().page; // not an expensive call
		}
	
		// else, have an action to take on all tables
		return this.iterator( 'table', function ( settings ) {
			_fnPageChange( settings, action );
		} );
	} );
	
	
	/**
	 * Paging information for the first table in the current context.
	 *
	 * If you require paging information for another table, use the `table()` method
	 * with a suitable selector.
	 *
	 * @return {object} Object with the following properties set:
	 *  * `page` - Current page index (zero based - i.e. the first page is `0`)
	 *  * `pages` - Total number of pages
	 *  * `start` - Display index for the first record shown on the current page
	 *  * `end` - Display index for the last record shown on the current page
	 *  * `length` - Display length (number of records). Note that generally `start
	 *    + length = end`, but this is not always true, for example if there are
	 *    only 2 records to show on the final page, with a length of 10.
	 *  * `recordsTotal` - Full data set length
	 *  * `recordsDisplay` - Data set length once the current filtering criterion
	 *    are applied.
	 */
	_api_register( 'page.info()', function () {
		if ( this.context.length === 0 ) {
			return undefined;
		}
	
		var
			settings   = this.context[0],
			start      = settings._iDisplayStart,
			len        = settings.oFeatures.bPaginate ? settings._iDisplayLength : -1,
			visRecords = settings.fnRecordsDisplay(),
			all        = len === -1;
	
		return {
			"page":           all ? 0 : Math.floor( start / len ),
			"pages":          all ? 1 : Math.ceil( visRecords / len ),
			"start":          start,
			"end":            settings.fnDisplayEnd(),
			"length":         len,
			"recordsTotal":   settings.fnRecordsTotal(),
			"recordsDisplay": visRecords,
			"serverSide":     _fnDataSource( settings ) === 'ssp'
		};
	} );
	
	
	/**
	 * Get the current page length.
	 *
	 * @return {integer} Current page length. Note `-1` indicates that all records
	 *   are to be shown.
	 *//**
	 * Set the current page length.
	 *
	 * @param {integer} Page length to set. Use `-1` to show all records.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'page.len()', function ( len ) {
		// Note that we can't call this function 'length()' because `length`
		// is a Javascript property of functions which defines how many arguments
		// the function expects.
		if ( len === undefined ) {
			return this.context.length !== 0 ?
				this.context[0]._iDisplayLength :
				undefined;
		}
	
		// else, set the page length
		return this.iterator( 'table', function ( settings ) {
			_fnLengthChange( settings, len );
		} );
	} );
	
	
	
	var __reload = function ( settings, holdPosition, callback ) {
		// Use the draw event to trigger a callback
		if ( callback ) {
			var api = new _Api( settings );
	
			api.one( 'draw', function () {
				callback( api.ajax.json() );
			} );
		}
	
		if ( _fnDataSource( settings ) == 'ssp' ) {
			_fnReDraw( settings, holdPosition );
		}
		else {
			_fnProcessingDisplay( settings, true );
	
			// Cancel an existing request
			var xhr = settings.jqXHR;
			if ( xhr && xhr.readyState !== 4 ) {
				xhr.abort();
			}
	
			// Trigger xhr
			_fnBuildAjax( settings, {}, function( json ) {
				_fnClearTable( settings );
	
				var data = _fnAjaxDataSrc( settings, json );
				for ( var i=0, ien=data.length ; i<ien ; i++ ) {
					_fnAddData( settings, data[i] );
				}
	
				_fnReDraw( settings, holdPosition );
				_fnInitComplete( settings );
				_fnProcessingDisplay( settings, false );
			} );
		}
	};
	
	
	/**
	 * Get the JSON response from the last Ajax request that DataTables made to the
	 * server. Note that this returns the JSON from the first table in the current
	 * context.
	 *
	 * @return {object} JSON received from the server.
	 */
	_api_register( 'ajax.json()', function () {
		var ctx = this.context;
	
		if ( ctx.length > 0 ) {
			return ctx[0].json;
		}
	
		// else return undefined;
	} );
	
	
	/**
	 * Get the data submitted in the last Ajax request
	 */
	_api_register( 'ajax.params()', function () {
		var ctx = this.context;
	
		if ( ctx.length > 0 ) {
			return ctx[0].oAjaxData;
		}
	
		// else return undefined;
	} );
	
	
	/**
	 * Reload tables from the Ajax data source. Note that this function will
	 * automatically re-draw the table when the remote data has been loaded.
	 *
	 * @param {boolean} [reset=true] Reset (default) or hold the current paging
	 *   position. A full re-sort and re-filter is performed when this method is
	 *   called, which is why the pagination reset is the default action.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'ajax.reload()', function ( callback, resetPaging ) {
		return this.iterator( 'table', function (settings) {
			__reload( settings, resetPaging===false, callback );
		} );
	} );
	
	
	/**
	 * Get the current Ajax URL. Note that this returns the URL from the first
	 * table in the current context.
	 *
	 * @return {string} Current Ajax source URL
	 *//**
	 * Set the Ajax URL. Note that this will set the URL for all tables in the
	 * current context.
	 *
	 * @param {string} url URL to set.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'ajax.url()', function ( url ) {
		var ctx = this.context;
	
		if ( url === undefined ) {
			// get
			if ( ctx.length === 0 ) {
				return undefined;
			}
			ctx = ctx[0];
	
			return $.isPlainObject( ctx.ajax ) ?
				ctx.ajax.url :
				ctx.ajax;
		}
	
		// set
		return this.iterator( 'table', function ( settings ) {
			if ( $.isPlainObject( settings.ajax ) ) {
				settings.ajax.url = url;
			}
			else {
				settings.ajax = url;
			}
		} );
	} );
	
	
	/**
	 * Load data from the newly set Ajax URL. Note that this method is only
	 * available when `ajax.url()` is used to set a URL. Additionally, this method
	 * has the same effect as calling `ajax.reload()` but is provided for
	 * convenience when setting a new URL. Like `ajax.reload()` it will
	 * automatically redraw the table once the remote data has been loaded.
	 *
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'ajax.url().load()', function ( callback, resetPaging ) {
		// Same as a reload, but makes sense to present it for easy access after a
		// url change
		return this.iterator( 'table', function ( ctx ) {
			__reload( ctx, resetPaging===false, callback );
		} );
	} );
	
	
	
	
	var _selector_run = function ( type, selector, selectFn, settings, opts )
	{
		var
			out = [], res,
			i, ien,
			selectorType = typeof selector;
	
		// Can't just check for isArray here, as an API or jQuery instance might be
		// given with their array like look
		if ( ! selector || selectorType === 'string' || selectorType === 'function' || selector.length === undefined ) {
			selector = [ selector ];
		}
	
		for ( i=0, ien=selector.length ; i<ien ; i++ ) {
			res = selectFn( typeof selector[i] === 'string' ? selector[i].trim() : selector[i] );
	
			// Remove empty items
			res = res.filter( function (item) {
				return item !== null && item !== undefined;
			});
	
			if ( res && res.length ) {
				out = out.concat( res );
			}
		}
	
		// selector extensions
		var ext = _ext.selector[ type ];
		if ( ext.length ) {
			for ( i=0, ien=ext.length ; i<ien ; i++ ) {
				out = ext[i]( settings, opts, out );
			}
		}
	
		return _unique( out );
	};
	
	
	var _selector_opts = function ( opts )
	{
		if ( ! opts ) {
			opts = {};
		}
	
		// Backwards compatibility for 1.9- which used the terminology filter rather
		// than search
		if ( opts.filter && opts.search === undefined ) {
			opts.search = opts.filter;
		}
	
		return $.extend( {
			columnOrder: 'implied',
			search: 'none',
			order: 'current',
			page: 'all'
		}, opts );
	};
	
	
	// Reduce the API instance to the first item found
	var _selector_first = function ( old )
	{
		var inst = new _Api(old.context[0]);
	
		// Use a push rather than passing to the constructor, since it will
		// merge arrays down automatically, which isn't what is wanted here
		if (old.length) {
			inst.push( old[0] );
		}
	
		inst.selector = old.selector;
	
		// Limit to a single row / column / cell
		if (inst.length && inst[0].length > 1) {
			inst[0].splice(1);
		}
	
		return inst;
	};
	
	
	var _selector_row_indexes = function ( settings, opts )
	{
		var
			i, ien, tmp, a=[],
			displayFiltered = settings.aiDisplay,
			displayMaster = settings.aiDisplayMaster;
	
		var
			search = opts.search,  // none, applied, removed
			order  = opts.order,   // applied, current, index (original - compatibility with 1.9)
			page   = opts.page;    // all, current
	
		if ( _fnDataSource( settings ) == 'ssp' ) {
			// In server-side processing mode, most options are irrelevant since
			// rows not shown don't exist and the index order is the applied order
			// Removed is a special case - for consistency just return an empty
			// array
			return search === 'removed' ?
				[] :
				_range( 0, displayMaster.length );
		}
	
		if ( page == 'current' ) {
			// Current page implies that order=current and filter=applied, since it is
			// fairly senseless otherwise, regardless of what order and search actually
			// are
			for ( i=settings._iDisplayStart, ien=settings.fnDisplayEnd() ; i<ien ; i++ ) {
				a.push( displayFiltered[i] );
			}
		}
		else if ( order == 'current' || order == 'applied' ) {
			if ( search == 'none') {
				a = displayMaster.slice();
			}
			else if ( search == 'applied' ) {
				a = displayFiltered.slice();
			}
			else if ( search == 'removed' ) {
				// O(n+m) solution by creating a hash map
				var displayFilteredMap = {};
	
				for ( i=0, ien=displayFiltered.length ; i<ien ; i++ ) {
					displayFilteredMap[displayFiltered[i]] = null;
				}
	
				displayMaster.forEach(function (item) {
					if (! Object.prototype.hasOwnProperty.call(displayFilteredMap, item)) {
						a.push(item);
					}
				});
			}
		}
		else if ( order == 'index' || order == 'original' ) {
			for ( i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
				if (! settings.aoData[i]) {
					continue;
				}
	
				if ( search == 'none' ) {
					a.push( i );
				}
				else { // applied | removed
					tmp = displayFiltered.indexOf(i);
	
					if ((tmp === -1 && search == 'removed') ||
						(tmp >= 0   && search == 'applied') )
					{
						a.push( i );
					}
				}
			}
		}
		else if ( typeof order === 'number' ) {
			// Order the rows by the given column
			var ordered = _fnSort(settings, order, 'asc');
	
			if (search === 'none') {
				a = ordered;
			}
			else { // applied | removed
				for (i=0; i<ordered.length; i++) {
					tmp = displayFiltered.indexOf(ordered[i]);
	
					if ((tmp === -1 && search == 'removed') ||
						(tmp >= 0   && search == 'applied') )
					{
						a.push( ordered[i] );
					}
				}
			}
		}
	
		return a;
	};
	
	
	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 * Rows
	 *
	 * {}          - no selector - use all available rows
	 * {integer}   - row aoData index
	 * {node}      - TR node
	 * {string}    - jQuery selector to apply to the TR elements
	 * {array}     - jQuery array of nodes, or simply an array of TR nodes
	 *
	 */
	var __row_selector = function ( settings, selector, opts )
	{
		var rows;
		var run = function ( sel ) {
			var selInt = _intVal( sel );
			var aoData = settings.aoData;
	
			// Short cut - selector is a number and no options provided (default is
			// all records, so no need to check if the index is in there, since it
			// must be - dev error if the index doesn't exist).
			if ( selInt !== null && ! opts ) {
				return [ selInt ];
			}
	
			if ( ! rows ) {
				rows = _selector_row_indexes( settings, opts );
			}
	
			if ( selInt !== null && rows.indexOf(selInt) !== -1 ) {
				// Selector - integer
				return [ selInt ];
			}
			else if ( sel === null || sel === undefined || sel === '' ) {
				// Selector - none
				return rows;
			}
	
			// Selector - function
			if ( typeof sel === 'function' ) {
				return rows.map( function (idx) {
					var row = aoData[ idx ];
					return sel( idx, row._aData, row.nTr ) ? idx : null;
				} );
			}
	
			// Selector - node
			if ( sel.nodeName ) {
				var rowIdx = sel._DT_RowIndex;  // Property added by DT for fast lookup
				var cellIdx = sel._DT_CellIndex;
	
				if ( rowIdx !== undefined ) {
					// Make sure that the row is actually still present in the table
					return aoData[ rowIdx ] && aoData[ rowIdx ].nTr === sel ?
						[ rowIdx ] :
						[];
				}
				else if ( cellIdx ) {
					return aoData[ cellIdx.row ] && aoData[ cellIdx.row ].nTr === sel.parentNode ?
						[ cellIdx.row ] :
						[];
				}
				else {
					var host = $(sel).closest('*[data-dt-row]');
					return host.length ?
						[ host.data('dt-row') ] :
						[];
				}
			}
	
			// ID selector. Want to always be able to select rows by id, regardless
			// of if the tr element has been created or not, so can't rely upon
			// jQuery here - hence a custom implementation. This does not match
			// Sizzle's fast selector or HTML4 - in HTML5 the ID can be anything,
			// but to select it using a CSS selector engine (like Sizzle or
			// querySelect) it would need to need to be escaped for some characters.
			// DataTables simplifies this for row selectors since you can select
			// only a row. A # indicates an id any anything that follows is the id -
			// unescaped.
			if ( typeof sel === 'string' && sel.charAt(0) === '#' ) {
				// get row index from id
				var rowObj = settings.aIds[ sel.replace( /^#/, '' ) ];
				if ( rowObj !== undefined ) {
					return [ rowObj.idx ];
				}
	
				// need to fall through to jQuery in case there is DOM id that
				// matches
			}
			
			// Get nodes in the order from the `rows` array with null values removed
			var nodes = _removeEmpty(
				_pluck_order( settings.aoData, rows, 'nTr' )
			);
	
			// Selector - jQuery selector string, array of nodes or jQuery object/
			// As jQuery's .filter() allows jQuery objects to be passed in filter,
			// it also allows arrays, so this will cope with all three options
			return $(nodes)
				.filter( sel )
				.map( function () {
					return this._DT_RowIndex;
				} )
				.toArray();
		};
	
		var matched = _selector_run( 'row', selector, run, settings, opts );
	
		if (opts.order === 'current' || opts.order === 'applied') {
			_fnSortDisplay(settings, matched);
		}
	
		return matched;
	};
	
	
	_api_register( 'rows()', function ( selector, opts ) {
		// argument shifting
		if ( selector === undefined ) {
			selector = '';
		}
		else if ( $.isPlainObject( selector ) ) {
			opts = selector;
			selector = '';
		}
	
		opts = _selector_opts( opts );
	
		var inst = this.iterator( 'table', function ( settings ) {
			return __row_selector( settings, selector, opts );
		}, 1 );
	
		// Want argument shifting here and in __row_selector?
		inst.selector.rows = selector;
		inst.selector.opts = opts;
	
		return inst;
	} );
	
	_api_register( 'rows().nodes()', function () {
		return this.iterator( 'row', function ( settings, row ) {
			return settings.aoData[ row ].nTr || undefined;
		}, 1 );
	} );
	
	_api_register( 'rows().data()', function () {
		return this.iterator( true, 'rows', function ( settings, rows ) {
			return _pluck_order( settings.aoData, rows, '_aData' );
		}, 1 );
	} );
	
	_api_registerPlural( 'rows().cache()', 'row().cache()', function ( type ) {
		return this.iterator( 'row', function ( settings, row ) {
			var r = settings.aoData[ row ];
			return type === 'search' ? r._aFilterData : r._aSortData;
		}, 1 );
	} );
	
	_api_registerPlural( 'rows().invalidate()', 'row().invalidate()', function ( src ) {
		return this.iterator( 'row', function ( settings, row ) {
			_fnInvalidate( settings, row, src );
		} );
	} );
	
	_api_registerPlural( 'rows().indexes()', 'row().index()', function () {
		return this.iterator( 'row', function ( settings, row ) {
			return row;
		}, 1 );
	} );
	
	_api_registerPlural( 'rows().ids()', 'row().id()', function ( hash ) {
		var a = [];
		var context = this.context;
	
		// `iterator` will drop undefined values, but in this case we want them
		for ( var i=0, ien=context.length ; i<ien ; i++ ) {
			for ( var j=0, jen=this[i].length ; j<jen ; j++ ) {
				var id = context[i].rowIdFn( context[i].aoData[ this[i][j] ]._aData );
				a.push( (hash === true ? '#' : '' )+ id );
			}
		}
	
		return new _Api( context, a );
	} );
	
	_api_registerPlural( 'rows().remove()', 'row().remove()', function () {
		this.iterator( 'row', function ( settings, row ) {
			var data = settings.aoData;
			var rowData = data[ row ];
	
			// Delete from the display arrays
			var idx = settings.aiDisplayMaster.indexOf(row);
			if (idx !== -1) {
				settings.aiDisplayMaster.splice(idx, 1);
			}
	
			// For server-side processing tables - subtract the deleted row from the count
			if ( settings._iRecordsDisplay > 0 ) {
				settings._iRecordsDisplay--;
			}
	
			// Check for an 'overflow' they case for displaying the table
			_fnLengthOverflow( settings );
	
			// Remove the row's ID reference if there is one
			var id = settings.rowIdFn( rowData._aData );
			if ( id !== undefined ) {
				delete settings.aIds[ id ];
			}
	
			data[row] = null;
		} );
	
		return this;
	} );
	
	
	_api_register( 'rows.add()', function ( rows ) {
		var newRows = this.iterator( 'table', function ( settings ) {
				var row, i, ien;
				var out = [];
	
				for ( i=0, ien=rows.length ; i<ien ; i++ ) {
					row = rows[i];
	
					if ( row.nodeName && row.nodeName.toUpperCase() === 'TR' ) {
						out.push( _fnAddTr( settings, row )[0] );
					}
					else {
						out.push( _fnAddData( settings, row ) );
					}
				}
	
				return out;
			}, 1 );
	
		// Return an Api.rows() extended instance, so rows().nodes() etc can be used
		var modRows = this.rows( -1 );
		modRows.pop();
		_fnArrayApply(modRows, newRows);
	
		return modRows;
	} );
	
	
	
	
	
	/**
	 *
	 */
	_api_register( 'row()', function ( selector, opts ) {
		return _selector_first( this.rows( selector, opts ) );
	} );
	
	
	_api_register( 'row().data()', function ( data ) {
		var ctx = this.context;
	
		if ( data === undefined ) {
			// Get
			return ctx.length && this.length && this[0].length ?
				ctx[0].aoData[ this[0] ]._aData :
				undefined;
		}
	
		// Set
		var row = ctx[0].aoData[ this[0] ];
		row._aData = data;
	
		// If the DOM has an id, and the data source is an array
		if ( Array.isArray( data ) && row.nTr && row.nTr.id ) {
			_fnSetObjectDataFn( ctx[0].rowId )( data, row.nTr.id );
		}
	
		// Automatically invalidate
		_fnInvalidate( ctx[0], this[0], 'data' );
	
		return this;
	} );
	
	
	_api_register( 'row().node()', function () {
		var ctx = this.context;
	
		if (ctx.length && this.length && this[0].length) {
			var row = ctx[0].aoData[ this[0] ];
	
			if (row && row.nTr) {
				return row.nTr;
			}
		}
	
		return null;
	} );
	
	
	_api_register( 'row.add()', function ( row ) {
		// Allow a jQuery object to be passed in - only a single row is added from
		// it though - the first element in the set
		if ( row instanceof $ && row.length ) {
			row = row[0];
		}
	
		var rows = this.iterator( 'table', function ( settings ) {
			if ( row.nodeName && row.nodeName.toUpperCase() === 'TR' ) {
				return _fnAddTr( settings, row )[0];
			}
			return _fnAddData( settings, row );
		} );
	
		// Return an Api.rows() extended instance, with the newly added row selected
		return this.row( rows[0] );
	} );
	
	
	$(document).on('plugin-init.dt', function (e, context) {
		var api = new _Api( context );
	
		api.on( 'stateSaveParams.DT', function ( e, settings, d ) {
			// This could be more compact with the API, but it is a lot faster as a simple
			// internal loop
			var idFn = settings.rowIdFn;
			var rows = settings.aiDisplayMaster;
			var ids = [];
	
			for (var i=0 ; i<rows.length ; i++) {
				var rowIdx = rows[i];
				var data = settings.aoData[rowIdx];
	
				if (data._detailsShow) {
					ids.push( '#' + idFn(data._aData) );
				}
			}
	
			d.childRows = ids;
		});
	
		// For future state loads (e.g. with StateRestore)
		api.on( 'stateLoaded.DT', function (e, settings, state) {
			__details_state_load( api, state );
		});
	
		// And the initial load state
		__details_state_load( api, api.state.loaded() );
	});
	
	var __details_state_load = function (api, state)
	{
		if ( state && state.childRows ) {
			api
				.rows( state.childRows.map(function (id) {
					// Escape any `:` characters from the row id. Accounts for
					// already escaped characters.
					return id.replace(/([^:\\]*(?:\\.[^:\\]*)*):/g, "$1\\:");
				}) )
				.every( function () {
					_fnCallbackFire( api.settings()[0], null, 'requestChild', [ this ] )
				});
		}
	}
	
	var __details_add = function ( ctx, row, data, klass )
	{
		// Convert to array of TR elements
		var rows = [];
		var addRow = function ( r, k ) {
			// Recursion to allow for arrays of jQuery objects
			if ( Array.isArray( r ) || r instanceof $ ) {
				for ( var i=0, ien=r.length ; i<ien ; i++ ) {
					addRow( r[i], k );
				}
				return;
			}
	
			// If we get a TR element, then just add it directly - up to the dev
			// to add the correct number of columns etc
			if ( r.nodeName && r.nodeName.toLowerCase() === 'tr' ) {
				r.setAttribute( 'data-dt-row', row.idx );
				rows.push( r );
			}
			else {
				// Otherwise create a row with a wrapper
				var created = $('<tr><td></td></tr>')
					.attr( 'data-dt-row', row.idx )
					.addClass( k );
				
				$('td', created)
					.addClass( k )
					.html( r )[0].colSpan = _fnVisbleColumns( ctx );
	
				rows.push( created[0] );
			}
		};
	
		addRow( data, klass );
	
		if ( row._details ) {
			row._details.detach();
		}
	
		row._details = $(rows);
	
		// If the children were already shown, that state should be retained
		if ( row._detailsShow ) {
			row._details.insertAfter( row.nTr );
		}
	};
	
	
	// Make state saving of child row details async to allow them to be batch processed
	var __details_state = DataTable.util.throttle(
		function (ctx) {
			_fnSaveState( ctx[0] )
		},
		500
	);
	
	
	var __details_remove = function ( api, idx )
	{
		var ctx = api.context;
	
		if ( ctx.length ) {
			var row = ctx[0].aoData[ idx !== undefined ? idx : api[0] ];
	
			if ( row && row._details ) {
				row._details.remove();
	
				row._detailsShow = undefined;
				row._details = undefined;
				$( row.nTr ).removeClass( 'dt-hasChild' );
				__details_state( ctx );
			}
		}
	};
	
	
	var __details_display = function ( api, show ) {
		var ctx = api.context;
	
		if ( ctx.length && api.length ) {
			var row = ctx[0].aoData[ api[0] ];
	
			if ( row._details ) {
				row._detailsShow = show;
	
				if ( show ) {
					row._details.insertAfter( row.nTr );
					$( row.nTr ).addClass( 'dt-hasChild' );
				}
				else {
					row._details.detach();
					$( row.nTr ).removeClass( 'dt-hasChild' );
				}
	
				_fnCallbackFire( ctx[0], null, 'childRow', [ show, api.row( api[0] ) ] )
	
				__details_events( ctx[0] );
				__details_state( ctx );
			}
		}
	};
	
	
	var __details_events = function ( settings )
	{
		var api = new _Api( settings );
		var namespace = '.dt.DT_details';
		var drawEvent = 'draw'+namespace;
		var colvisEvent = 'column-sizing'+namespace;
		var destroyEvent = 'destroy'+namespace;
		var data = settings.aoData;
	
		api.off( drawEvent +' '+ colvisEvent +' '+ destroyEvent );
	
		if ( _pluck( data, '_details' ).length > 0 ) {
			// On each draw, insert the required elements into the document
			api.on( drawEvent, function ( e, ctx ) {
				if ( settings !== ctx ) {
					return;
				}
	
				api.rows( {page:'current'} ).eq(0).each( function (idx) {
					// Internal data grab
					var row = data[ idx ];
	
					if ( row._detailsShow ) {
						row._details.insertAfter( row.nTr );
					}
				} );
			} );
	
			// Column visibility change - update the colspan
			api.on( colvisEvent, function ( e, ctx ) {
				if ( settings !== ctx ) {
					return;
				}
	
				// Update the colspan for the details rows (note, only if it already has
				// a colspan)
				var row, visible = _fnVisbleColumns( ctx );
	
				for ( var i=0, ien=data.length ; i<ien ; i++ ) {
					row = data[i];
	
					if ( row && row._details ) {
						row._details.each(function () {
							var el = $(this).children('td');
	
							if (el.length == 1) {
								el.attr('colspan', visible);
							}
						});
					}
				}
			} );
	
			// Table destroyed - nuke any child rows
			api.on( destroyEvent, function ( e, ctx ) {
				if ( settings !== ctx ) {
					return;
				}
	
				for ( var i=0, ien=data.length ; i<ien ; i++ ) {
					if ( data[i] && data[i]._details ) {
						__details_remove( api, i );
					}
				}
			} );
		}
	};
	
	// Strings for the method names to help minification
	var _emp = '';
	var _child_obj = _emp+'row().child';
	var _child_mth = _child_obj+'()';
	
	// data can be:
	//  tr
	//  string
	//  jQuery or array of any of the above
	_api_register( _child_mth, function ( data, klass ) {
		var ctx = this.context;
	
		if ( data === undefined ) {
			// get
			return ctx.length && this.length && ctx[0].aoData[ this[0] ]
				? ctx[0].aoData[ this[0] ]._details
				: undefined;
		}
		else if ( data === true ) {
			// show
			this.child.show();
		}
		else if ( data === false ) {
			// remove
			__details_remove( this );
		}
		else if ( ctx.length && this.length ) {
			// set
			__details_add( ctx[0], ctx[0].aoData[ this[0] ], data, klass );
		}
	
		return this;
	} );
	
	
	_api_register( [
		_child_obj+'.show()',
		_child_mth+'.show()' // only when `child()` was called with parameters (without
	], function () {         // it returns an object and this method is not executed)
		__details_display( this, true );
		return this;
	} );
	
	
	_api_register( [
		_child_obj+'.hide()',
		_child_mth+'.hide()' // only when `child()` was called with parameters (without
	], function () {         // it returns an object and this method is not executed)
		__details_display( this, false );
		return this;
	} );
	
	
	_api_register( [
		_child_obj+'.remove()',
		_child_mth+'.remove()' // only when `child()` was called with parameters (without
	], function () {           // it returns an object and this method is not executed)
		__details_remove( this );
		return this;
	} );
	
	
	_api_register( _child_obj+'.isShown()', function () {
		var ctx = this.context;
	
		if ( ctx.length && this.length && ctx[0].aoData[ this[0] ] ) {
			// _detailsShown as false or undefined will fall through to return false
			return ctx[0].aoData[ this[0] ]._detailsShow || false;
		}
		return false;
	} );
	
	
	
	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 * Columns
	 *
	 * {integer}           - column index (>=0 count from left, <0 count from right)
	 * "{integer}:visIdx"  - visible column index (i.e. translate to column index)  (>=0 count from left, <0 count from right)
	 * "{integer}:visible" - alias for {integer}:visIdx  (>=0 count from left, <0 count from right)
	 * "{string}:name"     - column name
	 * "{string}"          - jQuery selector on column header nodes
	 *
	 */
	
	// can be an array of these items, comma separated list, or an array of comma
	// separated lists
	
	var __re_column_selector = /^([^:]+)?:(name|title|visIdx|visible)$/;
	
	
	// r1 and r2 are redundant - but it means that the parameters match for the
	// iterator callback in columns().data()
	var __columnData = function ( settings, column, r1, r2, rows, type ) {
		var a = [];
		for ( var row=0, ien=rows.length ; row<ien ; row++ ) {
			a.push( _fnGetCellData( settings, rows[row], column, type ) );
		}
		return a;
	};
	
	
	var __column_header = function ( settings, column, row ) {
		var header = settings.aoHeader;
		var titleRow = settings.titleRow;
		var target = null;
	
		if (row !== undefined) {
			target = row;
		}
		else if (titleRow === true) { // legacy orderCellsTop support
			target = 0;
		}
		else if (titleRow === false) {
			target = header.length - 1;
		}
		else if (titleRow !== null) {
			target = titleRow;
		}
		else {
			// Automatic - find the _last_ unique cell from the top that is not empty (last for
			// backwards compatibility)
			for (var i=0 ; i<header.length ; i++) {
				if (header[i][column].unique && $('span.dt-column-title', header[i][column].cell).text()) {
					target = i;
				}
			}
	
			if (target === null) {
				target = 0;
			}
		}
	
		return header[target][column].cell;
	};
	
	var __column_header_cells = function (header) {
		var out = [];
	
		for (var i=0 ; i<header.length ; i++) {
			for (var j=0 ; j<header[i].length ; j++) {
				var cell = header[i][j].cell;
	
				if (!out.includes(cell)) {
					out.push(cell);
				}
			}
		}
	
		return out;
	}
	
	var __column_selector = function ( settings, selector, opts )
	{
		var
			columns = settings.aoColumns,
			names, titles,
			nodes = __column_header_cells(settings.aoHeader);
		
		var run = function ( s ) {
			var selInt = _intVal( s );
	
			// Selector - all
			if ( s === '' ) {
				return _range( columns.length );
			}
	
			// Selector - index
			if ( selInt !== null ) {
				return [ selInt >= 0 ?
					selInt : // Count from left
					columns.length + selInt // Count from right (+ because its a negative value)
				];
			}
	
			// Selector = function
			if ( typeof s === 'function' ) {
				var rows = _selector_row_indexes( settings, opts );
	
				return columns.map(function (col, idx) {
					return s(
							idx,
							__columnData( settings, idx, 0, 0, rows ),
							__column_header( settings, idx )
						) ? idx : null;
				});
			}
	
			// jQuery or string selector
			var match = typeof s === 'string' ?
				s.match( __re_column_selector ) :
				'';
	
			if ( match ) {
				switch( match[2] ) {
					case 'visIdx':
					case 'visible':
						// Selector is a column index
						if (match[1] && match[1].match(/^\d+$/)) {
							var idx = parseInt( match[1], 10 );
	
							// Visible index given, convert to column index
							if ( idx < 0 ) {
								// Counting from the right
								var visColumns = columns.map( function (col,i) {
									return col.bVisible ? i : null;
								} );
								return [ visColumns[ visColumns.length + idx ] ];
							}
							// Counting from the left
							return [ _fnVisibleToColumnIndex( settings, idx ) ];
						}
						
						return columns.map( function (col, idx) {
							// Not visible, can't match
							if (! col.bVisible) {
								return null;
							}
	
							// Selector
							if (match[1]) {
								return $(nodes[idx]).filter(match[1]).length > 0 ? idx : null;
							}
	
							// `:visible` on its own
							return idx;
						} );
	
					case 'name':
						// Don't get names, unless needed, and only get once if it is
						if (!names) {
							names = _pluck( columns, 'sName' );
						}
	
						// match by name. `names` is column index complete and in order
						return names.map( function (name, i) {
							return name === match[1] ? i : null;
						} );
	
					case 'title':
						if (!titles) {
							titles = _pluck( columns, 'sTitle' );
						}
	
						// match by column title
						return titles.map( function (title, i) {
							return title === match[1] ? i : null;
						} );
	
					default:
						return [];
				}
			}
	
			// Cell in the table body
			if ( s.nodeName && s._DT_CellIndex ) {
				return [ s._DT_CellIndex.column ];
			}
	
			// jQuery selector on the TH elements for the columns
			var jqResult = $( nodes )
				.filter( s )
				.map( function () {
					return _fnColumnsFromHeader( this ); // `nodes` is column index complete and in order
				} )
				.toArray()
				.sort(function (a, b) {
					return a - b;
				});
	
			if ( jqResult.length || ! s.nodeName ) {
				return jqResult;
			}
	
			// Otherwise a node which might have a `dt-column` data attribute, or be
			// a child or such an element
			var host = $(s).closest('*[data-dt-column]');
			return host.length ?
				[ host.data('dt-column') ] :
				[];
		};
	
		var selected = _selector_run( 'column', selector, run, settings, opts );
	
		return opts.columnOrder && opts.columnOrder === 'index'
			? selected.sort(function (a, b) { return a - b; })
			: selected; // implied
	};
	
	
	var __setColumnVis = function ( settings, column, vis ) {
		var
			cols = settings.aoColumns,
			col  = cols[ column ],
			data = settings.aoData,
			cells, i, ien, tr;
	
		// Get
		if ( vis === undefined ) {
			return col.bVisible;
		}
	
		// Set
		// No change
		if ( col.bVisible === vis ) {
			return false;
		}
	
		if ( vis ) {
			// Insert column
			// Need to decide if we should use appendChild or insertBefore
			var insertBefore = _pluck(cols, 'bVisible').indexOf(true, column+1);
	
			for ( i=0, ien=data.length ; i<ien ; i++ ) {
				if (data[i]) {
					tr = data[i].nTr;
					cells = data[i].anCells;
	
					if ( tr ) {
						// insertBefore can act like appendChild if 2nd arg is null
						tr.insertBefore( cells[ column ], cells[ insertBefore ] || null );
					}
				}
			}
		}
		else {
			// Remove column
			$( _pluck( settings.aoData, 'anCells', column ) ).detach();
		}
	
		// Common actions
		col.bVisible = vis;
	
		_colGroup(settings);
		
		return true;
	};
	
	
	_api_register( 'columns()', function ( selector, opts ) {
		// argument shifting
		if ( selector === undefined ) {
			selector = '';
		}
		else if ( $.isPlainObject( selector ) ) {
			opts = selector;
			selector = '';
		}
	
		opts = _selector_opts( opts );
	
		var inst = this.iterator( 'table', function ( settings ) {
			return __column_selector( settings, selector, opts );
		}, 1 );
	
		// Want argument shifting here and in _row_selector?
		inst.selector.cols = selector;
		inst.selector.opts = opts;
	
		return inst;
	} );
	
	_api_registerPlural( 'columns().header()', 'column().header()', function ( row ) {
		return this.iterator( 'column', function (settings, column) {
			return __column_header(settings, column, row);
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().footer()', 'column().footer()', function ( row ) {
		return this.iterator( 'column', function ( settings, column ) {
			var footer = settings.aoFooter;
	
			if (! footer.length) {
				return null;
			}
	
			return settings.aoFooter[row !== undefined ? row : 0][column].cell;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().data()', 'column().data()', function () {
		return this.iterator( 'column-rows', __columnData, 1 );
	} );
	
	_api_registerPlural( 'columns().render()', 'column().render()', function ( type ) {
		return this.iterator( 'column-rows', function ( settings, column, i, j, rows ) {
			return __columnData( settings, column, i, j, rows, type );
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().dataSrc()', 'column().dataSrc()', function () {
		return this.iterator( 'column', function ( settings, column ) {
			return settings.aoColumns[column].mData;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().cache()', 'column().cache()', function ( type ) {
		return this.iterator( 'column-rows', function ( settings, column, i, j, rows ) {
			return _pluck_order( settings.aoData, rows,
				type === 'search' ? '_aFilterData' : '_aSortData', column
			);
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().init()', 'column().init()', function () {
		return this.iterator( 'column', function ( settings, column ) {
			return settings.aoColumns[column];
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().names()', 'column().name()', function () {
		return this.iterator( 'column', function ( settings, column ) {
			return settings.aoColumns[column].sName;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().nodes()', 'column().nodes()', function () {
		return this.iterator( 'column-rows', function ( settings, column, i, j, rows ) {
			return _pluck_order( settings.aoData, rows, 'anCells', column ) ;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().titles()', 'column().title()', function (title, row) {
		return this.iterator( 'column', function ( settings, column ) {
			// Argument shifting
			if (typeof title === 'number') {
				row = title;
				title = undefined;
			}
	
			var span = $('span.dt-column-title', this.column(column).header(row));
	
			if (title !== undefined) {
				span.html(title);
				return this;
			}
	
			return span.html();
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().types()', 'column().type()', function () {
		return this.iterator( 'column', function ( settings, column ) {
			var type = settings.aoColumns[column].sType;
	
			// If the type was invalidated, then resolve it. This actually does
			// all columns at the moment. Would only happen once if getting all
			// column's data types.
			if (! type) {
				_fnColumnTypes(settings);
			}
	
			return type;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().visible()', 'column().visible()', function ( vis, calc ) {
		var that = this;
		var changed = [];
		var ret = this.iterator( 'column', function ( settings, column ) {
			if ( vis === undefined ) {
				return settings.aoColumns[ column ].bVisible;
			} // else
			
			if (__setColumnVis( settings, column, vis )) {
				changed.push(column);
			}
		} );
	
		// Group the column visibility changes
		if ( vis !== undefined ) {
			this.iterator( 'table', function ( settings ) {
				// Redraw the header after changes
				_fnDrawHead( settings, settings.aoHeader );
				_fnDrawHead( settings, settings.aoFooter );
		
				// Update colspan for no records display. Child rows and extensions will use their own
				// listeners to do this - only need to update the empty table item here
				if ( ! settings.aiDisplay.length ) {
					$(settings.nTBody).find('td[colspan]').attr('colspan', _fnVisbleColumns(settings));
				}
		
				_fnSaveState( settings );
	
				// Second loop once the first is done for events
				that.iterator( 'column', function ( settings, column ) {
					if (changed.includes(column)) {
						_fnCallbackFire( settings, null, 'column-visibility', [settings, column, vis, calc] );
					}
				} );
	
				if ( changed.length && (calc === undefined || calc) ) {
					that.columns.adjust();
				}
			});
		}
	
		return ret;
	} );
	
	_api_registerPlural( 'columns().widths()', 'column().width()', function () {
		// Injects a fake row into the table for just a moment so the widths can
		// be read, regardless of colspan in the header and rows being present in
		// the body
		var columns = this.columns(':visible').count();
		var row = $('<tr>').html('<td>' + Array(columns).join('</td><td>') + '</td>');
	
		$(this.table().body()).append(row);
	
		var widths = row.children().map(function () {
			return $(this).outerWidth();
		});
	
		row.remove();
		
		return this.iterator( 'column', function ( settings, column ) {
			var visIdx = _fnColumnIndexToVisible( settings, column );
	
			return visIdx !== null ? widths[visIdx] : 0;
		}, 1);
	} );
	
	_api_registerPlural( 'columns().indexes()', 'column().index()', function ( type ) {
		return this.iterator( 'column', function ( settings, column ) {
			return type === 'visible' ?
				_fnColumnIndexToVisible( settings, column ) :
				column;
		}, 1 );
	} );
	
	_api_register( 'columns.adjust()', function () {
		return this.iterator( 'table', function ( settings ) {
			// Force a column sizing to happen with a manual call - otherwise it can skip
			// if the size hasn't changed
			settings.containerWidth = -1;
	
			_fnAdjustColumnSizing( settings );
		}, 1 );
	} );
	
	_api_register( 'column.index()', function ( type, idx ) {
		if ( this.context.length !== 0 ) {
			var ctx = this.context[0];
	
			if ( type === 'fromVisible' || type === 'toData' ) {
				return _fnVisibleToColumnIndex( ctx, idx );
			}
			else if ( type === 'fromData' || type === 'toVisible' ) {
				return _fnColumnIndexToVisible( ctx, idx );
			}
		}
	} );
	
	_api_register( 'column()', function ( selector, opts ) {
		return _selector_first( this.columns( selector, opts ) );
	} );
	
	var __cell_selector = function ( settings, selector, opts )
	{
		var data = settings.aoData;
		var rows = _selector_row_indexes( settings, opts );
		var cells = _removeEmpty( _pluck_order( data, rows, 'anCells' ) );
		var allCells = $(_flatten( [], cells ));
		var row;
		var columns = settings.aoColumns.length;
		var a, i, ien, j, o, host;
	
		var run = function ( s ) {
			var fnSelector = typeof s === 'function';
	
			if ( s === null || s === undefined || fnSelector ) {
				// All cells and function selectors
				a = [];
	
				for ( i=0, ien=rows.length ; i<ien ; i++ ) {
					row = rows[i];
	
					for ( j=0 ; j<columns ; j++ ) {
						o = {
							row: row,
							column: j
						};
	
						if ( fnSelector ) {
							// Selector - function
							host = data[ row ];
	
							if ( s( o, _fnGetCellData(settings, row, j), host.anCells ? host.anCells[j] : null ) ) {
								a.push( o );
							}
						}
						else {
							// Selector - all
							a.push( o );
						}
					}
				}
	
				return a;
			}
			
			// Selector - index
			if ( $.isPlainObject( s ) ) {
				// Valid cell index and its in the array of selectable rows
				return s.column !== undefined && s.row !== undefined && rows.indexOf(s.row) !== -1 ?
					[s] :
					[];
			}
	
			// Selector - jQuery filtered cells
			var jqResult = allCells
				.filter( s )
				.map( function (i, el) {
					return { // use a new object, in case someone changes the values
						row:    el._DT_CellIndex.row,
						column: el._DT_CellIndex.column
					};
				} )
				.toArray();
	
			if ( jqResult.length || ! s.nodeName ) {
				return jqResult;
			}
	
			// Otherwise the selector is a node, and there is one last option - the
			// element might be a child of an element which has dt-row and dt-column
			// data attributes
			host = $(s).closest('*[data-dt-row]');
			return host.length ?
				[ {
					row: host.data('dt-row'),
					column: host.data('dt-column')
				} ] :
				[];
		};
	
		return _selector_run( 'cell', selector, run, settings, opts );
	};
	
	
	
	
	_api_register( 'cells()', function ( rowSelector, columnSelector, opts ) {
		// Argument shifting
		if ( $.isPlainObject( rowSelector ) ) {
			// Indexes
			if ( rowSelector.row === undefined ) {
				// Selector options in first parameter
				opts = rowSelector;
				rowSelector = null;
			}
			else {
				// Cell index objects in first parameter
				opts = columnSelector;
				columnSelector = null;
			}
		}
		if ( $.isPlainObject( columnSelector ) ) {
			opts = columnSelector;
			columnSelector = null;
		}
	
		// Cell selector
		if ( columnSelector === null || columnSelector === undefined ) {
			return this.iterator( 'table', function ( settings ) {
				return __cell_selector( settings, rowSelector, _selector_opts( opts ) );
			} );
		}
	
		// The default built in options need to apply to row and columns
		var internalOpts = opts ? {
			page: opts.page,
			order: opts.order,
			search: opts.search
		} : {};
	
		// Row + column selector
		var columns = this.columns( columnSelector, internalOpts );
		var rows = this.rows( rowSelector, internalOpts );
		var i, ien, j, jen;
	
		var cellsNoOpts = this.iterator( 'table', function ( settings, idx ) {
			var a = [];
	
			for ( i=0, ien=rows[idx].length ; i<ien ; i++ ) {
				for ( j=0, jen=columns[idx].length ; j<jen ; j++ ) {
					a.push( {
						row:    rows[idx][i],
						column: columns[idx][j]
					} );
				}
			}
	
			return a;
		}, 1 );
	
		// There is currently only one extension which uses a cell selector extension
		// It is a _major_ performance drag to run this if it isn't needed, so this is
		// an extension specific check at the moment
		var cells = opts && opts.selected ?
			this.cells( cellsNoOpts, opts ) :
			cellsNoOpts;
	
		$.extend( cells.selector, {
			cols: columnSelector,
			rows: rowSelector,
			opts: opts
		} );
	
		return cells;
	} );
	
	
	_api_registerPlural( 'cells().nodes()', 'cell().node()', function () {
		return this.iterator( 'cell', function ( settings, row, column ) {
			var data = settings.aoData[ row ];
	
			return data && data.anCells ?
				data.anCells[ column ] :
				undefined;
		}, 1 );
	} );
	
	
	_api_register( 'cells().data()', function () {
		return this.iterator( 'cell', function ( settings, row, column ) {
			return _fnGetCellData( settings, row, column );
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().cache()', 'cell().cache()', function ( type ) {
		type = type === 'search' ? '_aFilterData' : '_aSortData';
	
		return this.iterator( 'cell', function ( settings, row, column ) {
			return settings.aoData[ row ][ type ][ column ];
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().render()', 'cell().render()', function ( type ) {
		return this.iterator( 'cell', function ( settings, row, column ) {
			return _fnGetCellData( settings, row, column, type );
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().indexes()', 'cell().index()', function () {
		return this.iterator( 'cell', function ( settings, row, column ) {
			return {
				row: row,
				column: column,
				columnVisible: _fnColumnIndexToVisible( settings, column )
			};
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().invalidate()', 'cell().invalidate()', function ( src ) {
		return this.iterator( 'cell', function ( settings, row, column ) {
			_fnInvalidate( settings, row, src, column );
		} );
	} );
	
	
	
	_api_register( 'cell()', function ( rowSelector, columnSelector, opts ) {
		return _selector_first( this.cells( rowSelector, columnSelector, opts ) );
	} );
	
	
	_api_register( 'cell().data()', function ( data ) {
		var ctx = this.context;
		var cell = this[0];
	
		if ( data === undefined ) {
			// Get
			return ctx.length && cell.length ?
				_fnGetCellData( ctx[0], cell[0].row, cell[0].column ) :
				undefined;
		}
	
		// Set
		_fnSetCellData( ctx[0], cell[0].row, cell[0].column, data );
		_fnInvalidate( ctx[0], cell[0].row, 'data', cell[0].column );
	
		return this;
	} );
	
	
	
	/**
	 * Get current ordering (sorting) that has been applied to the table.
	 *
	 * @returns {array} 2D array containing the sorting information for the first
	 *   table in the current context. Each element in the parent array represents
	 *   a column being sorted upon (i.e. multi-sorting with two columns would have
	 *   2 inner arrays). The inner arrays may have 2 or 3 elements. The first is
	 *   the column index that the sorting condition applies to, the second is the
	 *   direction of the sort (`desc` or `asc`) and, optionally, the third is the
	 *   index of the sorting order from the `column.sorting` initialisation array.
	 *//**
	 * Set the ordering for the table.
	 *
	 * @param {integer} order Column index to sort upon.
	 * @param {string} direction Direction of the sort to be applied (`asc` or `desc`)
	 * @returns {DataTables.Api} this
	 *//**
	 * Set the ordering for the table.
	 *
	 * @param {array} order 1D array of sorting information to be applied.
	 * @param {array} [...] Optional additional sorting conditions
	 * @returns {DataTables.Api} this
	 *//**
	 * Set the ordering for the table.
	 *
	 * @param {array} order 2D array of sorting information to be applied.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'order()', function ( order, dir ) {
		var ctx = this.context;
		var args = Array.prototype.slice.call( arguments );
	
		if ( order === undefined ) {
			// get
			return ctx.length !== 0 ?
				ctx[0].aaSorting :
				undefined;
		}
	
		// set
		if ( typeof order === 'number' ) {
			// Simple column / direction passed in
			order = [ [ order, dir ] ];
		}
		else if ( args.length > 1 ) {
			// Arguments passed in (list of 1D arrays)
			order = args;
		}
		// otherwise a 2D array was passed in
	
		return this.iterator( 'table', function ( settings ) {
			var resolved = [];
			_fnSortResolve(settings, resolved, order);
	
			settings.aaSorting = resolved;
		} );
	} );
	
	
	/**
	 * Attach a sort listener to an element for a given column
	 *
	 * @param {node|jQuery|string} node Identifier for the element(s) to attach the
	 *   listener to. This can take the form of a single DOM node, a jQuery
	 *   collection of nodes or a jQuery selector which will identify the node(s).
	 * @param {integer} column the column that a click on this node will sort on
	 * @param {function} [callback] callback function when sort is run
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'order.listener()', function ( node, column, callback ) {
		return this.iterator( 'table', function ( settings ) {
			_fnSortAttachListener(settings, node, {}, column, callback);
		} );
	} );
	
	
	_api_register( 'order.fixed()', function ( set ) {
		if ( ! set ) {
			var ctx = this.context;
			var fixed = ctx.length ?
				ctx[0].aaSortingFixed :
				undefined;
	
			return Array.isArray( fixed ) ?
				{ pre: fixed } :
				fixed;
		}
	
		return this.iterator( 'table', function ( settings ) {
			settings.aaSortingFixed = $.extend( true, {}, set );
		} );
	} );
	
	
	// Order by the selected column(s)
	_api_register( [
		'columns().order()',
		'column().order()'
	], function ( dir ) {
		var that = this;
	
		if ( ! dir ) {
			return this.iterator( 'column', function ( settings, idx ) {
				var sort = _fnSortFlatten( settings );
	
				for ( var i=0, ien=sort.length ; i<ien ; i++ ) {
					if ( sort[i].col === idx ) {
						return sort[i].dir;
					}
				}
	
				return null;
			}, 1 );
		}
		else {
			return this.iterator( 'table', function ( settings, i ) {
				settings.aaSorting = that[i].map( function (col) {
					return [ col, dir ];
				} );
			} );
		}
	} );
	
	_api_registerPlural('columns().orderable()', 'column().orderable()', function ( directions ) {
		return this.iterator( 'column', function ( settings, idx ) {
			var col = settings.aoColumns[idx];
	
			return directions ?
				col.asSorting :
				col.bSortable;
		}, 1 );
	} );
	
	
	_api_register( 'processing()', function ( show ) {
		return this.iterator( 'table', function ( ctx ) {
			_fnProcessingDisplay( ctx, show );
		} );
	} );
	
	
	_api_register( 'search()', function ( input, regex, smart, caseInsen ) {
		var ctx = this.context;
	
		if ( input === undefined ) {
			// get
			return ctx.length !== 0 ?
				ctx[0].oPreviousSearch.search :
				undefined;
		}
	
		// set
		return this.iterator( 'table', function ( settings ) {
			if ( ! settings.oFeatures.bFilter ) {
				return;
			}
	
			if (typeof regex === 'object') {
				// New style options to pass to the search builder
				_fnFilterComplete( settings, $.extend( settings.oPreviousSearch, regex, {
					search: input
				} ) );
			}
			else {
				// Compat for the old options
				_fnFilterComplete( settings, $.extend( settings.oPreviousSearch, {
					search: input,
					regex:  regex === null ? false : regex,
					smart:  smart === null ? true  : smart,
					caseInsensitive: caseInsen === null ? true : caseInsen
				} ) );
			}
		} );
	} );
	
	_api_register( 'search.fixed()', function ( name, search ) {
		var ret = this.iterator( true, 'table', function ( settings ) {
			var fixed = settings.searchFixed;
	
			if (! name) {
				return Object.keys(fixed);
			}
			else if (search === undefined) {
				return fixed[name];
			}
			else if (search === null) {
				delete fixed[name];
			}
			else {
				fixed[name] = search;
			}
	
			return this;
		} );
	
		return name !== undefined && search === undefined
			? ret[0]
			: ret;
	} );
	
	_api_registerPlural(
		'columns().search()',
		'column().search()',
		function ( input, regex, smart, caseInsen ) {
			return this.iterator( 'column', function ( settings, column ) {
				var preSearch = settings.aoPreSearchCols;
	
				if ( input === undefined ) {
					// get
					return preSearch[ column ].search;
				}
	
				// set
				if ( ! settings.oFeatures.bFilter ) {
					return;
				}
	
				if (typeof regex === 'object') {
					// New style options to pass to the search builder
					$.extend( preSearch[ column ], regex, {
						search: input
					} );
				}
				else {
					// Old style (with not all options available)
					$.extend( preSearch[ column ], {
						search: input,
						regex:  regex === null ? false : regex,
						smart:  smart === null ? true  : smart,
						caseInsensitive: caseInsen === null ? true : caseInsen
					} );
				}
	
				_fnFilterComplete( settings, settings.oPreviousSearch );
			} );
		}
	);
	
	_api_register([
			'columns().search.fixed()',
			'column().search.fixed()'
		],
		function ( name, search ) {
			var ret = this.iterator( true, 'column', function ( settings, colIdx ) {
				var fixed = settings.aoColumns[colIdx].searchFixed;
	
				if (! name) {
					return Object.keys(fixed);
				}
				else if (search === undefined) {
					return fixed[name] || null;
				}
				else if (search === null) {
					delete fixed[name];
				}
				else {
					fixed[name] = search;
				}
	
				return this;
			} );
	
			return name !== undefined && search === undefined
				? ret[0]
				: ret;
		}
	);
	/*
	 * State API methods
	 */
	
	_api_register( 'state()', function ( set, ignoreTime ) {
		// getter
		if ( ! set ) {
			return this.context.length ?
				this.context[0].oSavedState :
				null;
		}
	
		var setMutate = $.extend( true, {}, set );
	
		// setter
		return this.iterator( 'table', function ( settings ) {
			if ( ignoreTime !== false ) {
				setMutate.time = +new Date() + 100;
			}
	
			_fnImplementState( settings, setMutate, function(){} );
		} );
	} );
	
	
	_api_register( 'state.clear()', function () {
		return this.iterator( 'table', function ( settings ) {
			// Save an empty object
			settings.fnStateSaveCallback.call( settings.oInstance, settings, {} );
		} );
	} );
	
	
	_api_register( 'state.loaded()', function () {
		return this.context.length ?
			this.context[0].oLoadedState :
			null;
	} );
	
	
	_api_register( 'state.save()', function () {
		return this.iterator( 'table', function ( settings ) {
			_fnSaveState( settings );
		} );
	} );
	
	// Can be assigned in DateTable.use() - note luxon and moment vars are in helpers.js
	var __bootstrap;
	var __foundation;
	
	/**
	 * Set the libraries that DataTables uses, or the global objects.
	 * Note that the arguments can be either way around (legacy support)
	 * and the second is optional. See docs.
	 */
	DataTable.use = function (arg1, arg2) {
		// Reverse arguments for legacy support
		var module = typeof arg1 === 'string'
			? arg2
			: arg1;
		var type = typeof arg2 === 'string'
			? arg2
			: arg1;
	
		// Getter
		if (module === undefined && typeof type === 'string') {
			switch (type) {
				case 'lib':
				case 'jq':
					return $;
	
				case 'win':
					return window;
	
				case 'datetime':
					return DataTable.DateTime;
	
				case 'luxon':
					return __luxon;
	
				case 'moment':
					return __moment;
	
				case 'bootstrap':
					// Use local if set, otherwise try window, which could be undefined
					return __bootstrap || window.bootstrap;
	
				case 'foundation':
					// Ditto
					return __foundation || window.Foundation;
	
				default:
					return null;
			}
		}
	
		// Setter
		if (type === 'lib' || type === 'jq' || (module && module.fn && module.fn.jquery)) {
			$ = module;
		}
		else if (type === 'win' || (module && module.document)) {
			window = module;
			document = module.document;
		}
		else if (type === 'datetime' || (module && module.type === 'DateTime')) {
			DataTable.DateTime = module;
		}
		else if (type === 'luxon' || (module && module.FixedOffsetZone)) {
			__luxon = module;
		}
		else if (type === 'moment' || (module && module.isMoment)) {
			__moment = module;
		}
		else if (type === 'bootstrap' || (module && module.Modal && module.Modal.NAME === 'modal'))
		{
			// This is currently for BS5 only. BS3/4 attach to jQuery, so no need to use `.use()`
			__bootstrap = module;
		}
		else if (type === 'foundation' || (module && module.Reveal)) {
			__foundation = module;
		}
	}
	
	/**
	 * CommonJS factory function pass through. This will check if the arguments
	 * given are a window object or a jQuery object. If so they are set
	 * accordingly.
	 * @param {*} root Window
	 * @param {*} jq jQUery
	 * @returns {boolean} Indicator
	 */
	DataTable.factory = function (root, jq) {
		var is = false;
	
		// Test if the first parameter is a window object
		if (root && root.document) {
			window = root;
			document = root.document;
		}
	
		// Test if the second parameter is a jQuery object
		if (jq && jq.fn && jq.fn.jquery) {
			$ = jq;
			is = true;
		}
	
		return is;
	}
	
	/**
	 * Provide a common method for plug-ins to check the version of DataTables being
	 * used, in order to ensure compatibility.
	 *
	 *  @param {string} version Version string to check for, in the format "X.Y.Z".
	 *    Note that the formats "X" and "X.Y" are also acceptable.
	 *  @param {string} [version2=current DataTables version] As above, but optional.
	 *   If not given the current DataTables version will be used.
	 *  @returns {boolean} true if this version of DataTables is greater or equal to
	 *    the required version, or false if this version of DataTales is not
	 *    suitable
	 *  @static
	 *  @dtopt API-Static
	 *
	 *  @example
	 *    alert( $.fn.dataTable.versionCheck( '1.9.0' ) );
	 */
	DataTable.versionCheck = function( version, version2 )
	{
		var aThis = version2 ?
			version2.split('.') :
			DataTable.version.split('.');
		var aThat = version.split('.');
		var iThis, iThat;
	
		for ( var i=0, iLen=aThat.length ; i<iLen ; i++ ) {
			iThis = parseInt( aThis[i], 10 ) || 0;
			iThat = parseInt( aThat[i], 10 ) || 0;
	
			// Parts are the same, keep comparing
			if (iThis === iThat) {
				continue;
			}
	
			// Parts are different, return immediately
			return iThis > iThat;
		}
	
		return true;
	};
	
	
	/**
	 * Check if a `<table>` node is a DataTable table already or not.
	 *
	 *  @param {node|jquery|string} table Table node, jQuery object or jQuery
	 *      selector for the table to test. Note that if more than more than one
	 *      table is passed on, only the first will be checked
	 *  @returns {boolean} true the table given is a DataTable, or false otherwise
	 *  @static
	 *  @dtopt API-Static
	 *
	 *  @example
	 *    if ( ! $.fn.DataTable.isDataTable( '#example' ) ) {
	 *      $('#example').dataTable();
	 *    }
	 */
	DataTable.isDataTable = function ( table )
	{
		var t = $(table).get(0);
		var is = false;
	
		if ( table instanceof DataTable.Api ) {
			return true;
		}
	
		$.each( DataTable.settings, function (i, o) {
			var head = o.nScrollHead ? $('table', o.nScrollHead)[0] : null;
			var foot = o.nScrollFoot ? $('table', o.nScrollFoot)[0] : null;
	
			if ( o.nTable === t || head === t || foot === t ) {
				is = true;
			}
		} );
	
		return is;
	};
	
	
	/**
	 * Get all DataTable tables that have been initialised - optionally you can
	 * select to get only currently visible tables.
	 *
	 *  @param {boolean} [visible=false] Flag to indicate if you want all (default)
	 *    or visible tables only.
	 *  @returns {array} Array of `table` nodes (not DataTable instances) which are
	 *    DataTables
	 *  @static
	 *  @dtopt API-Static
	 *
	 *  @example
	 *    $.each( $.fn.dataTable.tables(true), function () {
	 *      $(table).DataTable().columns.adjust();
	 *    } );
	 */
	DataTable.tables = function ( visible )
	{
		var api = false;
	
		if ( $.isPlainObject( visible ) ) {
			api = visible.api;
			visible = visible.visible;
		}
	
		var a = DataTable.settings
			.filter( function (o) {
				return !visible || (visible && $(o.nTable).is(':visible')) 
					? true
					: false;
			} )
			.map( function (o) {
				return o.nTable;
			});
	
		return api ?
			new _Api( a ) :
			a;
	};
	
	
	/**
	 * Convert from camel case parameters to Hungarian notation. This is made public
	 * for the extensions to provide the same ability as DataTables core to accept
	 * either the 1.9 style Hungarian notation, or the 1.10+ style camelCase
	 * parameters.
	 *
	 *  @param {object} src The model object which holds all parameters that can be
	 *    mapped.
	 *  @param {object} user The object to convert from camel case to Hungarian.
	 *  @param {boolean} force When set to `true`, properties which already have a
	 *    Hungarian value in the `user` object will be overwritten. Otherwise they
	 *    won't be.
	 */
	DataTable.camelToHungarian = _fnCamelToHungarian;
	
	
	
	/**
	 *
	 */
	_api_register( '$()', function ( selector, opts ) {
		var
			rows   = this.rows( opts ).nodes(), // Get all rows
			jqRows = $(rows);
	
		return $( [].concat(
			jqRows.filter( selector ).toArray(),
			jqRows.find( selector ).toArray()
		) );
	} );
	
	
	// jQuery functions to operate on the tables
	$.each( [ 'on', 'one', 'off' ], function (i, key) {
		_api_register( key+'()', function ( /* event, handler */ ) {
			var args = Array.prototype.slice.call(arguments);
	
			// Add the `dt` namespace automatically if it isn't already present
			args[0] = args[0].split( /\s/ ).map( function ( e ) {
				return ! e.match(/\.dt\b/) ?
					e+'.dt' :
					e;
				} ).join( ' ' );
	
			var inst = $( this.tables().nodes() );
			inst[key].apply( inst, args );
			return this;
		} );
	} );
	
	
	_api_register( 'clear()', function () {
		return this.iterator( 'table', function ( settings ) {
			_fnClearTable( settings );
		} );
	} );
	
	
	_api_register( 'error()', function (msg) {
		return this.iterator( 'table', function ( settings ) {
			_fnLog( settings, 0, msg );
		} );
	} );
	
	
	_api_register( 'settings()', function () {
		return new _Api( this.context, this.context );
	} );
	
	
	_api_register( 'init()', function () {
		var ctx = this.context;
		return ctx.length ? ctx[0].oInit : null;
	} );
	
	
	_api_register( 'data()', function () {
		return this.iterator( 'table', function ( settings ) {
			return _pluck( settings.aoData, '_aData' );
		} ).flatten();
	} );
	
	
	_api_register( 'trigger()', function ( name, args, bubbles ) {
		return this.iterator( 'table', function ( settings ) {
			return _fnCallbackFire( settings, null, name, args, bubbles );
		} ).flatten();
	} );
	
	
	_api_register( 'ready()', function ( fn ) {
		var ctx = this.context;
	
		// Get status of first table
		if (! fn) {
			return ctx.length
				? (ctx[0]._bInitComplete || false)
				: null;
		}
	
		// Function to run either once the table becomes ready or
		// immediately if it is already ready.
		return this.tables().every(function () {
			var api = this;
	
			if (this.context[0]._bInitComplete) {
				fn.call(api);
			}
			else {
				this.on('init.dt.DT', function () {
					fn.call(api);
				});
			}
		} );
	} );
	
	
	_api_register( 'destroy()', function ( remove ) {
		remove = remove || false;
	
		return this.iterator( 'table', function ( settings ) {
			var classes   = settings.oClasses;
			var table     = settings.nTable;
			var tbody     = settings.nTBody;
			var thead     = settings.nTHead;
			var tfoot     = settings.nTFoot;
			var jqTable   = $(table);
			var jqTbody   = $(tbody);
			var jqWrapper = $(settings.nTableWrapper);
			var rows      = settings.aoData.map( function (r) { return r ? r.nTr : null; } );
			var orderClasses = classes.order;
	
			// Flag to note that the table is currently being destroyed - no action
			// should be taken
			settings.bDestroying = true;
	
			// Fire off the destroy callbacks for plug-ins etc
			_fnCallbackFire( settings, "aoDestroyCallback", "destroy", [settings], true );
	
			// If not being removed from the document, make all columns visible
			if ( ! remove ) {
				new _Api( settings ).columns().visible( true );
			}
	
			// Container width change listener
			if (settings.resizeObserver) {
				settings.resizeObserver.disconnect();
			}
	
			// Blitz all `DT` namespaced events (these are internal events, the
			// lowercase, `dt` events are user subscribed and they are responsible
			// for removing them
			jqWrapper.off('.DT').find(':not(tbody *)').off('.DT');
			$(window).off('.DT-'+settings.sInstance);
	
			// When scrolling we had to break the table up - restore it
			if ( table != thead.parentNode ) {
				jqTable.children('thead').detach();
				jqTable.append( thead );
			}
	
			if ( tfoot && table != tfoot.parentNode ) {
				jqTable.children('tfoot').detach();
				jqTable.append( tfoot );
			}
	
			// Clean up the header / footer
			cleanHeader(thead, 'header');
			cleanHeader(tfoot, 'footer');
			settings.colgroup.remove();
	
			settings.aaSorting = [];
			settings.aaSortingFixed = [];
			_fnSortingClasses( settings );
	
			$(jqTable).find('th, td').removeClass(
				$.map(DataTable.ext.type.className, function (v) {
					return v;
				}).join(' ')
			);
	
			$('th, td', thead)
				.removeClass(
					orderClasses.none + ' ' +
					orderClasses.canAsc + ' ' +
					orderClasses.canDesc + ' ' +
					orderClasses.isAsc + ' ' +
					orderClasses.isDesc
				)
				.css('width', '')
				.removeAttr('aria-sort');
	
			// Add the TR elements back into the table in their original order
			jqTbody.children().detach();
			jqTbody.append( rows );
	
			var orig = settings.nTableWrapper.parentNode;
			var insertBefore = settings.nTableWrapper.nextSibling;
	
			// Remove the DataTables generated nodes, events and classes
			var removedMethod = remove ? 'remove' : 'detach';
			jqTable[ removedMethod ]();
			jqWrapper[ removedMethod ]();
	
			// If we need to reattach the table to the document
			if ( ! remove && orig ) {
				// insertBefore acts like appendChild if !arg[1]
				orig.insertBefore( table, insertBefore );
	
				// Restore the width of the original table - was read from the style property,
				// so we can restore directly to that
				jqTable
					.css( 'width', settings.sDestroyWidth )
					.removeClass( classes.table );
			}
	
			/* Remove the settings object from the settings array */
			var idx = DataTable.settings.indexOf(settings);
			if ( idx !== -1 ) {
				DataTable.settings.splice( idx, 1 );
			}
		} );
	} );
	
	
	// Add the `every()` method for rows, columns and cells in a compact form
	$.each( [ 'column', 'row', 'cell' ], function ( i, type ) {
		_api_register( type+'s().every()', function ( fn ) {
			var opts = this.selector.opts;
			var api = this;
			var inst;
			var counter = 0;
	
			return this.iterator( 'every', function ( settings, selectedIdx, tableIdx ) {
				inst = api[ type ](selectedIdx, opts);
	
				if (type === 'cell') {
					fn.call(inst, inst[0][0].row, inst[0][0].column, tableIdx, counter);
				}
				else {
					fn.call(inst, selectedIdx, tableIdx, counter);
				}
	
				counter++;
			} );
		} );
	} );
	
	
	// i18n method for extensions to be able to use the language object from the
	// DataTable
	_api_register( 'i18n()', function ( token, def, plural ) {
		var ctx = this.context[0];
		var resolved = _fnGetObjectDataFn( token )( ctx.oLanguage );
	
		if ( resolved === undefined ) {
			resolved = def;
		}
	
		if ( $.isPlainObject( resolved ) ) {
			resolved = plural !== undefined && resolved[ plural ] !== undefined ?
				resolved[ plural ] :
				resolved._;
		}
	
		return typeof resolved === 'string'
			? resolved.replace( '%d', plural ) // nb: plural might be undefined,
			: resolved;
	} );
	
	// Needed for header and footer, so pulled into its own function
	function cleanHeader(node, className) {
		$(node).find('span.dt-column-order').remove();
		$(node).find('span.dt-column-title').each(function () {
			var title = $(this).html();
			$(this).parent().parent().append(title);
			$(this).remove();
		});
		$(node).find('div.dt-column-' + className).remove();
	
		$('th, td', node).removeAttr('data-dt-column');
	}
	
	/**
	 * Version string for plug-ins to check compatibility. Allowed format is
	 * `a.b.c-d` where: a:int, b:int, c:int, d:string(dev|beta|alpha). `d` is used
	 * only for non-release builds. See https://semver.org/ for more information.
	 *  @member
	 *  @type string
	 *  @default Version number
	 */
	DataTable.version = "2.3.2";
	
	/**
	 * Private data store, containing all of the settings objects that are
	 * created for the tables on a given page.
	 *
	 * Note that the `DataTable.settings` object is aliased to
	 * `jQuery.fn.dataTableExt` through which it may be accessed and
	 * manipulated, or `jQuery.fn.dataTable.settings`.
	 *  @member
	 *  @type array
	 *  @default []
	 *  @private
	 */
	DataTable.settings = [];
	
	/**
	 * Object models container, for the various models that DataTables has
	 * available to it. These models define the objects that are used to hold
	 * the active state and configuration of the table.
	 *  @namespace
	 */
	DataTable.models = {};
	
	
	
	/**
	 * Template object for the way in which DataTables holds information about
	 * search information for the global filter and individual column filters.
	 *  @namespace
	 */
	DataTable.models.oSearch = {
		/**
		 * Flag to indicate if the filtering should be case insensitive or not
		 */
		"caseInsensitive": true,
	
		/**
		 * Applied search term
		 */
		"search": "",
	
		/**
		 * Flag to indicate if the search term should be interpreted as a
		 * regular expression (true) or not (false) and therefore and special
		 * regex characters escaped.
		 */
		"regex": false,
	
		/**
		 * Flag to indicate if DataTables is to use its smart filtering or not.
		 */
		"smart": true,
	
		/**
		 * Flag to indicate if DataTables should only trigger a search when
		 * the return key is pressed.
		 */
		"return": false
	};
	
	
	
	
	/**
	 * Template object for the way in which DataTables holds information about
	 * each individual row. This is the object format used for the settings
	 * aoData array.
	 *  @namespace
	 */
	DataTable.models.oRow = {
		/**
		 * TR element for the row
		 */
		"nTr": null,
	
		/**
		 * Array of TD elements for each row. This is null until the row has been
		 * created.
		 */
		"anCells": null,
	
		/**
		 * Data object from the original data source for the row. This is either
		 * an array if using the traditional form of DataTables, or an object if
		 * using mData options. The exact type will depend on the passed in
		 * data from the data source, or will be an array if using DOM a data
		 * source.
		 */
		"_aData": [],
	
		/**
		 * Sorting data cache - this array is ostensibly the same length as the
		 * number of columns (although each index is generated only as it is
		 * needed), and holds the data that is used for sorting each column in the
		 * row. We do this cache generation at the start of the sort in order that
		 * the formatting of the sort data need be done only once for each cell
		 * per sort. This array should not be read from or written to by anything
		 * other than the master sorting methods.
		 */
		"_aSortData": null,
	
		/**
		 * Per cell filtering data cache. As per the sort data cache, used to
		 * increase the performance of the filtering in DataTables
		 */
		"_aFilterData": null,
	
		/**
		 * Filtering data cache. This is the same as the cell filtering cache, but
		 * in this case a string rather than an array. This is easily computed with
		 * a join on `_aFilterData`, but is provided as a cache so the join isn't
		 * needed on every search (memory traded for performance)
		 */
		"_sFilterRow": null,
	
		/**
		 * Denote if the original data source was from the DOM, or the data source
		 * object. This is used for invalidating data, so DataTables can
		 * automatically read data from the original source, unless uninstructed
		 * otherwise.
		 */
		"src": null,
	
		/**
		 * Index in the aoData array. This saves an indexOf lookup when we have the
		 * object, but want to know the index
		 */
		"idx": -1,
	
		/**
		 * Cached display value
		 */
		displayData: null
	};
	
	
	/**
	 * Template object for the column information object in DataTables. This object
	 * is held in the settings aoColumns array and contains all the information that
	 * DataTables needs about each individual column.
	 *
	 * Note that this object is related to {@link DataTable.defaults.column}
	 * but this one is the internal data store for DataTables's cache of columns.
	 * It should NOT be manipulated outside of DataTables. Any configuration should
	 * be done through the initialisation options.
	 *  @namespace
	 */
	DataTable.models.oColumn = {
		/**
		 * Column index.
		 */
		"idx": null,
	
		/**
		 * A list of the columns that sorting should occur on when this column
		 * is sorted. That this property is an array allows multi-column sorting
		 * to be defined for a column (for example first name / last name columns
		 * would benefit from this). The values are integers pointing to the
		 * columns to be sorted on (typically it will be a single integer pointing
		 * at itself, but that doesn't need to be the case).
		 */
		"aDataSort": null,
	
		/**
		 * Define the sorting directions that are applied to the column, in sequence
		 * as the column is repeatedly sorted upon - i.e. the first value is used
		 * as the sorting direction when the column if first sorted (clicked on).
		 * Sort it again (click again) and it will move on to the next index.
		 * Repeat until loop.
		 */
		"asSorting": null,
	
		/**
		 * Flag to indicate if the column is searchable, and thus should be included
		 * in the filtering or not.
		 */
		"bSearchable": null,
	
		/**
		 * Flag to indicate if the column is sortable or not.
		 */
		"bSortable": null,
	
		/**
		 * Flag to indicate if the column is currently visible in the table or not
		 */
		"bVisible": null,
	
		/**
		 * Store for manual type assignment using the `column.type` option. This
		 * is held in store so we can manipulate the column's `sType` property.
		 */
		"_sManualType": null,
	
		/**
		 * Flag to indicate if HTML5 data attributes should be used as the data
		 * source for filtering or sorting. True is either are.
		 */
		"_bAttrSrc": false,
	
		/**
		 * Developer definable function that is called whenever a cell is created (Ajax source,
		 * etc) or processed for input (DOM source). This can be used as a compliment to mRender
		 * allowing you to modify the DOM element (add background colour for example) when the
		 * element is available.
		 */
		"fnCreatedCell": null,
	
		/**
		 * Function to get data from a cell in a column. You should <b>never</b>
		 * access data directly through _aData internally in DataTables - always use
		 * the method attached to this property. It allows mData to function as
		 * required. This function is automatically assigned by the column
		 * initialisation method
		 */
		"fnGetData": null,
	
		/**
		 * Function to set data for a cell in the column. You should <b>never</b>
		 * set the data directly to _aData internally in DataTables - always use
		 * this method. It allows mData to function as required. This function
		 * is automatically assigned by the column initialisation method
		 */
		"fnSetData": null,
	
		/**
		 * Property to read the value for the cells in the column from the data
		 * source array / object. If null, then the default content is used, if a
		 * function is given then the return from the function is used.
		 */
		"mData": null,
	
		/**
		 * Partner property to mData which is used (only when defined) to get
		 * the data - i.e. it is basically the same as mData, but without the
		 * 'set' option, and also the data fed to it is the result from mData.
		 * This is the rendering method to match the data method of mData.
		 */
		"mRender": null,
	
		/**
		 * The class to apply to all TD elements in the table's TBODY for the column
		 */
		"sClass": null,
	
		/**
		 * When DataTables calculates the column widths to assign to each column,
		 * it finds the longest string in each column and then constructs a
		 * temporary table and reads the widths from that. The problem with this
		 * is that "mmm" is much wider then "iiii", but the latter is a longer
		 * string - thus the calculation can go wrong (doing it properly and putting
		 * it into an DOM object and measuring that is horribly(!) slow). Thus as
		 * a "work around" we provide this option. It will append its value to the
		 * text that is found to be the longest string for the column - i.e. padding.
		 */
		"sContentPadding": null,
	
		/**
		 * Allows a default value to be given for a column's data, and will be used
		 * whenever a null data source is encountered (this can be because mData
		 * is set to null, or because the data source itself is null).
		 */
		"sDefaultContent": null,
	
		/**
		 * Name for the column, allowing reference to the column by name as well as
		 * by index (needs a lookup to work by name).
		 */
		"sName": null,
	
		/**
		 * Custom sorting data type - defines which of the available plug-ins in
		 * afnSortData the custom sorting will use - if any is defined.
		 */
		"sSortDataType": 'std',
	
		/**
		 * Class to be applied to the header element when sorting on this column
		 */
		"sSortingClass": null,
	
		/**
		 * Title of the column - what is seen in the TH element (nTh).
		 */
		"sTitle": null,
	
		/**
		 * Column sorting and filtering type
		 */
		"sType": null,
	
		/**
		 * Width of the column
		 */
		"sWidth": null,
	
		/**
		 * Width of the column when it was first "encountered"
		 */
		"sWidthOrig": null,
	
		/** Cached string which is the longest in the column */
		maxLenString: null,
	
		/**
		 * Store for named searches
		 */
		searchFixed: null
	};
	
	
	/*
	 * Developer note: The properties of the object below are given in Hungarian
	 * notation, that was used as the interface for DataTables prior to v1.10, however
	 * from v1.10 onwards the primary interface is camel case. In order to avoid
	 * breaking backwards compatibility utterly with this change, the Hungarian
	 * version is still, internally the primary interface, but is is not documented
	 * - hence the @name tags in each doc comment. This allows a Javascript function
	 * to create a map from Hungarian notation to camel case (going the other direction
	 * would require each property to be listed, which would add around 3K to the size
	 * of DataTables, while this method is about a 0.5K hit).
	 *
	 * Ultimately this does pave the way for Hungarian notation to be dropped
	 * completely, but that is a massive amount of work and will break current
	 * installs (therefore is on-hold until v2).
	 */
	
	/**
	 * Initialisation options that can be given to DataTables at initialisation
	 * time.
	 *  @namespace
	 */
	DataTable.defaults = {
		/**
		 * An array of data to use for the table, passed in at initialisation which
		 * will be used in preference to any data which is already in the DOM. This is
		 * particularly useful for constructing tables purely in Javascript, for
		 * example with a custom Ajax call.
		 */
		"aaData": null,
	
	
		/**
		 * If ordering is enabled, then DataTables will perform a first pass sort on
		 * initialisation. You can define which column(s) the sort is performed
		 * upon, and the sorting direction, with this variable. The `sorting` array
		 * should contain an array for each column to be sorted initially containing
		 * the column's index and a direction string ('asc' or 'desc').
		 */
		"aaSorting": [[0,'asc']],
	
	
		/**
		 * This parameter is basically identical to the `sorting` parameter, but
		 * cannot be overridden by user interaction with the table. What this means
		 * is that you could have a column (visible or hidden) which the sorting
		 * will always be forced on first - any sorting after that (from the user)
		 * will then be performed as required. This can be useful for grouping rows
		 * together.
		 */
		"aaSortingFixed": [],
	
	
		/**
		 * DataTables can be instructed to load data to display in the table from a
		 * Ajax source. This option defines how that Ajax call is made and where to.
		 *
		 * The `ajax` property has three different modes of operation, depending on
		 * how it is defined. These are:
		 *
		 * * `string` - Set the URL from where the data should be loaded from.
		 * * `object` - Define properties for `jQuery.ajax`.
		 * * `function` - Custom data get function
		 *
		 * `string`
		 * --------
		 *
		 * As a string, the `ajax` property simply defines the URL from which
		 * DataTables will load data.
		 *
		 * `object`
		 * --------
		 *
		 * As an object, the parameters in the object are passed to
		 * [jQuery.ajax](https://api.jquery.com/jQuery.ajax/) allowing fine control
		 * of the Ajax request. DataTables has a number of default parameters which
		 * you can override using this option. Please refer to the jQuery
		 * documentation for a full description of the options available, although
		 * the following parameters provide additional options in DataTables or
		 * require special consideration:
		 *
		 * * `data` - As with jQuery, `data` can be provided as an object, but it
		 *   can also be used as a function to manipulate the data DataTables sends
		 *   to the server. The function takes a single parameter, an object of
		 *   parameters with the values that DataTables has readied for sending. An
		 *   object may be returned which will be merged into the DataTables
		 *   defaults, or you can add the items to the object that was passed in and
		 *   not return anything from the function. This supersedes `fnServerParams`
		 *   from DataTables 1.9-.
		 *
		 * * `dataSrc` - By default DataTables will look for the property `data` (or
		 *   `aaData` for compatibility with DataTables 1.9-) when obtaining data
		 *   from an Ajax source or for server-side processing - this parameter
		 *   allows that property to be changed. You can use Javascript dotted
		 *   object notation to get a data source for multiple levels of nesting, or
		 *   it my be used as a function. As a function it takes a single parameter,
		 *   the JSON returned from the server, which can be manipulated as
		 *   required, with the returned value being that used by DataTables as the
		 *   data source for the table.
		 *
		 * * `success` - Should not be overridden it is used internally in
		 *   DataTables. To manipulate / transform the data returned by the server
		 *   use `ajax.dataSrc`, or use `ajax` as a function (see below).
		 *
		 * `function`
		 * ----------
		 *
		 * As a function, making the Ajax call is left up to yourself allowing
		 * complete control of the Ajax request. Indeed, if desired, a method other
		 * than Ajax could be used to obtain the required data, such as Web storage
		 * or an AIR database.
		 *
		 * The function is given four parameters and no return is required. The
		 * parameters are:
		 *
		 * 1. _object_ - Data to send to the server
		 * 2. _function_ - Callback function that must be executed when the required
		 *    data has been obtained. That data should be passed into the callback
		 *    as the only parameter
		 * 3. _object_ - DataTables settings object for the table
		 */
		"ajax": null,
	
	
		/**
		 * This parameter allows you to readily specify the entries in the length drop
		 * down menu that DataTables shows when pagination is enabled. It can be
		 * either a 1D array of options which will be used for both the displayed
		 * option and the value, or a 2D array which will use the array in the first
		 * position as the value, and the array in the second position as the
		 * displayed options (useful for language strings such as 'All').
		 *
		 * Note that the `pageLength` property will be automatically set to the
		 * first value given in this array, unless `pageLength` is also provided.
		 */
		"aLengthMenu": [ 10, 25, 50, 100 ],
	
	
		/**
		 * The `columns` option in the initialisation parameter allows you to define
		 * details about the way individual columns behave. For a full list of
		 * column options that can be set, please see
		 * {@link DataTable.defaults.column}. Note that if you use `columns` to
		 * define your columns, you must have an entry in the array for every single
		 * column that you have in your table (these can be null if you don't which
		 * to specify any options).
		 */
		"aoColumns": null,
	
		/**
		 * Very similar to `columns`, `columnDefs` allows you to target a specific
		 * column, multiple columns, or all columns, using the `targets` property of
		 * each object in the array. This allows great flexibility when creating
		 * tables, as the `columnDefs` arrays can be of any length, targeting the
		 * columns you specifically want. `columnDefs` may use any of the column
		 * options available: {@link DataTable.defaults.column}, but it _must_
		 * have `targets` defined in each object in the array. Values in the `targets`
		 * array may be:
		 *   <ul>
		 *     <li>a string - class name will be matched on the TH for the column</li>
		 *     <li>0 or a positive integer - column index counting from the left</li>
		 *     <li>a negative integer - column index counting from the right</li>
		 *     <li>the string "_all" - all columns (i.e. assign a default)</li>
		 *   </ul>
		 */
		"aoColumnDefs": null,
	
	
		/**
		 * Basically the same as `search`, this parameter defines the individual column
		 * filtering state at initialisation time. The array must be of the same size
		 * as the number of columns, and each element be an object with the parameters
		 * `search` and `escapeRegex` (the latter is optional). 'null' is also
		 * accepted and the default will be used.
		 */
		"aoSearchCols": [],
	
	
		/**
		 * Enable or disable automatic column width calculation. This can be disabled
		 * as an optimisation (it takes some time to calculate the widths) if the
		 * tables widths are passed in using `columns`.
		 */
		"bAutoWidth": true,
	
	
		/**
		 * Deferred rendering can provide DataTables with a huge speed boost when you
		 * are using an Ajax or JS data source for the table. This option, when set to
		 * true, will cause DataTables to defer the creation of the table elements for
		 * each row until they are needed for a draw - saving a significant amount of
		 * time.
		 */
		"bDeferRender": true,
	
	
		/**
		 * Replace a DataTable which matches the given selector and replace it with
		 * one which has the properties of the new initialisation object passed. If no
		 * table matches the selector, then the new DataTable will be constructed as
		 * per normal.
		 */
		"bDestroy": false,
	
	
		/**
		 * Enable or disable filtering of data. Filtering in DataTables is "smart" in
		 * that it allows the end user to input multiple words (space separated) and
		 * will match a row containing those words, even if not in the order that was
		 * specified (this allow matching across multiple columns). Note that if you
		 * wish to use filtering in DataTables this must remain 'true' - to remove the
		 * default filtering input box and retain filtering abilities, please use
		 * {@link DataTable.defaults.dom}.
		 */
		"bFilter": true,
	
		/**
		 * Used only for compatiblity with DT1
		 * @deprecated
		 */
		"bInfo": true,
	
		/**
		 * Used only for compatiblity with DT1
		 * @deprecated
		 */
		"bLengthChange": true,
	
		/**
		 * Enable or disable pagination.
		 */
		"bPaginate": true,
	
	
		/**
		 * Enable or disable the display of a 'processing' indicator when the table is
		 * being processed (e.g. a sort). This is particularly useful for tables with
		 * large amounts of data where it can take a noticeable amount of time to sort
		 * the entries.
		 */
		"bProcessing": false,
	
	
		/**
		 * Retrieve the DataTables object for the given selector. Note that if the
		 * table has already been initialised, this parameter will cause DataTables
		 * to simply return the object that has already been set up - it will not take
		 * account of any changes you might have made to the initialisation object
		 * passed to DataTables (setting this parameter to true is an acknowledgement
		 * that you understand this). `destroy` can be used to reinitialise a table if
		 * you need.
		 */
		"bRetrieve": false,
	
	
		/**
		 * When vertical (y) scrolling is enabled, DataTables will force the height of
		 * the table's viewport to the given height at all times (useful for layout).
		 * However, this can look odd when filtering data down to a small data set,
		 * and the footer is left "floating" further down. This parameter (when
		 * enabled) will cause DataTables to collapse the table's viewport down when
		 * the result set will fit within the given Y height.
		 */
		"bScrollCollapse": false,
	
	
		/**
		 * Configure DataTables to use server-side processing. Note that the
		 * `ajax` parameter must also be given in order to give DataTables a
		 * source to obtain the required data for each draw.
		 */
		"bServerSide": false,
	
	
		/**
		 * Enable or disable sorting of columns. Sorting of individual columns can be
		 * disabled by the `sortable` option for each column.
		 */
		"bSort": true,
	
	
		/**
		 * Enable or display DataTables' ability to sort multiple columns at the
		 * same time (activated by shift-click by the user).
		 */
		"bSortMulti": true,
	
	
		/**
		 * Allows control over whether DataTables should use the top (true) unique
		 * cell that is found for a single column, or the bottom (false - default).
		 * This is useful when using complex headers.
		 */
		"bSortCellsTop": null,
	
	
		/** Specify which row is the title row in the header. Replacement for bSortCellsTop */
		titleRow: null,
	
	
		/**
		 * Enable or disable the addition of the classes `sorting\_1`, `sorting\_2` and
		 * `sorting\_3` to the columns which are currently being sorted on. This is
		 * presented as a feature switch as it can increase processing time (while
		 * classes are removed and added) so for large data sets you might want to
		 * turn this off.
		 */
		"bSortClasses": true,
	
	
		/**
		 * Enable or disable state saving. When enabled HTML5 `localStorage` will be
		 * used to save table display information such as pagination information,
		 * display length, filtering and sorting. As such when the end user reloads
		 * the page the display display will match what thy had previously set up.
		 */
		"bStateSave": false,
	
	
		/**
		 * This function is called when a TR element is created (and all TD child
		 * elements have been inserted), or registered if using a DOM source, allowing
		 * manipulation of the TR element (adding classes etc).
		 */
		"fnCreatedRow": null,
	
	
		/**
		 * This function is called on every 'draw' event, and allows you to
		 * dynamically modify any aspect you want about the created DOM.
		 */
		"fnDrawCallback": null,
	
	
		/**
		 * Identical to fnHeaderCallback() but for the table footer this function
		 * allows you to modify the table footer on every 'draw' event.
		 */
		"fnFooterCallback": null,
	
	
		/**
		 * When rendering large numbers in the information element for the table
		 * (i.e. "Showing 1 to 10 of 57 entries") DataTables will render large numbers
		 * to have a comma separator for the 'thousands' units (e.g. 1 million is
		 * rendered as "1,000,000") to help readability for the end user. This
		 * function will override the default method DataTables uses.
		 */
		"fnFormatNumber": function ( toFormat ) {
			return toFormat.toString().replace(
				/\B(?=(\d{3})+(?!\d))/g,
				this.oLanguage.sThousands
			);
		},
	
	
		/**
		 * This function is called on every 'draw' event, and allows you to
		 * dynamically modify the header row. This can be used to calculate and
		 * display useful information about the table.
		 */
		"fnHeaderCallback": null,
	
	
		/**
		 * The information element can be used to convey information about the current
		 * state of the table. Although the internationalisation options presented by
		 * DataTables are quite capable of dealing with most customisations, there may
		 * be times where you wish to customise the string further. This callback
		 * allows you to do exactly that.
		 */
		"fnInfoCallback": null,
	
	
		/**
		 * Called when the table has been initialised. Normally DataTables will
		 * initialise sequentially and there will be no need for this function,
		 * however, this does not hold true when using external language information
		 * since that is obtained using an async XHR call.
		 */
		"fnInitComplete": null,
	
	
		/**
		 * Called at the very start of each table draw and can be used to cancel the
		 * draw by returning false, any other return (including undefined) results in
		 * the full draw occurring).
		 */
		"fnPreDrawCallback": null,
	
	
		/**
		 * This function allows you to 'post process' each row after it have been
		 * generated for each table draw, but before it is rendered on screen. This
		 * function might be used for setting the row class name etc.
		 */
		"fnRowCallback": null,
	
	
		/**
		 * Load the table state. With this function you can define from where, and how, the
		 * state of a table is loaded. By default DataTables will load from `localStorage`
		 * but you might wish to use a server-side database or cookies.
		 */
		"fnStateLoadCallback": function ( settings ) {
			try {
				return JSON.parse(
					(settings.iStateDuration === -1 ? sessionStorage : localStorage).getItem(
						'DataTables_'+settings.sInstance+'_'+location.pathname
					)
				);
			} catch (e) {
				return {};
			}
		},
	
	
		/**
		 * Callback which allows modification of the saved state prior to loading that state.
		 * This callback is called when the table is loading state from the stored data, but
		 * prior to the settings object being modified by the saved state. Note that for
		 * plug-in authors, you should use the `stateLoadParams` event to load parameters for
		 * a plug-in.
		 */
		"fnStateLoadParams": null,
	
	
		/**
		 * Callback that is called when the state has been loaded from the state saving method
		 * and the DataTables settings object has been modified as a result of the loaded state.
		 */
		"fnStateLoaded": null,
	
	
		/**
		 * Save the table state. This function allows you to define where and how the state
		 * information for the table is stored By default DataTables will use `localStorage`
		 * but you might wish to use a server-side database or cookies.
		 */
		"fnStateSaveCallback": function ( settings, data ) {
			try {
				(settings.iStateDuration === -1 ? sessionStorage : localStorage).setItem(
					'DataTables_'+settings.sInstance+'_'+location.pathname,
					JSON.stringify( data )
				);
			} catch (e) {
				// noop
			}
		},
	
	
		/**
		 * Callback which allows modification of the state to be saved. Called when the table
		 * has changed state a new state save is required. This method allows modification of
		 * the state saving object prior to actually doing the save, including addition or
		 * other state properties or modification. Note that for plug-in authors, you should
		 * use the `stateSaveParams` event to save parameters for a plug-in.
		 */
		"fnStateSaveParams": null,
	
	
		/**
		 * Duration for which the saved state information is considered valid. After this period
		 * has elapsed the state will be returned to the default.
		 * Value is given in seconds.
		 */
		"iStateDuration": 7200,
	
	
		/**
		 * Number of rows to display on a single page when using pagination. If
		 * feature enabled (`lengthChange`) then the end user will be able to override
		 * this to a custom setting using a pop-up menu.
		 */
		"iDisplayLength": 10,
	
	
		/**
		 * Define the starting point for data display when using DataTables with
		 * pagination. Note that this parameter is the number of records, rather than
		 * the page number, so if you have 10 records per page and want to start on
		 * the third page, it should be "20".
		 */
		"iDisplayStart": 0,
	
	
		/**
		 * By default DataTables allows keyboard navigation of the table (sorting, paging,
		 * and filtering) by adding a `tabindex` attribute to the required elements. This
		 * allows you to tab through the controls and press the enter key to activate them.
		 * The tabindex is default 0, meaning that the tab follows the flow of the document.
		 * You can overrule this using this parameter if you wish. Use a value of -1 to
		 * disable built-in keyboard navigation.
		 */
		"iTabIndex": 0,
	
	
		/**
		 * Classes that DataTables assigns to the various components and features
		 * that it adds to the HTML table. This allows classes to be configured
		 * during initialisation in addition to through the static
		 * {@link DataTable.ext.oStdClasses} object).
		 */
		"oClasses": {},
	
	
		/**
		 * All strings that DataTables uses in the user interface that it creates
		 * are defined in this object, allowing you to modified them individually or
		 * completely replace them all as required.
		 */
		"oLanguage": {
			/**
			 * Strings that are used for WAI-ARIA labels and controls only (these are not
			 * actually visible on the page, but will be read by screenreaders, and thus
			 * must be internationalised as well).
			 */
			"oAria": {
				/**
				 * ARIA label that is added to the table headers when the column may be sorted
				 */
				"orderable": ": Activate to sort",
	
				/**
				 * ARIA label that is added to the table headers when the column is currently being sorted
				 */
				"orderableReverse": ": Activate to invert sorting",
	
				/**
				 * ARIA label that is added to the table headers when the column is currently being 
				 * sorted and next step is to remove sorting
				 */
				"orderableRemove": ": Activate to remove sorting",
	
				paginate: {
					first: 'First',
					last: 'Last',
					next: 'Next',
					previous: 'Previous',
					number: ''
				}
			},
	
			/**
			 * Pagination string used by DataTables for the built-in pagination
			 * control types.
			 */
			"oPaginate": {
				/**
				 * Label and character for first page button («)
				 */
				"sFirst": "\u00AB",
	
				/**
				 * Last page button (»)
				 */
				"sLast": "\u00BB",
	
				/**
				 * Next page button (›)
				 */
				"sNext": "\u203A",
	
				/**
				 * Previous page button (‹)
				 */
				"sPrevious": "\u2039",
			},
	
			/**
			 * Plural object for the data type the table is showing
			 */
			entries: {
				_: "entries",
				1: "entry"
			},
	
			/**
			 * Page length options
			 */
			lengthLabels: {
				'-1': 'All'
			},
	
			/**
			 * This string is shown in preference to `zeroRecords` when the table is
			 * empty of data (regardless of filtering). Note that this is an optional
			 * parameter - if it is not given, the value of `zeroRecords` will be used
			 * instead (either the default or given value).
			 */
			"sEmptyTable": "No data available in table",
	
	
			/**
			 * This string gives information to the end user about the information
			 * that is current on display on the page. The following tokens can be
			 * used in the string and will be dynamically replaced as the table
			 * display updates. This tokens can be placed anywhere in the string, or
			 * removed as needed by the language requires:
			 *
			 * * `\_START\_` - Display index of the first record on the current page
			 * * `\_END\_` - Display index of the last record on the current page
			 * * `\_TOTAL\_` - Number of records in the table after filtering
			 * * `\_MAX\_` - Number of records in the table without filtering
			 * * `\_PAGE\_` - Current page number
			 * * `\_PAGES\_` - Total number of pages of data in the table
			 */
			"sInfo": "Showing _START_ to _END_ of _TOTAL_ _ENTRIES-TOTAL_",
	
	
			/**
			 * Display information string for when the table is empty. Typically the
			 * format of this string should match `info`.
			 */
			"sInfoEmpty": "Showing 0 to 0 of 0 _ENTRIES-TOTAL_",
	
	
			/**
			 * When a user filters the information in a table, this string is appended
			 * to the information (`info`) to give an idea of how strong the filtering
			 * is. The variable _MAX_ is dynamically updated.
			 */
			"sInfoFiltered": "(filtered from _MAX_ total _ENTRIES-MAX_)",
	
	
			/**
			 * If can be useful to append extra information to the info string at times,
			 * and this variable does exactly that. This information will be appended to
			 * the `info` (`infoEmpty` and `infoFiltered` in whatever combination they are
			 * being used) at all times.
			 */
			"sInfoPostFix": "",
	
	
			/**
			 * This decimal place operator is a little different from the other
			 * language options since DataTables doesn't output floating point
			 * numbers, so it won't ever use this for display of a number. Rather,
			 * what this parameter does is modify the sort methods of the table so
			 * that numbers which are in a format which has a character other than
			 * a period (`.`) as a decimal place will be sorted numerically.
			 *
			 * Note that numbers with different decimal places cannot be shown in
			 * the same table and still be sortable, the table must be consistent.
			 * However, multiple different tables on the page can use different
			 * decimal place characters.
			 */
			"sDecimal": "",
	
	
			/**
			 * DataTables has a build in number formatter (`formatNumber`) which is
			 * used to format large numbers that are used in the table information.
			 * By default a comma is used, but this can be trivially changed to any
			 * character you wish with this parameter.
			 */
			"sThousands": ",",
	
	
			/**
			 * Detail the action that will be taken when the drop down menu for the
			 * pagination length option is changed. The '_MENU_' variable is replaced
			 * with a default select list of 10, 25, 50 and 100, and can be replaced
			 * with a custom select box if required.
			 */
			"sLengthMenu": "_MENU_ _ENTRIES_ per page",
	
	
			/**
			 * When using Ajax sourced data and during the first draw when DataTables is
			 * gathering the data, this message is shown in an empty row in the table to
			 * indicate to the end user the the data is being loaded. Note that this
			 * parameter is not used when loading data by server-side processing, just
			 * Ajax sourced data with client-side processing.
			 */
			"sLoadingRecords": "Loading...",
	
	
			/**
			 * Text which is displayed when the table is processing a user action
			 * (usually a sort command or similar).
			 */
			"sProcessing": "",
	
	
			/**
			 * Details the actions that will be taken when the user types into the
			 * filtering input text box. The variable "_INPUT_", if used in the string,
			 * is replaced with the HTML text box for the filtering input allowing
			 * control over where it appears in the string. If "_INPUT_" is not given
			 * then the input box is appended to the string automatically.
			 */
			"sSearch": "Search:",
	
	
			/**
			 * Assign a `placeholder` attribute to the search `input` element
			 *  @type string
			 *  @default 
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.searchPlaceholder
			 */
			"sSearchPlaceholder": "",
	
	
			/**
			 * All of the language information can be stored in a file on the
			 * server-side, which DataTables will look up if this parameter is passed.
			 * It must store the URL of the language file, which is in a JSON format,
			 * and the object has the same properties as the oLanguage object in the
			 * initialiser object (i.e. the above parameters). Please refer to one of
			 * the example language files to see how this works in action.
			 */
			"sUrl": "",
	
	
			/**
			 * Text shown inside the table records when the is no information to be
			 * displayed after filtering. `emptyTable` is shown when there is simply no
			 * information in the table at all (regardless of filtering).
			 */
			"sZeroRecords": "No matching records found"
		},
	
	
		/** The initial data order is reversed when `desc` ordering */
		orderDescReverse: true,
	
	
		/**
		 * This parameter allows you to have define the global filtering state at
		 * initialisation time. As an object the `search` parameter must be
		 * defined, but all other parameters are optional. When `regex` is true,
		 * the search string will be treated as a regular expression, when false
		 * (default) it will be treated as a straight string. When `smart`
		 * DataTables will use it's smart filtering methods (to word match at
		 * any point in the data), when false this will not be done.
		 */
		"oSearch": $.extend( {}, DataTable.models.oSearch ),
	
	
		/**
		 * Table and control layout. This replaces the legacy `dom` option.
		 */
		layout: {
			topStart: 'pageLength',
			topEnd: 'search',
			bottomStart: 'info',
			bottomEnd: 'paging'
		},
	
	
		/**
		 * Legacy DOM layout option
		 */
		"sDom": null,
	
	
		/**
		 * Search delay option. This will throttle full table searches that use the
		 * DataTables provided search input element (it does not effect calls to
		 * `dt-api search()`, providing a delay before the search is made.
		 */
		"searchDelay": null,
	
	
		/**
		 * DataTables features six different built-in options for the buttons to
		 * display for pagination control:
		 *
		 * * `numbers` - Page number buttons only
		 * * `simple` - 'Previous' and 'Next' buttons only
		 * * 'simple_numbers` - 'Previous' and 'Next' buttons, plus page numbers
		 * * `full` - 'First', 'Previous', 'Next' and 'Last' buttons
		 * * `full_numbers` - 'First', 'Previous', 'Next' and 'Last' buttons, plus page numbers
		 * * `first_last_numbers` - 'First' and 'Last' buttons, plus page numbers
		 */
		"sPaginationType": "",
	
	
		/**
		 * Enable horizontal scrolling. When a table is too wide to fit into a
		 * certain layout, or you have a large number of columns in the table, you
		 * can enable x-scrolling to show the table in a viewport, which can be
		 * scrolled. This property can be `true` which will allow the table to
		 * scroll horizontally when needed, or any CSS unit, or a number (in which
		 * case it will be treated as a pixel measurement). Setting as simply `true`
		 * is recommended.
		 */
		"sScrollX": "",
	
	
		/**
		 * This property can be used to force a DataTable to use more width than it
		 * might otherwise do when x-scrolling is enabled. For example if you have a
		 * table which requires to be well spaced, this parameter is useful for
		 * "over-sizing" the table, and thus forcing scrolling. This property can by
		 * any CSS unit, or a number (in which case it will be treated as a pixel
		 * measurement).
		 */
		"sScrollXInner": "",
	
	
		/**
		 * Enable vertical scrolling. Vertical scrolling will constrain the DataTable
		 * to the given height, and enable scrolling for any data which overflows the
		 * current viewport. This can be used as an alternative to paging to display
		 * a lot of data in a small area (although paging and scrolling can both be
		 * enabled at the same time). This property can be any CSS unit, or a number
		 * (in which case it will be treated as a pixel measurement).
		 */
		"sScrollY": "",
	
	
		/**
		 * __Deprecated__ The functionality provided by this parameter has now been
		 * superseded by that provided through `ajax`, which should be used instead.
		 *
		 * Set the HTTP method that is used to make the Ajax call for server-side
		 * processing or Ajax sourced data.
		 */
		"sServerMethod": "GET",
	
	
		/**
		 * DataTables makes use of renderers when displaying HTML elements for
		 * a table. These renderers can be added or modified by plug-ins to
		 * generate suitable mark-up for a site. For example the Bootstrap
		 * integration plug-in for DataTables uses a paging button renderer to
		 * display pagination buttons in the mark-up required by Bootstrap.
		 *
		 * For further information about the renderers available see
		 * DataTable.ext.renderer
		 */
		"renderer": null,
	
	
		/**
		 * Set the data property name that DataTables should use to get a row's id
		 * to set as the `id` property in the node.
		 */
		"rowId": "DT_RowId",
	
	
		/**
		 * Caption value
		 */
		"caption": null,
	
	
		/**
		 * For server-side processing - use the data from the DOM for the first draw
		 */
		iDeferLoading: null,
	
		/** Event listeners */
		on: null
	};
	
	_fnHungarianMap( DataTable.defaults );
	
	
	
	/*
	 * Developer note - See note in model.defaults.js about the use of Hungarian
	 * notation and camel case.
	 */
	
	/**
	 * Column options that can be given to DataTables at initialisation time.
	 *  @namespace
	 */
	DataTable.defaults.column = {
		/**
		 * Define which column(s) an order will occur on for this column. This
		 * allows a column's ordering to take multiple columns into account when
		 * doing a sort or use the data from a different column. For example first
		 * name / last name columns make sense to do a multi-column sort over the
		 * two columns.
		 */
		"aDataSort": null,
		"iDataSort": -1,
	
		ariaTitle: '',
	
	
		/**
		 * You can control the default ordering direction, and even alter the
		 * behaviour of the sort handler (i.e. only allow ascending ordering etc)
		 * using this parameter.
		 */
		"asSorting": [ 'asc', 'desc', '' ],
	
	
		/**
		 * Enable or disable filtering on the data in this column.
		 */
		"bSearchable": true,
	
	
		/**
		 * Enable or disable ordering on this column.
		 */
		"bSortable": true,
	
	
		/**
		 * Enable or disable the display of this column.
		 */
		"bVisible": true,
	
	
		/**
		 * Developer definable function that is called whenever a cell is created (Ajax source,
		 * etc) or processed for input (DOM source). This can be used as a compliment to mRender
		 * allowing you to modify the DOM element (add background colour for example) when the
		 * element is available.
		 */
		"fnCreatedCell": null,
	
	
		/**
		 * This property can be used to read data from any data source property,
		 * including deeply nested objects / properties. `data` can be given in a
		 * number of different ways which effect its behaviour:
		 *
		 * * `integer` - treated as an array index for the data source. This is the
		 *   default that DataTables uses (incrementally increased for each column).
		 * * `string` - read an object property from the data source. There are
		 *   three 'special' options that can be used in the string to alter how
		 *   DataTables reads the data from the source object:
		 *    * `.` - Dotted Javascript notation. Just as you use a `.` in
		 *      Javascript to read from nested objects, so to can the options
		 *      specified in `data`. For example: `browser.version` or
		 *      `browser.name`. If your object parameter name contains a period, use
		 *      `\\` to escape it - i.e. `first\\.name`.
		 *    * `[]` - Array notation. DataTables can automatically combine data
		 *      from and array source, joining the data with the characters provided
		 *      between the two brackets. For example: `name[, ]` would provide a
		 *      comma-space separated list from the source array. If no characters
		 *      are provided between the brackets, the original array source is
		 *      returned.
		 *    * `()` - Function notation. Adding `()` to the end of a parameter will
		 *      execute a function of the name given. For example: `browser()` for a
		 *      simple function on the data source, `browser.version()` for a
		 *      function in a nested property or even `browser().version` to get an
		 *      object property if the function called returns an object. Note that
		 *      function notation is recommended for use in `render` rather than
		 *      `data` as it is much simpler to use as a renderer.
		 * * `null` - use the original data source for the row rather than plucking
		 *   data directly from it. This action has effects on two other
		 *   initialisation options:
		 *    * `defaultContent` - When null is given as the `data` option and
		 *      `defaultContent` is specified for the column, the value defined by
		 *      `defaultContent` will be used for the cell.
		 *    * `render` - When null is used for the `data` option and the `render`
		 *      option is specified for the column, the whole data source for the
		 *      row is used for the renderer.
		 * * `function` - the function given will be executed whenever DataTables
		 *   needs to set or get the data for a cell in the column. The function
		 *   takes three parameters:
		 *    * Parameters:
		 *      * `{array|object}` The data source for the row
		 *      * `{string}` The type call data requested - this will be 'set' when
		 *        setting data or 'filter', 'display', 'type', 'sort' or undefined
		 *        when gathering data. Note that when `undefined` is given for the
		 *        type DataTables expects to get the raw data for the object back<
		 *      * `{*}` Data to set when the second parameter is 'set'.
		 *    * Return:
		 *      * The return value from the function is not required when 'set' is
		 *        the type of call, but otherwise the return is what will be used
		 *        for the data requested.
		 *
		 * Note that `data` is a getter and setter option. If you just require
		 * formatting of data for output, you will likely want to use `render` which
		 * is simply a getter and thus simpler to use.
		 *
		 * Note that prior to DataTables 1.9.2 `data` was called `mDataProp`. The
		 * name change reflects the flexibility of this property and is consistent
		 * with the naming of mRender. If 'mDataProp' is given, then it will still
		 * be used by DataTables, as it automatically maps the old name to the new
		 * if required.
		 */
		"mData": null,
	
	
		/**
		 * This property is the rendering partner to `data` and it is suggested that
		 * when you want to manipulate data for display (including filtering,
		 * sorting etc) without altering the underlying data for the table, use this
		 * property. `render` can be considered to be the the read only companion to
		 * `data` which is read / write (then as such more complex). Like `data`
		 * this option can be given in a number of different ways to effect its
		 * behaviour:
		 *
		 * * `integer` - treated as an array index for the data source. This is the
		 *   default that DataTables uses (incrementally increased for each column).
		 * * `string` - read an object property from the data source. There are
		 *   three 'special' options that can be used in the string to alter how
		 *   DataTables reads the data from the source object:
		 *    * `.` - Dotted Javascript notation. Just as you use a `.` in
		 *      Javascript to read from nested objects, so to can the options
		 *      specified in `data`. For example: `browser.version` or
		 *      `browser.name`. If your object parameter name contains a period, use
		 *      `\\` to escape it - i.e. `first\\.name`.
		 *    * `[]` - Array notation. DataTables can automatically combine data
		 *      from and array source, joining the data with the characters provided
		 *      between the two brackets. For example: `name[, ]` would provide a
		 *      comma-space separated list from the source array. If no characters
		 *      are provided between the brackets, the original array source is
		 *      returned.
		 *    * `()` - Function notation. Adding `()` to the end of a parameter will
		 *      execute a function of the name given. For example: `browser()` for a
		 *      simple function on the data source, `browser.version()` for a
		 *      function in a nested property or even `browser().version` to get an
		 *      object property if the function called returns an object.
		 * * `object` - use different data for the different data types requested by
		 *   DataTables ('filter', 'display', 'type' or 'sort'). The property names
		 *   of the object is the data type the property refers to and the value can
		 *   defined using an integer, string or function using the same rules as
		 *   `render` normally does. Note that an `_` option _must_ be specified.
		 *   This is the default value to use if you haven't specified a value for
		 *   the data type requested by DataTables.
		 * * `function` - the function given will be executed whenever DataTables
		 *   needs to set or get the data for a cell in the column. The function
		 *   takes three parameters:
		 *    * Parameters:
		 *      * {array|object} The data source for the row (based on `data`)
		 *      * {string} The type call data requested - this will be 'filter',
		 *        'display', 'type' or 'sort'.
		 *      * {array|object} The full data source for the row (not based on
		 *        `data`)
		 *    * Return:
		 *      * The return value from the function is what will be used for the
		 *        data requested.
		 */
		"mRender": null,
	
	
		/**
		 * Change the cell type created for the column - either TD cells or TH cells. This
		 * can be useful as TH cells have semantic meaning in the table body, allowing them
		 * to act as a header for a row (you may wish to add scope='row' to the TH elements).
		 */
		"sCellType": "td",
	
	
		/**
		 * Class to give to each cell in this column.
		 */
		"sClass": "",
	
		/**
		 * When DataTables calculates the column widths to assign to each column,
		 * it finds the longest string in each column and then constructs a
		 * temporary table and reads the widths from that. The problem with this
		 * is that "mmm" is much wider then "iiii", but the latter is a longer
		 * string - thus the calculation can go wrong (doing it properly and putting
		 * it into an DOM object and measuring that is horribly(!) slow). Thus as
		 * a "work around" we provide this option. It will append its value to the
		 * text that is found to be the longest string for the column - i.e. padding.
		 * Generally you shouldn't need this!
		 */
		"sContentPadding": "",
	
	
		/**
		 * Allows a default value to be given for a column's data, and will be used
		 * whenever a null data source is encountered (this can be because `data`
		 * is set to null, or because the data source itself is null).
		 */
		"sDefaultContent": null,
	
	
		/**
		 * This parameter is only used in DataTables' server-side processing. It can
		 * be exceptionally useful to know what columns are being displayed on the
		 * client side, and to map these to database fields. When defined, the names
		 * also allow DataTables to reorder information from the server if it comes
		 * back in an unexpected order (i.e. if you switch your columns around on the
		 * client-side, your server-side code does not also need updating).
		 */
		"sName": "",
	
	
		/**
		 * Defines a data source type for the ordering which can be used to read
		 * real-time information from the table (updating the internally cached
		 * version) prior to ordering. This allows ordering to occur on user
		 * editable elements such as form inputs.
		 */
		"sSortDataType": "std",
	
	
		/**
		 * The title of this column.
		 */
		"sTitle": null,
	
	
		/**
		 * The type allows you to specify how the data for this column will be
		 * ordered. Four types (string, numeric, date and html (which will strip
		 * HTML tags before ordering)) are currently available. Note that only date
		 * formats understood by Javascript's Date() object will be accepted as type
		 * date. For example: "Mar 26, 2008 5:03 PM". May take the values: 'string',
		 * 'numeric', 'date' or 'html' (by default). Further types can be adding
		 * through plug-ins.
		 */
		"sType": null,
	
	
		/**
		 * Defining the width of the column, this parameter may take any CSS value
		 * (3em, 20px etc). DataTables applies 'smart' widths to columns which have not
		 * been given a specific width through this interface ensuring that the table
		 * remains readable.
		 */
		"sWidth": null
	};
	
	_fnHungarianMap( DataTable.defaults.column );
	
	
	
	/**
	 * DataTables settings object - this holds all the information needed for a
	 * given table, including configuration, data and current application of the
	 * table options. DataTables does not have a single instance for each DataTable
	 * with the settings attached to that instance, but rather instances of the
	 * DataTable "class" are created on-the-fly as needed (typically by a
	 * $().dataTable() call) and the settings object is then applied to that
	 * instance.
	 *
	 * Note that this object is related to {@link DataTable.defaults} but this
	 * one is the internal data store for DataTables's cache of columns. It should
	 * NOT be manipulated outside of DataTables. Any configuration should be done
	 * through the initialisation options.
	 */
	DataTable.models.oSettings = {
		/**
		 * Primary features of DataTables and their enablement state.
		 */
		"oFeatures": {
	
			/**
			 * Flag to say if DataTables should automatically try to calculate the
			 * optimum table and columns widths (true) or not (false).
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bAutoWidth": null,
	
			/**
			 * Delay the creation of TR and TD elements until they are actually
			 * needed by a driven page draw. This can give a significant speed
			 * increase for Ajax source and Javascript source data, but makes no
			 * difference at all for DOM and server-side processing tables.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bDeferRender": null,
	
			/**
			 * Enable filtering on the table or not. Note that if this is disabled
			 * then there is no filtering at all on the table, including fnFilter.
			 * To just remove the filtering input use sDom and remove the 'f' option.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bFilter": null,
	
			/**
			 * Used only for compatiblity with DT1
			 * @deprecated
			 */
			"bInfo": true,
	
			/**
			 * Used only for compatiblity with DT1
			 * @deprecated
			 */
			"bLengthChange": true,
	
			/**
			 * Pagination enabled or not. Note that if this is disabled then length
			 * changing must also be disabled.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bPaginate": null,
	
			/**
			 * Processing indicator enable flag whenever DataTables is enacting a
			 * user request - typically an Ajax request for server-side processing.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bProcessing": null,
	
			/**
			 * Server-side processing enabled flag - when enabled DataTables will
			 * get all data from the server for every draw - there is no filtering,
			 * sorting or paging done on the client-side.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bServerSide": null,
	
			/**
			 * Sorting enablement flag.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bSort": null,
	
			/**
			 * Multi-column sorting
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bSortMulti": null,
	
			/**
			 * Apply a class to the columns which are being sorted to provide a
			 * visual highlight or not. This can slow things down when enabled since
			 * there is a lot of DOM interaction.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bSortClasses": null,
	
			/**
			 * State saving enablement flag.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bStateSave": null
		},
	
	
		/**
		 * Scrolling settings for a table.
		 */
		"oScroll": {
			/**
			 * When the table is shorter in height than sScrollY, collapse the
			 * table container down to the height of the table (when true).
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"bCollapse": null,
	
			/**
			 * Width of the scrollbar for the web-browser's platform. Calculated
			 * during table initialisation.
			 */
			"iBarWidth": 0,
	
			/**
			 * Viewport width for horizontal scrolling. Horizontal scrolling is
			 * disabled if an empty string.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"sX": null,
	
			/**
			 * Width to expand the table to when using x-scrolling. Typically you
			 * should not need to use this.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @deprecated
			 */
			"sXInner": null,
	
			/**
			 * Viewport height for vertical scrolling. Vertical scrolling is disabled
			 * if an empty string.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 */
			"sY": null
		},
	
		/**
		 * Language information for the table.
		 */
		"oLanguage": {
			/**
			 * Information callback function. See
			 * {@link DataTable.defaults.fnInfoCallback}
			 */
			"fnInfoCallback": null
		},
	
		/**
		 * Browser support parameters
		 */
		"oBrowser": {
			/**
			 * Determine if the vertical scrollbar is on the right or left of the
			 * scrolling container - needed for rtl language layout, although not
			 * all browsers move the scrollbar (Safari).
			 */
			"bScrollbarLeft": false,
	
			/**
			 * Browser scrollbar width
			 */
			"barWidth": 0
		},
	
	
		"ajax": null,
	
	
		/**
		 * Array referencing the nodes which are used for the features. The
		 * parameters of this object match what is allowed by sDom - i.e.
		 *   <ul>
		 *     <li>'l' - Length changing</li>
		 *     <li>'f' - Filtering input</li>
		 *     <li>'t' - The table!</li>
		 *     <li>'i' - Information</li>
		 *     <li>'p' - Pagination</li>
		 *     <li>'r' - pRocessing</li>
		 *   </ul>
		 */
		"aanFeatures": [],
	
		/**
		 * Store data information - see {@link DataTable.models.oRow} for detailed
		 * information.
		 */
		"aoData": [],
	
		/**
		 * Array of indexes which are in the current display (after filtering etc)
		 */
		"aiDisplay": [],
	
		/**
		 * Array of indexes for display - no filtering
		 */
		"aiDisplayMaster": [],
	
		/**
		 * Map of row ids to data indexes
		 */
		"aIds": {},
	
		/**
		 * Store information about each column that is in use
		 */
		"aoColumns": [],
	
		/**
		 * Store information about the table's header
		 */
		"aoHeader": [],
	
		/**
		 * Store information about the table's footer
		 */
		"aoFooter": [],
	
		/**
		 * Store the applied global search information in case we want to force a
		 * research or compare the old search to a new one.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 */
		"oPreviousSearch": {},
	
		/**
		 * Store for named searches
		 */
		searchFixed: {},
	
		/**
		 * Store the applied search for each column - see
		 * {@link DataTable.models.oSearch} for the format that is used for the
		 * filtering information for each column.
		 */
		"aoPreSearchCols": [],
	
		/**
		 * Sorting that is applied to the table. Note that the inner arrays are
		 * used in the following manner:
		 * <ul>
		 *   <li>Index 0 - column number</li>
		 *   <li>Index 1 - current sorting direction</li>
		 * </ul>
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 */
		"aaSorting": null,
	
		/**
		 * Sorting that is always applied to the table (i.e. prefixed in front of
		 * aaSorting).
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 */
		"aaSortingFixed": [],
	
		/**
		 * If restoring a table - we should restore its width
		 */
		"sDestroyWidth": 0,
	
		/**
		 * Callback functions array for every time a row is inserted (i.e. on a draw).
		 */
		"aoRowCallback": [],
	
		/**
		 * Callback functions for the header on each draw.
		 */
		"aoHeaderCallback": [],
	
		/**
		 * Callback function for the footer on each draw.
		 */
		"aoFooterCallback": [],
	
		/**
		 * Array of callback functions for draw callback functions
		 */
		"aoDrawCallback": [],
	
		/**
		 * Array of callback functions for row created function
		 */
		"aoRowCreatedCallback": [],
	
		/**
		 * Callback functions for just before the table is redrawn. A return of
		 * false will be used to cancel the draw.
		 */
		"aoPreDrawCallback": [],
	
		/**
		 * Callback functions for when the table has been initialised.
		 */
		"aoInitComplete": [],
	
	
		/**
		 * Callbacks for modifying the settings to be stored for state saving, prior to
		 * saving state.
		 */
		"aoStateSaveParams": [],
	
		/**
		 * Callbacks for modifying the settings that have been stored for state saving
		 * prior to using the stored values to restore the state.
		 */
		"aoStateLoadParams": [],
	
		/**
		 * Callbacks for operating on the settings object once the saved state has been
		 * loaded
		 */
		"aoStateLoaded": [],
	
		/**
		 * Cache the table ID for quick access
		 */
		"sTableId": "",
	
		/**
		 * The TABLE node for the main table
		 */
		"nTable": null,
	
		/**
		 * Permanent ref to the thead element
		 */
		"nTHead": null,
	
		/**
		 * Permanent ref to the tfoot element - if it exists
		 */
		"nTFoot": null,
	
		/**
		 * Permanent ref to the tbody element
		 */
		"nTBody": null,
	
		/**
		 * Cache the wrapper node (contains all DataTables controlled elements)
		 */
		"nTableWrapper": null,
	
		/**
		 * Indicate if all required information has been read in
		 */
		"bInitialised": false,
	
		/**
		 * Information about open rows. Each object in the array has the parameters
		 * 'nTr' and 'nParent'
		 */
		"aoOpenRows": [],
	
		/**
		 * Dictate the positioning of DataTables' control elements - see
		 * {@link DataTable.model.oInit.sDom}.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 */
		"sDom": null,
	
		/**
		 * Search delay (in mS)
		 */
		"searchDelay": null,
	
		/**
		 * Which type of pagination should be used.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 */
		"sPaginationType": "two_button",
	
		/**
		 * Number of paging controls on the page. Only used for backwards compatibility
		 */
		pagingControls: 0,
	
		/**
		 * The state duration (for `stateSave`) in seconds.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 */
		"iStateDuration": 0,
	
		/**
		 * Array of callback functions for state saving. Each array element is an
		 * object with the following parameters:
		 *   <ul>
		 *     <li>function:fn - function to call. Takes two parameters, oSettings
		 *       and the JSON string to save that has been thus far created. Returns
		 *       a JSON string to be inserted into a json object
		 *       (i.e. '"param": [ 0, 1, 2]')</li>
		 *     <li>string:sName - name of callback</li>
		 *   </ul>
		 */
		"aoStateSave": [],
	
		/**
		 * Array of callback functions for state loading. Each array element is an
		 * object with the following parameters:
		 *   <ul>
		 *     <li>function:fn - function to call. Takes two parameters, oSettings
		 *       and the object stored. May return false to cancel state loading</li>
		 *     <li>string:sName - name of callback</li>
		 *   </ul>
		 */
		"aoStateLoad": [],
	
		/**
		 * State that was saved. Useful for back reference
		 */
		"oSavedState": null,
	
		/**
		 * State that was loaded. Useful for back reference
		 */
		"oLoadedState": null,
	
		/**
		 * Note if draw should be blocked while getting data
		 */
		"bAjaxDataGet": true,
	
		/**
		 * The last jQuery XHR object that was used for server-side data gathering.
		 * This can be used for working with the XHR information in one of the
		 * callbacks
		 */
		"jqXHR": null,
	
		/**
		 * JSON returned from the server in the last Ajax request
		 */
		"json": undefined,
	
		/**
		 * Data submitted as part of the last Ajax request
		 */
		"oAjaxData": undefined,
	
		/**
		 * Send the XHR HTTP method - GET or POST (could be PUT or DELETE if
		 * required).
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 */
		"sServerMethod": null,
	
		/**
		 * Format numbers for display.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 */
		"fnFormatNumber": null,
	
		/**
		 * List of options that can be used for the user selectable length menu.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 */
		"aLengthMenu": null,
	
		/**
		 * Counter for the draws that the table does. Also used as a tracker for
		 * server-side processing
		 */
		"iDraw": 0,
	
		/**
		 * Indicate if a redraw is being done - useful for Ajax
		 */
		"bDrawing": false,
	
		/**
		 * Draw index (iDraw) of the last error when parsing the returned data
		 */
		"iDrawError": -1,
	
		/**
		 * Paging display length
		 */
		"_iDisplayLength": 10,
	
		/**
		 * Paging start point - aiDisplay index
		 */
		"_iDisplayStart": 0,
	
		/**
		 * Server-side processing - number of records in the result set
		 * (i.e. before filtering), Use fnRecordsTotal rather than
		 * this property to get the value of the number of records, regardless of
		 * the server-side processing setting.
		 */
		"_iRecordsTotal": 0,
	
		/**
		 * Server-side processing - number of records in the current display set
		 * (i.e. after filtering). Use fnRecordsDisplay rather than
		 * this property to get the value of the number of records, regardless of
		 * the server-side processing setting.
		 */
		"_iRecordsDisplay": 0,
	
		/**
		 * The classes to use for the table
		 */
		"oClasses": {},
	
		/**
		 * Flag attached to the settings object so you can check in the draw
		 * callback if filtering has been done in the draw. Deprecated in favour of
		 * events.
		 *  @deprecated
		 */
		"bFiltered": false,
	
		/**
		 * Flag attached to the settings object so you can check in the draw
		 * callback if sorting has been done in the draw. Deprecated in favour of
		 * events.
		 *  @deprecated
		 */
		"bSorted": false,
	
		/**
		 * Indicate that if multiple rows are in the header and there is more than
		 * one unique cell per column. Replaced by titleRow
		 */
		"bSortCellsTop": null,
	
		/**
		 * Initialisation object that is used for the table
		 */
		"oInit": null,
	
		/**
		 * Destroy callback functions - for plug-ins to attach themselves to the
		 * destroy so they can clean up markup and events.
		 */
		"aoDestroyCallback": [],
	
	
		/**
		 * Get the number of records in the current record set, before filtering
		 */
		"fnRecordsTotal": function ()
		{
			return _fnDataSource( this ) == 'ssp' ?
				this._iRecordsTotal * 1 :
				this.aiDisplayMaster.length;
		},
	
		/**
		 * Get the number of records in the current record set, after filtering
		 */
		"fnRecordsDisplay": function ()
		{
			return _fnDataSource( this ) == 'ssp' ?
				this._iRecordsDisplay * 1 :
				this.aiDisplay.length;
		},
	
		/**
		 * Get the display end point - aiDisplay index
		 */
		"fnDisplayEnd": function ()
		{
			var
				len      = this._iDisplayLength,
				start    = this._iDisplayStart,
				calc     = start + len,
				records  = this.aiDisplay.length,
				features = this.oFeatures,
				paginate = features.bPaginate;
	
			if ( features.bServerSide ) {
				return paginate === false || len === -1 ?
					start + records :
					Math.min( start+len, this._iRecordsDisplay );
			}
			else {
				return ! paginate || calc>records || len===-1 ?
					records :
					calc;
			}
		},
	
		/**
		 * The DataTables object for this table
		 */
		"oInstance": null,
	
		/**
		 * Unique identifier for each instance of the DataTables object. If there
		 * is an ID on the table node, then it takes that value, otherwise an
		 * incrementing internal counter is used.
		 */
		"sInstance": null,
	
		/**
		 * tabindex attribute value that is added to DataTables control elements, allowing
		 * keyboard navigation of the table and its controls.
		 */
		"iTabIndex": 0,
	
		/**
		 * DIV container for the footer scrolling table if scrolling
		 */
		"nScrollHead": null,
	
		/**
		 * DIV container for the footer scrolling table if scrolling
		 */
		"nScrollFoot": null,
	
		/**
		 * Last applied sort
		 */
		"aLastSort": [],
	
		/**
		 * Stored plug-in instances
		 */
		"oPlugins": {},
	
		/**
		 * Function used to get a row's id from the row's data
		 */
		"rowIdFn": null,
	
		/**
		 * Data location where to store a row's id
		 */
		"rowId": null,
	
		caption: '',
	
		captionNode: null,
	
		colgroup: null,
	
		/** Delay loading of data */
		deferLoading: null,
	
		/** Allow auto type detection */
		typeDetect: true,
	
		/** ResizeObserver for the container div */
		resizeObserver: null,
	
		/** Keep a record of the last size of the container, so we can skip duplicates */
		containerWidth: -1,
	
		/** Reverse the initial order of the data set on desc ordering */
		orderDescReverse: null,
	
		/** Show / hide ordering indicators in headers */
		orderIndicators: true,
	
		/** Default ordering listener */
		orderHandler: true,
	
		/** Title row indicator */
		titleRow: null
	};
	
	/**
	 * Extension object for DataTables that is used to provide all extension
	 * options.
	 *
	 * Note that the `DataTable.ext` object is available through
	 * `jQuery.fn.dataTable.ext` where it may be accessed and manipulated. It is
	 * also aliased to `jQuery.fn.dataTableExt` for historic reasons.
	 *  @namespace
	 *  @extends DataTable.models.ext
	 */
	
	
	var extPagination = DataTable.ext.pager;
	
	// Paging buttons configuration
	$.extend( extPagination, {
		simple: function () {
			return [ 'previous', 'next' ];
		},
	
		full: function () {
			return [ 'first', 'previous', 'next', 'last' ];
		},
	
		numbers: function () {
			return [ 'numbers' ];
		},
	
		simple_numbers: function () {
			return [ 'previous', 'numbers', 'next' ];
		},
	
		full_numbers: function () {
			return [ 'first', 'previous', 'numbers', 'next', 'last' ];
		},
	
		first_last: function () {
			return ['first', 'last'];
		},
	
		first_last_numbers: function () {
			return ['first', 'numbers', 'last'];
		},
	
		// For testing and plug-ins to use
		_numbers: _pagingNumbers,
	
		// Number of number buttons - legacy, use `numbers` option for paging feature
		numbers_length: 7
	} );
	
	
	$.extend( true, DataTable.ext.renderer, {
		pagingButton: {
			_: function (settings, buttonType, content, active, disabled) {
				var classes = settings.oClasses.paging;
				var btnClasses = [classes.button];
				var btn;
	
				if (active) {
					btnClasses.push(classes.active);
				}
	
				if (disabled) {
					btnClasses.push(classes.disabled)
				}
	
				if (buttonType === 'ellipsis') {
					btn = $('<span class="ellipsis"></span>').html(content)[0];
				}
				else {
					btn = $('<button>', {
						class: btnClasses.join(' '),
						role: 'link',
						type: 'button'
					}).html(content);
				}
	
				return {
					display: btn,
					clicker: btn
				}
			}
		},
	
		pagingContainer: {
			_: function (settings, buttons) {
				// No wrapping element - just append directly to the host
				return buttons;
			}
		}
	} );
	
	// Common function to remove new lines, strip HTML and diacritic control
	var _filterString = function (stripHtml, normalize) {
		return function (str) {
			if (_empty(str) || typeof str !== 'string') {
				return str;
			}
	
			str = str.replace( _re_new_lines, " " );
	
			if (stripHtml) {
				str = _stripHtml(str);
			}
	
			if (normalize) {
				str = _normalize(str, false);
			}
	
			return str;
		};
	}
	
	/*
	 * Public helper functions. These aren't used internally by DataTables, or
	 * called by any of the options passed into DataTables, but they can be used
	 * externally by developers working with DataTables. They are helper functions
	 * to make working with DataTables a little bit easier.
	 */
	
	/**
	 * Common logic for moment, luxon or a date action.
	 *
	 * Happens after __mldObj, so don't need to call `resolveWindowsLibs` again
	 */
	function __mld( dtLib, momentFn, luxonFn, dateFn, arg1 ) {
		if (__moment) {
			return dtLib[momentFn]( arg1 );
		}
		else if (__luxon) {
			return dtLib[luxonFn]( arg1 );
		}
		
		return dateFn ? dtLib[dateFn]( arg1 ) : dtLib;
	}
	
	
	var __mlWarning = false;
	var __luxon; // Can be assigned in DateTable.use()
	var __moment; // Can be assigned in DateTable.use()
	
	/**
	 * 
	 */
	function resolveWindowLibs() {
		if (window.luxon && ! __luxon) {
			__luxon = window.luxon;
		}
		
		if (window.moment && ! __moment) {
			__moment = window.moment;
		}
	}
	
	function __mldObj (d, format, locale) {
		var dt;
	
		resolveWindowLibs();
	
		if (__moment) {
			dt = __moment.utc( d, format, locale, true );
	
			if (! dt.isValid()) {
				return null;
			}
		}
		else if (__luxon) {
			dt = format && typeof d === 'string'
				? __luxon.DateTime.fromFormat( d, format )
				: __luxon.DateTime.fromISO( d );
	
			if (! dt.isValid) {
				return null;
			}
	
			dt = dt.setLocale(locale);
		}
		else if (! format) {
			// No format given, must be ISO
			dt = new Date(d);
		}
		else {
			if (! __mlWarning) {
				alert('DataTables warning: Formatted date without Moment.js or Luxon - https://datatables.net/tn/17');
			}
	
			__mlWarning = true;
		}
	
		return dt;
	}
	
	// Wrapper for date, datetime and time which all operate the same way with the exception of
	// the output string for auto locale support
	function __mlHelper (localeString) {
		return function ( from, to, locale, def ) {
			// Luxon and Moment support
			// Argument shifting
			if ( arguments.length === 0 ) {
				locale = 'en';
				to = null; // means toLocaleString
				from = null; // means iso8601
			}
			else if ( arguments.length === 1 ) {
				locale = 'en';
				to = from;
				from = null;
			}
			else if ( arguments.length === 2 ) {
				locale = to;
				to = from;
				from = null;
			}
	
			var typeName = 'datetime' + (to ? '-' + to : '');
	
			// Add type detection and sorting specific to this date format - we need to be able to identify
			// date type columns as such, rather than as numbers in extensions. Hence the need for this.
			if (! DataTable.ext.type.order[typeName + '-pre']) {
				DataTable.type(typeName, {
					detect: function (d) {
						// The renderer will give the value to type detect as the type!
						return d === typeName ? typeName : false;
					},
					order: {
						pre: function (d) {
							// The renderer gives us Moment, Luxon or Date obects for the sorting, all of which have a
							// `valueOf` which gives milliseconds epoch
							return d.valueOf();
						}
					},
					className: 'dt-right'
				});
			}
		
			return function ( d, type ) {
				// Allow for a default value
				if (d === null || d === undefined) {
					if (def === '--now') {
						// We treat everything as UTC further down, so no changes are
						// made, as such need to get the local date / time as if it were
						// UTC
						var local = new Date();
						d = new Date( Date.UTC(
							local.getFullYear(), local.getMonth(), local.getDate(),
							local.getHours(), local.getMinutes(), local.getSeconds()
						) );
					}
					else {
						d = '';
					}
				}
	
				if (type === 'type') {
					// Typing uses the type name for fast matching
					return typeName;
				}
	
				if (d === '') {
					return type !== 'sort'
						? ''
						: __mldObj('0000-01-01 00:00:00', null, locale);
				}
	
				// Shortcut. If `from` and `to` are the same, we are using the renderer to
				// format for ordering, not display - its already in the display format.
				if ( to !== null && from === to && type !== 'sort' && type !== 'type' && ! (d instanceof Date) ) {
					return d;
				}
	
				var dt = __mldObj(d, from, locale);
	
				if (dt === null) {
					return d;
				}
	
				if (type === 'sort') {
					return dt;
				}
				
				var formatted = to === null
					? __mld(dt, 'toDate', 'toJSDate', '')[localeString]()
					: __mld(dt, 'format', 'toFormat', 'toISOString', to);
	
				// XSS protection
				return type === 'display' ?
					_escapeHtml( formatted ) :
					formatted;
			};
		}
	}
	
	// Based on locale, determine standard number formatting
	// Fallback for legacy browsers is US English
	var __thousands = ',';
	var __decimal = '.';
	
	if (window.Intl !== undefined) {
		try {
			var num = new Intl.NumberFormat().formatToParts(100000.1);
		
			for (var i=0 ; i<num.length ; i++) {
				if (num[i].type === 'group') {
					__thousands = num[i].value;
				}
				else if (num[i].type === 'decimal') {
					__decimal = num[i].value;
				}
			}
		}
		catch (e) {
			// noop
		}
	}
	
	// Formatted date time detection - use by declaring the formats you are going to use
	DataTable.datetime = function ( format, locale ) {
		var typeName = 'datetime-' + format;
	
		if (! locale) {
			locale = 'en';
		}
	
		if (! DataTable.ext.type.order[typeName]) {
			DataTable.type(typeName, {
				detect: function (d) {
					var dt = __mldObj(d, format, locale);
					return d === '' || dt ? typeName : false;
				},
				order: {
					pre: function (d) {
						return __mldObj(d, format, locale) || 0;
					}
				},
				className: 'dt-right'
			});
		}
	}
	
	/**
	 * Helpers for `columns.render`.
	 *
	 * The options defined here can be used with the `columns.render` initialisation
	 * option to provide a display renderer. The following functions are defined:
	 *
	 * * `moment` - Uses the MomentJS library to convert from a given format into another.
	 * This renderer has three overloads:
	 *   * 1 parameter:
	 *     * `string` - Format to convert to (assumes input is ISO8601 and locale is `en`)
	 *   * 2 parameters:
	 *     * `string` - Format to convert from
	 *     * `string` - Format to convert to. Assumes `en` locale
	 *   * 3 parameters:
	 *     * `string` - Format to convert from
	 *     * `string` - Format to convert to
	 *     * `string` - Locale
	 * * `number` - Will format numeric data (defined by `columns.data`) for
	 *   display, retaining the original unformatted data for sorting and filtering.
	 *   It takes 5 parameters:
	 *   * `string` - Thousands grouping separator
	 *   * `string` - Decimal point indicator
	 *   * `integer` - Number of decimal points to show
	 *   * `string` (optional) - Prefix.
	 *   * `string` (optional) - Postfix (/suffix).
	 * * `text` - Escape HTML to help prevent XSS attacks. It has no optional
	 *   parameters.
	 *
	 * @example
	 *   // Column definition using the number renderer
	 *   {
	 *     data: "salary",
	 *     render: $.fn.dataTable.render.number( '\'', '.', 0, '$' )
	 *   }
	 *
	 * @namespace
	 */
	DataTable.render = {
		date: __mlHelper('toLocaleDateString'),
		datetime: __mlHelper('toLocaleString'),
		time: __mlHelper('toLocaleTimeString'),
		number: function ( thousands, decimal, precision, prefix, postfix ) {
			// Auto locale detection
			if (thousands === null || thousands === undefined) {
				thousands = __thousands;
			}
	
			if (decimal === null || decimal === undefined) {
				decimal = __decimal;
			}
	
			return {
				display: function ( d ) {
					if ( typeof d !== 'number' && typeof d !== 'string' ) {
						return d;
					}
	
					if (d === '' || d === null) {
						return d;
					}
	
					var negative = d < 0 ? '-' : '';
					var flo = parseFloat( d );
					var abs = Math.abs(flo);
	
					// Scientific notation for large and small numbers
					if (abs >= 100000000000 || (abs < 0.0001 && abs !== 0) ) {
						var exp = flo.toExponential(precision).split(/e\+?/);
						return exp[0] + ' x 10<sup>' + exp[1] + '</sup>';
					}
	
					// If NaN then there isn't much formatting that we can do - just
					// return immediately, escaping any HTML (this was supposed to
					// be a number after all)
					if ( isNaN( flo ) ) {
						return _escapeHtml( d );
					}
	
					flo = flo.toFixed( precision );
					d = Math.abs( flo );
	
					var intPart = parseInt( d, 10 );
					var floatPart = precision ?
						decimal+(d - intPart).toFixed( precision ).substring( 2 ):
						'';
	
					// If zero, then can't have a negative prefix
					if (intPart === 0 && parseFloat(floatPart) === 0) {
						negative = '';
					}
	
					return negative + (prefix||'') +
						intPart.toString().replace(
							/\B(?=(\d{3})+(?!\d))/g, thousands
						) +
						floatPart +
						(postfix||'');
				}
			};
		},
	
		text: function () {
			return {
				display: _escapeHtml,
				filter: _escapeHtml
			};
		}
	};
	
	
	var _extTypes = DataTable.ext.type;
	
	// Get / set type
	DataTable.type = function (name, prop, val) {
		if (! prop) {
			return {
				className: _extTypes.className[name],
				detect: _extTypes.detect.find(function (fn) {
					return fn._name === name;
				}),
				order: {
					pre: _extTypes.order[name + '-pre'],
					asc: _extTypes.order[name + '-asc'],
					desc: _extTypes.order[name + '-desc']
				},
				render: _extTypes.render[name],
				search: _extTypes.search[name]
			};
		}
	
		var setProp = function(prop, propVal) {
			_extTypes[prop][name] = propVal;
		};
		var setDetect = function (detect) {
			// `detect` can be a function or an object - we set a name
			// property for either - that is used for the detection
			Object.defineProperty(detect, "_name", {value: name});
	
			var idx = _extTypes.detect.findIndex(function (item) {
				return item._name === name;
			});
	
			if (idx === -1) {
				_extTypes.detect.unshift(detect);
			}
			else {
				_extTypes.detect.splice(idx, 1, detect);
			}
		};
		var setOrder = function (obj) {
			_extTypes.order[name + '-pre'] = obj.pre; // can be undefined
			_extTypes.order[name + '-asc'] = obj.asc; // can be undefined
			_extTypes.order[name + '-desc'] = obj.desc; // can be undefined
		};
	
		// prop is optional
		if (val === undefined) {
			val = prop;
			prop = null;
		}
	
		if (prop === 'className') {
			setProp('className', val);
		}
		else if (prop === 'detect') {
			setDetect(val);
		}
		else if (prop === 'order') {
			setOrder(val);
		}
		else if (prop === 'render') {
			setProp('render', val);
		}
		else if (prop === 'search') {
			setProp('search', val);
		}
		else if (! prop) {
			if (val.className) {
				setProp('className', val.className);
			}
	
			if (val.detect !== undefined) {
				setDetect(val.detect);
			}
	
			if (val.order) {
				setOrder(val.order);
			}
	
			if (val.render !== undefined) {
				setProp('render', val.render);
			}
	
			if (val.search !== undefined) {
				setProp('search', val.search);
			}
		}
	}
	
	// Get a list of types
	DataTable.types = function () {
		return _extTypes.detect.map(function (fn) {
			return fn._name;
		});
	};
	
	var __diacriticSort = function (a, b) {
		a = a !== null && a !== undefined ? a.toString().toLowerCase() : '';
		b = b !== null && b !== undefined ? b.toString().toLowerCase() : '';
	
		// Checked for `navigator.languages` support in `oneOf` so this code can't execute in old
		// Safari and thus can disable this check
		// eslint-disable-next-line compat/compat
		return a.localeCompare(b, navigator.languages[0] || navigator.language, {
			numeric: true,
			ignorePunctuation: true,
		});
	}
	
	var __diacriticHtmlSort = function (a, b) {
		a = _stripHtml(a);
		b = _stripHtml(b);
	
		return __diacriticSort(a, b);
	}
	
	//
	// Built in data types
	//
	
	DataTable.type('string', {
		detect: function () {
			return 'string';
		},
		order: {
			pre: function ( a ) {
				// This is a little complex, but faster than always calling toString,
				// http://jsperf.com/tostring-v-check
				return _empty(a) && typeof a !== 'boolean' ?
					'' :
					typeof a === 'string' ?
						a.toLowerCase() :
						! a.toString ?
							'' :
							a.toString();
			}
		},
		search: _filterString(false, true)
	});
	
	DataTable.type('string-utf8', {
		detect: {
			allOf: function ( d ) {
				return true;
			},
			oneOf: function ( d ) {
				// At least one data point must contain a non-ASCII character
				// This line will also check if navigator.languages is supported or not. If not (Safari 10.0-)
				// this data type won't be supported.
				// eslint-disable-next-line compat/compat
				return ! _empty( d ) && navigator.languages && typeof d === 'string' && d.match(/[^\x00-\x7F]/);
			}
		},
		order: {
			asc: __diacriticSort,
			desc: function (a, b) {
				return __diacriticSort(a, b) * -1;
			}
		},
		search: _filterString(false, true)
	});
	
	
	DataTable.type('html', {
		detect: {
			allOf: function ( d ) {
				return _empty( d ) || (typeof d === 'string' && d.indexOf('<') !== -1);
			},
			oneOf: function ( d ) {
				// At least one data point must contain a `<`
				return ! _empty( d ) && typeof d === 'string' && d.indexOf('<') !== -1;
			}
		},
		order: {
			pre: function ( a ) {
				return _empty(a) ?
					'' :
					a.replace ?
						_stripHtml(a).trim().toLowerCase() :
						a+'';
			}
		},
		search: _filterString(true, true)
	});
	
	
	DataTable.type('html-utf8', {
		detect: {
			allOf: function ( d ) {
				return _empty( d ) || (typeof d === 'string' && d.indexOf('<') !== -1);
			},
			oneOf: function ( d ) {
				// At least one data point must contain a `<` and a non-ASCII character
				// eslint-disable-next-line compat/compat
				return navigator.languages &&
					! _empty( d ) &&
					typeof d === 'string' &&
					d.indexOf('<') !== -1 &&
					typeof d === 'string' && d.match(/[^\x00-\x7F]/);
			}
		},
		order: {
			asc: __diacriticHtmlSort,
			desc: function (a, b) {
				return __diacriticHtmlSort(a, b) * -1;
			}
		},
		search: _filterString(true, true)
	});
	
	
	DataTable.type('date', {
		className: 'dt-type-date',
		detect: {
			allOf: function ( d ) {
				// V8 tries _very_ hard to make a string passed into `Date.parse()`
				// valid, so we need to use a regex to restrict date formats. Use a
				// plug-in for anything other than ISO8601 style strings
				if ( d && !(d instanceof Date) && ! _re_date.test(d) ) {
					return null;
				}
				var parsed = Date.parse(d);
				return (parsed !== null && !isNaN(parsed)) || _empty(d);
			},
			oneOf: function ( d ) {
				// At least one entry must be a date or a string with a date
				return (d instanceof Date) || (typeof d === 'string' && _re_date.test(d));
			}
		},
		order: {
			pre: function ( d ) {
				var ts = Date.parse( d );
				return isNaN(ts) ? -Infinity : ts;
			}
		}
	});
	
	
	DataTable.type('html-num-fmt', {
		className: 'dt-type-numeric',
		detect: {
			allOf: function ( d, settings ) {
				var decimal = settings.oLanguage.sDecimal;
				return _htmlNumeric( d, decimal, true, false );
			},
			oneOf: function (d, settings) {
				// At least one data point must contain a numeric value
				var decimal = settings.oLanguage.sDecimal;
				return _htmlNumeric( d, decimal, true, false );
			}
		},
		order: {
			pre: function ( d, s ) {
				var dp = s.oLanguage.sDecimal;
				return __numericReplace( d, dp, _re_html, _re_formatted_numeric );
			}
		},
		search: _filterString(true, true)
	});
	
	
	DataTable.type('html-num', {
		className: 'dt-type-numeric',
		detect: {
			allOf: function ( d, settings ) {
				var decimal = settings.oLanguage.sDecimal;
				return _htmlNumeric( d, decimal, false, true );
			},
			oneOf: function (d, settings) {
				// At least one data point must contain a numeric value
				var decimal = settings.oLanguage.sDecimal;
				return _htmlNumeric( d, decimal, false, false );
			}
		},
		order: {
			pre: function ( d, s ) {
				var dp = s.oLanguage.sDecimal;
				return __numericReplace( d, dp, _re_html );
			}
		},
		search: _filterString(true, true)
	});
	
	
	DataTable.type('num-fmt', {
		className: 'dt-type-numeric',
		detect: {
			allOf: function ( d, settings ) {
				var decimal = settings.oLanguage.sDecimal;
				return _isNumber( d, decimal, true, true );
			},
			oneOf: function (d, settings) {
				// At least one data point must contain a numeric value
				var decimal = settings.oLanguage.sDecimal;
				return _isNumber( d, decimal, true, false );
			}
		},
		order: {
			pre: function ( d, s ) {
				var dp = s.oLanguage.sDecimal;
				return __numericReplace( d, dp, _re_formatted_numeric );
			}
		}
	});
	
	
	DataTable.type('num', {
		className: 'dt-type-numeric',
		detect: {
			allOf: function ( d, settings ) {
				var decimal = settings.oLanguage.sDecimal;
				return _isNumber( d, decimal, false, true );
			},
			oneOf: function (d, settings) {
				// At least one data point must contain a numeric value
				var decimal = settings.oLanguage.sDecimal;
				return _isNumber( d, decimal, false, false );
			}
		},
		order: {
			pre: function (d, s) {
				var dp = s.oLanguage.sDecimal;
				return __numericReplace( d, dp );
			}
		}
	});
	
	
	
	
	var __numericReplace = function ( d, decimalPlace, re1, re2 ) {
		if ( d !== 0 && (!d || d === '-') ) {
			return -Infinity;
		}
		
		var type = typeof d;
	
		if (type === 'number' || type === 'bigint') {
			return d;
		}
	
		// If a decimal place other than `.` is used, it needs to be given to the
		// function so we can detect it and replace with a `.` which is the only
		// decimal place Javascript recognises - it is not locale aware.
		if ( decimalPlace ) {
			d = _numToDecimal( d, decimalPlace );
		}
	
		if ( d.replace ) {
			if ( re1 ) {
				d = d.replace( re1, '' );
			}
	
			if ( re2 ) {
				d = d.replace( re2, '' );
			}
		}
	
		return d * 1;
	};
	
	
	$.extend( true, DataTable.ext.renderer, {
		footer: {
			_: function ( settings, cell, classes ) {
				cell.addClass(classes.tfoot.cell);
			}
		},
	
		header: {
			_: function ( settings, cell, classes ) {
				cell.addClass(classes.thead.cell);
	
				if (! settings.oFeatures.bSort) {
					cell.addClass(classes.order.none);
				}
	
				var titleRow = settings.titleRow;
				var headerRows = cell.closest('thead').find('tr');
				var rowIdx = cell.parent().index();
	
				// Conditions to not apply the ordering icons
				if (
					// Cells and rows which have the attribute to disable the icons
					cell.attr('data-dt-order') === 'disable' ||
					cell.parent().attr('data-dt-order') === 'disable' ||
	
					// titleRow support, for defining a specific row in the header
					(titleRow === true && rowIdx !== 0) ||
					(titleRow === false && rowIdx !== headerRows.length - 1) ||
					(typeof titleRow === 'number' && rowIdx !== titleRow)
				) {
					return;
				}
	
				// No additional mark-up required
				// Attach a sort listener to update on sort - note that using the
				// `DT` namespace will allow the event to be removed automatically
				// on destroy, while the `dt` namespaced event is the one we are
				// listening for
				$(settings.nTable).on( 'order.dt.DT column-visibility.dt.DT', function ( e, ctx, column ) {
					if ( settings !== ctx ) { // need to check this this is the host
						return;               // table, not a nested one
					}
	
					var sorting = ctx.sortDetails;
	
					if (! sorting) {
						return;
					}
	
					var orderedColumns = _pluck(sorting, 'col');
	
					// This handler is only needed on column visibility if the column is part of the
					// ordering. If it isn't, then we can bail out to save performance. It could be a
					// separate event handler, but this is a balance between code reuse / size and performance
					// console.log(e, e.name, column, orderedColumns, orderedColumns.includes(column))
					if (e.type === 'column-visibility' && ! orderedColumns.includes(column)) {
						return;
					}
	
					var i;
					var orderClasses = classes.order;
					var columns = ctx.api.columns( cell );
					var col = settings.aoColumns[columns.flatten()[0]];
					var orderable = columns.orderable().includes(true);
					var ariaType = '';
					var indexes = columns.indexes();
					var sortDirs = columns.orderable(true).flatten();
					var tabIndex = settings.iTabIndex;
					var canOrder = ctx.orderHandler && orderable;
	
					cell
						.removeClass(
							orderClasses.isAsc +' '+
							orderClasses.isDesc
						)
						.toggleClass( orderClasses.none, ! orderable )
						.toggleClass( orderClasses.canAsc, canOrder && sortDirs.includes('asc') )
						.toggleClass( orderClasses.canDesc, canOrder && sortDirs.includes('desc') );
	
					// Determine if all of the columns that this cell covers are included in the
					// current ordering
					var isOrdering = true;
					
					for (i=0; i<indexes.length; i++) {
						if (! orderedColumns.includes(indexes[i])) {
							isOrdering = false;
						}
					}
	
					if ( isOrdering ) {
						// Get the ordering direction for the columns under this cell
						// Note that it is possible for a cell to be asc and desc sorting
						// (column spanning cells)
						var orderDirs = columns.order();
	
						cell.addClass(
							orderDirs.includes('asc') ? orderClasses.isAsc : '' +
							orderDirs.includes('desc') ? orderClasses.isDesc : ''
						);
					}
	
					// Find the first visible column that has ordering applied to it - it get's
					// the aria information, as the ARIA spec says that only one column should
					// be marked with aria-sort
					var firstVis = -1; // column index
	
					for (i=0; i<orderedColumns.length; i++) {
						if (settings.aoColumns[orderedColumns[i]].bVisible) {
							firstVis = orderedColumns[i];
							break;
						}
					}
	
					if (indexes[0] == firstVis) {
						var firstSort = sorting[0];
						var sortOrder = col.asSorting;
	
						cell.attr('aria-sort', firstSort.dir === 'asc' ? 'ascending' : 'descending');
	
						// Determine if the next click will remove sorting or change the sort
						ariaType = ! sortOrder[firstSort.index + 1] ? 'Remove' : 'Reverse';
					}
					else {
						cell.removeAttr('aria-sort');
					}
	
					// Make the headers tab-able for keyboard navigation
					if (orderable) {
						var orderSpan = cell.find('.dt-column-order');
						
						orderSpan
							.attr('role', 'button')
							.attr('aria-label', orderable
								? col.ariaTitle + ctx.api.i18n('oAria.orderable' + ariaType)
								: col.ariaTitle
							);
	
						if (tabIndex !== -1) {
							orderSpan.attr('tabindex', tabIndex);
						}
					}
				} );
			}
		},
	
		layout: {
			_: function ( settings, container, items ) {
				var classes = settings.oClasses.layout;
				var row = $('<div/>')
					.attr('id', items.id || null)
					.addClass(items.className || classes.row)
					.appendTo( container );
	
				DataTable.ext.renderer.layout._forLayoutRow(items, function (key, val) {
					if (key === 'id' || key === 'className') {
						return;
					}
	
					var klass = '';
	
					if (val.table) {
						row.addClass(classes.tableRow);
						klass += classes.tableCell + ' ';
					}
	
					if (key === 'start') {
						klass += classes.start;
					}
					else if (key === 'end') {
						klass += classes.end;
					}
					else {
						klass += classes.full;
					}
	
					$('<div/>')
						.attr({
							id: val.id || null,
							"class": val.className
								? val.className
								: classes.cell + ' ' + klass
						})
						.append( val.contents )
						.appendTo( row );
				});
			},
	
			// Shared for use by the styling frameworks
			_forLayoutRow: function (items, fn) {
				// As we are inserting dom elements, we need start / end in a
				// specific order, this function is used for sorting the layout
				// keys.
				var layoutEnum = function (x) {
					switch (x) {
						case '': return 0;
						case 'start': return 1;
						case 'end': return 2;
						default: return 3;
					}
				};
	
				Object
					.keys(items)
					.sort(function (a, b) {
						return layoutEnum(a) - layoutEnum(b);
					})
					.forEach(function (key) {
						fn(key, items[key]);
					});
			}
		}
	} );
	
	
	DataTable.feature = {};
	
	// Third parameter is internal only!
	DataTable.feature.register = function ( name, cb, legacy ) {
		DataTable.ext.features[ name ] = cb;
	
		if (legacy) {
			_ext.feature.push({
				cFeature: legacy,
				fnInit: cb
			});
		}
	};
	
	function _divProp(el, prop, val) {
		if (val) {
			el[prop] = val;
		}
	}
	
	DataTable.feature.register( 'div', function ( settings, opts ) {
		var n = $('<div>')[0];
	
		if (opts) {
			_divProp(n, 'className', opts.className);
			_divProp(n, 'id', opts.id);
			_divProp(n, 'innerHTML', opts.html);
			_divProp(n, 'textContent', opts.text);
		}
	
		return n;
	} );
	
	DataTable.feature.register( 'info', function ( settings, opts ) {
		// For compatibility with the legacy `info` top level option
		if (! settings.oFeatures.bInfo) {
			return null;
		}
	
		var
			lang  = settings.oLanguage,
			tid = settings.sTableId,
			n = $('<div/>', {
				'class': settings.oClasses.info.container,
			} );
	
		opts = $.extend({
			callback: lang.fnInfoCallback,
			empty: lang.sInfoEmpty,
			postfix: lang.sInfoPostFix,
			search: lang.sInfoFiltered,
			text: lang.sInfo,
		}, opts);
	
	
		// Update display on each draw
		settings.aoDrawCallback.push(function (s) {
			_fnUpdateInfo(s, opts, n);
		});
	
		// For the first info display in the table, we add a callback and aria information.
		if (! settings._infoEl) {
			n.attr({
				'aria-live': 'polite',
				id: tid+'_info',
				role: 'status'
			});
	
			// Table is described by our info div
			$(settings.nTable).attr( 'aria-describedby', tid+'_info' );
	
			settings._infoEl = n;
		}
	
		return n;
	}, 'i' );
	
	/**
	 * Update the information elements in the display
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnUpdateInfo ( settings, opts, node )
	{
		var
			start = settings._iDisplayStart+1,
			end   = settings.fnDisplayEnd(),
			max   = settings.fnRecordsTotal(),
			total = settings.fnRecordsDisplay(),
			out   = total
				? opts.text
				: opts.empty;
	
		if ( total !== max ) {
			// Record set after filtering
			out += ' ' + opts.search;
		}
	
		// Convert the macros
		out += opts.postfix;
		out = _fnMacros( settings, out );
	
		if ( opts.callback ) {
			out = opts.callback.call( settings.oInstance,
				settings, start, end, max, total, out
			);
		}
	
		node.html( out );
	
		_fnCallbackFire(settings, null, 'info', [settings, node[0], out]);
	}
	
	var __searchCounter = 0;
	
	// opts
	// - text
	// - placeholder
	DataTable.feature.register( 'search', function ( settings, opts ) {
		// Don't show the input if filtering isn't available on the table
		if (! settings.oFeatures.bFilter) {
			return null;
		}
	
		var classes = settings.oClasses.search;
		var tableId = settings.sTableId;
		var language = settings.oLanguage;
		var previousSearch = settings.oPreviousSearch;
		var input = '<input type="search" class="'+classes.input+'"/>';
	
		opts = $.extend({
			placeholder: language.sSearchPlaceholder,
			processing: false,
			text: language.sSearch
		}, opts);
	
		// The _INPUT_ is optional - is appended if not present
		if (opts.text.indexOf('_INPUT_') === -1) {
			opts.text += '_INPUT_';
		}
	
		opts.text = _fnMacros(settings, opts.text);
	
		// We can put the <input> outside of the label if it is at the start or end
		// which helps improve accessability (not all screen readers like implicit
		// for elements).
		var end = opts.text.match(/_INPUT_$/);
		var start = opts.text.match(/^_INPUT_/);
		var removed = opts.text.replace(/_INPUT_/, '');
		var str = '<label>' + opts.text + '</label>';
	
		if (start) {
			str = '_INPUT_<label>' + removed + '</label>';
		}
		else if (end) {
			str = '<label>' + removed + '</label>_INPUT_';
		}
	
		var filter = $('<div>')
			.addClass(classes.container)
			.append(str.replace(/_INPUT_/, input));
	
		// add for and id to label and input
		filter.find('label').attr('for', 'dt-search-' + __searchCounter);
		filter.find('input').attr('id', 'dt-search-' + __searchCounter);
		__searchCounter++;
	
		var searchFn = function(event) {
			var val = this.value;
	
			if(previousSearch.return && event.key !== "Enter") {
				return;
			}
	
			/* Now do the filter */
			if ( val != previousSearch.search ) {
				_fnProcessingRun(settings, opts.processing, function () {
					previousSearch.search = val;
			
					_fnFilterComplete( settings, previousSearch );
			
					// Need to redraw, without resorting
					settings._iDisplayStart = 0;
					_fnDraw( settings );
				});
			}
		};
	
		var searchDelay = settings.searchDelay !== null ?
			settings.searchDelay :
			0;
	
		var jqFilter = $('input', filter)
			.val( previousSearch.search )
			.attr( 'placeholder', opts.placeholder )
			.on(
				'keyup.DT search.DT input.DT paste.DT cut.DT',
				searchDelay ?
					DataTable.util.debounce( searchFn, searchDelay ) :
					searchFn
			)
			.on( 'mouseup.DT', function(e) {
				// Edge fix! Edge 17 does not trigger anything other than mouse events when clicking
				// on the clear icon (Edge bug 17584515). This is safe in other browsers as `searchFn`
				// checks the value to see if it has changed. In other browsers it won't have.
				setTimeout( function () {
					searchFn.call(jqFilter[0], e);
				}, 10);
			} )
			.on( 'keypress.DT', function(e) {
				/* Prevent form submission */
				if ( e.keyCode == 13 ) {
					return false;
				}
			} )
			.attr('aria-controls', tableId);
	
		// Update the input elements whenever the table is filtered
		$(settings.nTable).on( 'search.dt.DT', function ( ev, s ) {
			if ( settings === s && jqFilter[0] !== document.activeElement ) {
				jqFilter.val( typeof previousSearch.search !== 'function'
					? previousSearch.search
					: ''
				);
			}
		} );
	
		return filter;
	}, 'f' );
	
	// opts
	// - type - button configuration
	// - buttons - number of buttons to show - must be odd
	DataTable.feature.register( 'paging', function ( settings, opts ) {
		// Don't show the paging input if the table doesn't have paging enabled
		if (! settings.oFeatures.bPaginate) {
			return null;
		}
	
		opts = $.extend({
			buttons: DataTable.ext.pager.numbers_length,
			type: settings.sPaginationType,
			boundaryNumbers: true,
			firstLast: true,
			previousNext: true,
			numbers: true
		}, opts);
	
		var host = $('<div/>')
			.addClass(settings.oClasses.paging.container + (opts.type ? ' paging_' + opts.type : ''))
			.append(
				$('<nav>')
					.attr('aria-label', 'pagination')
					.addClass(settings.oClasses.paging.nav)
			);
		var draw = function () {
			_pagingDraw(settings, host.children(), opts);
		};
	
		settings.aoDrawCallback.push(draw);
	
		// Responsive redraw of paging control
		$(settings.nTable).on('column-sizing.dt.DT', draw);
	
		return host;
	}, 'p' );
	
	/**
	 * Dynamically create the button type array based on the configuration options.
	 * This will only happen if the paging type is not defined.
	 */
	function _pagingDynamic(opts) {
		var out = [];
	
		if (opts.numbers) {
			out.push('numbers');
		}
	
		if (opts.previousNext) {
			out.unshift('previous');
			out.push('next');
		}
	
		if (opts.firstLast) {
			out.unshift('first');
			out.push('last');
		}
	
		return out;
	}
	
	function _pagingDraw(settings, host, opts) {
		if (! settings._bInitComplete) {
			return;
		}
	
		var
			plugin = opts.type
				? DataTable.ext.pager[ opts.type ]
				: _pagingDynamic,
			aria = settings.oLanguage.oAria.paginate || {},
			start      = settings._iDisplayStart,
			len        = settings._iDisplayLength,
			visRecords = settings.fnRecordsDisplay(),
			all        = len === -1,
			page = all ? 0 : Math.ceil( start / len ),
			pages = all ? 1 : Math.ceil( visRecords / len ),
			buttons = [],
			buttonEls = [],
			buttonsNested = plugin(opts)
				.map(function (val) {
					return val === 'numbers'
						? _pagingNumbers(page, pages, opts.buttons, opts.boundaryNumbers)
						: val;
				});
	
		// .flat() would be better, but not supported in old Safari
		buttons = buttons.concat.apply(buttons, buttonsNested);
	
		for (var i=0 ; i<buttons.length ; i++) {
			var button = buttons[i];
	
			var btnInfo = _pagingButtonInfo(settings, button, page, pages);
			var btn = _fnRenderer( settings, 'pagingButton' )(
				settings,
				button,
				btnInfo.display,
				btnInfo.active,
				btnInfo.disabled
			);
	
			var ariaLabel = typeof button === 'string'
				? aria[ button ]
				: aria.number
					? aria.number + (button+1)
					: null;
	
			// Common attributes
			$(btn.clicker).attr({
				'aria-controls': settings.sTableId,
				'aria-disabled': btnInfo.disabled ? 'true' : null,
				'aria-current': btnInfo.active ? 'page' : null,
				'aria-label': ariaLabel,
				'data-dt-idx': button,
				'tabIndex': btnInfo.disabled
					? -1
					: settings.iTabIndex && btn.clicker[0].nodeName.toLowerCase() !== 'span'
						? settings.iTabIndex
						: null, // `0` doesn't need a tabIndex since it is the default
			});
	
			if (typeof button !== 'number') {
				$(btn.clicker).addClass(button);
			}
	
			_fnBindAction(
				btn.clicker, {action: button}, function(e) {
					e.preventDefault();
	
					_fnPageChange( settings, e.data.action, true );
				}
			);
	
			buttonEls.push(btn.display);
		}
	
		var wrapped = _fnRenderer(settings, 'pagingContainer')(
			settings, buttonEls
		);
	
		var activeEl = host.find(document.activeElement).data('dt-idx');
	
		host.empty().append(wrapped);
	
		if ( activeEl !== undefined ) {
			host.find( '[data-dt-idx='+activeEl+']' ).trigger('focus');
		}
	
		// Responsive - check if the buttons are over two lines based on the
		// height of the buttons and the container.
		if (buttonEls.length) {
			var outerHeight = $(buttonEls[0]).outerHeight();
		
			if (
				opts.buttons > 1 && // prevent infinite
				outerHeight > 0 && // will be 0 if hidden
				$(host).height() >= (outerHeight * 2) - 10
			) {
				_pagingDraw(settings, host, $.extend({}, opts, { buttons: opts.buttons - 2 }));
			}
		}
	}
	
	/**
	 * Get properties for a button based on the current paging state of the table
	 *
	 * @param {*} settings DT settings object
	 * @param {*} button The button type in question
	 * @param {*} page Table's current page
	 * @param {*} pages Number of pages
	 * @returns Info object
	 */
	function _pagingButtonInfo(settings, button, page, pages) {
		var lang = settings.oLanguage.oPaginate;
		var o = {
			display: '',
			active: false,
			disabled: false
		};
	
		switch ( button ) {
			case 'ellipsis':
				o.display = '&#x2026;';
				break;
	
			case 'first':
				o.display = lang.sFirst;
	
				if (page === 0) {
					o.disabled = true;
				}
				break;
	
			case 'previous':
				o.display = lang.sPrevious;
	
				if ( page === 0 ) {
					o.disabled = true;
				}
				break;
	
			case 'next':
				o.display = lang.sNext;
	
				if ( pages === 0 || page === pages-1 ) {
					o.disabled = true;
				}
				break;
	
			case 'last':
				o.display = lang.sLast;
	
				if ( pages === 0 || page === pages-1 ) {
					o.disabled = true;
				}
				break;
	
			default:
				if ( typeof button === 'number' ) {
					o.display = settings.fnFormatNumber( button + 1 );
					
					if (page === button) {
						o.active = true;
					}
				}
				break;
		}
	
		return o;
	}
	
	/**
	 * Compute what number buttons to show in the paging control
	 *
	 * @param {*} page Current page
	 * @param {*} pages Total number of pages
	 * @param {*} buttons Target number of number buttons
	 * @param {boolean} addFirstLast Indicate if page 1 and end should be included
	 * @returns Buttons to show
	 */
	function _pagingNumbers ( page, pages, buttons, addFirstLast ) {
		var
			numbers = [],
			half = Math.floor(buttons / 2),
			before = addFirstLast ? 2 : 1,
			after = addFirstLast ? 1 : 0;
	
		if ( pages <= buttons ) {
			numbers = _range(0, pages);
		}
		else if (buttons === 1) {
			// Single button - current page only
			numbers = [page];
		}
		else if (buttons === 3) {
			// Special logic for just three buttons
			if (page <= 1) {
				numbers = [0, 1, 'ellipsis'];
			}
			else if (page >= pages - 2) {
				numbers = _range(pages-2, pages);
				numbers.unshift('ellipsis');
			}
			else {
				numbers = ['ellipsis', page, 'ellipsis'];
			}
		}
		else if ( page <= half ) {
			numbers = _range(0, buttons-before);
			numbers.push('ellipsis');
	
			if (addFirstLast) {
				numbers.push(pages-1);
			}
		}
		else if ( page >= pages - 1 - half ) {
			numbers = _range(pages-(buttons-before), pages);
			numbers.unshift('ellipsis');
	
			if (addFirstLast) {
				numbers.unshift(0);
			}
		}
		else {
			numbers = _range(page-half+before, page+half-after);
			numbers.push('ellipsis');
			numbers.unshift('ellipsis');
	
			if (addFirstLast) {
				numbers.push(pages-1);
				numbers.unshift(0);
			}
		}
	
		return numbers;
	}
	
	var __lengthCounter = 0;
	
	// opts
	// - menu
	// - text
	DataTable.feature.register( 'pageLength', function ( settings, opts ) {
		var features = settings.oFeatures;
	
		// For compatibility with the legacy `pageLength` top level option
		if (! features.bPaginate || ! features.bLengthChange) {
			return null;
		}
	
		opts = $.extend({
			menu: settings.aLengthMenu,
			text: settings.oLanguage.sLengthMenu
		}, opts);
	
		var
			classes  = settings.oClasses.length,
			tableId  = settings.sTableId,
			menu     = opts.menu,
			lengths  = [],
			language = [],
			i;
	
		// Options can be given in a number of ways
		if (Array.isArray( menu[0] )) {
			// Old 1.x style - 2D array
			lengths = menu[0];
			language = menu[1];
		}
		else {
			for ( i=0 ; i<menu.length ; i++ ) {
				// An object with different label and value
				if ($.isPlainObject(menu[i])) {
					lengths.push(menu[i].value);
					language.push(menu[i].label);
				}
				else {
					// Or just a number to display and use
					lengths.push(menu[i]);
					language.push(menu[i]);
				}
			}
		}
	
		// We can put the <select> outside of the label if it is at the start or
		// end which helps improve accessability (not all screen readers like
		// implicit for elements).
		var end = opts.text.match(/_MENU_$/);
		var start = opts.text.match(/^_MENU_/);
		var removed = opts.text.replace(/_MENU_/, '');
		var str = '<label>' + opts.text + '</label>';
	
		if (start) {
			str = '_MENU_<label>' + removed + '</label>';
		}
		else if (end) {
			str = '<label>' + removed + '</label>_MENU_';
		}
	
		// Wrapper element - use a span as a holder for where the select will go
		var tmpId = 'tmp-' + (+new Date())
		var div = $('<div/>')
			.addClass( classes.container )
			.append(
				str.replace( '_MENU_', '<span id="'+tmpId+'"></span>' )
			);
	
		// Save text node content for macro updating
		var textNodes = [];
		Array.prototype.slice.call(div.find('label')[0].childNodes).forEach(function (el) {
			if (el.nodeType === Node.TEXT_NODE) {
				textNodes.push({
					el: el,
					text: el.textContent
				});
			}
		});
	
		// Update the label text in case it has an entries value
		var updateEntries = function (len) {
			textNodes.forEach(function (node) {
				node.el.textContent = _fnMacros(settings, node.text, len);
			});
		}
	
		// Next, the select itself, along with the options
		var select = $('<select/>', {
			'aria-controls': tableId,
			'class':         classes.select
		} );
	
		for ( i=0 ; i<lengths.length ; i++ ) {
			// Attempt to look up the length from the i18n options
			var label = settings.api.i18n('lengthLabels.' + lengths[i], null);
	
			if (label === null) {
				// If not present, fallback to old style
				label = typeof language[i] === 'number' ?
					settings.fnFormatNumber( language[i] ) :
					language[i];
			}
	
			select[0][ i ] = new Option(label, lengths[i]);
		}
	
		// add for and id to label and input
		div.find('label').attr('for', 'dt-length-' + __lengthCounter);
		select.attr('id', 'dt-length-' + __lengthCounter);
		__lengthCounter++;
	
		// Swap in the select list
		div.find('#' + tmpId).replaceWith(select);
	
		// Can't use `select` variable as user might provide their own and the
		// reference is broken by the use of outerHTML
		$('select', div)
			.val( settings._iDisplayLength )
			.on( 'change.DT', function() {
				_fnLengthChange( settings, $(this).val() );
				_fnDraw( settings );
			} );
	
		// Update node value whenever anything changes the table's length
		$(settings.nTable).on( 'length.dt.DT', function (e, s, len) {
			if ( settings === s ) {
				$('select', div).val( len );
	
				// Resolve plurals in the text for the new length
				updateEntries(len);
			}
		} );
	
		updateEntries(settings._iDisplayLength);
	
		return div;
	}, 'l' );
	
	// jQuery access
	$.fn.dataTable = DataTable;
	
	// Provide access to the host jQuery object (circular reference)
	DataTable.$ = $;
	
	// Legacy aliases
	$.fn.dataTableSettings = DataTable.settings;
	$.fn.dataTableExt = DataTable.ext;
	
	// With a capital `D` we return a DataTables API instance rather than a
	// jQuery object
	$.fn.DataTable = function ( opts ) {
		return $(this).dataTable( opts ).api();
	};
	
	// All properties that are available to $.fn.dataTable should also be
	// available on $.fn.DataTable
	$.each( DataTable, function ( prop, val ) {
		$.fn.DataTable[ prop ] = val;
	} );

	return DataTable;
}));

/*! DataTables Bootstrap 5 integration
 * © SpryMedia Ltd - datatables.net/license
 */

(function( factory ){
	if ( typeof define === 'function' && define.amd ) {
		// AMD
		define( ['jquery', 'datatables.net'], function ( $ ) {
			return factory( $, window, document );
		} );
	}
	else if ( typeof exports === 'object' ) {
		// CommonJS
		var jq = require('jquery');
		var cjsRequires = function (root, $) {
			if ( ! $.fn.dataTable ) {
				require('datatables.net')(root, $);
			}
		};

		if (typeof window === 'undefined') {
			module.exports = function (root, $) {
				if ( ! root ) {
					// CommonJS environments without a window global must pass a
					// root. This will give an error otherwise
					root = window;
				}

				if ( ! $ ) {
					$ = jq( root );
				}

				cjsRequires( root, $ );
				return factory( $, root, root.document );
			};
		}
		else {
			cjsRequires( window, jq );
			module.exports = factory( jq, window, window.document );
		}
	}
	else {
		// Browser
		factory( jQuery, window, document );
	}
}(function( $, window, document ) {
'use strict';
var DataTable = $.fn.dataTable;



/**
 * DataTables integration for Bootstrap 5.
 *
 * This file sets the defaults and adds options to DataTables to style its
 * controls using Bootstrap. See https://datatables.net/manual/styling/bootstrap
 * for further information.
 */

/* Set the defaults for DataTables initialisation */
$.extend( true, DataTable.defaults, {
	renderer: 'bootstrap'
} );


/* Default class modification */
$.extend( true, DataTable.ext.classes, {
	container: "dt-container dt-bootstrap5",
	search: {
		input: "form-control form-control-sm"
	},
	length: {
		select: "form-select form-select-sm"
	},
	processing: {
		container: "dt-processing card"
	},
	layout: {
		row: 'row mt-2 justify-content-between',
		cell: 'd-md-flex justify-content-between align-items-center',
		tableCell: 'col-12',
		start: 'dt-layout-start col-md-auto me-auto',
		end: 'dt-layout-end col-md-auto ms-auto',
		full: 'dt-layout-full col-md'
	}
} );


/* Bootstrap paging button renderer */
DataTable.ext.renderer.pagingButton.bootstrap = function (settings, buttonType, content, active, disabled) {
	var btnClasses = ['dt-paging-button', 'page-item'];

	if (active) {
		btnClasses.push('active');
	}

	if (disabled) {
		btnClasses.push('disabled')
	}

	var li = $('<li>').addClass(btnClasses.join(' '));
	var a = $('<button>', {
		'class': 'page-link',
		role: 'link',
		type: 'button'
	})
		.html(content)
		.appendTo(li);

	return {
		display: li,
		clicker: a
	};
};

DataTable.ext.renderer.pagingContainer.bootstrap = function (settings, buttonEls) {
	return $('<ul/>').addClass('pagination').append(buttonEls);
};


return DataTable;
}));

"use strict";

//
// Datatables.net Initialization
//

// Set Defaults

var defaults = {
	"language": {		
		"info": "Showing _START_ to _END_ of _TOTAL_ records",
    "infoEmpty": "Showing no records",
		"lengthMenu": "_MENU_",
		"processing": '<span class="spinner-border w-15px h-15px text-muted align-middle me-2"></span> <span class="text-gray-600">Loading...</span>',
		"paginate": {
			"first": '<i class="kt-outline kt-double-left"></i>',
			"last": '<i class="kt-outline kt-double-right"></i>',
			"next": '<i class="next"></i>',
			"previous": '<i class="previous"></i>'
		}
	}
};

$.extend(true, $.fn.dataTable.defaults, defaults);

/*! DataTables Bootstrap 4 integration
 * ©2011-2017 SpryMedia Ltd - datatables.net/license
 */

/**
 * DataTables integration for Bootstrap 4. This requires Bootstrap 4 and
 * DataTables 1.10 or newer.
 *
 * This file sets the defaults and adds options to DataTables to style its
 * controls using Bootstrap. See http://datatables.net/manual/styling/bootstrap
 * for further information.
 */
(function( factory ){
	if ( typeof define === 'function' && define.amd ) {
		// AMD
		define( ['jquery', 'datatables.net'], function ( $ ) {
			return factory( $, window, document );
		} );
	}
	else if ( typeof exports === 'object' ) {
		// CommonJS
		module.exports = function (root, $) {
			if ( ! root ) {
				root = window;
			}

			if ( ! $ || ! $.fn.dataTable ) {
				// Require DataTables, which attaches to jQuery, including
				// jQuery if needed and have a $ property so we can access the
				// jQuery object that is used
				$ = require('datatables.net')(root, $).$;
			}

			return factory( $, root, root.document );
		};
	}
	else {
		// Browser
		factory( jQuery, window, document );
	}
}(function( $, window, document, undefined ) {
'use strict';
var DataTable = $.fn.dataTable;


/* Set the defaults for DataTables initialisation */
$.extend( true, DataTable.defaults, {
	pagingType: 'simple_numbers',
	dom: 
		"<'table-responsive'tr>" +
		"<'row'" + 
		"<'col-sm-12 col-md-5 d-flex align-items-center justify-content-center justify-content-md-start dt-toolbar'li>" + 
		"<'col-sm-12 col-md-7 d-flex align-items-center justify-content-center justify-content-md-end'p>" +
		">",	
	renderer: 'bootstrap'
} );


/* Default class modification */
$.extend( DataTable.ext.classes, {
	sWrapper:      "dataTables_wrapper dt-bootstrap4",
	sFilterInput:  "form-control form-control-sm form-control-solid",
	sLengthSelect: "form-select form-select-sm form-select-solid",
	sProcessing:   "dataTables_processing",
	sPageButton:   "paginate_button page-item",
	search: {
		input: "form-control form-control-solid form-control-sm"
	},
	length: {
		select: "form-select form-select-solid form-select-sm"
	}
} );

/* Bootstrap paging button renderer */
DataTable.ext.renderer.pageButton.bootstrap = function ( settings, host, idx, buttons, page, pages ) {
	var api     = new DataTable.Api( settings );
	var classes = settings.oClasses;
	var lang    = settings.oLanguage.oPaginate;
	var aria = settings.oLanguage.oAria.paginate || {};
	var btnDisplay, btnClass, counter=0;

	var attach = function( container, buttons ) {
		var i, ien, node, button;
		var clickHandler = function ( e ) {
			e.preventDefault();
			if ( !$(e.currentTarget).hasClass('disabled') && api.page() != e.data.action ) {
				api.page( e.data.action ).draw( 'page' );
			}
		};

		for ( i=0, ien=buttons.length ; i<ien ; i++ ) {
			button = buttons[i];

			if ( Array.isArray( button ) ) {	
				attach( container, button );
			} else {
				btnDisplay = '';
				btnClass = '';

				switch ( button ) {
					case 'ellipsis':
						btnDisplay = '&#x2026;';
						btnClass = 'disabled';
						break;

					case 'first':
						btnDisplay = lang.sFirst;
						btnClass = button + (page > 0 ?
							'' : ' disabled');
						break;

					case 'previous':
						btnDisplay = lang.sPrevious;
						btnClass = button + (page > 0 ?
							'' : ' disabled');
						break;

					case 'next':
						btnDisplay = lang.sNext;
						btnClass = button + (page < pages-1 ?
							'' : ' disabled');
						break;

					case 'last':
						btnDisplay = lang.sLast;
						btnClass = button + (page < pages-1 ?
							'' : ' disabled');
						break;

					default:
						btnDisplay = button + 1;
						btnClass = page === button ?
							'active' : '';
						break;
				}

				if ( btnDisplay ) {
					node = $('<li>', {
							'class': classes.sPageButton+' '+btnClass,
							'id': idx === 0 && typeof button === 'string' ?
								settings.sTableId +'_'+ button :
								null
						} )
						.append( $('<a>', {
								'href': '#',
								'aria-controls': settings.sTableId,
								'aria-label': aria[ button ],
								'data-dt-idx': counter,
								'tabindex': settings.iTabIndex,
								'class': 'page-link'
							} )
							.html( btnDisplay )
						)
						.appendTo( container );

					settings.oApi._fnBindAction(
						node, {action: button}, clickHandler
					);

					counter++;
				}
			}
		}
	};

	// IE9 throws an 'unknown error' if document.activeElement is used
	// inside an iframe or frame.
	var activeEl;

	try {
		// Because this approach is destroying and recreating the paging
		// elements, focus is lost on the select button which is bad for
		// accessibility. So we want to restore focus once the draw has
		// completed
		activeEl = $(host).find(document.activeElement).data('dt-idx');
	}
	catch (e) {}

	attach(
		$(host).empty().html('<ul class="pagination"/>').children('ul'),
		buttons
	);

	if ( activeEl !== undefined ) {
		$(host).find( '[data-dt-idx='+activeEl+']' ).trigger('focus');
	}
};


return DataTable;
}));

/*!

JSZip v3.10.1 - A JavaScript class for generating and reading zip files
<http://stuartk.com/jszip>

(c) 2009-2016 Stuart Knightley <stuart [at] stuartk.com>
Dual licenced under the MIT license or GPLv3. See https://raw.github.com/Stuk/jszip/main/LICENSE.markdown.

JSZip uses the library pako released under the MIT license :
https://github.com/nodeca/pako/blob/main/LICENSE
*/

!function(e){if("object"==typeof exports&&"undefined"!=typeof module)module.exports=e();else if("function"==typeof define&&define.amd)define([],e);else{("undefined"!=typeof window?window:"undefined"!=typeof global?global:"undefined"!=typeof self?self:this).JSZip=e()}}(function(){return function s(a,o,h){function u(r,e){if(!o[r]){if(!a[r]){var t="function"==typeof require&&require;if(!e&&t)return t(r,!0);if(l)return l(r,!0);var n=new Error("Cannot find module '"+r+"'");throw n.code="MODULE_NOT_FOUND",n}var i=o[r]={exports:{}};a[r][0].call(i.exports,function(e){var t=a[r][1][e];return u(t||e)},i,i.exports,s,a,o,h)}return o[r].exports}for(var l="function"==typeof require&&require,e=0;e<h.length;e++)u(h[e]);return u}({1:[function(e,t,r){"use strict";var d=e("./utils"),c=e("./support"),p="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";r.encode=function(e){for(var t,r,n,i,s,a,o,h=[],u=0,l=e.length,f=l,c="string"!==d.getTypeOf(e);u<e.length;)f=l-u,n=c?(t=e[u++],r=u<l?e[u++]:0,u<l?e[u++]:0):(t=e.charCodeAt(u++),r=u<l?e.charCodeAt(u++):0,u<l?e.charCodeAt(u++):0),i=t>>2,s=(3&t)<<4|r>>4,a=1<f?(15&r)<<2|n>>6:64,o=2<f?63&n:64,h.push(p.charAt(i)+p.charAt(s)+p.charAt(a)+p.charAt(o));return h.join("")},r.decode=function(e){var t,r,n,i,s,a,o=0,h=0,u="data:";if(e.substr(0,u.length)===u)throw new Error("Invalid base64 input, it looks like a data url.");var l,f=3*(e=e.replace(/[^A-Za-z0-9+/=]/g,"")).length/4;if(e.charAt(e.length-1)===p.charAt(64)&&f--,e.charAt(e.length-2)===p.charAt(64)&&f--,f%1!=0)throw new Error("Invalid base64 input, bad content length.");for(l=c.uint8array?new Uint8Array(0|f):new Array(0|f);o<e.length;)t=p.indexOf(e.charAt(o++))<<2|(i=p.indexOf(e.charAt(o++)))>>4,r=(15&i)<<4|(s=p.indexOf(e.charAt(o++)))>>2,n=(3&s)<<6|(a=p.indexOf(e.charAt(o++))),l[h++]=t,64!==s&&(l[h++]=r),64!==a&&(l[h++]=n);return l}},{"./support":30,"./utils":32}],2:[function(e,t,r){"use strict";var n=e("./external"),i=e("./stream/DataWorker"),s=e("./stream/Crc32Probe"),a=e("./stream/DataLengthProbe");function o(e,t,r,n,i){this.compressedSize=e,this.uncompressedSize=t,this.crc32=r,this.compression=n,this.compressedContent=i}o.prototype={getContentWorker:function(){var e=new i(n.Promise.resolve(this.compressedContent)).pipe(this.compression.uncompressWorker()).pipe(new a("data_length")),t=this;return e.on("end",function(){if(this.streamInfo.data_length!==t.uncompressedSize)throw new Error("Bug : uncompressed data size mismatch")}),e},getCompressedWorker:function(){return new i(n.Promise.resolve(this.compressedContent)).withStreamInfo("compressedSize",this.compressedSize).withStreamInfo("uncompressedSize",this.uncompressedSize).withStreamInfo("crc32",this.crc32).withStreamInfo("compression",this.compression)}},o.createWorkerFrom=function(e,t,r){return e.pipe(new s).pipe(new a("uncompressedSize")).pipe(t.compressWorker(r)).pipe(new a("compressedSize")).withStreamInfo("compression",t)},t.exports=o},{"./external":6,"./stream/Crc32Probe":25,"./stream/DataLengthProbe":26,"./stream/DataWorker":27}],3:[function(e,t,r){"use strict";var n=e("./stream/GenericWorker");r.STORE={magic:"\0\0",compressWorker:function(){return new n("STORE compression")},uncompressWorker:function(){return new n("STORE decompression")}},r.DEFLATE=e("./flate")},{"./flate":7,"./stream/GenericWorker":28}],4:[function(e,t,r){"use strict";var n=e("./utils");var o=function(){for(var e,t=[],r=0;r<256;r++){e=r;for(var n=0;n<8;n++)e=1&e?3988292384^e>>>1:e>>>1;t[r]=e}return t}();t.exports=function(e,t){return void 0!==e&&e.length?"string"!==n.getTypeOf(e)?function(e,t,r,n){var i=o,s=n+r;e^=-1;for(var a=n;a<s;a++)e=e>>>8^i[255&(e^t[a])];return-1^e}(0|t,e,e.length,0):function(e,t,r,n){var i=o,s=n+r;e^=-1;for(var a=n;a<s;a++)e=e>>>8^i[255&(e^t.charCodeAt(a))];return-1^e}(0|t,e,e.length,0):0}},{"./utils":32}],5:[function(e,t,r){"use strict";r.base64=!1,r.binary=!1,r.dir=!1,r.createFolders=!0,r.date=null,r.compression=null,r.compressionOptions=null,r.comment=null,r.unixPermissions=null,r.dosPermissions=null},{}],6:[function(e,t,r){"use strict";var n=null;n="undefined"!=typeof Promise?Promise:e("lie"),t.exports={Promise:n}},{lie:37}],7:[function(e,t,r){"use strict";var n="undefined"!=typeof Uint8Array&&"undefined"!=typeof Uint16Array&&"undefined"!=typeof Uint32Array,i=e("pako"),s=e("./utils"),a=e("./stream/GenericWorker"),o=n?"uint8array":"array";function h(e,t){a.call(this,"FlateWorker/"+e),this._pako=null,this._pakoAction=e,this._pakoOptions=t,this.meta={}}r.magic="\b\0",s.inherits(h,a),h.prototype.processChunk=function(e){this.meta=e.meta,null===this._pako&&this._createPako(),this._pako.push(s.transformTo(o,e.data),!1)},h.prototype.flush=function(){a.prototype.flush.call(this),null===this._pako&&this._createPako(),this._pako.push([],!0)},h.prototype.cleanUp=function(){a.prototype.cleanUp.call(this),this._pako=null},h.prototype._createPako=function(){this._pako=new i[this._pakoAction]({raw:!0,level:this._pakoOptions.level||-1});var t=this;this._pako.onData=function(e){t.push({data:e,meta:t.meta})}},r.compressWorker=function(e){return new h("Deflate",e)},r.uncompressWorker=function(){return new h("Inflate",{})}},{"./stream/GenericWorker":28,"./utils":32,pako:38}],8:[function(e,t,r){"use strict";function A(e,t){var r,n="";for(r=0;r<t;r++)n+=String.fromCharCode(255&e),e>>>=8;return n}function n(e,t,r,n,i,s){var a,o,h=e.file,u=e.compression,l=s!==O.utf8encode,f=I.transformTo("string",s(h.name)),c=I.transformTo("string",O.utf8encode(h.name)),d=h.comment,p=I.transformTo("string",s(d)),m=I.transformTo("string",O.utf8encode(d)),_=c.length!==h.name.length,g=m.length!==d.length,b="",v="",y="",w=h.dir,k=h.date,x={crc32:0,compressedSize:0,uncompressedSize:0};t&&!r||(x.crc32=e.crc32,x.compressedSize=e.compressedSize,x.uncompressedSize=e.uncompressedSize);var S=0;t&&(S|=8),l||!_&&!g||(S|=2048);var z=0,C=0;w&&(z|=16),"UNIX"===i?(C=798,z|=function(e,t){var r=e;return e||(r=t?16893:33204),(65535&r)<<16}(h.unixPermissions,w)):(C=20,z|=function(e){return 63&(e||0)}(h.dosPermissions)),a=k.getUTCHours(),a<<=6,a|=k.getUTCMinutes(),a<<=5,a|=k.getUTCSeconds()/2,o=k.getUTCFullYear()-1980,o<<=4,o|=k.getUTCMonth()+1,o<<=5,o|=k.getUTCDate(),_&&(v=A(1,1)+A(B(f),4)+c,b+="up"+A(v.length,2)+v),g&&(y=A(1,1)+A(B(p),4)+m,b+="uc"+A(y.length,2)+y);var E="";return E+="\n\0",E+=A(S,2),E+=u.magic,E+=A(a,2),E+=A(o,2),E+=A(x.crc32,4),E+=A(x.compressedSize,4),E+=A(x.uncompressedSize,4),E+=A(f.length,2),E+=A(b.length,2),{fileRecord:R.LOCAL_FILE_HEADER+E+f+b,dirRecord:R.CENTRAL_FILE_HEADER+A(C,2)+E+A(p.length,2)+"\0\0\0\0"+A(z,4)+A(n,4)+f+b+p}}var I=e("../utils"),i=e("../stream/GenericWorker"),O=e("../utf8"),B=e("../crc32"),R=e("../signature");function s(e,t,r,n){i.call(this,"ZipFileWorker"),this.bytesWritten=0,this.zipComment=t,this.zipPlatform=r,this.encodeFileName=n,this.streamFiles=e,this.accumulate=!1,this.contentBuffer=[],this.dirRecords=[],this.currentSourceOffset=0,this.entriesCount=0,this.currentFile=null,this._sources=[]}I.inherits(s,i),s.prototype.push=function(e){var t=e.meta.percent||0,r=this.entriesCount,n=this._sources.length;this.accumulate?this.contentBuffer.push(e):(this.bytesWritten+=e.data.length,i.prototype.push.call(this,{data:e.data,meta:{currentFile:this.currentFile,percent:r?(t+100*(r-n-1))/r:100}}))},s.prototype.openedSource=function(e){this.currentSourceOffset=this.bytesWritten,this.currentFile=e.file.name;var t=this.streamFiles&&!e.file.dir;if(t){var r=n(e,t,!1,this.currentSourceOffset,this.zipPlatform,this.encodeFileName);this.push({data:r.fileRecord,meta:{percent:0}})}else this.accumulate=!0},s.prototype.closedSource=function(e){this.accumulate=!1;var t=this.streamFiles&&!e.file.dir,r=n(e,t,!0,this.currentSourceOffset,this.zipPlatform,this.encodeFileName);if(this.dirRecords.push(r.dirRecord),t)this.push({data:function(e){return R.DATA_DESCRIPTOR+A(e.crc32,4)+A(e.compressedSize,4)+A(e.uncompressedSize,4)}(e),meta:{percent:100}});else for(this.push({data:r.fileRecord,meta:{percent:0}});this.contentBuffer.length;)this.push(this.contentBuffer.shift());this.currentFile=null},s.prototype.flush=function(){for(var e=this.bytesWritten,t=0;t<this.dirRecords.length;t++)this.push({data:this.dirRecords[t],meta:{percent:100}});var r=this.bytesWritten-e,n=function(e,t,r,n,i){var s=I.transformTo("string",i(n));return R.CENTRAL_DIRECTORY_END+"\0\0\0\0"+A(e,2)+A(e,2)+A(t,4)+A(r,4)+A(s.length,2)+s}(this.dirRecords.length,r,e,this.zipComment,this.encodeFileName);this.push({data:n,meta:{percent:100}})},s.prototype.prepareNextSource=function(){this.previous=this._sources.shift(),this.openedSource(this.previous.streamInfo),this.isPaused?this.previous.pause():this.previous.resume()},s.prototype.registerPrevious=function(e){this._sources.push(e);var t=this;return e.on("data",function(e){t.processChunk(e)}),e.on("end",function(){t.closedSource(t.previous.streamInfo),t._sources.length?t.prepareNextSource():t.end()}),e.on("error",function(e){t.error(e)}),this},s.prototype.resume=function(){return!!i.prototype.resume.call(this)&&(!this.previous&&this._sources.length?(this.prepareNextSource(),!0):this.previous||this._sources.length||this.generatedError?void 0:(this.end(),!0))},s.prototype.error=function(e){var t=this._sources;if(!i.prototype.error.call(this,e))return!1;for(var r=0;r<t.length;r++)try{t[r].error(e)}catch(e){}return!0},s.prototype.lock=function(){i.prototype.lock.call(this);for(var e=this._sources,t=0;t<e.length;t++)e[t].lock()},t.exports=s},{"../crc32":4,"../signature":23,"../stream/GenericWorker":28,"../utf8":31,"../utils":32}],9:[function(e,t,r){"use strict";var u=e("../compressions"),n=e("./ZipFileWorker");r.generateWorker=function(e,a,t){var o=new n(a.streamFiles,t,a.platform,a.encodeFileName),h=0;try{e.forEach(function(e,t){h++;var r=function(e,t){var r=e||t,n=u[r];if(!n)throw new Error(r+" is not a valid compression method !");return n}(t.options.compression,a.compression),n=t.options.compressionOptions||a.compressionOptions||{},i=t.dir,s=t.date;t._compressWorker(r,n).withStreamInfo("file",{name:e,dir:i,date:s,comment:t.comment||"",unixPermissions:t.unixPermissions,dosPermissions:t.dosPermissions}).pipe(o)}),o.entriesCount=h}catch(e){o.error(e)}return o}},{"../compressions":3,"./ZipFileWorker":8}],10:[function(e,t,r){"use strict";function n(){if(!(this instanceof n))return new n;if(arguments.length)throw new Error("The constructor with parameters has been removed in JSZip 3.0, please check the upgrade guide.");this.files=Object.create(null),this.comment=null,this.root="",this.clone=function(){var e=new n;for(var t in this)"function"!=typeof this[t]&&(e[t]=this[t]);return e}}(n.prototype=e("./object")).loadAsync=e("./load"),n.support=e("./support"),n.defaults=e("./defaults"),n.version="3.10.1",n.loadAsync=function(e,t){return(new n).loadAsync(e,t)},n.external=e("./external"),t.exports=n},{"./defaults":5,"./external":6,"./load":11,"./object":15,"./support":30}],11:[function(e,t,r){"use strict";var u=e("./utils"),i=e("./external"),n=e("./utf8"),s=e("./zipEntries"),a=e("./stream/Crc32Probe"),l=e("./nodejsUtils");function f(n){return new i.Promise(function(e,t){var r=n.decompressed.getContentWorker().pipe(new a);r.on("error",function(e){t(e)}).on("end",function(){r.streamInfo.crc32!==n.decompressed.crc32?t(new Error("Corrupted zip : CRC32 mismatch")):e()}).resume()})}t.exports=function(e,o){var h=this;return o=u.extend(o||{},{base64:!1,checkCRC32:!1,optimizedBinaryString:!1,createFolders:!1,decodeFileName:n.utf8decode}),l.isNode&&l.isStream(e)?i.Promise.reject(new Error("JSZip can't accept a stream when loading a zip file.")):u.prepareContent("the loaded zip file",e,!0,o.optimizedBinaryString,o.base64).then(function(e){var t=new s(o);return t.load(e),t}).then(function(e){var t=[i.Promise.resolve(e)],r=e.files;if(o.checkCRC32)for(var n=0;n<r.length;n++)t.push(f(r[n]));return i.Promise.all(t)}).then(function(e){for(var t=e.shift(),r=t.files,n=0;n<r.length;n++){var i=r[n],s=i.fileNameStr,a=u.resolve(i.fileNameStr);h.file(a,i.decompressed,{binary:!0,optimizedBinaryString:!0,date:i.date,dir:i.dir,comment:i.fileCommentStr.length?i.fileCommentStr:null,unixPermissions:i.unixPermissions,dosPermissions:i.dosPermissions,createFolders:o.createFolders}),i.dir||(h.file(a).unsafeOriginalName=s)}return t.zipComment.length&&(h.comment=t.zipComment),h})}},{"./external":6,"./nodejsUtils":14,"./stream/Crc32Probe":25,"./utf8":31,"./utils":32,"./zipEntries":33}],12:[function(e,t,r){"use strict";var n=e("../utils"),i=e("../stream/GenericWorker");function s(e,t){i.call(this,"Nodejs stream input adapter for "+e),this._upstreamEnded=!1,this._bindStream(t)}n.inherits(s,i),s.prototype._bindStream=function(e){var t=this;(this._stream=e).pause(),e.on("data",function(e){t.push({data:e,meta:{percent:0}})}).on("error",function(e){t.isPaused?this.generatedError=e:t.error(e)}).on("end",function(){t.isPaused?t._upstreamEnded=!0:t.end()})},s.prototype.pause=function(){return!!i.prototype.pause.call(this)&&(this._stream.pause(),!0)},s.prototype.resume=function(){return!!i.prototype.resume.call(this)&&(this._upstreamEnded?this.end():this._stream.resume(),!0)},t.exports=s},{"../stream/GenericWorker":28,"../utils":32}],13:[function(e,t,r){"use strict";var i=e("readable-stream").Readable;function n(e,t,r){i.call(this,t),this._helper=e;var n=this;e.on("data",function(e,t){n.push(e)||n._helper.pause(),r&&r(t)}).on("error",function(e){n.emit("error",e)}).on("end",function(){n.push(null)})}e("../utils").inherits(n,i),n.prototype._read=function(){this._helper.resume()},t.exports=n},{"../utils":32,"readable-stream":16}],14:[function(e,t,r){"use strict";t.exports={isNode:"undefined"!=typeof Buffer,newBufferFrom:function(e,t){if(Buffer.from&&Buffer.from!==Uint8Array.from)return Buffer.from(e,t);if("number"==typeof e)throw new Error('The "data" argument must not be a number');return new Buffer(e,t)},allocBuffer:function(e){if(Buffer.alloc)return Buffer.alloc(e);var t=new Buffer(e);return t.fill(0),t},isBuffer:function(e){return Buffer.isBuffer(e)},isStream:function(e){return e&&"function"==typeof e.on&&"function"==typeof e.pause&&"function"==typeof e.resume}}},{}],15:[function(e,t,r){"use strict";function s(e,t,r){var n,i=u.getTypeOf(t),s=u.extend(r||{},f);s.date=s.date||new Date,null!==s.compression&&(s.compression=s.compression.toUpperCase()),"string"==typeof s.unixPermissions&&(s.unixPermissions=parseInt(s.unixPermissions,8)),s.unixPermissions&&16384&s.unixPermissions&&(s.dir=!0),s.dosPermissions&&16&s.dosPermissions&&(s.dir=!0),s.dir&&(e=g(e)),s.createFolders&&(n=_(e))&&b.call(this,n,!0);var a="string"===i&&!1===s.binary&&!1===s.base64;r&&void 0!==r.binary||(s.binary=!a),(t instanceof c&&0===t.uncompressedSize||s.dir||!t||0===t.length)&&(s.base64=!1,s.binary=!0,t="",s.compression="STORE",i="string");var o=null;o=t instanceof c||t instanceof l?t:p.isNode&&p.isStream(t)?new m(e,t):u.prepareContent(e,t,s.binary,s.optimizedBinaryString,s.base64);var h=new d(e,o,s);this.files[e]=h}var i=e("./utf8"),u=e("./utils"),l=e("./stream/GenericWorker"),a=e("./stream/StreamHelper"),f=e("./defaults"),c=e("./compressedObject"),d=e("./zipObject"),o=e("./generate"),p=e("./nodejsUtils"),m=e("./nodejs/NodejsStreamInputAdapter"),_=function(e){"/"===e.slice(-1)&&(e=e.substring(0,e.length-1));var t=e.lastIndexOf("/");return 0<t?e.substring(0,t):""},g=function(e){return"/"!==e.slice(-1)&&(e+="/"),e},b=function(e,t){return t=void 0!==t?t:f.createFolders,e=g(e),this.files[e]||s.call(this,e,null,{dir:!0,createFolders:t}),this.files[e]};function h(e){return"[object RegExp]"===Object.prototype.toString.call(e)}var n={load:function(){throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.")},forEach:function(e){var t,r,n;for(t in this.files)n=this.files[t],(r=t.slice(this.root.length,t.length))&&t.slice(0,this.root.length)===this.root&&e(r,n)},filter:function(r){var n=[];return this.forEach(function(e,t){r(e,t)&&n.push(t)}),n},file:function(e,t,r){if(1!==arguments.length)return e=this.root+e,s.call(this,e,t,r),this;if(h(e)){var n=e;return this.filter(function(e,t){return!t.dir&&n.test(e)})}var i=this.files[this.root+e];return i&&!i.dir?i:null},folder:function(r){if(!r)return this;if(h(r))return this.filter(function(e,t){return t.dir&&r.test(e)});var e=this.root+r,t=b.call(this,e),n=this.clone();return n.root=t.name,n},remove:function(r){r=this.root+r;var e=this.files[r];if(e||("/"!==r.slice(-1)&&(r+="/"),e=this.files[r]),e&&!e.dir)delete this.files[r];else for(var t=this.filter(function(e,t){return t.name.slice(0,r.length)===r}),n=0;n<t.length;n++)delete this.files[t[n].name];return this},generate:function(){throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.")},generateInternalStream:function(e){var t,r={};try{if((r=u.extend(e||{},{streamFiles:!1,compression:"STORE",compressionOptions:null,type:"",platform:"DOS",comment:null,mimeType:"application/zip",encodeFileName:i.utf8encode})).type=r.type.toLowerCase(),r.compression=r.compression.toUpperCase(),"binarystring"===r.type&&(r.type="string"),!r.type)throw new Error("No output type specified.");u.checkSupport(r.type),"darwin"!==r.platform&&"freebsd"!==r.platform&&"linux"!==r.platform&&"sunos"!==r.platform||(r.platform="UNIX"),"win32"===r.platform&&(r.platform="DOS");var n=r.comment||this.comment||"";t=o.generateWorker(this,r,n)}catch(e){(t=new l("error")).error(e)}return new a(t,r.type||"string",r.mimeType)},generateAsync:function(e,t){return this.generateInternalStream(e).accumulate(t)},generateNodeStream:function(e,t){return(e=e||{}).type||(e.type="nodebuffer"),this.generateInternalStream(e).toNodejsStream(t)}};t.exports=n},{"./compressedObject":2,"./defaults":5,"./generate":9,"./nodejs/NodejsStreamInputAdapter":12,"./nodejsUtils":14,"./stream/GenericWorker":28,"./stream/StreamHelper":29,"./utf8":31,"./utils":32,"./zipObject":35}],16:[function(e,t,r){"use strict";t.exports=e("stream")},{stream:void 0}],17:[function(e,t,r){"use strict";var n=e("./DataReader");function i(e){n.call(this,e);for(var t=0;t<this.data.length;t++)e[t]=255&e[t]}e("../utils").inherits(i,n),i.prototype.byteAt=function(e){return this.data[this.zero+e]},i.prototype.lastIndexOfSignature=function(e){for(var t=e.charCodeAt(0),r=e.charCodeAt(1),n=e.charCodeAt(2),i=e.charCodeAt(3),s=this.length-4;0<=s;--s)if(this.data[s]===t&&this.data[s+1]===r&&this.data[s+2]===n&&this.data[s+3]===i)return s-this.zero;return-1},i.prototype.readAndCheckSignature=function(e){var t=e.charCodeAt(0),r=e.charCodeAt(1),n=e.charCodeAt(2),i=e.charCodeAt(3),s=this.readData(4);return t===s[0]&&r===s[1]&&n===s[2]&&i===s[3]},i.prototype.readData=function(e){if(this.checkOffset(e),0===e)return[];var t=this.data.slice(this.zero+this.index,this.zero+this.index+e);return this.index+=e,t},t.exports=i},{"../utils":32,"./DataReader":18}],18:[function(e,t,r){"use strict";var n=e("../utils");function i(e){this.data=e,this.length=e.length,this.index=0,this.zero=0}i.prototype={checkOffset:function(e){this.checkIndex(this.index+e)},checkIndex:function(e){if(this.length<this.zero+e||e<0)throw new Error("End of data reached (data length = "+this.length+", asked index = "+e+"). Corrupted zip ?")},setIndex:function(e){this.checkIndex(e),this.index=e},skip:function(e){this.setIndex(this.index+e)},byteAt:function(){},readInt:function(e){var t,r=0;for(this.checkOffset(e),t=this.index+e-1;t>=this.index;t--)r=(r<<8)+this.byteAt(t);return this.index+=e,r},readString:function(e){return n.transformTo("string",this.readData(e))},readData:function(){},lastIndexOfSignature:function(){},readAndCheckSignature:function(){},readDate:function(){var e=this.readInt(4);return new Date(Date.UTC(1980+(e>>25&127),(e>>21&15)-1,e>>16&31,e>>11&31,e>>5&63,(31&e)<<1))}},t.exports=i},{"../utils":32}],19:[function(e,t,r){"use strict";var n=e("./Uint8ArrayReader");function i(e){n.call(this,e)}e("../utils").inherits(i,n),i.prototype.readData=function(e){this.checkOffset(e);var t=this.data.slice(this.zero+this.index,this.zero+this.index+e);return this.index+=e,t},t.exports=i},{"../utils":32,"./Uint8ArrayReader":21}],20:[function(e,t,r){"use strict";var n=e("./DataReader");function i(e){n.call(this,e)}e("../utils").inherits(i,n),i.prototype.byteAt=function(e){return this.data.charCodeAt(this.zero+e)},i.prototype.lastIndexOfSignature=function(e){return this.data.lastIndexOf(e)-this.zero},i.prototype.readAndCheckSignature=function(e){return e===this.readData(4)},i.prototype.readData=function(e){this.checkOffset(e);var t=this.data.slice(this.zero+this.index,this.zero+this.index+e);return this.index+=e,t},t.exports=i},{"../utils":32,"./DataReader":18}],21:[function(e,t,r){"use strict";var n=e("./ArrayReader");function i(e){n.call(this,e)}e("../utils").inherits(i,n),i.prototype.readData=function(e){if(this.checkOffset(e),0===e)return new Uint8Array(0);var t=this.data.subarray(this.zero+this.index,this.zero+this.index+e);return this.index+=e,t},t.exports=i},{"../utils":32,"./ArrayReader":17}],22:[function(e,t,r){"use strict";var n=e("../utils"),i=e("../support"),s=e("./ArrayReader"),a=e("./StringReader"),o=e("./NodeBufferReader"),h=e("./Uint8ArrayReader");t.exports=function(e){var t=n.getTypeOf(e);return n.checkSupport(t),"string"!==t||i.uint8array?"nodebuffer"===t?new o(e):i.uint8array?new h(n.transformTo("uint8array",e)):new s(n.transformTo("array",e)):new a(e)}},{"../support":30,"../utils":32,"./ArrayReader":17,"./NodeBufferReader":19,"./StringReader":20,"./Uint8ArrayReader":21}],23:[function(e,t,r){"use strict";r.LOCAL_FILE_HEADER="PK",r.CENTRAL_FILE_HEADER="PK",r.CENTRAL_DIRECTORY_END="PK",r.ZIP64_CENTRAL_DIRECTORY_LOCATOR="PK",r.ZIP64_CENTRAL_DIRECTORY_END="PK",r.DATA_DESCRIPTOR="PK\b"},{}],24:[function(e,t,r){"use strict";var n=e("./GenericWorker"),i=e("../utils");function s(e){n.call(this,"ConvertWorker to "+e),this.destType=e}i.inherits(s,n),s.prototype.processChunk=function(e){this.push({data:i.transformTo(this.destType,e.data),meta:e.meta})},t.exports=s},{"../utils":32,"./GenericWorker":28}],25:[function(e,t,r){"use strict";var n=e("./GenericWorker"),i=e("../crc32");function s(){n.call(this,"Crc32Probe"),this.withStreamInfo("crc32",0)}e("../utils").inherits(s,n),s.prototype.processChunk=function(e){this.streamInfo.crc32=i(e.data,this.streamInfo.crc32||0),this.push(e)},t.exports=s},{"../crc32":4,"../utils":32,"./GenericWorker":28}],26:[function(e,t,r){"use strict";var n=e("../utils"),i=e("./GenericWorker");function s(e){i.call(this,"DataLengthProbe for "+e),this.propName=e,this.withStreamInfo(e,0)}n.inherits(s,i),s.prototype.processChunk=function(e){if(e){var t=this.streamInfo[this.propName]||0;this.streamInfo[this.propName]=t+e.data.length}i.prototype.processChunk.call(this,e)},t.exports=s},{"../utils":32,"./GenericWorker":28}],27:[function(e,t,r){"use strict";var n=e("../utils"),i=e("./GenericWorker");function s(e){i.call(this,"DataWorker");var t=this;this.dataIsReady=!1,this.index=0,this.max=0,this.data=null,this.type="",this._tickScheduled=!1,e.then(function(e){t.dataIsReady=!0,t.data=e,t.max=e&&e.length||0,t.type=n.getTypeOf(e),t.isPaused||t._tickAndRepeat()},function(e){t.error(e)})}n.inherits(s,i),s.prototype.cleanUp=function(){i.prototype.cleanUp.call(this),this.data=null},s.prototype.resume=function(){return!!i.prototype.resume.call(this)&&(!this._tickScheduled&&this.dataIsReady&&(this._tickScheduled=!0,n.delay(this._tickAndRepeat,[],this)),!0)},s.prototype._tickAndRepeat=function(){this._tickScheduled=!1,this.isPaused||this.isFinished||(this._tick(),this.isFinished||(n.delay(this._tickAndRepeat,[],this),this._tickScheduled=!0))},s.prototype._tick=function(){if(this.isPaused||this.isFinished)return!1;var e=null,t=Math.min(this.max,this.index+16384);if(this.index>=this.max)return this.end();switch(this.type){case"string":e=this.data.substring(this.index,t);break;case"uint8array":e=this.data.subarray(this.index,t);break;case"array":case"nodebuffer":e=this.data.slice(this.index,t)}return this.index=t,this.push({data:e,meta:{percent:this.max?this.index/this.max*100:0}})},t.exports=s},{"../utils":32,"./GenericWorker":28}],28:[function(e,t,r){"use strict";function n(e){this.name=e||"default",this.streamInfo={},this.generatedError=null,this.extraStreamInfo={},this.isPaused=!0,this.isFinished=!1,this.isLocked=!1,this._listeners={data:[],end:[],error:[]},this.previous=null}n.prototype={push:function(e){this.emit("data",e)},end:function(){if(this.isFinished)return!1;this.flush();try{this.emit("end"),this.cleanUp(),this.isFinished=!0}catch(e){this.emit("error",e)}return!0},error:function(e){return!this.isFinished&&(this.isPaused?this.generatedError=e:(this.isFinished=!0,this.emit("error",e),this.previous&&this.previous.error(e),this.cleanUp()),!0)},on:function(e,t){return this._listeners[e].push(t),this},cleanUp:function(){this.streamInfo=this.generatedError=this.extraStreamInfo=null,this._listeners=[]},emit:function(e,t){if(this._listeners[e])for(var r=0;r<this._listeners[e].length;r++)this._listeners[e][r].call(this,t)},pipe:function(e){return e.registerPrevious(this)},registerPrevious:function(e){if(this.isLocked)throw new Error("The stream '"+this+"' has already been used.");this.streamInfo=e.streamInfo,this.mergeStreamInfo(),this.previous=e;var t=this;return e.on("data",function(e){t.processChunk(e)}),e.on("end",function(){t.end()}),e.on("error",function(e){t.error(e)}),this},pause:function(){return!this.isPaused&&!this.isFinished&&(this.isPaused=!0,this.previous&&this.previous.pause(),!0)},resume:function(){if(!this.isPaused||this.isFinished)return!1;var e=this.isPaused=!1;return this.generatedError&&(this.error(this.generatedError),e=!0),this.previous&&this.previous.resume(),!e},flush:function(){},processChunk:function(e){this.push(e)},withStreamInfo:function(e,t){return this.extraStreamInfo[e]=t,this.mergeStreamInfo(),this},mergeStreamInfo:function(){for(var e in this.extraStreamInfo)Object.prototype.hasOwnProperty.call(this.extraStreamInfo,e)&&(this.streamInfo[e]=this.extraStreamInfo[e])},lock:function(){if(this.isLocked)throw new Error("The stream '"+this+"' has already been used.");this.isLocked=!0,this.previous&&this.previous.lock()},toString:function(){var e="Worker "+this.name;return this.previous?this.previous+" -> "+e:e}},t.exports=n},{}],29:[function(e,t,r){"use strict";var h=e("../utils"),i=e("./ConvertWorker"),s=e("./GenericWorker"),u=e("../base64"),n=e("../support"),a=e("../external"),o=null;if(n.nodestream)try{o=e("../nodejs/NodejsStreamOutputAdapter")}catch(e){}function l(e,o){return new a.Promise(function(t,r){var n=[],i=e._internalType,s=e._outputType,a=e._mimeType;e.on("data",function(e,t){n.push(e),o&&o(t)}).on("error",function(e){n=[],r(e)}).on("end",function(){try{var e=function(e,t,r){switch(e){case"blob":return h.newBlob(h.transformTo("arraybuffer",t),r);case"base64":return u.encode(t);default:return h.transformTo(e,t)}}(s,function(e,t){var r,n=0,i=null,s=0;for(r=0;r<t.length;r++)s+=t[r].length;switch(e){case"string":return t.join("");case"array":return Array.prototype.concat.apply([],t);case"uint8array":for(i=new Uint8Array(s),r=0;r<t.length;r++)i.set(t[r],n),n+=t[r].length;return i;case"nodebuffer":return Buffer.concat(t);default:throw new Error("concat : unsupported type '"+e+"'")}}(i,n),a);t(e)}catch(e){r(e)}n=[]}).resume()})}function f(e,t,r){var n=t;switch(t){case"blob":case"arraybuffer":n="uint8array";break;case"base64":n="string"}try{this._internalType=n,this._outputType=t,this._mimeType=r,h.checkSupport(n),this._worker=e.pipe(new i(n)),e.lock()}catch(e){this._worker=new s("error"),this._worker.error(e)}}f.prototype={accumulate:function(e){return l(this,e)},on:function(e,t){var r=this;return"data"===e?this._worker.on(e,function(e){t.call(r,e.data,e.meta)}):this._worker.on(e,function(){h.delay(t,arguments,r)}),this},resume:function(){return h.delay(this._worker.resume,[],this._worker),this},pause:function(){return this._worker.pause(),this},toNodejsStream:function(e){if(h.checkSupport("nodestream"),"nodebuffer"!==this._outputType)throw new Error(this._outputType+" is not supported by this method");return new o(this,{objectMode:"nodebuffer"!==this._outputType},e)}},t.exports=f},{"../base64":1,"../external":6,"../nodejs/NodejsStreamOutputAdapter":13,"../support":30,"../utils":32,"./ConvertWorker":24,"./GenericWorker":28}],30:[function(e,t,r){"use strict";if(r.base64=!0,r.array=!0,r.string=!0,r.arraybuffer="undefined"!=typeof ArrayBuffer&&"undefined"!=typeof Uint8Array,r.nodebuffer="undefined"!=typeof Buffer,r.uint8array="undefined"!=typeof Uint8Array,"undefined"==typeof ArrayBuffer)r.blob=!1;else{var n=new ArrayBuffer(0);try{r.blob=0===new Blob([n],{type:"application/zip"}).size}catch(e){try{var i=new(self.BlobBuilder||self.WebKitBlobBuilder||self.MozBlobBuilder||self.MSBlobBuilder);i.append(n),r.blob=0===i.getBlob("application/zip").size}catch(e){r.blob=!1}}}try{r.nodestream=!!e("readable-stream").Readable}catch(e){r.nodestream=!1}},{"readable-stream":16}],31:[function(e,t,s){"use strict";for(var o=e("./utils"),h=e("./support"),r=e("./nodejsUtils"),n=e("./stream/GenericWorker"),u=new Array(256),i=0;i<256;i++)u[i]=252<=i?6:248<=i?5:240<=i?4:224<=i?3:192<=i?2:1;u[254]=u[254]=1;function a(){n.call(this,"utf-8 decode"),this.leftOver=null}function l(){n.call(this,"utf-8 encode")}s.utf8encode=function(e){return h.nodebuffer?r.newBufferFrom(e,"utf-8"):function(e){var t,r,n,i,s,a=e.length,o=0;for(i=0;i<a;i++)55296==(64512&(r=e.charCodeAt(i)))&&i+1<a&&56320==(64512&(n=e.charCodeAt(i+1)))&&(r=65536+(r-55296<<10)+(n-56320),i++),o+=r<128?1:r<2048?2:r<65536?3:4;for(t=h.uint8array?new Uint8Array(o):new Array(o),i=s=0;s<o;i++)55296==(64512&(r=e.charCodeAt(i)))&&i+1<a&&56320==(64512&(n=e.charCodeAt(i+1)))&&(r=65536+(r-55296<<10)+(n-56320),i++),r<128?t[s++]=r:(r<2048?t[s++]=192|r>>>6:(r<65536?t[s++]=224|r>>>12:(t[s++]=240|r>>>18,t[s++]=128|r>>>12&63),t[s++]=128|r>>>6&63),t[s++]=128|63&r);return t}(e)},s.utf8decode=function(e){return h.nodebuffer?o.transformTo("nodebuffer",e).toString("utf-8"):function(e){var t,r,n,i,s=e.length,a=new Array(2*s);for(t=r=0;t<s;)if((n=e[t++])<128)a[r++]=n;else if(4<(i=u[n]))a[r++]=65533,t+=i-1;else{for(n&=2===i?31:3===i?15:7;1<i&&t<s;)n=n<<6|63&e[t++],i--;1<i?a[r++]=65533:n<65536?a[r++]=n:(n-=65536,a[r++]=55296|n>>10&1023,a[r++]=56320|1023&n)}return a.length!==r&&(a.subarray?a=a.subarray(0,r):a.length=r),o.applyFromCharCode(a)}(e=o.transformTo(h.uint8array?"uint8array":"array",e))},o.inherits(a,n),a.prototype.processChunk=function(e){var t=o.transformTo(h.uint8array?"uint8array":"array",e.data);if(this.leftOver&&this.leftOver.length){if(h.uint8array){var r=t;(t=new Uint8Array(r.length+this.leftOver.length)).set(this.leftOver,0),t.set(r,this.leftOver.length)}else t=this.leftOver.concat(t);this.leftOver=null}var n=function(e,t){var r;for((t=t||e.length)>e.length&&(t=e.length),r=t-1;0<=r&&128==(192&e[r]);)r--;return r<0?t:0===r?t:r+u[e[r]]>t?r:t}(t),i=t;n!==t.length&&(h.uint8array?(i=t.subarray(0,n),this.leftOver=t.subarray(n,t.length)):(i=t.slice(0,n),this.leftOver=t.slice(n,t.length))),this.push({data:s.utf8decode(i),meta:e.meta})},a.prototype.flush=function(){this.leftOver&&this.leftOver.length&&(this.push({data:s.utf8decode(this.leftOver),meta:{}}),this.leftOver=null)},s.Utf8DecodeWorker=a,o.inherits(l,n),l.prototype.processChunk=function(e){this.push({data:s.utf8encode(e.data),meta:e.meta})},s.Utf8EncodeWorker=l},{"./nodejsUtils":14,"./stream/GenericWorker":28,"./support":30,"./utils":32}],32:[function(e,t,a){"use strict";var o=e("./support"),h=e("./base64"),r=e("./nodejsUtils"),u=e("./external");function n(e){return e}function l(e,t){for(var r=0;r<e.length;++r)t[r]=255&e.charCodeAt(r);return t}e("setimmediate"),a.newBlob=function(t,r){a.checkSupport("blob");try{return new Blob([t],{type:r})}catch(e){try{var n=new(self.BlobBuilder||self.WebKitBlobBuilder||self.MozBlobBuilder||self.MSBlobBuilder);return n.append(t),n.getBlob(r)}catch(e){throw new Error("Bug : can't construct the Blob.")}}};var i={stringifyByChunk:function(e,t,r){var n=[],i=0,s=e.length;if(s<=r)return String.fromCharCode.apply(null,e);for(;i<s;)"array"===t||"nodebuffer"===t?n.push(String.fromCharCode.apply(null,e.slice(i,Math.min(i+r,s)))):n.push(String.fromCharCode.apply(null,e.subarray(i,Math.min(i+r,s)))),i+=r;return n.join("")},stringifyByChar:function(e){for(var t="",r=0;r<e.length;r++)t+=String.fromCharCode(e[r]);return t},applyCanBeUsed:{uint8array:function(){try{return o.uint8array&&1===String.fromCharCode.apply(null,new Uint8Array(1)).length}catch(e){return!1}}(),nodebuffer:function(){try{return o.nodebuffer&&1===String.fromCharCode.apply(null,r.allocBuffer(1)).length}catch(e){return!1}}()}};function s(e){var t=65536,r=a.getTypeOf(e),n=!0;if("uint8array"===r?n=i.applyCanBeUsed.uint8array:"nodebuffer"===r&&(n=i.applyCanBeUsed.nodebuffer),n)for(;1<t;)try{return i.stringifyByChunk(e,r,t)}catch(e){t=Math.floor(t/2)}return i.stringifyByChar(e)}function f(e,t){for(var r=0;r<e.length;r++)t[r]=e[r];return t}a.applyFromCharCode=s;var c={};c.string={string:n,array:function(e){return l(e,new Array(e.length))},arraybuffer:function(e){return c.string.uint8array(e).buffer},uint8array:function(e){return l(e,new Uint8Array(e.length))},nodebuffer:function(e){return l(e,r.allocBuffer(e.length))}},c.array={string:s,array:n,arraybuffer:function(e){return new Uint8Array(e).buffer},uint8array:function(e){return new Uint8Array(e)},nodebuffer:function(e){return r.newBufferFrom(e)}},c.arraybuffer={string:function(e){return s(new Uint8Array(e))},array:function(e){return f(new Uint8Array(e),new Array(e.byteLength))},arraybuffer:n,uint8array:function(e){return new Uint8Array(e)},nodebuffer:function(e){return r.newBufferFrom(new Uint8Array(e))}},c.uint8array={string:s,array:function(e){return f(e,new Array(e.length))},arraybuffer:function(e){return e.buffer},uint8array:n,nodebuffer:function(e){return r.newBufferFrom(e)}},c.nodebuffer={string:s,array:function(e){return f(e,new Array(e.length))},arraybuffer:function(e){return c.nodebuffer.uint8array(e).buffer},uint8array:function(e){return f(e,new Uint8Array(e.length))},nodebuffer:n},a.transformTo=function(e,t){if(t=t||"",!e)return t;a.checkSupport(e);var r=a.getTypeOf(t);return c[r][e](t)},a.resolve=function(e){for(var t=e.split("/"),r=[],n=0;n<t.length;n++){var i=t[n];"."===i||""===i&&0!==n&&n!==t.length-1||(".."===i?r.pop():r.push(i))}return r.join("/")},a.getTypeOf=function(e){return"string"==typeof e?"string":"[object Array]"===Object.prototype.toString.call(e)?"array":o.nodebuffer&&r.isBuffer(e)?"nodebuffer":o.uint8array&&e instanceof Uint8Array?"uint8array":o.arraybuffer&&e instanceof ArrayBuffer?"arraybuffer":void 0},a.checkSupport=function(e){if(!o[e.toLowerCase()])throw new Error(e+" is not supported by this platform")},a.MAX_VALUE_16BITS=65535,a.MAX_VALUE_32BITS=-1,a.pretty=function(e){var t,r,n="";for(r=0;r<(e||"").length;r++)n+="\\x"+((t=e.charCodeAt(r))<16?"0":"")+t.toString(16).toUpperCase();return n},a.delay=function(e,t,r){setImmediate(function(){e.apply(r||null,t||[])})},a.inherits=function(e,t){function r(){}r.prototype=t.prototype,e.prototype=new r},a.extend=function(){var e,t,r={};for(e=0;e<arguments.length;e++)for(t in arguments[e])Object.prototype.hasOwnProperty.call(arguments[e],t)&&void 0===r[t]&&(r[t]=arguments[e][t]);return r},a.prepareContent=function(r,e,n,i,s){return u.Promise.resolve(e).then(function(n){return o.blob&&(n instanceof Blob||-1!==["[object File]","[object Blob]"].indexOf(Object.prototype.toString.call(n)))&&"undefined"!=typeof FileReader?new u.Promise(function(t,r){var e=new FileReader;e.onload=function(e){t(e.target.result)},e.onerror=function(e){r(e.target.error)},e.readAsArrayBuffer(n)}):n}).then(function(e){var t=a.getTypeOf(e);return t?("arraybuffer"===t?e=a.transformTo("uint8array",e):"string"===t&&(s?e=h.decode(e):n&&!0!==i&&(e=function(e){return l(e,o.uint8array?new Uint8Array(e.length):new Array(e.length))}(e))),e):u.Promise.reject(new Error("Can't read the data of '"+r+"'. Is it in a supported JavaScript type (String, Blob, ArrayBuffer, etc) ?"))})}},{"./base64":1,"./external":6,"./nodejsUtils":14,"./support":30,setimmediate:54}],33:[function(e,t,r){"use strict";var n=e("./reader/readerFor"),i=e("./utils"),s=e("./signature"),a=e("./zipEntry"),o=e("./support");function h(e){this.files=[],this.loadOptions=e}h.prototype={checkSignature:function(e){if(!this.reader.readAndCheckSignature(e)){this.reader.index-=4;var t=this.reader.readString(4);throw new Error("Corrupted zip or bug: unexpected signature ("+i.pretty(t)+", expected "+i.pretty(e)+")")}},isSignature:function(e,t){var r=this.reader.index;this.reader.setIndex(e);var n=this.reader.readString(4)===t;return this.reader.setIndex(r),n},readBlockEndOfCentral:function(){this.diskNumber=this.reader.readInt(2),this.diskWithCentralDirStart=this.reader.readInt(2),this.centralDirRecordsOnThisDisk=this.reader.readInt(2),this.centralDirRecords=this.reader.readInt(2),this.centralDirSize=this.reader.readInt(4),this.centralDirOffset=this.reader.readInt(4),this.zipCommentLength=this.reader.readInt(2);var e=this.reader.readData(this.zipCommentLength),t=o.uint8array?"uint8array":"array",r=i.transformTo(t,e);this.zipComment=this.loadOptions.decodeFileName(r)},readBlockZip64EndOfCentral:function(){this.zip64EndOfCentralSize=this.reader.readInt(8),this.reader.skip(4),this.diskNumber=this.reader.readInt(4),this.diskWithCentralDirStart=this.reader.readInt(4),this.centralDirRecordsOnThisDisk=this.reader.readInt(8),this.centralDirRecords=this.reader.readInt(8),this.centralDirSize=this.reader.readInt(8),this.centralDirOffset=this.reader.readInt(8),this.zip64ExtensibleData={};for(var e,t,r,n=this.zip64EndOfCentralSize-44;0<n;)e=this.reader.readInt(2),t=this.reader.readInt(4),r=this.reader.readData(t),this.zip64ExtensibleData[e]={id:e,length:t,value:r}},readBlockZip64EndOfCentralLocator:function(){if(this.diskWithZip64CentralDirStart=this.reader.readInt(4),this.relativeOffsetEndOfZip64CentralDir=this.reader.readInt(8),this.disksCount=this.reader.readInt(4),1<this.disksCount)throw new Error("Multi-volumes zip are not supported")},readLocalFiles:function(){var e,t;for(e=0;e<this.files.length;e++)t=this.files[e],this.reader.setIndex(t.localHeaderOffset),this.checkSignature(s.LOCAL_FILE_HEADER),t.readLocalPart(this.reader),t.handleUTF8(),t.processAttributes()},readCentralDir:function(){var e;for(this.reader.setIndex(this.centralDirOffset);this.reader.readAndCheckSignature(s.CENTRAL_FILE_HEADER);)(e=new a({zip64:this.zip64},this.loadOptions)).readCentralPart(this.reader),this.files.push(e);if(this.centralDirRecords!==this.files.length&&0!==this.centralDirRecords&&0===this.files.length)throw new Error("Corrupted zip or bug: expected "+this.centralDirRecords+" records in central dir, got "+this.files.length)},readEndOfCentral:function(){var e=this.reader.lastIndexOfSignature(s.CENTRAL_DIRECTORY_END);if(e<0)throw!this.isSignature(0,s.LOCAL_FILE_HEADER)?new Error("Can't find end of central directory : is this a zip file ? If it is, see https://stuk.github.io/jszip/documentation/howto/read_zip.html"):new Error("Corrupted zip: can't find end of central directory");this.reader.setIndex(e);var t=e;if(this.checkSignature(s.CENTRAL_DIRECTORY_END),this.readBlockEndOfCentral(),this.diskNumber===i.MAX_VALUE_16BITS||this.diskWithCentralDirStart===i.MAX_VALUE_16BITS||this.centralDirRecordsOnThisDisk===i.MAX_VALUE_16BITS||this.centralDirRecords===i.MAX_VALUE_16BITS||this.centralDirSize===i.MAX_VALUE_32BITS||this.centralDirOffset===i.MAX_VALUE_32BITS){if(this.zip64=!0,(e=this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR))<0)throw new Error("Corrupted zip: can't find the ZIP64 end of central directory locator");if(this.reader.setIndex(e),this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR),this.readBlockZip64EndOfCentralLocator(),!this.isSignature(this.relativeOffsetEndOfZip64CentralDir,s.ZIP64_CENTRAL_DIRECTORY_END)&&(this.relativeOffsetEndOfZip64CentralDir=this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_END),this.relativeOffsetEndOfZip64CentralDir<0))throw new Error("Corrupted zip: can't find the ZIP64 end of central directory");this.reader.setIndex(this.relativeOffsetEndOfZip64CentralDir),this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_END),this.readBlockZip64EndOfCentral()}var r=this.centralDirOffset+this.centralDirSize;this.zip64&&(r+=20,r+=12+this.zip64EndOfCentralSize);var n=t-r;if(0<n)this.isSignature(t,s.CENTRAL_FILE_HEADER)||(this.reader.zero=n);else if(n<0)throw new Error("Corrupted zip: missing "+Math.abs(n)+" bytes.")},prepareReader:function(e){this.reader=n(e)},load:function(e){this.prepareReader(e),this.readEndOfCentral(),this.readCentralDir(),this.readLocalFiles()}},t.exports=h},{"./reader/readerFor":22,"./signature":23,"./support":30,"./utils":32,"./zipEntry":34}],34:[function(e,t,r){"use strict";var n=e("./reader/readerFor"),s=e("./utils"),i=e("./compressedObject"),a=e("./crc32"),o=e("./utf8"),h=e("./compressions"),u=e("./support");function l(e,t){this.options=e,this.loadOptions=t}l.prototype={isEncrypted:function(){return 1==(1&this.bitFlag)},useUTF8:function(){return 2048==(2048&this.bitFlag)},readLocalPart:function(e){var t,r;if(e.skip(22),this.fileNameLength=e.readInt(2),r=e.readInt(2),this.fileName=e.readData(this.fileNameLength),e.skip(r),-1===this.compressedSize||-1===this.uncompressedSize)throw new Error("Bug or corrupted zip : didn't get enough information from the central directory (compressedSize === -1 || uncompressedSize === -1)");if(null===(t=function(e){for(var t in h)if(Object.prototype.hasOwnProperty.call(h,t)&&h[t].magic===e)return h[t];return null}(this.compressionMethod)))throw new Error("Corrupted zip : compression "+s.pretty(this.compressionMethod)+" unknown (inner file : "+s.transformTo("string",this.fileName)+")");this.decompressed=new i(this.compressedSize,this.uncompressedSize,this.crc32,t,e.readData(this.compressedSize))},readCentralPart:function(e){this.versionMadeBy=e.readInt(2),e.skip(2),this.bitFlag=e.readInt(2),this.compressionMethod=e.readString(2),this.date=e.readDate(),this.crc32=e.readInt(4),this.compressedSize=e.readInt(4),this.uncompressedSize=e.readInt(4);var t=e.readInt(2);if(this.extraFieldsLength=e.readInt(2),this.fileCommentLength=e.readInt(2),this.diskNumberStart=e.readInt(2),this.internalFileAttributes=e.readInt(2),this.externalFileAttributes=e.readInt(4),this.localHeaderOffset=e.readInt(4),this.isEncrypted())throw new Error("Encrypted zip are not supported");e.skip(t),this.readExtraFields(e),this.parseZIP64ExtraField(e),this.fileComment=e.readData(this.fileCommentLength)},processAttributes:function(){this.unixPermissions=null,this.dosPermissions=null;var e=this.versionMadeBy>>8;this.dir=!!(16&this.externalFileAttributes),0==e&&(this.dosPermissions=63&this.externalFileAttributes),3==e&&(this.unixPermissions=this.externalFileAttributes>>16&65535),this.dir||"/"!==this.fileNameStr.slice(-1)||(this.dir=!0)},parseZIP64ExtraField:function(){if(this.extraFields[1]){var e=n(this.extraFields[1].value);this.uncompressedSize===s.MAX_VALUE_32BITS&&(this.uncompressedSize=e.readInt(8)),this.compressedSize===s.MAX_VALUE_32BITS&&(this.compressedSize=e.readInt(8)),this.localHeaderOffset===s.MAX_VALUE_32BITS&&(this.localHeaderOffset=e.readInt(8)),this.diskNumberStart===s.MAX_VALUE_32BITS&&(this.diskNumberStart=e.readInt(4))}},readExtraFields:function(e){var t,r,n,i=e.index+this.extraFieldsLength;for(this.extraFields||(this.extraFields={});e.index+4<i;)t=e.readInt(2),r=e.readInt(2),n=e.readData(r),this.extraFields[t]={id:t,length:r,value:n};e.setIndex(i)},handleUTF8:function(){var e=u.uint8array?"uint8array":"array";if(this.useUTF8())this.fileNameStr=o.utf8decode(this.fileName),this.fileCommentStr=o.utf8decode(this.fileComment);else{var t=this.findExtraFieldUnicodePath();if(null!==t)this.fileNameStr=t;else{var r=s.transformTo(e,this.fileName);this.fileNameStr=this.loadOptions.decodeFileName(r)}var n=this.findExtraFieldUnicodeComment();if(null!==n)this.fileCommentStr=n;else{var i=s.transformTo(e,this.fileComment);this.fileCommentStr=this.loadOptions.decodeFileName(i)}}},findExtraFieldUnicodePath:function(){var e=this.extraFields[28789];if(e){var t=n(e.value);return 1!==t.readInt(1)?null:a(this.fileName)!==t.readInt(4)?null:o.utf8decode(t.readData(e.length-5))}return null},findExtraFieldUnicodeComment:function(){var e=this.extraFields[25461];if(e){var t=n(e.value);return 1!==t.readInt(1)?null:a(this.fileComment)!==t.readInt(4)?null:o.utf8decode(t.readData(e.length-5))}return null}},t.exports=l},{"./compressedObject":2,"./compressions":3,"./crc32":4,"./reader/readerFor":22,"./support":30,"./utf8":31,"./utils":32}],35:[function(e,t,r){"use strict";function n(e,t,r){this.name=e,this.dir=r.dir,this.date=r.date,this.comment=r.comment,this.unixPermissions=r.unixPermissions,this.dosPermissions=r.dosPermissions,this._data=t,this._dataBinary=r.binary,this.options={compression:r.compression,compressionOptions:r.compressionOptions}}var s=e("./stream/StreamHelper"),i=e("./stream/DataWorker"),a=e("./utf8"),o=e("./compressedObject"),h=e("./stream/GenericWorker");n.prototype={internalStream:function(e){var t=null,r="string";try{if(!e)throw new Error("No output type specified.");var n="string"===(r=e.toLowerCase())||"text"===r;"binarystring"!==r&&"text"!==r||(r="string"),t=this._decompressWorker();var i=!this._dataBinary;i&&!n&&(t=t.pipe(new a.Utf8EncodeWorker)),!i&&n&&(t=t.pipe(new a.Utf8DecodeWorker))}catch(e){(t=new h("error")).error(e)}return new s(t,r,"")},async:function(e,t){return this.internalStream(e).accumulate(t)},nodeStream:function(e,t){return this.internalStream(e||"nodebuffer").toNodejsStream(t)},_compressWorker:function(e,t){if(this._data instanceof o&&this._data.compression.magic===e.magic)return this._data.getCompressedWorker();var r=this._decompressWorker();return this._dataBinary||(r=r.pipe(new a.Utf8EncodeWorker)),o.createWorkerFrom(r,e,t)},_decompressWorker:function(){return this._data instanceof o?this._data.getContentWorker():this._data instanceof h?this._data:new i(this._data)}};for(var u=["asText","asBinary","asNodeBuffer","asUint8Array","asArrayBuffer"],l=function(){throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.")},f=0;f<u.length;f++)n.prototype[u[f]]=l;t.exports=n},{"./compressedObject":2,"./stream/DataWorker":27,"./stream/GenericWorker":28,"./stream/StreamHelper":29,"./utf8":31}],36:[function(e,l,t){(function(t){"use strict";var r,n,e=t.MutationObserver||t.WebKitMutationObserver;if(e){var i=0,s=new e(u),a=t.document.createTextNode("");s.observe(a,{characterData:!0}),r=function(){a.data=i=++i%2}}else if(t.setImmediate||void 0===t.MessageChannel)r="document"in t&&"onreadystatechange"in t.document.createElement("script")?function(){var e=t.document.createElement("script");e.onreadystatechange=function(){u(),e.onreadystatechange=null,e.parentNode.removeChild(e),e=null},t.document.documentElement.appendChild(e)}:function(){setTimeout(u,0)};else{var o=new t.MessageChannel;o.port1.onmessage=u,r=function(){o.port2.postMessage(0)}}var h=[];function u(){var e,t;n=!0;for(var r=h.length;r;){for(t=h,h=[],e=-1;++e<r;)t[e]();r=h.length}n=!1}l.exports=function(e){1!==h.push(e)||n||r()}}).call(this,"undefined"!=typeof global?global:"undefined"!=typeof self?self:"undefined"!=typeof window?window:{})},{}],37:[function(e,t,r){"use strict";var i=e("immediate");function u(){}var l={},s=["REJECTED"],a=["FULFILLED"],n=["PENDING"];function o(e){if("function"!=typeof e)throw new TypeError("resolver must be a function");this.state=n,this.queue=[],this.outcome=void 0,e!==u&&d(this,e)}function h(e,t,r){this.promise=e,"function"==typeof t&&(this.onFulfilled=t,this.callFulfilled=this.otherCallFulfilled),"function"==typeof r&&(this.onRejected=r,this.callRejected=this.otherCallRejected)}function f(t,r,n){i(function(){var e;try{e=r(n)}catch(e){return l.reject(t,e)}e===t?l.reject(t,new TypeError("Cannot resolve promise with itself")):l.resolve(t,e)})}function c(e){var t=e&&e.then;if(e&&("object"==typeof e||"function"==typeof e)&&"function"==typeof t)return function(){t.apply(e,arguments)}}function d(t,e){var r=!1;function n(e){r||(r=!0,l.reject(t,e))}function i(e){r||(r=!0,l.resolve(t,e))}var s=p(function(){e(i,n)});"error"===s.status&&n(s.value)}function p(e,t){var r={};try{r.value=e(t),r.status="success"}catch(e){r.status="error",r.value=e}return r}(t.exports=o).prototype.finally=function(t){if("function"!=typeof t)return this;var r=this.constructor;return this.then(function(e){return r.resolve(t()).then(function(){return e})},function(e){return r.resolve(t()).then(function(){throw e})})},o.prototype.catch=function(e){return this.then(null,e)},o.prototype.then=function(e,t){if("function"!=typeof e&&this.state===a||"function"!=typeof t&&this.state===s)return this;var r=new this.constructor(u);this.state!==n?f(r,this.state===a?e:t,this.outcome):this.queue.push(new h(r,e,t));return r},h.prototype.callFulfilled=function(e){l.resolve(this.promise,e)},h.prototype.otherCallFulfilled=function(e){f(this.promise,this.onFulfilled,e)},h.prototype.callRejected=function(e){l.reject(this.promise,e)},h.prototype.otherCallRejected=function(e){f(this.promise,this.onRejected,e)},l.resolve=function(e,t){var r=p(c,t);if("error"===r.status)return l.reject(e,r.value);var n=r.value;if(n)d(e,n);else{e.state=a,e.outcome=t;for(var i=-1,s=e.queue.length;++i<s;)e.queue[i].callFulfilled(t)}return e},l.reject=function(e,t){e.state=s,e.outcome=t;for(var r=-1,n=e.queue.length;++r<n;)e.queue[r].callRejected(t);return e},o.resolve=function(e){if(e instanceof this)return e;return l.resolve(new this(u),e)},o.reject=function(e){var t=new this(u);return l.reject(t,e)},o.all=function(e){var r=this;if("[object Array]"!==Object.prototype.toString.call(e))return this.reject(new TypeError("must be an array"));var n=e.length,i=!1;if(!n)return this.resolve([]);var s=new Array(n),a=0,t=-1,o=new this(u);for(;++t<n;)h(e[t],t);return o;function h(e,t){r.resolve(e).then(function(e){s[t]=e,++a!==n||i||(i=!0,l.resolve(o,s))},function(e){i||(i=!0,l.reject(o,e))})}},o.race=function(e){var t=this;if("[object Array]"!==Object.prototype.toString.call(e))return this.reject(new TypeError("must be an array"));var r=e.length,n=!1;if(!r)return this.resolve([]);var i=-1,s=new this(u);for(;++i<r;)a=e[i],t.resolve(a).then(function(e){n||(n=!0,l.resolve(s,e))},function(e){n||(n=!0,l.reject(s,e))});var a;return s}},{immediate:36}],38:[function(e,t,r){"use strict";var n={};(0,e("./lib/utils/common").assign)(n,e("./lib/deflate"),e("./lib/inflate"),e("./lib/zlib/constants")),t.exports=n},{"./lib/deflate":39,"./lib/inflate":40,"./lib/utils/common":41,"./lib/zlib/constants":44}],39:[function(e,t,r){"use strict";var a=e("./zlib/deflate"),o=e("./utils/common"),h=e("./utils/strings"),i=e("./zlib/messages"),s=e("./zlib/zstream"),u=Object.prototype.toString,l=0,f=-1,c=0,d=8;function p(e){if(!(this instanceof p))return new p(e);this.options=o.assign({level:f,method:d,chunkSize:16384,windowBits:15,memLevel:8,strategy:c,to:""},e||{});var t=this.options;t.raw&&0<t.windowBits?t.windowBits=-t.windowBits:t.gzip&&0<t.windowBits&&t.windowBits<16&&(t.windowBits+=16),this.err=0,this.msg="",this.ended=!1,this.chunks=[],this.strm=new s,this.strm.avail_out=0;var r=a.deflateInit2(this.strm,t.level,t.method,t.windowBits,t.memLevel,t.strategy);if(r!==l)throw new Error(i[r]);if(t.header&&a.deflateSetHeader(this.strm,t.header),t.dictionary){var n;if(n="string"==typeof t.dictionary?h.string2buf(t.dictionary):"[object ArrayBuffer]"===u.call(t.dictionary)?new Uint8Array(t.dictionary):t.dictionary,(r=a.deflateSetDictionary(this.strm,n))!==l)throw new Error(i[r]);this._dict_set=!0}}function n(e,t){var r=new p(t);if(r.push(e,!0),r.err)throw r.msg||i[r.err];return r.result}p.prototype.push=function(e,t){var r,n,i=this.strm,s=this.options.chunkSize;if(this.ended)return!1;n=t===~~t?t:!0===t?4:0,"string"==typeof e?i.input=h.string2buf(e):"[object ArrayBuffer]"===u.call(e)?i.input=new Uint8Array(e):i.input=e,i.next_in=0,i.avail_in=i.input.length;do{if(0===i.avail_out&&(i.output=new o.Buf8(s),i.next_out=0,i.avail_out=s),1!==(r=a.deflate(i,n))&&r!==l)return this.onEnd(r),!(this.ended=!0);0!==i.avail_out&&(0!==i.avail_in||4!==n&&2!==n)||("string"===this.options.to?this.onData(h.buf2binstring(o.shrinkBuf(i.output,i.next_out))):this.onData(o.shrinkBuf(i.output,i.next_out)))}while((0<i.avail_in||0===i.avail_out)&&1!==r);return 4===n?(r=a.deflateEnd(this.strm),this.onEnd(r),this.ended=!0,r===l):2!==n||(this.onEnd(l),!(i.avail_out=0))},p.prototype.onData=function(e){this.chunks.push(e)},p.prototype.onEnd=function(e){e===l&&("string"===this.options.to?this.result=this.chunks.join(""):this.result=o.flattenChunks(this.chunks)),this.chunks=[],this.err=e,this.msg=this.strm.msg},r.Deflate=p,r.deflate=n,r.deflateRaw=function(e,t){return(t=t||{}).raw=!0,n(e,t)},r.gzip=function(e,t){return(t=t||{}).gzip=!0,n(e,t)}},{"./utils/common":41,"./utils/strings":42,"./zlib/deflate":46,"./zlib/messages":51,"./zlib/zstream":53}],40:[function(e,t,r){"use strict";var c=e("./zlib/inflate"),d=e("./utils/common"),p=e("./utils/strings"),m=e("./zlib/constants"),n=e("./zlib/messages"),i=e("./zlib/zstream"),s=e("./zlib/gzheader"),_=Object.prototype.toString;function a(e){if(!(this instanceof a))return new a(e);this.options=d.assign({chunkSize:16384,windowBits:0,to:""},e||{});var t=this.options;t.raw&&0<=t.windowBits&&t.windowBits<16&&(t.windowBits=-t.windowBits,0===t.windowBits&&(t.windowBits=-15)),!(0<=t.windowBits&&t.windowBits<16)||e&&e.windowBits||(t.windowBits+=32),15<t.windowBits&&t.windowBits<48&&0==(15&t.windowBits)&&(t.windowBits|=15),this.err=0,this.msg="",this.ended=!1,this.chunks=[],this.strm=new i,this.strm.avail_out=0;var r=c.inflateInit2(this.strm,t.windowBits);if(r!==m.Z_OK)throw new Error(n[r]);this.header=new s,c.inflateGetHeader(this.strm,this.header)}function o(e,t){var r=new a(t);if(r.push(e,!0),r.err)throw r.msg||n[r.err];return r.result}a.prototype.push=function(e,t){var r,n,i,s,a,o,h=this.strm,u=this.options.chunkSize,l=this.options.dictionary,f=!1;if(this.ended)return!1;n=t===~~t?t:!0===t?m.Z_FINISH:m.Z_NO_FLUSH,"string"==typeof e?h.input=p.binstring2buf(e):"[object ArrayBuffer]"===_.call(e)?h.input=new Uint8Array(e):h.input=e,h.next_in=0,h.avail_in=h.input.length;do{if(0===h.avail_out&&(h.output=new d.Buf8(u),h.next_out=0,h.avail_out=u),(r=c.inflate(h,m.Z_NO_FLUSH))===m.Z_NEED_DICT&&l&&(o="string"==typeof l?p.string2buf(l):"[object ArrayBuffer]"===_.call(l)?new Uint8Array(l):l,r=c.inflateSetDictionary(this.strm,o)),r===m.Z_BUF_ERROR&&!0===f&&(r=m.Z_OK,f=!1),r!==m.Z_STREAM_END&&r!==m.Z_OK)return this.onEnd(r),!(this.ended=!0);h.next_out&&(0!==h.avail_out&&r!==m.Z_STREAM_END&&(0!==h.avail_in||n!==m.Z_FINISH&&n!==m.Z_SYNC_FLUSH)||("string"===this.options.to?(i=p.utf8border(h.output,h.next_out),s=h.next_out-i,a=p.buf2string(h.output,i),h.next_out=s,h.avail_out=u-s,s&&d.arraySet(h.output,h.output,i,s,0),this.onData(a)):this.onData(d.shrinkBuf(h.output,h.next_out)))),0===h.avail_in&&0===h.avail_out&&(f=!0)}while((0<h.avail_in||0===h.avail_out)&&r!==m.Z_STREAM_END);return r===m.Z_STREAM_END&&(n=m.Z_FINISH),n===m.Z_FINISH?(r=c.inflateEnd(this.strm),this.onEnd(r),this.ended=!0,r===m.Z_OK):n!==m.Z_SYNC_FLUSH||(this.onEnd(m.Z_OK),!(h.avail_out=0))},a.prototype.onData=function(e){this.chunks.push(e)},a.prototype.onEnd=function(e){e===m.Z_OK&&("string"===this.options.to?this.result=this.chunks.join(""):this.result=d.flattenChunks(this.chunks)),this.chunks=[],this.err=e,this.msg=this.strm.msg},r.Inflate=a,r.inflate=o,r.inflateRaw=function(e,t){return(t=t||{}).raw=!0,o(e,t)},r.ungzip=o},{"./utils/common":41,"./utils/strings":42,"./zlib/constants":44,"./zlib/gzheader":47,"./zlib/inflate":49,"./zlib/messages":51,"./zlib/zstream":53}],41:[function(e,t,r){"use strict";var n="undefined"!=typeof Uint8Array&&"undefined"!=typeof Uint16Array&&"undefined"!=typeof Int32Array;r.assign=function(e){for(var t=Array.prototype.slice.call(arguments,1);t.length;){var r=t.shift();if(r){if("object"!=typeof r)throw new TypeError(r+"must be non-object");for(var n in r)r.hasOwnProperty(n)&&(e[n]=r[n])}}return e},r.shrinkBuf=function(e,t){return e.length===t?e:e.subarray?e.subarray(0,t):(e.length=t,e)};var i={arraySet:function(e,t,r,n,i){if(t.subarray&&e.subarray)e.set(t.subarray(r,r+n),i);else for(var s=0;s<n;s++)e[i+s]=t[r+s]},flattenChunks:function(e){var t,r,n,i,s,a;for(t=n=0,r=e.length;t<r;t++)n+=e[t].length;for(a=new Uint8Array(n),t=i=0,r=e.length;t<r;t++)s=e[t],a.set(s,i),i+=s.length;return a}},s={arraySet:function(e,t,r,n,i){for(var s=0;s<n;s++)e[i+s]=t[r+s]},flattenChunks:function(e){return[].concat.apply([],e)}};r.setTyped=function(e){e?(r.Buf8=Uint8Array,r.Buf16=Uint16Array,r.Buf32=Int32Array,r.assign(r,i)):(r.Buf8=Array,r.Buf16=Array,r.Buf32=Array,r.assign(r,s))},r.setTyped(n)},{}],42:[function(e,t,r){"use strict";var h=e("./common"),i=!0,s=!0;try{String.fromCharCode.apply(null,[0])}catch(e){i=!1}try{String.fromCharCode.apply(null,new Uint8Array(1))}catch(e){s=!1}for(var u=new h.Buf8(256),n=0;n<256;n++)u[n]=252<=n?6:248<=n?5:240<=n?4:224<=n?3:192<=n?2:1;function l(e,t){if(t<65537&&(e.subarray&&s||!e.subarray&&i))return String.fromCharCode.apply(null,h.shrinkBuf(e,t));for(var r="",n=0;n<t;n++)r+=String.fromCharCode(e[n]);return r}u[254]=u[254]=1,r.string2buf=function(e){var t,r,n,i,s,a=e.length,o=0;for(i=0;i<a;i++)55296==(64512&(r=e.charCodeAt(i)))&&i+1<a&&56320==(64512&(n=e.charCodeAt(i+1)))&&(r=65536+(r-55296<<10)+(n-56320),i++),o+=r<128?1:r<2048?2:r<65536?3:4;for(t=new h.Buf8(o),i=s=0;s<o;i++)55296==(64512&(r=e.charCodeAt(i)))&&i+1<a&&56320==(64512&(n=e.charCodeAt(i+1)))&&(r=65536+(r-55296<<10)+(n-56320),i++),r<128?t[s++]=r:(r<2048?t[s++]=192|r>>>6:(r<65536?t[s++]=224|r>>>12:(t[s++]=240|r>>>18,t[s++]=128|r>>>12&63),t[s++]=128|r>>>6&63),t[s++]=128|63&r);return t},r.buf2binstring=function(e){return l(e,e.length)},r.binstring2buf=function(e){for(var t=new h.Buf8(e.length),r=0,n=t.length;r<n;r++)t[r]=e.charCodeAt(r);return t},r.buf2string=function(e,t){var r,n,i,s,a=t||e.length,o=new Array(2*a);for(r=n=0;r<a;)if((i=e[r++])<128)o[n++]=i;else if(4<(s=u[i]))o[n++]=65533,r+=s-1;else{for(i&=2===s?31:3===s?15:7;1<s&&r<a;)i=i<<6|63&e[r++],s--;1<s?o[n++]=65533:i<65536?o[n++]=i:(i-=65536,o[n++]=55296|i>>10&1023,o[n++]=56320|1023&i)}return l(o,n)},r.utf8border=function(e,t){var r;for((t=t||e.length)>e.length&&(t=e.length),r=t-1;0<=r&&128==(192&e[r]);)r--;return r<0?t:0===r?t:r+u[e[r]]>t?r:t}},{"./common":41}],43:[function(e,t,r){"use strict";t.exports=function(e,t,r,n){for(var i=65535&e|0,s=e>>>16&65535|0,a=0;0!==r;){for(r-=a=2e3<r?2e3:r;s=s+(i=i+t[n++]|0)|0,--a;);i%=65521,s%=65521}return i|s<<16|0}},{}],44:[function(e,t,r){"use strict";t.exports={Z_NO_FLUSH:0,Z_PARTIAL_FLUSH:1,Z_SYNC_FLUSH:2,Z_FULL_FLUSH:3,Z_FINISH:4,Z_BLOCK:5,Z_TREES:6,Z_OK:0,Z_STREAM_END:1,Z_NEED_DICT:2,Z_ERRNO:-1,Z_STREAM_ERROR:-2,Z_DATA_ERROR:-3,Z_BUF_ERROR:-5,Z_NO_COMPRESSION:0,Z_BEST_SPEED:1,Z_BEST_COMPRESSION:9,Z_DEFAULT_COMPRESSION:-1,Z_FILTERED:1,Z_HUFFMAN_ONLY:2,Z_RLE:3,Z_FIXED:4,Z_DEFAULT_STRATEGY:0,Z_BINARY:0,Z_TEXT:1,Z_UNKNOWN:2,Z_DEFLATED:8}},{}],45:[function(e,t,r){"use strict";var o=function(){for(var e,t=[],r=0;r<256;r++){e=r;for(var n=0;n<8;n++)e=1&e?3988292384^e>>>1:e>>>1;t[r]=e}return t}();t.exports=function(e,t,r,n){var i=o,s=n+r;e^=-1;for(var a=n;a<s;a++)e=e>>>8^i[255&(e^t[a])];return-1^e}},{}],46:[function(e,t,r){"use strict";var h,c=e("../utils/common"),u=e("./trees"),d=e("./adler32"),p=e("./crc32"),n=e("./messages"),l=0,f=4,m=0,_=-2,g=-1,b=4,i=2,v=8,y=9,s=286,a=30,o=19,w=2*s+1,k=15,x=3,S=258,z=S+x+1,C=42,E=113,A=1,I=2,O=3,B=4;function R(e,t){return e.msg=n[t],t}function T(e){return(e<<1)-(4<e?9:0)}function D(e){for(var t=e.length;0<=--t;)e[t]=0}function F(e){var t=e.state,r=t.pending;r>e.avail_out&&(r=e.avail_out),0!==r&&(c.arraySet(e.output,t.pending_buf,t.pending_out,r,e.next_out),e.next_out+=r,t.pending_out+=r,e.total_out+=r,e.avail_out-=r,t.pending-=r,0===t.pending&&(t.pending_out=0))}function N(e,t){u._tr_flush_block(e,0<=e.block_start?e.block_start:-1,e.strstart-e.block_start,t),e.block_start=e.strstart,F(e.strm)}function U(e,t){e.pending_buf[e.pending++]=t}function P(e,t){e.pending_buf[e.pending++]=t>>>8&255,e.pending_buf[e.pending++]=255&t}function L(e,t){var r,n,i=e.max_chain_length,s=e.strstart,a=e.prev_length,o=e.nice_match,h=e.strstart>e.w_size-z?e.strstart-(e.w_size-z):0,u=e.window,l=e.w_mask,f=e.prev,c=e.strstart+S,d=u[s+a-1],p=u[s+a];e.prev_length>=e.good_match&&(i>>=2),o>e.lookahead&&(o=e.lookahead);do{if(u[(r=t)+a]===p&&u[r+a-1]===d&&u[r]===u[s]&&u[++r]===u[s+1]){s+=2,r++;do{}while(u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&s<c);if(n=S-(c-s),s=c-S,a<n){if(e.match_start=t,o<=(a=n))break;d=u[s+a-1],p=u[s+a]}}}while((t=f[t&l])>h&&0!=--i);return a<=e.lookahead?a:e.lookahead}function j(e){var t,r,n,i,s,a,o,h,u,l,f=e.w_size;do{if(i=e.window_size-e.lookahead-e.strstart,e.strstart>=f+(f-z)){for(c.arraySet(e.window,e.window,f,f,0),e.match_start-=f,e.strstart-=f,e.block_start-=f,t=r=e.hash_size;n=e.head[--t],e.head[t]=f<=n?n-f:0,--r;);for(t=r=f;n=e.prev[--t],e.prev[t]=f<=n?n-f:0,--r;);i+=f}if(0===e.strm.avail_in)break;if(a=e.strm,o=e.window,h=e.strstart+e.lookahead,u=i,l=void 0,l=a.avail_in,u<l&&(l=u),r=0===l?0:(a.avail_in-=l,c.arraySet(o,a.input,a.next_in,l,h),1===a.state.wrap?a.adler=d(a.adler,o,l,h):2===a.state.wrap&&(a.adler=p(a.adler,o,l,h)),a.next_in+=l,a.total_in+=l,l),e.lookahead+=r,e.lookahead+e.insert>=x)for(s=e.strstart-e.insert,e.ins_h=e.window[s],e.ins_h=(e.ins_h<<e.hash_shift^e.window[s+1])&e.hash_mask;e.insert&&(e.ins_h=(e.ins_h<<e.hash_shift^e.window[s+x-1])&e.hash_mask,e.prev[s&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=s,s++,e.insert--,!(e.lookahead+e.insert<x)););}while(e.lookahead<z&&0!==e.strm.avail_in)}function Z(e,t){for(var r,n;;){if(e.lookahead<z){if(j(e),e.lookahead<z&&t===l)return A;if(0===e.lookahead)break}if(r=0,e.lookahead>=x&&(e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+x-1])&e.hash_mask,r=e.prev[e.strstart&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=e.strstart),0!==r&&e.strstart-r<=e.w_size-z&&(e.match_length=L(e,r)),e.match_length>=x)if(n=u._tr_tally(e,e.strstart-e.match_start,e.match_length-x),e.lookahead-=e.match_length,e.match_length<=e.max_lazy_match&&e.lookahead>=x){for(e.match_length--;e.strstart++,e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+x-1])&e.hash_mask,r=e.prev[e.strstart&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=e.strstart,0!=--e.match_length;);e.strstart++}else e.strstart+=e.match_length,e.match_length=0,e.ins_h=e.window[e.strstart],e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+1])&e.hash_mask;else n=u._tr_tally(e,0,e.window[e.strstart]),e.lookahead--,e.strstart++;if(n&&(N(e,!1),0===e.strm.avail_out))return A}return e.insert=e.strstart<x-1?e.strstart:x-1,t===f?(N(e,!0),0===e.strm.avail_out?O:B):e.last_lit&&(N(e,!1),0===e.strm.avail_out)?A:I}function W(e,t){for(var r,n,i;;){if(e.lookahead<z){if(j(e),e.lookahead<z&&t===l)return A;if(0===e.lookahead)break}if(r=0,e.lookahead>=x&&(e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+x-1])&e.hash_mask,r=e.prev[e.strstart&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=e.strstart),e.prev_length=e.match_length,e.prev_match=e.match_start,e.match_length=x-1,0!==r&&e.prev_length<e.max_lazy_match&&e.strstart-r<=e.w_size-z&&(e.match_length=L(e,r),e.match_length<=5&&(1===e.strategy||e.match_length===x&&4096<e.strstart-e.match_start)&&(e.match_length=x-1)),e.prev_length>=x&&e.match_length<=e.prev_length){for(i=e.strstart+e.lookahead-x,n=u._tr_tally(e,e.strstart-1-e.prev_match,e.prev_length-x),e.lookahead-=e.prev_length-1,e.prev_length-=2;++e.strstart<=i&&(e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+x-1])&e.hash_mask,r=e.prev[e.strstart&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=e.strstart),0!=--e.prev_length;);if(e.match_available=0,e.match_length=x-1,e.strstart++,n&&(N(e,!1),0===e.strm.avail_out))return A}else if(e.match_available){if((n=u._tr_tally(e,0,e.window[e.strstart-1]))&&N(e,!1),e.strstart++,e.lookahead--,0===e.strm.avail_out)return A}else e.match_available=1,e.strstart++,e.lookahead--}return e.match_available&&(n=u._tr_tally(e,0,e.window[e.strstart-1]),e.match_available=0),e.insert=e.strstart<x-1?e.strstart:x-1,t===f?(N(e,!0),0===e.strm.avail_out?O:B):e.last_lit&&(N(e,!1),0===e.strm.avail_out)?A:I}function M(e,t,r,n,i){this.good_length=e,this.max_lazy=t,this.nice_length=r,this.max_chain=n,this.func=i}function H(){this.strm=null,this.status=0,this.pending_buf=null,this.pending_buf_size=0,this.pending_out=0,this.pending=0,this.wrap=0,this.gzhead=null,this.gzindex=0,this.method=v,this.last_flush=-1,this.w_size=0,this.w_bits=0,this.w_mask=0,this.window=null,this.window_size=0,this.prev=null,this.head=null,this.ins_h=0,this.hash_size=0,this.hash_bits=0,this.hash_mask=0,this.hash_shift=0,this.block_start=0,this.match_length=0,this.prev_match=0,this.match_available=0,this.strstart=0,this.match_start=0,this.lookahead=0,this.prev_length=0,this.max_chain_length=0,this.max_lazy_match=0,this.level=0,this.strategy=0,this.good_match=0,this.nice_match=0,this.dyn_ltree=new c.Buf16(2*w),this.dyn_dtree=new c.Buf16(2*(2*a+1)),this.bl_tree=new c.Buf16(2*(2*o+1)),D(this.dyn_ltree),D(this.dyn_dtree),D(this.bl_tree),this.l_desc=null,this.d_desc=null,this.bl_desc=null,this.bl_count=new c.Buf16(k+1),this.heap=new c.Buf16(2*s+1),D(this.heap),this.heap_len=0,this.heap_max=0,this.depth=new c.Buf16(2*s+1),D(this.depth),this.l_buf=0,this.lit_bufsize=0,this.last_lit=0,this.d_buf=0,this.opt_len=0,this.static_len=0,this.matches=0,this.insert=0,this.bi_buf=0,this.bi_valid=0}function G(e){var t;return e&&e.state?(e.total_in=e.total_out=0,e.data_type=i,(t=e.state).pending=0,t.pending_out=0,t.wrap<0&&(t.wrap=-t.wrap),t.status=t.wrap?C:E,e.adler=2===t.wrap?0:1,t.last_flush=l,u._tr_init(t),m):R(e,_)}function K(e){var t=G(e);return t===m&&function(e){e.window_size=2*e.w_size,D(e.head),e.max_lazy_match=h[e.level].max_lazy,e.good_match=h[e.level].good_length,e.nice_match=h[e.level].nice_length,e.max_chain_length=h[e.level].max_chain,e.strstart=0,e.block_start=0,e.lookahead=0,e.insert=0,e.match_length=e.prev_length=x-1,e.match_available=0,e.ins_h=0}(e.state),t}function Y(e,t,r,n,i,s){if(!e)return _;var a=1;if(t===g&&(t=6),n<0?(a=0,n=-n):15<n&&(a=2,n-=16),i<1||y<i||r!==v||n<8||15<n||t<0||9<t||s<0||b<s)return R(e,_);8===n&&(n=9);var o=new H;return(e.state=o).strm=e,o.wrap=a,o.gzhead=null,o.w_bits=n,o.w_size=1<<o.w_bits,o.w_mask=o.w_size-1,o.hash_bits=i+7,o.hash_size=1<<o.hash_bits,o.hash_mask=o.hash_size-1,o.hash_shift=~~((o.hash_bits+x-1)/x),o.window=new c.Buf8(2*o.w_size),o.head=new c.Buf16(o.hash_size),o.prev=new c.Buf16(o.w_size),o.lit_bufsize=1<<i+6,o.pending_buf_size=4*o.lit_bufsize,o.pending_buf=new c.Buf8(o.pending_buf_size),o.d_buf=1*o.lit_bufsize,o.l_buf=3*o.lit_bufsize,o.level=t,o.strategy=s,o.method=r,K(e)}h=[new M(0,0,0,0,function(e,t){var r=65535;for(r>e.pending_buf_size-5&&(r=e.pending_buf_size-5);;){if(e.lookahead<=1){if(j(e),0===e.lookahead&&t===l)return A;if(0===e.lookahead)break}e.strstart+=e.lookahead,e.lookahead=0;var n=e.block_start+r;if((0===e.strstart||e.strstart>=n)&&(e.lookahead=e.strstart-n,e.strstart=n,N(e,!1),0===e.strm.avail_out))return A;if(e.strstart-e.block_start>=e.w_size-z&&(N(e,!1),0===e.strm.avail_out))return A}return e.insert=0,t===f?(N(e,!0),0===e.strm.avail_out?O:B):(e.strstart>e.block_start&&(N(e,!1),e.strm.avail_out),A)}),new M(4,4,8,4,Z),new M(4,5,16,8,Z),new M(4,6,32,32,Z),new M(4,4,16,16,W),new M(8,16,32,32,W),new M(8,16,128,128,W),new M(8,32,128,256,W),new M(32,128,258,1024,W),new M(32,258,258,4096,W)],r.deflateInit=function(e,t){return Y(e,t,v,15,8,0)},r.deflateInit2=Y,r.deflateReset=K,r.deflateResetKeep=G,r.deflateSetHeader=function(e,t){return e&&e.state?2!==e.state.wrap?_:(e.state.gzhead=t,m):_},r.deflate=function(e,t){var r,n,i,s;if(!e||!e.state||5<t||t<0)return e?R(e,_):_;if(n=e.state,!e.output||!e.input&&0!==e.avail_in||666===n.status&&t!==f)return R(e,0===e.avail_out?-5:_);if(n.strm=e,r=n.last_flush,n.last_flush=t,n.status===C)if(2===n.wrap)e.adler=0,U(n,31),U(n,139),U(n,8),n.gzhead?(U(n,(n.gzhead.text?1:0)+(n.gzhead.hcrc?2:0)+(n.gzhead.extra?4:0)+(n.gzhead.name?8:0)+(n.gzhead.comment?16:0)),U(n,255&n.gzhead.time),U(n,n.gzhead.time>>8&255),U(n,n.gzhead.time>>16&255),U(n,n.gzhead.time>>24&255),U(n,9===n.level?2:2<=n.strategy||n.level<2?4:0),U(n,255&n.gzhead.os),n.gzhead.extra&&n.gzhead.extra.length&&(U(n,255&n.gzhead.extra.length),U(n,n.gzhead.extra.length>>8&255)),n.gzhead.hcrc&&(e.adler=p(e.adler,n.pending_buf,n.pending,0)),n.gzindex=0,n.status=69):(U(n,0),U(n,0),U(n,0),U(n,0),U(n,0),U(n,9===n.level?2:2<=n.strategy||n.level<2?4:0),U(n,3),n.status=E);else{var a=v+(n.w_bits-8<<4)<<8;a|=(2<=n.strategy||n.level<2?0:n.level<6?1:6===n.level?2:3)<<6,0!==n.strstart&&(a|=32),a+=31-a%31,n.status=E,P(n,a),0!==n.strstart&&(P(n,e.adler>>>16),P(n,65535&e.adler)),e.adler=1}if(69===n.status)if(n.gzhead.extra){for(i=n.pending;n.gzindex<(65535&n.gzhead.extra.length)&&(n.pending!==n.pending_buf_size||(n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),F(e),i=n.pending,n.pending!==n.pending_buf_size));)U(n,255&n.gzhead.extra[n.gzindex]),n.gzindex++;n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),n.gzindex===n.gzhead.extra.length&&(n.gzindex=0,n.status=73)}else n.status=73;if(73===n.status)if(n.gzhead.name){i=n.pending;do{if(n.pending===n.pending_buf_size&&(n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),F(e),i=n.pending,n.pending===n.pending_buf_size)){s=1;break}s=n.gzindex<n.gzhead.name.length?255&n.gzhead.name.charCodeAt(n.gzindex++):0,U(n,s)}while(0!==s);n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),0===s&&(n.gzindex=0,n.status=91)}else n.status=91;if(91===n.status)if(n.gzhead.comment){i=n.pending;do{if(n.pending===n.pending_buf_size&&(n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),F(e),i=n.pending,n.pending===n.pending_buf_size)){s=1;break}s=n.gzindex<n.gzhead.comment.length?255&n.gzhead.comment.charCodeAt(n.gzindex++):0,U(n,s)}while(0!==s);n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),0===s&&(n.status=103)}else n.status=103;if(103===n.status&&(n.gzhead.hcrc?(n.pending+2>n.pending_buf_size&&F(e),n.pending+2<=n.pending_buf_size&&(U(n,255&e.adler),U(n,e.adler>>8&255),e.adler=0,n.status=E)):n.status=E),0!==n.pending){if(F(e),0===e.avail_out)return n.last_flush=-1,m}else if(0===e.avail_in&&T(t)<=T(r)&&t!==f)return R(e,-5);if(666===n.status&&0!==e.avail_in)return R(e,-5);if(0!==e.avail_in||0!==n.lookahead||t!==l&&666!==n.status){var o=2===n.strategy?function(e,t){for(var r;;){if(0===e.lookahead&&(j(e),0===e.lookahead)){if(t===l)return A;break}if(e.match_length=0,r=u._tr_tally(e,0,e.window[e.strstart]),e.lookahead--,e.strstart++,r&&(N(e,!1),0===e.strm.avail_out))return A}return e.insert=0,t===f?(N(e,!0),0===e.strm.avail_out?O:B):e.last_lit&&(N(e,!1),0===e.strm.avail_out)?A:I}(n,t):3===n.strategy?function(e,t){for(var r,n,i,s,a=e.window;;){if(e.lookahead<=S){if(j(e),e.lookahead<=S&&t===l)return A;if(0===e.lookahead)break}if(e.match_length=0,e.lookahead>=x&&0<e.strstart&&(n=a[i=e.strstart-1])===a[++i]&&n===a[++i]&&n===a[++i]){s=e.strstart+S;do{}while(n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&i<s);e.match_length=S-(s-i),e.match_length>e.lookahead&&(e.match_length=e.lookahead)}if(e.match_length>=x?(r=u._tr_tally(e,1,e.match_length-x),e.lookahead-=e.match_length,e.strstart+=e.match_length,e.match_length=0):(r=u._tr_tally(e,0,e.window[e.strstart]),e.lookahead--,e.strstart++),r&&(N(e,!1),0===e.strm.avail_out))return A}return e.insert=0,t===f?(N(e,!0),0===e.strm.avail_out?O:B):e.last_lit&&(N(e,!1),0===e.strm.avail_out)?A:I}(n,t):h[n.level].func(n,t);if(o!==O&&o!==B||(n.status=666),o===A||o===O)return 0===e.avail_out&&(n.last_flush=-1),m;if(o===I&&(1===t?u._tr_align(n):5!==t&&(u._tr_stored_block(n,0,0,!1),3===t&&(D(n.head),0===n.lookahead&&(n.strstart=0,n.block_start=0,n.insert=0))),F(e),0===e.avail_out))return n.last_flush=-1,m}return t!==f?m:n.wrap<=0?1:(2===n.wrap?(U(n,255&e.adler),U(n,e.adler>>8&255),U(n,e.adler>>16&255),U(n,e.adler>>24&255),U(n,255&e.total_in),U(n,e.total_in>>8&255),U(n,e.total_in>>16&255),U(n,e.total_in>>24&255)):(P(n,e.adler>>>16),P(n,65535&e.adler)),F(e),0<n.wrap&&(n.wrap=-n.wrap),0!==n.pending?m:1)},r.deflateEnd=function(e){var t;return e&&e.state?(t=e.state.status)!==C&&69!==t&&73!==t&&91!==t&&103!==t&&t!==E&&666!==t?R(e,_):(e.state=null,t===E?R(e,-3):m):_},r.deflateSetDictionary=function(e,t){var r,n,i,s,a,o,h,u,l=t.length;if(!e||!e.state)return _;if(2===(s=(r=e.state).wrap)||1===s&&r.status!==C||r.lookahead)return _;for(1===s&&(e.adler=d(e.adler,t,l,0)),r.wrap=0,l>=r.w_size&&(0===s&&(D(r.head),r.strstart=0,r.block_start=0,r.insert=0),u=new c.Buf8(r.w_size),c.arraySet(u,t,l-r.w_size,r.w_size,0),t=u,l=r.w_size),a=e.avail_in,o=e.next_in,h=e.input,e.avail_in=l,e.next_in=0,e.input=t,j(r);r.lookahead>=x;){for(n=r.strstart,i=r.lookahead-(x-1);r.ins_h=(r.ins_h<<r.hash_shift^r.window[n+x-1])&r.hash_mask,r.prev[n&r.w_mask]=r.head[r.ins_h],r.head[r.ins_h]=n,n++,--i;);r.strstart=n,r.lookahead=x-1,j(r)}return r.strstart+=r.lookahead,r.block_start=r.strstart,r.insert=r.lookahead,r.lookahead=0,r.match_length=r.prev_length=x-1,r.match_available=0,e.next_in=o,e.input=h,e.avail_in=a,r.wrap=s,m},r.deflateInfo="pako deflate (from Nodeca project)"},{"../utils/common":41,"./adler32":43,"./crc32":45,"./messages":51,"./trees":52}],47:[function(e,t,r){"use strict";t.exports=function(){this.text=0,this.time=0,this.xflags=0,this.os=0,this.extra=null,this.extra_len=0,this.name="",this.comment="",this.hcrc=0,this.done=!1}},{}],48:[function(e,t,r){"use strict";t.exports=function(e,t){var r,n,i,s,a,o,h,u,l,f,c,d,p,m,_,g,b,v,y,w,k,x,S,z,C;r=e.state,n=e.next_in,z=e.input,i=n+(e.avail_in-5),s=e.next_out,C=e.output,a=s-(t-e.avail_out),o=s+(e.avail_out-257),h=r.dmax,u=r.wsize,l=r.whave,f=r.wnext,c=r.window,d=r.hold,p=r.bits,m=r.lencode,_=r.distcode,g=(1<<r.lenbits)-1,b=(1<<r.distbits)-1;e:do{p<15&&(d+=z[n++]<<p,p+=8,d+=z[n++]<<p,p+=8),v=m[d&g];t:for(;;){if(d>>>=y=v>>>24,p-=y,0===(y=v>>>16&255))C[s++]=65535&v;else{if(!(16&y)){if(0==(64&y)){v=m[(65535&v)+(d&(1<<y)-1)];continue t}if(32&y){r.mode=12;break e}e.msg="invalid literal/length code",r.mode=30;break e}w=65535&v,(y&=15)&&(p<y&&(d+=z[n++]<<p,p+=8),w+=d&(1<<y)-1,d>>>=y,p-=y),p<15&&(d+=z[n++]<<p,p+=8,d+=z[n++]<<p,p+=8),v=_[d&b];r:for(;;){if(d>>>=y=v>>>24,p-=y,!(16&(y=v>>>16&255))){if(0==(64&y)){v=_[(65535&v)+(d&(1<<y)-1)];continue r}e.msg="invalid distance code",r.mode=30;break e}if(k=65535&v,p<(y&=15)&&(d+=z[n++]<<p,(p+=8)<y&&(d+=z[n++]<<p,p+=8)),h<(k+=d&(1<<y)-1)){e.msg="invalid distance too far back",r.mode=30;break e}if(d>>>=y,p-=y,(y=s-a)<k){if(l<(y=k-y)&&r.sane){e.msg="invalid distance too far back",r.mode=30;break e}if(S=c,(x=0)===f){if(x+=u-y,y<w){for(w-=y;C[s++]=c[x++],--y;);x=s-k,S=C}}else if(f<y){if(x+=u+f-y,(y-=f)<w){for(w-=y;C[s++]=c[x++],--y;);if(x=0,f<w){for(w-=y=f;C[s++]=c[x++],--y;);x=s-k,S=C}}}else if(x+=f-y,y<w){for(w-=y;C[s++]=c[x++],--y;);x=s-k,S=C}for(;2<w;)C[s++]=S[x++],C[s++]=S[x++],C[s++]=S[x++],w-=3;w&&(C[s++]=S[x++],1<w&&(C[s++]=S[x++]))}else{for(x=s-k;C[s++]=C[x++],C[s++]=C[x++],C[s++]=C[x++],2<(w-=3););w&&(C[s++]=C[x++],1<w&&(C[s++]=C[x++]))}break}}break}}while(n<i&&s<o);n-=w=p>>3,d&=(1<<(p-=w<<3))-1,e.next_in=n,e.next_out=s,e.avail_in=n<i?i-n+5:5-(n-i),e.avail_out=s<o?o-s+257:257-(s-o),r.hold=d,r.bits=p}},{}],49:[function(e,t,r){"use strict";var I=e("../utils/common"),O=e("./adler32"),B=e("./crc32"),R=e("./inffast"),T=e("./inftrees"),D=1,F=2,N=0,U=-2,P=1,n=852,i=592;function L(e){return(e>>>24&255)+(e>>>8&65280)+((65280&e)<<8)+((255&e)<<24)}function s(){this.mode=0,this.last=!1,this.wrap=0,this.havedict=!1,this.flags=0,this.dmax=0,this.check=0,this.total=0,this.head=null,this.wbits=0,this.wsize=0,this.whave=0,this.wnext=0,this.window=null,this.hold=0,this.bits=0,this.length=0,this.offset=0,this.extra=0,this.lencode=null,this.distcode=null,this.lenbits=0,this.distbits=0,this.ncode=0,this.nlen=0,this.ndist=0,this.have=0,this.next=null,this.lens=new I.Buf16(320),this.work=new I.Buf16(288),this.lendyn=null,this.distdyn=null,this.sane=0,this.back=0,this.was=0}function a(e){var t;return e&&e.state?(t=e.state,e.total_in=e.total_out=t.total=0,e.msg="",t.wrap&&(e.adler=1&t.wrap),t.mode=P,t.last=0,t.havedict=0,t.dmax=32768,t.head=null,t.hold=0,t.bits=0,t.lencode=t.lendyn=new I.Buf32(n),t.distcode=t.distdyn=new I.Buf32(i),t.sane=1,t.back=-1,N):U}function o(e){var t;return e&&e.state?((t=e.state).wsize=0,t.whave=0,t.wnext=0,a(e)):U}function h(e,t){var r,n;return e&&e.state?(n=e.state,t<0?(r=0,t=-t):(r=1+(t>>4),t<48&&(t&=15)),t&&(t<8||15<t)?U:(null!==n.window&&n.wbits!==t&&(n.window=null),n.wrap=r,n.wbits=t,o(e))):U}function u(e,t){var r,n;return e?(n=new s,(e.state=n).window=null,(r=h(e,t))!==N&&(e.state=null),r):U}var l,f,c=!0;function j(e){if(c){var t;for(l=new I.Buf32(512),f=new I.Buf32(32),t=0;t<144;)e.lens[t++]=8;for(;t<256;)e.lens[t++]=9;for(;t<280;)e.lens[t++]=7;for(;t<288;)e.lens[t++]=8;for(T(D,e.lens,0,288,l,0,e.work,{bits:9}),t=0;t<32;)e.lens[t++]=5;T(F,e.lens,0,32,f,0,e.work,{bits:5}),c=!1}e.lencode=l,e.lenbits=9,e.distcode=f,e.distbits=5}function Z(e,t,r,n){var i,s=e.state;return null===s.window&&(s.wsize=1<<s.wbits,s.wnext=0,s.whave=0,s.window=new I.Buf8(s.wsize)),n>=s.wsize?(I.arraySet(s.window,t,r-s.wsize,s.wsize,0),s.wnext=0,s.whave=s.wsize):(n<(i=s.wsize-s.wnext)&&(i=n),I.arraySet(s.window,t,r-n,i,s.wnext),(n-=i)?(I.arraySet(s.window,t,r-n,n,0),s.wnext=n,s.whave=s.wsize):(s.wnext+=i,s.wnext===s.wsize&&(s.wnext=0),s.whave<s.wsize&&(s.whave+=i))),0}r.inflateReset=o,r.inflateReset2=h,r.inflateResetKeep=a,r.inflateInit=function(e){return u(e,15)},r.inflateInit2=u,r.inflate=function(e,t){var r,n,i,s,a,o,h,u,l,f,c,d,p,m,_,g,b,v,y,w,k,x,S,z,C=0,E=new I.Buf8(4),A=[16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15];if(!e||!e.state||!e.output||!e.input&&0!==e.avail_in)return U;12===(r=e.state).mode&&(r.mode=13),a=e.next_out,i=e.output,h=e.avail_out,s=e.next_in,n=e.input,o=e.avail_in,u=r.hold,l=r.bits,f=o,c=h,x=N;e:for(;;)switch(r.mode){case P:if(0===r.wrap){r.mode=13;break}for(;l<16;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(2&r.wrap&&35615===u){E[r.check=0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0),l=u=0,r.mode=2;break}if(r.flags=0,r.head&&(r.head.done=!1),!(1&r.wrap)||(((255&u)<<8)+(u>>8))%31){e.msg="incorrect header check",r.mode=30;break}if(8!=(15&u)){e.msg="unknown compression method",r.mode=30;break}if(l-=4,k=8+(15&(u>>>=4)),0===r.wbits)r.wbits=k;else if(k>r.wbits){e.msg="invalid window size",r.mode=30;break}r.dmax=1<<k,e.adler=r.check=1,r.mode=512&u?10:12,l=u=0;break;case 2:for(;l<16;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(r.flags=u,8!=(255&r.flags)){e.msg="unknown compression method",r.mode=30;break}if(57344&r.flags){e.msg="unknown header flags set",r.mode=30;break}r.head&&(r.head.text=u>>8&1),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0)),l=u=0,r.mode=3;case 3:for(;l<32;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}r.head&&(r.head.time=u),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,E[2]=u>>>16&255,E[3]=u>>>24&255,r.check=B(r.check,E,4,0)),l=u=0,r.mode=4;case 4:for(;l<16;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}r.head&&(r.head.xflags=255&u,r.head.os=u>>8),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0)),l=u=0,r.mode=5;case 5:if(1024&r.flags){for(;l<16;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}r.length=u,r.head&&(r.head.extra_len=u),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0)),l=u=0}else r.head&&(r.head.extra=null);r.mode=6;case 6:if(1024&r.flags&&(o<(d=r.length)&&(d=o),d&&(r.head&&(k=r.head.extra_len-r.length,r.head.extra||(r.head.extra=new Array(r.head.extra_len)),I.arraySet(r.head.extra,n,s,d,k)),512&r.flags&&(r.check=B(r.check,n,d,s)),o-=d,s+=d,r.length-=d),r.length))break e;r.length=0,r.mode=7;case 7:if(2048&r.flags){if(0===o)break e;for(d=0;k=n[s+d++],r.head&&k&&r.length<65536&&(r.head.name+=String.fromCharCode(k)),k&&d<o;);if(512&r.flags&&(r.check=B(r.check,n,d,s)),o-=d,s+=d,k)break e}else r.head&&(r.head.name=null);r.length=0,r.mode=8;case 8:if(4096&r.flags){if(0===o)break e;for(d=0;k=n[s+d++],r.head&&k&&r.length<65536&&(r.head.comment+=String.fromCharCode(k)),k&&d<o;);if(512&r.flags&&(r.check=B(r.check,n,d,s)),o-=d,s+=d,k)break e}else r.head&&(r.head.comment=null);r.mode=9;case 9:if(512&r.flags){for(;l<16;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(u!==(65535&r.check)){e.msg="header crc mismatch",r.mode=30;break}l=u=0}r.head&&(r.head.hcrc=r.flags>>9&1,r.head.done=!0),e.adler=r.check=0,r.mode=12;break;case 10:for(;l<32;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}e.adler=r.check=L(u),l=u=0,r.mode=11;case 11:if(0===r.havedict)return e.next_out=a,e.avail_out=h,e.next_in=s,e.avail_in=o,r.hold=u,r.bits=l,2;e.adler=r.check=1,r.mode=12;case 12:if(5===t||6===t)break e;case 13:if(r.last){u>>>=7&l,l-=7&l,r.mode=27;break}for(;l<3;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}switch(r.last=1&u,l-=1,3&(u>>>=1)){case 0:r.mode=14;break;case 1:if(j(r),r.mode=20,6!==t)break;u>>>=2,l-=2;break e;case 2:r.mode=17;break;case 3:e.msg="invalid block type",r.mode=30}u>>>=2,l-=2;break;case 14:for(u>>>=7&l,l-=7&l;l<32;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if((65535&u)!=(u>>>16^65535)){e.msg="invalid stored block lengths",r.mode=30;break}if(r.length=65535&u,l=u=0,r.mode=15,6===t)break e;case 15:r.mode=16;case 16:if(d=r.length){if(o<d&&(d=o),h<d&&(d=h),0===d)break e;I.arraySet(i,n,s,d,a),o-=d,s+=d,h-=d,a+=d,r.length-=d;break}r.mode=12;break;case 17:for(;l<14;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(r.nlen=257+(31&u),u>>>=5,l-=5,r.ndist=1+(31&u),u>>>=5,l-=5,r.ncode=4+(15&u),u>>>=4,l-=4,286<r.nlen||30<r.ndist){e.msg="too many length or distance symbols",r.mode=30;break}r.have=0,r.mode=18;case 18:for(;r.have<r.ncode;){for(;l<3;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}r.lens[A[r.have++]]=7&u,u>>>=3,l-=3}for(;r.have<19;)r.lens[A[r.have++]]=0;if(r.lencode=r.lendyn,r.lenbits=7,S={bits:r.lenbits},x=T(0,r.lens,0,19,r.lencode,0,r.work,S),r.lenbits=S.bits,x){e.msg="invalid code lengths set",r.mode=30;break}r.have=0,r.mode=19;case 19:for(;r.have<r.nlen+r.ndist;){for(;g=(C=r.lencode[u&(1<<r.lenbits)-1])>>>16&255,b=65535&C,!((_=C>>>24)<=l);){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(b<16)u>>>=_,l-=_,r.lens[r.have++]=b;else{if(16===b){for(z=_+2;l<z;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(u>>>=_,l-=_,0===r.have){e.msg="invalid bit length repeat",r.mode=30;break}k=r.lens[r.have-1],d=3+(3&u),u>>>=2,l-=2}else if(17===b){for(z=_+3;l<z;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}l-=_,k=0,d=3+(7&(u>>>=_)),u>>>=3,l-=3}else{for(z=_+7;l<z;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}l-=_,k=0,d=11+(127&(u>>>=_)),u>>>=7,l-=7}if(r.have+d>r.nlen+r.ndist){e.msg="invalid bit length repeat",r.mode=30;break}for(;d--;)r.lens[r.have++]=k}}if(30===r.mode)break;if(0===r.lens[256]){e.msg="invalid code -- missing end-of-block",r.mode=30;break}if(r.lenbits=9,S={bits:r.lenbits},x=T(D,r.lens,0,r.nlen,r.lencode,0,r.work,S),r.lenbits=S.bits,x){e.msg="invalid literal/lengths set",r.mode=30;break}if(r.distbits=6,r.distcode=r.distdyn,S={bits:r.distbits},x=T(F,r.lens,r.nlen,r.ndist,r.distcode,0,r.work,S),r.distbits=S.bits,x){e.msg="invalid distances set",r.mode=30;break}if(r.mode=20,6===t)break e;case 20:r.mode=21;case 21:if(6<=o&&258<=h){e.next_out=a,e.avail_out=h,e.next_in=s,e.avail_in=o,r.hold=u,r.bits=l,R(e,c),a=e.next_out,i=e.output,h=e.avail_out,s=e.next_in,n=e.input,o=e.avail_in,u=r.hold,l=r.bits,12===r.mode&&(r.back=-1);break}for(r.back=0;g=(C=r.lencode[u&(1<<r.lenbits)-1])>>>16&255,b=65535&C,!((_=C>>>24)<=l);){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(g&&0==(240&g)){for(v=_,y=g,w=b;g=(C=r.lencode[w+((u&(1<<v+y)-1)>>v)])>>>16&255,b=65535&C,!(v+(_=C>>>24)<=l);){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}u>>>=v,l-=v,r.back+=v}if(u>>>=_,l-=_,r.back+=_,r.length=b,0===g){r.mode=26;break}if(32&g){r.back=-1,r.mode=12;break}if(64&g){e.msg="invalid literal/length code",r.mode=30;break}r.extra=15&g,r.mode=22;case 22:if(r.extra){for(z=r.extra;l<z;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}r.length+=u&(1<<r.extra)-1,u>>>=r.extra,l-=r.extra,r.back+=r.extra}r.was=r.length,r.mode=23;case 23:for(;g=(C=r.distcode[u&(1<<r.distbits)-1])>>>16&255,b=65535&C,!((_=C>>>24)<=l);){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(0==(240&g)){for(v=_,y=g,w=b;g=(C=r.distcode[w+((u&(1<<v+y)-1)>>v)])>>>16&255,b=65535&C,!(v+(_=C>>>24)<=l);){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}u>>>=v,l-=v,r.back+=v}if(u>>>=_,l-=_,r.back+=_,64&g){e.msg="invalid distance code",r.mode=30;break}r.offset=b,r.extra=15&g,r.mode=24;case 24:if(r.extra){for(z=r.extra;l<z;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}r.offset+=u&(1<<r.extra)-1,u>>>=r.extra,l-=r.extra,r.back+=r.extra}if(r.offset>r.dmax){e.msg="invalid distance too far back",r.mode=30;break}r.mode=25;case 25:if(0===h)break e;if(d=c-h,r.offset>d){if((d=r.offset-d)>r.whave&&r.sane){e.msg="invalid distance too far back",r.mode=30;break}p=d>r.wnext?(d-=r.wnext,r.wsize-d):r.wnext-d,d>r.length&&(d=r.length),m=r.window}else m=i,p=a-r.offset,d=r.length;for(h<d&&(d=h),h-=d,r.length-=d;i[a++]=m[p++],--d;);0===r.length&&(r.mode=21);break;case 26:if(0===h)break e;i[a++]=r.length,h--,r.mode=21;break;case 27:if(r.wrap){for(;l<32;){if(0===o)break e;o--,u|=n[s++]<<l,l+=8}if(c-=h,e.total_out+=c,r.total+=c,c&&(e.adler=r.check=r.flags?B(r.check,i,c,a-c):O(r.check,i,c,a-c)),c=h,(r.flags?u:L(u))!==r.check){e.msg="incorrect data check",r.mode=30;break}l=u=0}r.mode=28;case 28:if(r.wrap&&r.flags){for(;l<32;){if(0===o)break e;o--,u+=n[s++]<<l,l+=8}if(u!==(4294967295&r.total)){e.msg="incorrect length check",r.mode=30;break}l=u=0}r.mode=29;case 29:x=1;break e;case 30:x=-3;break e;case 31:return-4;case 32:default:return U}return e.next_out=a,e.avail_out=h,e.next_in=s,e.avail_in=o,r.hold=u,r.bits=l,(r.wsize||c!==e.avail_out&&r.mode<30&&(r.mode<27||4!==t))&&Z(e,e.output,e.next_out,c-e.avail_out)?(r.mode=31,-4):(f-=e.avail_in,c-=e.avail_out,e.total_in+=f,e.total_out+=c,r.total+=c,r.wrap&&c&&(e.adler=r.check=r.flags?B(r.check,i,c,e.next_out-c):O(r.check,i,c,e.next_out-c)),e.data_type=r.bits+(r.last?64:0)+(12===r.mode?128:0)+(20===r.mode||15===r.mode?256:0),(0==f&&0===c||4===t)&&x===N&&(x=-5),x)},r.inflateEnd=function(e){if(!e||!e.state)return U;var t=e.state;return t.window&&(t.window=null),e.state=null,N},r.inflateGetHeader=function(e,t){var r;return e&&e.state?0==(2&(r=e.state).wrap)?U:((r.head=t).done=!1,N):U},r.inflateSetDictionary=function(e,t){var r,n=t.length;return e&&e.state?0!==(r=e.state).wrap&&11!==r.mode?U:11===r.mode&&O(1,t,n,0)!==r.check?-3:Z(e,t,n,n)?(r.mode=31,-4):(r.havedict=1,N):U},r.inflateInfo="pako inflate (from Nodeca project)"},{"../utils/common":41,"./adler32":43,"./crc32":45,"./inffast":48,"./inftrees":50}],50:[function(e,t,r){"use strict";var D=e("../utils/common"),F=[3,4,5,6,7,8,9,10,11,13,15,17,19,23,27,31,35,43,51,59,67,83,99,115,131,163,195,227,258,0,0],N=[16,16,16,16,16,16,16,16,17,17,17,17,18,18,18,18,19,19,19,19,20,20,20,20,21,21,21,21,16,72,78],U=[1,2,3,4,5,7,9,13,17,25,33,49,65,97,129,193,257,385,513,769,1025,1537,2049,3073,4097,6145,8193,12289,16385,24577,0,0],P=[16,16,16,16,17,17,18,18,19,19,20,20,21,21,22,22,23,23,24,24,25,25,26,26,27,27,28,28,29,29,64,64];t.exports=function(e,t,r,n,i,s,a,o){var h,u,l,f,c,d,p,m,_,g=o.bits,b=0,v=0,y=0,w=0,k=0,x=0,S=0,z=0,C=0,E=0,A=null,I=0,O=new D.Buf16(16),B=new D.Buf16(16),R=null,T=0;for(b=0;b<=15;b++)O[b]=0;for(v=0;v<n;v++)O[t[r+v]]++;for(k=g,w=15;1<=w&&0===O[w];w--);if(w<k&&(k=w),0===w)return i[s++]=20971520,i[s++]=20971520,o.bits=1,0;for(y=1;y<w&&0===O[y];y++);for(k<y&&(k=y),b=z=1;b<=15;b++)if(z<<=1,(z-=O[b])<0)return-1;if(0<z&&(0===e||1!==w))return-1;for(B[1]=0,b=1;b<15;b++)B[b+1]=B[b]+O[b];for(v=0;v<n;v++)0!==t[r+v]&&(a[B[t[r+v]]++]=v);if(d=0===e?(A=R=a,19):1===e?(A=F,I-=257,R=N,T-=257,256):(A=U,R=P,-1),b=y,c=s,S=v=E=0,l=-1,f=(C=1<<(x=k))-1,1===e&&852<C||2===e&&592<C)return 1;for(;;){for(p=b-S,_=a[v]<d?(m=0,a[v]):a[v]>d?(m=R[T+a[v]],A[I+a[v]]):(m=96,0),h=1<<b-S,y=u=1<<x;i[c+(E>>S)+(u-=h)]=p<<24|m<<16|_|0,0!==u;);for(h=1<<b-1;E&h;)h>>=1;if(0!==h?(E&=h-1,E+=h):E=0,v++,0==--O[b]){if(b===w)break;b=t[r+a[v]]}if(k<b&&(E&f)!==l){for(0===S&&(S=k),c+=y,z=1<<(x=b-S);x+S<w&&!((z-=O[x+S])<=0);)x++,z<<=1;if(C+=1<<x,1===e&&852<C||2===e&&592<C)return 1;i[l=E&f]=k<<24|x<<16|c-s|0}}return 0!==E&&(i[c+E]=b-S<<24|64<<16|0),o.bits=k,0}},{"../utils/common":41}],51:[function(e,t,r){"use strict";t.exports={2:"need dictionary",1:"stream end",0:"","-1":"file error","-2":"stream error","-3":"data error","-4":"insufficient memory","-5":"buffer error","-6":"incompatible version"}},{}],52:[function(e,t,r){"use strict";var i=e("../utils/common"),o=0,h=1;function n(e){for(var t=e.length;0<=--t;)e[t]=0}var s=0,a=29,u=256,l=u+1+a,f=30,c=19,_=2*l+1,g=15,d=16,p=7,m=256,b=16,v=17,y=18,w=[0,0,0,0,0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4,5,5,5,5,0],k=[0,0,0,0,1,1,2,2,3,3,4,4,5,5,6,6,7,7,8,8,9,9,10,10,11,11,12,12,13,13],x=[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,7],S=[16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15],z=new Array(2*(l+2));n(z);var C=new Array(2*f);n(C);var E=new Array(512);n(E);var A=new Array(256);n(A);var I=new Array(a);n(I);var O,B,R,T=new Array(f);function D(e,t,r,n,i){this.static_tree=e,this.extra_bits=t,this.extra_base=r,this.elems=n,this.max_length=i,this.has_stree=e&&e.length}function F(e,t){this.dyn_tree=e,this.max_code=0,this.stat_desc=t}function N(e){return e<256?E[e]:E[256+(e>>>7)]}function U(e,t){e.pending_buf[e.pending++]=255&t,e.pending_buf[e.pending++]=t>>>8&255}function P(e,t,r){e.bi_valid>d-r?(e.bi_buf|=t<<e.bi_valid&65535,U(e,e.bi_buf),e.bi_buf=t>>d-e.bi_valid,e.bi_valid+=r-d):(e.bi_buf|=t<<e.bi_valid&65535,e.bi_valid+=r)}function L(e,t,r){P(e,r[2*t],r[2*t+1])}function j(e,t){for(var r=0;r|=1&e,e>>>=1,r<<=1,0<--t;);return r>>>1}function Z(e,t,r){var n,i,s=new Array(g+1),a=0;for(n=1;n<=g;n++)s[n]=a=a+r[n-1]<<1;for(i=0;i<=t;i++){var o=e[2*i+1];0!==o&&(e[2*i]=j(s[o]++,o))}}function W(e){var t;for(t=0;t<l;t++)e.dyn_ltree[2*t]=0;for(t=0;t<f;t++)e.dyn_dtree[2*t]=0;for(t=0;t<c;t++)e.bl_tree[2*t]=0;e.dyn_ltree[2*m]=1,e.opt_len=e.static_len=0,e.last_lit=e.matches=0}function M(e){8<e.bi_valid?U(e,e.bi_buf):0<e.bi_valid&&(e.pending_buf[e.pending++]=e.bi_buf),e.bi_buf=0,e.bi_valid=0}function H(e,t,r,n){var i=2*t,s=2*r;return e[i]<e[s]||e[i]===e[s]&&n[t]<=n[r]}function G(e,t,r){for(var n=e.heap[r],i=r<<1;i<=e.heap_len&&(i<e.heap_len&&H(t,e.heap[i+1],e.heap[i],e.depth)&&i++,!H(t,n,e.heap[i],e.depth));)e.heap[r]=e.heap[i],r=i,i<<=1;e.heap[r]=n}function K(e,t,r){var n,i,s,a,o=0;if(0!==e.last_lit)for(;n=e.pending_buf[e.d_buf+2*o]<<8|e.pending_buf[e.d_buf+2*o+1],i=e.pending_buf[e.l_buf+o],o++,0===n?L(e,i,t):(L(e,(s=A[i])+u+1,t),0!==(a=w[s])&&P(e,i-=I[s],a),L(e,s=N(--n),r),0!==(a=k[s])&&P(e,n-=T[s],a)),o<e.last_lit;);L(e,m,t)}function Y(e,t){var r,n,i,s=t.dyn_tree,a=t.stat_desc.static_tree,o=t.stat_desc.has_stree,h=t.stat_desc.elems,u=-1;for(e.heap_len=0,e.heap_max=_,r=0;r<h;r++)0!==s[2*r]?(e.heap[++e.heap_len]=u=r,e.depth[r]=0):s[2*r+1]=0;for(;e.heap_len<2;)s[2*(i=e.heap[++e.heap_len]=u<2?++u:0)]=1,e.depth[i]=0,e.opt_len--,o&&(e.static_len-=a[2*i+1]);for(t.max_code=u,r=e.heap_len>>1;1<=r;r--)G(e,s,r);for(i=h;r=e.heap[1],e.heap[1]=e.heap[e.heap_len--],G(e,s,1),n=e.heap[1],e.heap[--e.heap_max]=r,e.heap[--e.heap_max]=n,s[2*i]=s[2*r]+s[2*n],e.depth[i]=(e.depth[r]>=e.depth[n]?e.depth[r]:e.depth[n])+1,s[2*r+1]=s[2*n+1]=i,e.heap[1]=i++,G(e,s,1),2<=e.heap_len;);e.heap[--e.heap_max]=e.heap[1],function(e,t){var r,n,i,s,a,o,h=t.dyn_tree,u=t.max_code,l=t.stat_desc.static_tree,f=t.stat_desc.has_stree,c=t.stat_desc.extra_bits,d=t.stat_desc.extra_base,p=t.stat_desc.max_length,m=0;for(s=0;s<=g;s++)e.bl_count[s]=0;for(h[2*e.heap[e.heap_max]+1]=0,r=e.heap_max+1;r<_;r++)p<(s=h[2*h[2*(n=e.heap[r])+1]+1]+1)&&(s=p,m++),h[2*n+1]=s,u<n||(e.bl_count[s]++,a=0,d<=n&&(a=c[n-d]),o=h[2*n],e.opt_len+=o*(s+a),f&&(e.static_len+=o*(l[2*n+1]+a)));if(0!==m){do{for(s=p-1;0===e.bl_count[s];)s--;e.bl_count[s]--,e.bl_count[s+1]+=2,e.bl_count[p]--,m-=2}while(0<m);for(s=p;0!==s;s--)for(n=e.bl_count[s];0!==n;)u<(i=e.heap[--r])||(h[2*i+1]!==s&&(e.opt_len+=(s-h[2*i+1])*h[2*i],h[2*i+1]=s),n--)}}(e,t),Z(s,u,e.bl_count)}function X(e,t,r){var n,i,s=-1,a=t[1],o=0,h=7,u=4;for(0===a&&(h=138,u=3),t[2*(r+1)+1]=65535,n=0;n<=r;n++)i=a,a=t[2*(n+1)+1],++o<h&&i===a||(o<u?e.bl_tree[2*i]+=o:0!==i?(i!==s&&e.bl_tree[2*i]++,e.bl_tree[2*b]++):o<=10?e.bl_tree[2*v]++:e.bl_tree[2*y]++,s=i,u=(o=0)===a?(h=138,3):i===a?(h=6,3):(h=7,4))}function V(e,t,r){var n,i,s=-1,a=t[1],o=0,h=7,u=4;for(0===a&&(h=138,u=3),n=0;n<=r;n++)if(i=a,a=t[2*(n+1)+1],!(++o<h&&i===a)){if(o<u)for(;L(e,i,e.bl_tree),0!=--o;);else 0!==i?(i!==s&&(L(e,i,e.bl_tree),o--),L(e,b,e.bl_tree),P(e,o-3,2)):o<=10?(L(e,v,e.bl_tree),P(e,o-3,3)):(L(e,y,e.bl_tree),P(e,o-11,7));s=i,u=(o=0)===a?(h=138,3):i===a?(h=6,3):(h=7,4)}}n(T);var q=!1;function J(e,t,r,n){P(e,(s<<1)+(n?1:0),3),function(e,t,r,n){M(e),n&&(U(e,r),U(e,~r)),i.arraySet(e.pending_buf,e.window,t,r,e.pending),e.pending+=r}(e,t,r,!0)}r._tr_init=function(e){q||(function(){var e,t,r,n,i,s=new Array(g+1);for(n=r=0;n<a-1;n++)for(I[n]=r,e=0;e<1<<w[n];e++)A[r++]=n;for(A[r-1]=n,n=i=0;n<16;n++)for(T[n]=i,e=0;e<1<<k[n];e++)E[i++]=n;for(i>>=7;n<f;n++)for(T[n]=i<<7,e=0;e<1<<k[n]-7;e++)E[256+i++]=n;for(t=0;t<=g;t++)s[t]=0;for(e=0;e<=143;)z[2*e+1]=8,e++,s[8]++;for(;e<=255;)z[2*e+1]=9,e++,s[9]++;for(;e<=279;)z[2*e+1]=7,e++,s[7]++;for(;e<=287;)z[2*e+1]=8,e++,s[8]++;for(Z(z,l+1,s),e=0;e<f;e++)C[2*e+1]=5,C[2*e]=j(e,5);O=new D(z,w,u+1,l,g),B=new D(C,k,0,f,g),R=new D(new Array(0),x,0,c,p)}(),q=!0),e.l_desc=new F(e.dyn_ltree,O),e.d_desc=new F(e.dyn_dtree,B),e.bl_desc=new F(e.bl_tree,R),e.bi_buf=0,e.bi_valid=0,W(e)},r._tr_stored_block=J,r._tr_flush_block=function(e,t,r,n){var i,s,a=0;0<e.level?(2===e.strm.data_type&&(e.strm.data_type=function(e){var t,r=4093624447;for(t=0;t<=31;t++,r>>>=1)if(1&r&&0!==e.dyn_ltree[2*t])return o;if(0!==e.dyn_ltree[18]||0!==e.dyn_ltree[20]||0!==e.dyn_ltree[26])return h;for(t=32;t<u;t++)if(0!==e.dyn_ltree[2*t])return h;return o}(e)),Y(e,e.l_desc),Y(e,e.d_desc),a=function(e){var t;for(X(e,e.dyn_ltree,e.l_desc.max_code),X(e,e.dyn_dtree,e.d_desc.max_code),Y(e,e.bl_desc),t=c-1;3<=t&&0===e.bl_tree[2*S[t]+1];t--);return e.opt_len+=3*(t+1)+5+5+4,t}(e),i=e.opt_len+3+7>>>3,(s=e.static_len+3+7>>>3)<=i&&(i=s)):i=s=r+5,r+4<=i&&-1!==t?J(e,t,r,n):4===e.strategy||s===i?(P(e,2+(n?1:0),3),K(e,z,C)):(P(e,4+(n?1:0),3),function(e,t,r,n){var i;for(P(e,t-257,5),P(e,r-1,5),P(e,n-4,4),i=0;i<n;i++)P(e,e.bl_tree[2*S[i]+1],3);V(e,e.dyn_ltree,t-1),V(e,e.dyn_dtree,r-1)}(e,e.l_desc.max_code+1,e.d_desc.max_code+1,a+1),K(e,e.dyn_ltree,e.dyn_dtree)),W(e),n&&M(e)},r._tr_tally=function(e,t,r){return e.pending_buf[e.d_buf+2*e.last_lit]=t>>>8&255,e.pending_buf[e.d_buf+2*e.last_lit+1]=255&t,e.pending_buf[e.l_buf+e.last_lit]=255&r,e.last_lit++,0===t?e.dyn_ltree[2*r]++:(e.matches++,t--,e.dyn_ltree[2*(A[r]+u+1)]++,e.dyn_dtree[2*N(t)]++),e.last_lit===e.lit_bufsize-1},r._tr_align=function(e){P(e,2,3),L(e,m,z),function(e){16===e.bi_valid?(U(e,e.bi_buf),e.bi_buf=0,e.bi_valid=0):8<=e.bi_valid&&(e.pending_buf[e.pending++]=255&e.bi_buf,e.bi_buf>>=8,e.bi_valid-=8)}(e)}},{"../utils/common":41}],53:[function(e,t,r){"use strict";t.exports=function(){this.input=null,this.next_in=0,this.avail_in=0,this.total_in=0,this.output=null,this.next_out=0,this.avail_out=0,this.total_out=0,this.msg="",this.state=null,this.data_type=2,this.adler=0}},{}],54:[function(e,t,r){(function(e){!function(r,n){"use strict";if(!r.setImmediate){var i,s,t,a,o=1,h={},u=!1,l=r.document,e=Object.getPrototypeOf&&Object.getPrototypeOf(r);e=e&&e.setTimeout?e:r,i="[object process]"==={}.toString.call(r.process)?function(e){process.nextTick(function(){c(e)})}:function(){if(r.postMessage&&!r.importScripts){var e=!0,t=r.onmessage;return r.onmessage=function(){e=!1},r.postMessage("","*"),r.onmessage=t,e}}()?(a="setImmediate$"+Math.random()+"$",r.addEventListener?r.addEventListener("message",d,!1):r.attachEvent("onmessage",d),function(e){r.postMessage(a+e,"*")}):r.MessageChannel?((t=new MessageChannel).port1.onmessage=function(e){c(e.data)},function(e){t.port2.postMessage(e)}):l&&"onreadystatechange"in l.createElement("script")?(s=l.documentElement,function(e){var t=l.createElement("script");t.onreadystatechange=function(){c(e),t.onreadystatechange=null,s.removeChild(t),t=null},s.appendChild(t)}):function(e){setTimeout(c,0,e)},e.setImmediate=function(e){"function"!=typeof e&&(e=new Function(""+e));for(var t=new Array(arguments.length-1),r=0;r<t.length;r++)t[r]=arguments[r+1];var n={callback:e,args:t};return h[o]=n,i(o),o++},e.clearImmediate=f}function f(e){delete h[e]}function c(e){if(u)setTimeout(c,0,e);else{var t=h[e];if(t){u=!0;try{!function(e){var t=e.callback,r=e.args;switch(r.length){case 0:t();break;case 1:t(r[0]);break;case 2:t(r[0],r[1]);break;case 3:t(r[0],r[1],r[2]);break;default:t.apply(n,r)}}(t)}finally{f(e),u=!1}}}}function d(e){e.source===r&&"string"==typeof e.data&&0===e.data.indexOf(a)&&c(+e.data.slice(a.length))}}("undefined"==typeof self?void 0===e?this:e:self)}).call(this,"undefined"!=typeof global?global:"undefined"!=typeof self?self:"undefined"!=typeof window?window:{})},{}]},{},[10])(10)});
/*! pdfmake v0.2.20, @license MIT, @link http://pdfmake.org */
!function webpackUniversalModuleDefinition(r,o){if("object"==typeof exports&&"object"==typeof module)module.exports=o();else if("function"==typeof define&&define.amd)define([],o);else{var u=o();for(var p in u)("object"==typeof exports?exports:r)[p]=u[p]}}("undefined"!=typeof self?self:this,(function(){return function(){var r={464:function(r,o,u){var p=u(63432),y=u(55480);(r.exports=function(r,o){return y[r]||(y[r]=void 0!==o?o:{})})("versions",[]).push({version:"3.19.0",mode:p?"pure":"global",copyright:"© 2021 Denis Pushkarev (zloirock.ru)"})},520:function(r,o,u){var p=u(38486);r.exports=p("document","documentElement")},664:function(r,o,u){var p=u(40715);r.exports=/web0s(?!.*chrome)/i.test(p)},1083:function(r,o,u){u(71768)},1239:function(r,o,u){"use strict";var p;r.exports=(p=u(48352),function(){var r=p,o=r.lib,u=o.WordArray,y=o.Hasher,_=r.algo,P=u.create([0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,7,4,13,1,10,6,15,3,12,0,9,5,2,14,11,8,3,10,14,4,9,15,8,1,2,7,0,6,13,11,5,12,1,9,11,10,0,8,12,4,13,3,7,15,14,5,6,2,4,0,5,9,7,12,2,10,14,1,3,8,11,6,15,13]),k=u.create([5,14,7,0,9,2,11,4,13,6,15,8,1,10,3,12,6,11,3,7,0,13,5,10,14,15,8,12,4,9,1,2,15,5,1,3,7,14,6,9,11,8,12,2,10,0,4,13,8,6,4,1,3,11,15,0,5,12,2,13,9,7,10,14,12,15,10,4,1,5,8,7,6,2,13,14,0,3,9,11]),I=u.create([11,14,15,12,5,8,7,9,11,13,14,15,6,7,9,8,7,6,8,13,11,9,7,15,7,12,15,9,11,7,13,12,11,13,6,7,14,9,13,15,14,8,13,6,5,12,7,5,11,12,14,15,14,15,9,8,9,14,5,6,8,6,5,12,9,15,5,11,6,8,13,12,5,12,13,14,11,8,5,6]),O=u.create([8,9,9,11,13,15,15,5,7,7,8,11,14,14,12,6,9,13,15,7,12,8,9,11,7,7,12,7,6,15,13,11,9,7,15,11,8,6,6,14,12,13,5,14,13,13,7,5,15,5,8,11,14,14,6,14,6,9,12,9,12,5,15,8,8,5,12,9,12,5,14,6,8,13,6,5,15,13,11,11]),B=u.create([0,1518500249,1859775393,2400959708,2840853838]),D=u.create([1352829926,1548603684,1836072691,2053994217,0]),R=_.RIPEMD160=y.extend({_doReset:function _doReset(){this._hash=u.create([1732584193,4023233417,2562383102,271733878,3285377520])},_doProcessBlock:function _doProcessBlock(r,o){for(var u=0;u<16;u++){var p=o+u,y=r[p];r[p]=16711935&(y<<8|y>>>24)|4278255360&(y<<24|y>>>8)}var _,R,N,U,W,j,G,X,Y,K,J,$=this._hash.words,ee=B.words,te=D.words,re=P.words,ne=k.words,ie=I.words,oe=O.words;for(j=_=$[0],G=R=$[1],X=N=$[2],Y=U=$[3],K=W=$[4],u=0;u<80;u+=1)J=_+r[o+re[u]]|0,J+=u<16?f1(R,N,U)+ee[0]:u<32?f2(R,N,U)+ee[1]:u<48?f3(R,N,U)+ee[2]:u<64?f4(R,N,U)+ee[3]:f5(R,N,U)+ee[4],J=(J=rotl(J|=0,ie[u]))+W|0,_=W,W=U,U=rotl(N,10),N=R,R=J,J=j+r[o+ne[u]]|0,J+=u<16?f5(G,X,Y)+te[0]:u<32?f4(G,X,Y)+te[1]:u<48?f3(G,X,Y)+te[2]:u<64?f2(G,X,Y)+te[3]:f1(G,X,Y)+te[4],J=(J=rotl(J|=0,oe[u]))+K|0,j=K,K=Y,Y=rotl(X,10),X=G,G=J;J=$[1]+N+Y|0,$[1]=$[2]+U+K|0,$[2]=$[3]+W+j|0,$[3]=$[4]+_+G|0,$[4]=$[0]+R+X|0,$[0]=J},_doFinalize:function _doFinalize(){var r=this._data,o=r.words,u=8*this._nDataBytes,p=8*r.sigBytes;o[p>>>5]|=128<<24-p%32,o[14+(p+64>>>9<<4)]=16711935&(u<<8|u>>>24)|4278255360&(u<<24|u>>>8),r.sigBytes=4*(o.length+1),this._process();for(var y=this._hash,_=y.words,P=0;P<5;P++){var k=_[P];_[P]=16711935&(k<<8|k>>>24)|4278255360&(k<<24|k>>>8)}return y},clone:function clone(){var clone=y.clone.call(this);return clone._hash=this._hash.clone(),clone}});function f1(r,o,u){return r^o^u}function f2(r,o,u){return r&o|~r&u}function f3(r,o,u){return(r|~o)^u}function f4(r,o,u){return r&u|o&~u}function f5(r,o,u){return r^(o|~u)}function rotl(r,o){return r<<o|r>>>32-o}r.RIPEMD160=y._createHelper(R),r.HmacRIPEMD160=y._createHmacHelper(R)}(Math),p.RIPEMD160)},1593:function(r,o,u){"use strict";var p=u(56475),y=u(32010),_=u(2834),P=u(38347),k=u(83943),I=u(94578),O=u(28831),B=u(25096),D=u(51839),R=u(21182),N=u(29519),U=u(38688),W=u(63432),j=U("replace"),G=RegExp.prototype,X=y.TypeError,Y=P(R),K=P("".indexOf),J=P("".replace),$=P("".slice),ee=Math.max,stringIndexOf=function(r,o,u){return u>r.length?-1:""===o?u:K(r,o,u)};p({target:"String",proto:!0},{replaceAll:function replaceAll(r,o){var u,p,y,P,R,U,te,re,ne,ie=k(this),oe=0,ae=0,se="";if(null!=r){if((u=O(r))&&(p=B(k("flags"in G?r.flags:Y(r))),!~K(p,"g")))throw X("`.replaceAll` does not allow non-global regexes");if(y=D(r,j))return _(y,r,ie,o);if(W&&u)return J(B(ie),r,o)}for(P=B(ie),R=B(r),(U=I(o))||(o=B(o)),te=R.length,re=ee(1,te),oe=stringIndexOf(P,R,0);-1!==oe;)ne=U?B(o(R,oe,P)):N(R,P,oe,[],void 0,o),se+=$(P,ae,oe)+ne,ae=oe+te,oe=stringIndexOf(P,R,oe+re);return ae<P.length&&(se+=$(P,ae)),se}})},2256:function(r,o,u){"use strict";var p;u(94845),r.exports=(p=u(48352),function(){var r=p,o=r.lib.WordArray,u=r.enc;function swapEndian(r){return r<<8&4278255360|r>>>8&16711935}u.Utf16=u.Utf16BE={stringify:function stringify(r){for(var o=r.words,u=r.sigBytes,p=[],y=0;y<u;y+=2){var _=o[y>>>2]>>>16-y%4*8&65535;p.push(String.fromCharCode(_))}return p.join("")},parse:function parse(r){for(var u=r.length,p=[],y=0;y<u;y++)p[y>>>1]|=r.charCodeAt(y)<<16-y%2*16;return o.create(p,2*u)}},u.Utf16LE={stringify:function stringify(r){for(var o=r.words,u=r.sigBytes,p=[],y=0;y<u;y+=2){var _=swapEndian(o[y>>>2]>>>16-y%4*8&65535);p.push(String.fromCharCode(_))}return p.join("")},parse:function parse(r){for(var u=r.length,p=[],y=0;y<u;y++)p[y>>>1]|=swapEndian(r.charCodeAt(y)<<16-y%2*16);return o.create(p,2*u)}}}(),p.enc.Utf16)},2269:function(r,o,u){"use strict";var p=u(72519),y=u(46911),_=u(99049),P=u(96395),k=u(92920),I=-2,O=12,B=30;function zswap32(r){return(r>>>24&255)+(r>>>8&65280)+((65280&r)<<8)+((255&r)<<24)}function InflateState(){this.mode=0,this.last=!1,this.wrap=0,this.havedict=!1,this.flags=0,this.dmax=0,this.check=0,this.total=0,this.head=null,this.wbits=0,this.wsize=0,this.whave=0,this.wnext=0,this.window=null,this.hold=0,this.bits=0,this.length=0,this.offset=0,this.extra=0,this.lencode=null,this.distcode=null,this.lenbits=0,this.distbits=0,this.ncode=0,this.nlen=0,this.ndist=0,this.have=0,this.next=null,this.lens=new p.Buf16(320),this.work=new p.Buf16(288),this.lendyn=null,this.distdyn=null,this.sane=0,this.back=0,this.was=0}function inflateResetKeep(r){var o;return r&&r.state?(o=r.state,r.total_in=r.total_out=o.total=0,r.msg="",o.wrap&&(r.adler=1&o.wrap),o.mode=1,o.last=0,o.havedict=0,o.dmax=32768,o.head=null,o.hold=0,o.bits=0,o.lencode=o.lendyn=new p.Buf32(852),o.distcode=o.distdyn=new p.Buf32(592),o.sane=1,o.back=-1,0):I}function inflateReset(r){var o;return r&&r.state?((o=r.state).wsize=0,o.whave=0,o.wnext=0,inflateResetKeep(r)):I}function inflateReset2(r,o){var u,p;return r&&r.state?(p=r.state,o<0?(u=0,o=-o):(u=1+(o>>4),o<48&&(o&=15)),o&&(o<8||o>15)?I:(null!==p.window&&p.wbits!==o&&(p.window=null),p.wrap=u,p.wbits=o,inflateReset(r))):I}function inflateInit2(r,o){var u,p;return r?(p=new InflateState,r.state=p,p.window=null,0!==(u=inflateReset2(r,o))&&(r.state=null),u):I}var D,R,N=!0;function fixedtables(r){if(N){var o;for(D=new p.Buf32(512),R=new p.Buf32(32),o=0;o<144;)r.lens[o++]=8;for(;o<256;)r.lens[o++]=9;for(;o<280;)r.lens[o++]=7;for(;o<288;)r.lens[o++]=8;for(k(1,r.lens,0,288,D,0,r.work,{bits:9}),o=0;o<32;)r.lens[o++]=5;k(2,r.lens,0,32,R,0,r.work,{bits:5}),N=!1}r.lencode=D,r.lenbits=9,r.distcode=R,r.distbits=5}function updatewindow(r,o,u,y){var _,P=r.state;return null===P.window&&(P.wsize=1<<P.wbits,P.wnext=0,P.whave=0,P.window=new p.Buf8(P.wsize)),y>=P.wsize?(p.arraySet(P.window,o,u-P.wsize,P.wsize,0),P.wnext=0,P.whave=P.wsize):((_=P.wsize-P.wnext)>y&&(_=y),p.arraySet(P.window,o,u-y,_,P.wnext),(y-=_)?(p.arraySet(P.window,o,u-y,y,0),P.wnext=y,P.whave=P.wsize):(P.wnext+=_,P.wnext===P.wsize&&(P.wnext=0),P.whave<P.wsize&&(P.whave+=_))),0}o.inflateReset=inflateReset,o.inflateReset2=inflateReset2,o.inflateResetKeep=inflateResetKeep,o.inflateInit=function inflateInit(r){return inflateInit2(r,15)},o.inflateInit2=inflateInit2,o.inflate=function inflate(r,o){var u,D,R,N,U,W,j,G,X,Y,K,J,$,ee,te,re,ne,ie,oe,ae,se,ce,le,ue,fe=0,he=new p.Buf8(4),de=[16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15];if(!r||!r.state||!r.output||!r.input&&0!==r.avail_in)return I;(u=r.state).mode===O&&(u.mode=13),U=r.next_out,R=r.output,j=r.avail_out,N=r.next_in,D=r.input,W=r.avail_in,G=u.hold,X=u.bits,Y=W,K=j,ce=0;e:for(;;)switch(u.mode){case 1:if(0===u.wrap){u.mode=13;break}for(;X<16;){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}if(2&u.wrap&&35615===G){u.check=0,he[0]=255&G,he[1]=G>>>8&255,u.check=_(u.check,he,2,0),G=0,X=0,u.mode=2;break}if(u.flags=0,u.head&&(u.head.done=!1),!(1&u.wrap)||(((255&G)<<8)+(G>>8))%31){r.msg="incorrect header check",u.mode=B;break}if(8!=(15&G)){r.msg="unknown compression method",u.mode=B;break}if(X-=4,se=8+(15&(G>>>=4)),0===u.wbits)u.wbits=se;else if(se>u.wbits){r.msg="invalid window size",u.mode=B;break}u.dmax=1<<se,r.adler=u.check=1,u.mode=512&G?10:O,G=0,X=0;break;case 2:for(;X<16;){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}if(u.flags=G,8!=(255&u.flags)){r.msg="unknown compression method",u.mode=B;break}if(57344&u.flags){r.msg="unknown header flags set",u.mode=B;break}u.head&&(u.head.text=G>>8&1),512&u.flags&&(he[0]=255&G,he[1]=G>>>8&255,u.check=_(u.check,he,2,0)),G=0,X=0,u.mode=3;case 3:for(;X<32;){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}u.head&&(u.head.time=G),512&u.flags&&(he[0]=255&G,he[1]=G>>>8&255,he[2]=G>>>16&255,he[3]=G>>>24&255,u.check=_(u.check,he,4,0)),G=0,X=0,u.mode=4;case 4:for(;X<16;){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}u.head&&(u.head.xflags=255&G,u.head.os=G>>8),512&u.flags&&(he[0]=255&G,he[1]=G>>>8&255,u.check=_(u.check,he,2,0)),G=0,X=0,u.mode=5;case 5:if(1024&u.flags){for(;X<16;){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}u.length=G,u.head&&(u.head.extra_len=G),512&u.flags&&(he[0]=255&G,he[1]=G>>>8&255,u.check=_(u.check,he,2,0)),G=0,X=0}else u.head&&(u.head.extra=null);u.mode=6;case 6:if(1024&u.flags&&((J=u.length)>W&&(J=W),J&&(u.head&&(se=u.head.extra_len-u.length,u.head.extra||(u.head.extra=new Array(u.head.extra_len)),p.arraySet(u.head.extra,D,N,J,se)),512&u.flags&&(u.check=_(u.check,D,J,N)),W-=J,N+=J,u.length-=J),u.length))break e;u.length=0,u.mode=7;case 7:if(2048&u.flags){if(0===W)break e;J=0;do{se=D[N+J++],u.head&&se&&u.length<65536&&(u.head.name+=String.fromCharCode(se))}while(se&&J<W);if(512&u.flags&&(u.check=_(u.check,D,J,N)),W-=J,N+=J,se)break e}else u.head&&(u.head.name=null);u.length=0,u.mode=8;case 8:if(4096&u.flags){if(0===W)break e;J=0;do{se=D[N+J++],u.head&&se&&u.length<65536&&(u.head.comment+=String.fromCharCode(se))}while(se&&J<W);if(512&u.flags&&(u.check=_(u.check,D,J,N)),W-=J,N+=J,se)break e}else u.head&&(u.head.comment=null);u.mode=9;case 9:if(512&u.flags){for(;X<16;){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}if(G!==(65535&u.check)){r.msg="header crc mismatch",u.mode=B;break}G=0,X=0}u.head&&(u.head.hcrc=u.flags>>9&1,u.head.done=!0),r.adler=u.check=0,u.mode=O;break;case 10:for(;X<32;){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}r.adler=u.check=zswap32(G),G=0,X=0,u.mode=11;case 11:if(0===u.havedict)return r.next_out=U,r.avail_out=j,r.next_in=N,r.avail_in=W,u.hold=G,u.bits=X,2;r.adler=u.check=1,u.mode=O;case O:if(5===o||6===o)break e;case 13:if(u.last){G>>>=7&X,X-=7&X,u.mode=27;break}for(;X<3;){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}switch(u.last=1&G,X-=1,3&(G>>>=1)){case 0:u.mode=14;break;case 1:if(fixedtables(u),u.mode=20,6===o){G>>>=2,X-=2;break e}break;case 2:u.mode=17;break;case 3:r.msg="invalid block type",u.mode=B}G>>>=2,X-=2;break;case 14:for(G>>>=7&X,X-=7&X;X<32;){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}if((65535&G)!=(G>>>16^65535)){r.msg="invalid stored block lengths",u.mode=B;break}if(u.length=65535&G,G=0,X=0,u.mode=15,6===o)break e;case 15:u.mode=16;case 16:if(J=u.length){if(J>W&&(J=W),J>j&&(J=j),0===J)break e;p.arraySet(R,D,N,J,U),W-=J,N+=J,j-=J,U+=J,u.length-=J;break}u.mode=O;break;case 17:for(;X<14;){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}if(u.nlen=257+(31&G),G>>>=5,X-=5,u.ndist=1+(31&G),G>>>=5,X-=5,u.ncode=4+(15&G),G>>>=4,X-=4,u.nlen>286||u.ndist>30){r.msg="too many length or distance symbols",u.mode=B;break}u.have=0,u.mode=18;case 18:for(;u.have<u.ncode;){for(;X<3;){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}u.lens[de[u.have++]]=7&G,G>>>=3,X-=3}for(;u.have<19;)u.lens[de[u.have++]]=0;if(u.lencode=u.lendyn,u.lenbits=7,le={bits:u.lenbits},ce=k(0,u.lens,0,19,u.lencode,0,u.work,le),u.lenbits=le.bits,ce){r.msg="invalid code lengths set",u.mode=B;break}u.have=0,u.mode=19;case 19:for(;u.have<u.nlen+u.ndist;){for(;re=(fe=u.lencode[G&(1<<u.lenbits)-1])>>>16&255,ne=65535&fe,!((te=fe>>>24)<=X);){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}if(ne<16)G>>>=te,X-=te,u.lens[u.have++]=ne;else{if(16===ne){for(ue=te+2;X<ue;){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}if(G>>>=te,X-=te,0===u.have){r.msg="invalid bit length repeat",u.mode=B;break}se=u.lens[u.have-1],J=3+(3&G),G>>>=2,X-=2}else if(17===ne){for(ue=te+3;X<ue;){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}X-=te,se=0,J=3+(7&(G>>>=te)),G>>>=3,X-=3}else{for(ue=te+7;X<ue;){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}X-=te,se=0,J=11+(127&(G>>>=te)),G>>>=7,X-=7}if(u.have+J>u.nlen+u.ndist){r.msg="invalid bit length repeat",u.mode=B;break}for(;J--;)u.lens[u.have++]=se}}if(u.mode===B)break;if(0===u.lens[256]){r.msg="invalid code -- missing end-of-block",u.mode=B;break}if(u.lenbits=9,le={bits:u.lenbits},ce=k(1,u.lens,0,u.nlen,u.lencode,0,u.work,le),u.lenbits=le.bits,ce){r.msg="invalid literal/lengths set",u.mode=B;break}if(u.distbits=6,u.distcode=u.distdyn,le={bits:u.distbits},ce=k(2,u.lens,u.nlen,u.ndist,u.distcode,0,u.work,le),u.distbits=le.bits,ce){r.msg="invalid distances set",u.mode=B;break}if(u.mode=20,6===o)break e;case 20:u.mode=21;case 21:if(W>=6&&j>=258){r.next_out=U,r.avail_out=j,r.next_in=N,r.avail_in=W,u.hold=G,u.bits=X,P(r,K),U=r.next_out,R=r.output,j=r.avail_out,N=r.next_in,D=r.input,W=r.avail_in,G=u.hold,X=u.bits,u.mode===O&&(u.back=-1);break}for(u.back=0;re=(fe=u.lencode[G&(1<<u.lenbits)-1])>>>16&255,ne=65535&fe,!((te=fe>>>24)<=X);){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}if(re&&!(240&re)){for(ie=te,oe=re,ae=ne;re=(fe=u.lencode[ae+((G&(1<<ie+oe)-1)>>ie)])>>>16&255,ne=65535&fe,!(ie+(te=fe>>>24)<=X);){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}G>>>=ie,X-=ie,u.back+=ie}if(G>>>=te,X-=te,u.back+=te,u.length=ne,0===re){u.mode=26;break}if(32&re){u.back=-1,u.mode=O;break}if(64&re){r.msg="invalid literal/length code",u.mode=B;break}u.extra=15&re,u.mode=22;case 22:if(u.extra){for(ue=u.extra;X<ue;){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}u.length+=G&(1<<u.extra)-1,G>>>=u.extra,X-=u.extra,u.back+=u.extra}u.was=u.length,u.mode=23;case 23:for(;re=(fe=u.distcode[G&(1<<u.distbits)-1])>>>16&255,ne=65535&fe,!((te=fe>>>24)<=X);){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}if(!(240&re)){for(ie=te,oe=re,ae=ne;re=(fe=u.distcode[ae+((G&(1<<ie+oe)-1)>>ie)])>>>16&255,ne=65535&fe,!(ie+(te=fe>>>24)<=X);){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}G>>>=ie,X-=ie,u.back+=ie}if(G>>>=te,X-=te,u.back+=te,64&re){r.msg="invalid distance code",u.mode=B;break}u.offset=ne,u.extra=15&re,u.mode=24;case 24:if(u.extra){for(ue=u.extra;X<ue;){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}u.offset+=G&(1<<u.extra)-1,G>>>=u.extra,X-=u.extra,u.back+=u.extra}if(u.offset>u.dmax){r.msg="invalid distance too far back",u.mode=B;break}u.mode=25;case 25:if(0===j)break e;if(J=K-j,u.offset>J){if((J=u.offset-J)>u.whave&&u.sane){r.msg="invalid distance too far back",u.mode=B;break}J>u.wnext?(J-=u.wnext,$=u.wsize-J):$=u.wnext-J,J>u.length&&(J=u.length),ee=u.window}else ee=R,$=U-u.offset,J=u.length;J>j&&(J=j),j-=J,u.length-=J;do{R[U++]=ee[$++]}while(--J);0===u.length&&(u.mode=21);break;case 26:if(0===j)break e;R[U++]=u.length,j--,u.mode=21;break;case 27:if(u.wrap){for(;X<32;){if(0===W)break e;W--,G|=D[N++]<<X,X+=8}if(K-=j,r.total_out+=K,u.total+=K,K&&(r.adler=u.check=u.flags?_(u.check,R,K,U-K):y(u.check,R,K,U-K)),K=j,(u.flags?G:zswap32(G))!==u.check){r.msg="incorrect data check",u.mode=B;break}G=0,X=0}u.mode=28;case 28:if(u.wrap&&u.flags){for(;X<32;){if(0===W)break e;W--,G+=D[N++]<<X,X+=8}if(G!==(4294967295&u.total)){r.msg="incorrect length check",u.mode=B;break}G=0,X=0}u.mode=29;case 29:ce=1;break e;case B:ce=-3;break e;case 31:return-4;default:return I}return r.next_out=U,r.avail_out=j,r.next_in=N,r.avail_in=W,u.hold=G,u.bits=X,(u.wsize||K!==r.avail_out&&u.mode<B&&(u.mode<27||4!==o))&&updatewindow(r,r.output,r.next_out,K-r.avail_out)?(u.mode=31,-4):(Y-=r.avail_in,K-=r.avail_out,r.total_in+=Y,r.total_out+=K,u.total+=K,u.wrap&&K&&(r.adler=u.check=u.flags?_(u.check,R,K,r.next_out-K):y(u.check,R,K,r.next_out-K)),r.data_type=u.bits+(u.last?64:0)+(u.mode===O?128:0)+(20===u.mode||15===u.mode?256:0),(0===Y&&0===K||4===o)&&0===ce&&(ce=-5),ce)},o.inflateEnd=function inflateEnd(r){if(!r||!r.state)return I;var o=r.state;return o.window&&(o.window=null),r.state=null,0},o.inflateGetHeader=function inflateGetHeader(r,o){var u;return r&&r.state&&2&(u=r.state).wrap?(u.head=o,o.done=!1,0):I},o.inflateSetDictionary=function inflateSetDictionary(r,o){var u,p=o.length;return r&&r.state?0!==(u=r.state).wrap&&11!==u.mode?I:11===u.mode&&y(1,o,p,0)!==u.check?-3:updatewindow(r,o,p,p)?(u.mode=31,-4):(u.havedict=1,0):I},o.inflateInfo="pako inflate (from Nodeca project)"},2318:function(r){"use strict";function TraversalTracker(){this.events={}}TraversalTracker.prototype.startTracking=function(r,o){var u=this.events[r]||(this.events[r]=[]);u.indexOf(o)<0&&u.push(o)},TraversalTracker.prototype.stopTracking=function(r,o){var u=this.events[r];if(u){var p=u.indexOf(o);p>=0&&u.splice(p,1)}},TraversalTracker.prototype.emit=function(r){var o=Array.prototype.slice.call(arguments,1),u=this.events[r];u&&u.forEach((function(r){r.apply(this,o)}))},TraversalTracker.prototype.auto=function(r,o,u){this.startTracking(r,o),u(),this.stopTracking(r,o)},r.exports=TraversalTracker},2416:function(r){r.exports=["constructor","hasOwnProperty","isPrototypeOf","propertyIsEnumerable","toLocaleString","toString","valueOf"]},2644:function(r,o,u){var p=u(32010),y=u(2834),_=u(24517),P=u(46290),k=u(51839),I=u(39629),O=u(38688),B=p.TypeError,D=O("toPrimitive");r.exports=function(r,o){if(!_(r)||P(r))return r;var u,p=k(r,D);if(p){if(void 0===o&&(o="default"),u=y(p,r,o),!_(u)||P(u))return u;throw B("Can't convert object to primitive value")}return void 0===o&&(o="number"),I(r,o)}},2675:function(r,o,u){var p=u(20340),y=u(21594),_=u(72062),P=u(95892);r.exports=function(r,o){for(var u=y(o),k=P.f,I=_.f,O=0;O<u.length;O++){var B=u[O];p(r,B)||k(r,B,I(o,B))}}},2834:function(r){var o=Function.prototype.call;r.exports=o.bind?o.bind(o):function(){return o.apply(o,arguments)}},2857:function(r,o,u){"use strict";var p=u(59754),y=u(91102).forEach,_=p.aTypedArray;(0,p.exportTypedArrayMethod)("forEach",(function forEach(r){y(_(this),r,arguments.length>1?arguments[1]:void 0)}))},2868:function(r,o,u){var p=u(32010),y=u(70176),_=p.TypeError;r.exports=function(r,o){if(y(o,r))return r;throw _("Incorrect invocation")}},2876:function(r,o,u){u(56475)({target:"Number",stat:!0},{isInteger:u(17506)})},3131:function(r,o,u){u(98828)("Float32",(function(r){return function Float32Array(o,u,p){return r(this,o,u,p)}}))},3157:function(r){r.exports="object"==typeof window},3483:function(r){function Tree(){this.table=new Uint16Array(16),this.trans=new Uint16Array(288)}function Data(r,o){this.source=r,this.sourceIndex=0,this.tag=0,this.bitcount=0,this.dest=o,this.destLen=0,this.ltree=new Tree,this.dtree=new Tree}var o=new Tree,u=new Tree,p=new Uint8Array(30),y=new Uint16Array(30),_=new Uint8Array(30),P=new Uint16Array(30),k=new Uint8Array([16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15]),I=new Tree,O=new Uint8Array(320);function tinf_build_bits_base(r,o,u,p){var y,_;for(y=0;y<u;++y)r[y]=0;for(y=0;y<30-u;++y)r[y+u]=y/u|0;for(_=p,y=0;y<30;++y)o[y]=_,_+=1<<r[y]}var B=new Uint16Array(16);function tinf_build_tree(r,o,u,p){var y,_;for(y=0;y<16;++y)r.table[y]=0;for(y=0;y<p;++y)r.table[o[u+y]]++;for(r.table[0]=0,_=0,y=0;y<16;++y)B[y]=_,_+=r.table[y];for(y=0;y<p;++y)o[u+y]&&(r.trans[B[o[u+y]]++]=y)}function tinf_getbit(r){r.bitcount--||(r.tag=r.source[r.sourceIndex++],r.bitcount=7);var o=1&r.tag;return r.tag>>>=1,o}function tinf_read_bits(r,o,u){if(!o)return u;for(;r.bitcount<24;)r.tag|=r.source[r.sourceIndex++]<<r.bitcount,r.bitcount+=8;var p=r.tag&65535>>>16-o;return r.tag>>>=o,r.bitcount-=o,p+u}function tinf_decode_symbol(r,o){for(;r.bitcount<24;)r.tag|=r.source[r.sourceIndex++]<<r.bitcount,r.bitcount+=8;var u=0,p=0,y=0,_=r.tag;do{p=2*p+(1&_),_>>>=1,++y,u+=o.table[y],p-=o.table[y]}while(p>=0);return r.tag=_,r.bitcount-=y,o.trans[u+p]}function tinf_decode_trees(r,o,u){var p,y,_,P,B,D;for(p=tinf_read_bits(r,5,257),y=tinf_read_bits(r,5,1),_=tinf_read_bits(r,4,4),P=0;P<19;++P)O[P]=0;for(P=0;P<_;++P){var R=tinf_read_bits(r,3,0);O[k[P]]=R}for(tinf_build_tree(I,O,0,19),B=0;B<p+y;){var N=tinf_decode_symbol(r,I);switch(N){case 16:var U=O[B-1];for(D=tinf_read_bits(r,2,3);D;--D)O[B++]=U;break;case 17:for(D=tinf_read_bits(r,3,3);D;--D)O[B++]=0;break;case 18:for(D=tinf_read_bits(r,7,11);D;--D)O[B++]=0;break;default:O[B++]=N}}tinf_build_tree(o,O,0,p),tinf_build_tree(u,O,p,y)}function tinf_inflate_block_data(r,o,u){for(;;){var k,I,O,B,D=tinf_decode_symbol(r,o);if(256===D)return 0;if(D<256)r.dest[r.destLen++]=D;else for(k=tinf_read_bits(r,p[D-=257],y[D]),I=tinf_decode_symbol(r,u),B=O=r.destLen-tinf_read_bits(r,_[I],P[I]);B<O+k;++B)r.dest[r.destLen++]=r.dest[B]}}function tinf_inflate_uncompressed_block(r){for(var o,u;r.bitcount>8;)r.sourceIndex--,r.bitcount-=8;if((o=256*(o=r.source[r.sourceIndex+1])+r.source[r.sourceIndex])!==(65535&~(256*r.source[r.sourceIndex+3]+r.source[r.sourceIndex+2])))return-3;for(r.sourceIndex+=4,u=o;u;--u)r.dest[r.destLen++]=r.source[r.sourceIndex++];return r.bitcount=0,0}!function tinf_build_fixed_trees(r,o){var u;for(u=0;u<7;++u)r.table[u]=0;for(r.table[7]=24,r.table[8]=152,r.table[9]=112,u=0;u<24;++u)r.trans[u]=256+u;for(u=0;u<144;++u)r.trans[24+u]=u;for(u=0;u<8;++u)r.trans[168+u]=280+u;for(u=0;u<112;++u)r.trans[176+u]=144+u;for(u=0;u<5;++u)o.table[u]=0;for(o.table[5]=32,u=0;u<32;++u)o.trans[u]=u}(o,u),tinf_build_bits_base(p,y,4,3),tinf_build_bits_base(_,P,2,1),p[28]=0,y[28]=258,r.exports=function tinf_uncompress(r,p){var y,_,P=new Data(r,p);do{switch(y=tinf_getbit(P),tinf_read_bits(P,2,0)){case 0:_=tinf_inflate_uncompressed_block(P);break;case 1:_=tinf_inflate_block_data(P,o,u);break;case 2:tinf_decode_trees(P,P.ltree,P.dtree),_=tinf_inflate_block_data(P,P.ltree,P.dtree);break;default:_=-3}if(0!==_)throw new Error("Data error")}while(!y);return P.destLen<P.dest.length?"function"==typeof P.dest.slice?P.dest.slice(0,P.destLen):P.dest.subarray(0,P.destLen):P.dest}},3534:function(r,o,u){"use strict";var p=u(89636),y=u(77802);r.exports=function shimObjectIs(){var r=p();return y(Object,{is:r},{is:function testObjectIs(){return Object.is!==r}}),r}},3746:function(r){"use strict";var o,u,p=Function.prototype.toString,y="object"==typeof Reflect&&null!==Reflect&&Reflect.apply;if("function"==typeof y&&"function"==typeof Object.defineProperty)try{o=Object.defineProperty({},"length",{get:function(){throw u}}),u={},y((function(){throw 42}),null,o)}catch(r){r!==u&&(y=null)}else y=null;var _=/^\s*class\b/,P=function isES6ClassFunction(r){try{var o=p.call(r);return _.test(o)}catch(r){return!1}},k=function tryFunctionToStr(r){try{return!P(r)&&(p.call(r),!0)}catch(r){return!1}},I=Object.prototype.toString,O="function"==typeof Symbol&&!!Symbol.toStringTag,B=!(0 in[,]),D=function isDocumentDotAll(){return!1};if("object"==typeof document){var R=document.all;I.call(R)===I.call(document.all)&&(D=function isDocumentDotAll(r){if((B||!r)&&(void 0===r||"object"==typeof r))try{var o=I.call(r);return("[object HTMLAllCollection]"===o||"[object HTML document.all class]"===o||"[object HTMLCollection]"===o||"[object Object]"===o)&&null==r("")}catch(r){}return!1})}r.exports=y?function isCallable(r){if(D(r))return!0;if(!r)return!1;if("function"!=typeof r&&"object"!=typeof r)return!1;try{y(r,null,o)}catch(r){if(r!==u)return!1}return!P(r)&&k(r)}:function isCallable(r){if(D(r))return!0;if(!r)return!1;if("function"!=typeof r&&"object"!=typeof r)return!1;if(O)return k(r);if(P(r))return!1;var o=I.call(r);return!("[object Function]"!==o&&"[object GeneratorFunction]"!==o&&!/^\[object HTML/.test(o))&&k(r)}},3774:function(r){"use strict";r.exports=Math.max},3809:function(r,o,u){var p=u(40715).match(/firefox\/(\d+)/i);r.exports=!!p&&+p[1]},3840:function(r,o,u){var p=u(38347),y=u(34984),_=u(58659);r.exports=Object.setPrototypeOf||("__proto__"in{}?function(){var r,o=!1,u={};try{(r=p(Object.getOwnPropertyDescriptor(Object.prototype,"__proto__").set))(u,[]),o=u instanceof Array}catch(r){}return function setPrototypeOf(u,p){return y(u),_(p),o?r(u,p):u.__proto__=p,u}}():void 0)},3858:function(r,o,u){"use strict";var p=u(59754),y=u(91102).some,_=p.aTypedArray;(0,p.exportTypedArrayMethod)("some",(function some(r){return y(_(this),r,arguments.length>1?arguments[1]:void 0)}))},4420:function(r,o,u){"use strict";var p,y,_,P;u(39081),r.exports=(p=u(48352),_=(y=p).lib.Base,P=y.enc.Utf8,void(y.algo.HMAC=_.extend({init:function init(r,o){r=this._hasher=new r.init,"string"==typeof o&&(o=P.parse(o));var u=r.blockSize,p=4*u;o.sigBytes>p&&(o=r.finalize(o)),o.clamp();for(var y=this._oKey=o.clone(),_=this._iKey=o.clone(),k=y.words,I=_.words,O=0;O<u;O++)k[O]^=1549556828,I[O]^=909522486;y.sigBytes=_.sigBytes=p,this.reset()},reset:function reset(){var r=this._hasher;r.reset(),r.update(this._iKey)},update:function update(r){return this._hasher.update(r),this},finalize:function finalize(r){var o=this._hasher,u=o.finalize(r);return o.reset(),o.finalize(this._oKey.clone().concat(u))}})))},4660:function(r,o,u){"use strict";var p=u(59754),y=u(32843).left,_=p.aTypedArray;(0,p.exportTypedArrayMethod)("reduce",(function reduce(r){var o=arguments.length;return y(_(this),r,o,o>1?arguments[1]:void 0)}))},5049:function(r,o,u){"use strict";var p=u(12719);r.exports=Function.prototype.bind||p},5155:function(r,o,u){var p=u(32010);r.exports=p.Promise},5303:function(r,o){"use strict";o.DI_BRK=0,o.IN_BRK=1,o.CI_BRK=2,o.CP_BRK=3,o.PR_BRK=4,o.pairTable=[[4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,3,4,4,4,4,4,4,4,4,4,4,4],[0,4,4,1,1,4,4,4,4,1,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,4,4,4,4,1,1,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[4,4,4,1,1,1,4,4,4,1,1,1,1,1,1,1,1,1,1,1,4,2,4,1,1,1,1,1,1,1,1,1,1],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,1,1,1,1,1,1,4,2,4,1,1,1,1,1,1,1,1,1,1],[0,4,4,1,1,1,4,4,4,0,0,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,0,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,0,1,0,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,0,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,0,0,1,1,1,1,1,1,1,0,0,4,2,4,1,1,1,1,1,0,1,1,1,0],[1,4,4,1,1,1,4,4,4,0,0,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,0,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,0,1,4,4,4,0,0,1,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,0,1,4,4,4,0,0,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,1,1,1,1,1,1,4,2,4,1,1,1,1,1,1,1,1,1,0],[0,4,4,1,1,1,4,4,4,0,0,0,0,0,0,1,1,1,0,4,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,1,1,1,1,1,1,4,2,4,1,1,1,1,1,1,1,1,1,1],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,1,1,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,1,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,1,1,1,1,0,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,1,1,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,1,0,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,0,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,1,0,0,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,1,1,0],[0,4,4,1,1,1,4,4,4,0,1,0,0,0,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[1,4,4,1,1,1,4,4,4,1,1,1,1,1,0,1,1,1,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0],[0,4,4,1,1,0,4,4,4,0,0,0,0,0,0,0,0,0,0,0,4,2,4,0,0,0,0,0,0,0,0,1,0]]},5417:function(r,o,u){"use strict";var p=u(59006),y=u(32504),_=u(84490),P=_.BK,k=_.CR,I=_.LF,O=_.NL,B=_.SG,D=_.WJ,R=_.SP,N=_.ZWJ,U=_.BA,W=_.HY,j=_.NS,G=_.AI,X=_.AL,Y=_.CJ,K=_.HL,J=_.RI,$=_.SA,ee=_.XX,te=u(5303),re=te.DI_BRK,ne=te.IN_BRK,ie=te.CI_BRK,oe=te.CP_BRK,ae=(te.PR_BRK,te.pairTable),se=new p(y.toByteArray("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")),ce=function mapClass(r){switch(r){case G:case $:case B:case ee:return X;case Y:return j;default:return r}},le=function mapFirst(r){switch(r){case I:case O:return P;case R:return D;default:return r}},ue=function Break(r,o){void 0===o&&(o=!1),this.position=r,this.required=o},fe=function(){function LineBreaker(r){this.string=r,this.pos=0,this.lastPos=0,this.curClass=null,this.nextClass=null,this.LB8a=!1,this.LB21a=!1,this.LB30a=0}var r=LineBreaker.prototype;return r.nextCodePoint=function nextCodePoint(){var r=this.string.charCodeAt(this.pos++),o=this.string.charCodeAt(this.pos);return 55296<=r&&r<=56319&&56320<=o&&o<=57343?(this.pos++,1024*(r-55296)+(o-56320)+65536):r},r.nextCharClass=function nextCharClass(){return ce(se.get(this.nextCodePoint()))},r.getSimpleBreak=function getSimpleBreak(){switch(this.nextClass){case R:return!1;case P:case I:case O:return this.curClass=P,!1;case k:return this.curClass=k,!1}return null},r.getPairTableBreak=function getPairTableBreak(r){var o=!1;switch(ae[this.curClass][this.nextClass]){case re:o=!0;break;case ne:o=r===R;break;case ie:if(!(o=r===R))return o=!1;break;case oe:if(r!==R)return o}return this.LB8a&&(o=!1),!this.LB21a||this.curClass!==W&&this.curClass!==U?this.LB21a=this.curClass===K:(o=!1,this.LB21a=!1),this.curClass===J?(this.LB30a++,2==this.LB30a&&this.nextClass===J&&(o=!0,this.LB30a=0)):this.LB30a=0,this.curClass=this.nextClass,o},r.nextBreak=function nextBreak(){if(null==this.curClass){var r=this.nextCharClass();this.curClass=le(r),this.nextClass=r,this.LB8a=r===N,this.LB30a=0}for(;this.pos<this.string.length;){this.lastPos=this.pos;var o=this.nextClass;if(this.nextClass=this.nextCharClass(),this.curClass===P||this.curClass===k&&this.nextClass!==I)return this.curClass=le(ce(this.nextClass)),new ue(this.lastPos,!0);var u=this.getSimpleBreak();if(null===u&&(u=this.getPairTableBreak(o)),this.LB8a=this.nextClass===N,u)return new ue(this.lastPos)}return this.lastPos<this.string.length?(this.lastPos=this.string.length,new ue(this.string.length)):null},LineBreaker}();r.exports=fe},5557:function(r,o,u){"use strict";var p=u(64781);r.exports=p},5597:function(r,o,u){u(46042)("toPrimitive")},5844:function(r,o,u){"use strict";var p,y,_,P=u(47044),k=u(94578),I=u(10819),O=u(69548),B=u(13711),D=u(38688),R=u(63432),N=D("iterator"),U=!1;[].keys&&("next"in(_=[].keys())?(y=O(O(_)))!==Object.prototype&&(p=y):U=!0),null==p||P((function(){var r={};return p[N].call(r)!==r}))?p={}:R&&(p=I(p)),k(p[N])||B(p,N,(function(){return this})),r.exports={IteratorPrototype:p,BUGGY_SAFARI_ITERATORS:U}},6422:function(r,o,u){"use strict";var p=u(2834),y=u(11813),_=u(34984),P=u(23417),k=u(25096),I=u(83943),O=u(51839),B=u(36352),D=u(66723);y("match",(function(r,o,u){return[function match(o){var u=I(this),y=null==o?void 0:O(o,r);return y?p(y,o,u):new RegExp(o)[r](k(u))},function(r){var p=_(this),y=k(r),I=u(o,p,y);if(I.done)return I.value;if(!p.global)return D(p,y);var O=p.unicode;p.lastIndex=0;for(var R,N=[],U=0;null!==(R=D(p,y));){var W=k(R[0]);N[U]=W,""===W&&(p.lastIndex=B(y,P(p.lastIndex),O)),U++}return 0===U?null:N}]}))},6564:function(r,o,u){"use strict";var p,y,_;u(20731),r.exports=(_=u(48352),u(51270),_.mode.OFB=(p=_.lib.BlockCipherMode.extend(),y=p.Encryptor=p.extend({processBlock:function processBlock(r,o){var u=this._cipher,p=u.blockSize,y=this._iv,_=this._keystream;y&&(_=this._keystream=y.slice(0),this._iv=void 0),u.encryptBlock(_,0);for(var P=0;P<p;P++)r[o+P]^=_[P]}}),p.Decryptor=y,p),_.mode.OFB)},6611:function(r,o,u){var p=u(64429),y=u(2416).concat("length","prototype");o.f=Object.getOwnPropertyNames||function getOwnPropertyNames(r){return p(r,y)}},6616:function(r,o,u){var p,y,_,P,k=u(32010),I=u(58448),O=u(25567),B=u(94578),D=u(20340),R=u(47044),N=u(520),U=u(73163),W=u(12072),j=u(17716),G=u(95053),X=k.setImmediate,Y=k.clearImmediate,K=k.process,J=k.Dispatch,$=k.Function,ee=k.MessageChannel,te=k.String,re=0,ne={},ie="onreadystatechange";try{p=k.location}catch(r){}var run=function(r){if(D(ne,r)){var o=ne[r];delete ne[r],o()}},runner=function(r){return function(){run(r)}},listener=function(r){run(r.data)},post=function(r){k.postMessage(te(r),p.protocol+"//"+p.host)};X&&Y||(X=function setImmediate(r){var o=U(arguments,1);return ne[++re]=function(){I(B(r)?r:$(r),void 0,o)},y(re),re},Y=function clearImmediate(r){delete ne[r]},G?y=function(r){K.nextTick(runner(r))}:J&&J.now?y=function(r){J.now(runner(r))}:ee&&!j?(P=(_=new ee).port2,_.port1.onmessage=listener,y=O(P.postMessage,P)):k.addEventListener&&B(k.postMessage)&&!k.importScripts&&p&&"file:"!==p.protocol&&!R(post)?(y=post,k.addEventListener("message",listener,!1)):y=ie in W("script")?function(r){N.appendChild(W("script"))[ie]=function(){N.removeChild(this),run(r)}}:function(r){setTimeout(runner(r),0)}),r.exports={set:X,clear:Y}},6729:function(r,o,u){"use strict";var p=u(9964),y=u(14598).Buffer,_=u(9760).Transform,P=u(72908),k=u(7187),I=u(80182).ok,O=u(14598).kMaxLength,B="Cannot create final Buffer. It would be larger than 0x"+O.toString(16)+" bytes";P.Z_MIN_WINDOWBITS=8,P.Z_MAX_WINDOWBITS=15,P.Z_DEFAULT_WINDOWBITS=15,P.Z_MIN_CHUNK=64,P.Z_MAX_CHUNK=1/0,P.Z_DEFAULT_CHUNK=16384,P.Z_MIN_MEMLEVEL=1,P.Z_MAX_MEMLEVEL=9,P.Z_DEFAULT_MEMLEVEL=8,P.Z_MIN_LEVEL=-1,P.Z_MAX_LEVEL=9,P.Z_DEFAULT_LEVEL=P.Z_DEFAULT_COMPRESSION;for(var D=Object.keys(P),R=0;R<D.length;R++){var N=D[R];N.match(/^Z/)&&Object.defineProperty(o,N,{enumerable:!0,value:P[N],writable:!1})}for(var U={Z_OK:P.Z_OK,Z_STREAM_END:P.Z_STREAM_END,Z_NEED_DICT:P.Z_NEED_DICT,Z_ERRNO:P.Z_ERRNO,Z_STREAM_ERROR:P.Z_STREAM_ERROR,Z_DATA_ERROR:P.Z_DATA_ERROR,Z_MEM_ERROR:P.Z_MEM_ERROR,Z_BUF_ERROR:P.Z_BUF_ERROR,Z_VERSION_ERROR:P.Z_VERSION_ERROR},W=Object.keys(U),j=0;j<W.length;j++){var G=W[j];U[U[G]]=G}function zlibBuffer(r,o,u){var p=[],_=0;function flow(){for(var o;null!==(o=r.read());)p.push(o),_+=o.length;r.once("readable",flow)}function onEnd(){var o,P=null;_>=O?P=new RangeError(B):o=y.concat(p,_),p=[],r.close(),u(P,o)}r.on("error",(function onError(o){r.removeListener("end",onEnd),r.removeListener("readable",flow),u(o)})),r.on("end",onEnd),r.end(o),flow()}function zlibBufferSync(r,o){if("string"==typeof o&&(o=y.from(o)),!y.isBuffer(o))throw new TypeError("Not a string or buffer");var u=r._finishFlushFlag;return r._processChunk(o,u)}function Deflate(r){if(!(this instanceof Deflate))return new Deflate(r);Zlib.call(this,r,P.DEFLATE)}function Inflate(r){if(!(this instanceof Inflate))return new Inflate(r);Zlib.call(this,r,P.INFLATE)}function Gzip(r){if(!(this instanceof Gzip))return new Gzip(r);Zlib.call(this,r,P.GZIP)}function Gunzip(r){if(!(this instanceof Gunzip))return new Gunzip(r);Zlib.call(this,r,P.GUNZIP)}function DeflateRaw(r){if(!(this instanceof DeflateRaw))return new DeflateRaw(r);Zlib.call(this,r,P.DEFLATERAW)}function InflateRaw(r){if(!(this instanceof InflateRaw))return new InflateRaw(r);Zlib.call(this,r,P.INFLATERAW)}function Unzip(r){if(!(this instanceof Unzip))return new Unzip(r);Zlib.call(this,r,P.UNZIP)}function isValidFlushFlag(r){return r===P.Z_NO_FLUSH||r===P.Z_PARTIAL_FLUSH||r===P.Z_SYNC_FLUSH||r===P.Z_FULL_FLUSH||r===P.Z_FINISH||r===P.Z_BLOCK}function Zlib(r,u){var p=this;if(this._opts=r=r||{},this._chunkSize=r.chunkSize||o.Z_DEFAULT_CHUNK,_.call(this,r),r.flush&&!isValidFlushFlag(r.flush))throw new Error("Invalid flush flag: "+r.flush);if(r.finishFlush&&!isValidFlushFlag(r.finishFlush))throw new Error("Invalid flush flag: "+r.finishFlush);if(this._flushFlag=r.flush||P.Z_NO_FLUSH,this._finishFlushFlag=void 0!==r.finishFlush?r.finishFlush:P.Z_FINISH,r.chunkSize&&(r.chunkSize<o.Z_MIN_CHUNK||r.chunkSize>o.Z_MAX_CHUNK))throw new Error("Invalid chunk size: "+r.chunkSize);if(r.windowBits&&(r.windowBits<o.Z_MIN_WINDOWBITS||r.windowBits>o.Z_MAX_WINDOWBITS))throw new Error("Invalid windowBits: "+r.windowBits);if(r.level&&(r.level<o.Z_MIN_LEVEL||r.level>o.Z_MAX_LEVEL))throw new Error("Invalid compression level: "+r.level);if(r.memLevel&&(r.memLevel<o.Z_MIN_MEMLEVEL||r.memLevel>o.Z_MAX_MEMLEVEL))throw new Error("Invalid memLevel: "+r.memLevel);if(r.strategy&&r.strategy!=o.Z_FILTERED&&r.strategy!=o.Z_HUFFMAN_ONLY&&r.strategy!=o.Z_RLE&&r.strategy!=o.Z_FIXED&&r.strategy!=o.Z_DEFAULT_STRATEGY)throw new Error("Invalid strategy: "+r.strategy);if(r.dictionary&&!y.isBuffer(r.dictionary))throw new Error("Invalid dictionary: it should be a Buffer instance");this._handle=new P.Zlib(u);var k=this;this._hadError=!1,this._handle.onerror=function(r,u){_close(k),k._hadError=!0;var p=new Error(r);p.errno=u,p.code=o.codes[u],k.emit("error",p)};var I=o.Z_DEFAULT_COMPRESSION;"number"==typeof r.level&&(I=r.level);var O=o.Z_DEFAULT_STRATEGY;"number"==typeof r.strategy&&(O=r.strategy),this._handle.init(r.windowBits||o.Z_DEFAULT_WINDOWBITS,I,r.memLevel||o.Z_DEFAULT_MEMLEVEL,O,r.dictionary),this._buffer=y.allocUnsafe(this._chunkSize),this._offset=0,this._level=I,this._strategy=O,this.once("end",this.close),Object.defineProperty(this,"_closed",{get:function(){return!p._handle},configurable:!0,enumerable:!0})}function _close(r,o){o&&p.nextTick(o),r._handle&&(r._handle.close(),r._handle=null)}function emitCloseNT(r){r.emit("close")}Object.defineProperty(o,"codes",{enumerable:!0,value:Object.freeze(U),writable:!1}),o.Deflate=Deflate,o.Inflate=Inflate,o.Gzip=Gzip,o.Gunzip=Gunzip,o.DeflateRaw=DeflateRaw,o.InflateRaw=InflateRaw,o.Unzip=Unzip,o.createDeflate=function(r){return new Deflate(r)},o.createInflate=function(r){return new Inflate(r)},o.createDeflateRaw=function(r){return new DeflateRaw(r)},o.createInflateRaw=function(r){return new InflateRaw(r)},o.createGzip=function(r){return new Gzip(r)},o.createGunzip=function(r){return new Gunzip(r)},o.createUnzip=function(r){return new Unzip(r)},o.deflate=function(r,o,u){return"function"==typeof o&&(u=o,o={}),zlibBuffer(new Deflate(o),r,u)},o.deflateSync=function(r,o){return zlibBufferSync(new Deflate(o),r)},o.gzip=function(r,o,u){return"function"==typeof o&&(u=o,o={}),zlibBuffer(new Gzip(o),r,u)},o.gzipSync=function(r,o){return zlibBufferSync(new Gzip(o),r)},o.deflateRaw=function(r,o,u){return"function"==typeof o&&(u=o,o={}),zlibBuffer(new DeflateRaw(o),r,u)},o.deflateRawSync=function(r,o){return zlibBufferSync(new DeflateRaw(o),r)},o.unzip=function(r,o,u){return"function"==typeof o&&(u=o,o={}),zlibBuffer(new Unzip(o),r,u)},o.unzipSync=function(r,o){return zlibBufferSync(new Unzip(o),r)},o.inflate=function(r,o,u){return"function"==typeof o&&(u=o,o={}),zlibBuffer(new Inflate(o),r,u)},o.inflateSync=function(r,o){return zlibBufferSync(new Inflate(o),r)},o.gunzip=function(r,o,u){return"function"==typeof o&&(u=o,o={}),zlibBuffer(new Gunzip(o),r,u)},o.gunzipSync=function(r,o){return zlibBufferSync(new Gunzip(o),r)},o.inflateRaw=function(r,o,u){return"function"==typeof o&&(u=o,o={}),zlibBuffer(new InflateRaw(o),r,u)},o.inflateRawSync=function(r,o){return zlibBufferSync(new InflateRaw(o),r)},k.inherits(Zlib,_),Zlib.prototype.params=function(r,u,y){if(r<o.Z_MIN_LEVEL||r>o.Z_MAX_LEVEL)throw new RangeError("Invalid compression level: "+r);if(u!=o.Z_FILTERED&&u!=o.Z_HUFFMAN_ONLY&&u!=o.Z_RLE&&u!=o.Z_FIXED&&u!=o.Z_DEFAULT_STRATEGY)throw new TypeError("Invalid strategy: "+u);if(this._level!==r||this._strategy!==u){var _=this;this.flush(P.Z_SYNC_FLUSH,(function(){I(_._handle,"zlib binding closed"),_._handle.params(r,u),_._hadError||(_._level=r,_._strategy=u,y&&y())}))}else p.nextTick(y)},Zlib.prototype.reset=function(){return I(this._handle,"zlib binding closed"),this._handle.reset()},Zlib.prototype._flush=function(r){this._transform(y.alloc(0),"",r)},Zlib.prototype.flush=function(r,o){var u=this,_=this._writableState;("function"==typeof r||void 0===r&&!o)&&(o=r,r=P.Z_FULL_FLUSH),_.ended?o&&p.nextTick(o):_.ending?o&&this.once("end",o):_.needDrain?o&&this.once("drain",(function(){return u.flush(r,o)})):(this._flushFlag=r,this.write(y.alloc(0),"",o))},Zlib.prototype.close=function(r){_close(this,r),p.nextTick(emitCloseNT,this)},Zlib.prototype._transform=function(r,o,u){var p,_=this._writableState,k=(_.ending||_.ended)&&(!r||_.length===r.length);return null===r||y.isBuffer(r)?this._handle?(k?p=this._finishFlushFlag:(p=this._flushFlag,r.length>=_.length&&(this._flushFlag=this._opts.flush||P.Z_NO_FLUSH)),void this._processChunk(r,p,u)):u(new Error("zlib binding closed")):u(new Error("invalid input"))},Zlib.prototype._processChunk=function(r,o,u){var p=r&&r.length,_=this._chunkSize-this._offset,P=0,k=this,D="function"==typeof u;if(!D){var R,N=[],U=0;this.on("error",(function(r){R=r})),I(this._handle,"zlib binding closed");do{var W=this._handle.writeSync(o,r,P,p,this._buffer,this._offset,_)}while(!this._hadError&&callback(W[0],W[1]));if(this._hadError)throw R;if(U>=O)throw _close(this),new RangeError(B);var j=y.concat(N,U);return _close(this),j}I(this._handle,"zlib binding closed");var G=this._handle.write(o,r,P,p,this._buffer,this._offset,_);function callback(O,B){if(this&&(this.buffer=null,this.callback=null),!k._hadError){var R=_-B;if(I(R>=0,"have should not go down"),R>0){var W=k._buffer.slice(k._offset,k._offset+R);k._offset+=R,D?k.push(W):(N.push(W),U+=W.length)}if((0===B||k._offset>=k._chunkSize)&&(_=k._chunkSize,k._offset=0,k._buffer=y.allocUnsafe(k._chunkSize)),0===B){if(P+=p-O,p=O,!D)return!0;var j=k._handle.write(o,r,P,p,k._buffer,k._offset,k._chunkSize);return j.callback=callback,void(j.buffer=r)}if(!D)return!1;u()}}G.buffer=r,G.callback=callback},k.inherits(Deflate,Zlib),k.inherits(Inflate,Zlib),k.inherits(Gzip,Zlib),k.inherits(Gunzip,Zlib),k.inherits(DeflateRaw,Zlib),k.inherits(InflateRaw,Zlib),k.inherits(Unzip,Zlib)},6819:function(r,o,u){"use strict";var p;r.exports=(p=u(48352),u(66947),u(68319),u(82747),u(51270),function(){var r=p,o=r.lib.StreamCipher,u=r.algo,y=[],_=[],P=[],k=u.Rabbit=o.extend({_doReset:function _doReset(){for(var r=this._key.words,o=this.cfg.iv,u=0;u<4;u++)r[u]=16711935&(r[u]<<8|r[u]>>>24)|4278255360&(r[u]<<24|r[u]>>>8);var p=this._X=[r[0],r[3]<<16|r[2]>>>16,r[1],r[0]<<16|r[3]>>>16,r[2],r[1]<<16|r[0]>>>16,r[3],r[2]<<16|r[1]>>>16],y=this._C=[r[2]<<16|r[2]>>>16,4294901760&r[0]|65535&r[1],r[3]<<16|r[3]>>>16,4294901760&r[1]|65535&r[2],r[0]<<16|r[0]>>>16,4294901760&r[2]|65535&r[3],r[1]<<16|r[1]>>>16,4294901760&r[3]|65535&r[0]];for(this._b=0,u=0;u<4;u++)nextState.call(this);for(u=0;u<8;u++)y[u]^=p[u+4&7];if(o){var _=o.words,P=_[0],k=_[1],I=16711935&(P<<8|P>>>24)|4278255360&(P<<24|P>>>8),O=16711935&(k<<8|k>>>24)|4278255360&(k<<24|k>>>8),B=I>>>16|4294901760&O,D=O<<16|65535&I;for(y[0]^=I,y[1]^=B,y[2]^=O,y[3]^=D,y[4]^=I,y[5]^=B,y[6]^=O,y[7]^=D,u=0;u<4;u++)nextState.call(this)}},_doProcessBlock:function _doProcessBlock(r,o){var u=this._X;nextState.call(this),y[0]=u[0]^u[5]>>>16^u[3]<<16,y[1]=u[2]^u[7]>>>16^u[5]<<16,y[2]=u[4]^u[1]>>>16^u[7]<<16,y[3]=u[6]^u[3]>>>16^u[1]<<16;for(var p=0;p<4;p++)y[p]=16711935&(y[p]<<8|y[p]>>>24)|4278255360&(y[p]<<24|y[p]>>>8),r[o+p]^=y[p]},blockSize:4,ivSize:2});function nextState(){for(var r=this._X,o=this._C,u=0;u<8;u++)_[u]=o[u];for(o[0]=o[0]+1295307597+this._b|0,o[1]=o[1]+3545052371+(o[0]>>>0<_[0]>>>0?1:0)|0,o[2]=o[2]+886263092+(o[1]>>>0<_[1]>>>0?1:0)|0,o[3]=o[3]+1295307597+(o[2]>>>0<_[2]>>>0?1:0)|0,o[4]=o[4]+3545052371+(o[3]>>>0<_[3]>>>0?1:0)|0,o[5]=o[5]+886263092+(o[4]>>>0<_[4]>>>0?1:0)|0,o[6]=o[6]+1295307597+(o[5]>>>0<_[5]>>>0?1:0)|0,o[7]=o[7]+3545052371+(o[6]>>>0<_[6]>>>0?1:0)|0,this._b=o[7]>>>0<_[7]>>>0?1:0,u=0;u<8;u++){var p=r[u]+o[u],y=65535&p,k=p>>>16,I=((y*y>>>17)+y*k>>>15)+k*k,O=((4294901760&p)*p|0)+((65535&p)*p|0);P[u]=I^O}r[0]=P[0]+(P[7]<<16|P[7]>>>16)+(P[6]<<16|P[6]>>>16)|0,r[1]=P[1]+(P[0]<<8|P[0]>>>24)+P[7]|0,r[2]=P[2]+(P[1]<<16|P[1]>>>16)+(P[0]<<16|P[0]>>>16)|0,r[3]=P[3]+(P[2]<<8|P[2]>>>24)+P[1]|0,r[4]=P[4]+(P[3]<<16|P[3]>>>16)+(P[2]<<16|P[2]>>>16)|0,r[5]=P[5]+(P[4]<<8|P[4]>>>24)+P[3]|0,r[6]=P[6]+(P[5]<<16|P[5]>>>16)+(P[4]<<16|P[4]>>>16)|0,r[7]=P[7]+(P[6]<<8|P[6]>>>24)+P[5]|0}r.Rabbit=o._createHelper(k)}(),p.Rabbit)},7043:function(r,o){o.lookup=new Uint8Array([0,0,0,0,0,0,0,0,0,4,4,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,12,16,12,12,20,12,16,24,28,12,12,32,12,36,12,44,44,44,44,44,44,44,44,44,44,32,32,24,40,28,12,12,48,52,52,52,48,52,52,52,48,52,52,52,52,52,48,52,52,52,52,52,48,52,52,52,52,52,24,12,28,12,12,12,56,60,60,60,56,60,60,60,56,60,60,60,60,60,56,60,60,60,60,60,56,60,60,60,60,60,24,12,28,12,0,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,2,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,2,2,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,1,1,1,1,1,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,7,0,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,24,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,32,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,40,48,48,48,48,48,48,48,48,48,48,48,48,48,48,48,56,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59,60,61,62,63,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59,60,61,62,63,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59,60,61,62,63,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59,60,61,62,63,0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4,5,5,5,5,6,6,6,6,7,7,7,7,8,8,8,8,9,9,9,9,10,10,10,10,11,11,11,11,12,12,12,12,13,13,13,13,14,14,14,14,15,15,15,15,16,16,16,16,17,17,17,17,18,18,18,18,19,19,19,19,20,20,20,20,21,21,21,21,22,22,22,22,23,23,23,23,24,24,24,24,25,25,25,25,26,26,26,26,27,27,27,27,28,28,28,28,29,29,29,29,30,30,30,30,31,31,31,31,32,32,32,32,33,33,33,33,34,34,34,34,35,35,35,35,36,36,36,36,37,37,37,37,38,38,38,38,39,39,39,39,40,40,40,40,41,41,41,41,42,42,42,42,43,43,43,43,44,44,44,44,45,45,45,45,46,46,46,46,47,47,47,47,48,48,48,48,49,49,49,49,50,50,50,50,51,51,51,51,52,52,52,52,53,53,53,53,54,54,54,54,55,55,55,55,56,56,56,56,57,57,57,57,58,58,58,58,59,59,59,59,60,60,60,60,61,61,61,61,62,62,62,62,63,63,63,63,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]),o.lookupOffsets=new Uint16Array([1024,1536,1280,1536,0,256,768,512])},7051:function(r,o,u){"use strict";var p=u(26601),y=u(77802),_=u(82621),P=u(61320),k=u(35074),I=p(P(),Number);y(I,{getPolyfill:P,implementation:_,shim:k}),r.exports=I},7081:function(r,o,u){var p=u(15567),y=u(20340),_=Function.prototype,P=p&&Object.getOwnPropertyDescriptor,k=y(_,"name"),I=k&&"something"===function something(){}.name,O=k&&(!p||p&&P(_,"name").configurable);r.exports={EXISTS:k,PROPER:I,CONFIGURABLE:O}},7187:function(r,o,u){var p=u(9964),y=Object.getOwnPropertyDescriptors||function getOwnPropertyDescriptors(r){for(var o=Object.keys(r),u={},p=0;p<o.length;p++)u[o[p]]=Object.getOwnPropertyDescriptor(r,o[p]);return u},_=/%[sdj%]/g;o.format=function(r){if(!isString(r)){for(var o=[],u=0;u<arguments.length;u++)o.push(inspect(arguments[u]));return o.join(" ")}u=1;for(var p=arguments,y=p.length,P=String(r).replace(_,(function(r){if("%%"===r)return"%";if(u>=y)return r;switch(r){case"%s":return String(p[u++]);case"%d":return Number(p[u++]);case"%j":try{return JSON.stringify(p[u++])}catch(r){return"[Circular]"}default:return r}})),k=p[u];u<y;k=p[++u])isNull(k)||!isObject(k)?P+=" "+k:P+=" "+inspect(k);return P},o.deprecate=function(r,u){if(void 0!==p&&!0===p.noDeprecation)return r;if(void 0===p)return function(){return o.deprecate(r,u).apply(this,arguments)};var y=!1;return function deprecated(){if(!y){if(p.throwDeprecation)throw new Error(u);p.traceDeprecation,y=!0}return r.apply(this,arguments)}};var P={},k=/^$/;if(p.env.NODE_DEBUG){var I=p.env.NODE_DEBUG;I=I.replace(/[|\\{}()[\]^$+?.]/g,"\\$&").replace(/\*/g,".*").replace(/,/g,"$|^").toUpperCase(),k=new RegExp("^"+I+"$","i")}function inspect(r,u){var p={seen:[],stylize:stylizeNoColor};return arguments.length>=3&&(p.depth=arguments[2]),arguments.length>=4&&(p.colors=arguments[3]),isBoolean(u)?p.showHidden=u:u&&o._extend(p,u),isUndefined(p.showHidden)&&(p.showHidden=!1),isUndefined(p.depth)&&(p.depth=2),isUndefined(p.colors)&&(p.colors=!1),isUndefined(p.customInspect)&&(p.customInspect=!0),p.colors&&(p.stylize=stylizeWithColor),formatValue(p,r,p.depth)}function stylizeWithColor(r,o){var u=inspect.styles[o];return u?"["+inspect.colors[u][0]+"m"+r+"["+inspect.colors[u][1]+"m":r}function stylizeNoColor(r,o){return r}function formatValue(r,u,p){if(r.customInspect&&u&&isFunction(u.inspect)&&u.inspect!==o.inspect&&(!u.constructor||u.constructor.prototype!==u)){var y=u.inspect(p,r);return isString(y)||(y=formatValue(r,y,p)),y}var _=function formatPrimitive(r,o){if(isUndefined(o))return r.stylize("undefined","undefined");if(isString(o)){var u="'"+JSON.stringify(o).replace(/^"|"$/g,"").replace(/'/g,"\\'").replace(/\\"/g,'"')+"'";return r.stylize(u,"string")}if(isNumber(o))return r.stylize(""+o,"number");if(isBoolean(o))return r.stylize(""+o,"boolean");if(isNull(o))return r.stylize("null","null")}(r,u);if(_)return _;var P=Object.keys(u),k=function arrayToHash(r){var o={};return r.forEach((function(r,u){o[r]=!0})),o}(P);if(r.showHidden&&(P=Object.getOwnPropertyNames(u)),isError(u)&&(P.indexOf("message")>=0||P.indexOf("description")>=0))return formatError(u);if(0===P.length){if(isFunction(u)){var I=u.name?": "+u.name:"";return r.stylize("[Function"+I+"]","special")}if(isRegExp(u))return r.stylize(RegExp.prototype.toString.call(u),"regexp");if(isDate(u))return r.stylize(Date.prototype.toString.call(u),"date");if(isError(u))return formatError(u)}var O,B="",D=!1,R=["{","}"];(isArray(u)&&(D=!0,R=["[","]"]),isFunction(u))&&(B=" [Function"+(u.name?": "+u.name:"")+"]");return isRegExp(u)&&(B=" "+RegExp.prototype.toString.call(u)),isDate(u)&&(B=" "+Date.prototype.toUTCString.call(u)),isError(u)&&(B=" "+formatError(u)),0!==P.length||D&&0!=u.length?p<0?isRegExp(u)?r.stylize(RegExp.prototype.toString.call(u),"regexp"):r.stylize("[Object]","special"):(r.seen.push(u),O=D?function formatArray(r,o,u,p,y){for(var _=[],P=0,k=o.length;P<k;++P)hasOwnProperty(o,String(P))?_.push(formatProperty(r,o,u,p,String(P),!0)):_.push("");return y.forEach((function(y){y.match(/^\d+$/)||_.push(formatProperty(r,o,u,p,y,!0))})),_}(r,u,p,k,P):P.map((function(o){return formatProperty(r,u,p,k,o,D)})),r.seen.pop(),function reduceToSingleString(r,o,u){var p=r.reduce((function(r,o){return o.indexOf("\n")>=0&&0,r+o.replace(/\u001b\[\d\d?m/g,"").length+1}),0);if(p>60)return u[0]+(""===o?"":o+"\n ")+" "+r.join(",\n  ")+" "+u[1];return u[0]+o+" "+r.join(", ")+" "+u[1]}(O,B,R)):R[0]+B+R[1]}function formatError(r){return"["+Error.prototype.toString.call(r)+"]"}function formatProperty(r,o,u,p,y,_){var P,k,I;if((I=Object.getOwnPropertyDescriptor(o,y)||{value:o[y]}).get?k=I.set?r.stylize("[Getter/Setter]","special"):r.stylize("[Getter]","special"):I.set&&(k=r.stylize("[Setter]","special")),hasOwnProperty(p,y)||(P="["+y+"]"),k||(r.seen.indexOf(I.value)<0?(k=isNull(u)?formatValue(r,I.value,null):formatValue(r,I.value,u-1)).indexOf("\n")>-1&&(k=_?k.split("\n").map((function(r){return"  "+r})).join("\n").slice(2):"\n"+k.split("\n").map((function(r){return"   "+r})).join("\n")):k=r.stylize("[Circular]","special")),isUndefined(P)){if(_&&y.match(/^\d+$/))return k;(P=JSON.stringify(""+y)).match(/^"([a-zA-Z_][a-zA-Z_0-9]*)"$/)?(P=P.slice(1,-1),P=r.stylize(P,"name")):(P=P.replace(/'/g,"\\'").replace(/\\"/g,'"').replace(/(^"|"$)/g,"'"),P=r.stylize(P,"string"))}return P+": "+k}function isArray(r){return Array.isArray(r)}function isBoolean(r){return"boolean"==typeof r}function isNull(r){return null===r}function isNumber(r){return"number"==typeof r}function isString(r){return"string"==typeof r}function isUndefined(r){return void 0===r}function isRegExp(r){return isObject(r)&&"[object RegExp]"===objectToString(r)}function isObject(r){return"object"==typeof r&&null!==r}function isDate(r){return isObject(r)&&"[object Date]"===objectToString(r)}function isError(r){return isObject(r)&&("[object Error]"===objectToString(r)||r instanceof Error)}function isFunction(r){return"function"==typeof r}function objectToString(r){return Object.prototype.toString.call(r)}o.debuglog=function(r){if(r=r.toUpperCase(),!P[r])if(k.test(r)){p.pid;P[r]=function(){o.format.apply(o,arguments)}}else P[r]=function(){};return P[r]},o.inspect=inspect,inspect.colors={bold:[1,22],italic:[3,23],underline:[4,24],inverse:[7,27],white:[37,39],grey:[90,39],black:[30,39],blue:[34,39],cyan:[36,39],green:[32,39],magenta:[35,39],red:[31,39],yellow:[33,39]},inspect.styles={special:"cyan",number:"yellow",boolean:"yellow",undefined:"grey",null:"bold",string:"green",date:"magenta",regexp:"red"},o.types=u(29490),o.isArray=isArray,o.isBoolean=isBoolean,o.isNull=isNull,o.isNullOrUndefined=function isNullOrUndefined(r){return null==r},o.isNumber=isNumber,o.isString=isString,o.isSymbol=function isSymbol(r){return"symbol"==typeof r},o.isUndefined=isUndefined,o.isRegExp=isRegExp,o.types.isRegExp=isRegExp,o.isObject=isObject,o.isDate=isDate,o.types.isDate=isDate,o.isError=isError,o.types.isNativeError=isError,o.isFunction=isFunction,o.isPrimitive=function isPrimitive(r){return null===r||"boolean"==typeof r||"number"==typeof r||"string"==typeof r||"symbol"==typeof r||void 0===r},o.isBuffer=u(41201);function hasOwnProperty(r,o){return Object.prototype.hasOwnProperty.call(r,o)}o.log=function(){},o.inherits=u(89784),o._extend=function(r,o){if(!o||!isObject(o))return r;for(var u=Object.keys(o),p=u.length;p--;)r[u[p]]=o[u[p]];return r};var O="undefined"!=typeof Symbol?Symbol("util.promisify.custom"):void 0;function callbackifyOnRejected(r,o){if(!r){var u=new Error("Promise was rejected with a falsy value");u.reason=r,r=u}return o(r)}o.promisify=function promisify(r){if("function"!=typeof r)throw new TypeError('The "original" argument must be of type Function');if(O&&r[O]){var o;if("function"!=typeof(o=r[O]))throw new TypeError('The "util.promisify.custom" argument must be of type Function');return Object.defineProperty(o,O,{value:o,enumerable:!1,writable:!1,configurable:!0}),o}function o(){for(var o,u,p=new Promise((function(r,p){o=r,u=p})),y=[],_=0;_<arguments.length;_++)y.push(arguments[_]);y.push((function(r,p){r?u(r):o(p)}));try{r.apply(this,y)}catch(r){u(r)}return p}return Object.setPrototypeOf(o,Object.getPrototypeOf(r)),O&&Object.defineProperty(o,O,{value:o,enumerable:!1,writable:!1,configurable:!0}),Object.defineProperties(o,y(r))},o.promisify.custom=O,o.callbackify=function callbackify(r){if("function"!=typeof r)throw new TypeError('The "original" argument must be of type Function');function callbackified(){for(var o=[],u=0;u<arguments.length;u++)o.push(arguments[u]);var y=o.pop();if("function"!=typeof y)throw new TypeError("The last argument must be of type Function");var _=this,cb=function(){return y.apply(_,arguments)};r.apply(this,o).then((function(r){p.nextTick(cb.bind(null,null,r))}),(function(r){p.nextTick(callbackifyOnRejected.bind(null,r,cb))}))}return Object.setPrototypeOf(callbackified,Object.getPrototypeOf(r)),Object.defineProperties(callbackified,y(r)),callbackified}},7283:function(r,o,u){"use strict";var p=u(15567),y=u(32010),_=u(38347),P=u(39599),k=u(13711),I=u(20340),O=u(51868),B=u(70176),D=u(46290),R=u(2644),N=u(47044),U=u(6611).f,W=u(72062).f,j=u(95892).f,G=u(16679),X=u(29841).trim,Y="Number",K=y[Y],J=K.prototype,$=y.TypeError,ee=_("".slice),te=_("".charCodeAt),toNumber=function(r){var o,u,p,y,_,P,k,I,O=R(r,"number");if(D(O))throw $("Cannot convert a Symbol value to a number");if("string"==typeof O&&O.length>2)if(O=X(O),43===(o=te(O,0))||45===o){if(88===(u=te(O,2))||120===u)return NaN}else if(48===o){switch(te(O,1)){case 66:case 98:p=2,y=49;break;case 79:case 111:p=8,y=55;break;default:return+O}for(P=(_=ee(O,2)).length,k=0;k<P;k++)if((I=te(_,k))<48||I>y)return NaN;return parseInt(_,p)}return+O};if(P(Y,!K(" 0o1")||!K("0b1")||K("+0x1"))){for(var re,ne=function Number(r){var o=arguments.length<1?0:K(function(r){var o=R(r,"number");return"bigint"==typeof o?o:toNumber(o)}(r)),u=this;return B(J,u)&&N((function(){G(u)}))?O(Object(o),u,ne):o},ie=p?U(K):"MAX_VALUE,MIN_VALUE,NaN,NEGATIVE_INFINITY,POSITIVE_INFINITY,EPSILON,MAX_SAFE_INTEGER,MIN_SAFE_INTEGER,isFinite,isInteger,isNaN,isSafeInteger,parseFloat,parseInt,fromString,range".split(","),oe=0;ie.length>oe;oe++)I(K,re=ie[oe])&&!I(ne,re)&&j(ne,re,W(K,re));ne.prototype=J,J.constructor=ne,k(y,Y,ne)}},7421:function(r,o,u){var p=u(32010),y=Object.defineProperty;r.exports=function(r,o){try{y(p,r,{value:o,configurable:!0,writable:!0})}catch(u){p[r]=o}return o}},7452:function(r,o,u){var p=u(47044);r.exports=function(r){return p((function(){var o=""[r]('"');return o!==o.toLowerCase()||o.split('"').length>3}))}},7514:function(r,o,u){var p=u(32010),y=u(38347),_=u(47044),P=u(93975),k=p.Object,I=y("".split);r.exports=_((function(){return!k("z").propertyIsEnumerable(0)}))?function(r){return"String"==P(r)?I(r,""):k(r)}:k},7585:function(r,o,u){var p=u(20340),y=u(13711),_=u(53087),P=u(38688)("toPrimitive"),k=Date.prototype;p(k,P)||y(k,P,_)},7785:function(r,o,u){"use strict";"undefined"==typeof window||window.Promise||u(98168),u(83043);function URLBrowserResolver(r){this.fs=r,this.resolving={}}URLBrowserResolver.prototype.resolve=function(r,o){if(!this.resolving[r]){var u=this;this.resolving[r]=new Promise((function(p,y){0===r.toLowerCase().indexOf("https://")||0===r.toLowerCase().indexOf("http://")?u.fs.existsSync(r)?p():function(r,o){return new Promise((function(u,p){var y=new XMLHttpRequest;for(var _ in y.open("GET",r,!0),o)y.setRequestHeader(_,o[_]);y.responseType="arraybuffer",y.onreadystatechange=function(){4===y.readyState&&(y.status>=200&&y.status<300||setTimeout((function(){p(new TypeError('Failed to fetch (url: "'+r+'")'))}),0))},y.onload=function(){y.status>=200&&y.status<300&&u(y.response)},y.onerror=function(){setTimeout((function(){p(new TypeError('Network request failed (url: "'+r+'")'))}),0)},y.ontimeout=function(){setTimeout((function(){p(new TypeError('Network request failed (url: "'+r+'")'))}),0)},y.send()}))}(r,o).then((function(o){u.fs.writeFileSync(r,o),p()}),(function(r){y(r)})):p()}))}return this.resolving[r]},URLBrowserResolver.prototype.resolved=function(){var r=this;return new Promise((function(o,u){Promise.all(Object.values(r.resolving)).then((function(){o()}),(function(r){u(r)}))}))},r.exports=URLBrowserResolver},7844:function(r,o,u){"use strict";var p=u(83089),y=u(77802).supportsDescriptors,_=Object.getOwnPropertyDescriptor;r.exports=function getPolyfill(){if(y&&"gim"===/a/gim.flags){var r=_(RegExp.prototype,"flags");if(r&&"function"==typeof r.get&&"dotAll"in RegExp.prototype&&"hasIndices"in RegExp.prototype){var o="",u={};if(Object.defineProperty(u,"hasIndices",{get:function(){o+="d"}}),Object.defineProperty(u,"sticky",{get:function(){o+="y"}}),r.get.call(u),"dy"===o)return r.get}}return p}},7851:function(r,o,u){"use strict";var p=u(58448),y=u(2834),_=u(38347),P=u(11813),k=u(28831),I=u(34984),O=u(83943),B=u(27754),D=u(36352),R=u(23417),N=u(25096),U=u(51839),W=u(73163),j=u(66723),G=u(49820),X=u(74846),Y=u(47044),K=X.UNSUPPORTED_Y,J=4294967295,$=Math.min,ee=[].push,te=_(/./.exec),re=_(ee),ne=_("".slice),ie=!Y((function(){var r=/(?:)/,o=r.exec;r.exec=function(){return o.apply(this,arguments)};var u="ab".split(r);return 2!==u.length||"a"!==u[0]||"b"!==u[1]}));P("split",(function(r,o,u){var _;return _="c"=="abbc".split(/(b)*/)[1]||4!="test".split(/(?:)/,-1).length||2!="ab".split(/(?:ab)*/).length||4!=".".split(/(.?)(.?)/).length||".".split(/()()/).length>1||"".split(/.?/).length?function(r,u){var _=N(O(this)),P=void 0===u?J:u>>>0;if(0===P)return[];if(void 0===r)return[_];if(!k(r))return y(o,_,r,P);for(var I,B,D,R=[],U=(r.ignoreCase?"i":"")+(r.multiline?"m":"")+(r.unicode?"u":"")+(r.sticky?"y":""),j=0,X=new RegExp(r.source,U+"g");(I=y(G,X,_))&&!((B=X.lastIndex)>j&&(re(R,ne(_,j,I.index)),I.length>1&&I.index<_.length&&p(ee,R,W(I,1)),D=I[0].length,j=B,R.length>=P));)X.lastIndex===I.index&&X.lastIndex++;return j===_.length?!D&&te(X,"")||re(R,""):re(R,ne(_,j)),R.length>P?W(R,0,P):R}:"0".split(void 0,0).length?function(r,u){return void 0===r&&0===u?[]:y(o,this,r,u)}:o,[function split(o,u){var p=O(this),P=null==o?void 0:U(o,r);return P?y(P,o,p,u):y(_,N(p),o,u)},function(r,p){var y=I(this),P=N(r),k=u(_,y,P,p,_!==o);if(k.done)return k.value;var O=B(y,RegExp),U=y.unicode,W=(y.ignoreCase?"i":"")+(y.multiline?"m":"")+(y.unicode?"u":"")+(K?"g":"y"),G=new O(K?"^(?:"+y.source+")":y,W),X=void 0===p?J:p>>>0;if(0===X)return[];if(0===P.length)return null===j(G,P)?[P]:[];for(var Y=0,ee=0,te=[];ee<P.length;){G.lastIndex=K?0:ee;var ie,oe=j(G,K?ne(P,ee):P);if(null===oe||(ie=$(R(G.lastIndex+(K?ee:0)),P.length))===Y)ee=D(P,ee,U);else{if(re(te,ne(P,Y,ee)),te.length===X)return te;for(var ae=1;ae<=oe.length-1;ae++)if(re(te,oe[ae]),te.length===X)return te;ee=Y=ie}}return re(te,ne(P,Y)),te}]}),!ie,K)},8807:function(r,o,u){var p=u(93975),y=u(98086),_=u(6611).f,P=u(73163),k="object"==typeof window&&window&&Object.getOwnPropertyNames?Object.getOwnPropertyNames(window):[];r.exports.f=function getOwnPropertyNames(r){return k&&"Window"==p(r)?function(r){try{return _(r)}catch(r){return P(k)}}(r):_(y(r))}},8953:function(r,o,u){"use strict";var p=u(56475),y=u(32010),_=u(83124),P=u(51334),k="ArrayBuffer",I=_[k];p({global:!0,forced:y[k]!==I},{ArrayBuffer:I}),P(k)},9240:function(r){"use strict";r.exports=JSON.parse('{"ifd":{"8298":"Copyright","8769":"ExifIFDPointer","8822":"ExposureProgram","8824":"SpectralSensitivity","8825":"GPSInfoIFDPointer","8827":"PhotographicSensitivity","8828":"OECF","8830":"SensitivityType","8831":"StandardOutputSensitivity","8832":"RecommendedExposureIndex","8833":"ISOSpeed","8834":"ISOSpeedLatitudeyyy","8835":"ISOSpeedLatitudezzz","9000":"ExifVersion","9003":"DateTimeOriginal","9004":"DateTimeDigitized","9101":"ComponentsConfiguration","9102":"CompressedBitsPerPixel","9201":"ShutterSpeedValue","9202":"ApertureValue","9203":"BrightnessValue","9204":"ExposureBiasValue","9205":"MaxApertureValue","9206":"SubjectDistance","9207":"MeteringMode","9208":"LightSource","9209":"Flash","9214":"SubjectArea","9286":"UserComment","9290":"SubSecTime","9291":"SubSecTimeOriginal","9292":"SubSecTimeDigitized","010e":"ImageDescription","010f":"Make","011a":"XResolution","011b":"YResolution","011c":"PlanarConfiguration","012d":"TransferFunction","013b":"Artist","013e":"WhitePoint","013f":"PrimaryChromaticities","0100":"ImageWidth","0101":"ImageHeight","0102":"BitsPerSample","0103":"Compression","0106":"PhotometricInterpretation","0110":"Model","0111":"StripOffsets","0112":"Orientation","0115":"SamplesPerPixel","0116":"RowsPerStrip","0117":"StripByteCounts","0128":"ResolutionUnit","0131":"Software","0132":"DateTime","0201":"JPEGInterchangeFormat","0202":"JPEGInterchangeFormatLength","0211":"YCbCrCoefficients","0212":"YCbCrSubSampling","0213":"YCbCrPositioning","0214":"ReferenceBlackWhite","829a":"ExposureTime","829d":"FNumber","920a":"FocalLength","927c":"MakerNote","a000":"FlashpixVersion","a001":"ColorSpace","a002":"PixelXDimension","a003":"PixelYDimension","a004":"RelatedSoundFile","a005":"InteroperabilityIFDPointer","a20b":"FlashEnergy","a20c":"SpatialFrequencyResponse","a20e":"FocalPlaneXResolution","a20f":"FocalPlaneYResolution","a40a":"Sharpness","a40b":"DeviceSettingDescription","a40c":"SubjectDistanceRange","a210":"FocalPlaneResolutionUnit","a214":"SubjectLocation","a215":"ExposureIndex","a217":"SensingMethod","a300":"FileSource","a301":"SceneType","a302":"CFAPattern","a401":"CustomRendered","a402":"ExposureMode","a403":"WhiteBalance","a404":"DigitalZoomRatio","a405":"FocalLengthIn35mmFilm","a406":"SceneCaptureType","a407":"GainControl","a408":"Contrast","a409":"Saturation","a420":"ImageUniqueID","a430":"CameraOwnerName","a431":"BodySerialNumber","a432":"LensSpecification","a433":"LensMake","a434":"LensModel","a435":"LensSerialNumber","a500":"Gamma"},"gps":{"0000":"GPSVersionID","0001":"GPSLatitudeRef","0002":"GPSLatitude","0003":"GPSLongitudeRef","0004":"GPSLongitude","0005":"GPSAltitudeRef","0006":"GPSAltitude","0007":"GPSTimeStamp","0008":"GPSSatellites","0009":"GPSStatus","000a":"GPSMeasureMode","000b":"GPSDOP","000c":"GPSSpeedRef","000d":"GPSSpeed","000e":"GPSTrackRef","000f":"GPSTrack","0010":"GPSImgDirectionRef","0011":"GPSImgDirection","0012":"GPSMapDatum","0013":"GPSDestLatitudeRef","0014":"GPSDestLatitude","0015":"GPSDestLongitudeRef","0016":"GPSDestLongitude","0017":"GPSDestBearingRef","0018":"GPSDestBearing","0019":"GPSDestDistanceRef","001a":"GPSDestDistance","001b":"GPSProcessingMethod","001c":"GPSAreaInformation","001d":"GPSDateStamp","001e":"GPSDifferential","001f":"GPSHPositioningError"}}')},9324:function(r,o,u){"use strict";var p=u(18128),y=function(){function Pointer(r,o,u){if(void 0===u&&(u={}),this.offsetType=r,this.type=o,this.options=u,"void"===this.type&&(this.type=null),null==this.options.type&&(this.options.type="local"),null==this.options.allowNull&&(this.options.allowNull=!0),null==this.options.nullValue&&(this.options.nullValue=0),null==this.options.lazy&&(this.options.lazy=!1),this.options.relativeTo){if("function"!=typeof this.options.relativeTo)throw new Error("relativeTo option must be a function");this.relativeToGetter=u.relativeTo}}var r=Pointer.prototype;return r.decode=function decode(r,o){var u,y=this,_=this.offsetType.decode(r,o);if(_===this.options.nullValue&&this.options.allowNull)return null;switch(this.options.type){case"local":u=o._startOffset;break;case"immediate":u=r.pos-this.offsetType.size();break;case"parent":u=o.parent._startOffset;break;default:for(var P=o;P.parent;)P=P.parent;u=P._startOffset||0}this.options.relativeTo&&(u+=this.relativeToGetter(o));var k=_+u;if(null!=this.type){var I=null,O=function decodeValue(){if(null!=I)return I;var u=r.pos;return r.pos=k,I=y.type.decode(r,o),r.pos=u,I};return this.options.lazy?new p.PropertyDescriptor({get:O}):O()}return k},r.size=function size(r,o){var u=o;switch(this.options.type){case"local":case"immediate":break;case"parent":o=o.parent;break;default:for(;o.parent;)o=o.parent}var p=this.type;if(null==p){if(!(r instanceof _))throw new Error("Must be a VoidPointer");p=r.type,r=r.value}return r&&o&&(o.pointerSize+=p.size(r,u)),this.offsetType.size()},r.encode=function encode(r,o,u){var p,y=u;if(null!=o){switch(this.options.type){case"local":p=u.startOffset;break;case"immediate":p=r.pos+this.offsetType.size(o,y);break;case"parent":p=(u=u.parent).startOffset;break;default:for(p=0;u.parent;)u=u.parent}this.options.relativeTo&&(p+=this.relativeToGetter(y.val)),this.offsetType.encode(r,u.pointerOffset-p);var P=this.type;if(null==P){if(!(o instanceof _))throw new Error("Must be a VoidPointer");P=o.type,o=o.value}return u.pointers.push({type:P,val:o,parent:y}),u.pointerOffset+=P.size(o,y)}this.offsetType.encode(r,this.options.nullValue)},Pointer}(),_=function VoidPointer(r,o){this.type=r,this.value=o};o.Pointer=y,o.VoidPointer=_},9567:function(r,o,u){var p=u(46887);r.exports=p&&!Symbol.sham&&"symbol"==typeof Symbol.iterator},9649:function(r,o,u){"use strict";var p=u(95892).f,y=u(10819),_=u(15341),P=u(25567),k=u(2868),I=u(80383),O=u(97001),B=u(51334),D=u(15567),R=u(62148).fastKey,N=u(70172),U=N.set,W=N.getterFor;r.exports={getConstructor:function(r,o,u,O){var B=r((function(r,p){k(r,N),U(r,{type:o,index:y(null),first:void 0,last:void 0,size:0}),D||(r.size=0),null!=p&&I(p,r[O],{that:r,AS_ENTRIES:u})})),N=B.prototype,j=W(o),define=function(r,o,u){var p,y,_=j(r),P=getEntry(r,o);return P?P.value=u:(_.last=P={index:y=R(o,!0),key:o,value:u,previous:p=_.last,next:void 0,removed:!1},_.first||(_.first=P),p&&(p.next=P),D?_.size++:r.size++,"F"!==y&&(_.index[y]=P)),r},getEntry=function(r,o){var u,p=j(r),y=R(o);if("F"!==y)return p.index[y];for(u=p.first;u;u=u.next)if(u.key==o)return u};return _(N,{clear:function clear(){for(var r=j(this),o=r.index,u=r.first;u;)u.removed=!0,u.previous&&(u.previous=u.previous.next=void 0),delete o[u.index],u=u.next;r.first=r.last=void 0,D?r.size=0:this.size=0},delete:function(r){var o=this,u=j(o),p=getEntry(o,r);if(p){var y=p.next,_=p.previous;delete u.index[p.index],p.removed=!0,_&&(_.next=y),y&&(y.previous=_),u.first==p&&(u.first=y),u.last==p&&(u.last=_),D?u.size--:o.size--}return!!p},forEach:function forEach(r){for(var o,u=j(this),p=P(r,arguments.length>1?arguments[1]:void 0);o=o?o.next:u.first;)for(p(o.value,o.key,this);o&&o.removed;)o=o.previous},has:function has(r){return!!getEntry(this,r)}}),_(N,u?{get:function get(r){var o=getEntry(this,r);return o&&o.value},set:function set(r,o){return define(this,0===r?0:r,o)}}:{add:function add(r){return define(this,r=0===r?0:r,r)}}),D&&p(N,"size",{get:function(){return j(this).size}}),B},setStrong:function(r,o,u){var p=o+" Iterator",y=W(o),_=W(p);O(r,o,(function(r,o){U(this,{type:p,target:r,state:y(r),kind:o,last:void 0})}),(function(){for(var r=_(this),o=r.kind,u=r.last;u&&u.removed;)u=u.previous;return r.target&&(r.last=u=u?u.next:r.state.first)?"keys"==o?{value:u.key,done:!1}:"values"==o?{value:u.value,done:!1}:{value:[u.key,u.value],done:!1}:(r.target=void 0,{value:void 0,done:!0})}),u?"entries":"values",!u,!0),B(o)}}},9760:function(r,o,u){r.exports=Stream;var p=u(64785).EventEmitter;function Stream(){p.call(this)}u(89784)(Stream,p),Stream.Readable=u(88261),Stream.Writable=u(29781),Stream.Duplex=u(14903),Stream.Transform=u(48569),Stream.PassThrough=u(17723),Stream.finished=u(12167),Stream.pipeline=u(43765),Stream.Stream=Stream,Stream.prototype.pipe=function(r,o){var u=this;function ondata(o){r.writable&&!1===r.write(o)&&u.pause&&u.pause()}function ondrain(){u.readable&&u.resume&&u.resume()}u.on("data",ondata),r.on("drain",ondrain),r._isStdio||o&&!1===o.end||(u.on("end",onend),u.on("close",onclose));var y=!1;function onend(){y||(y=!0,r.end())}function onclose(){y||(y=!0,"function"==typeof r.destroy&&r.destroy())}function onerror(r){if(cleanup(),0===p.listenerCount(this,"error"))throw r}function cleanup(){u.removeListener("data",ondata),r.removeListener("drain",ondrain),u.removeListener("end",onend),u.removeListener("close",onclose),u.removeListener("error",onerror),r.removeListener("error",onerror),u.removeListener("end",cleanup),u.removeListener("close",cleanup),r.removeListener("close",cleanup)}return u.on("error",onerror),r.on("error",onerror),u.on("end",cleanup),u.on("close",cleanup),r.on("close",cleanup),r.emit("pipe",u),r}},9964:function(r){var o,u,p=r.exports={};function defaultSetTimout(){throw new Error("setTimeout has not been defined")}function defaultClearTimeout(){throw new Error("clearTimeout has not been defined")}function runTimeout(r){if(o===setTimeout)return setTimeout(r,0);if((o===defaultSetTimout||!o)&&setTimeout)return o=setTimeout,setTimeout(r,0);try{return o(r,0)}catch(u){try{return o.call(null,r,0)}catch(u){return o.call(this,r,0)}}}!function(){try{o="function"==typeof setTimeout?setTimeout:defaultSetTimout}catch(r){o=defaultSetTimout}try{u="function"==typeof clearTimeout?clearTimeout:defaultClearTimeout}catch(r){u=defaultClearTimeout}}();var y,_=[],P=!1,k=-1;function cleanUpNextTick(){P&&y&&(P=!1,y.length?_=y.concat(_):k=-1,_.length&&drainQueue())}function drainQueue(){if(!P){var r=runTimeout(cleanUpNextTick);P=!0;for(var o=_.length;o;){for(y=_,_=[];++k<o;)y&&y[k].run();k=-1,o=_.length}y=null,P=!1,function runClearTimeout(r){if(u===clearTimeout)return clearTimeout(r);if((u===defaultClearTimeout||!u)&&clearTimeout)return u=clearTimeout,clearTimeout(r);try{return u(r)}catch(o){try{return u.call(null,r)}catch(o){return u.call(this,r)}}}(r)}}function Item(r,o){this.fun=r,this.array=o}function noop(){}p.nextTick=function(r){var o=new Array(arguments.length-1);if(arguments.length>1)for(var u=1;u<arguments.length;u++)o[u-1]=arguments[u];_.push(new Item(r,o)),1!==_.length||P||runTimeout(drainQueue)},Item.prototype.run=function(){this.fun.apply(null,this.array)},p.title="browser",p.browser=!0,p.env={},p.argv=[],p.version="",p.versions={},p.on=noop,p.addListener=noop,p.once=noop,p.off=noop,p.removeListener=noop,p.removeAllListeners=noop,p.emit=noop,p.prependListener=noop,p.prependOnceListener=noop,p.listeners=function(r){return[]},p.binding=function(r){throw new Error("process.binding is not supported")},p.cwd=function(){return"/"},p.chdir=function(r){throw new Error("process.chdir is not supported")},p.umask=function(){return 0}},10078:function(r){"use strict";r.exports=Function.prototype.call},10196:function(r,o,u){var p=u(15567),y=u(95892),_=u(34984),P=u(98086),k=u(84675);r.exports=p?Object.defineProperties:function defineProperties(r,o){_(r);for(var u,p=P(o),I=k(o),O=I.length,B=0;O>B;)y.f(r,u=I[B++],p[u]);return r}},10447:function(r,o,u){var p=u(38347),y=u(94578),_=u(55480),P=p(Function.toString);y(_.inspectSource)||(_.inspectSource=function(r){return P(r)}),r.exports=_.inspectSource},10713:function(r){"use strict";r.exports=Number.isNaN||function isNaN(r){return r!=r}},10740:function(r,o,u){"use strict";var p,y=u(14598).Buffer;function _defineProperties(r,o){for(var u=0;u<o.length;u++){var p=o[u];p.enumerable=p.enumerable||!1,p.configurable=!0,"value"in p&&(p.writable=!0),Object.defineProperty(r,_toPropertyKey(p.key),p)}}function _createClass(r,o,u){return o&&_defineProperties(r.prototype,o),u&&_defineProperties(r,u),Object.defineProperty(r,"prototype",{writable:!1}),r}function _toPropertyKey(r){var o=function _toPrimitive(r,o){if("object"!=typeof r||!r)return r;var u=r[Symbol.toPrimitive];if(void 0!==u){var p=u.call(r,o||"default");if("object"!=typeof p)return p;throw new TypeError("@@toPrimitive must return a primitive value.")}return("string"===o?String:Number)(r)}(r,"string");return"symbol"==typeof o?o:o+""}function _inheritsLoose(r,o){r.prototype=Object.create(o.prototype),r.prototype.constructor=r,_setPrototypeOf(r,o)}function _setPrototypeOf(r,o){return _setPrototypeOf=Object.setPrototypeOf?Object.setPrototypeOf.bind():function(r,o){return r.__proto__=o,r},_setPrototypeOf(r,o)}function _createForOfIteratorHelperLoose(r,o){var u="undefined"!=typeof Symbol&&r[Symbol.iterator]||r["@@iterator"];if(u)return(u=u.call(r)).next.bind(u);if(Array.isArray(r)||(u=function _unsupportedIterableToArray(r,o){if(r){if("string"==typeof r)return _arrayLikeToArray(r,o);var u={}.toString.call(r).slice(8,-1);return"Object"===u&&r.constructor&&(u=r.constructor.name),"Map"===u||"Set"===u?Array.from(r):"Arguments"===u||/^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(u)?_arrayLikeToArray(r,o):void 0}}(r))||o&&r&&"number"==typeof r.length){u&&(r=u);var p=0;return function(){return p>=r.length?{done:!0}:{done:!1,value:r[p++]}}}throw new TypeError("Invalid attempt to iterate non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.")}function _arrayLikeToArray(r,o){(null==o||o>r.length)&&(o=r.length);for(var u=0,p=Array(o);u<o;u++)p[u]=r[u];return p}function _interopDefault(r){return r&&"object"==typeof r&&"default"in r?r.default:r}u(65292),u(73844),u(69330),u(5597),u(7585),u(39081),u(41584),u(11765),u(81755),u(94845),u(80055),u(20731),u(45337),u(23913),u(8953),u(24863),u(18828),u(7283),u(58549),u(18756),u(75174),u(57444),u(37309),u(14032),u(68067),u(61726),u(74516),u(57114),u(76014),u(65578),u(58281),u(6422),u(28264),u(72095),u(75626),u(47969),u(73663),u(29883),u(35471),u(21012),u(88997),u(97464),u(2857),u(94715),u(13624),u(91132),u(62514),u(24597),u(88042),u(4660),u(92451),u(44206),u(66288),u(13250),u(3858),u(84538),u(64793),u(74202),u(52529),u(42437),u(94712);var _=_interopDefault(u(62420)),P=u(18128),k=_interopDefault(u(45728)),I=_interopDefault(u(24766)),O=_interopDefault(u(59006)),B=_interopDefault(u(18645)),D=_interopDefault(u(41613)),R=_interopDefault(u(3483)),N=_interopDefault(u(34460)),U=u(48181),W={logErrors:!1},j=[];function _defineProperty(r,o,u){return o in r?Object.defineProperty(r,o,{value:u,enumerable:!0,configurable:!0,writable:!0}):r[o]=u,r}function _applyDecoratedDescriptor(r,o,u,p,y){var _={};return Object.keys(p).forEach((function(r){_[r]=p[r]})),_.enumerable=!!_.enumerable,_.configurable=!!_.configurable,("value"in _||_.initializer)&&(_.writable=!0),_=u.slice().reverse().reduce((function(u,p){return p(r,o,u)||u}),_),y&&void 0!==_.initializer&&(_.value=_.initializer?_.initializer.call(y):void 0,_.initializer=void 0),void 0===_.initializer&&(Object.defineProperty(r,o,_),_=null),_}function cache(r,o,u){if(u.get){var p=u.get;u.get=function(){var r=p.call(this);return Object.defineProperty(this,o,{value:r}),r}}else if("function"==typeof u.value){var y=u.value;return{get:function get(){var r=new Map;function memoized(){for(var o=arguments.length,u=new Array(o),p=0;p<o;p++)u[p]=arguments[p];var _=u.length>0?u[0]:"value";if(r.has(_))return r.get(_);var P=y.apply(this,u);return r.set(_,P),P}return Object.defineProperty(this,o,{value:memoized}),memoized}}}}W.registerFormat=function(r){j.push(r)},W.openSync=function(r,o){var u=U.readFileSync(r);return W.create(u,o)},W.open=function(r,o,u){"function"==typeof o&&(u=o,o=null),U.readFile(r,(function(r,p){if(r)return u(r);try{var y=W.create(p,o)}catch(r){return u(r)}return u(null,y)}))},W.create=function(r,o){for(var u=0;u<j.length;u++){var p=j[u];if(p.probe(r)){var y=new p(new _.DecodeStream(r));return o?y.getFont(o):y}}throw new Error("Unknown font format")},W.defaultLanguage="en",W.setDefaultLanguage=function(r){void 0===r&&(r="en"),W.defaultLanguage=r};var G=new _.Struct({firstCode:_.uint16,entryCount:_.uint16,idDelta:_.int16,idRangeOffset:_.uint16}),X=new _.Struct({startCharCode:_.uint32,endCharCode:_.uint32,glyphID:_.uint32}),Y=new _.Struct({startUnicodeValue:_.uint24,additionalCount:_.uint8}),K=new _.Struct({unicodeValue:_.uint24,glyphID:_.uint16}),J=new _.Array(Y,_.uint32),$=new _.Array(K,_.uint32),ee=new _.Struct({varSelector:_.uint24,defaultUVS:new _.Pointer(_.uint32,J,{type:"parent"}),nonDefaultUVS:new _.Pointer(_.uint32,$,{type:"parent"})}),te=new _.VersionedStruct(_.uint16,{0:{length:_.uint16,language:_.uint16,codeMap:new _.LazyArray(_.uint8,256)},2:{length:_.uint16,language:_.uint16,subHeaderKeys:new _.Array(_.uint16,256),subHeaderCount:function subHeaderCount(r){return Math.max.apply(Math,r.subHeaderKeys)},subHeaders:new _.LazyArray(G,"subHeaderCount"),glyphIndexArray:new _.LazyArray(_.uint16,"subHeaderCount")},4:{length:_.uint16,language:_.uint16,segCountX2:_.uint16,segCount:function segCount(r){return r.segCountX2>>1},searchRange:_.uint16,entrySelector:_.uint16,rangeShift:_.uint16,endCode:new _.LazyArray(_.uint16,"segCount"),reservedPad:new _.Reserved(_.uint16),startCode:new _.LazyArray(_.uint16,"segCount"),idDelta:new _.LazyArray(_.int16,"segCount"),idRangeOffset:new _.LazyArray(_.uint16,"segCount"),glyphIndexArray:new _.LazyArray(_.uint16,(function(r){return(r.length-r._currentOffset)/2}))},6:{length:_.uint16,language:_.uint16,firstCode:_.uint16,entryCount:_.uint16,glyphIndices:new _.LazyArray(_.uint16,"entryCount")},8:{reserved:new _.Reserved(_.uint16),length:_.uint32,language:_.uint16,is32:new _.LazyArray(_.uint8,8192),nGroups:_.uint32,groups:new _.LazyArray(X,"nGroups")},10:{reserved:new _.Reserved(_.uint16),length:_.uint32,language:_.uint32,firstCode:_.uint32,entryCount:_.uint32,glyphIndices:new _.LazyArray(_.uint16,"numChars")},12:{reserved:new _.Reserved(_.uint16),length:_.uint32,language:_.uint32,nGroups:_.uint32,groups:new _.LazyArray(X,"nGroups")},13:{reserved:new _.Reserved(_.uint16),length:_.uint32,language:_.uint32,nGroups:_.uint32,groups:new _.LazyArray(X,"nGroups")},14:{length:_.uint32,numRecords:_.uint32,varSelectors:new _.LazyArray(ee,"numRecords")}}),re=new _.Struct({platformID:_.uint16,encodingID:_.uint16,table:new _.Pointer(_.uint32,te,{type:"parent",lazy:!0})}),ne=new _.Struct({version:_.uint16,numSubtables:_.uint16,tables:new _.Array(re,"numSubtables")}),ie=new _.Struct({version:_.int32,revision:_.int32,checkSumAdjustment:_.uint32,magicNumber:_.uint32,flags:_.uint16,unitsPerEm:_.uint16,created:new _.Array(_.int32,2),modified:new _.Array(_.int32,2),xMin:_.int16,yMin:_.int16,xMax:_.int16,yMax:_.int16,macStyle:new _.Bitfield(_.uint16,["bold","italic","underline","outline","shadow","condensed","extended"]),lowestRecPPEM:_.uint16,fontDirectionHint:_.int16,indexToLocFormat:_.int16,glyphDataFormat:_.int16}),oe=new _.Struct({version:_.int32,ascent:_.int16,descent:_.int16,lineGap:_.int16,advanceWidthMax:_.uint16,minLeftSideBearing:_.int16,minRightSideBearing:_.int16,xMaxExtent:_.int16,caretSlopeRise:_.int16,caretSlopeRun:_.int16,caretOffset:_.int16,reserved:new _.Reserved(_.int16,4),metricDataFormat:_.int16,numberOfMetrics:_.uint16}),ae=new _.Struct({advance:_.uint16,bearing:_.int16}),se=new _.Struct({metrics:new _.LazyArray(ae,(function(r){return r.parent.hhea.numberOfMetrics})),bearings:new _.LazyArray(_.int16,(function(r){return r.parent.maxp.numGlyphs-r.parent.hhea.numberOfMetrics}))}),ce=new _.Struct({version:_.int32,numGlyphs:_.uint16,maxPoints:_.uint16,maxContours:_.uint16,maxComponentPoints:_.uint16,maxComponentContours:_.uint16,maxZones:_.uint16,maxTwilightPoints:_.uint16,maxStorage:_.uint16,maxFunctionDefs:_.uint16,maxInstructionDefs:_.uint16,maxStackElements:_.uint16,maxSizeOfInstructions:_.uint16,maxComponentElements:_.uint16,maxComponentDepth:_.uint16});function getEncoding(r,o,u){return void 0===u&&(u=0),1===r&&ue[u]?ue[u]:le[r][o]}var le=[["utf16be","utf16be","utf16be","utf16be","utf16be","utf16be"],["macroman","shift-jis","big5","euc-kr","iso-8859-6","iso-8859-8","macgreek","maccyrillic","symbol","Devanagari","Gurmukhi","Gujarati","Oriya","Bengali","Tamil","Telugu","Kannada","Malayalam","Sinhalese","Burmese","Khmer","macthai","Laotian","Georgian","Armenian","gb-2312-80","Tibetan","Mongolian","Geez","maccenteuro","Vietnamese","Sindhi"],["ascii"],["symbol","utf16be","shift-jis","gb18030","big5","wansung","johab",null,null,null,"utf16be"]],ue={15:"maciceland",17:"macturkish",18:"maccroatian",24:"maccenteuro",25:"maccenteuro",26:"maccenteuro",27:"maccenteuro",28:"maccenteuro",30:"maciceland",37:"macromania",38:"maccenteuro",39:"maccenteuro",40:"maccenteuro",143:"macinuit",146:"macgaelic"},fe=[[],{0:"en",30:"fo",60:"ks",90:"rw",1:"fr",31:"fa",61:"ku",91:"rn",2:"de",32:"ru",62:"sd",92:"ny",3:"it",33:"zh",63:"bo",93:"mg",4:"nl",34:"nl-BE",64:"ne",94:"eo",5:"sv",35:"ga",65:"sa",128:"cy",6:"es",36:"sq",66:"mr",129:"eu",7:"da",37:"ro",67:"bn",130:"ca",8:"pt",38:"cz",68:"as",131:"la",9:"no",39:"sk",69:"gu",132:"qu",10:"he",40:"si",70:"pa",133:"gn",11:"ja",41:"yi",71:"or",134:"ay",12:"ar",42:"sr",72:"ml",135:"tt",13:"fi",43:"mk",73:"kn",136:"ug",14:"el",44:"bg",74:"ta",137:"dz",15:"is",45:"uk",75:"te",138:"jv",16:"mt",46:"be",76:"si",139:"su",17:"tr",47:"uz",77:"my",140:"gl",18:"hr",48:"kk",78:"km",141:"af",19:"zh-Hant",49:"az-Cyrl",79:"lo",142:"br",20:"ur",50:"az-Arab",80:"vi",143:"iu",21:"hi",51:"hy",81:"id",144:"gd",22:"th",52:"ka",82:"tl",145:"gv",23:"ko",53:"mo",83:"ms",146:"ga",24:"lt",54:"ky",84:"ms-Arab",147:"to",25:"pl",55:"tg",85:"am",148:"el-polyton",26:"hu",56:"tk",86:"ti",149:"kl",27:"es",57:"mn-CN",87:"om",150:"az",28:"lv",58:"mn",88:"so",151:"nn",29:"se",59:"ps",89:"sw"},[],{1078:"af",16393:"en-IN",1159:"rw",1074:"tn",1052:"sq",6153:"en-IE",1089:"sw",1115:"si",1156:"gsw",8201:"en-JM",1111:"kok",1051:"sk",1118:"am",17417:"en-MY",1042:"ko",1060:"sl",5121:"ar-DZ",5129:"en-NZ",1088:"ky",11274:"es-AR",15361:"ar-BH",13321:"en-PH",1108:"lo",16394:"es-BO",3073:"ar",18441:"en-SG",1062:"lv",13322:"es-CL",2049:"ar-IQ",7177:"en-ZA",1063:"lt",9226:"es-CO",11265:"ar-JO",11273:"en-TT",2094:"dsb",5130:"es-CR",13313:"ar-KW",2057:"en-GB",1134:"lb",7178:"es-DO",12289:"ar-LB",1033:"en",1071:"mk",12298:"es-EC",4097:"ar-LY",12297:"en-ZW",2110:"ms-BN",17418:"es-SV",6145:"ary",1061:"et",1086:"ms",4106:"es-GT",8193:"ar-OM",1080:"fo",1100:"ml",18442:"es-HN",16385:"ar-QA",1124:"fil",1082:"mt",2058:"es-MX",1025:"ar-SA",1035:"fi",1153:"mi",19466:"es-NI",10241:"ar-SY",2060:"fr-BE",1146:"arn",6154:"es-PA",7169:"aeb",3084:"fr-CA",1102:"mr",15370:"es-PY",14337:"ar-AE",1036:"fr",1148:"moh",10250:"es-PE",9217:"ar-YE",5132:"fr-LU",1104:"mn",20490:"es-PR",1067:"hy",6156:"fr-MC",2128:"mn-CN",3082:"es",1101:"as",4108:"fr-CH",1121:"ne",1034:"es",2092:"az-Cyrl",1122:"fy",1044:"nb",21514:"es-US",1068:"az",1110:"gl",2068:"nn",14346:"es-UY",1133:"ba",1079:"ka",1154:"oc",8202:"es-VE",1069:"eu",3079:"de-AT",1096:"or",2077:"sv-FI",1059:"be",1031:"de",1123:"ps",1053:"sv",2117:"bn",5127:"de-LI",1045:"pl",1114:"syr",1093:"bn-IN",4103:"de-LU",1046:"pt",1064:"tg",8218:"bs-Cyrl",2055:"de-CH",2070:"pt-PT",2143:"tzm",5146:"bs",1032:"el",1094:"pa",1097:"ta",1150:"br",1135:"kl",1131:"qu-BO",1092:"tt",1026:"bg",1095:"gu",2155:"qu-EC",1098:"te",1027:"ca",1128:"ha",3179:"qu",1054:"th",3076:"zh-HK",1037:"he",1048:"ro",1105:"bo",5124:"zh-MO",1081:"hi",1047:"rm",1055:"tr",2052:"zh",1038:"hu",1049:"ru",1090:"tk",4100:"zh-SG",1039:"is",9275:"smn",1152:"ug",1028:"zh-TW",1136:"ig",4155:"smj-NO",1058:"uk",1155:"co",1057:"id",5179:"smj",1070:"hsb",1050:"hr",1117:"iu",3131:"se-FI",1056:"ur",4122:"hr-BA",2141:"iu-Latn",1083:"se",2115:"uz-Cyrl",1029:"cs",2108:"ga",2107:"se-SE",1091:"uz",1030:"da",1076:"xh",8251:"sms",1066:"vi",1164:"prs",1077:"zu",6203:"sma-NO",1106:"cy",1125:"dv",1040:"it",7227:"sms",1160:"wo",2067:"nl-BE",2064:"it-CH",1103:"sa",1157:"sah",1043:"nl",1041:"ja",7194:"sr-Cyrl-BA",1144:"ii",3081:"en-AU",1099:"kn",3098:"sr",1130:"yo",10249:"en-BZ",1087:"kk",6170:"sr-Latn-BA",4105:"en-CA",1107:"km",2074:"sr-Latn",9225:"en-029",1158:"quc",1132:"nso"}],he=new _.Struct({platformID:_.uint16,encodingID:_.uint16,languageID:_.uint16,nameID:_.uint16,length:_.uint16,string:new _.Pointer(_.uint16,new _.String("length",(function(r){return getEncoding(r.platformID,r.encodingID,r.languageID)})),{type:"parent",relativeTo:function relativeTo(r){return r.parent.stringOffset},allowNull:!1})}),de=new _.Struct({length:_.uint16,tag:new _.Pointer(_.uint16,new _.String("length","utf16be"),{type:"parent",relativeTo:function relativeTo(r){return r.stringOffset}})}),pe=new _.VersionedStruct(_.uint16,{0:{count:_.uint16,stringOffset:_.uint16,records:new _.Array(he,"count")},1:{count:_.uint16,stringOffset:_.uint16,records:new _.Array(he,"count"),langTagCount:_.uint16,langTags:new _.Array(de,"langTagCount")}}),ge=["copyright","fontFamily","fontSubfamily","uniqueSubfamily","fullName","version","postscriptName","trademark","manufacturer","designer","description","vendorURL","designerURL","license","licenseURL",null,"preferredFamily","preferredSubfamily","compatibleFull","sampleText","postscriptCIDFontName","wwsFamilyName","wwsSubfamilyName"];pe.process=function(r){for(var o,u={},p=_createForOfIteratorHelperLoose(this.records);!(o=p()).done;){var y=o.value,_=fe[y.platformID][y.languageID];null==_&&null!=this.langTags&&y.languageID>=32768&&(_=this.langTags[y.languageID-32768].tag),null==_&&(_=y.platformID+"-"+y.languageID);var P=y.nameID>=256?"fontFeatures":ge[y.nameID]||y.nameID;null==u[P]&&(u[P]={});var k=u[P];y.nameID>=256&&(k=k[y.nameID]||(k[y.nameID]={})),"string"!=typeof y.string&&"string"==typeof k[_]||(k[_]=y.string)}this.records=u},pe.preEncode=function(){if(!Array.isArray(this.records)){this.version=0;var r=[];for(var o in this.records){var u=this.records[o];"fontFeatures"!==o&&(r.push({platformID:3,encodingID:1,languageID:1033,nameID:ge.indexOf(o),length:y.byteLength(u.en,"utf16le"),string:u.en}),"postscriptName"===o&&r.push({platformID:1,encodingID:0,languageID:0,nameID:ge.indexOf(o),length:u.en.length,string:u.en}))}this.records=r,this.count=r.length,this.stringOffset=pe.size(this,null,!1)}};var ye=new _.VersionedStruct(_.uint16,{header:{xAvgCharWidth:_.int16,usWeightClass:_.uint16,usWidthClass:_.uint16,fsType:new _.Bitfield(_.uint16,[null,"noEmbedding","viewOnly","editable",null,null,null,null,"noSubsetting","bitmapOnly"]),ySubscriptXSize:_.int16,ySubscriptYSize:_.int16,ySubscriptXOffset:_.int16,ySubscriptYOffset:_.int16,ySuperscriptXSize:_.int16,ySuperscriptYSize:_.int16,ySuperscriptXOffset:_.int16,ySuperscriptYOffset:_.int16,yStrikeoutSize:_.int16,yStrikeoutPosition:_.int16,sFamilyClass:_.int16,panose:new _.Array(_.uint8,10),ulCharRange:new _.Array(_.uint32,4),vendorID:new _.String(4),fsSelection:new _.Bitfield(_.uint16,["italic","underscore","negative","outlined","strikeout","bold","regular","useTypoMetrics","wws","oblique"]),usFirstCharIndex:_.uint16,usLastCharIndex:_.uint16},0:{},1:{typoAscender:_.int16,typoDescender:_.int16,typoLineGap:_.int16,winAscent:_.uint16,winDescent:_.uint16,codePageRange:new _.Array(_.uint32,2)},2:{typoAscender:_.int16,typoDescender:_.int16,typoLineGap:_.int16,winAscent:_.uint16,winDescent:_.uint16,codePageRange:new _.Array(_.uint32,2),xHeight:_.int16,capHeight:_.int16,defaultChar:_.uint16,breakChar:_.uint16,maxContent:_.uint16},5:{typoAscender:_.int16,typoDescender:_.int16,typoLineGap:_.int16,winAscent:_.uint16,winDescent:_.uint16,codePageRange:new _.Array(_.uint32,2),xHeight:_.int16,capHeight:_.int16,defaultChar:_.uint16,breakChar:_.uint16,maxContent:_.uint16,usLowerOpticalPointSize:_.uint16,usUpperOpticalPointSize:_.uint16}}),ve=ye.versions;ve[3]=ve[4]=ve[2];var me=new _.VersionedStruct(_.fixed32,{header:{italicAngle:_.fixed32,underlinePosition:_.int16,underlineThickness:_.int16,isFixedPitch:_.uint32,minMemType42:_.uint32,maxMemType42:_.uint32,minMemType1:_.uint32,maxMemType1:_.uint32},1:{},2:{numberOfGlyphs:_.uint16,glyphNameIndex:new _.Array(_.uint16,"numberOfGlyphs"),names:new _.Array(new _.String(_.uint8))},2.5:{numberOfGlyphs:_.uint16,offsets:new _.Array(_.uint8,"numberOfGlyphs")},3:{},4:{map:new _.Array(_.uint32,(function(r){return r.parent.maxp.numGlyphs}))}}),be=new _.Struct({controlValues:new _.Array(_.int16)}),we=new _.Struct({instructions:new _.Array(_.uint8)}),_e=new _.VersionedStruct("head.indexToLocFormat",{0:{offsets:new _.Array(_.uint16)},1:{offsets:new _.Array(_.uint32)}});_e.process=function(){if(0===this.version)for(var r=0;r<this.offsets.length;r++)this.offsets[r]<<=1},_e.preEncode=function(){if(0===this.version)for(var r=0;r<this.offsets.length;r++)this.offsets[r]>>>=1};var Se=new _.Struct({controlValueProgram:new _.Array(_.uint8)}),xe=new _.Array(new _.Buffer),Ae=function(){function CFFIndex(r){this.type=r}var r=CFFIndex.prototype;return r.getCFFVersion=function getCFFVersion(r){for(;r&&!r.hdrSize;)r=r.parent;return r?r.version:-1},r.decode=function decode(r,o){var u=this.getCFFVersion(o)>=2?r.readUInt32BE():r.readUInt16BE();if(0===u)return[];var p,y=r.readUInt8();if(1===y)p=_.uint8;else if(2===y)p=_.uint16;else if(3===y)p=_.uint24;else{if(4!==y)throw new Error("Bad offset size in CFFIndex: ".concat(y," ").concat(r.pos));p=_.uint32}for(var P=[],k=r.pos+(u+1)*y-1,I=p.decode(r),O=0;O<u;O++){var B=p.decode(r);if(null!=this.type){var D=r.pos;r.pos=k+I,o.length=B-I,P.push(this.type.decode(r,o)),r.pos=D}else P.push({offset:k+I,length:B-I});I=B}return r.pos=k+I,P},r.size=function size(r,o){var size=2;if(0===r.length)return size;for(var u,p=this.type||new _.Buffer,y=1,P=0;P<r.length;P++){var k=r[P];y+=p.size(k,o)}if(y<=255)u=_.uint8;else if(y<=65535)u=_.uint16;else if(y<=16777215)u=_.uint24;else{if(!(y<=4294967295))throw new Error("Bad offset in CFFIndex");u=_.uint32}return size+=1+u.size()*(r.length+1),size+=y-1,size},r.encode=function encode(r,o,u){if(r.writeUInt16BE(o.length),0!==o.length){for(var p,y,P=this.type||new _.Buffer,k=[],I=1,O=_createForOfIteratorHelperLoose(o);!(p=O()).done;){var B=p.value,D=P.size(B,u);k.push(D),I+=D}if(I<=255)y=_.uint8;else if(I<=65535)y=_.uint16;else if(I<=16777215)y=_.uint24;else{if(!(I<=4294967295))throw new Error("Bad offset in CFFIndex");y=_.uint32}r.writeUInt8(y.size()),I=1,y.encode(r,I);for(var R=0,N=k;R<N.length;R++){I+=N[R],y.encode(r,I)}for(var U,W=_createForOfIteratorHelperLoose(o);!(U=W()).done;){var j=U.value;P.encode(r,j,u)}}},CFFIndex}(),Ce=["0","1","2","3","4","5","6","7","8","9",".","E","E-",null,"-"],Pe={".":10,E:11,"E-":12,"-":14},ke=function(){function CFFOperand(){}return CFFOperand.decode=function decode(r,o){if(32<=o&&o<=246)return o-139;if(247<=o&&o<=250)return 256*(o-247)+r.readUInt8()+108;if(251<=o&&o<=254)return 256*-(o-251)-r.readUInt8()-108;if(28===o)return r.readInt16BE();if(29===o)return r.readInt32BE();if(30===o){for(var u="";;){var p=r.readUInt8(),y=p>>4;if(15===y)break;u+=Ce[y];var _=15&p;if(15===_)break;u+=Ce[_]}return parseFloat(u)}return null},CFFOperand.size=function size(r){if(r.forceLarge&&(r=32768),(0|r)!==r){var o=""+r;return 1+Math.ceil((o.length+1)/2)}return-107<=r&&r<=107?1:108<=r&&r<=1131||-1131<=r&&r<=-108?2:-32768<=r&&r<=32767?3:5},CFFOperand.encode=function encode(r,o){var u=Number(o);if(o.forceLarge)return r.writeUInt8(29),r.writeInt32BE(u);if((0|u)===u)return-107<=u&&u<=107?r.writeUInt8(u+139):108<=u&&u<=1131?(u-=108,r.writeUInt8(247+(u>>8)),r.writeUInt8(255&u)):-1131<=u&&u<=-108?(u=-u-108,r.writeUInt8(251+(u>>8)),r.writeUInt8(255&u)):-32768<=u&&u<=32767?(r.writeUInt8(28),r.writeInt16BE(u)):(r.writeUInt8(29),r.writeInt32BE(u));r.writeUInt8(30);for(var p=""+u,y=0;y<p.length;y+=2){var _=p[y],P=Pe[_]||+_;if(y===p.length-1)var k=15;else{var I=p[y+1];k=Pe[I]||+I}r.writeUInt8(P<<4|15&k)}return 15!==k?r.writeUInt8(240):void 0},CFFOperand}(),Ee=function(){function CFFDict(r){void 0===r&&(r=[]),this.ops=r,this.fields={};for(var o,u=_createForOfIteratorHelperLoose(r);!(o=u()).done;){var p=o.value,y=Array.isArray(p[0])?p[0][0]<<8|p[0][1]:p[0];this.fields[y]=p}}var r=CFFDict.prototype;return r.decodeOperands=function decodeOperands(r,o,u,p){var y=this;if(Array.isArray(r))return p.map((function(p,_){return y.decodeOperands(r[_],o,u,[p])}));if(null!=r.decode)return r.decode(o,u,p);switch(r){case"number":case"offset":case"sid":return p[0];case"boolean":return!!p[0];default:return p}},r.encodeOperands=function encodeOperands(r,o,u,p){var y=this;return Array.isArray(r)?p.map((function(p,_){return y.encodeOperands(r[_],o,u,p)[0]})):null!=r.encode?r.encode(o,p,u):"number"==typeof p?[p]:"boolean"==typeof p?[+p]:Array.isArray(p)?p:[p]},r.decode=function decode(r,o){var u=r.pos+o.length,p={},y=[];for(var _ in Object.defineProperties(p,{parent:{value:o},_startOffset:{value:r.pos}}),this.fields){var k=this.fields[_];p[k[1]]=k[3]}for(;r.pos<u;){var I=r.readUInt8();if(I<28){12===I&&(I=I<<8|r.readUInt8());var O=this.fields[I];if(!O)throw new Error("Unknown operator ".concat(I));var B=this.decodeOperands(O[2],r,p,y);null!=B&&(B instanceof P.PropertyDescriptor?Object.defineProperty(p,O[1],B):p[O[1]]=B),y=[]}else y.push(ke.decode(r,I))}return p},r.size=function size(r,o,u){void 0===u&&(u=!0);var p={parent:o,val:r,pointerSize:0,startOffset:o.startOffset||0},y=0;for(var _ in this.fields){var P=this.fields[_],I=r[P[1]];if(null!=I&&!k(I,P[3])){for(var O,B=_createForOfIteratorHelperLoose(this.encodeOperands(P[2],null,p,I));!(O=B()).done;){var D=O.value;y+=ke.size(D)}y+=(Array.isArray(P[0])?P[0]:[P[0]]).length}}return u&&(y+=p.pointerSize),y},r.encode=function encode(r,o,u){var p={pointers:[],startOffset:r.pos,parent:u,val:o,pointerSize:0};p.pointerOffset=r.pos+this.size(o,p,!1);for(var y,_=_createForOfIteratorHelperLoose(this.ops);!(y=_()).done;){var P=y.value,I=o[P[1]];if(null!=I&&!k(I,P[3])){for(var O,B=_createForOfIteratorHelperLoose(this.encodeOperands(P[2],r,p,I));!(O=B()).done;){var D=O.value;ke.encode(r,D)}for(var R,N=_createForOfIteratorHelperLoose(Array.isArray(P[0])?P[0]:[P[0]]);!(R=N()).done;){var U=R.value;r.writeUInt8(U)}}}for(var W=0;W<p.pointers.length;){var j=p.pointers[W++];j.type.encode(r,j.val,j.parent)}},CFFDict}(),Te=function(r){function CFFPointer(o,u){return void 0===u&&(u={}),null==u.type&&(u.type="global"),r.call(this,null,o,u)||this}_inheritsLoose(CFFPointer,r);var o=CFFPointer.prototype;return o.decode=function decode(o,u,p){return this.offsetType={decode:function decode(){return p[0]}},r.prototype.decode.call(this,o,u,p)},o.encode=function encode(o,u,p){if(!o)return this.offsetType={size:function size(){return 0}},this.size(u,p),[new Ie(0)];var y=null;return this.offsetType={encode:function encode(r,o){return y=o}},r.prototype.encode.call(this,o,u,p),[new Ie(y)]},CFFPointer}(_.Pointer),Ie=function(){function Ptr(r){this.val=r,this.forceLarge=!0}return Ptr.prototype.valueOf=function valueOf(){return this.val},Ptr}(),Oe=new Ee([[6,"BlueValues","delta",null],[7,"OtherBlues","delta",null],[8,"FamilyBlues","delta",null],[9,"FamilyOtherBlues","delta",null],[[12,9],"BlueScale","number",.039625],[[12,10],"BlueShift","number",7],[[12,11],"BlueFuzz","number",1],[10,"StdHW","number",null],[11,"StdVW","number",null],[[12,12],"StemSnapH","delta",null],[[12,13],"StemSnapV","delta",null],[[12,14],"ForceBold","boolean",!1],[[12,17],"LanguageGroup","number",0],[[12,18],"ExpansionFactor","number",.06],[[12,19],"initialRandomSeed","number",0],[20,"defaultWidthX","number",0],[21,"nominalWidthX","number",0],[22,"vsindex","number",0],[23,"blend",function(){function CFFBlendOp(){}return CFFBlendOp.decode=function decode(r,o,u){for(var p=u.pop();u.length>p;)u.pop()},CFFBlendOp}(),null],[19,"Subrs",new Te(new Ae,{type:"local"}),null]]),Be=[".notdef","space","exclam","quotedbl","numbersign","dollar","percent","ampersand","quoteright","parenleft","parenright","asterisk","plus","comma","hyphen","period","slash","zero","one","two","three","four","five","six","seven","eight","nine","colon","semicolon","less","equal","greater","question","at","A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","W","X","Y","Z","bracketleft","backslash","bracketright","asciicircum","underscore","quoteleft","a","b","c","d","e","f","g","h","i","j","k","l","m","n","o","p","q","r","s","t","u","v","w","x","y","z","braceleft","bar","braceright","asciitilde","exclamdown","cent","sterling","fraction","yen","florin","section","currency","quotesingle","quotedblleft","guillemotleft","guilsinglleft","guilsinglright","fi","fl","endash","dagger","daggerdbl","periodcentered","paragraph","bullet","quotesinglbase","quotedblbase","quotedblright","guillemotright","ellipsis","perthousand","questiondown","grave","acute","circumflex","tilde","macron","breve","dotaccent","dieresis","ring","cedilla","hungarumlaut","ogonek","caron","emdash","AE","ordfeminine","Lslash","Oslash","OE","ordmasculine","ae","dotlessi","lslash","oslash","oe","germandbls","onesuperior","logicalnot","mu","trademark","Eth","onehalf","plusminus","Thorn","onequarter","divide","brokenbar","degree","thorn","threequarters","twosuperior","registered","minus","eth","multiply","threesuperior","copyright","Aacute","Acircumflex","Adieresis","Agrave","Aring","Atilde","Ccedilla","Eacute","Ecircumflex","Edieresis","Egrave","Iacute","Icircumflex","Idieresis","Igrave","Ntilde","Oacute","Ocircumflex","Odieresis","Ograve","Otilde","Scaron","Uacute","Ucircumflex","Udieresis","Ugrave","Yacute","Ydieresis","Zcaron","aacute","acircumflex","adieresis","agrave","aring","atilde","ccedilla","eacute","ecircumflex","edieresis","egrave","iacute","icircumflex","idieresis","igrave","ntilde","oacute","ocircumflex","odieresis","ograve","otilde","scaron","uacute","ucircumflex","udieresis","ugrave","yacute","ydieresis","zcaron","exclamsmall","Hungarumlautsmall","dollaroldstyle","dollarsuperior","ampersandsmall","Acutesmall","parenleftsuperior","parenrightsuperior","twodotenleader","onedotenleader","zerooldstyle","oneoldstyle","twooldstyle","threeoldstyle","fouroldstyle","fiveoldstyle","sixoldstyle","sevenoldstyle","eightoldstyle","nineoldstyle","commasuperior","threequartersemdash","periodsuperior","questionsmall","asuperior","bsuperior","centsuperior","dsuperior","esuperior","isuperior","lsuperior","msuperior","nsuperior","osuperior","rsuperior","ssuperior","tsuperior","ff","ffi","ffl","parenleftinferior","parenrightinferior","Circumflexsmall","hyphensuperior","Gravesmall","Asmall","Bsmall","Csmall","Dsmall","Esmall","Fsmall","Gsmall","Hsmall","Ismall","Jsmall","Ksmall","Lsmall","Msmall","Nsmall","Osmall","Psmall","Qsmall","Rsmall","Ssmall","Tsmall","Usmall","Vsmall","Wsmall","Xsmall","Ysmall","Zsmall","colonmonetary","onefitted","rupiah","Tildesmall","exclamdownsmall","centoldstyle","Lslashsmall","Scaronsmall","Zcaronsmall","Dieresissmall","Brevesmall","Caronsmall","Dotaccentsmall","Macronsmall","figuredash","hypheninferior","Ogoneksmall","Ringsmall","Cedillasmall","questiondownsmall","oneeighth","threeeighths","fiveeighths","seveneighths","onethird","twothirds","zerosuperior","foursuperior","fivesuperior","sixsuperior","sevensuperior","eightsuperior","ninesuperior","zeroinferior","oneinferior","twoinferior","threeinferior","fourinferior","fiveinferior","sixinferior","seveninferior","eightinferior","nineinferior","centinferior","dollarinferior","periodinferior","commainferior","Agravesmall","Aacutesmall","Acircumflexsmall","Atildesmall","Adieresissmall","Aringsmall","AEsmall","Ccedillasmall","Egravesmall","Eacutesmall","Ecircumflexsmall","Edieresissmall","Igravesmall","Iacutesmall","Icircumflexsmall","Idieresissmall","Ethsmall","Ntildesmall","Ogravesmall","Oacutesmall","Ocircumflexsmall","Otildesmall","Odieresissmall","OEsmall","Oslashsmall","Ugravesmall","Uacutesmall","Ucircumflexsmall","Udieresissmall","Yacutesmall","Thornsmall","Ydieresissmall","001.000","001.001","001.002","001.003","Black","Bold","Book","Light","Medium","Regular","Roman","Semibold"],Fe=["","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","space","exclam","quotedbl","numbersign","dollar","percent","ampersand","quoteright","parenleft","parenright","asterisk","plus","comma","hyphen","period","slash","zero","one","two","three","four","five","six","seven","eight","nine","colon","semicolon","less","equal","greater","question","at","A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","W","X","Y","Z","bracketleft","backslash","bracketright","asciicircum","underscore","quoteleft","a","b","c","d","e","f","g","h","i","j","k","l","m","n","o","p","q","r","s","t","u","v","w","x","y","z","braceleft","bar","braceright","asciitilde","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","exclamdown","cent","sterling","fraction","yen","florin","section","currency","quotesingle","quotedblleft","guillemotleft","guilsinglleft","guilsinglright","fi","fl","","endash","dagger","daggerdbl","periodcentered","","paragraph","bullet","quotesinglbase","quotedblbase","quotedblright","guillemotright","ellipsis","perthousand","","questiondown","","grave","acute","circumflex","tilde","macron","breve","dotaccent","dieresis","","ring","cedilla","","hungarumlaut","ogonek","caron","emdash","","","","","","","","","","","","","","","","","AE","","ordfeminine","","","","","Lslash","Oslash","OE","ordmasculine","","","","","","ae","","","","dotlessi","","","lslash","oslash","oe","germandbls"],Le=[".notdef","space","exclam","quotedbl","numbersign","dollar","percent","ampersand","quoteright","parenleft","parenright","asterisk","plus","comma","hyphen","period","slash","zero","one","two","three","four","five","six","seven","eight","nine","colon","semicolon","less","equal","greater","question","at","A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","W","X","Y","Z","bracketleft","backslash","bracketright","asciicircum","underscore","quoteleft","a","b","c","d","e","f","g","h","i","j","k","l","m","n","o","p","q","r","s","t","u","v","w","x","y","z","braceleft","bar","braceright","asciitilde","exclamdown","cent","sterling","fraction","yen","florin","section","currency","quotesingle","quotedblleft","guillemotleft","guilsinglleft","guilsinglright","fi","fl","endash","dagger","daggerdbl","periodcentered","paragraph","bullet","quotesinglbase","quotedblbase","quotedblright","guillemotright","ellipsis","perthousand","questiondown","grave","acute","circumflex","tilde","macron","breve","dotaccent","dieresis","ring","cedilla","hungarumlaut","ogonek","caron","emdash","AE","ordfeminine","Lslash","Oslash","OE","ordmasculine","ae","dotlessi","lslash","oslash","oe","germandbls","onesuperior","logicalnot","mu","trademark","Eth","onehalf","plusminus","Thorn","onequarter","divide","brokenbar","degree","thorn","threequarters","twosuperior","registered","minus","eth","multiply","threesuperior","copyright","Aacute","Acircumflex","Adieresis","Agrave","Aring","Atilde","Ccedilla","Eacute","Ecircumflex","Edieresis","Egrave","Iacute","Icircumflex","Idieresis","Igrave","Ntilde","Oacute","Ocircumflex","Odieresis","Ograve","Otilde","Scaron","Uacute","Ucircumflex","Udieresis","Ugrave","Yacute","Ydieresis","Zcaron","aacute","acircumflex","adieresis","agrave","aring","atilde","ccedilla","eacute","ecircumflex","edieresis","egrave","iacute","icircumflex","idieresis","igrave","ntilde","oacute","ocircumflex","odieresis","ograve","otilde","scaron","uacute","ucircumflex","udieresis","ugrave","yacute","ydieresis","zcaron"],De=new _.Struct({reserved:new _.Reserved(_.uint16),reqFeatureIndex:_.uint16,featureCount:_.uint16,featureIndexes:new _.Array(_.uint16,"featureCount")}),Re=new _.Struct({tag:new _.String(4),langSys:new _.Pointer(_.uint16,De,{type:"parent"})}),Me=new _.Struct({defaultLangSys:new _.Pointer(_.uint16,De),count:_.uint16,langSysRecords:new _.Array(Re,"count")}),Ne=new _.Struct({tag:new _.String(4),script:new _.Pointer(_.uint16,Me,{type:"parent"})}),Ue=new _.Array(Ne,_.uint16),We=new _.Struct({featureParams:_.uint16,lookupCount:_.uint16,lookupListIndexes:new _.Array(_.uint16,"lookupCount")}),ze=new _.Struct({tag:new _.String(4),feature:new _.Pointer(_.uint16,We,{type:"parent"})}),je=new _.Array(ze,_.uint16),Ge=new _.Struct({markAttachmentType:_.uint8,flags:new _.Bitfield(_.uint8,["rightToLeft","ignoreBaseGlyphs","ignoreLigatures","ignoreMarks","useMarkFilteringSet"])});function LookupList(r){var o=new _.Struct({lookupType:_.uint16,flags:Ge,subTableCount:_.uint16,subTables:new _.Array(new _.Pointer(_.uint16,r),"subTableCount"),markFilteringSet:new _.Optional(_.uint16,(function(r){return r.flags.flags.useMarkFilteringSet}))});return new _.LazyArray(new _.Pointer(_.uint16,o),_.uint16)}var Ve=new _.Struct({start:_.uint16,end:_.uint16,startCoverageIndex:_.uint16}),He=new _.VersionedStruct(_.uint16,{1:{glyphCount:_.uint16,glyphs:new _.Array(_.uint16,"glyphCount")},2:{rangeCount:_.uint16,rangeRecords:new _.Array(Ve,"rangeCount")}}),qe=new _.Struct({start:_.uint16,end:_.uint16,class:_.uint16}),Xe=new _.VersionedStruct(_.uint16,{1:{startGlyph:_.uint16,glyphCount:_.uint16,classValueArray:new _.Array(_.uint16,"glyphCount")},2:{classRangeCount:_.uint16,classRangeRecord:new _.Array(qe,"classRangeCount")}}),Ze=new _.Struct({a:_.uint16,b:_.uint16,deltaFormat:_.uint16}),Ye=new _.Struct({sequenceIndex:_.uint16,lookupListIndex:_.uint16}),Ke=new _.Struct({glyphCount:_.uint16,lookupCount:_.uint16,input:new _.Array(_.uint16,(function(r){return r.glyphCount-1})),lookupRecords:new _.Array(Ye,"lookupCount")}),Je=new _.Array(new _.Pointer(_.uint16,Ke),_.uint16),Qe=new _.Struct({glyphCount:_.uint16,lookupCount:_.uint16,classes:new _.Array(_.uint16,(function(r){return r.glyphCount-1})),lookupRecords:new _.Array(Ye,"lookupCount")}),$e=new _.Array(new _.Pointer(_.uint16,Qe),_.uint16),et=new _.VersionedStruct(_.uint16,{1:{coverage:new _.Pointer(_.uint16,He),ruleSetCount:_.uint16,ruleSets:new _.Array(new _.Pointer(_.uint16,Je),"ruleSetCount")},2:{coverage:new _.Pointer(_.uint16,He),classDef:new _.Pointer(_.uint16,Xe),classSetCnt:_.uint16,classSet:new _.Array(new _.Pointer(_.uint16,$e),"classSetCnt")},3:{glyphCount:_.uint16,lookupCount:_.uint16,coverages:new _.Array(new _.Pointer(_.uint16,He),"glyphCount"),lookupRecords:new _.Array(Ye,"lookupCount")}}),tt=new _.Struct({backtrackGlyphCount:_.uint16,backtrack:new _.Array(_.uint16,"backtrackGlyphCount"),inputGlyphCount:_.uint16,input:new _.Array(_.uint16,(function(r){return r.inputGlyphCount-1})),lookaheadGlyphCount:_.uint16,lookahead:new _.Array(_.uint16,"lookaheadGlyphCount"),lookupCount:_.uint16,lookupRecords:new _.Array(Ye,"lookupCount")}),rt=new _.Array(new _.Pointer(_.uint16,tt),_.uint16),nt=new _.VersionedStruct(_.uint16,{1:{coverage:new _.Pointer(_.uint16,He),chainCount:_.uint16,chainRuleSets:new _.Array(new _.Pointer(_.uint16,rt),"chainCount")},2:{coverage:new _.Pointer(_.uint16,He),backtrackClassDef:new _.Pointer(_.uint16,Xe),inputClassDef:new _.Pointer(_.uint16,Xe),lookaheadClassDef:new _.Pointer(_.uint16,Xe),chainCount:_.uint16,chainClassSet:new _.Array(new _.Pointer(_.uint16,rt),"chainCount")},3:{backtrackGlyphCount:_.uint16,backtrackCoverage:new _.Array(new _.Pointer(_.uint16,He),"backtrackGlyphCount"),inputGlyphCount:_.uint16,inputCoverage:new _.Array(new _.Pointer(_.uint16,He),"inputGlyphCount"),lookaheadGlyphCount:_.uint16,lookaheadCoverage:new _.Array(new _.Pointer(_.uint16,He),"lookaheadGlyphCount"),lookupCount:_.uint16,lookupRecords:new _.Array(Ye,"lookupCount")}}),it=new _.Fixed(16,"BE",14),ot=new _.Struct({startCoord:it,peakCoord:it,endCoord:it}),at=new _.Struct({axisCount:_.uint16,regionCount:_.uint16,variationRegions:new _.Array(new _.Array(ot,"axisCount"),"regionCount")}),st=new _.Struct({shortDeltas:new _.Array(_.int16,(function(r){return r.parent.shortDeltaCount})),regionDeltas:new _.Array(_.int8,(function(r){return r.parent.regionIndexCount-r.parent.shortDeltaCount})),deltas:function deltas(r){return r.shortDeltas.concat(r.regionDeltas)}}),ct=new _.Struct({itemCount:_.uint16,shortDeltaCount:_.uint16,regionIndexCount:_.uint16,regionIndexes:new _.Array(_.uint16,"regionIndexCount"),deltaSets:new _.Array(st,"itemCount")}),lt=new _.Struct({format:_.uint16,variationRegionList:new _.Pointer(_.uint32,at),variationDataCount:_.uint16,itemVariationData:new _.Array(new _.Pointer(_.uint32,ct),"variationDataCount")}),ut=new _.VersionedStruct(_.uint16,{1:(p={axisIndex:_.uint16},p.axisIndex=_.uint16,p.filterRangeMinValue=it,p.filterRangeMaxValue=it,p)}),ft=new _.Struct({conditionCount:_.uint16,conditionTable:new _.Array(new _.Pointer(_.uint32,ut),"conditionCount")}),ht=new _.Struct({featureIndex:_.uint16,alternateFeatureTable:new _.Pointer(_.uint32,We,{type:"parent"})}),dt=new _.Struct({version:_.fixed32,substitutionCount:_.uint16,substitutions:new _.Array(ht,"substitutionCount")}),pt=new _.Struct({conditionSet:new _.Pointer(_.uint32,ft,{type:"parent"}),featureTableSubstitution:new _.Pointer(_.uint32,dt,{type:"parent"})}),gt=new _.Struct({majorVersion:_.uint16,minorVersion:_.uint16,featureVariationRecordCount:_.uint32,featureVariationRecords:new _.Array(pt,"featureVariationRecordCount")}),yt=function(){function PredefinedOp(r,o){this.predefinedOps=r,this.type=o}var r=PredefinedOp.prototype;return r.decode=function decode(r,o,u){return this.predefinedOps[u[0]]?this.predefinedOps[u[0]]:this.type.decode(r,o,u)},r.size=function size(r,o){return this.type.size(r,o)},r.encode=function encode(r,o,u){var p=this.predefinedOps.indexOf(o);return-1!==p?p:this.type.encode(r,o,u)},PredefinedOp}(),vt=function(r){function CFFEncodingVersion(){return r.call(this,"UInt8")||this}return _inheritsLoose(CFFEncodingVersion,r),CFFEncodingVersion.prototype.decode=function decode(r){return 127&_.uint8.decode(r)},CFFEncodingVersion}(_.Number),mt=new _.Struct({first:_.uint16,nLeft:_.uint8}),bt=new _.Struct({first:_.uint16,nLeft:_.uint16}),wt=new yt([Fe,["","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","space","exclamsmall","Hungarumlautsmall","","dollaroldstyle","dollarsuperior","ampersandsmall","Acutesmall","parenleftsuperior","parenrightsuperior","twodotenleader","onedotenleader","comma","hyphen","period","fraction","zerooldstyle","oneoldstyle","twooldstyle","threeoldstyle","fouroldstyle","fiveoldstyle","sixoldstyle","sevenoldstyle","eightoldstyle","nineoldstyle","colon","semicolon","commasuperior","threequartersemdash","periodsuperior","questionsmall","","asuperior","bsuperior","centsuperior","dsuperior","esuperior","","","isuperior","","","lsuperior","msuperior","nsuperior","osuperior","","","rsuperior","ssuperior","tsuperior","","ff","fi","fl","ffi","ffl","parenleftinferior","","parenrightinferior","Circumflexsmall","hyphensuperior","Gravesmall","Asmall","Bsmall","Csmall","Dsmall","Esmall","Fsmall","Gsmall","Hsmall","Ismall","Jsmall","Ksmall","Lsmall","Msmall","Nsmall","Osmall","Psmall","Qsmall","Rsmall","Ssmall","Tsmall","Usmall","Vsmall","Wsmall","Xsmall","Ysmall","Zsmall","colonmonetary","onefitted","rupiah","Tildesmall","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","","exclamdownsmall","centoldstyle","Lslashsmall","","","Scaronsmall","Zcaronsmall","Dieresissmall","Brevesmall","Caronsmall","","Dotaccentsmall","","","Macronsmall","","","figuredash","hypheninferior","","","Ogoneksmall","Ringsmall","Cedillasmall","","","","onequarter","onehalf","threequarters","questiondownsmall","oneeighth","threeeighths","fiveeighths","seveneighths","onethird","twothirds","","","zerosuperior","onesuperior","twosuperior","threesuperior","foursuperior","fivesuperior","sixsuperior","sevensuperior","eightsuperior","ninesuperior","zeroinferior","oneinferior","twoinferior","threeinferior","fourinferior","fiveinferior","sixinferior","seveninferior","eightinferior","nineinferior","centinferior","dollarinferior","periodinferior","commainferior","Agravesmall","Aacutesmall","Acircumflexsmall","Atildesmall","Adieresissmall","Aringsmall","AEsmall","Ccedillasmall","Egravesmall","Eacutesmall","Ecircumflexsmall","Edieresissmall","Igravesmall","Iacutesmall","Icircumflexsmall","Idieresissmall","Ethsmall","Ntildesmall","Ogravesmall","Oacutesmall","Ocircumflexsmall","Otildesmall","Odieresissmall","OEsmall","Oslashsmall","Ugravesmall","Uacutesmall","Ucircumflexsmall","Udieresissmall","Yacutesmall","Thornsmall","Ydieresissmall"]],new Te(new _.VersionedStruct(new vt,{0:{nCodes:_.uint8,codes:new _.Array(_.uint8,"nCodes")},1:{nRanges:_.uint8,ranges:new _.Array(mt,"nRanges")}}),{lazy:!0})),_t=function(r){function RangeArray(){return r.apply(this,arguments)||this}return _inheritsLoose(RangeArray,r),RangeArray.prototype.decode=function decode(r,o){for(var u=P.resolveLength(this.length,r,o),p=0,y=[];p<u;){var _=this.type.decode(r,o);_.offset=p,p+=_.nLeft+1,y.push(_)}return y},RangeArray}(_.Array),St=new _.VersionedStruct(_.uint8,{0:{glyphs:new _.Array(_.uint16,(function(r){return r.parent.CharStrings.length-1}))},1:{ranges:new _t(mt,(function(r){return r.parent.CharStrings.length-1}))},2:{ranges:new _t(bt,(function(r){return r.parent.CharStrings.length-1}))}}),xt=new yt([Le,[".notdef","space","exclamsmall","Hungarumlautsmall","dollaroldstyle","dollarsuperior","ampersandsmall","Acutesmall","parenleftsuperior","parenrightsuperior","twodotenleader","onedotenleader","comma","hyphen","period","fraction","zerooldstyle","oneoldstyle","twooldstyle","threeoldstyle","fouroldstyle","fiveoldstyle","sixoldstyle","sevenoldstyle","eightoldstyle","nineoldstyle","colon","semicolon","commasuperior","threequartersemdash","periodsuperior","questionsmall","asuperior","bsuperior","centsuperior","dsuperior","esuperior","isuperior","lsuperior","msuperior","nsuperior","osuperior","rsuperior","ssuperior","tsuperior","ff","fi","fl","ffi","ffl","parenleftinferior","parenrightinferior","Circumflexsmall","hyphensuperior","Gravesmall","Asmall","Bsmall","Csmall","Dsmall","Esmall","Fsmall","Gsmall","Hsmall","Ismall","Jsmall","Ksmall","Lsmall","Msmall","Nsmall","Osmall","Psmall","Qsmall","Rsmall","Ssmall","Tsmall","Usmall","Vsmall","Wsmall","Xsmall","Ysmall","Zsmall","colonmonetary","onefitted","rupiah","Tildesmall","exclamdownsmall","centoldstyle","Lslashsmall","Scaronsmall","Zcaronsmall","Dieresissmall","Brevesmall","Caronsmall","Dotaccentsmall","Macronsmall","figuredash","hypheninferior","Ogoneksmall","Ringsmall","Cedillasmall","onequarter","onehalf","threequarters","questiondownsmall","oneeighth","threeeighths","fiveeighths","seveneighths","onethird","twothirds","zerosuperior","onesuperior","twosuperior","threesuperior","foursuperior","fivesuperior","sixsuperior","sevensuperior","eightsuperior","ninesuperior","zeroinferior","oneinferior","twoinferior","threeinferior","fourinferior","fiveinferior","sixinferior","seveninferior","eightinferior","nineinferior","centinferior","dollarinferior","periodinferior","commainferior","Agravesmall","Aacutesmall","Acircumflexsmall","Atildesmall","Adieresissmall","Aringsmall","AEsmall","Ccedillasmall","Egravesmall","Eacutesmall","Ecircumflexsmall","Edieresissmall","Igravesmall","Iacutesmall","Icircumflexsmall","Idieresissmall","Ethsmall","Ntildesmall","Ogravesmall","Oacutesmall","Ocircumflexsmall","Otildesmall","Odieresissmall","OEsmall","Oslashsmall","Ugravesmall","Uacutesmall","Ucircumflexsmall","Udieresissmall","Yacutesmall","Thornsmall","Ydieresissmall"],[".notdef","space","dollaroldstyle","dollarsuperior","parenleftsuperior","parenrightsuperior","twodotenleader","onedotenleader","comma","hyphen","period","fraction","zerooldstyle","oneoldstyle","twooldstyle","threeoldstyle","fouroldstyle","fiveoldstyle","sixoldstyle","sevenoldstyle","eightoldstyle","nineoldstyle","colon","semicolon","commasuperior","threequartersemdash","periodsuperior","asuperior","bsuperior","centsuperior","dsuperior","esuperior","isuperior","lsuperior","msuperior","nsuperior","osuperior","rsuperior","ssuperior","tsuperior","ff","fi","fl","ffi","ffl","parenleftinferior","parenrightinferior","hyphensuperior","colonmonetary","onefitted","rupiah","centoldstyle","figuredash","hypheninferior","onequarter","onehalf","threequarters","oneeighth","threeeighths","fiveeighths","seveneighths","onethird","twothirds","zerosuperior","onesuperior","twosuperior","threesuperior","foursuperior","fivesuperior","sixsuperior","sevensuperior","eightsuperior","ninesuperior","zeroinferior","oneinferior","twoinferior","threeinferior","fourinferior","fiveinferior","sixinferior","seveninferior","eightinferior","nineinferior","centinferior","dollarinferior","periodinferior","commainferior"]],new Te(St,{lazy:!0})),At=new _.Struct({first:_.uint16,fd:_.uint8}),Ct=new _.Struct({first:_.uint32,fd:_.uint16}),Pt=new _.VersionedStruct(_.uint8,{0:{fds:new _.Array(_.uint8,(function(r){return r.parent.CharStrings.length}))},3:{nRanges:_.uint16,ranges:new _.Array(At,"nRanges"),sentinel:_.uint16},4:{nRanges:_.uint32,ranges:new _.Array(Ct,"nRanges"),sentinel:_.uint32}}),kt=new Te(Oe),Et=function(){function CFFPrivateOp(){}var r=CFFPrivateOp.prototype;return r.decode=function decode(r,o,u){return o.length=u[0],kt.decode(r,o,[u[1]])},r.size=function size(r,o){return[Oe.size(r,o,!1),kt.size(r,o)[0]]},r.encode=function encode(r,o,u){return[Oe.size(o,u,!1),kt.encode(r,o,u)[0]]},CFFPrivateOp}(),Tt=new Ee([[18,"Private",new Et,null],[[12,38],"FontName","sid",null],[[12,7],"FontMatrix","array",[.001,0,0,.001,0,0]],[[12,5],"PaintType","number",0]]),It=new Ee([[[12,30],"ROS",["sid","sid","number"],null],[0,"version","sid",null],[1,"Notice","sid",null],[[12,0],"Copyright","sid",null],[2,"FullName","sid",null],[3,"FamilyName","sid",null],[4,"Weight","sid",null],[[12,1],"isFixedPitch","boolean",!1],[[12,2],"ItalicAngle","number",0],[[12,3],"UnderlinePosition","number",-100],[[12,4],"UnderlineThickness","number",50],[[12,5],"PaintType","number",0],[[12,6],"CharstringType","number",2],[[12,7],"FontMatrix","array",[.001,0,0,.001,0,0]],[13,"UniqueID","number",null],[5,"FontBBox","array",[0,0,0,0]],[[12,8],"StrokeWidth","number",0],[14,"XUID","array",null],[15,"charset",xt,Le],[16,"Encoding",wt,Fe],[17,"CharStrings",new Te(new Ae),null],[18,"Private",new Et,null],[[12,20],"SyntheticBase","number",null],[[12,21],"PostScript","sid",null],[[12,22],"BaseFontName","sid",null],[[12,23],"BaseFontBlend","delta",null],[[12,31],"CIDFontVersion","number",0],[[12,32],"CIDFontRevision","number",0],[[12,33],"CIDFontType","number",0],[[12,34],"CIDCount","number",8720],[[12,35],"UIDBase","number",null],[[12,37],"FDSelect",new Te(Pt),null],[[12,36],"FDArray",new Te(new Ae(Tt)),null],[[12,38],"FontName","sid",null]]),Ot=new _.Struct({length:_.uint16,itemVariationStore:lt}),Bt=new Ee([[[12,7],"FontMatrix","array",[.001,0,0,.001,0,0]],[17,"CharStrings",new Te(new Ae),null],[[12,37],"FDSelect",new Te(Pt),null],[[12,36],"FDArray",new Te(new Ae(Tt)),null],[24,"vstore",new Te(Ot),null],[25,"maxstack","number",193]]),Ft=new _.VersionedStruct(_.fixed16,{1:{hdrSize:_.uint8,offSize:_.uint8,nameIndex:new Ae(new _.String("length")),topDictIndex:new Ae(It),stringIndex:new Ae(new _.String("length")),globalSubrIndex:new Ae},2:{hdrSize:_.uint8,length:_.uint16,topDict:Bt,globalSubrIndex:new Ae}}),Lt=function(){function CFFFont(r){this.stream=r,this.decode()}CFFFont.decode=function decode(r){return new CFFFont(r)};var r=CFFFont.prototype;return r.decode=function decode(){this.stream.pos;var r=Ft.decode(this.stream);for(var o in r){var u=r[o];this[o]=u}if(this.version<2){if(1!==this.topDictIndex.length)throw new Error("Only a single font is allowed in CFF");this.topDict=this.topDictIndex[0]}return this.isCIDFont=null!=this.topDict.ROS,this},r.string=function string(r){return this.version>=2?null:r<Be.length?Be[r]:this.stringIndex[r-Be.length]},r.getCharString=function getCharString(r){return this.stream.pos=this.topDict.CharStrings[r].offset,this.stream.readBuffer(this.topDict.CharStrings[r].length)},r.getGlyphName=function getGlyphName(r){if(this.version>=2)return null;if(this.isCIDFont)return null;var o=this.topDict.charset;if(Array.isArray(o))return o[r];if(0===r)return".notdef";switch(r-=1,o.version){case 0:return this.string(o.glyphs[r]);case 1:case 2:for(var u=0;u<o.ranges.length;u++){var p=o.ranges[u];if(p.offset<=r&&r<=p.offset+p.nLeft)return this.string(p.first+(r-p.offset))}}return null},r.fdForGlyph=function fdForGlyph(r){if(!this.topDict.FDSelect)return null;switch(this.topDict.FDSelect.version){case 0:return this.topDict.FDSelect.fds[r];case 3:case 4:for(var o=this.topDict.FDSelect.ranges,u=0,p=o.length-1;u<=p;){var y=u+p>>1;if(r<o[y].first)p=y-1;else{if(!(y<p&&r>=o[y+1].first))return o[y].fd;u=y+1}}default:throw new Error("Unknown FDSelect version: ".concat(this.topDict.FDSelect.version))}},r.privateDictForGlyph=function privateDictForGlyph(r){if(this.topDict.FDSelect){var o=this.fdForGlyph(r);return this.topDict.FDArray[o]?this.topDict.FDArray[o].Private:null}return this.version<2?this.topDict.Private:this.topDict.FDArray[0].Private},_createClass(CFFFont,[{key:"postscriptName",get:function get(){return this.version<2?this.nameIndex[0]:null}},{key:"fullName",get:function get(){return this.string(this.topDict.FullName)}},{key:"familyName",get:function get(){return this.string(this.topDict.FamilyName)}}])}(),Dt=new _.Struct({glyphIndex:_.uint16,vertOriginY:_.int16}),Rt=new _.Struct({majorVersion:_.uint16,minorVersion:_.uint16,defaultVertOriginY:_.int16,numVertOriginYMetrics:_.uint16,metrics:new _.Array(Dt,"numVertOriginYMetrics")}),Mt=new _.Struct({height:_.uint8,width:_.uint8,horiBearingX:_.int8,horiBearingY:_.int8,horiAdvance:_.uint8,vertBearingX:_.int8,vertBearingY:_.int8,vertAdvance:_.uint8}),Nt=new _.Struct({height:_.uint8,width:_.uint8,bearingX:_.int8,bearingY:_.int8,advance:_.uint8}),Ut=new _.Struct({glyph:_.uint16,xOffset:_.int8,yOffset:_.int8}),Wt=function ByteAligned(){},zt=function BitAligned(){},jt=(new _.VersionedStruct("version",{1:{metrics:Nt,data:Wt},2:{metrics:Nt,data:zt},5:{data:zt},6:{metrics:Mt,data:Wt},7:{metrics:Mt,data:zt},8:{metrics:Nt,pad:new _.Reserved(_.uint8),numComponents:_.uint16,components:new _.Array(Ut,"numComponents")},9:{metrics:Mt,pad:new _.Reserved(_.uint8),numComponents:_.uint16,components:new _.Array(Ut,"numComponents")},17:{metrics:Nt,dataLen:_.uint32,data:new _.Buffer("dataLen")},18:{metrics:Mt,dataLen:_.uint32,data:new _.Buffer("dataLen")},19:{dataLen:_.uint32,data:new _.Buffer("dataLen")}}),new _.Struct({ascender:_.int8,descender:_.int8,widthMax:_.uint8,caretSlopeNumerator:_.int8,caretSlopeDenominator:_.int8,caretOffset:_.int8,minOriginSB:_.int8,minAdvanceSB:_.int8,maxBeforeBL:_.int8,minAfterBL:_.int8,pad:new _.Reserved(_.int8,2)})),Gt=new _.Struct({glyphCode:_.uint16,offset:_.uint16}),Vt=new _.VersionedStruct(_.uint16,{header:{imageFormat:_.uint16,imageDataOffset:_.uint32},1:{offsetArray:new _.Array(_.uint32,(function(r){return r.parent.lastGlyphIndex-r.parent.firstGlyphIndex+1}))},2:{imageSize:_.uint32,bigMetrics:Mt},3:{offsetArray:new _.Array(_.uint16,(function(r){return r.parent.lastGlyphIndex-r.parent.firstGlyphIndex+1}))},4:{numGlyphs:_.uint32,glyphArray:new _.Array(Gt,(function(r){return r.numGlyphs+1}))},5:{imageSize:_.uint32,bigMetrics:Mt,numGlyphs:_.uint32,glyphCodeArray:new _.Array(_.uint16,"numGlyphs")}}),Ht=new _.Struct({firstGlyphIndex:_.uint16,lastGlyphIndex:_.uint16,subtable:new _.Pointer(_.uint32,Vt)}),qt=new _.Struct({indexSubTableArray:new _.Pointer(_.uint32,new _.Array(Ht,1),{type:"parent"}),indexTablesSize:_.uint32,numberOfIndexSubTables:_.uint32,colorRef:_.uint32,hori:jt,vert:jt,startGlyphIndex:_.uint16,endGlyphIndex:_.uint16,ppemX:_.uint8,ppemY:_.uint8,bitDepth:_.uint8,flags:new _.Bitfield(_.uint8,["horizontal","vertical"])}),Xt=new _.Struct({version:_.uint32,numSizes:_.uint32,sizes:new _.Array(qt,"numSizes")}),Zt=new _.Struct({ppem:_.uint16,resolution:_.uint16,imageOffsets:new _.Array(new _.Pointer(_.uint32,"void"),(function(r){return r.parent.parent.maxp.numGlyphs+1}))}),Yt=new _.Struct({version:_.uint16,flags:new _.Bitfield(_.uint16,["renderOutlines"]),numImgTables:_.uint32,imageTables:new _.Array(new _.Pointer(_.uint32,Zt),"numImgTables")}),Kt=new _.Struct({gid:_.uint16,paletteIndex:_.uint16}),Jt=new _.Struct({gid:_.uint16,firstLayerIndex:_.uint16,numLayers:_.uint16}),Qt=new _.Struct({version:_.uint16,numBaseGlyphRecords:_.uint16,baseGlyphRecord:new _.Pointer(_.uint32,new _.Array(Jt,"numBaseGlyphRecords")),layerRecords:new _.Pointer(_.uint32,new _.Array(Kt,"numLayerRecords"),{lazy:!0}),numLayerRecords:_.uint16}),$t=new _.Struct({blue:_.uint8,green:_.uint8,red:_.uint8,alpha:_.uint8}),er=new _.VersionedStruct(_.uint16,{header:{numPaletteEntries:_.uint16,numPalettes:_.uint16,numColorRecords:_.uint16,colorRecords:new _.Pointer(_.uint32,new _.Array($t,"numColorRecords")),colorRecordIndices:new _.Array(_.uint16,"numPalettes")},0:{},1:{offsetPaletteTypeArray:new _.Pointer(_.uint32,new _.Array(_.uint32,"numPalettes")),offsetPaletteLabelArray:new _.Pointer(_.uint32,new _.Array(_.uint16,"numPalettes")),offsetPaletteEntryLabelArray:new _.Pointer(_.uint32,new _.Array(_.uint16,"numPaletteEntries"))}}),tr=new _.VersionedStruct(_.uint16,{1:{coordinate:_.int16},2:{coordinate:_.int16,referenceGlyph:_.uint16,baseCoordPoint:_.uint16},3:{coordinate:_.int16,deviceTable:new _.Pointer(_.uint16,Ze)}}),rr=new _.Struct({defaultIndex:_.uint16,baseCoordCount:_.uint16,baseCoords:new _.Array(new _.Pointer(_.uint16,tr),"baseCoordCount")}),nr=new _.Struct({tag:new _.String(4),minCoord:new _.Pointer(_.uint16,tr,{type:"parent"}),maxCoord:new _.Pointer(_.uint16,tr,{type:"parent"})}),ir=new _.Struct({minCoord:new _.Pointer(_.uint16,tr),maxCoord:new _.Pointer(_.uint16,tr),featMinMaxCount:_.uint16,featMinMaxRecords:new _.Array(nr,"featMinMaxCount")}),or=new _.Struct({tag:new _.String(4),minMax:new _.Pointer(_.uint16,ir,{type:"parent"})}),ar=new _.Struct({baseValues:new _.Pointer(_.uint16,rr),defaultMinMax:new _.Pointer(_.uint16,ir),baseLangSysCount:_.uint16,baseLangSysRecords:new _.Array(or,"baseLangSysCount")}),sr=new _.Struct({tag:new _.String(4),script:new _.Pointer(_.uint16,ar,{type:"parent"})}),cr=new _.Array(sr,_.uint16),lr=new _.Array(new _.String(4),_.uint16),ur=new _.Struct({baseTagList:new _.Pointer(_.uint16,lr),baseScriptList:new _.Pointer(_.uint16,cr)}),fr=new _.VersionedStruct(_.uint32,{header:{horizAxis:new _.Pointer(_.uint16,ur),vertAxis:new _.Pointer(_.uint16,ur)},65536:{},65537:{itemVariationStore:new _.Pointer(_.uint32,lt)}}),hr=new _.Array(_.uint16,_.uint16),dr=new _.Struct({coverage:new _.Pointer(_.uint16,He),glyphCount:_.uint16,attachPoints:new _.Array(new _.Pointer(_.uint16,hr),"glyphCount")}),pr=new _.VersionedStruct(_.uint16,{1:{coordinate:_.int16},2:{caretValuePoint:_.uint16},3:{coordinate:_.int16,deviceTable:new _.Pointer(_.uint16,Ze)}}),gr=new _.Array(new _.Pointer(_.uint16,pr),_.uint16),yr=new _.Struct({coverage:new _.Pointer(_.uint16,He),ligGlyphCount:_.uint16,ligGlyphs:new _.Array(new _.Pointer(_.uint16,gr),"ligGlyphCount")}),vr=new _.Struct({markSetTableFormat:_.uint16,markSetCount:_.uint16,coverage:new _.Array(new _.Pointer(_.uint32,He),"markSetCount")}),mr=new _.VersionedStruct(_.uint32,{header:{glyphClassDef:new _.Pointer(_.uint16,Xe),attachList:new _.Pointer(_.uint16,dr),ligCaretList:new _.Pointer(_.uint16,yr),markAttachClassDef:new _.Pointer(_.uint16,Xe)},65536:{},65538:{markGlyphSetsDef:new _.Pointer(_.uint16,vr)},65539:{markGlyphSetsDef:new _.Pointer(_.uint16,vr),itemVariationStore:new _.Pointer(_.uint32,lt)}}),br=new _.Bitfield(_.uint16,["xPlacement","yPlacement","xAdvance","yAdvance","xPlaDevice","yPlaDevice","xAdvDevice","yAdvDevice"]),wr={xPlacement:_.int16,yPlacement:_.int16,xAdvance:_.int16,yAdvance:_.int16,xPlaDevice:new _.Pointer(_.uint16,Ze,{type:"global",relativeTo:function relativeTo(r){return r.rel}}),yPlaDevice:new _.Pointer(_.uint16,Ze,{type:"global",relativeTo:function relativeTo(r){return r.rel}}),xAdvDevice:new _.Pointer(_.uint16,Ze,{type:"global",relativeTo:function relativeTo(r){return r.rel}}),yAdvDevice:new _.Pointer(_.uint16,Ze,{type:"global",relativeTo:function relativeTo(r){return r.rel}})},_r=function(){function ValueRecord(r){void 0===r&&(r="valueFormat"),this.key=r}var r=ValueRecord.prototype;return r.buildStruct=function buildStruct(r){for(var o=r;!o[this.key]&&o.parent;)o=o.parent;if(o[this.key]){var u={rel:function(){return o._startOffset}},p=o[this.key];for(var y in p)p[y]&&(u[y]=wr[y]);return new _.Struct(u)}},r.size=function size(r,o){return this.buildStruct(o).size(r,o)},r.decode=function decode(r,o){var u=this.buildStruct(o).decode(r,o);return delete u.rel,u},ValueRecord}(),Sr=new _.Struct({secondGlyph:_.uint16,value1:new _r("valueFormat1"),value2:new _r("valueFormat2")}),xr=new _.Array(Sr,_.uint16),Ar=new _.Struct({value1:new _r("valueFormat1"),value2:new _r("valueFormat2")}),Cr=new _.VersionedStruct(_.uint16,{1:{xCoordinate:_.int16,yCoordinate:_.int16},2:{xCoordinate:_.int16,yCoordinate:_.int16,anchorPoint:_.uint16},3:{xCoordinate:_.int16,yCoordinate:_.int16,xDeviceTable:new _.Pointer(_.uint16,Ze),yDeviceTable:new _.Pointer(_.uint16,Ze)}}),Pr=new _.Struct({entryAnchor:new _.Pointer(_.uint16,Cr,{type:"parent"}),exitAnchor:new _.Pointer(_.uint16,Cr,{type:"parent"})}),kr=new _.Struct({class:_.uint16,markAnchor:new _.Pointer(_.uint16,Cr,{type:"parent"})}),Er=new _.Array(kr,_.uint16),Tr=new _.Array(new _.Pointer(_.uint16,Cr),(function(r){return r.parent.classCount})),Ir=new _.Array(Tr,_.uint16),Or=new _.Array(new _.Pointer(_.uint16,Cr),(function(r){return r.parent.parent.classCount})),Br=new _.Array(Or,_.uint16),Fr=new _.Array(new _.Pointer(_.uint16,Br),_.uint16),Lr=new _.VersionedStruct("lookupType",{1:new _.VersionedStruct(_.uint16,{1:{coverage:new _.Pointer(_.uint16,He),valueFormat:br,value:new _r},2:{coverage:new _.Pointer(_.uint16,He),valueFormat:br,valueCount:_.uint16,values:new _.LazyArray(new _r,"valueCount")}}),2:new _.VersionedStruct(_.uint16,{1:{coverage:new _.Pointer(_.uint16,He),valueFormat1:br,valueFormat2:br,pairSetCount:_.uint16,pairSets:new _.LazyArray(new _.Pointer(_.uint16,xr),"pairSetCount")},2:{coverage:new _.Pointer(_.uint16,He),valueFormat1:br,valueFormat2:br,classDef1:new _.Pointer(_.uint16,Xe),classDef2:new _.Pointer(_.uint16,Xe),class1Count:_.uint16,class2Count:_.uint16,classRecords:new _.LazyArray(new _.LazyArray(Ar,"class2Count"),"class1Count")}}),3:{format:_.uint16,coverage:new _.Pointer(_.uint16,He),entryExitCount:_.uint16,entryExitRecords:new _.Array(Pr,"entryExitCount")},4:{format:_.uint16,markCoverage:new _.Pointer(_.uint16,He),baseCoverage:new _.Pointer(_.uint16,He),classCount:_.uint16,markArray:new _.Pointer(_.uint16,Er),baseArray:new _.Pointer(_.uint16,Ir)},5:{format:_.uint16,markCoverage:new _.Pointer(_.uint16,He),ligatureCoverage:new _.Pointer(_.uint16,He),classCount:_.uint16,markArray:new _.Pointer(_.uint16,Er),ligatureArray:new _.Pointer(_.uint16,Fr)},6:{format:_.uint16,mark1Coverage:new _.Pointer(_.uint16,He),mark2Coverage:new _.Pointer(_.uint16,He),classCount:_.uint16,mark1Array:new _.Pointer(_.uint16,Er),mark2Array:new _.Pointer(_.uint16,Ir)},7:et,8:nt,9:{posFormat:_.uint16,lookupType:_.uint16,extension:new _.Pointer(_.uint32,Lr)}});Lr.versions[9].extension.type=Lr;var Dr=new _.VersionedStruct(_.uint32,{header:{scriptList:new _.Pointer(_.uint16,Ue),featureList:new _.Pointer(_.uint16,je),lookupList:new _.Pointer(_.uint16,new LookupList(Lr))},65536:{},65537:{featureVariations:new _.Pointer(_.uint32,gt)}}),Rr=new _.Array(_.uint16,_.uint16),Mr=Rr,Nr=new _.Struct({glyph:_.uint16,compCount:_.uint16,components:new _.Array(_.uint16,(function(r){return r.compCount-1}))}),Ur=new _.Array(new _.Pointer(_.uint16,Nr),_.uint16),Wr=new _.VersionedStruct("lookupType",{1:new _.VersionedStruct(_.uint16,{1:{coverage:new _.Pointer(_.uint16,He),deltaGlyphID:_.int16},2:{coverage:new _.Pointer(_.uint16,He),glyphCount:_.uint16,substitute:new _.LazyArray(_.uint16,"glyphCount")}}),2:{substFormat:_.uint16,coverage:new _.Pointer(_.uint16,He),count:_.uint16,sequences:new _.LazyArray(new _.Pointer(_.uint16,Rr),"count")},3:{substFormat:_.uint16,coverage:new _.Pointer(_.uint16,He),count:_.uint16,alternateSet:new _.LazyArray(new _.Pointer(_.uint16,Mr),"count")},4:{substFormat:_.uint16,coverage:new _.Pointer(_.uint16,He),count:_.uint16,ligatureSets:new _.LazyArray(new _.Pointer(_.uint16,Ur),"count")},5:et,6:nt,7:{substFormat:_.uint16,lookupType:_.uint16,extension:new _.Pointer(_.uint32,Wr)},8:{substFormat:_.uint16,coverage:new _.Pointer(_.uint16,He),backtrackCoverage:new _.Array(new _.Pointer(_.uint16,He),"backtrackGlyphCount"),lookaheadGlyphCount:_.uint16,lookaheadCoverage:new _.Array(new _.Pointer(_.uint16,He),"lookaheadGlyphCount"),glyphCount:_.uint16,substitutes:new _.Array(_.uint16,"glyphCount")}});Wr.versions[7].extension.type=Wr;var zr=new _.VersionedStruct(_.uint32,{header:{scriptList:new _.Pointer(_.uint16,Ue),featureList:new _.Pointer(_.uint16,je),lookupList:new _.Pointer(_.uint16,new LookupList(Wr))},65536:{},65537:{featureVariations:new _.Pointer(_.uint32,gt)}}),jr=new _.Array(_.uint16,_.uint16),Gr=new _.Struct({shrinkageEnableGSUB:new _.Pointer(_.uint16,jr),shrinkageDisableGSUB:new _.Pointer(_.uint16,jr),shrinkageEnableGPOS:new _.Pointer(_.uint16,jr),shrinkageDisableGPOS:new _.Pointer(_.uint16,jr),shrinkageJstfMax:new _.Pointer(_.uint16,new LookupList(Lr)),extensionEnableGSUB:new _.Pointer(_.uint16,jr),extensionDisableGSUB:new _.Pointer(_.uint16,jr),extensionEnableGPOS:new _.Pointer(_.uint16,jr),extensionDisableGPOS:new _.Pointer(_.uint16,jr),extensionJstfMax:new _.Pointer(_.uint16,new LookupList(Lr))}),Vr=new _.Array(new _.Pointer(_.uint16,Gr),_.uint16),Hr=new _.Struct({tag:new _.String(4),jstfLangSys:new _.Pointer(_.uint16,Vr)}),qr=new _.Struct({extenderGlyphs:new _.Pointer(_.uint16,new _.Array(_.uint16,_.uint16)),defaultLangSys:new _.Pointer(_.uint16,Vr),langSysCount:_.uint16,langSysRecords:new _.Array(Hr,"langSysCount")}),Xr=new _.Struct({tag:new _.String(4),script:new _.Pointer(_.uint16,qr,{type:"parent"})}),Zr=new _.Struct({version:_.uint32,scriptCount:_.uint16,scriptList:new _.Array(Xr,"scriptCount")}),Yr=function(){function VariableSizeNumber(r){this._size=r}var r=VariableSizeNumber.prototype;return r.decode=function decode(r,o){switch(this.size(0,o)){case 1:return r.readUInt8();case 2:return r.readUInt16BE();case 3:return r.readUInt24BE();case 4:return r.readUInt32BE()}},r.size=function size(r,o){return P.resolveLength(this._size,null,o)},VariableSizeNumber}(),Kr=new _.Struct({entry:new Yr((function(r){return 1+((48&r.parent.entryFormat)>>4)})),outerIndex:function outerIndex(r){return r.entry>>1+(15&r.parent.entryFormat)},innerIndex:function innerIndex(r){return r.entry&(1<<1+(15&r.parent.entryFormat))-1}}),Jr=new _.Struct({entryFormat:_.uint16,mapCount:_.uint16,mapData:new _.Array(Kr,"mapCount")}),Qr=new _.Struct({majorVersion:_.uint16,minorVersion:_.uint16,itemVariationStore:new _.Pointer(_.uint32,lt),advanceWidthMapping:new _.Pointer(_.uint32,Jr),LSBMapping:new _.Pointer(_.uint32,Jr),RSBMapping:new _.Pointer(_.uint32,Jr)}),$r=new _.Struct({format:_.uint32,length:_.uint32,offset:_.uint32}),en=new _.Struct({reserved:new _.Reserved(_.uint16,2),cbSignature:_.uint32,signature:new _.Buffer("cbSignature")}),tn=new _.Struct({ulVersion:_.uint32,usNumSigs:_.uint16,usFlag:_.uint16,signatures:new _.Array($r,"usNumSigs"),signatureBlocks:new _.Array(en,"usNumSigs")}),rn=new _.Struct({rangeMaxPPEM:_.uint16,rangeGaspBehavior:new _.Bitfield(_.uint16,["grayscale","gridfit","symmetricSmoothing","symmetricGridfit"])}),nn=new _.Struct({version:_.uint16,numRanges:_.uint16,gaspRanges:new _.Array(rn,"numRanges")}),on=new _.Struct({pixelSize:_.uint8,maximumWidth:_.uint8,widths:new _.Array(_.uint8,(function(r){return r.parent.parent.maxp.numGlyphs}))}),an=new _.Struct({version:_.uint16,numRecords:_.int16,sizeDeviceRecord:_.int32,records:new _.Array(on,"numRecords")}),sn=new _.Struct({left:_.uint16,right:_.uint16,value:_.int16}),cn=new _.Struct({firstGlyph:_.uint16,nGlyphs:_.uint16,offsets:new _.Array(_.uint16,"nGlyphs"),max:function max(r){return r.offsets.length&&Math.max.apply(Math,r.offsets)}}),ln=new _.Struct({off:function off(r){return r._startOffset-r.parent.parent._startOffset},len:function len(r){return((r.parent.leftTable.max-r.off)/r.parent.rowWidth+1)*(r.parent.rowWidth/2)},values:new _.LazyArray(_.int16,"len")}),un=new _.VersionedStruct("format",{0:{nPairs:_.uint16,searchRange:_.uint16,entrySelector:_.uint16,rangeShift:_.uint16,pairs:new _.Array(sn,"nPairs")},2:{rowWidth:_.uint16,leftTable:new _.Pointer(_.uint16,cn,{type:"parent"}),rightTable:new _.Pointer(_.uint16,cn,{type:"parent"}),array:new _.Pointer(_.uint16,ln,{type:"parent"})},3:{glyphCount:_.uint16,kernValueCount:_.uint8,leftClassCount:_.uint8,rightClassCount:_.uint8,flags:_.uint8,kernValue:new _.Array(_.int16,"kernValueCount"),leftClass:new _.Array(_.uint8,"glyphCount"),rightClass:new _.Array(_.uint8,"glyphCount"),kernIndex:new _.Array(_.uint8,(function(r){return r.leftClassCount*r.rightClassCount}))}}),fn=new _.VersionedStruct("version",{0:{subVersion:_.uint16,length:_.uint16,format:_.uint8,coverage:new _.Bitfield(_.uint8,["horizontal","minimum","crossStream","override"]),subtable:un,padding:new _.Reserved(_.uint8,(function(r){return r.length-r._currentOffset}))},1:{length:_.uint32,coverage:new _.Bitfield(_.uint8,[null,null,null,null,null,"variation","crossStream","vertical"]),format:_.uint8,tupleIndex:_.uint16,subtable:un,padding:new _.Reserved(_.uint8,(function(r){return r.length-r._currentOffset}))}}),hn=new _.VersionedStruct(_.uint16,{0:{nTables:_.uint16,tables:new _.Array(fn,"nTables")},1:{reserved:new _.Reserved(_.uint16),nTables:_.uint32,tables:new _.Array(fn,"nTables")}}),dn=new _.Struct({version:_.uint16,numGlyphs:_.uint16,yPels:new _.Array(_.uint8,"numGlyphs")}),pn=new _.Struct({version:_.uint16,fontNumber:_.uint32,pitch:_.uint16,xHeight:_.uint16,style:_.uint16,typeFamily:_.uint16,capHeight:_.uint16,symbolSet:_.uint16,typeface:new _.String(16),characterComplement:new _.String(8),fileName:new _.String(6),strokeWeight:new _.String(1),widthType:new _.String(1),serifStyle:_.uint8,reserved:new _.Reserved(_.uint8)}),gn=new _.Struct({bCharSet:_.uint8,xRatio:_.uint8,yStartRatio:_.uint8,yEndRatio:_.uint8}),yn=new _.Struct({yPelHeight:_.uint16,yMax:_.int16,yMin:_.int16}),vn=new _.Struct({recs:_.uint16,startsz:_.uint8,endsz:_.uint8,entries:new _.Array(yn,"recs")}),mn=new _.Struct({version:_.uint16,numRecs:_.uint16,numRatios:_.uint16,ratioRanges:new _.Array(gn,"numRatios"),offsets:new _.Array(_.uint16,"numRatios"),groups:new _.Array(vn,"numRecs")}),bn=new _.Struct({version:_.uint16,ascent:_.int16,descent:_.int16,lineGap:_.int16,advanceHeightMax:_.int16,minTopSideBearing:_.int16,minBottomSideBearing:_.int16,yMaxExtent:_.int16,caretSlopeRise:_.int16,caretSlopeRun:_.int16,caretOffset:_.int16,reserved:new _.Reserved(_.int16,4),metricDataFormat:_.int16,numberOfMetrics:_.uint16}),wn=new _.Struct({advance:_.uint16,bearing:_.int16}),_n=new _.Struct({metrics:new _.LazyArray(wn,(function(r){return r.parent.vhea.numberOfMetrics})),bearings:new _.LazyArray(_.int16,(function(r){return r.parent.maxp.numGlyphs-r.parent.vhea.numberOfMetrics}))}),Sn=new _.Fixed(16,"BE",14),xn=new _.Struct({fromCoord:Sn,toCoord:Sn}),An=new _.Struct({pairCount:_.uint16,correspondence:new _.Array(xn,"pairCount")}),Cn=new _.Struct({version:_.fixed32,axisCount:_.uint32,segment:new _.Array(An,"axisCount")}),Pn=function(){function UnboundedArrayAccessor(r,o,u){this.type=r,this.stream=o,this.parent=u,this.base=this.stream.pos,this._items=[]}var r=UnboundedArrayAccessor.prototype;return r.getItem=function getItem(r){if(null==this._items[r]){var o=this.stream.pos;this.stream.pos=this.base+this.type.size(null,this.parent)*r,this._items[r]=this.type.decode(this.stream,this.parent),this.stream.pos=o}return this._items[r]},r.inspect=function inspect(){return"[UnboundedArray ".concat(this.type.constructor.name,"]")},UnboundedArrayAccessor}(),kn=function(r){function UnboundedArray(o){return r.call(this,o,0)||this}return _inheritsLoose(UnboundedArray,r),UnboundedArray.prototype.decode=function decode(r,o){return new Pn(this.type,r,o)},UnboundedArray}(_.Array),En=function LookupTable(r){void 0===r&&(r=_.uint16),r=new(function(){function Shadow(r){this.type=r}var r=Shadow.prototype;return r.decode=function decode(r,o){return o=o.parent.parent,this.type.decode(r,o)},r.size=function size(r,o){return o=o.parent.parent,this.type.size(r,o)},r.encode=function encode(r,o,u){return u=u.parent.parent,this.type.encode(r,o,u)},Shadow}())(r);var o=new _.Struct({unitSize:_.uint16,nUnits:_.uint16,searchRange:_.uint16,entrySelector:_.uint16,rangeShift:_.uint16}),u=new _.Struct({lastGlyph:_.uint16,firstGlyph:_.uint16,value:r}),p=new _.Struct({lastGlyph:_.uint16,firstGlyph:_.uint16,values:new _.Pointer(_.uint16,new _.Array(r,(function(r){return r.lastGlyph-r.firstGlyph+1})),{type:"parent"})}),y=new _.Struct({glyph:_.uint16,value:r});return new _.VersionedStruct(_.uint16,{0:{values:new kn(r)},2:{binarySearchHeader:o,segments:new _.Array(u,(function(r){return r.binarySearchHeader.nUnits}))},4:{binarySearchHeader:o,segments:new _.Array(p,(function(r){return r.binarySearchHeader.nUnits}))},6:{binarySearchHeader:o,segments:new _.Array(y,(function(r){return r.binarySearchHeader.nUnits}))},8:{firstGlyph:_.uint16,count:_.uint16,values:new _.Array(r,"count")}})};function StateTable(r,o){void 0===r&&(r={}),void 0===o&&(o=_.uint16);var u=Object.assign({newState:_.uint16,flags:_.uint16},r),p=new _.Struct(u),y=new kn(new _.Array(_.uint16,(function(r){return r.nClasses})));return new _.Struct({nClasses:_.uint32,classTable:new _.Pointer(_.uint32,new En(o)),stateArray:new _.Pointer(_.uint32,y),entryTable:new _.Pointer(_.uint32,new kn(p))})}var Tn=new _.VersionedStruct("format",{0:{deltas:new _.Array(_.int16,32)},1:{deltas:new _.Array(_.int16,32),mappingData:new En(_.uint16)},2:{standardGlyph:_.uint16,controlPoints:new _.Array(_.uint16,32)},3:{standardGlyph:_.uint16,controlPoints:new _.Array(_.uint16,32),mappingData:new En(_.uint16)}}),In=new _.Struct({version:_.fixed32,format:_.uint16,defaultBaseline:_.uint16,subtable:Tn}),On=new _.Struct({setting:_.uint16,nameIndex:_.int16,name:function name(r){return r.parent.parent.parent.name.records.fontFeatures[r.nameIndex]}}),Bn=new _.Struct({feature:_.uint16,nSettings:_.uint16,settingTable:new _.Pointer(_.uint32,new _.Array(On,"nSettings"),{type:"parent"}),featureFlags:new _.Bitfield(_.uint8,[null,null,null,null,null,null,"hasDefault","exclusive"]),defaultSetting:_.uint8,nameIndex:_.int16,name:function name(r){return r.parent.parent.name.records.fontFeatures[r.nameIndex]}}),Fn=new _.Struct({version:_.fixed32,featureNameCount:_.uint16,reserved1:new _.Reserved(_.uint16),reserved2:new _.Reserved(_.uint32),featureNames:new _.Array(Bn,"featureNameCount")}),Ln=new _.Struct({axisTag:new _.String(4),minValue:_.fixed32,defaultValue:_.fixed32,maxValue:_.fixed32,flags:_.uint16,nameID:_.uint16,name:function name(r){return r.parent.parent.name.records.fontFeatures[r.nameID]}}),Dn=new _.Struct({nameID:_.uint16,name:function name(r){return r.parent.parent.name.records.fontFeatures[r.nameID]},flags:_.uint16,coord:new _.Array(_.fixed32,(function(r){return r.parent.axisCount})),postscriptNameID:new _.Optional(_.uint16,(function(r){return r.parent.instanceSize-r._currentOffset>0}))}),Rn=new _.Struct({version:_.fixed32,offsetToData:_.uint16,countSizePairs:_.uint16,axisCount:_.uint16,axisSize:_.uint16,instanceCount:_.uint16,instanceSize:_.uint16,axis:new _.Array(Ln,"axisCount"),instance:new _.Array(Dn,"instanceCount")}),Mn=new _.Fixed(16,"BE",14),Nn=function(){function Offset(){}return Offset.decode=function decode(r,o){return o.flags?r.readUInt32BE():2*r.readUInt16BE()},Offset}(),Un=new _.Struct({version:_.uint16,reserved:new _.Reserved(_.uint16),axisCount:_.uint16,globalCoordCount:_.uint16,globalCoords:new _.Pointer(_.uint32,new _.Array(new _.Array(Mn,"axisCount"),"globalCoordCount")),glyphCount:_.uint16,flags:_.uint16,offsetToData:_.uint32,offsets:new _.Array(new _.Pointer(Nn,"void",{relativeTo:function relativeTo(r){return r.offsetToData},allowNull:!1}),(function(r){return r.glyphCount+1}))}),Wn=new _.Struct({length:_.uint16,coverage:_.uint16,subFeatureFlags:_.uint32,stateTable:new function StateTable1(r,o){void 0===r&&(r={}),void 0===o&&(o=_.uint16);var u=new _.Struct({version:function version(){return 8},firstGlyph:_.uint16,values:new _.Array(_.uint8,_.uint16)}),p=Object.assign({newStateOffset:_.uint16,newState:function newState(r){return(r.newStateOffset-(r.parent.stateArray.base-r.parent._startOffset))/r.parent.nClasses},flags:_.uint16},r),y=new _.Struct(p),P=new kn(new _.Array(_.uint8,(function(r){return r.nClasses})));return new _.Struct({nClasses:_.uint16,classTable:new _.Pointer(_.uint16,u),stateArray:new _.Pointer(_.uint16,P),entryTable:new _.Pointer(_.uint16,new kn(y))})}}),zn=new _.Struct({justClass:_.uint32,beforeGrowLimit:_.fixed32,beforeShrinkLimit:_.fixed32,afterGrowLimit:_.fixed32,afterShrinkLimit:_.fixed32,growFlags:_.uint16,shrinkFlags:_.uint16}),jn=new _.Array(zn,_.uint32),Gn=new _.VersionedStruct("actionType",{0:{lowerLimit:_.fixed32,upperLimit:_.fixed32,order:_.uint16,glyphs:new _.Array(_.uint16,_.uint16)},1:{addGlyph:_.uint16},2:{substThreshold:_.fixed32,addGlyph:_.uint16,substGlyph:_.uint16},3:{},4:{variationAxis:_.uint32,minimumLimit:_.fixed32,noStretchValue:_.fixed32,maximumLimit:_.fixed32},5:{flags:_.uint16,glyph:_.uint16}}),Vn=new _.Struct({actionClass:_.uint16,actionType:_.uint16,actionLength:_.uint32,actionData:Gn,padding:new _.Reserved(_.uint8,(function(r){return r.actionLength-r._currentOffset}))}),Hn=new _.Array(Vn,_.uint32),qn=new _.Struct({lookupTable:new En(new _.Pointer(_.uint16,Hn))}),Xn=new _.Struct({classTable:new _.Pointer(_.uint16,Wn,{type:"parent"}),wdcOffset:_.uint16,postCompensationTable:new _.Pointer(_.uint16,qn,{type:"parent"}),widthDeltaClusters:new En(new _.Pointer(_.uint16,jn,{type:"parent",relativeTo:function relativeTo(r){return r.wdcOffset}}))}),Zn=new _.Struct({version:_.uint32,format:_.uint16,horizontal:new _.Pointer(_.uint16,Xn),vertical:new _.Pointer(_.uint16,Xn)}),Yn={action:_.uint16},Kn={markIndex:_.uint16,currentIndex:_.uint16},Jn={currentInsertIndex:_.uint16,markedInsertIndex:_.uint16},Qn=new _.Struct({items:new kn(new _.Pointer(_.uint32,new En))}),$n=new _.VersionedStruct("type",{0:{stateTable:new StateTable},1:{stateTable:new StateTable(Kn),substitutionTable:new _.Pointer(_.uint32,Qn)},2:{stateTable:new StateTable(Yn),ligatureActions:new _.Pointer(_.uint32,new kn(_.uint32)),components:new _.Pointer(_.uint32,new kn(_.uint16)),ligatureList:new _.Pointer(_.uint32,new kn(_.uint16))},4:{lookupTable:new En},5:{stateTable:new StateTable(Jn),insertionActions:new _.Pointer(_.uint32,new kn(_.uint16))}}),ei=new _.Struct({length:_.uint32,coverage:_.uint24,type:_.uint8,subFeatureFlags:_.uint32,table:$n,padding:new _.Reserved(_.uint8,(function(r){return r.length-r._currentOffset}))}),ti=new _.Struct({featureType:_.uint16,featureSetting:_.uint16,enableFlags:_.uint32,disableFlags:_.uint32}),ri=new _.Struct({defaultFlags:_.uint32,chainLength:_.uint32,nFeatureEntries:_.uint32,nSubtables:_.uint32,features:new _.Array(ti,"nFeatureEntries"),subtables:new _.Array(ei,"nSubtables")}),ni=new _.Struct({version:_.uint16,unused:new _.Reserved(_.uint16),nChains:_.uint32,chains:new _.Array(ri,"nChains")}),ii=new _.Struct({left:_.int16,top:_.int16,right:_.int16,bottom:_.int16}),oi=new _.Struct({version:_.fixed32,format:_.uint16,lookupTable:new En(ii)}),ai={};ai.cmap=ne,ai.head=ie,ai.hhea=oe,ai.hmtx=se,ai.maxp=ce,ai.name=pe,ai["OS/2"]=ye,ai.post=me,ai.fpgm=we,ai.loca=_e,ai.prep=Se,ai["cvt "]=be,ai.glyf=xe,ai["CFF "]=Lt,ai.CFF2=Lt,ai.VORG=Rt,ai.EBLC=Xt,ai.CBLC=ai.EBLC,ai.sbix=Yt,ai.COLR=Qt,ai.CPAL=er,ai.BASE=fr,ai.GDEF=mr,ai.GPOS=Dr,ai.GSUB=zr,ai.JSTF=Zr,ai.HVAR=Qr,ai.DSIG=tn,ai.gasp=nn,ai.hdmx=an,ai.kern=hn,ai.LTSH=dn,ai.PCLT=pn,ai.VDMX=mn,ai.vhea=bn,ai.vmtx=_n,ai.avar=Cn,ai.bsln=In,ai.feat=Fn,ai.fvar=Rn,ai.gvar=Un,ai.just=Zn,ai.morx=ni,ai.opbd=oi;var si,ci=new _.Struct({tag:new _.String(4),checkSum:_.uint32,offset:new _.Pointer(_.uint32,"void",{type:"global"}),length:_.uint32}),li=new _.Struct({tag:new _.String(4),numTables:_.uint16,searchRange:_.uint16,entrySelector:_.uint16,rangeShift:_.uint16,tables:new _.Array(ci,"numTables")});function binarySearch(r,o){for(var u=0,p=r.length-1;u<=p;){var y=u+p>>1,_=o(r[y]);if(_<0)p=y-1;else{if(!(_>0))return y;u=y+1}}return-1}function range(r,o){for(var u=[];r<o;)u.push(r++);return u}li.process=function(){for(var r,o={},u=_createForOfIteratorHelperLoose(this.tables);!(r=u()).done;){var p=r.value;o[p.tag]=p}this.tables=o},li.preEncode=function(r){var o=[];for(var u in this.tables){var p=this.tables[u];p&&o.push({tag:u,checkSum:0,offset:new _.VoidPointer(ai[u],p),length:ai[u].size(p)})}this.tag="true",this.numTables=o.length,this.tables=o;var y=Math.floor(Math.log(this.numTables)/Math.LN2),P=Math.pow(2,y);this.searchRange=16*P,this.entrySelector=Math.log(P)/Math.LN2,this.rangeShift=16*this.numTables-this.searchRange};try{var ui=u(54171)}catch(r){}var fi=(si=function(){function CmapProcessor(r){if(this.encoding=null,this.cmap=this.findSubtable(r,[[3,10],[0,6],[0,4],[3,1],[0,3],[0,2],[0,1],[0,0]]),!this.cmap&&ui)for(var o,u=_createForOfIteratorHelperLoose(r.tables);!(o=u()).done;){var p=o.value,y=getEncoding(p.platformID,p.encodingID,p.table.language-1);ui.encodingExists(y)&&(this.cmap=p.table,this.encoding=y)}if(!this.cmap)throw new Error("Could not find a supported cmap table");this.uvs=this.findSubtable(r,[[0,5]]),this.uvs&&14!==this.uvs.version&&(this.uvs=null)}var r=CmapProcessor.prototype;return r.findSubtable=function findSubtable(r,o){for(var u,p=_createForOfIteratorHelperLoose(o);!(u=p()).done;)for(var y,_=u.value,P=_[0],k=_[1],I=_createForOfIteratorHelperLoose(r.tables);!(y=I()).done;){var O=y.value;if(O.platformID===P&&O.encodingID===k)return O.table}return null},r.lookup=function lookup(r,o){if(this.encoding){var u=ui.encode(String.fromCodePoint(r),this.encoding);r=0;for(var p=0;p<u.length;p++)r=r<<8|u[p]}else if(o){var y=this.getVariationSelector(r,o);if(y)return y}var _=this.cmap;switch(_.version){case 0:return _.codeMap.get(r)||0;case 4:for(var P=0,k=_.segCount-1;P<=k;){var I=P+k>>1;if(r<_.startCode.get(I))k=I-1;else{if(!(r>_.endCode.get(I))){var O=_.idRangeOffset.get(I),B=void 0;if(0===O)B=r+_.idDelta.get(I);else{var D=O/2+(r-_.startCode.get(I))-(_.segCount-I);0!==(B=_.glyphIndexArray.get(D)||0)&&(B+=_.idDelta.get(I))}return 65535&B}P=I+1}}return 0;case 8:throw new Error("TODO: cmap format 8");case 6:case 10:return _.glyphIndices.get(r-_.firstCode)||0;case 12:case 13:for(var R=0,N=_.nGroups-1;R<=N;){var U=R+N>>1,W=_.groups.get(U);if(r<W.startCharCode)N=U-1;else{if(!(r>W.endCharCode))return 12===_.version?W.glyphID+(r-W.startCharCode):W.glyphID;R=U+1}}return 0;case 14:throw new Error("TODO: cmap format 14");default:throw new Error("Unknown cmap format ".concat(_.version))}},r.getVariationSelector=function getVariationSelector(r,o){if(!this.uvs)return 0;var u=this.uvs.varSelectors.toArray(),p=binarySearch(u,(function(r){return o-r.varSelector})),y=u[p];return-1!==p&&y.defaultUVS&&(p=binarySearch(y.defaultUVS,(function(o){return r<o.startUnicodeValue?-1:r>o.startUnicodeValue+o.additionalCount?1:0}))),-1!==p&&y.nonDefaultUVS&&-1!==(p=binarySearch(y.nonDefaultUVS,(function(o){return r-o.unicodeValue})))?y.nonDefaultUVS[p].glyphID:0},r.getCharacterSet=function getCharacterSet(){var r=this.cmap;switch(r.version){case 0:return range(0,r.codeMap.length);case 4:for(var o=[],u=r.endCode.toArray(),p=0;p<u.length;p++){var y=u[p]+1,_=r.startCode.get(p);o.push.apply(o,range(_,y))}return o;case 8:throw new Error("TODO: cmap format 8");case 6:case 10:return range(r.firstCode,r.firstCode+r.glyphIndices.length);case 12:case 13:for(var P,k=[],I=_createForOfIteratorHelperLoose(r.groups.toArray());!(P=I()).done;){var O=P.value;k.push.apply(k,range(O.startCharCode,O.endCharCode+1))}return k;case 14:throw new Error("TODO: cmap format 14");default:throw new Error("Unknown cmap format ".concat(r.version))}},r.codePointsForGlyph=function codePointsForGlyph(r){var o=this.cmap;switch(o.version){case 0:for(var u=[],p=0;p<256;p++)o.codeMap.get(p)===r&&u.push(p);return u;case 4:for(var y=[],_=0;_<o.segCount;_++)for(var P=o.endCode.get(_),k=o.startCode.get(_),I=o.idRangeOffset.get(_),O=o.idDelta.get(_),B=k;B<=P;B++){var D=0;if(0===I)D=B+O;else{var R=I/2+(B-k)-(o.segCount-_);0!==(D=o.glyphIndexArray.get(R)||0)&&(D+=O)}D===r&&y.push(B)}return y;case 12:for(var N,U=[],W=_createForOfIteratorHelperLoose(o.groups.toArray());!(N=W()).done;){var j=N.value;r>=j.glyphID&&r<=j.glyphID+(j.endCharCode-j.startCharCode)&&U.push(j.startCharCode+(r-j.glyphID))}return U;case 13:for(var G,X=[],Y=_createForOfIteratorHelperLoose(o.groups.toArray());!(G=Y()).done;){var K=G.value;r===K.glyphID&&X.push.apply(X,range(K.startCharCode,K.endCharCode+1))}return X;default:throw new Error("Unknown cmap format ".concat(o.version))}},CmapProcessor}(),_applyDecoratedDescriptor(si.prototype,"getCharacterSet",[cache],Object.getOwnPropertyDescriptor(si.prototype,"getCharacterSet"),si.prototype),_applyDecoratedDescriptor(si.prototype,"codePointsForGlyph",[cache],Object.getOwnPropertyDescriptor(si.prototype,"codePointsForGlyph"),si.prototype),si),hi=function(){function KernProcessor(r){this.kern=r.kern}var r=KernProcessor.prototype;return r.process=function process(r,o){for(var u=0;u<r.length-1;u++){var p=r[u].id,y=r[u+1].id;o[u].xAdvance+=this.getKerning(p,y)}},r.getKerning=function getKerning(r,o){for(var u,p=0,y=_createForOfIteratorHelperLoose(this.kern.tables);!(u=y()).done;){var _=u.value;if(!_.coverage.crossStream){switch(_.version){case 0:if(!_.coverage.horizontal)continue;break;case 1:if(_.coverage.vertical||_.coverage.variation)continue;break;default:throw new Error("Unsupported kerning table version ".concat(_.version))}var P=0,k=_.subtable;switch(_.format){case 0:var I=binarySearch(k.pairs,(function(u){return r-u.left||o-u.right}));I>=0&&(P=k.pairs[I].value);break;case 2:var O=0,B=0;O=r>=k.leftTable.firstGlyph&&r<k.leftTable.firstGlyph+k.leftTable.nGlyphs?k.leftTable.offsets[r-k.leftTable.firstGlyph]:k.array.off,o>=k.rightTable.firstGlyph&&o<k.rightTable.firstGlyph+k.rightTable.nGlyphs&&(B=k.rightTable.offsets[o-k.rightTable.firstGlyph]);var D=(O+B-k.array.off)/2;P=k.array.values.get(D);break;case 3:if(r>=k.glyphCount||o>=k.glyphCount)return 0;P=k.kernValue[k.kernIndex[k.leftClass[r]*k.rightClassCount+k.rightClass[o]]];break;default:throw new Error("Unsupported kerning sub-table format ".concat(_.format))}_.coverage.override?p=P:p+=P}}return p},KernProcessor}(),di=function(){function UnicodeLayoutEngine(r){this.font=r}var r=UnicodeLayoutEngine.prototype;return r.positionGlyphs=function positionGlyphs(r,o){for(var u=0,p=0,y=0;y<r.length;y++){r[y].isMark?p=y:(u!==p&&this.positionCluster(r,o,u,p),u=p=y)}return u!==p&&this.positionCluster(r,o,u,p),o},r.positionCluster=function positionCluster(r,o,u,p){var y=r[u],_=y.cbox.copy();y.codePoints.length>1&&(_.minX+=(y.codePoints.length-1)*_.width/y.codePoints.length);for(var P=-o[u].xAdvance,k=0,I=this.font.unitsPerEm/16,O=u+1;O<=p;O++){var B=r[O],D=B.cbox,R=o[O],N=this.getCombiningClass(B.codePoints[0]);if("Not_Reordered"!==N){switch(R.xOffset=R.yOffset=0,N){case"Double_Above":case"Double_Below":R.xOffset+=_.minX-D.width/2-D.minX;break;case"Attached_Below_Left":case"Below_Left":case"Above_Left":R.xOffset+=_.minX-D.minX;break;case"Attached_Above_Right":case"Below_Right":case"Above_Right":R.xOffset+=_.maxX-D.width-D.minX;break;default:R.xOffset+=_.minX+(_.width-D.width)/2-D.minX}switch(N){case"Double_Below":case"Below_Left":case"Below":case"Below_Right":case"Attached_Below_Left":case"Attached_Below":"Attached_Below_Left"!==N&&"Attached_Below"!==N||(_.minY+=I),R.yOffset=-_.minY-D.maxY,_.minY+=D.height;break;case"Double_Above":case"Above_Left":case"Above":case"Above_Right":case"Attached_Above":case"Attached_Above_Right":"Attached_Above"!==N&&"Attached_Above_Right"!==N||(_.maxY+=I),R.yOffset=_.maxY-D.minY,_.maxY+=D.height}R.xAdvance=R.yAdvance=0,R.xOffset+=P,R.yOffset+=k}else P-=R.xAdvance,k-=R.yAdvance}},r.getCombiningClass=function getCombiningClass(r){var o=I.getCombiningClass(r);if(3584==(-256&r))if("Not_Reordered"===o)switch(r){case 3633:case 3636:case 3637:case 3638:case 3639:case 3655:case 3660:case 3645:case 3662:return"Above_Right";case 3761:case 3764:case 3765:case 3766:case 3767:case 3771:case 3788:case 3789:return"Above";case 3772:return"Below"}else if(3642===r)return"Below_Right";switch(o){case"CCC10":case"CCC11":case"CCC12":case"CCC13":case"CCC14":case"CCC15":case"CCC16":case"CCC17":case"CCC18":case"CCC20":case"CCC22":case"CCC29":case"CCC32":case"CCC118":case"CCC129":case"CCC132":return"Below";case"CCC23":return"Attached_Above";case"CCC24":case"CCC107":return"Above_Right";case"CCC25":case"CCC19":return"Above_Left";case"CCC26":case"CCC27":case"CCC28":case"CCC30":case"CCC31":case"CCC33":case"CCC34":case"CCC35":case"CCC36":case"CCC122":case"CCC130":return"Above";case"CCC21":break;case"CCC103":return"Below_Right"}return o},UnicodeLayoutEngine}(),pi=function(){function BBox(r,o,u,p){void 0===r&&(r=1/0),void 0===o&&(o=1/0),void 0===u&&(u=-1/0),void 0===p&&(p=-1/0),this.minX=r,this.minY=o,this.maxX=u,this.maxY=p}var r=BBox.prototype;return r.addPoint=function addPoint(r,o){Math.abs(r)!==1/0&&(r<this.minX&&(this.minX=r),r>this.maxX&&(this.maxX=r)),Math.abs(o)!==1/0&&(o<this.minY&&(this.minY=o),o>this.maxY&&(this.maxY=o))},r.copy=function copy(){return new BBox(this.minX,this.minY,this.maxX,this.maxY)},_createClass(BBox,[{key:"width",get:function get(){return this.maxX-this.minX}},{key:"height",get:function get(){return this.maxY-this.minY}}])}(),gi={Caucasian_Albanian:"aghb",Arabic:"arab",Imperial_Aramaic:"armi",Armenian:"armn",Avestan:"avst",Balinese:"bali",Bamum:"bamu",Bassa_Vah:"bass",Batak:"batk",Bengali:["bng2","beng"],Bopomofo:"bopo",Brahmi:"brah",Braille:"brai",Buginese:"bugi",Buhid:"buhd",Chakma:"cakm",Canadian_Aboriginal:"cans",Carian:"cari",Cham:"cham",Cherokee:"cher",Coptic:"copt",Cypriot:"cprt",Cyrillic:"cyrl",Devanagari:["dev2","deva"],Deseret:"dsrt",Duployan:"dupl",Egyptian_Hieroglyphs:"egyp",Elbasan:"elba",Ethiopic:"ethi",Georgian:"geor",Glagolitic:"glag",Gothic:"goth",Grantha:"gran",Greek:"grek",Gujarati:["gjr2","gujr"],Gurmukhi:["gur2","guru"],Hangul:"hang",Han:"hani",Hanunoo:"hano",Hebrew:"hebr",Hiragana:"hira",Pahawh_Hmong:"hmng",Katakana_Or_Hiragana:"hrkt",Old_Italic:"ital",Javanese:"java",Kayah_Li:"kali",Katakana:"kana",Kharoshthi:"khar",Khmer:"khmr",Khojki:"khoj",Kannada:["knd2","knda"],Kaithi:"kthi",Tai_Tham:"lana",Lao:"lao ",Latin:"latn",Lepcha:"lepc",Limbu:"limb",Linear_A:"lina",Linear_B:"linb",Lisu:"lisu",Lycian:"lyci",Lydian:"lydi",Mahajani:"mahj",Mandaic:"mand",Manichaean:"mani",Mende_Kikakui:"mend",Meroitic_Cursive:"merc",Meroitic_Hieroglyphs:"mero",Malayalam:["mlm2","mlym"],Modi:"modi",Mongolian:"mong",Mro:"mroo",Meetei_Mayek:"mtei",Myanmar:["mym2","mymr"],Old_North_Arabian:"narb",Nabataean:"nbat",Nko:"nko ",Ogham:"ogam",Ol_Chiki:"olck",Old_Turkic:"orkh",Oriya:["ory2","orya"],Osmanya:"osma",Palmyrene:"palm",Pau_Cin_Hau:"pauc",Old_Permic:"perm",Phags_Pa:"phag",Inscriptional_Pahlavi:"phli",Psalter_Pahlavi:"phlp",Phoenician:"phnx",Miao:"plrd",Inscriptional_Parthian:"prti",Rejang:"rjng",Runic:"runr",Samaritan:"samr",Old_South_Arabian:"sarb",Saurashtra:"saur",Shavian:"shaw",Sharada:"shrd",Siddham:"sidd",Khudawadi:"sind",Sinhala:"sinh",Sora_Sompeng:"sora",Sundanese:"sund",Syloti_Nagri:"sylo",Syriac:"syrc",Tagbanwa:"tagb",Takri:"takr",Tai_Le:"tale",New_Tai_Lue:"talu",Tamil:["tml2","taml"],Tai_Viet:"tavt",Telugu:["tel2","telu"],Tifinagh:"tfng",Tagalog:"tglg",Thaana:"thaa",Thai:"thai",Tibetan:"tibt",Tirhuta:"tirh",Ugaritic:"ugar",Vai:"vai ",Warang_Citi:"wara",Old_Persian:"xpeo",Cuneiform:"xsux",Yi:"yi  ",Inherited:"zinh",Common:"zyyy",Unknown:"zzzz"},yi={};for(var vi in gi){var mi=gi[vi];if(Array.isArray(mi))for(var bi,wi=_createForOfIteratorHelperLoose(mi);!(bi=wi()).done;){var _i=bi.value;yi[_i]=vi}else yi[mi]=vi}var Si={arab:!0,hebr:!0,syrc:!0,thaa:!0,cprt:!0,khar:!0,phnx:!0,"nko ":!0,lydi:!0,avst:!0,armi:!0,phli:!0,prti:!0,sarb:!0,orkh:!0,samr:!0,mand:!0,merc:!0,mero:!0,mani:!0,mend:!0,nbat:!0,narb:!0,palm:!0,phlp:!0};function direction(r){return Si[r]?"rtl":"ltr"}for(var xi=function(){return _createClass((function GlyphRun(r,o,u,p,y){if(this.glyphs=r,this.positions=null,this.script=u,this.language=p||null,this.direction=y||direction(u),this.features={},Array.isArray(o))for(var _,P=_createForOfIteratorHelperLoose(o);!(_=P()).done;){var k=_.value;this.features[k]=!0}else"object"==typeof o&&(this.features=o)}),[{key:"advanceWidth",get:function get(){for(var r,o=0,u=_createForOfIteratorHelperLoose(this.positions);!(r=u()).done;){o+=r.value.xAdvance}return o}},{key:"advanceHeight",get:function get(){for(var r,o=0,u=_createForOfIteratorHelperLoose(this.positions);!(r=u()).done;){o+=r.value.yAdvance}return o}},{key:"bbox",get:function get(){for(var r=new pi,o=0,u=0,p=0;p<this.glyphs.length;p++){var y=this.glyphs[p],_=this.positions[p],P=y.bbox;r.addPoint(P.minX+o+_.xOffset,P.minY+u+_.yOffset),r.addPoint(P.maxX+o+_.xOffset,P.maxY+u+_.yOffset),o+=_.xAdvance,u+=_.yAdvance}return r}}])}(),Ai=function GlyphPosition(r,o,u,p){void 0===r&&(r=0),void 0===o&&(o=0),void 0===u&&(u=0),void 0===p&&(p=0),this.xAdvance=r,this.yAdvance=o,this.xOffset=u,this.yOffset=p},Ci={allTypographicFeatures:{code:0,exclusive:!1,allTypeFeatures:0},ligatures:{code:1,exclusive:!1,requiredLigatures:0,commonLigatures:2,rareLigatures:4,rebusPictures:8,diphthongLigatures:10,squaredLigatures:12,abbrevSquaredLigatures:14,symbolLigatures:16,contextualLigatures:18,historicalLigatures:20},cursiveConnection:{code:2,exclusive:!0,unconnected:0,partiallyConnected:1,cursive:2},letterCase:{code:3,exclusive:!0},verticalSubstitution:{code:4,exclusive:!1,substituteVerticalForms:0},linguisticRearrangement:{code:5,exclusive:!1,linguisticRearrangement:0},numberSpacing:{code:6,exclusive:!0,monospacedNumbers:0,proportionalNumbers:1,thirdWidthNumbers:2,quarterWidthNumbers:3},smartSwash:{code:8,exclusive:!1,wordInitialSwashes:0,wordFinalSwashes:2,nonFinalSwashes:8},diacritics:{code:9,exclusive:!0,showDiacritics:0,hideDiacritics:1,decomposeDiacritics:2},verticalPosition:{code:10,exclusive:!0,normalPosition:0,superiors:1,inferiors:2,ordinals:3,scientificInferiors:4},fractions:{code:11,exclusive:!0,noFractions:0,verticalFractions:1,diagonalFractions:2},overlappingCharacters:{code:13,exclusive:!1,preventOverlap:0},typographicExtras:{code:14,exclusive:!1,slashedZero:4},mathematicalExtras:{code:15,exclusive:!1,mathematicalGreek:10},ornamentSets:{code:16,exclusive:!0,noOrnaments:0,dingbats:1,piCharacters:2,fleurons:3,decorativeBorders:4,internationalSymbols:5,mathSymbols:6},characterAlternatives:{code:17,exclusive:!0,noAlternates:0},designComplexity:{code:18,exclusive:!0,designLevel1:0,designLevel2:1,designLevel3:2,designLevel4:3,designLevel5:4},styleOptions:{code:19,exclusive:!0,noStyleOptions:0,displayText:1,engravedText:2,illuminatedCaps:3,titlingCaps:4,tallCaps:5},characterShape:{code:20,exclusive:!0,traditionalCharacters:0,simplifiedCharacters:1,JIS1978Characters:2,JIS1983Characters:3,JIS1990Characters:4,traditionalAltOne:5,traditionalAltTwo:6,traditionalAltThree:7,traditionalAltFour:8,traditionalAltFive:9,expertCharacters:10,JIS2004Characters:11,hojoCharacters:12,NLCCharacters:13,traditionalNamesCharacters:14},numberCase:{code:21,exclusive:!0,lowerCaseNumbers:0,upperCaseNumbers:1},textSpacing:{code:22,exclusive:!0,proportionalText:0,monospacedText:1,halfWidthText:2,thirdWidthText:3,quarterWidthText:4,altProportionalText:5,altHalfWidthText:6},transliteration:{code:23,exclusive:!0,noTransliteration:0},annotation:{code:24,exclusive:!0,noAnnotation:0,boxAnnotation:1,roundedBoxAnnotation:2,circleAnnotation:3,invertedCircleAnnotation:4,parenthesisAnnotation:5,periodAnnotation:6,romanNumeralAnnotation:7,diamondAnnotation:8,invertedBoxAnnotation:9,invertedRoundedBoxAnnotation:10},kanaSpacing:{code:25,exclusive:!0,fullWidthKana:0,proportionalKana:1},ideographicSpacing:{code:26,exclusive:!0,fullWidthIdeographs:0,proportionalIdeographs:1,halfWidthIdeographs:2},unicodeDecomposition:{code:27,exclusive:!1,canonicalComposition:0,compatibilityComposition:2,transcodingComposition:4},rubyKana:{code:28,exclusive:!1,rubyKana:2},CJKSymbolAlternatives:{code:29,exclusive:!0,noCJKSymbolAlternatives:0,CJKSymbolAltOne:1,CJKSymbolAltTwo:2,CJKSymbolAltThree:3,CJKSymbolAltFour:4,CJKSymbolAltFive:5},ideographicAlternatives:{code:30,exclusive:!0,noIdeographicAlternatives:0,ideographicAltOne:1,ideographicAltTwo:2,ideographicAltThree:3,ideographicAltFour:4,ideographicAltFive:5},CJKVerticalRomanPlacement:{code:31,exclusive:!0,CJKVerticalRomanCentered:0,CJKVerticalRomanHBaseline:1},italicCJKRoman:{code:32,exclusive:!1,CJKItalicRoman:2},caseSensitiveLayout:{code:33,exclusive:!1,caseSensitiveLayout:0,caseSensitiveSpacing:2},alternateKana:{code:34,exclusive:!1,alternateHorizKana:0,alternateVertKana:2},stylisticAlternatives:{code:35,exclusive:!1,noStylisticAlternates:0,stylisticAltOne:2,stylisticAltTwo:4,stylisticAltThree:6,stylisticAltFour:8,stylisticAltFive:10,stylisticAltSix:12,stylisticAltSeven:14,stylisticAltEight:16,stylisticAltNine:18,stylisticAltTen:20,stylisticAltEleven:22,stylisticAltTwelve:24,stylisticAltThirteen:26,stylisticAltFourteen:28,stylisticAltFifteen:30,stylisticAltSixteen:32,stylisticAltSeventeen:34,stylisticAltEighteen:36,stylisticAltNineteen:38,stylisticAltTwenty:40},contextualAlternates:{code:36,exclusive:!1,contextualAlternates:0,swashAlternates:2,contextualSwashAlternates:4},lowerCase:{code:37,exclusive:!0,defaultLowerCase:0,lowerCaseSmallCaps:1,lowerCasePetiteCaps:2},upperCase:{code:38,exclusive:!0,defaultUpperCase:0,upperCaseSmallCaps:1,upperCasePetiteCaps:2},languageTag:{code:39,exclusive:!0},CJKRomanSpacing:{code:103,exclusive:!0,halfWidthCJKRoman:0,proportionalCJKRoman:1,defaultCJKRoman:2,fullWidthCJKRoman:3}},Pi=function feature(r,o){return[Ci[r].code,Ci[r][o]]},ki={rlig:Pi("ligatures","requiredLigatures"),clig:Pi("ligatures","contextualLigatures"),dlig:Pi("ligatures","rareLigatures"),hlig:Pi("ligatures","historicalLigatures"),liga:Pi("ligatures","commonLigatures"),hist:Pi("ligatures","historicalLigatures"),smcp:Pi("lowerCase","lowerCaseSmallCaps"),pcap:Pi("lowerCase","lowerCasePetiteCaps"),frac:Pi("fractions","diagonalFractions"),dnom:Pi("fractions","diagonalFractions"),numr:Pi("fractions","diagonalFractions"),afrc:Pi("fractions","verticalFractions"),case:Pi("caseSensitiveLayout","caseSensitiveLayout"),ccmp:Pi("unicodeDecomposition","canonicalComposition"),cpct:Pi("CJKVerticalRomanPlacement","CJKVerticalRomanCentered"),valt:Pi("CJKVerticalRomanPlacement","CJKVerticalRomanCentered"),swsh:Pi("contextualAlternates","swashAlternates"),cswh:Pi("contextualAlternates","contextualSwashAlternates"),curs:Pi("cursiveConnection","cursive"),c2pc:Pi("upperCase","upperCasePetiteCaps"),c2sc:Pi("upperCase","upperCaseSmallCaps"),init:Pi("smartSwash","wordInitialSwashes"),fin2:Pi("smartSwash","wordFinalSwashes"),medi:Pi("smartSwash","nonFinalSwashes"),med2:Pi("smartSwash","nonFinalSwashes"),fin3:Pi("smartSwash","wordFinalSwashes"),fina:Pi("smartSwash","wordFinalSwashes"),pkna:Pi("kanaSpacing","proportionalKana"),half:Pi("textSpacing","halfWidthText"),halt:Pi("textSpacing","altHalfWidthText"),hkna:Pi("alternateKana","alternateHorizKana"),vkna:Pi("alternateKana","alternateVertKana"),ital:Pi("italicCJKRoman","CJKItalicRoman"),lnum:Pi("numberCase","upperCaseNumbers"),onum:Pi("numberCase","lowerCaseNumbers"),mgrk:Pi("mathematicalExtras","mathematicalGreek"),calt:Pi("contextualAlternates","contextualAlternates"),vrt2:Pi("verticalSubstitution","substituteVerticalForms"),vert:Pi("verticalSubstitution","substituteVerticalForms"),tnum:Pi("numberSpacing","monospacedNumbers"),pnum:Pi("numberSpacing","proportionalNumbers"),sups:Pi("verticalPosition","superiors"),subs:Pi("verticalPosition","inferiors"),ordn:Pi("verticalPosition","ordinals"),pwid:Pi("textSpacing","proportionalText"),hwid:Pi("textSpacing","halfWidthText"),qwid:Pi("textSpacing","quarterWidthText"),twid:Pi("textSpacing","thirdWidthText"),fwid:Pi("textSpacing","proportionalText"),palt:Pi("textSpacing","altProportionalText"),trad:Pi("characterShape","traditionalCharacters"),smpl:Pi("characterShape","simplifiedCharacters"),jp78:Pi("characterShape","JIS1978Characters"),jp83:Pi("characterShape","JIS1983Characters"),jp90:Pi("characterShape","JIS1990Characters"),jp04:Pi("characterShape","JIS2004Characters"),expt:Pi("characterShape","expertCharacters"),hojo:Pi("characterShape","hojoCharacters"),nlck:Pi("characterShape","NLCCharacters"),tnam:Pi("characterShape","traditionalNamesCharacters"),ruby:Pi("rubyKana","rubyKana"),titl:Pi("styleOptions","titlingCaps"),zero:Pi("typographicExtras","slashedZero"),ss01:Pi("stylisticAlternatives","stylisticAltOne"),ss02:Pi("stylisticAlternatives","stylisticAltTwo"),ss03:Pi("stylisticAlternatives","stylisticAltThree"),ss04:Pi("stylisticAlternatives","stylisticAltFour"),ss05:Pi("stylisticAlternatives","stylisticAltFive"),ss06:Pi("stylisticAlternatives","stylisticAltSix"),ss07:Pi("stylisticAlternatives","stylisticAltSeven"),ss08:Pi("stylisticAlternatives","stylisticAltEight"),ss09:Pi("stylisticAlternatives","stylisticAltNine"),ss10:Pi("stylisticAlternatives","stylisticAltTen"),ss11:Pi("stylisticAlternatives","stylisticAltEleven"),ss12:Pi("stylisticAlternatives","stylisticAltTwelve"),ss13:Pi("stylisticAlternatives","stylisticAltThirteen"),ss14:Pi("stylisticAlternatives","stylisticAltFourteen"),ss15:Pi("stylisticAlternatives","stylisticAltFifteen"),ss16:Pi("stylisticAlternatives","stylisticAltSixteen"),ss17:Pi("stylisticAlternatives","stylisticAltSeventeen"),ss18:Pi("stylisticAlternatives","stylisticAltEighteen"),ss19:Pi("stylisticAlternatives","stylisticAltNineteen"),ss20:Pi("stylisticAlternatives","stylisticAltTwenty")},Ei=1;Ei<=99;Ei++)ki["cv".concat("00".concat(Ei).slice(-2))]=[Ci.characterAlternatives.code,Ei];var Ti,Ii={};for(var Oi in ki){var Bi=ki[Oi];null==Ii[Bi[0]]&&(Ii[Bi[0]]={}),Ii[Bi[0]][Bi[1]]=Oi}function mapFeatureStrings(r){var o=r[0],u=r[1];if(isNaN(o))var p=Ci[o]&&Ci[o].code;else p=o;if(isNaN(u))var y=Ci[o]&&Ci[o][u];else y=u;return[p,y]}var Fi,Li=(Ti=function(){function AATLookupTable(r){this.table=r}var r=AATLookupTable.prototype;return r.lookup=function lookup(r){switch(this.table.version){case 0:return this.table.values.getItem(r);case 2:case 4:for(var o=0,u=this.table.binarySearchHeader.nUnits-1;o<=u;){var p=o+u>>1;if(65535===(P=this.table.segments[p]).firstGlyph)return null;if(r<P.firstGlyph)u=p-1;else{if(!(r>P.lastGlyph))return 2===this.table.version?P.value:P.values[r-P.firstGlyph];o=p+1}}return null;case 6:for(var y=0,_=this.table.binarySearchHeader.nUnits-1;y<=_;){var P;p=y+_>>1;if(65535===(P=this.table.segments[p]).glyph)return null;if(r<P.glyph)_=p-1;else{if(!(r>P.glyph))return P.value;y=p+1}}return null;case 8:return this.table.values[r-this.table.firstGlyph];default:throw new Error("Unknown lookup table format: ".concat(this.table.version))}},r.glyphsForValue=function glyphsForValue(r){var o=[];switch(this.table.version){case 2:case 4:for(var u,p=_createForOfIteratorHelperLoose(this.table.segments);!(u=p()).done;){var y=u.value;if(2===this.table.version&&y.value===r)o.push.apply(o,range(y.firstGlyph,y.lastGlyph+1));else for(var _=0;_<y.values.length;_++)y.values[_]===r&&o.push(y.firstGlyph+_)}break;case 6:for(var P,k=_createForOfIteratorHelperLoose(this.table.segments);!(P=k()).done;){var I=P.value;I.value===r&&o.push(I.glyph)}break;case 8:for(var O=0;O<this.table.values.length;O++)this.table.values[O]===r&&o.push(this.table.firstGlyph+O);break;default:throw new Error("Unknown lookup table format: ".concat(this.table.version))}return o},AATLookupTable}(),_applyDecoratedDescriptor(Ti.prototype,"glyphsForValue",[cache],Object.getOwnPropertyDescriptor(Ti.prototype,"glyphsForValue"),Ti.prototype),Ti),Di=function(){function AATStateMachine(r){this.stateTable=r,this.lookupTable=new Li(r.classTable)}var r=AATStateMachine.prototype;return r.process=function process(r,o,u){for(var p=0,y=o?r.length-1:0,_=o?-1:1;1===_&&y<=r.length||-1===_&&y>=-1;){var P=null,k=1,I=!0;y===r.length||-1===y?k=0:65535===(P=r[y]).id?k=2:null==(k=this.lookupTable.lookup(P.id))&&(k=1);var O=this.stateTable.stateArray.getItem(p)[k],B=this.stateTable.entryTable.getItem(O);0!==k&&2!==k&&(u(P,B,y),I=!(16384&B.flags)),p=B.newState,I&&(y+=_)}return r},r.traverse=function traverse(r,o,u){if(void 0===o&&(o=0),void 0===u&&(u=new Set),!u.has(o)){u.add(o);for(var p=this.stateTable,y=p.nClasses,_=p.stateArray,P=p.entryTable,k=_.getItem(o),I=4;I<y;I++)for(var O,B=k[I],D=P.getItem(B),R=_createForOfIteratorHelperLoose(this.lookupTable.glyphsForValue(I));!(O=R()).done;){var N=O.value;r.enter&&r.enter(N,D),0!==D.newState&&this.traverse(r,D.newState,u),r.exit&&r.exit(N,D)}}},AATStateMachine}(),Ri=4194304,Mi=(Fi=function(){function AATMorxProcessor(r){this.processIndicRearragement=this.processIndicRearragement.bind(this),this.processContextualSubstitution=this.processContextualSubstitution.bind(this),this.processLigature=this.processLigature.bind(this),this.processNoncontextualSubstitutions=this.processNoncontextualSubstitutions.bind(this),this.processGlyphInsertion=this.processGlyphInsertion.bind(this),this.font=r,this.morx=r.morx,this.inputCache=null}var r=AATMorxProcessor.prototype;return r.process=function process(r,o){void 0===o&&(o={});for(var u,p=_createForOfIteratorHelperLoose(this.morx.chains);!(u=p()).done;){for(var y,_=u.value,P=_.defaultFlags,k=_createForOfIteratorHelperLoose(_.features);!(y=k()).done;){var I,O=y.value;(I=o[O.featureType])&&(I[O.featureSetting]?(P&=O.disableFlags,P|=O.enableFlags):!1===I[O.featureSetting]&&(P|=~O.disableFlags,P&=~O.enableFlags))}for(var B,D=_createForOfIteratorHelperLoose(_.subtables);!(B=D()).done;){var R=B.value;R.subFeatureFlags&P&&this.processSubtable(R,r)}}for(var N=r.length-1;N>=0;)65535===r[N].id&&r.splice(N,1),N--;return r},r.processSubtable=function processSubtable(r,o){if(this.subtable=r,this.glyphs=o,4!==this.subtable.type){this.ligatureStack=[],this.markedGlyph=null,this.firstGlyph=null,this.lastGlyph=null,this.markedIndex=null;var u=this.getStateMachine(r),p=this.getProcessor(),y=!!(this.subtable.coverage&Ri);return u.process(this.glyphs,y,p)}this.processNoncontextualSubstitutions(this.subtable,this.glyphs)},r.getStateMachine=function getStateMachine(r){return new Di(r.table.stateTable)},r.getProcessor=function getProcessor(){switch(this.subtable.type){case 0:return this.processIndicRearragement;case 1:return this.processContextualSubstitution;case 2:return this.processLigature;case 4:return this.processNoncontextualSubstitutions;case 5:return this.processGlyphInsertion;default:throw new Error("Invalid morx subtable type: ".concat(this.subtable.type))}},r.processIndicRearragement=function processIndicRearragement(r,o,u){32768&o.flags&&(this.firstGlyph=u),8192&o.flags&&(this.lastGlyph=u),function reorderGlyphs(r,o,u,p){switch(o){case 0:return r;case 1:return swap(r,[u,1],[p,0]);case 2:return swap(r,[u,0],[p,1]);case 3:return swap(r,[u,1],[p,1]);case 4:return swap(r,[u,2],[p,0]);case 5:return swap(r,[u,2],[p,0],!0,!1);case 6:return swap(r,[u,0],[p,2]);case 7:return swap(r,[u,0],[p,2],!1,!0);case 8:return swap(r,[u,1],[p,2]);case 9:return swap(r,[u,1],[p,2],!1,!0);case 10:return swap(r,[u,2],[p,1]);case 11:return swap(r,[u,2],[p,1],!0,!1);case 12:return swap(r,[u,2],[p,2]);case 13:return swap(r,[u,2],[p,2],!0,!1);case 14:return swap(r,[u,2],[p,2],!1,!0);case 15:return swap(r,[u,2],[p,2],!0,!0);default:throw new Error("Unknown verb: ".concat(o))}}(this.glyphs,15&o.flags,this.firstGlyph,this.lastGlyph)},r.processContextualSubstitution=function processContextualSubstitution(r,o,u){var p=this.subtable.table.substitutionTable.items;if(65535!==o.markIndex){var y=p.getItem(o.markIndex),_=new Li(y);r=this.glyphs[this.markedGlyph],(P=_.lookup(r.id))&&(this.glyphs[this.markedGlyph]=this.font.getGlyph(P,r.codePoints))}if(65535!==o.currentIndex){var P,k=p.getItem(o.currentIndex),I=new Li(k);r=this.glyphs[u],(P=I.lookup(r.id))&&(this.glyphs[u]=this.font.getGlyph(P,r.codePoints))}32768&o.flags&&(this.markedGlyph=u)},r.processLigature=function processLigature(r,o,u){if(32768&o.flags&&this.ligatureStack.push(u),8192&o.flags){for(var p,y=this.subtable.table.ligatureActions,_=this.subtable.table.components,P=this.subtable.table.ligatureList,k=o.action,I=!1,O=0,B=[],D=[];!I;){var R,N=this.ligatureStack.pop();(R=B).unshift.apply(R,this.glyphs[N].codePoints);var U=y.getItem(k++);I=!!(2147483648&U);var W=!!(1073741824&U),j=(1073741823&U)<<2>>2;if(j+=this.glyphs[N].id,O+=_.getItem(j),I||W){var G=P.getItem(O);this.glyphs[N]=this.font.getGlyph(G,B),D.push(N),O=0,B=[]}else this.glyphs[N]=this.font.getGlyph(65535)}(p=this.ligatureStack).push.apply(p,D)}},r.processNoncontextualSubstitutions=function processNoncontextualSubstitutions(r,o,u){var p=new Li(r.table.lookupTable);for(u=0;u<o.length;u++){var y=o[u];if(65535!==y.id){var _=p.lookup(y.id);_&&(o[u]=this.font.getGlyph(_,y.codePoints))}}},r._insertGlyphs=function _insertGlyphs(r,o,u,p){for(var y,_=[];u--;){var P=this.subtable.table.insertionActions.getItem(o++);_.push(this.font.getGlyph(P))}p||r++,(y=this.glyphs).splice.apply(y,[r,0].concat(_))},r.processGlyphInsertion=function processGlyphInsertion(r,o,u){if(32768&o.flags&&(this.markedIndex=u),65535!==o.markedInsertIndex){var p=(31&o.flags)>>>5,y=!!(1024&o.flags);this._insertGlyphs(this.markedIndex,o.markedInsertIndex,p,y)}if(65535!==o.currentInsertIndex){var _=(992&o.flags)>>>5,P=!!(2048&o.flags);this._insertGlyphs(u,o.currentInsertIndex,_,P)}},r.getSupportedFeatures=function getSupportedFeatures(){for(var r,o=[],u=_createForOfIteratorHelperLoose(this.morx.chains);!(r=u()).done;)for(var p,y=_createForOfIteratorHelperLoose(r.value.features);!(p=y()).done;){var _=p.value;o.push([_.featureType,_.featureSetting])}return o},r.generateInputs=function generateInputs(r){return this.inputCache||this.generateInputCache(),this.inputCache[r]||[]},r.generateInputCache=function generateInputCache(){this.inputCache={};for(var r,o=_createForOfIteratorHelperLoose(this.morx.chains);!(r=o()).done;)for(var u,p=r.value,y=p.defaultFlags,_=_createForOfIteratorHelperLoose(p.subtables);!(u=_()).done;){var P=u.value;P.subFeatureFlags&y&&this.generateInputsForSubtable(P)}},r.generateInputsForSubtable=function generateInputsForSubtable(r){var o=this;if(2===r.type){if(!!(r.coverage&Ri))throw new Error("Reverse subtable, not supported.");this.subtable=r,this.ligatureStack=[];var u=this.getStateMachine(r),p=this.getProcessor(),y=[],_=[];this.glyphs=[],u.traverse({enter:function enter(r,u){var P=o.glyphs;_.push({glyphs:P.slice(),ligatureStack:o.ligatureStack.slice()});var k=o.font.getGlyph(r);y.push(k),P.push(y[y.length-1]),p(P[P.length-1],u,P.length-1);for(var I=0,O=0,B=0;B<P.length&&I<=1;B++)65535!==P[B].id&&(I++,O=P[B].id);if(1===I){var D=y.map((function(r){return r.id})),R=o.inputCache[O];R?R.push(D):o.inputCache[O]=[D]}},exit:function exit(){var r=_.pop();o.glyphs=r.glyphs,o.ligatureStack=r.ligatureStack,y.pop()}})}},AATMorxProcessor}(),_applyDecoratedDescriptor(Fi.prototype,"getStateMachine",[cache],Object.getOwnPropertyDescriptor(Fi.prototype,"getStateMachine"),Fi.prototype),Fi);function swap(r,o,u,p,y){void 0===p&&(p=!1),void 0===y&&(y=!1);var _=r.splice(u[0]-(u[1]-1),u[1]);y&&_.reverse();var P=r.splice.apply(r,[o[0],o[1]].concat(_));return p&&P.reverse(),r.splice.apply(r,[u[0]-(o[1]-1),0].concat(P)),r}var Ni=function(){function AATLayoutEngine(r){this.font=r,this.morxProcessor=new Mi(r),this.fallbackPosition=!1}var r=AATLayoutEngine.prototype;return r.substitute=function substitute(r){"rtl"===r.direction&&r.glyphs.reverse(),this.morxProcessor.process(r.glyphs,function mapOTToAAT(r){var o={};for(var u in r){var p;(p=ki[u])&&(null==o[p[0]]&&(o[p[0]]={}),o[p[0]][p[1]]=r[u])}return o}(r.features))},r.getAvailableFeatures=function getAvailableFeatures(r,o){return function mapAATToOT(r){var o={};if(Array.isArray(r))for(var u=0;u<r.length;u++){var p,y=mapFeatureStrings(r[u]);(p=Ii[y[0]]&&Ii[y[0]][y[1]])&&(o[p]=!0)}else if("object"==typeof r)for(var _ in r){var P=r[_];for(var k in P){var I=void 0,O=mapFeatureStrings([_,k]);P[k]&&(I=Ii[O[0]]&&Ii[O[0]][O[1]])&&(o[I]=!0)}}return Object.keys(o)}(this.morxProcessor.getSupportedFeatures())},r.stringsForGlyph=function stringsForGlyph(r){for(var o,u=this.morxProcessor.generateInputs(r),p=new Set,y=_createForOfIteratorHelperLoose(u);!(o=y()).done;){var _=o.value;this._addStrings(_,0,p,"")}return p},r._addStrings=function _addStrings(r,o,u,p){for(var y,_=_createForOfIteratorHelperLoose(this.font._cmapProcessor.codePointsForGlyph(r[o]));!(y=_()).done;){var P=y.value,k=p+String.fromCodePoint(P);o<r.length-1?this._addStrings(r,o+1,u,k):u.add(k)}},AATLayoutEngine}(),Ui=function(){function ShapingPlan(r,o,u){this.font=r,this.script=o,this.direction=u,this.stages=[],this.globalFeatures={},this.allFeatures={}}var r=ShapingPlan.prototype;return r._addFeatures=function _addFeatures(r,o){for(var u,p=this.stages.length-1,y=this.stages[p],_=_createForOfIteratorHelperLoose(r);!(u=_()).done;){var P=u.value;null==this.allFeatures[P]&&(y.push(P),this.allFeatures[P]=p,o&&(this.globalFeatures[P]=!0))}},r.add=function add(r,o){if(void 0===o&&(o=!0),0===this.stages.length&&this.stages.push([]),"string"==typeof r&&(r=[r]),Array.isArray(r))this._addFeatures(r,o);else{if("object"!=typeof r)throw new Error("Unsupported argument to ShapingPlan#add");this._addFeatures(r.global||[],!0),this._addFeatures(r.local||[],!1)}},r.addStage=function addStage(r,o){"function"==typeof r?this.stages.push(r,[]):(this.stages.push([]),this.add(r,o))},r.setFeatureOverrides=function setFeatureOverrides(r){if(Array.isArray(r))this.add(r);else if("object"==typeof r)for(var o in r)if(r[o])this.add(o);else if(null!=this.allFeatures[o]){var u=this.stages[this.allFeatures[o]];u.splice(u.indexOf(o),1),delete this.allFeatures[o],delete this.globalFeatures[o]}},r.assignGlobalFeatures=function assignGlobalFeatures(r){for(var o,u=_createForOfIteratorHelperLoose(r);!(o=u()).done;){var p=o.value;for(var y in this.globalFeatures)p.features[y]=!0}},r.process=function process(r,o,u){for(var p,y=_createForOfIteratorHelperLoose(this.stages);!(p=y()).done;){var _=p.value;"function"==typeof _?u||_(this.font,o,this):_.length>0&&r.applyFeatures(_,o,u)}},ShapingPlan}(),Wi=["rvrn"],zi=["ccmp","locl","rlig","mark","mkmk"],ji=["frac","numr","dnom"],Gi=["calt","clig","liga","rclt","curs","kern"],Vi={ltr:["ltra","ltrm"],rtl:["rtla","rtlm"]},Hi=function(){function DefaultShaper(){}return DefaultShaper.plan=function plan(r,o,u){this.planPreprocessing(r),this.planFeatures(r),this.planPostprocessing(r,u),r.assignGlobalFeatures(o),this.assignFeatures(r,o)},DefaultShaper.planPreprocessing=function planPreprocessing(r){r.add({global:[].concat(Wi,Vi[r.direction]),local:ji})},DefaultShaper.planFeatures=function planFeatures(r){},DefaultShaper.planPostprocessing=function planPostprocessing(r,o){r.add([].concat(zi,Gi)),r.setFeatureOverrides(o)},DefaultShaper.assignFeatures=function assignFeatures(r,o){for(var u=0;u<o.length;u++){var p=o[u];if(8260===p.codePoints[0]){for(var y=u,_=u+1;y>0&&I.isDigit(o[y-1].codePoints[0]);)o[y-1].features.numr=!0,o[y-1].features.frac=!0,y--;for(;_<o.length&&I.isDigit(o[_].codePoints[0]);)o[_].features.dnom=!0,o[_].features.frac=!0,_++;p.features.frac=!0,u=_-1}}},DefaultShaper}();_defineProperty(Hi,"zeroMarkWidths","AFTER_GPOS");var qi=new O(y("APABAAAAAAAAOAAAAf0BAv7tmi1MxDAUx7vtvjhAgcDgkEgEAnmXEBIMCYaEcygEiqBQ4FAkCE4ikUgMiiBJSAgSiUQSDMn9L9eSl6bddddug9t7yS/trevre+3r27pcNxZiG+yCfdCVv/9LeQxOwRm4AJegD27ALbgD9+ABPJF+z+BN/h7yDj5k/VOWX6SdmU5+wLWknggxDxaS8u0qiiX4uiz9XamQ3wzDMAzDMAzDMAzDVI/h959V/v7BMAzDMAzDMLlyNTNiMSdewVxbiA44B4/guz1qW58VYlMI0WsJ0W+N6kXw0spvPtdwhtkwnGM6uLaV4Xyzg3v3PM9DPfQ/sOg4xPWjipy31P8LTqbU304c/cLCUmWJLNB2Uz2U1KTeRKNmKHVMfbJC+/0loTZRH/W5cvEvBJPMbREkWt3FD1NcqXZBSpuE2Ad0PBehPtNrPtIEdYP+hiRt/V1jIiE69X4NT/uVZI3PUHE9bm5M7ePGdZWy951v7Nn6j8v1WWKP3mt6ttnsigx6VN7Vc0VomSSGqW2mGNP1muZPl7LfjNUaKNFtDGVf2fvE9O7VlBS5j333c5p/eeoOqcs1R/hIqDWLJ7TTlksirVT1SI7l8k4Yp+g3jafGcrU1RM6l9th80XOpnlN97bDNY4i4s61B0Si/ipa0uHMl6zqEjlFfCZm/TM8KmzQDjmuTAQ==","base64")),Xi=["isol","fina","fin2","fin3","medi","med2","init"],Zi=0,Yi=6,Ki="isol",Ji="fina",Qi="fin2",$i="medi",eo="med2",to="init",ro=null,no=[[[ro,ro,0],[ro,Ki,2],[ro,Ki,1],[ro,Ki,2],[ro,Ki,1],[ro,Ki,6]],[[ro,ro,0],[ro,Ki,2],[ro,Ki,1],[ro,Ki,2],[ro,Qi,5],[ro,Ki,6]],[[ro,ro,0],[ro,Ki,2],[to,Ji,1],[to,Ji,3],[to,Ji,4],[to,Ji,6]],[[ro,ro,0],[ro,Ki,2],[$i,Ji,1],[$i,Ji,3],[$i,Ji,4],[$i,Ji,6]],[[ro,ro,0],[ro,Ki,2],[eo,Ki,1],[eo,Ki,2],[eo,Qi,5],[eo,Ki,6]],[[ro,ro,0],[ro,Ki,2],[Ki,Ki,1],[Ki,Ki,2],[Ki,Qi,5],[Ki,Ki,6]],[[ro,ro,0],[ro,Ki,2],[ro,Ki,1],[ro,Ki,2],[ro,"fin3",5],[ro,Ki,6]]],io=function(r){function ArabicShaper(){return r.apply(this,arguments)||this}return _inheritsLoose(ArabicShaper,r),ArabicShaper.planFeatures=function planFeatures(r){r.add(["ccmp","locl"]);for(var o=0;o<Xi.length;o++){var u=Xi[o];r.addStage(u,!1)}r.addStage("mset")},ArabicShaper.assignFeatures=function assignFeatures(o,u){r.assignFeatures.call(this,o,u);for(var p=-1,y=0,_=[],P=0;P<u.length;P++){var k,I,O=getShapingClass((N=u[P]).codePoints[0]);if(O!==Yi){var B=no[y][O];I=B[0],k=B[1],y=B[2],I!==ro&&-1!==p&&(_[p]=I),_[P]=k,p=P}else _[P]=ro}for(var D=0;D<u.length;D++){var R,N=u[D];(R=_[D])&&(N.features[R]=!0)}},ArabicShaper}(Hi);function getShapingClass(r){var o=qi.get(r);if(o)return o-1;var u=I.getCategory(r);return"Mn"===u||"Me"===u||"Cf"===u?Yi:Zi}var oo=function(){function GlyphIterator(r,o){this.glyphs=r,this.reset(o)}var r=GlyphIterator.prototype;return r.reset=function reset(r,o){void 0===r&&(r={}),void 0===o&&(o=0),this.options=r,this.flags=r.flags||{},this.markAttachmentType=r.markAttachmentType||0,this.index=o},r.shouldIgnore=function shouldIgnore(r){return this.flags.ignoreMarks&&r.isMark||this.flags.ignoreBaseGlyphs&&r.isBase||this.flags.ignoreLigatures&&r.isLigature||this.markAttachmentType&&r.isMark&&r.markAttachmentType!==this.markAttachmentType},r.move=function move(r){for(this.index+=r;0<=this.index&&this.index<this.glyphs.length&&this.shouldIgnore(this.glyphs[this.index]);)this.index+=r;return 0>this.index||this.index>=this.glyphs.length?null:this.glyphs[this.index]},r.next=function next(){return this.move(1)},r.prev=function prev(){return this.move(-1)},r.peek=function peek(r){void 0===r&&(r=1);var o=this.index,u=this.increment(r);return this.index=o,u},r.peekIndex=function peekIndex(r){void 0===r&&(r=1);var o=this.index;this.increment(r);var u=this.index;return this.index=o,u},r.increment=function increment(r){void 0===r&&(r=1);var o=r<0?-1:1;for(r=Math.abs(r);r--;)this.move(o);return this.glyphs[this.index]},_createClass(GlyphIterator,[{key:"cur",get:function get(){return this.glyphs[this.index]||null}}])}(),ao=["DFLT","dflt","latn"],so=function(){function OTProcessor(r,o){this.font=r,this.table=o,this.script=null,this.scriptTag=null,this.language=null,this.languageTag=null,this.features={},this.lookups={},this.variationsIndex=r._variationProcessor?this.findVariationsIndex(r._variationProcessor.normalizedCoords):-1,this.selectScript(),this.glyphs=[],this.positions=[],this.ligatureID=1,this.currentFeature=null}var r=OTProcessor.prototype;return r.findScript=function findScript(r){if(null==this.table.scriptList)return null;Array.isArray(r)||(r=[r]);for(var o,u=_createForOfIteratorHelperLoose(r);!(o=u()).done;)for(var p,y=o.value,_=_createForOfIteratorHelperLoose(this.table.scriptList);!(p=_()).done;){var P=p.value;if(P.tag===y)return P}return null},r.selectScript=function selectScript(r,o,u){var p,y=!1;if(!this.script||r!==this.scriptTag){if((p=this.findScript(r))||(p=this.findScript(ao)),!p)return this.scriptTag;this.scriptTag=p.tag,this.script=p.script,this.language=null,this.languageTag=null,y=!0}if(u&&u===this.direction||(this.direction=u||direction(r)),o&&o.length<4&&(o+=" ".repeat(4-o.length)),!o||o!==this.languageTag){this.language=null;for(var _,P=_createForOfIteratorHelperLoose(this.script.langSysRecords);!(_=P()).done;){var k=_.value;if(k.tag===o){this.language=k.langSys,this.languageTag=k.tag;break}}this.language||(this.language=this.script.defaultLangSys,this.languageTag=null),y=!0}if(y&&(this.features={},this.language))for(var I,O=_createForOfIteratorHelperLoose(this.language.featureIndexes);!(I=O()).done;){var B=I.value,D=this.table.featureList[B],R=this.substituteFeatureForVariations(B);this.features[D.tag]=R||D.feature}return this.scriptTag},r.lookupsForFeatures=function lookupsForFeatures(r,o){void 0===r&&(r=[]);for(var u,p=[],y=_createForOfIteratorHelperLoose(r);!(u=y()).done;){var _=u.value,P=this.features[_];if(P)for(var k,I=_createForOfIteratorHelperLoose(P.lookupListIndexes);!(k=I()).done;){var O=k.value;o&&-1!==o.indexOf(O)||p.push({feature:_,index:O,lookup:this.table.lookupList.get(O)})}}return p.sort((function(r,o){return r.index-o.index})),p},r.substituteFeatureForVariations=function substituteFeatureForVariations(r){if(-1===this.variationsIndex)return null;for(var o,u=_createForOfIteratorHelperLoose(this.table.featureVariations.featureVariationRecords[this.variationsIndex].featureTableSubstitution.substitutions);!(o=u()).done;){var p=o.value;if(p.featureIndex===r)return p.alternateFeatureTable}return null},r.findVariationsIndex=function findVariationsIndex(r){var o=this.table.featureVariations;if(!o)return-1;for(var u=o.featureVariationRecords,p=0;p<u.length;p++){var y=u[p].conditionSet.conditionTable;if(this.variationConditionsMatch(y,r))return p}return-1},r.variationConditionsMatch=function variationConditionsMatch(r,o){return r.every((function(r){var u=r.axisIndex<o.length?o[r.axisIndex]:0;return r.filterRangeMinValue<=u&&u<=r.filterRangeMaxValue}))},r.applyFeatures=function applyFeatures(r,o,u){var p=this.lookupsForFeatures(r);this.applyLookups(p,o,u)},r.applyLookups=function applyLookups(r,o,u){this.glyphs=o,this.positions=u,this.glyphIterator=new oo(o);for(var p,y=_createForOfIteratorHelperLoose(r);!(p=y()).done;){var _=p.value,P=_.feature,k=_.lookup;for(this.currentFeature=P,this.glyphIterator.reset(k.flags);this.glyphIterator.index<o.length;)if(P in this.glyphIterator.cur.features){for(var I,O=_createForOfIteratorHelperLoose(k.subTables);!(I=O()).done;){var B=I.value;if(this.applyLookup(k.lookupType,B))break}this.glyphIterator.next()}else this.glyphIterator.next()}},r.applyLookup=function applyLookup(r,o){throw new Error("applyLookup must be implemented by subclasses")},r.applyLookupList=function applyLookupList(r){for(var o,u=this.glyphIterator.options,p=this.glyphIterator.index,y=_createForOfIteratorHelperLoose(r);!(o=y()).done;){var _=o.value;this.glyphIterator.reset(u,p),this.glyphIterator.increment(_.sequenceIndex);var P=this.table.lookupList.get(_.lookupListIndex);this.glyphIterator.reset(P.flags,this.glyphIterator.index);for(var k,I=_createForOfIteratorHelperLoose(P.subTables);!(k=I()).done;){var O=k.value;if(this.applyLookup(P.lookupType,O))break}}return this.glyphIterator.reset(u,p),!0},r.coverageIndex=function coverageIndex(r,o){switch(null==o&&(o=this.glyphIterator.cur.id),r.version){case 1:return r.glyphs.indexOf(o);case 2:for(var u,p=_createForOfIteratorHelperLoose(r.rangeRecords);!(u=p()).done;){var y=u.value;if(y.start<=o&&o<=y.end)return y.startCoverageIndex+o-y.start}}return-1},r.match=function match(r,o,u,p){for(var y=this.glyphIterator.index,_=this.glyphIterator.increment(r),P=0;P<o.length&&_&&u(o[P],_);)p&&p.push(this.glyphIterator.index),P++,_=this.glyphIterator.next();return this.glyphIterator.index=y,!(P<o.length)&&(p||!0)},r.sequenceMatches=function sequenceMatches(r,o){return this.match(r,o,(function(r,o){return r===o.id}))},r.sequenceMatchIndices=function sequenceMatchIndices(r,o){var u=this;return this.match(r,o,(function(r,o){return u.currentFeature in o.features&&r===o.id}),[])},r.coverageSequenceMatches=function coverageSequenceMatches(r,o){var u=this;return this.match(r,o,(function(r,o){return u.coverageIndex(r,o.id)>=0}))},r.getClassID=function getClassID(r,o){switch(o.version){case 1:var u=r-o.startGlyph;if(u>=0&&u<o.classValueArray.length)return o.classValueArray[u];break;case 2:for(var p,y=_createForOfIteratorHelperLoose(o.classRangeRecord);!(p=y()).done;){var _=p.value;if(_.start<=r&&r<=_.end)return _.class}}return 0},r.classSequenceMatches=function classSequenceMatches(r,o,u){var p=this;return this.match(r,o,(function(r,o){return r===p.getClassID(o.id,u)}))},r.applyContext=function applyContext(r){switch(r.version){case 1:var o=this.coverageIndex(r.coverage);if(-1===o)return!1;for(var u,p=r.ruleSets[o],y=_createForOfIteratorHelperLoose(p);!(u=y()).done;){var _=u.value;if(this.sequenceMatches(1,_.input))return this.applyLookupList(_.lookupRecords)}break;case 2:if(-1===this.coverageIndex(r.coverage))return!1;if(-1===(o=this.getClassID(this.glyphIterator.cur.id,r.classDef)))return!1;for(var P,k=_createForOfIteratorHelperLoose(p=r.classSet[o]);!(P=k()).done;){var I=P.value;if(this.classSequenceMatches(1,I.classes,r.classDef))return this.applyLookupList(I.lookupRecords)}break;case 3:if(this.coverageSequenceMatches(0,r.coverages))return this.applyLookupList(r.lookupRecords)}return!1},r.applyChainingContext=function applyChainingContext(r){switch(r.version){case 1:var o=this.coverageIndex(r.coverage);if(-1===o)return!1;for(var u,p=_createForOfIteratorHelperLoose(r.chainRuleSets[o]);!(u=p()).done;){var y=u.value;if(this.sequenceMatches(-y.backtrack.length,y.backtrack)&&this.sequenceMatches(1,y.input)&&this.sequenceMatches(1+y.input.length,y.lookahead))return this.applyLookupList(y.lookupRecords)}break;case 2:if(-1===this.coverageIndex(r.coverage))return!1;o=this.getClassID(this.glyphIterator.cur.id,r.inputClassDef);var _=r.chainClassSet[o];if(!_)return!1;for(var P,k=_createForOfIteratorHelperLoose(_);!(P=k()).done;){var I=P.value;if(this.classSequenceMatches(-I.backtrack.length,I.backtrack,r.backtrackClassDef)&&this.classSequenceMatches(1,I.input,r.inputClassDef)&&this.classSequenceMatches(1+I.input.length,I.lookahead,r.lookaheadClassDef))return this.applyLookupList(I.lookupRecords)}break;case 3:if(this.coverageSequenceMatches(-r.backtrackGlyphCount,r.backtrackCoverage)&&this.coverageSequenceMatches(0,r.inputCoverage)&&this.coverageSequenceMatches(r.inputGlyphCount,r.lookaheadCoverage))return this.applyLookupList(r.lookupRecords)}return!1},OTProcessor}(),co=function(){function GlyphInfo(r,o,u,p){if(void 0===u&&(u=[]),this._font=r,this.codePoints=u,this.id=o,this.features={},Array.isArray(p))for(var y=0;y<p.length;y++){var _=p[y];this.features[_]=!0}else"object"==typeof p&&Object.assign(this.features,p);this.ligatureID=null,this.ligatureComponent=null,this.isLigated=!1,this.cursiveAttachment=null,this.markAttachment=null,this.shaperInfo=null,this.substituted=!1,this.isMultiplied=!1}return GlyphInfo.prototype.copy=function copy(){return new GlyphInfo(this._font,this.id,this.codePoints,this.features)},_createClass(GlyphInfo,[{key:"id",get:function get(){return this._id},set:function set(r){this._id=r,this.substituted=!0;var o=this._font.GDEF;if(o&&o.glyphClassDef){var u=so.prototype.getClassID(r,o.glyphClassDef);this.isBase=1===u,this.isLigature=2===u,this.isMark=3===u,this.markAttachmentType=o.markAttachClassDef?so.prototype.getClassID(r,o.markAttachClassDef):0}else this.isMark=this.codePoints.length>0&&this.codePoints.every(I.isMark),this.isBase=!this.isMark,this.isLigature=this.codePoints.length>1,this.markAttachmentType=0}}])}(),lo=function(r){function HangulShaper(){return r.apply(this,arguments)||this}return _inheritsLoose(HangulShaper,r),HangulShaper.planFeatures=function planFeatures(r){r.add(["ljmo","vjmo","tjmo"],!1)},HangulShaper.assignFeatures=function assignFeatures(r,o){for(var u=0,p=0;p<o.length;){var y,_=o[p].codePoints[0],P=getType(_),k=jo[u][P];switch(y=k[0],u=k[1],y){case No:r.font.hasGlyphForCodePoint(_)||(p=decompose(o,p,r.font));break;case Uo:p=compose(o,p,r.font);break;case Wo:reorderToneMark(o,p,r.font);break;case zo:p=insertDottedCircle(o,p,r.font)}p++}},HangulShaper}(Hi);_defineProperty(lo,"zeroMarkWidths","NONE");var uo=44032,fo=55204-uo+1,ho=4352,po=4449,go=4519,yo=21,vo=28,mo=ho+19-1,bo=po+yo-1,wo=go+vo-1,_o=9676,So=function isL(r){return 4352<=r&&r<=4447||43360<=r&&r<=43388},xo=function isV(r){return 4448<=r&&r<=4519||55216<=r&&r<=55238},Ao=function isT(r){return 4520<=r&&r<=4607||55243<=r&&r<=55291},Co=function isTone(r){return 12334<=r&&r<=12335},Po=function isLVT(r){return uo<=r&&r<=55204},ko=function isLV(r){return r-uo<fo&&(r-uo)%vo==0},Eo=function isCombiningL(r){return ho<=r&&r<=mo},To=function isCombiningV(r){return po<=r&&r<=bo},Io=function isCombiningT(r){return 1<=r&&r<=wo},Oo=0,Bo=1,Fo=2,Lo=3,Do=4,Ro=5,Mo=6;function getType(r){return So(r)?Bo:xo(r)?Fo:Ao(r)?Lo:ko(r)?Do:Po(r)?Ro:Co(r)?Mo:Oo}var No=1,Uo=2,Wo=4,zo=5,jo=[[[0,0],[0,1],[0,0],[0,0],[No,2],[No,3],[zo,0]],[[0,0],[0,1],[Uo,2],[0,0],[No,2],[No,3],[zo,0]],[[0,0],[0,1],[0,0],[Uo,3],[No,2],[No,3],[Wo,0]],[[0,0],[0,1],[0,0],[0,0],[No,2],[No,3],[Wo,0]]];function getGlyph(r,o,u){return new co(r,r.glyphForCodePoint(o).id,[o],u)}function decompose(r,o,u){var p=r[o],y=p.codePoints[0]-uo,_=go+y%vo,P=ho+(y=y/vo|0)/yo|0,k=po+y%yo;if(!u.hasGlyphForCodePoint(P)||!u.hasGlyphForCodePoint(k)||_!==go&&!u.hasGlyphForCodePoint(_))return o;var I=getGlyph(u,P,p.features);I.features.ljmo=!0;var O=getGlyph(u,k,p.features);O.features.vjmo=!0;var B=[I,O];if(_>go){var D=getGlyph(u,_,p.features);D.features.tjmo=!0,B.push(D)}return r.splice.apply(r,[o,1].concat(B)),o+B.length-1}function compose(r,o,u){var p,y,_,P,k=r[o],I=getType(r[o].codePoints[0]),O=r[o-1].codePoints[0],B=getType(O);if(B===Do&&I===Lo)p=O,P=k;else{I===Fo?(y=r[o-1],_=k):(y=r[o-2],_=r[o-1],P=k);var D=y.codePoints[0],R=_.codePoints[0];Eo(D)&&To(R)&&(p=uo+((D-ho)*yo+(R-po))*vo)}var N=P&&P.codePoints[0]||go;if(null!=p&&(N===go||Io(N))){var U=p+(N-go);if(u.hasGlyphForCodePoint(U)){var W=B===Fo?3:2;return r.splice(o-W+1,W,getGlyph(u,U,k.features)),o-W+1}}return y&&(y.features.ljmo=!0),_&&(_.features.vjmo=!0),P&&(P.features.tjmo=!0),B===Do?(decompose(r,o-1,u),o+1):o}function reorderToneMark(r,o,u){var p=r[o],y=r[o].codePoints[0];if(0!==u.glyphForCodePoint(y).advanceWidth){var _=function getLength(r){switch(getType(r)){case Do:case Ro:return 1;case Fo:return 2;case Lo:return 3}}(r[o-1].codePoints[0]);return r.splice(o,1),r.splice(o-_,0,p)}}function insertDottedCircle(r,o,u){var p=r[o],y=r[o].codePoints[0];if(u.hasGlyphForCodePoint(_o)){var _=getGlyph(u,_o,p.features),P=0===u.glyphForCodePoint(y).advanceWidth?o:o+1;r.splice(P,0,_),o++}return o}var Go={stateTable:[[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,2,3,4,5,6,7,8,9,0,10,11,11,12,13,14,15,16,17],[0,0,0,18,19,20,21,22,23,0,24,0,0,25,26,0,0,27,0],[0,0,0,28,29,30,31,32,33,0,34,0,0,35,36,0,0,37,0],[0,0,0,38,5,7,7,8,9,0,10,0,0,0,13,0,0,16,0],[0,39,0,0,0,40,41,0,9,0,10,0,0,0,42,0,39,0,0],[0,0,0,0,43,44,44,8,9,0,0,0,0,12,43,0,0,0,0],[0,0,0,0,43,44,44,8,9,0,0,0,0,0,43,0,0,0,0],[0,0,0,45,46,47,48,49,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,50,0,0,51,0,10,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,52,0,0,0,0,0,0,0,0],[0,0,0,53,54,55,56,57,58,0,59,0,0,60,61,0,0,62,0],[0,0,0,4,5,7,7,8,9,0,10,0,0,0,13,0,0,16,0],[0,63,64,0,0,40,41,0,9,0,10,0,0,0,42,0,63,0,0],[0,2,3,4,5,6,7,8,9,0,10,11,11,12,13,0,2,16,0],[0,0,0,18,65,20,21,22,23,0,24,0,0,25,26,0,0,27,0],[0,0,0,0,66,67,67,8,9,0,10,0,0,0,68,0,0,0,0],[0,0,0,69,0,70,70,0,71,0,72,0,0,0,0,0,0,0,0],[0,0,0,73,19,74,74,22,23,0,24,0,0,0,26,0,0,27,0],[0,75,0,0,0,76,77,0,23,0,24,0,0,0,78,0,75,0,0],[0,0,0,0,79,80,80,22,23,0,0,0,0,25,79,0,0,0,0],[0,0,0,18,19,20,74,22,23,0,24,0,0,25,26,0,0,27,0],[0,0,0,81,82,83,84,85,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,86,0,0,87,0,24,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,88,0,0,0,0,0,0,0,0],[0,0,0,18,19,74,74,22,23,0,24,0,0,0,26,0,0,27,0],[0,89,90,0,0,76,77,0,23,0,24,0,0,0,78,0,89,0,0],[0,0,0,0,91,92,92,22,23,0,24,0,0,0,93,0,0,0,0],[0,0,0,94,29,95,31,32,33,0,34,0,0,0,36,0,0,37,0],[0,96,0,0,0,97,98,0,33,0,34,0,0,0,99,0,96,0,0],[0,0,0,0,100,101,101,32,33,0,0,0,0,35,100,0,0,0,0],[0,0,0,0,100,101,101,32,33,0,0,0,0,0,100,0,0,0,0],[0,0,0,102,103,104,105,106,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,107,0,0,108,0,34,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,109,0,0,0,0,0,0,0,0],[0,0,0,28,29,95,31,32,33,0,34,0,0,0,36,0,0,37,0],[0,110,111,0,0,97,98,0,33,0,34,0,0,0,99,0,110,0,0],[0,0,0,0,112,113,113,32,33,0,34,0,0,0,114,0,0,0,0],[0,0,0,0,5,7,7,8,9,0,10,0,0,0,13,0,0,16,0],[0,0,0,115,116,117,118,8,9,0,10,0,0,119,120,0,0,16,0],[0,0,0,0,0,121,121,0,9,0,10,0,0,0,42,0,0,0,0],[0,39,0,122,0,123,123,8,9,0,10,0,0,0,42,0,39,0,0],[0,124,64,0,0,0,0,0,0,0,0,0,0,0,0,0,124,0,0],[0,39,0,0,0,121,125,0,9,0,10,0,0,0,42,0,39,0,0],[0,0,0,0,0,126,126,8,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,46,47,48,49,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,47,47,49,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,127,127,49,9,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,128,127,127,49,9,0,0,0,0,0,0,0,0,0,0],[0,0,0,129,130,131,132,133,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,10,0,0,0,0,0,0,0,0],[0,0,0,0,0,50,0,0,0,0,10,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,134,0,0,0,0,0,0,0,0],[0,0,0,135,54,56,56,57,58,0,59,0,0,0,61,0,0,62,0],[0,136,0,0,0,137,138,0,58,0,59,0,0,0,139,0,136,0,0],[0,0,0,0,140,141,141,57,58,0,0,0,0,60,140,0,0,0,0],[0,0,0,0,140,141,141,57,58,0,0,0,0,0,140,0,0,0,0],[0,0,0,142,143,144,145,146,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,147,0,0,148,0,59,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,149,0,0,0,0,0,0,0,0],[0,0,0,53,54,56,56,57,58,0,59,0,0,0,61,0,0,62,0],[0,150,151,0,0,137,138,0,58,0,59,0,0,0,139,0,150,0,0],[0,0,0,0,152,153,153,57,58,0,59,0,0,0,154,0,0,0,0],[0,0,0,155,116,156,157,8,9,0,10,0,0,158,120,0,0,16,0],[0,0,0,0,0,121,121,0,9,0,10,0,0,0,0,0,0,0,0],[0,75,3,4,5,159,160,8,161,0,162,0,11,12,163,0,75,16,0],[0,0,0,0,0,40,164,0,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,165,44,44,8,9,0,0,0,0,0,165,0,0,0,0],[0,124,64,0,0,40,164,0,9,0,10,0,0,0,42,0,124,0,0],[0,0,0,0,0,70,70,0,71,0,72,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,71,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,166,0,0,167,0,72,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,168,0,0,0,0,0,0,0,0],[0,0,0,0,19,74,74,22,23,0,24,0,0,0,26,0,0,27,0],[0,0,0,0,79,80,80,22,23,0,0,0,0,0,79,0,0,0,0],[0,0,0,169,170,171,172,22,23,0,24,0,0,173,174,0,0,27,0],[0,0,0,0,0,175,175,0,23,0,24,0,0,0,78,0,0,0,0],[0,75,0,176,0,177,177,22,23,0,24,0,0,0,78,0,75,0,0],[0,178,90,0,0,0,0,0,0,0,0,0,0,0,0,0,178,0,0],[0,75,0,0,0,175,179,0,23,0,24,0,0,0,78,0,75,0,0],[0,0,0,0,0,180,180,22,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,82,83,84,85,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,83,83,85,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,181,181,85,23,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,182,181,181,85,23,0,0,0,0,0,0,0,0,0,0],[0,0,0,183,184,185,186,187,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,24,0,0,0,0,0,0,0,0],[0,0,0,0,0,86,0,0,0,0,24,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,188,0,0,0,0,0,0,0,0],[0,0,0,189,170,190,191,22,23,0,24,0,0,192,174,0,0,27,0],[0,0,0,0,0,175,175,0,23,0,24,0,0,0,0,0,0,0,0],[0,0,0,0,0,76,193,0,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,194,80,80,22,23,0,0,0,0,0,194,0,0,0,0],[0,178,90,0,0,76,193,0,23,0,24,0,0,0,78,0,178,0,0],[0,0,0,0,29,95,31,32,33,0,34,0,0,0,36,0,0,37,0],[0,0,0,0,100,101,101,32,33,0,0,0,0,0,100,0,0,0,0],[0,0,0,195,196,197,198,32,33,0,34,0,0,199,200,0,0,37,0],[0,0,0,0,0,201,201,0,33,0,34,0,0,0,99,0,0,0,0],[0,96,0,202,0,203,203,32,33,0,34,0,0,0,99,0,96,0,0],[0,204,111,0,0,0,0,0,0,0,0,0,0,0,0,0,204,0,0],[0,96,0,0,0,201,205,0,33,0,34,0,0,0,99,0,96,0,0],[0,0,0,0,0,206,206,32,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,103,104,105,106,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,104,104,106,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,207,207,106,33,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,208,207,207,106,33,0,0,0,0,0,0,0,0,0,0],[0,0,0,209,210,211,212,213,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,34,0,0,0,0,0,0,0,0],[0,0,0,0,0,107,0,0,0,0,34,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,214,0,0,0,0,0,0,0,0],[0,0,0,215,196,216,217,32,33,0,34,0,0,218,200,0,0,37,0],[0,0,0,0,0,201,201,0,33,0,34,0,0,0,0,0,0,0,0],[0,0,0,0,0,97,219,0,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,220,101,101,32,33,0,0,0,0,0,220,0,0,0,0],[0,204,111,0,0,97,219,0,33,0,34,0,0,0,99,0,204,0,0],[0,0,0,221,116,222,222,8,9,0,10,0,0,0,120,0,0,16,0],[0,223,0,0,0,40,224,0,9,0,10,0,0,0,42,0,223,0,0],[0,0,0,0,225,44,44,8,9,0,0,0,0,119,225,0,0,0,0],[0,0,0,115,116,117,222,8,9,0,10,0,0,119,120,0,0,16,0],[0,0,0,115,116,222,222,8,9,0,10,0,0,0,120,0,0,16,0],[0,226,64,0,0,40,224,0,9,0,10,0,0,0,42,0,226,0,0],[0,0,0,0,0,0,0,0,9,0,0,0,0,0,0,0,0,0,0],[0,39,0,0,0,121,121,0,9,0,10,0,0,0,42,0,39,0,0],[0,0,0,0,0,44,44,8,9,0,0,0,0,0,0,0,0,0,0],[0,0,0,227,0,228,229,0,9,0,10,0,0,230,0,0,0,0,0],[0,39,0,122,0,121,121,0,9,0,10,0,0,0,42,0,39,0,0],[0,0,0,0,0,0,0,8,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,231,231,49,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,232,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,130,131,132,133,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,131,131,133,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,233,233,133,9,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,234,233,233,133,9,0,0,0,0,0,0,0,0,0,0],[0,0,0,235,236,237,238,239,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,54,56,56,57,58,0,59,0,0,0,61,0,0,62,0],[0,0,0,240,241,242,243,57,58,0,59,0,0,244,245,0,0,62,0],[0,0,0,0,0,246,246,0,58,0,59,0,0,0,139,0,0,0,0],[0,136,0,247,0,248,248,57,58,0,59,0,0,0,139,0,136,0,0],[0,249,151,0,0,0,0,0,0,0,0,0,0,0,0,0,249,0,0],[0,136,0,0,0,246,250,0,58,0,59,0,0,0,139,0,136,0,0],[0,0,0,0,0,251,251,57,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,143,144,145,146,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,144,144,146,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,252,252,146,58,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,253,252,252,146,58,0,0,0,0,0,0,0,0,0,0],[0,0,0,254,255,256,257,258,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,59,0,0,0,0,0,0,0,0],[0,0,0,0,0,147,0,0,0,0,59,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,259,0,0,0,0,0,0,0,0],[0,0,0,260,241,261,262,57,58,0,59,0,0,263,245,0,0,62,0],[0,0,0,0,0,246,246,0,58,0,59,0,0,0,0,0,0,0,0],[0,0,0,0,0,137,264,0,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,265,141,141,57,58,0,0,0,0,0,265,0,0,0,0],[0,249,151,0,0,137,264,0,58,0,59,0,0,0,139,0,249,0,0],[0,0,0,221,116,222,222,8,9,0,10,0,0,0,120,0,0,16,0],[0,0,0,0,225,44,44,8,9,0,0,0,0,158,225,0,0,0,0],[0,0,0,155,116,156,222,8,9,0,10,0,0,158,120,0,0,16,0],[0,0,0,155,116,222,222,8,9,0,10,0,0,0,120,0,0,16,0],[0,0,0,0,43,266,266,8,161,0,24,0,0,12,267,0,0,0,0],[0,75,0,176,43,268,268,269,161,0,24,0,0,0,267,0,75,0,0],[0,0,0,0,0,270,0,0,271,0,162,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,272,0,0,0,0,0,0,0,0],[0,273,274,0,0,40,41,0,9,0,10,0,0,0,42,0,273,0,0],[0,0,0,40,0,123,123,8,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,121,275,0,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,72,0,0,0,0,0,0,0,0],[0,0,0,0,0,166,0,0,0,0,72,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,276,0,0,0,0,0,0,0,0],[0,0,0,277,170,278,278,22,23,0,24,0,0,0,174,0,0,27,0],[0,279,0,0,0,76,280,0,23,0,24,0,0,0,78,0,279,0,0],[0,0,0,0,281,80,80,22,23,0,0,0,0,173,281,0,0,0,0],[0,0,0,169,170,171,278,22,23,0,24,0,0,173,174,0,0,27,0],[0,0,0,169,170,278,278,22,23,0,24,0,0,0,174,0,0,27,0],[0,282,90,0,0,76,280,0,23,0,24,0,0,0,78,0,282,0,0],[0,0,0,0,0,0,0,0,23,0,0,0,0,0,0,0,0,0,0],[0,75,0,0,0,175,175,0,23,0,24,0,0,0,78,0,75,0,0],[0,0,0,0,0,80,80,22,23,0,0,0,0,0,0,0,0,0,0],[0,0,0,283,0,284,285,0,23,0,24,0,0,286,0,0,0,0,0],[0,75,0,176,0,175,175,0,23,0,24,0,0,0,78,0,75,0,0],[0,0,0,0,0,0,0,22,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,287,287,85,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,288,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,184,185,186,187,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,185,185,187,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,289,289,187,23,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,290,289,289,187,23,0,0,0,0,0,0,0,0,0,0],[0,0,0,291,292,293,294,295,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,277,170,278,278,22,23,0,24,0,0,0,174,0,0,27,0],[0,0,0,0,281,80,80,22,23,0,0,0,0,192,281,0,0,0,0],[0,0,0,189,170,190,278,22,23,0,24,0,0,192,174,0,0,27,0],[0,0,0,189,170,278,278,22,23,0,24,0,0,0,174,0,0,27,0],[0,0,0,76,0,177,177,22,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,175,296,0,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,297,196,298,298,32,33,0,34,0,0,0,200,0,0,37,0],[0,299,0,0,0,97,300,0,33,0,34,0,0,0,99,0,299,0,0],[0,0,0,0,301,101,101,32,33,0,0,0,0,199,301,0,0,0,0],[0,0,0,195,196,197,298,32,33,0,34,0,0,199,200,0,0,37,0],[0,0,0,195,196,298,298,32,33,0,34,0,0,0,200,0,0,37,0],[0,302,111,0,0,97,300,0,33,0,34,0,0,0,99,0,302,0,0],[0,0,0,0,0,0,0,0,33,0,0,0,0,0,0,0,0,0,0],[0,96,0,0,0,201,201,0,33,0,34,0,0,0,99,0,96,0,0],[0,0,0,0,0,101,101,32,33,0,0,0,0,0,0,0,0,0,0],[0,0,0,303,0,304,305,0,33,0,34,0,0,306,0,0,0,0,0],[0,96,0,202,0,201,201,0,33,0,34,0,0,0,99,0,96,0,0],[0,0,0,0,0,0,0,32,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,307,307,106,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,308,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,210,211,212,213,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,211,211,213,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,309,309,213,33,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,310,309,309,213,33,0,0,0,0,0,0,0,0,0,0],[0,0,0,311,312,313,314,315,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,297,196,298,298,32,33,0,34,0,0,0,200,0,0,37,0],[0,0,0,0,301,101,101,32,33,0,0,0,0,218,301,0,0,0,0],[0,0,0,215,196,216,298,32,33,0,34,0,0,218,200,0,0,37,0],[0,0,0,215,196,298,298,32,33,0,34,0,0,0,200,0,0,37,0],[0,0,0,97,0,203,203,32,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,201,316,0,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,116,222,222,8,9,0,10,0,0,0,120,0,0,16,0],[0,0,0,0,225,44,44,8,9,0,0,0,0,0,225,0,0,0,0],[0,0,0,317,318,319,320,8,9,0,10,0,0,321,322,0,0,16,0],[0,223,0,323,0,123,123,8,9,0,10,0,0,0,42,0,223,0,0],[0,223,0,0,0,121,324,0,9,0,10,0,0,0,42,0,223,0,0],[0,0,0,325,318,326,327,8,9,0,10,0,0,328,322,0,0,16,0],[0,0,0,64,0,121,121,0,9,0,10,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,9,0,0,0,0,230,0,0,0,0,0],[0,0,0,227,0,228,121,0,9,0,10,0,0,230,0,0,0,0,0],[0,0,0,227,0,121,121,0,9,0,10,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,49,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,46,0,0],[0,0,0,0,0,329,329,133,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,330,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,236,237,238,239,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,237,237,239,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,331,331,239,9,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,332,331,331,239,9,0,0,0,0,0,0,0,0,0,0],[0,0,0,333,40,121,334,0,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,335,241,336,336,57,58,0,59,0,0,0,245,0,0,62,0],[0,337,0,0,0,137,338,0,58,0,59,0,0,0,139,0,337,0,0],[0,0,0,0,339,141,141,57,58,0,0,0,0,244,339,0,0,0,0],[0,0,0,240,241,242,336,57,58,0,59,0,0,244,245,0,0,62,0],[0,0,0,240,241,336,336,57,58,0,59,0,0,0,245,0,0,62,0],[0,340,151,0,0,137,338,0,58,0,59,0,0,0,139,0,340,0,0],[0,0,0,0,0,0,0,0,58,0,0,0,0,0,0,0,0,0,0],[0,136,0,0,0,246,246,0,58,0,59,0,0,0,139,0,136,0,0],[0,0,0,0,0,141,141,57,58,0,0,0,0,0,0,0,0,0,0],[0,0,0,341,0,342,343,0,58,0,59,0,0,344,0,0,0,0,0],[0,136,0,247,0,246,246,0,58,0,59,0,0,0,139,0,136,0,0],[0,0,0,0,0,0,0,57,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,345,345,146,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,346,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,255,256,257,258,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,256,256,258,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,347,347,258,58,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,348,347,347,258,58,0,0,0,0,0,0,0,0,0,0],[0,0,0,349,350,351,352,353,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,335,241,336,336,57,58,0,59,0,0,0,245,0,0,62,0],[0,0,0,0,339,141,141,57,58,0,0,0,0,263,339,0,0,0,0],[0,0,0,260,241,261,336,57,58,0,59,0,0,263,245,0,0,62,0],[0,0,0,260,241,336,336,57,58,0,59,0,0,0,245,0,0,62,0],[0,0,0,137,0,248,248,57,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,246,354,0,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,126,126,8,23,0,0,0,0,0,0,0,0,0,0],[0,355,90,0,0,121,125,0,9,0,10,0,0,0,42,0,355,0,0],[0,0,0,0,0,356,356,269,23,0,0,0,0,0,0,0,0,0,0],[0,0,0,357,358,359,360,361,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,162,0,0,0,0,0,0,0,0],[0,0,0,0,0,270,0,0,0,0,162,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,363,0,0,0,0,0,0,0,0],[0,0,0,364,116,365,366,8,161,0,162,0,0,367,120,0,0,16,0],[0,0,0,0,0,368,368,0,161,0,162,0,0,0,0,0,0,0,0],[0,0,0,40,0,121,121,0,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,170,278,278,22,23,0,24,0,0,0,174,0,0,27,0],[0,0,0,0,281,80,80,22,23,0,0,0,0,0,281,0,0,0,0],[0,0,0,369,370,371,372,22,23,0,24,0,0,373,374,0,0,27,0],[0,279,0,375,0,177,177,22,23,0,24,0,0,0,78,0,279,0,0],[0,279,0,0,0,175,376,0,23,0,24,0,0,0,78,0,279,0,0],[0,0,0,377,370,378,379,22,23,0,24,0,0,380,374,0,0,27,0],[0,0,0,90,0,175,175,0,23,0,24,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,23,0,0,0,0,286,0,0,0,0,0],[0,0,0,283,0,284,175,0,23,0,24,0,0,286,0,0,0,0,0],[0,0,0,283,0,175,175,0,23,0,24,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,85,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,82,0,0],[0,0,0,0,0,381,381,187,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,382,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,292,293,294,295,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,293,293,295,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,0,383,383,295,23,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,384,383,383,295,23,0,0,0,0,0,0,0,0,0,0],[0,0,0,385,76,175,386,0,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,76,0,175,175,0,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,196,298,298,32,33,0,34,0,0,0,200,0,0,37,0],[0,0,0,0,301,101,101,32,33,0,0,0,0,0,301,0,0,0,0],[0,0,0,387,388,389,390,32,33,0,34,0,0,391,392,0,0,37,0],[0,299,0,393,0,203,203,32,33,0,34,0,0,0,99,0,299,0,0],[0,299,0,0,0,201,394,0,33,0,34,0,0,0,99,0,299,0,0],[0,0,0,395,388,396,397,32,33,0,34,0,0,398,392,0,0,37,0],[0,0,0,111,0,201,201,0,33,0,34,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,33,0,0,0,0,306,0,0,0,0,0],[0,0,0,303,0,304,201,0,33,0,34,0,0,306,0,0,0,0,0],[0,0,0,303,0,201,201,0,33,0,34,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,106,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,103,0,0],[0,0,0,0,0,399,399,213,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,400,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,312,313,314,315,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,313,313,315,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,0,401,401,315,33,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,402,401,401,315,33,0,0,0,0,0,0,0,0,0,0],[0,0,0,403,97,201,404,0,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,97,0,201,201,0,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,405,318,406,406,8,9,0,10,0,0,0,322,0,0,16,0],[0,407,0,0,0,40,408,0,9,0,10,0,0,0,42,0,407,0,0],[0,0,0,0,409,44,44,8,9,0,0,0,0,321,409,0,0,0,0],[0,0,0,317,318,319,406,8,9,0,10,0,0,321,322,0,0,16,0],[0,0,0,317,318,406,406,8,9,0,10,0,0,0,322,0,0,16,0],[0,410,64,0,0,40,408,0,9,0,10,0,0,0,42,0,410,0,0],[0,223,0,0,0,121,121,0,9,0,10,0,0,0,42,0,223,0,0],[0,223,0,323,0,121,121,0,9,0,10,0,0,0,42,0,223,0,0],[0,0,0,405,318,406,406,8,9,0,10,0,0,0,322,0,0,16,0],[0,0,0,0,409,44,44,8,9,0,0,0,0,328,409,0,0,0,0],[0,0,0,325,318,326,406,8,9,0,10,0,0,328,322,0,0,16,0],[0,0,0,325,318,406,406,8,9,0,10,0,0,0,322,0,0,16,0],[0,0,0,0,0,0,0,133,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,130,0,0],[0,0,0,0,0,411,411,239,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,412,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,40,121,334,0,9,0,10,0,0,0,42,0,0,0,0],[0,0,0,0,413,0,0,0,9,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,241,336,336,57,58,0,59,0,0,0,245,0,0,62,0],[0,0,0,0,339,141,141,57,58,0,0,0,0,0,339,0,0,0,0],[0,0,0,414,415,416,417,57,58,0,59,0,0,418,419,0,0,62,0],[0,337,0,420,0,248,248,57,58,0,59,0,0,0,139,0,337,0,0],[0,337,0,0,0,246,421,0,58,0,59,0,0,0,139,0,337,0,0],[0,0,0,422,415,423,424,57,58,0,59,0,0,425,419,0,0,62,0],[0,0,0,151,0,246,246,0,58,0,59,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,58,0,0,0,0,344,0,0,0,0,0],[0,0,0,341,0,342,246,0,58,0,59,0,0,344,0,0,0,0,0],[0,0,0,341,0,246,246,0,58,0,59,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,146,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,143,0,0],[0,0,0,0,0,426,426,258,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,427,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,350,351,352,353,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,351,351,353,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,0,428,428,353,58,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,429,428,428,353,58,0,0,0,0,0,0,0,0,0,0],[0,0,0,430,137,246,431,0,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,137,0,246,246,0,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,432,116,433,434,8,161,0,162,0,0,435,120,0,0,16,0],[0,0,0,0,0,180,180,269,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,358,359,360,361,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,0,0,359,359,361,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,0,0,436,436,361,161,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,437,436,436,361,161,0,0,0,0,0,0,0,0,0,0],[0,0,0,438,439,440,441,442,161,0,162,0,0,0,362,0,0,0,0],[0,443,274,0,0,0,0,0,0,0,0,0,0,0,0,0,443,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,444,116,445,445,8,161,0,162,0,0,0,120,0,0,16,0],[0,0,0,0,225,44,44,8,161,0,0,0,0,367,225,0,0,0,0],[0,0,0,364,116,365,445,8,161,0,162,0,0,367,120,0,0,16,0],[0,0,0,364,116,445,445,8,161,0,162,0,0,0,120,0,0,16,0],[0,0,0,0,0,0,0,0,161,0,0,0,0,0,0,0,0,0,0],[0,0,0,446,370,447,447,22,23,0,24,0,0,0,374,0,0,27,0],[0,448,0,0,0,76,449,0,23,0,24,0,0,0,78,0,448,0,0],[0,0,0,0,450,80,80,22,23,0,0,0,0,373,450,0,0,0,0],[0,0,0,369,370,371,447,22,23,0,24,0,0,373,374,0,0,27,0],[0,0,0,369,370,447,447,22,23,0,24,0,0,0,374,0,0,27,0],[0,451,90,0,0,76,449,0,23,0,24,0,0,0,78,0,451,0,0],[0,279,0,0,0,175,175,0,23,0,24,0,0,0,78,0,279,0,0],[0,279,0,375,0,175,175,0,23,0,24,0,0,0,78,0,279,0,0],[0,0,0,446,370,447,447,22,23,0,24,0,0,0,374,0,0,27,0],[0,0,0,0,450,80,80,22,23,0,0,0,0,380,450,0,0,0,0],[0,0,0,377,370,378,447,22,23,0,24,0,0,380,374,0,0,27,0],[0,0,0,377,370,447,447,22,23,0,24,0,0,0,374,0,0,27,0],[0,0,0,0,0,0,0,187,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,184,0,0],[0,0,0,0,0,452,452,295,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,453,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,76,175,386,0,23,0,24,0,0,0,78,0,0,0,0],[0,0,0,0,454,0,0,0,23,0,0,0,0,0,0,0,0,0,0],[0,0,0,455,388,456,456,32,33,0,34,0,0,0,392,0,0,37,0],[0,457,0,0,0,97,458,0,33,0,34,0,0,0,99,0,457,0,0],[0,0,0,0,459,101,101,32,33,0,0,0,0,391,459,0,0,0,0],[0,0,0,387,388,389,456,32,33,0,34,0,0,391,392,0,0,37,0],[0,0,0,387,388,456,456,32,33,0,34,0,0,0,392,0,0,37,0],[0,460,111,0,0,97,458,0,33,0,34,0,0,0,99,0,460,0,0],[0,299,0,0,0,201,201,0,33,0,34,0,0,0,99,0,299,0,0],[0,299,0,393,0,201,201,0,33,0,34,0,0,0,99,0,299,0,0],[0,0,0,455,388,456,456,32,33,0,34,0,0,0,392,0,0,37,0],[0,0,0,0,459,101,101,32,33,0,0,0,0,398,459,0,0,0,0],[0,0,0,395,388,396,456,32,33,0,34,0,0,398,392,0,0,37,0],[0,0,0,395,388,456,456,32,33,0,34,0,0,0,392,0,0,37,0],[0,0,0,0,0,0,0,213,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,210,0,0],[0,0,0,0,0,461,461,315,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,462,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,97,201,404,0,33,0,34,0,0,0,99,0,0,0,0],[0,0,0,0,463,0,0,0,33,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,318,406,406,8,9,0,10,0,0,0,322,0,0,16,0],[0,0,0,0,409,44,44,8,9,0,0,0,0,0,409,0,0,0,0],[0,0,0,464,465,466,467,8,9,0,10,0,0,468,469,0,0,16,0],[0,407,0,470,0,123,123,8,9,0,10,0,0,0,42,0,407,0,0],[0,407,0,0,0,121,471,0,9,0,10,0,0,0,42,0,407,0,0],[0,0,0,472,465,473,474,8,9,0,10,0,0,475,469,0,0,16,0],[0,0,0,0,0,0,0,239,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,236,0,0],[0,0,0,0,0,0,476,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,477,415,478,478,57,58,0,59,0,0,0,419,0,0,62,0],[0,479,0,0,0,137,480,0,58,0,59,0,0,0,139,0,479,0,0],[0,0,0,0,481,141,141,57,58,0,0,0,0,418,481,0,0,0,0],[0,0,0,414,415,416,478,57,58,0,59,0,0,418,419,0,0,62,0],[0,0,0,414,415,478,478,57,58,0,59,0,0,0,419,0,0,62,0],[0,482,151,0,0,137,480,0,58,0,59,0,0,0,139,0,482,0,0],[0,337,0,0,0,246,246,0,58,0,59,0,0,0,139,0,337,0,0],[0,337,0,420,0,246,246,0,58,0,59,0,0,0,139,0,337,0,0],[0,0,0,477,415,478,478,57,58,0,59,0,0,0,419,0,0,62,0],[0,0,0,0,481,141,141,57,58,0,0,0,0,425,481,0,0,0,0],[0,0,0,422,415,423,478,57,58,0,59,0,0,425,419,0,0,62,0],[0,0,0,422,415,478,478,57,58,0,59,0,0,0,419,0,0,62,0],[0,0,0,0,0,0,0,258,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0],[0,0,0,0,0,483,483,353,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,484,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,137,246,431,0,58,0,59,0,0,0,139,0,0,0,0],[0,0,0,0,485,0,0,0,58,0,0,0,0,0,0,0,0,0,0],[0,0,0,444,116,445,445,8,161,0,162,0,0,0,120,0,0,16,0],[0,0,0,0,225,44,44,8,161,0,0,0,0,435,225,0,0,0,0],[0,0,0,432,116,433,445,8,161,0,162,0,0,435,120,0,0,16,0],[0,0,0,432,116,445,445,8,161,0,162,0,0,0,120,0,0,16,0],[0,0,0,0,0,486,486,361,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,487,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,439,440,441,442,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,0,0,440,440,442,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,0,0,488,488,442,161,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,489,488,488,442,161,0,0,0,0,0,0,0,0,0,0],[0,0,0,490,491,492,493,494,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,495,0,496,497,0,161,0,162,0,0,498,0,0,0,0,0],[0,0,0,0,116,445,445,8,161,0,162,0,0,0,120,0,0,16,0],[0,0,0,0,225,44,44,8,161,0,0,0,0,0,225,0,0,0,0],[0,0,0,0,370,447,447,22,23,0,24,0,0,0,374,0,0,27,0],[0,0,0,0,450,80,80,22,23,0,0,0,0,0,450,0,0,0,0],[0,0,0,499,500,501,502,22,23,0,24,0,0,503,504,0,0,27,0],[0,448,0,505,0,177,177,22,23,0,24,0,0,0,78,0,448,0,0],[0,448,0,0,0,175,506,0,23,0,24,0,0,0,78,0,448,0,0],[0,0,0,507,500,508,509,22,23,0,24,0,0,510,504,0,0,27,0],[0,0,0,0,0,0,0,295,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,292,0,0],[0,0,0,0,0,0,511,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,388,456,456,32,33,0,34,0,0,0,392,0,0,37,0],[0,0,0,0,459,101,101,32,33,0,0,0,0,0,459,0,0,0,0],[0,0,0,512,513,514,515,32,33,0,34,0,0,516,517,0,0,37,0],[0,457,0,518,0,203,203,32,33,0,34,0,0,0,99,0,457,0,0],[0,457,0,0,0,201,519,0,33,0,34,0,0,0,99,0,457,0,0],[0,0,0,520,513,521,522,32,33,0,34,0,0,523,517,0,0,37,0],[0,0,0,0,0,0,0,315,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,312,0,0],[0,0,0,0,0,0,524,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,525,465,526,526,8,9,0,10,0,0,0,469,0,0,16,0],[0,527,0,0,0,40,528,0,9,0,10,0,0,0,42,0,527,0,0],[0,0,0,0,529,44,44,8,9,0,0,0,0,468,529,0,0,0,0],[0,0,0,464,465,466,526,8,9,0,10,0,0,468,469,0,0,16,0],[0,0,0,464,465,526,526,8,9,0,10,0,0,0,469,0,0,16,0],[0,530,64,0,0,40,528,0,9,0,10,0,0,0,42,0,530,0,0],[0,407,0,0,0,121,121,0,9,0,10,0,0,0,42,0,407,0,0],[0,407,0,470,0,121,121,0,9,0,10,0,0,0,42,0,407,0,0],[0,0,0,525,465,526,526,8,9,0,10,0,0,0,469,0,0,16,0],[0,0,0,0,529,44,44,8,9,0,0,0,0,475,529,0,0,0,0],[0,0,0,472,465,473,526,8,9,0,10,0,0,475,469,0,0,16,0],[0,0,0,472,465,526,526,8,9,0,10,0,0,0,469,0,0,16,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,40,0,0],[0,0,0,0,415,478,478,57,58,0,59,0,0,0,419,0,0,62,0],[0,0,0,0,481,141,141,57,58,0,0,0,0,0,481,0,0,0,0],[0,0,0,531,532,533,534,57,58,0,59,0,0,535,536,0,0,62,0],[0,479,0,537,0,248,248,57,58,0,59,0,0,0,139,0,479,0,0],[0,479,0,0,0,246,538,0,58,0,59,0,0,0,139,0,479,0,0],[0,0,0,539,532,540,541,57,58,0,59,0,0,542,536,0,0,62,0],[0,0,0,0,0,0,0,353,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,350,0,0],[0,0,0,0,0,0,543,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,361,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,358,0,0],[0,0,0,0,0,544,544,442,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,545,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,491,492,493,494,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,0,0,492,492,494,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,0,0,546,546,494,161,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,547,546,546,494,161,0,0,0,0,0,0,0,0,0,0],[0,0,0,548,549,368,550,0,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,274,0,368,368,0,161,0,162,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,161,0,0,0,0,498,0,0,0,0,0],[0,0,0,495,0,496,368,0,161,0,162,0,0,498,0,0,0,0,0],[0,0,0,495,0,368,368,0,161,0,162,0,0,0,0,0,0,0,0],[0,0,0,551,500,552,552,22,23,0,24,0,0,0,504,0,0,27,0],[0,553,0,0,0,76,554,0,23,0,24,0,0,0,78,0,553,0,0],[0,0,0,0,555,80,80,22,23,0,0,0,0,503,555,0,0,0,0],[0,0,0,499,500,501,552,22,23,0,24,0,0,503,504,0,0,27,0],[0,0,0,499,500,552,552,22,23,0,24,0,0,0,504,0,0,27,0],[0,556,90,0,0,76,554,0,23,0,24,0,0,0,78,0,556,0,0],[0,448,0,0,0,175,175,0,23,0,24,0,0,0,78,0,448,0,0],[0,448,0,505,0,175,175,0,23,0,24,0,0,0,78,0,448,0,0],[0,0,0,551,500,552,552,22,23,0,24,0,0,0,504,0,0,27,0],[0,0,0,0,555,80,80,22,23,0,0,0,0,510,555,0,0,0,0],[0,0,0,507,500,508,552,22,23,0,24,0,0,510,504,0,0,27,0],[0,0,0,507,500,552,552,22,23,0,24,0,0,0,504,0,0,27,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,76,0,0],[0,0,0,557,513,558,558,32,33,0,34,0,0,0,517,0,0,37,0],[0,559,0,0,0,97,560,0,33,0,34,0,0,0,99,0,559,0,0],[0,0,0,0,561,101,101,32,33,0,0,0,0,516,561,0,0,0,0],[0,0,0,512,513,514,558,32,33,0,34,0,0,516,517,0,0,37,0],[0,0,0,512,513,558,558,32,33,0,34,0,0,0,517,0,0,37,0],[0,562,111,0,0,97,560,0,33,0,34,0,0,0,99,0,562,0,0],[0,457,0,0,0,201,201,0,33,0,34,0,0,0,99,0,457,0,0],[0,457,0,518,0,201,201,0,33,0,34,0,0,0,99,0,457,0,0],[0,0,0,557,513,558,558,32,33,0,34,0,0,0,517,0,0,37,0],[0,0,0,0,561,101,101,32,33,0,0,0,0,523,561,0,0,0,0],[0,0,0,520,513,521,558,32,33,0,34,0,0,523,517,0,0,37,0],[0,0,0,520,513,558,558,32,33,0,34,0,0,0,517,0,0,37,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,97,0,0],[0,0,0,0,465,526,526,8,9,0,10,0,0,0,469,0,0,16,0],[0,0,0,0,529,44,44,8,9,0,0,0,0,0,529,0,0,0,0],[0,0,0,563,66,564,565,8,9,0,10,0,0,566,68,0,0,16,0],[0,527,0,567,0,123,123,8,9,0,10,0,0,0,42,0,527,0,0],[0,527,0,0,0,121,568,0,9,0,10,0,0,0,42,0,527,0,0],[0,0,0,569,66,570,571,8,9,0,10,0,0,572,68,0,0,16,0],[0,0,0,573,532,574,574,57,58,0,59,0,0,0,536,0,0,62,0],[0,575,0,0,0,137,576,0,58,0,59,0,0,0,139,0,575,0,0],[0,0,0,0,577,141,141,57,58,0,0,0,0,535,577,0,0,0,0],[0,0,0,531,532,533,574,57,58,0,59,0,0,535,536,0,0,62,0],[0,0,0,531,532,574,574,57,58,0,59,0,0,0,536,0,0,62,0],[0,578,151,0,0,137,576,0,58,0,59,0,0,0,139,0,578,0,0],[0,479,0,0,0,246,246,0,58,0,59,0,0,0,139,0,479,0,0],[0,479,0,537,0,246,246,0,58,0,59,0,0,0,139,0,479,0,0],[0,0,0,573,532,574,574,57,58,0,59,0,0,0,536,0,0,62,0],[0,0,0,0,577,141,141,57,58,0,0,0,0,542,577,0,0,0,0],[0,0,0,539,532,540,574,57,58,0,59,0,0,542,536,0,0,62,0],[0,0,0,539,532,574,574,57,58,0,59,0,0,0,536,0,0,62,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,137,0,0],[0,0,0,0,0,0,0,442,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,439,0,0],[0,0,0,0,0,579,579,494,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,580,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,549,368,550,0,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,0,0,368,368,0,161,0,162,0,0,0,362,0,0,0,0],[0,0,0,0,581,0,0,0,161,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,500,552,552,22,23,0,24,0,0,0,504,0,0,27,0],[0,0,0,0,555,80,80,22,23,0,0,0,0,0,555,0,0,0,0],[0,0,0,582,91,583,584,22,23,0,24,0,0,585,93,0,0,27,0],[0,553,0,586,0,177,177,22,23,0,24,0,0,0,78,0,553,0,0],[0,553,0,0,0,175,587,0,23,0,24,0,0,0,78,0,553,0,0],[0,0,0,588,91,589,590,22,23,0,24,0,0,591,93,0,0,27,0],[0,0,0,0,513,558,558,32,33,0,34,0,0,0,517,0,0,37,0],[0,0,0,0,561,101,101,32,33,0,0,0,0,0,561,0,0,0,0],[0,0,0,592,112,593,594,32,33,0,34,0,0,595,114,0,0,37,0],[0,559,0,596,0,203,203,32,33,0,34,0,0,0,99,0,559,0,0],[0,559,0,0,0,201,597,0,33,0,34,0,0,0,99,0,559,0,0],[0,0,0,598,112,599,600,32,33,0,34,0,0,601,114,0,0,37,0],[0,0,0,602,66,67,67,8,9,0,10,0,0,0,68,0,0,16,0],[0,0,0,0,165,44,44,8,9,0,0,0,0,566,165,0,0,0,0],[0,0,0,563,66,564,67,8,9,0,10,0,0,566,68,0,0,16,0],[0,0,0,563,66,67,67,8,9,0,10,0,0,0,68,0,0,16,0],[0,527,0,0,0,121,121,0,9,0,10,0,0,0,42,0,527,0,0],[0,527,0,567,0,121,121,0,9,0,10,0,0,0,42,0,527,0,0],[0,0,0,602,66,67,67,8,9,0,10,0,0,0,68,0,0,16,0],[0,0,0,0,165,44,44,8,9,0,0,0,0,572,165,0,0,0,0],[0,0,0,569,66,570,67,8,9,0,10,0,0,572,68,0,0,16,0],[0,0,0,569,66,67,67,8,9,0,10,0,0,0,68,0,0,16,0],[0,0,0,0,532,574,574,57,58,0,59,0,0,0,536,0,0,62,0],[0,0,0,0,577,141,141,57,58,0,0,0,0,0,577,0,0,0,0],[0,0,0,603,152,604,605,57,58,0,59,0,0,606,154,0,0,62,0],[0,575,0,607,0,248,248,57,58,0,59,0,0,0,139,0,575,0,0],[0,575,0,0,0,246,608,0,58,0,59,0,0,0,139,0,575,0,0],[0,0,0,609,152,610,611,57,58,0,59,0,0,612,154,0,0,62,0],[0,0,0,0,0,0,0,494,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,491,0,0],[0,0,0,0,0,0,613,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,614,91,92,92,22,23,0,24,0,0,0,93,0,0,27,0],[0,0,0,0,194,80,80,22,23,0,0,0,0,585,194,0,0,0,0],[0,0,0,582,91,583,92,22,23,0,24,0,0,585,93,0,0,27,0],[0,0,0,582,91,92,92,22,23,0,24,0,0,0,93,0,0,27,0],[0,553,0,0,0,175,175,0,23,0,24,0,0,0,78,0,553,0,0],[0,553,0,586,0,175,175,0,23,0,24,0,0,0,78,0,553,0,0],[0,0,0,614,91,92,92,22,23,0,24,0,0,0,93,0,0,27,0],[0,0,0,0,194,80,80,22,23,0,0,0,0,591,194,0,0,0,0],[0,0,0,588,91,589,92,22,23,0,24,0,0,591,93,0,0,27,0],[0,0,0,588,91,92,92,22,23,0,24,0,0,0,93,0,0,27,0],[0,0,0,615,112,113,113,32,33,0,34,0,0,0,114,0,0,37,0],[0,0,0,0,220,101,101,32,33,0,0,0,0,595,220,0,0,0,0],[0,0,0,592,112,593,113,32,33,0,34,0,0,595,114,0,0,37,0],[0,0,0,592,112,113,113,32,33,0,34,0,0,0,114,0,0,37,0],[0,559,0,0,0,201,201,0,33,0,34,0,0,0,99,0,559,0,0],[0,559,0,596,0,201,201,0,33,0,34,0,0,0,99,0,559,0,0],[0,0,0,615,112,113,113,32,33,0,34,0,0,0,114,0,0,37,0],[0,0,0,0,220,101,101,32,33,0,0,0,0,601,220,0,0,0,0],[0,0,0,598,112,599,113,32,33,0,34,0,0,601,114,0,0,37,0],[0,0,0,598,112,113,113,32,33,0,34,0,0,0,114,0,0,37,0],[0,0,0,0,66,67,67,8,9,0,10,0,0,0,68,0,0,16,0],[0,0,0,616,152,153,153,57,58,0,59,0,0,0,154,0,0,62,0],[0,0,0,0,265,141,141,57,58,0,0,0,0,606,265,0,0,0,0],[0,0,0,603,152,604,153,57,58,0,59,0,0,606,154,0,0,62,0],[0,0,0,603,152,153,153,57,58,0,59,0,0,0,154,0,0,62,0],[0,575,0,0,0,246,246,0,58,0,59,0,0,0,139,0,575,0,0],[0,575,0,607,0,246,246,0,58,0,59,0,0,0,139,0,575,0,0],[0,0,0,616,152,153,153,57,58,0,59,0,0,0,154,0,0,62,0],[0,0,0,0,265,141,141,57,58,0,0,0,0,612,265,0,0,0,0],[0,0,0,609,152,610,153,57,58,0,59,0,0,612,154,0,0,62,0],[0,0,0,609,152,153,153,57,58,0,59,0,0,0,154,0,0,62,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,549,0,0],[0,0,0,0,91,92,92,22,23,0,24,0,0,0,93,0,0,27,0],[0,0,0,0,112,113,113,32,33,0,34,0,0,0,114,0,0,37,0],[0,0,0,0,152,153,153,57,58,0,59,0,0,0,154,0,0,62,0]],accepting:[!1,!0,!0,!0,!0,!0,!1,!1,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!1,!0,!1,!0,!0,!1,!1,!0,!0,!0,!0,!0,!0,!1,!1,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!0,!0,!0,!1,!0,!0,!0,!1,!0,!1,!0,!0,!1,!1,!0,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!0,!0,!0,!1,!0,!1,!0,!0,!1,!1,!0,!0,!0,!0,!0,!0,!0,!1,!0,!0,!0,!1,!0,!0,!0,!1,!0,!1,!0,!0,!1,!1,!1,!0,!0,!1,!1,!0,!0,!0,!0,!0,!0,!1,!0,!1,!0,!0,!1,!1,!0,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!1,!0,!0,!0,!1,!0,!1,!0,!0,!1,!1,!1,!0,!0,!1,!1,!0,!0,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!0,!1,!0,!1,!0,!0,!1,!1,!1,!0,!0,!1,!1,!0,!0,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!1,!1,!1,!1,!0,!0,!1,!1,!0,!0,!0,!1,!0,!0,!0,!1,!0,!1,!0,!0,!1,!1,!1,!0,!0,!1,!1,!0,!0,!0,!1,!0,!0,!0,!0,!1,!0,!1,!0,!0,!0,!0,!0,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!1,!1,!1,!1,!0,!0,!1,!1,!0,!0,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!1,!1,!1,!1,!0,!0,!1,!1,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!1,!1,!1,!0,!1,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!1,!1,!1,!1,!0,!0,!1,!1,!0,!0,!0,!1,!0,!0,!1,!1,!0,!1,!0,!0,!1,!0,!0,!1,!0,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!1,!1,!1,!0,!1,!0,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!1,!1,!1,!0,!1,!0,!1,!0,!0,!0,!0,!1,!1,!1,!0,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!1,!1,!1,!0,!1,!0,!1,!0,!0,!1,!1,!0,!0,!1,!1,!0,!0,!0,!1,!0,!1,!0,!0,!0,!0,!1,!1,!1,!0,!1,!0,!0,!0,!0,!1,!1,!1,!0,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!0,!1,!0,!0,!0,!0,!1,!1,!1,!1,!1,!1,!1,!0,!0,!1,!1,!0,!0,!1,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!0,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!0,!1,!0,!0,!1,!1,!1,!1,!1,!0,!0,!1,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!0,!1,!0,!0,!0,!0,!1,!1,!1,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!1,!0,!0,!1,!0,!0,!0],tags:[[],["broken_cluster"],["consonant_syllable"],["vowel_syllable"],["broken_cluster"],["broken_cluster"],[],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],["standalone_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["consonant_syllable"],["broken_cluster"],["symbol_cluster"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],[],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["standalone_cluster"],["standalone_cluster"],[],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["broken_cluster"],["broken_cluster"],["consonant_syllable","broken_cluster"],["broken_cluster"],[],["broken_cluster"],["symbol_cluster"],[],["symbol_cluster"],["symbol_cluster"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],[],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],[],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],[],[],[],["broken_cluster"],["broken_cluster"],[],[],["broken_cluster"],["broken_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],[],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["symbol_cluster"],["symbol_cluster"],["symbol_cluster"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],[],[],[],["consonant_syllable"],["consonant_syllable"],[],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],[],[],[],["vowel_syllable"],["vowel_syllable"],[],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],[],[],[],[],["broken_cluster"],["broken_cluster"],[],[],["broken_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],[],[],[],["standalone_cluster"],["standalone_cluster"],[],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],["broken_cluster"],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],["broken_cluster"],["symbol_cluster"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],[],[],[],[],["consonant_syllable"],["consonant_syllable"],[],[],["consonant_syllable"],["consonant_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],[],[],[],[],["vowel_syllable"],["vowel_syllable"],[],[],["vowel_syllable"],["vowel_syllable"],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],[],[],[],[],["broken_cluster"],[],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],[],[],[],[],["standalone_cluster"],["standalone_cluster"],[],[],["standalone_cluster"],["standalone_cluster"],["consonant_syllable","broken_cluster"],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],[],[],["consonant_syllable","broken_cluster"],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],[],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],[],[],[],[],["consonant_syllable"],[],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],[],[],[],[],["vowel_syllable"],[],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],[],[],[],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],[],[],[],[],["standalone_cluster"],[],["consonant_syllable","broken_cluster"],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],[],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],[],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],[],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],[],[],[],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],[],[],[],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],[],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],[],[],[],[],[],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],[],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],[],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],[],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],[],[],[],[],[],["consonant_syllable","broken_cluster"],["consonant_syllable","broken_cluster"],[],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],[],[],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],["consonant_syllable"],[],["consonant_syllable"],["consonant_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],["vowel_syllable"],[],["vowel_syllable"],["vowel_syllable"],["broken_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],["standalone_cluster"],[],["standalone_cluster"],["standalone_cluster"],[],["consonant_syllable"],["vowel_syllable"],["standalone_cluster"]]},Vo={categories:["O","IND","S","GB","B","FM","CGJ","VMAbv","VMPst","VAbv","VPst","CMBlw","VPre","VBlw","H","VMBlw","CMAbv","MBlw","CS","R","SUB","MPst","MPre","FAbv","FPst","FBlw","null","SMAbv","SMBlw","VMPre","ZWNJ","ZWJ","WJ","M","VS","N","HN","MAbv"],decompositions:{2507:[2503,2494],2508:[2503,2519],2888:[2887,2902],2891:[2887,2878],2892:[2887,2903],3018:[3014,3006],3019:[3015,3006],3020:[3014,3031],3144:[3142,3158],3264:[3263,3285],3271:[3270,3285],3272:[3270,3286],3274:[3270,3266],3275:[3270,3266,3285],3402:[3398,3390],3403:[3399,3390],3404:[3398,3415],3546:[3545,3530],3548:[3545,3535],3549:[3545,3535,3530],3550:[3545,3551],3635:[3661,3634],3763:[3789,3762],3955:[3953,3954],3957:[3953,3956],3958:[4018,3968],3959:[4018,3953,3968],3960:[4019,3968],3961:[4019,3953,3968],3969:[3953,3968],6971:[6970,6965],6973:[6972,6965],6976:[6974,6965],6977:[6975,6965],6979:[6978,6965],69934:[69937,69927],69935:[69938,69927],70475:[70471,70462],70476:[70471,70487],70843:[70841,70842],70844:[70841,70832],70846:[70841,70845],71098:[71096,71087],71099:[71097,71087]},stateTable:[[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[2,2,3,4,4,5,0,6,7,8,9,10,11,12,13,14,15,16,0,17,18,11,19,20,21,22,0,0,0,23,0,0,2,0,0,24,0,25],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,26,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,27,28,0,0,0,0,0,27,0,0,0],[0,0,0,0,0,29,0,30,31,32,33,34,35,36,37,38,39,40,0,0,41,35,42,43,44,45,0,0,0,46,0,0,0,0,39,0,0,47],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,6,7,0,0,0,0,0,0,14,0,0,0,0,0,0,0,20,21,22,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,0,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,20,21,22,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,6,7,8,9,0,0,12,0,14,0,0,0,0,0,0,0,20,21,22,0,0,0,23,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,6,7,0,9,0,0,0,0,14,0,0,0,0,0,0,0,20,21,22,0,0,0,23,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,6,7,8,9,10,11,12,13,14,0,16,0,0,18,11,19,20,21,22,0,0,0,23,0,0,0,0,0,0,0,25],[0,0,0,0,0,5,0,6,7,8,9,0,11,12,0,14,0,0,0,0,0,0,0,20,21,22,0,0,0,23,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,6,7,0,9,0,0,12,0,14,0,0,0,0,0,0,0,20,21,22,0,0,0,23,0,0,0,0,0,0,0,0],[0,0,0,0,18,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,0,7,0,0,0,0,0,0,14,0,0,0,0,0,0,0,20,21,22,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,6,7,8,9,10,11,12,13,14,15,16,0,0,18,11,19,20,21,22,0,0,0,23,0,0,0,0,0,0,0,25],[0,0,0,0,0,5,0,6,7,8,9,0,11,12,0,14,0,0,0,0,0,11,0,20,21,22,0,0,0,23,0,0,0,0,0,0,0,0],[0,0,0,4,4,5,0,6,7,8,9,10,11,12,13,14,15,16,0,0,18,11,19,20,21,22,0,0,0,23,0,0,0,0,0,0,0,25],[0,0,0,0,0,5,0,6,7,8,9,48,11,12,13,14,48,16,0,0,18,11,19,20,21,22,0,0,0,23,0,0,0,0,49,0,0,25],[0,0,0,0,0,5,0,6,7,8,9,0,11,12,0,14,0,16,0,0,0,11,0,20,21,22,0,0,0,23,0,0,0,0,0,0,0,25],[0,0,0,0,0,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,20,21,22,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,21,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,21,22,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,6,7,0,0,0,0,0,0,14,0,0,0,0,0,0,0,20,21,22,0,0,0,23,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,50,0,51,0],[0,0,0,0,0,5,0,6,7,8,9,0,11,12,0,14,0,16,0,0,0,11,0,20,21,22,0,0,0,23,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,27,28,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,28,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,30,31,0,0,0,0,0,0,38,0,0,0,0,0,0,0,43,44,45,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,0,31,0,0,0,0,0,0,0,0,0,0,0,0,0,0,43,44,45,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,30,31,32,33,0,0,36,0,38,0,0,0,0,0,0,0,43,44,45,0,0,0,46,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,30,31,0,33,0,0,0,0,38,0,0,0,0,0,0,0,43,44,45,0,0,0,46,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,30,31,32,33,34,35,36,37,38,0,40,0,0,41,35,42,43,44,45,0,0,0,46,0,0,0,0,0,0,0,47],[0,0,0,0,0,29,0,30,31,32,33,0,35,36,0,38,0,0,0,0,0,0,0,43,44,45,0,0,0,46,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,30,31,0,33,0,0,36,0,38,0,0,0,0,0,0,0,43,44,45,0,0,0,46,0,0,0,0,0,0,0,0],[0,0,0,0,41,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,0,31,0,0,0,0,0,0,38,0,0,0,0,0,0,0,43,44,45,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,30,31,32,33,34,35,36,37,38,39,40,0,0,41,35,42,43,44,45,0,0,0,46,0,0,0,0,0,0,0,47],[0,0,0,0,0,29,0,30,31,32,33,0,35,36,0,38,0,0,0,0,0,35,0,43,44,45,0,0,0,46,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,30,31,32,33,52,35,36,37,38,52,40,0,0,41,35,42,43,44,45,0,0,0,46,0,0,0,0,53,0,0,47],[0,0,0,0,0,29,0,30,31,32,33,0,35,36,0,38,0,40,0,0,0,35,0,43,44,45,0,0,0,46,0,0,0,0,0,0,0,47],[0,0,0,0,0,29,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,43,44,45,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,44,0,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,44,45,0,0,0,0,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,30,31,0,0,0,0,0,0,38,0,0,0,0,0,0,0,43,44,45,0,0,0,46,0,0,0,0,0,0,0,0],[0,0,0,0,0,29,0,30,31,32,33,0,35,36,0,38,0,40,0,0,0,35,0,43,44,45,0,0,0,46,0,0,0,0,0,0,0,0],[0,0,0,0,0,5,0,6,7,8,9,48,11,12,13,14,0,16,0,0,18,11,19,20,21,22,0,0,0,23,0,0,0,0,0,0,0,25],[0,0,0,0,0,5,0,6,7,8,9,48,11,12,13,14,48,16,0,0,18,11,19,20,21,22,0,0,0,23,0,0,0,0,0,0,0,25],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,51,0],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,54,0,0],[0,0,0,0,0,29,0,30,31,32,33,52,35,36,37,38,0,40,0,0,41,35,42,43,44,45,0,0,0,46,0,0,0,0,0,0,0,47],[0,0,0,0,0,29,0,30,31,32,33,52,35,36,37,38,52,40,0,0,41,35,42,43,44,45,0,0,0,46,0,0,0,0,0,0,0,47],[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,50,0,51,0]],accepting:[!1,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!1,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0,!0],tags:[[],["broken_cluster"],["independent_cluster"],["symbol_cluster"],["standard_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],[],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["broken_cluster"],["numeral_cluster"],["broken_cluster"],["independent_cluster"],["symbol_cluster"],["symbol_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["virama_terminated_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["standard_cluster"],["broken_cluster"],["broken_cluster"],["numeral_cluster"],["number_joiner_terminated_cluster"],["standard_cluster"],["standard_cluster"],["numeral_cluster"]]},Ho=1,qo=8,Xo=16,Zo=32,Yo=64,Ko=128,Jo=8192,Qo=32768,$o=65536,ea=1<<17,ta={Start:1,Ra_To_Become_Reph:2,Pre_M:4,Pre_C:8,Base_C:16,After_Main:32,Above_C:64,Before_Sub:128,Below_C:256,After_Sub:512,Before_Post:1024,Post_C:2048,After_Post:4096,Final_C:8192,SMVD:16384,End:32768},ra=2|$o|ea|4|2048|4096,na=Yo|Zo,ia=Xo|16384,oa={Default:{hasOldSpec:!1,virama:0,basePos:"Last",rephPos:ta.Before_Post,rephMode:"Implicit",blwfMode:"Pre_And_Post"},Devanagari:{hasOldSpec:!0,virama:2381,basePos:"Last",rephPos:ta.Before_Post,rephMode:"Implicit",blwfMode:"Pre_And_Post"},Bengali:{hasOldSpec:!0,virama:2509,basePos:"Last",rephPos:ta.After_Sub,rephMode:"Implicit",blwfMode:"Pre_And_Post"},Gurmukhi:{hasOldSpec:!0,virama:2637,basePos:"Last",rephPos:ta.Before_Sub,rephMode:"Implicit",blwfMode:"Pre_And_Post"},Gujarati:{hasOldSpec:!0,virama:2765,basePos:"Last",rephPos:ta.Before_Post,rephMode:"Implicit",blwfMode:"Pre_And_Post"},Oriya:{hasOldSpec:!0,virama:2893,basePos:"Last",rephPos:ta.After_Main,rephMode:"Implicit",blwfMode:"Pre_And_Post"},Tamil:{hasOldSpec:!0,virama:3021,basePos:"Last",rephPos:ta.After_Post,rephMode:"Implicit",blwfMode:"Pre_And_Post"},Telugu:{hasOldSpec:!0,virama:3149,basePos:"Last",rephPos:ta.After_Post,rephMode:"Explicit",blwfMode:"Post_Only"},Kannada:{hasOldSpec:!0,virama:3277,basePos:"Last",rephPos:ta.After_Post,rephMode:"Implicit",blwfMode:"Post_Only"},Malayalam:{hasOldSpec:!0,virama:3405,basePos:"Last",rephPos:ta.After_Main,rephMode:"Log_Repha",blwfMode:"Pre_And_Post"},Khmer:{hasOldSpec:!1,virama:6098,basePos:"First",rephPos:ta.Ra_To_Become_Reph,rephMode:"Vis_Repha",blwfMode:"Pre_And_Post"}},aa={6078:[6081,6078],6079:[6081,6079],6080:[6081,6080],6084:[6081,6084],6085:[6081,6085]},sa=Vo.decompositions,ca=new O(y("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","base64")),la=new B(Go),ua=function(r){function IndicShaper(){return r.apply(this,arguments)||this}return _inheritsLoose(IndicShaper,r),IndicShaper.planFeatures=function planFeatures(r){r.addStage(setupSyllables),r.addStage(["locl","ccmp"]),r.addStage(initialReordering),r.addStage("nukt"),r.addStage("akhn"),r.addStage("rphf",!1),r.addStage("rkrf"),r.addStage("pref",!1),r.addStage("blwf",!1),r.addStage("abvf",!1),r.addStage("half",!1),r.addStage("pstf",!1),r.addStage("vatu"),r.addStage("cjct"),r.addStage("cfar",!1),r.addStage(finalReordering),r.addStage({local:["init"],global:["pres","abvs","blws","psts","haln","dist","abvm","blwm","calt","clig"]}),r.unicodeScript=function fromOpenType(r){return yi[r]}(r.script),r.indicConfig=oa[r.unicodeScript]||oa.Default,r.isOldSpec=r.indicConfig.hasOldSpec&&"2"!==r.script[r.script.length-1]},IndicShaper.assignFeatures=function assignFeatures(r,o){for(var u=function _loop(u){var p=o[u].codePoints[0],y=aa[p]||sa[p];if(y){var _=y.map((function(p){var y=r.font.glyphForCodePoint(p);return new co(r.font,y.id,[p],o[u].features)}));o.splice.apply(o,[u,1].concat(_))}},p=o.length-1;p>=0;p--)u(p)},IndicShaper}(Hi);function indicCategory(r){return ca.get(r.codePoints[0])>>8}function indicPosition(r){return 1<<(255&ca.get(r.codePoints[0]))}_defineProperty(ua,"zeroMarkWidths","NONE");var fa=function IndicInfo(r,o,u,p){this.category=r,this.position=o,this.syllableType=u,this.syllable=p};function setupSyllables(r,o){for(var u,p=0,y=0,_=_createForOfIteratorHelperLoose(la.match(o.map(indicCategory)));!(u=_()).done;){var P=u.value,k=P[0],I=P[1],O=P[2];if(k>y){++p;for(var B=y;B<k;B++)o[B].shaperInfo=new fa(Ho,ta.End,"non_indic_cluster",p)}++p;for(var D=k;D<=I;D++)o[D].shaperInfo=new fa(1<<indicCategory(o[D]),indicPosition(o[D]),O[0],p);y=I+1}if(y<o.length){++p;for(var R=y;R<o.length;R++)o[R].shaperInfo=new fa(Ho,ta.End,"non_indic_cluster",p)}}function isConsonant(r){return r.shaperInfo.category&ra}function isJoiner(r){return r.shaperInfo.category&na}function isHalantOrCoeng(r){return r.shaperInfo.category&ia}function wouldSubstitute(r,o){for(var u,p=_createForOfIteratorHelperLoose(r);!(u=p()).done;){var y;u.value.features=((y={})[o]=!0,y)}return r[0]._font._layoutEngine.engine.GSUBProcessor.applyFeatures([o],r),1===r.length}function consonantPosition(r,o,u){var p=[u,o,u];return wouldSubstitute(p.slice(0,2),"blwf")||wouldSubstitute(p.slice(1,3),"blwf")?ta.Below_C:wouldSubstitute(p.slice(0,2),"pstf")||wouldSubstitute(p.slice(1,3),"pstf")||wouldSubstitute(p.slice(0,2),"pref")||wouldSubstitute(p.slice(1,3),"pref")?ta.Post_C:ta.Base_C}function initialReordering(r,o,u){var p=u.indicConfig,y=r._layoutEngine.engine.GSUBProcessor.features,_=r.glyphForCodePoint(9676).id,P=r.glyphForCodePoint(p.virama).id;if(P)for(var k=new co(r,P,[p.virama]),I=0;I<o.length;I++)o[I].shaperInfo.position===ta.Base_C&&(o[I].shaperInfo.position=consonantPosition(0,o[I].copy(),k));for(var O=0,B=nextSyllable(o,0);O<o.length;B=nextSyllable(o,O=B)){var D=o[O].shaperInfo,R=(D.category,D.syllableType);if("symbol_cluster"!==R&&"non_indic_cluster"!==R){if("broken_cluster"===R&&_){var N=new co(r,_,[9676]);N.shaperInfo=new fa(1<<indicCategory(N),indicPosition(N),o[O].shaperInfo.syllableType,o[O].shaperInfo.syllable);for(var U=O;U<B&&o[U].shaperInfo.category===Qo;)U++;o.splice(U++,0,N),B++}var W=B,j=O,G=!1;if(p.rephPos!==ta.Ra_To_Become_Reph&&y.rphf&&O+3<=B&&("Implicit"===p.rephMode&&!isJoiner(o[O+2])||"Explicit"===p.rephMode&&o[O+2].shaperInfo.category===Yo)){var X=[o[O].copy(),o[O+1].copy(),o[O+2].copy()];if(wouldSubstitute(X.slice(0,2),"rphf")||"Explicit"===p.rephMode&&wouldSubstitute(X,"rphf")){for(j+=2;j<B&&isJoiner(o[j]);)j++;W=O,G=!0}}else if("Log_Repha"===p.rephMode&&o[O].shaperInfo.category===Qo){for(j++;j<B&&isJoiner(o[j]);)j++;W=O,G=!0}switch(p.basePos){case"Last":var Y=B,K=!1;do{var J=o[--Y].shaperInfo;if(isConsonant(o[Y])){if(J.position!==ta.Below_C&&(J.position!==ta.Post_C||K)){W=Y;break}J.position===ta.Below_C&&(K=!0),W=Y}else if(O<Y&&J.category===Yo&&o[Y-1].shaperInfo.category===Xo)break}while(Y>j);break;case"First":for(var $=(W=O)+1;$<B;$++)isConsonant(o[$])&&(o[$].shaperInfo.position=ta.Below_C)}G&&W===O&&j-W<=2&&(G=!1);for(var ee=O;ee<W;ee++){var te=o[ee].shaperInfo;te.position=Math.min(ta.Pre_C,te.position)}W<B&&(o[W].shaperInfo.position=ta.Base_C);for(var re=W+1;re<B;re++)if(o[re].shaperInfo.category===Ko){for(var ne=re+1;ne<B;ne++)if(isConsonant(o[ne])){o[ne].shaperInfo.position=ta.Final_C;break}break}if(G&&(o[O].shaperInfo.position=ta.Ra_To_Become_Reph),u.isOldSpec)for(var ie="Malayalam"!==u.unicodeScript,oe=W+1;oe<B;oe++)if(o[oe].shaperInfo.category===Xo){var ae=void 0;for(ae=B-1;ae>oe&&!(isConsonant(o[ae])||ie&&o[ae].shaperInfo.category===Xo);ae--);if(o[ae].shaperInfo.category!==Xo&&ae>oe){var se=o[oe];o.splice.apply(o,[oe,0].concat(o.splice(oe+1,ae-oe))),o[ae]=se}break}for(var ce=ta.Start,le=O;le<B;le++){var ue=o[le].shaperInfo;if(ue.category&(na|qo|Jo|ea|ia&ue.category)){if(ue.position=ce,ue.category===Xo&&ue.position===ta.Pre_M)for(var fe=le;fe>O;fe--)if(o[fe-1].shaperInfo.position!==ta.Pre_M){ue.position=o[fe-1].shaperInfo.position;break}}else ue.position!==ta.SMVD&&(ce=ue.position)}for(var he=W,de=W+1;de<B;de++)if(isConsonant(o[de])){for(var pe=he+1;pe<de;pe++)o[pe].shaperInfo.position<ta.SMVD&&(o[pe].shaperInfo.position=o[de].shaperInfo.position);he=de}else o[de].shaperInfo.category===Ko&&(he=de);var ge=o.slice(O,B);ge.sort((function(r,o){return r.shaperInfo.position-o.shaperInfo.position})),o.splice.apply(o,[O,ge.length].concat(ge));for(var ye=O;ye<B;ye++)if(o[ye].shaperInfo.position===ta.Base_C){W=ye;break}for(var ve=O;ve<B&&o[ve].shaperInfo.position===ta.Ra_To_Become_Reph;ve++)o[ve].features.rphf=!0;for(var me=!u.isOldSpec&&"Pre_And_Post"===p.blwfMode,be=O;be<W;be++)o[be].features.half=!0,me&&(o[be].features.blwf=!0);for(var we=W+1;we<B;we++)o[we].features.abvf=!0,o[we].features.pstf=!0,o[we].features.blwf=!0;if(u.isOldSpec&&"Devanagari"===u.unicodeScript)for(var _e=O;_e+1<W;_e++)o[_e].shaperInfo.category!==$o||o[_e+1].shaperInfo.category!==Xo||_e+1!==W&&o[_e+2].shaperInfo.category!==Yo||(o[_e].features.blwf=!0,o[_e+1].features.blwf=!0);if(y.pref&&W+2<B)for(var Se=W+1;Se+2-1<B;Se++){if(wouldSubstitute([o[Se].copy(),o[Se+1].copy()],"pref")){for(var xe=0;xe<2;xe++)o[Se++].features.pref=!0;if(y.cfar)for(;Se<B;Se++)o[Se].features.cfar=!0;break}}for(var Ae=O+1;Ae<B;Ae++)if(isJoiner(o[Ae])){var Ce=o[Ae].shaperInfo.category===Zo,Pe=Ae;do{Pe--,Ce&&delete o[Pe].features.half}while(Pe>O&&!isConsonant(o[Pe]))}}}}function finalReordering(r,o,u){for(var p=u.indicConfig,y=r._layoutEngine.engine.GSUBProcessor.features,_=0,P=nextSyllable(o,0);_<o.length;P=nextSyllable(o,_=P)){for(var k=!!y.pref,O=_;O<P;O++)if(o[O].shaperInfo.position>=ta.Base_C){if(k&&O+1<P)for(var B=O+1;B<P;B++)if(o[B].features.pref){if(!o[B].substituted||!o[B].isLigated||o[B].isMultiplied){for(O=B;O<P&&isHalantOrCoeng(o[O]);)O++;o[O].shaperInfo.position=ta.BASE_C,k=!1}break}if("Malayalam"===u.unicodeScript)for(var D=O+1;D<P;D++){for(;D<P&&isJoiner(o[D]);)D++;if(D===P||!isHalantOrCoeng(o[D]))break;for(D++;D<P&&isJoiner(o[D]);)D++;D<P&&isConsonant(o[D])&&o[D].shaperInfo.position===ta.Below_C&&(o[O=D].shaperInfo.position=ta.Base_C)}_<O&&o[O].shaperInfo.position>ta.Base_C&&O--;break}if(O===P&&_<O&&o[O-1].shaperInfo.category===Yo&&O--,O<P)for(;_<O&&o[O].shaperInfo.category&(qo|ia);)O--;if(_+1<P&&_<O){var R=O===P?O-2:O-1;if("Malayalam"!==u.unicodeScript&&"Tamil"!==u.unicodeScript){for(;R>_&&!(o[R].shaperInfo.category&(Ko|ia));)R--;isHalantOrCoeng(o[R])&&o[R].shaperInfo.position!==ta.Pre_M?R+1<P&&isJoiner(o[R+1])&&R++:R=_}if(_<R&&o[R].shaperInfo.position!==ta.Pre_M)for(var N=R;N>_;N--)if(o[N-1].shaperInfo.position===ta.Pre_M){var U=N-1;U<O&&O<=R&&O--;var W=o[U];o.splice.apply(o,[U,0].concat(o.splice(U+1,R-U))),o[R]=W,R--}}if(_+1<P&&o[_].shaperInfo.position===ta.Ra_To_Become_Reph&&o[_].shaperInfo.category===Qo!==(o[_].isLigated&&!o[_].isMultiplied)){var j=void 0,G=p.rephPos,X=!1;if(G!==ta.After_Post){for(j=_+1;j<O&&!isHalantOrCoeng(o[j]);)j++;if(j<O&&isHalantOrCoeng(o[j])&&(j+1<O&&isJoiner(o[j+1])&&j++,X=!0),!X&&G===ta.After_Main){for(j=O;j+1<P&&o[j+1].shaperInfo.position<=ta.After_Main;)j++;X=j<P}if(!X&&G===ta.After_Sub){for(j=O;j+1<P&&!(o[j+1].shaperInfo.position&(ta.Post_C|ta.After_Post|ta.SMVD));)j++;X=j<P}}if(!X){for(j=_+1;j<O&&!isHalantOrCoeng(o[j]);)j++;j<O&&isHalantOrCoeng(o[j])&&(j+1<O&&isJoiner(o[j+1])&&j++,X=!0)}if(!X){for(j=P-1;j>_&&o[j].shaperInfo.position===ta.SMVD;)j--;if(isHalantOrCoeng(o[j]))for(var Y=O+1;Y<j;Y++)o[Y].shaperInfo.category===Ko&&j--}var K=o[_];o.splice.apply(o,[_,0].concat(o.splice(_+1,j-_))),o[j]=K,_<O&&O<=j&&O--}if(k&&O+1<P)for(var J=O+1;J<P;J++)if(o[J].features.pref){if(o[J].isLigated&&!o[J].isMultiplied){var $=O;if("Malayalam"!==u.unicodeScript&&"Tamil"!==u.unicodeScript){for(;$>_&&!(o[$-1].shaperInfo.category&(Ko|ia));)$--;if($>_&&o[$-1].shaperInfo.category===Ko)for(var ee=J,te=O+1;te<ee;te++)if(o[te].shaperInfo.category===Ko){$--;break}}$>_&&isHalantOrCoeng(o[$-1])&&$<P&&isJoiner(o[$])&&$++;var re=J,ne=o[re];o.splice.apply(o,[$+1,0].concat(o.splice($,re-$))),o[$]=ne,$<=O&&O<re&&O++}break}o[_].shaperInfo.position!==ta.Pre_M||_&&/Cf|Mn/.test(I.getCategory(o[_-1].codePoints[0]))||(o[_].features.init=!0)}}function nextSyllable(r,o){if(o>=r.length)return o;for(var u=r[o].shaperInfo.syllable;++o<r.length&&r[o].shaperInfo.syllable===u;);return o}var ha=Vo.categories,da=Vo.decompositions,pa=new O(y("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","base64")),ga=new B(Vo),ya=function(r){function UniversalShaper(){return r.apply(this,arguments)||this}return _inheritsLoose(UniversalShaper,r),UniversalShaper.planFeatures=function planFeatures(r){r.addStage(setupSyllables$1),r.addStage(["locl","ccmp","nukt","akhn"]),r.addStage(clearSubstitutionFlags),r.addStage(["rphf"],!1),r.addStage(recordRphf),r.addStage(clearSubstitutionFlags),r.addStage(["pref"]),r.addStage(recordPref),r.addStage(["rkrf","abvf","blwf","half","pstf","vatu","cjct"]),r.addStage(reorder),r.addStage(["abvs","blws","pres","psts","dist","abvm","blwm"])},UniversalShaper.assignFeatures=function assignFeatures(r,o){for(var u=function _loop(u){var p=o[u].codePoints[0];if(da[p]){var y=da[p].map((function(p){var y=r.font.glyphForCodePoint(p);return new co(r.font,y.id,[p],o[u].features)}));o.splice.apply(o,[u,1].concat(y))}},p=o.length-1;p>=0;p--)u(p)},UniversalShaper}(Hi);function useCategory(r){return pa.get(r.codePoints[0])}_defineProperty(ya,"zeroMarkWidths","BEFORE_GPOS");var va=function USEInfo(r,o,u){this.category=r,this.syllableType=o,this.syllable=u};function setupSyllables$1(r,o){for(var u,p=0,y=_createForOfIteratorHelperLoose(ga.match(o.map(useCategory)));!(u=y()).done;){var _=u.value,P=_[0],k=_[1],I=_[2];++p;for(var O=P;O<=k;O++)o[O].shaperInfo=new va(ha[useCategory(o[O])],I[0],p);for(var B="R"===o[P].shaperInfo.category?1:Math.min(3,k-P),D=P;D<P+B;D++)o[D].features.rphf=!0}}function clearSubstitutionFlags(r,o){for(var u,p=_createForOfIteratorHelperLoose(o);!(u=p()).done;){u.value.substituted=!1}}function recordRphf(r,o){for(var u,p=_createForOfIteratorHelperLoose(o);!(u=p()).done;){var y=u.value;y.substituted&&y.features.rphf&&(y.shaperInfo.category="R")}}function recordPref(r,o){for(var u,p=_createForOfIteratorHelperLoose(o);!(u=p()).done;){var y=u.value;y.substituted&&(y.shaperInfo.category="VPre")}}function reorder(r,o){for(var u=r.glyphForCodePoint(9676).id,p=0,y=nextSyllable$1(o,0);p<o.length;y=nextSyllable$1(o,p=y)){var _=void 0,P=void 0,k=o[p].shaperInfo,I=k.syllableType;if("virama_terminated_cluster"===I||"standard_cluster"===I||"broken_cluster"===I){if("broken_cluster"===I&&u){var O=new co(r,u,[9676]);for(O.shaperInfo=k,_=p;_<y&&"R"===o[_].shaperInfo.category;_++);o.splice(++_,0,O),y++}if("R"===k.category&&y-p>1)for(_=p+1;_<y;_++)if(isBase(k=o[_].shaperInfo)||isHalant(o[_])){isHalant(o[_])&&_--,o.splice.apply(o,[p,0].concat(o.splice(p+1,_-p),[o[_]]));break}for(_=p,P=y;_<y;_++)isBase(k=o[_].shaperInfo)||isHalant(o[_])?P=isHalant(o[_])?_+1:_:("VPre"===k.category||"VMPre"===k.category)&&P<_&&o.splice.apply(o,[P,1,o[_]].concat(o.splice(P,_-P)))}}}function nextSyllable$1(r,o){if(o>=r.length)return o;for(var u=r[o].shaperInfo.syllable;++o<r.length&&r[o].shaperInfo.syllable===u;);return o}function isHalant(r){return"H"===r.shaperInfo.category&&!r.isLigated}function isBase(r){return"B"===r.category||"GB"===r.category}var ma={arab:io,mong:io,syrc:io,"nko ":io,phag:io,mand:io,mani:io,phlp:io,hang:lo,bng2:ua,beng:ua,dev2:ua,deva:ua,gjr2:ua,gujr:ua,guru:ua,gur2:ua,knda:ua,knd2:ua,mlm2:ua,mlym:ua,ory2:ua,orya:ua,taml:ua,tml2:ua,telu:ua,tel2:ua,khmr:ua,bali:ya,batk:ya,brah:ya,bugi:ya,buhd:ya,cakm:ya,cham:ya,dupl:ya,egyp:ya,gran:ya,hano:ya,java:ya,kthi:ya,kali:ya,khar:ya,khoj:ya,sind:ya,lepc:ya,limb:ya,mahj:ya,mtei:ya,modi:ya,hmng:ya,rjng:ya,saur:ya,shrd:ya,sidd:ya,sinh:ya,sund:ya,sylo:ya,tglg:ya,tagb:ya,tale:ya,lana:ya,tavt:ya,takr:ya,tibt:ya,tfng:ya,tirh:ya,latn:Hi,DFLT:Hi};for(var ba=function(r){function GSUBProcessor(){return r.apply(this,arguments)||this}return _inheritsLoose(GSUBProcessor,r),GSUBProcessor.prototype.applyLookup=function applyLookup(r,o){var u=this;switch(r){case 1:var p=this.coverageIndex(o.coverage);if(-1===p)return!1;var y=this.glyphIterator.cur;switch(o.version){case 1:y.id=y.id+o.deltaGlyphID&65535;break;case 2:y.id=o.substitute.get(p)}return!0;case 2:var _=this.coverageIndex(o.coverage);if(-1!==_){var P,k=o.sequences.get(_);if(0===k.length)return this.glyphs.splice(this.glyphIterator.index,1),!0;this.glyphIterator.cur.id=k[0],this.glyphIterator.cur.ligatureComponent=0;var I=this.glyphIterator.cur.features,O=this.glyphIterator.cur,B=k.slice(1).map((function(r,o){var p=new co(u.font,r,void 0,I);return p.shaperInfo=O.shaperInfo,p.isLigated=O.isLigated,p.ligatureComponent=o+1,p.substituted=!0,p.isMultiplied=!0,p}));return(P=this.glyphs).splice.apply(P,[this.glyphIterator.index+1,0].concat(B)),!0}return!1;case 3:var D=this.coverageIndex(o.coverage);if(-1!==D){return this.glyphIterator.cur.id=o.alternateSet.get(D)[0],!0}return!1;case 4:var R=this.coverageIndex(o.coverage);if(-1===R)return!1;for(var N,U=_createForOfIteratorHelperLoose(o.ligatureSets.get(R));!(N=U()).done;){var W=N.value,j=this.sequenceMatchIndices(1,W.components);if(j){for(var G,X=this.glyphIterator.cur,Y=X.codePoints.slice(),K=_createForOfIteratorHelperLoose(j);!(G=K()).done;){var J,$=G.value;(J=Y).push.apply(J,this.glyphs[$].codePoints)}var ee=new co(this.font,W.glyph,Y,X.features);ee.shaperInfo=X.shaperInfo,ee.isLigated=!0,ee.substituted=!0;for(var te=X.isMark,re=0;re<j.length&&te;re++)te=this.glyphs[j[re]].isMark;ee.ligatureID=te?null:this.ligatureID++;for(var ne,ie=X.ligatureID,oe=X.codePoints.length,ae=oe,se=this.glyphIterator.index+1,ce=_createForOfIteratorHelperLoose(j);!(ne=ce()).done;){var le=ne.value;if(te)se=le;else for(;se<le;){var ue=ae-oe+Math.min(this.glyphs[se].ligatureComponent||1,oe);this.glyphs[se].ligatureID=ee.ligatureID,this.glyphs[se].ligatureComponent=ue,se++}ie=this.glyphs[se].ligatureID,ae+=oe=this.glyphs[se].codePoints.length,se++}if(ie&&!te)for(var fe=se;fe<this.glyphs.length&&this.glyphs[fe].ligatureID===ie;fe++){ue=ae-oe+Math.min(this.glyphs[fe].ligatureComponent||1,oe);this.glyphs[fe].ligatureComponent=ue}for(var he=j.length-1;he>=0;he--)this.glyphs.splice(j[he],1);return this.glyphs[this.glyphIterator.index]=ee,!0}}return!1;case 5:return this.applyContext(o);case 6:return this.applyChainingContext(o);case 7:return this.applyLookup(o.lookupType,o.extension);default:throw new Error("GSUB lookupType ".concat(r," is not supported"))}},GSUBProcessor}(so),wa=function(r){function GPOSProcessor(){return r.apply(this,arguments)||this}_inheritsLoose(GPOSProcessor,r);var o=GPOSProcessor.prototype;return o.applyPositionValue=function applyPositionValue(r,o){var u=this.positions[this.glyphIterator.peekIndex(r)];null!=o.xAdvance&&(u.xAdvance+=o.xAdvance),null!=o.yAdvance&&(u.yAdvance+=o.yAdvance),null!=o.xPlacement&&(u.xOffset+=o.xPlacement),null!=o.yPlacement&&(u.yOffset+=o.yPlacement);var p=this.font._variationProcessor,y=this.font.GDEF&&this.font.GDEF.itemVariationStore;p&&y&&(o.xPlaDevice&&(u.xOffset+=p.getDelta(y,o.xPlaDevice.a,o.xPlaDevice.b)),o.yPlaDevice&&(u.yOffset+=p.getDelta(y,o.yPlaDevice.a,o.yPlaDevice.b)),o.xAdvDevice&&(u.xAdvance+=p.getDelta(y,o.xAdvDevice.a,o.xAdvDevice.b)),o.yAdvDevice&&(u.yAdvance+=p.getDelta(y,o.yAdvDevice.a,o.yAdvDevice.b)))},o.applyLookup=function applyLookup(r,o){switch(r){case 1:var u=this.coverageIndex(o.coverage);if(-1===u)return!1;switch(o.version){case 1:this.applyPositionValue(0,o.value);break;case 2:this.applyPositionValue(0,o.values.get(u))}return!0;case 2:var p=this.glyphIterator.peek();if(!p)return!1;var y=this.coverageIndex(o.coverage);if(-1===y)return!1;switch(o.version){case 1:for(var _,P=_createForOfIteratorHelperLoose(o.pairSets.get(y));!(_=P()).done;){var k=_.value;if(k.secondGlyph===p.id)return this.applyPositionValue(0,k.value1),this.applyPositionValue(1,k.value2),!0}return!1;case 2:var I=this.getClassID(this.glyphIterator.cur.id,o.classDef1),O=this.getClassID(p.id,o.classDef2);if(-1===I||-1===O)return!1;var B=o.classRecords.get(I).get(O);return this.applyPositionValue(0,B.value1),this.applyPositionValue(1,B.value2),!0}case 3:var D=this.glyphIterator.peekIndex(),R=this.glyphs[D];if(!R)return!1;var N=o.entryExitRecords[this.coverageIndex(o.coverage)];if(!N||!N.exitAnchor)return!1;var U=o.entryExitRecords[this.coverageIndex(o.coverage,R.id)];if(!U||!U.entryAnchor)return!1;var W=this.getAnchor(U.entryAnchor),j=this.getAnchor(N.exitAnchor),G=this.positions[this.glyphIterator.index],X=this.positions[D];switch(this.direction){case"ltr":G.xAdvance=j.x+G.xOffset;var Y=W.x+X.xOffset;X.xAdvance-=Y,X.xOffset-=Y;break;case"rtl":Y=j.x+G.xOffset,G.xAdvance-=Y,G.xOffset-=Y,X.xAdvance=W.x+X.xOffset}return this.glyphIterator.flags.rightToLeft?(this.glyphIterator.cur.cursiveAttachment=D,G.yOffset=W.y-j.y):(R.cursiveAttachment=this.glyphIterator.index,G.yOffset=j.y-W.y),!0;case 4:var K=this.coverageIndex(o.markCoverage);if(-1===K)return!1;for(var J=this.glyphIterator.index;--J>=0&&(this.glyphs[J].isMark||this.glyphs[J].ligatureComponent>0););if(J<0)return!1;var $=this.coverageIndex(o.baseCoverage,this.glyphs[J].id);if(-1===$)return!1;var ee=o.markArray[K],te=o.baseArray[$][ee.class];return this.applyAnchor(ee,te,J),!0;case 5:var re=this.coverageIndex(o.markCoverage);if(-1===re)return!1;for(var ne=this.glyphIterator.index;--ne>=0&&this.glyphs[ne].isMark;);if(ne<0)return!1;var ie=this.coverageIndex(o.ligatureCoverage,this.glyphs[ne].id);if(-1===ie)return!1;var oe=o.ligatureArray[ie],ae=this.glyphIterator.cur,se=this.glyphs[ne],ce=se.ligatureID&&se.ligatureID===ae.ligatureID&&ae.ligatureComponent>0?Math.min(ae.ligatureComponent,se.codePoints.length)-1:se.codePoints.length-1,le=o.markArray[re],ue=oe[ce][le.class];return this.applyAnchor(le,ue,ne),!0;case 6:var fe=this.coverageIndex(o.mark1Coverage);if(-1===fe)return!1;var he=this.glyphIterator.peekIndex(-1),de=this.glyphs[he];if(!de||!de.isMark)return!1;var pe=this.glyphIterator.cur,ge=!1;if(pe.ligatureID===de.ligatureID?pe.ligatureID?pe.ligatureComponent===de.ligatureComponent&&(ge=!0):ge=!0:(pe.ligatureID&&!pe.ligatureComponent||de.ligatureID&&!de.ligatureComponent)&&(ge=!0),!ge)return!1;var ye=this.coverageIndex(o.mark2Coverage,de.id);if(-1===ye)return!1;var ve=o.mark1Array[fe],me=o.mark2Array[ye][ve.class];return this.applyAnchor(ve,me,he),!0;case 7:return this.applyContext(o);case 8:return this.applyChainingContext(o);case 9:return this.applyLookup(o.lookupType,o.extension);default:throw new Error("Unsupported GPOS table: ".concat(r))}},o.applyAnchor=function applyAnchor(r,o,u){var p=this.getAnchor(o),y=this.getAnchor(r.markAnchor),_=(this.positions[u],this.positions[this.glyphIterator.index]);_.xOffset=p.x-y.x,_.yOffset=p.y-y.y,this.glyphIterator.cur.markAttachment=u},o.getAnchor=function getAnchor(r){var o=r.xCoordinate,u=r.yCoordinate,p=this.font._variationProcessor,y=this.font.GDEF&&this.font.GDEF.itemVariationStore;return p&&y&&(r.xDeviceTable&&(o+=p.getDelta(y,r.xDeviceTable.a,r.xDeviceTable.b)),r.yDeviceTable&&(u+=p.getDelta(y,r.yDeviceTable.a,r.yDeviceTable.b))),{x:o,y:u}},o.applyFeatures=function applyFeatures(o,u,p){r.prototype.applyFeatures.call(this,o,u,p);for(var y=0;y<this.glyphs.length;y++)this.fixCursiveAttachment(y);this.fixMarkAttachment()},o.fixCursiveAttachment=function fixCursiveAttachment(r){var o=this.glyphs[r];if(null!=o.cursiveAttachment){var u=o.cursiveAttachment;o.cursiveAttachment=null,this.fixCursiveAttachment(u),this.positions[r].yOffset+=this.positions[u].yOffset}},o.fixMarkAttachment=function fixMarkAttachment(){for(var r=0;r<this.glyphs.length;r++){var o=this.glyphs[r];if(null!=o.markAttachment){var u=o.markAttachment;if(this.positions[r].xOffset+=this.positions[u].xOffset,this.positions[r].yOffset+=this.positions[u].yOffset,"ltr"===this.direction)for(var p=u;p<r;p++)this.positions[r].xOffset-=this.positions[p].xAdvance,this.positions[r].yOffset-=this.positions[p].yAdvance;else for(var y=u+1;y<r+1;y++)this.positions[r].xOffset+=this.positions[y].xAdvance,this.positions[r].yOffset+=this.positions[y].yAdvance}}},GPOSProcessor}(so),_a=function(){function OTLayoutEngine(r){this.font=r,this.glyphInfos=null,this.plan=null,this.GSUBProcessor=null,this.GPOSProcessor=null,this.fallbackPosition=!0,r.GSUB&&(this.GSUBProcessor=new ba(r,r.GSUB)),r.GPOS&&(this.GPOSProcessor=new wa(r,r.GPOS))}var r=OTLayoutEngine.prototype;return r.setup=function setup(r){var o=this;this.glyphInfos=r.glyphs.map((function(r){return new co(o.font,r.id,[].concat(r.codePoints))}));var u=null;for(var p in this.GPOSProcessor&&(u=this.GPOSProcessor.selectScript(r.script,r.language,r.direction)),this.GSUBProcessor&&(u=this.GSUBProcessor.selectScript(r.script,r.language,r.direction)),this.shaper=function choose(r){Array.isArray(r)||(r=[r]);for(var o,u=_createForOfIteratorHelperLoose(r);!(o=u()).done;){var p=o.value,y=ma[p];if(y)return y}return Hi}(u),this.plan=new Ui(this.font,u,r.direction),this.shaper.plan(this.plan,this.glyphInfos,r.features),this.plan.allFeatures)r.features[p]=!0},r.substitute=function substitute(r){var o=this;this.GSUBProcessor&&(this.plan.process(this.GSUBProcessor,this.glyphInfos),r.glyphs=this.glyphInfos.map((function(r){return o.font.getGlyph(r.id,r.codePoints)})))},r.position=function position(r){return"BEFORE_GPOS"===this.shaper.zeroMarkWidths&&this.zeroMarkAdvances(r.positions),this.GPOSProcessor&&this.plan.process(this.GPOSProcessor,this.glyphInfos,r.positions),"AFTER_GPOS"===this.shaper.zeroMarkWidths&&this.zeroMarkAdvances(r.positions),"rtl"===r.direction&&(r.glyphs.reverse(),r.positions.reverse()),this.GPOSProcessor&&this.GPOSProcessor.features},r.zeroMarkAdvances=function zeroMarkAdvances(r){for(var o=0;o<this.glyphInfos.length;o++)this.glyphInfos[o].isMark&&(r[o].xAdvance=0,r[o].yAdvance=0)},r.cleanup=function cleanup(){this.glyphInfos=null,this.plan=null,this.shaper=null},r.getAvailableFeatures=function getAvailableFeatures(r,o){var u=[];return this.GSUBProcessor&&(this.GSUBProcessor.selectScript(r,o),u.push.apply(u,Object.keys(this.GSUBProcessor.features))),this.GPOSProcessor&&(this.GPOSProcessor.selectScript(r,o),u.push.apply(u,Object.keys(this.GPOSProcessor.features))),u},OTLayoutEngine}(),Sa=function(){function LayoutEngine(r){this.font=r,this.unicodeLayoutEngine=null,this.kernProcessor=null,this.font.morx?this.engine=new Ni(this.font):(this.font.GSUB||this.font.GPOS)&&(this.engine=new _a(this.font))}var r=LayoutEngine.prototype;return r.layout=function layout(r,o,u,p,y){if("string"==typeof o&&(y=p,p=u,u=o,o=[]),"string"==typeof r){null==u&&(u=function forString(r){for(var o=r.length,u=0;u<o;){var p=r.charCodeAt(u++);if(55296<=p&&p<=56319&&u<o){var y=r.charCodeAt(u);56320<=y&&y<=57343&&(u++,p=((1023&p)<<10)+(1023&y)+65536)}var _=I.getScript(p);if("Common"!==_&&"Inherited"!==_&&"Unknown"!==_)return gi[_]}return gi.Unknown}(r));var _=this.font.glyphsForString(r)}else{if(null==u){for(var P,k=[],O=_createForOfIteratorHelperLoose(r);!(P=O()).done;){var B=P.value;k.push.apply(k,B.codePoints)}u=function forCodePoints(r){for(var o=0;o<r.length;o++){var u=r[o],p=I.getScript(u);if("Common"!==p&&"Inherited"!==p&&"Unknown"!==p)return gi[p]}return gi.Unknown}(k)}_=r}var D=new xi(_,o,u,p,y);return 0===_.length?(D.positions=[],D):(this.engine&&this.engine.setup&&this.engine.setup(D),this.substitute(D),this.position(D),this.hideDefaultIgnorables(D.glyphs,D.positions),this.engine&&this.engine.cleanup&&this.engine.cleanup(),D)},r.substitute=function substitute(r){this.engine&&this.engine.substitute&&this.engine.substitute(r)},r.position=function position(r){r.positions=r.glyphs.map((function(r){return new Ai(r.advanceWidth)}));var o=null;this.engine&&this.engine.position&&(o=this.engine.position(r)),o||this.engine&&!this.engine.fallbackPosition||(this.unicodeLayoutEngine||(this.unicodeLayoutEngine=new di(this.font)),this.unicodeLayoutEngine.positionGlyphs(r.glyphs,r.positions)),o&&o.kern||!1===r.features.kern||!this.font.kern||(this.kernProcessor||(this.kernProcessor=new hi(this.font)),this.kernProcessor.process(r.glyphs,r.positions),r.features.kern=!0)},r.hideDefaultIgnorables=function hideDefaultIgnorables(r,o){for(var u=this.font.glyphForCodePoint(32),p=0;p<r.length;p++)this.isDefaultIgnorable(r[p].codePoints[0])&&(r[p]=u,o[p].xAdvance=0,o[p].yAdvance=0)},r.isDefaultIgnorable=function isDefaultIgnorable(r){var o=r>>16;if(0===o)switch(r>>8){case 0:return 173===r;case 3:return 847===r;case 6:return 1564===r;case 23:return 6068<=r&&r<=6069;case 24:return 6155<=r&&r<=6158;case 32:return 8203<=r&&r<=8207||8234<=r&&r<=8238||8288<=r&&r<=8303;case 254:return 65024<=r&&r<=65039||65279===r;case 255:return 65520<=r&&r<=65528;default:return!1}else switch(o){case 1:return 113824<=r&&r<=113827||119155<=r&&r<=119162;case 14:return 917504<=r&&r<=921599;default:return!1}},r.getAvailableFeatures=function getAvailableFeatures(r,o){var u=[];return this.engine&&u.push.apply(u,this.engine.getAvailableFeatures(r,o)),this.font.kern&&-1===u.indexOf("kern")&&u.push("kern"),u},r.stringsForGlyph=function stringsForGlyph(r){for(var o,u=new Set,p=_createForOfIteratorHelperLoose(this.font._cmapProcessor.codePointsForGlyph(r));!(o=p()).done;){var y=o.value;u.add(String.fromCodePoint(y))}if(this.engine&&this.engine.stringsForGlyph)for(var _,P=_createForOfIteratorHelperLoose(this.engine.stringsForGlyph(r));!(_=P()).done;){var k=_.value;u.add(k)}return Array.from(u)},LayoutEngine}(),xa={moveTo:"M",lineTo:"L",quadraticCurveTo:"Q",bezierCurveTo:"C",closePath:"Z"},Aa=function(){function Path(){this.commands=[],this._bbox=null,this._cbox=null}var r=Path.prototype;return r.toFunction=function toFunction(){var r=this;return function(o){r.commands.forEach((function(r){return o[r.command].apply(o,r.args)}))}},r.toSVG=function toSVG(){return this.commands.map((function(r){var o=r.args.map((function(r){return Math.round(100*r)/100}));return"".concat(xa[r.command]).concat(o.join(" "))})).join("")},r.mapPoints=function mapPoints(r){for(var o,u=new Path,p=_createForOfIteratorHelperLoose(this.commands);!(o=p()).done;){for(var y=o.value,_=[],P=0;P<y.args.length;P+=2){var k=r(y.args[P],y.args[P+1]),I=k[0],O=k[1];_.push(I,O)}u[y.command].apply(u,_)}return u},r.transform=function transform(r,o,u,p,y,_){return this.mapPoints((function(P,k){return[P=r*P+u*k+y,k=o*P+p*k+_]}))},r.translate=function translate(r,o){return this.transform(1,0,0,1,r,o)},r.rotate=function rotate(r){var o=Math.cos(r),u=Math.sin(r);return this.transform(o,u,-u,o,0,0)},r.scale=function scale(r,o){return void 0===o&&(o=r),this.transform(r,0,0,o,0,0)},_createClass(Path,[{key:"cbox",get:function get(){if(!this._cbox){for(var r,o=new pi,u=_createForOfIteratorHelperLoose(this.commands);!(r=u()).done;)for(var p=r.value,y=0;y<p.args.length;y+=2)o.addPoint(p.args[y],p.args[y+1]);this._cbox=Object.freeze(o)}return this._cbox}},{key:"bbox",get:function get(){if(this._bbox)return this._bbox;for(var r,o=new pi,u=0,p=0,y=function f(r){return Math.pow(1-r,3)*K[te]+3*Math.pow(1-r,2)*r*J[te]+3*(1-r)*Math.pow(r,2)*$[te]+Math.pow(r,3)*ee[te]},_=_createForOfIteratorHelperLoose(this.commands);!(r=_()).done;){var P=r.value;switch(P.command){case"moveTo":case"lineTo":var k=P.args,I=k[0],O=k[1];o.addPoint(I,O),u=I,p=O;break;case"quadraticCurveTo":case"bezierCurveTo":if("quadraticCurveTo"===P.command)var B=P.args,D=B[0],R=B[1],N=u+2/3*(D-u),U=p+2/3*(R-p),W=(X=B[2])+2/3*(D-X),j=(Y=B[3])+2/3*(R-Y);else var G=P.args,X=(N=G[0],U=G[1],W=G[2],j=G[3],G[4]),Y=G[5];o.addPoint(X,Y);for(var K=[u,p],J=[N,U],$=[W,j],ee=[X,Y],te=0;te<=1;te++){var re=6*K[te]-12*J[te]+6*$[te],ne=-3*K[te]+9*J[te]-9*$[te]+3*ee[te];if(P=3*J[te]-3*K[te],0!==ne){var ie=Math.pow(re,2)-4*P*ne;if(!(ie<0)){var oe=(-re+Math.sqrt(ie))/(2*ne);0<oe&&oe<1&&(0===te?o.addPoint(y(oe),o.maxY):1===te&&o.addPoint(o.maxX,y(oe)));var ae=(-re-Math.sqrt(ie))/(2*ne);0<ae&&ae<1&&(0===te?o.addPoint(y(ae),o.maxY):1===te&&o.addPoint(o.maxX,y(ae)))}}else{if(0===re)continue;var se=-P/re;0<se&&se<1&&(0===te?o.addPoint(y(se),o.maxY):1===te&&o.addPoint(o.maxX,y(se)))}}u=X,p=Y}}return this._bbox=Object.freeze(o)}}])}(),Ca=function _loop(r){Aa.prototype[r]=function(){this._bbox=this._cbox=null;for(var o=arguments.length,u=new Array(o),p=0;p<o;p++)u[p]=arguments[p];return this.commands.push({command:r,args:u}),this}},Pa=0,ka=["moveTo","lineTo","quadraticCurveTo","bezierCurveTo","closePath"];Pa<ka.length;Pa++){Ca(ka[Pa])}var Ea,Ta,Ia=[".notdef",".null","nonmarkingreturn","space","exclam","quotedbl","numbersign","dollar","percent","ampersand","quotesingle","parenleft","parenright","asterisk","plus","comma","hyphen","period","slash","zero","one","two","three","four","five","six","seven","eight","nine","colon","semicolon","less","equal","greater","question","at","A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","W","X","Y","Z","bracketleft","backslash","bracketright","asciicircum","underscore","grave","a","b","c","d","e","f","g","h","i","j","k","l","m","n","o","p","q","r","s","t","u","v","w","x","y","z","braceleft","bar","braceright","asciitilde","Adieresis","Aring","Ccedilla","Eacute","Ntilde","Odieresis","Udieresis","aacute","agrave","acircumflex","adieresis","atilde","aring","ccedilla","eacute","egrave","ecircumflex","edieresis","iacute","igrave","icircumflex","idieresis","ntilde","oacute","ograve","ocircumflex","odieresis","otilde","uacute","ugrave","ucircumflex","udieresis","dagger","degree","cent","sterling","section","bullet","paragraph","germandbls","registered","copyright","trademark","acute","dieresis","notequal","AE","Oslash","infinity","plusminus","lessequal","greaterequal","yen","mu","partialdiff","summation","product","pi","integral","ordfeminine","ordmasculine","Omega","ae","oslash","questiondown","exclamdown","logicalnot","radical","florin","approxequal","Delta","guillemotleft","guillemotright","ellipsis","nonbreakingspace","Agrave","Atilde","Otilde","OE","oe","endash","emdash","quotedblleft","quotedblright","quoteleft","quoteright","divide","lozenge","ydieresis","Ydieresis","fraction","currency","guilsinglleft","guilsinglright","fi","fl","daggerdbl","periodcentered","quotesinglbase","quotedblbase","perthousand","Acircumflex","Ecircumflex","Aacute","Edieresis","Egrave","Iacute","Icircumflex","Idieresis","Igrave","Oacute","Ocircumflex","apple","Ograve","Uacute","Ucircumflex","Ugrave","dotlessi","circumflex","tilde","macron","breve","dotaccent","ring","cedilla","hungarumlaut","ogonek","caron","Lslash","lslash","Scaron","scaron","Zcaron","zcaron","brokenbar","Eth","eth","Yacute","yacute","Thorn","thorn","minus","multiply","onesuperior","twosuperior","threesuperior","onehalf","onequarter","threequarters","franc","Gbreve","gbreve","Idotaccent","Scedilla","scedilla","Cacute","cacute","Ccaron","ccaron","dcroat"],Oa=(Ea=function(){function Glyph(r,o,u){this.id=r,this.codePoints=o,this._font=u,this.isMark=this.codePoints.length>0&&this.codePoints.every(I.isMark),this.isLigature=this.codePoints.length>1}var r=Glyph.prototype;return r._getPath=function _getPath(){return new Aa},r._getCBox=function _getCBox(){return this.path.cbox},r._getBBox=function _getBBox(){return this.path.bbox},r._getTableMetrics=function _getTableMetrics(r){if(this.id<r.metrics.length)return r.metrics.get(this.id);var o=r.metrics.get(r.metrics.length-1);return{advance:o?o.advance:0,bearing:r.bearings.get(this.id-r.metrics.length)||0}},r._getMetrics=function _getMetrics(r){if(this._metrics)return this._metrics;var o,u=this._getTableMetrics(this._font.hmtx),p=u.advance,y=u.bearing;if(this._font.vmtx)var _=this._getTableMetrics(this._font.vmtx),P=_.advance,k=_.bearing;else if(null==r&&(r=this.cbox),(o=this._font["OS/2"])&&o.version>0)P=Math.abs(o.typoAscender-o.typoDescender),k=o.typoAscender-r.maxY;else{var I=this._font.hhea;P=Math.abs(I.ascent-I.descent),k=I.ascent-r.maxY}return this._font._variationProcessor&&this._font.HVAR&&(p+=this._font._variationProcessor.getAdvanceAdjustment(this.id,this._font.HVAR)),this._metrics={advanceWidth:p,advanceHeight:P,leftBearing:y,topBearing:k}},r.getScaledPath=function getScaledPath(r){var o=1/this._font.unitsPerEm*r;return this.path.scale(o)},r._getName=function _getName(){var r=this._font.post;if(!r)return null;switch(r.version){case 1:return Ia[this.id];case 2:var o=r.glyphNameIndex[this.id];return o<Ia.length?Ia[o]:r.names[o-Ia.length];case 2.5:return Ia[this.id+r.offsets[this.id]];case 4:return String.fromCharCode(r.map[this.id])}},r.render=function render(r,o){r.save();var u=1/this._font.head.unitsPerEm*o;r.scale(u,u),this.path.toFunction()(r),r.fill(),r.restore()},_createClass(Glyph,[{key:"cbox",get:function get(){return this._getCBox()}},{key:"bbox",get:function get(){return this._getBBox()}},{key:"path",get:function get(){return this._getPath()}},{key:"advanceWidth",get:function get(){return this._getMetrics().advanceWidth}},{key:"advanceHeight",get:function get(){return this._getMetrics().advanceHeight}},{key:"ligatureCaretPositions",get:function get(){}},{key:"name",get:function get(){return this._getName()}}])}(),_applyDecoratedDescriptor(Ea.prototype,"cbox",[cache],Object.getOwnPropertyDescriptor(Ea.prototype,"cbox"),Ea.prototype),_applyDecoratedDescriptor(Ea.prototype,"bbox",[cache],Object.getOwnPropertyDescriptor(Ea.prototype,"bbox"),Ea.prototype),_applyDecoratedDescriptor(Ea.prototype,"path",[cache],Object.getOwnPropertyDescriptor(Ea.prototype,"path"),Ea.prototype),_applyDecoratedDescriptor(Ea.prototype,"advanceWidth",[cache],Object.getOwnPropertyDescriptor(Ea.prototype,"advanceWidth"),Ea.prototype),_applyDecoratedDescriptor(Ea.prototype,"advanceHeight",[cache],Object.getOwnPropertyDescriptor(Ea.prototype,"advanceHeight"),Ea.prototype),_applyDecoratedDescriptor(Ea.prototype,"name",[cache],Object.getOwnPropertyDescriptor(Ea.prototype,"name"),Ea.prototype),Ea),Ba=new _.Struct({numberOfContours:_.int16,xMin:_.int16,yMin:_.int16,xMax:_.int16,yMax:_.int16}),Fa=function(){function Point(r,o,u,p){void 0===u&&(u=0),void 0===p&&(p=0),this.onCurve=r,this.endContour=o,this.x=u,this.y=p}return Point.prototype.copy=function copy(){return new Point(this.onCurve,this.endContour,this.x,this.y)},Point}(),La=function Component(r,o,u){this.glyphID=r,this.dx=o,this.dy=u,this.pos=0,this.scaleX=this.scaleY=1,this.scale01=this.scale10=0},Da=function(r){function TTFGlyph(){return r.apply(this,arguments)||this}_inheritsLoose(TTFGlyph,r);var o=TTFGlyph.prototype;return o._getCBox=function _getCBox(r){if(this._font._variationProcessor&&!r)return this.path.cbox;var o=this._font._getTableStream("glyf");o.pos+=this._font.loca.offsets[this.id];var u=Ba.decode(o),p=new pi(u.xMin,u.yMin,u.xMax,u.yMax);return Object.freeze(p)},o._parseGlyphCoord=function _parseGlyphCoord(r,o,u,p){if(u){var y=r.readUInt8();p||(y=-y),y+=o}else if(p)y=o;else y=o+r.readInt16BE();return y},o._decode=function _decode(){var r=this._font.loca.offsets[this.id];if(r===this._font.loca.offsets[this.id+1])return null;var o=this._font._getTableStream("glyf");o.pos+=r;var u=o.pos,p=Ba.decode(o);return p.numberOfContours>0?this._decodeSimple(p,o):p.numberOfContours<0&&this._decodeComposite(p,o,u),p},o._decodeSimple=function _decodeSimple(r,o){r.points=[];var u=new _.Array(_.uint16,r.numberOfContours).decode(o);r.instructions=new _.Array(_.uint8,_.uint16).decode(o);for(var p=[],y=u[u.length-1]+1;p.length<y;){var P=o.readUInt8();if(p.push(P),8&P)for(var k=o.readUInt8(),I=0;I<k;I++)p.push(P)}for(var O=0;O<p.length;O++){P=p[O];var B=new Fa(!!(1&P),u.indexOf(O)>=0,0,0);r.points.push(B)}var D=0;for(O=0;O<p.length;O++){P=p[O];r.points[O].x=D=this._parseGlyphCoord(o,D,2&P,16&P)}var R=0;for(O=0;O<p.length;O++){P=p[O];r.points[O].y=R=this._parseGlyphCoord(o,R,4&P,32&P)}if(this._font._variationProcessor){var N=r.points.slice();N.push.apply(N,this._getPhantomPoints(r)),this._font._variationProcessor.transformPoints(this.id,N),r.phantomPoints=N.slice(-4)}},o._decodeComposite=function _decodeComposite(r,o,u){void 0===u&&(u=0),r.components=[];for(var p=!1,y=32;32&y;){y=o.readUInt16BE();var _=o.pos-u,P=o.readUInt16BE();if(p||(p=!!(256&y)),1&y)var k=o.readInt16BE(),I=o.readInt16BE();else k=o.readInt8(),I=o.readInt8();(D=new La(P,k,I)).pos=_,8&y?D.scaleX=D.scaleY=(o.readUInt8()<<24|o.readUInt8()<<16)/1073741824:64&y?(D.scaleX=(o.readUInt8()<<24|o.readUInt8()<<16)/1073741824,D.scaleY=(o.readUInt8()<<24|o.readUInt8()<<16)/1073741824):128&y&&(D.scaleX=(o.readUInt8()<<24|o.readUInt8()<<16)/1073741824,D.scale01=(o.readUInt8()<<24|o.readUInt8()<<16)/1073741824,D.scale10=(o.readUInt8()<<24|o.readUInt8()<<16)/1073741824,D.scaleY=(o.readUInt8()<<24|o.readUInt8()<<16)/1073741824),r.components.push(D)}if(this._font._variationProcessor){for(var O=[],B=0;B<r.components.length;B++){var D=r.components[B];O.push(new Fa(!0,!0,D.dx,D.dy))}O.push.apply(O,this._getPhantomPoints(r)),this._font._variationProcessor.transformPoints(this.id,O),r.phantomPoints=O.splice(-4,4);for(var R=0;R<O.length;R++){var N=O[R];r.components[R].dx=N.x,r.components[R].dy=N.y}}return p},o._getPhantomPoints=function _getPhantomPoints(r){var o=this._getCBox(!0);null==this._metrics&&(this._metrics=Oa.prototype._getMetrics.call(this,o));var u=this._metrics,p=u.advanceWidth,y=u.advanceHeight,_=u.leftBearing,P=u.topBearing;return[new Fa(!1,!0,r.xMin-_,0),new Fa(!1,!0,r.xMin-_+p,0),new Fa(!1,!0,0,r.yMax+P),new Fa(!1,!0,0,r.yMax+P+y)]},o._getContours=function _getContours(){var r=this._decode();if(!r)return[];var o=[];if(r.numberOfContours<0)for(var u,p=_createForOfIteratorHelperLoose(r.components);!(u=p()).done;)for(var y=u.value,_=this._font.getGlyph(y.glyphID)._getContours(),P=0;P<_.length;P++)for(var k=_[P],I=0;I<k.length;I++){var O=k[I],B=O.x*y.scaleX+O.y*y.scale01+y.dx,D=O.y*y.scaleY+O.x*y.scale10+y.dy;o.push(new Fa(O.onCurve,O.endContour,B,D))}else o=r.points||[];r.phantomPoints&&!this._font.directory.tables.HVAR&&(this._metrics.advanceWidth=r.phantomPoints[1].x-r.phantomPoints[0].x,this._metrics.advanceHeight=r.phantomPoints[3].y-r.phantomPoints[2].y,this._metrics.leftBearing=r.xMin-r.phantomPoints[0].x,this._metrics.topBearing=r.phantomPoints[2].y-r.yMax);for(var R=[],N=[],U=0;U<o.length;U++){var W=o[U];N.push(W),W.endContour&&(R.push(N),N=[])}return R},o._getMetrics=function _getMetrics(){if(this._metrics)return this._metrics;var o=this._getCBox(!0);return r.prototype._getMetrics.call(this,o),this._font._variationProcessor&&!this._font.HVAR&&this.path,this._metrics},o._getPath=function _getPath(){for(var r=this._getContours(),o=new Aa,u=0;u<r.length;u++){var p=r[u],y=p[0],_=p[p.length-1],P=0;if(y.onCurve){var k=null;P=1}else k=y=_.onCurve?_:new Fa(!1,!1,(y.x+_.x)/2,(y.y+_.y)/2);o.moveTo(y.x,y.y);for(var I=P;I<p.length;I++){var O=p[I],B=0===I?y:p[I-1];if(B.onCurve&&O.onCurve)o.lineTo(O.x,O.y);else if(B.onCurve&&!O.onCurve)k=O;else if(B.onCurve||O.onCurve){if(B.onCurve||!O.onCurve)throw new Error("Unknown TTF path state");o.quadraticCurveTo(k.x,k.y,O.x,O.y);k=null}else{var D=(B.x+O.x)/2,R=(B.y+O.y)/2;o.quadraticCurveTo(B.x,B.y,D,R);var k=O}}k&&o.quadraticCurveTo(k.x,k.y,y.x,y.y),o.closePath()}return o},TTFGlyph}(Oa),Ra=function(r){function CFFGlyph(){return r.apply(this,arguments)||this}_inheritsLoose(CFFGlyph,r);var o=CFFGlyph.prototype;return o._getName=function _getName(){return this._font.CFF2?r.prototype._getName.call(this):this._font["CFF "].getGlyphName(this.id)},o.bias=function bias(r){return r.length<1240?107:r.length<33900?1131:32768},o._getPath=function _getPath(){var r=this._font.CFF2||this._font["CFF "],o=r.stream,u=r.topDict.CharStrings[this.id],p=u.offset+u.length;o.pos=u.offset;var y,_,P=new Aa,k=[],I=[],O=null,B=0,D=0,R=0,N=!1;this._usedGsubrs=y={},this._usedSubrs=_={};var U=r.globalSubrIndex||[],W=this.bias(U),j=r.privateDictForGlyph(this.id)||{},G=j.Subrs||[],X=this.bias(G),Y=r.topDict.vstore&&r.topDict.vstore.itemVariationStore,K=j.vsindex,J=this._font._variationProcessor;function checkWidth(){null==O&&(O=k.shift()+j.nominalWidthX)}function parseStems(){return k.length%2!=0&&checkWidth(),B+=k.length>>1,k.length=0}function moveTo(r,o){N&&P.closePath(),P.moveTo(r,o),N=!0}var $=function parse(){for(;o.pos<p;){var u=o.readUInt8();if(u<32)switch(u){case 1:case 3:case 18:case 23:parseStems();break;case 4:k.length>1&&checkWidth(),R+=k.shift(),moveTo(D,R);break;case 5:for(;k.length>=2;)D+=k.shift(),R+=k.shift(),P.lineTo(D,R);break;case 6:case 7:for(var O=6===u;k.length>=1;)O?D+=k.shift():R+=k.shift(),P.lineTo(D,R),O=!O;break;case 8:for(;k.length>0;){var j=D+k.shift(),ee=R+k.shift(),te=j+k.shift(),re=ee+k.shift();D=te+k.shift(),R=re+k.shift(),P.bezierCurveTo(j,ee,te,re,D,R)}break;case 10:var ne=k.pop()+X,ie=G[ne];if(ie){_[ne]=!0;var oe=o.pos,ae=p;o.pos=ie.offset,p=ie.offset+ie.length,$(),o.pos=oe,p=ae}break;case 11:if(r.version>=2)break;return;case 14:if(r.version>=2)break;k.length>0&&checkWidth(),N&&(P.closePath(),N=!1);break;case 15:if(r.version<2)throw new Error("vsindex operator not supported in CFF v1");K=k.pop();break;case 16:if(r.version<2)throw new Error("blend operator not supported in CFF v1");if(!J)throw new Error("blend operator in non-variation font");for(var se=J.getBlendVector(Y,K),ce=k.pop(),le=ce*se.length,ue=k.length-le,fe=ue-ce,he=0;he<ce;he++){for(var de=k[fe+he],pe=0;pe<se.length;pe++)de+=se[pe]*k[ue++];k[fe+he]=de}for(;le--;)k.pop();break;case 19:case 20:parseStems(),o.pos+=B+7>>3;break;case 21:k.length>2&&checkWidth(),D+=k.shift(),R+=k.shift(),moveTo(D,R);break;case 22:k.length>1&&checkWidth(),moveTo(D+=k.shift(),R);break;case 24:for(;k.length>=8;){j=D+k.shift(),ee=R+k.shift(),te=j+k.shift(),re=ee+k.shift();D=te+k.shift(),R=re+k.shift(),P.bezierCurveTo(j,ee,te,re,D,R)}D+=k.shift(),R+=k.shift(),P.lineTo(D,R);break;case 25:for(;k.length>=8;)D+=k.shift(),R+=k.shift(),P.lineTo(D,R);j=D+k.shift(),ee=R+k.shift(),te=j+k.shift(),re=ee+k.shift();D=te+k.shift(),R=re+k.shift(),P.bezierCurveTo(j,ee,te,re,D,R);break;case 26:for(k.length%2&&(D+=k.shift());k.length>=4;)j=D,ee=R+k.shift(),te=j+k.shift(),re=ee+k.shift(),D=te,R=re+k.shift(),P.bezierCurveTo(j,ee,te,re,D,R);break;case 27:for(k.length%2&&(R+=k.shift());k.length>=4;)j=D+k.shift(),ee=R,te=j+k.shift(),re=ee+k.shift(),D=te+k.shift(),R=re,P.bezierCurveTo(j,ee,te,re,D,R);break;case 28:k.push(o.readInt16BE());break;case 29:if(ne=k.pop()+W,ie=U[ne]){y[ne]=!0;oe=o.pos,ae=p;o.pos=ie.offset,p=ie.offset+ie.length,$(),o.pos=oe,p=ae}break;case 30:case 31:for(O=31===u;k.length>=4;)O?(j=D+k.shift(),ee=R,te=j+k.shift(),re=ee+k.shift(),R=re+k.shift(),D=te+(1===k.length?k.shift():0)):(j=D,ee=R+k.shift(),te=j+k.shift(),re=ee+k.shift(),D=te+k.shift(),R=re+(1===k.length?k.shift():0)),P.bezierCurveTo(j,ee,te,re,D,R),O=!O;break;case 12:switch(u=o.readUInt8()){case 3:var ge=k.pop(),ye=k.pop();k.push(ge&&ye?1:0);break;case 4:ge=k.pop(),ye=k.pop(),k.push(ge||ye?1:0);break;case 5:ge=k.pop(),k.push(ge?0:1);break;case 9:ge=k.pop(),k.push(Math.abs(ge));break;case 10:ge=k.pop(),ye=k.pop(),k.push(ge+ye);break;case 11:ge=k.pop(),ye=k.pop(),k.push(ge-ye);break;case 12:ge=k.pop(),ye=k.pop(),k.push(ge/ye);break;case 14:ge=k.pop(),k.push(-ge);break;case 15:ge=k.pop(),ye=k.pop(),k.push(ge===ye?1:0);break;case 18:k.pop();break;case 20:var ve=k.pop(),me=k.pop();I[me]=ve;break;case 21:me=k.pop(),k.push(I[me]||0);break;case 22:var be=k.pop(),we=k.pop(),_e=k.pop(),Se=k.pop();k.push(_e<=Se?be:we);break;case 23:k.push(Math.random());break;case 24:ge=k.pop(),ye=k.pop(),k.push(ge*ye);break;case 26:ge=k.pop(),k.push(Math.sqrt(ge));break;case 27:ge=k.pop(),k.push(ge,ge);break;case 28:ge=k.pop(),ye=k.pop(),k.push(ye,ge);break;case 29:(me=k.pop())<0?me=0:me>k.length-1&&(me=k.length-1),k.push(k[me]);break;case 30:var xe=k.pop(),Ae=k.pop();if(Ae>=0)for(;Ae>0;){for(var Ce=k[xe-1],Pe=xe-2;Pe>=0;Pe--)k[Pe+1]=k[Pe];k[0]=Ce,Ae--}else for(;Ae<0;){Ce=k[0];for(var ke=0;ke<=xe;ke++)k[ke]=k[ke+1];k[xe-1]=Ce,Ae++}break;case 34:j=D+k.shift(),ee=R,te=j+k.shift(),re=ee+k.shift();var Ee=te+k.shift(),Te=re,Ie=Ee+k.shift(),Oe=Te,Be=Ie+k.shift(),Fe=Oe,Le=Be+k.shift(),De=Fe;D=Le,R=De,P.bezierCurveTo(j,ee,te,re,Ee,Te),P.bezierCurveTo(Ie,Oe,Be,Fe,Le,De);break;case 35:for(var Re=[],Me=0;Me<=5;Me++)D+=k.shift(),R+=k.shift(),Re.push(D,R);P.bezierCurveTo.apply(P,Re.slice(0,6)),P.bezierCurveTo.apply(P,Re.slice(6)),k.shift();break;case 36:j=D+k.shift(),ee=R+k.shift(),te=j+k.shift(),Oe=Te=re=ee+k.shift(),Be=(Ie=(Ee=te+k.shift())+k.shift())+k.shift(),Fe=Oe+k.shift(),Le=Be+k.shift(),D=Le,R=De=Fe,P.bezierCurveTo(j,ee,te,re,Ee,Te),P.bezierCurveTo(Ie,Oe,Be,Fe,Le,De);break;case 37:var Ne=D,Ue=R;Re=[];for(var We=0;We<=4;We++)D+=k.shift(),R+=k.shift(),Re.push(D,R);Math.abs(D-Ne)>Math.abs(R-Ue)?(D+=k.shift(),R=Ue):(D=Ne,R+=k.shift()),Re.push(D,R),P.bezierCurveTo.apply(P,Re.slice(0,6)),P.bezierCurveTo.apply(P,Re.slice(6));break;default:throw new Error("Unknown op: 12 ".concat(u))}break;default:throw new Error("Unknown op: ".concat(u))}else if(u<247)k.push(u-139);else if(u<251){var ze=o.readUInt8();k.push(256*(u-247)+ze+108)}else if(u<255){ze=o.readUInt8();k.push(256*-(u-251)-ze-108)}else k.push(o.readInt32BE()/65536)}};return $(),N&&P.closePath(),P},CFFGlyph}(Oa),Ma=new _.Struct({originX:_.uint16,originY:_.uint16,type:new _.String(4),data:new _.Buffer((function(r){return r.parent.buflen-r._currentOffset}))}),Na=function(r){function SBIXGlyph(){return r.apply(this,arguments)||this}_inheritsLoose(SBIXGlyph,r);var o=SBIXGlyph.prototype;return o.getImageForSize=function getImageForSize(r){for(var o=0;o<this._font.sbix.imageTables.length;o++){var u=this._font.sbix.imageTables[o];if(u.ppem>=r)break}var p=u.imageOffsets,y=p[this.id],_=p[this.id+1];return y===_?null:(this._font.stream.pos=y,Ma.decode(this._font.stream,{buflen:_-y}))},o.render=function render(o,u){var p=this.getImageForSize(u);if(null!=p){var y=u/this._font.unitsPerEm;o.image(p.data,{height:u,x:p.originX,y:(this.bbox.minY-p.originY)*y})}this._font.sbix.flags.renderOutlines&&r.prototype.render.call(this,o,u)},SBIXGlyph}(Da),Ua=function COLRLayer(r,o){this.glyph=r,this.color=o},Wa=function(r){function COLRGlyph(){return r.apply(this,arguments)||this}_inheritsLoose(COLRGlyph,r);var o=COLRGlyph.prototype;return o._getBBox=function _getBBox(){for(var r=new pi,o=0;o<this.layers.length;o++){var u=this.layers[o].glyph.bbox;r.addPoint(u.minX,u.minY),r.addPoint(u.maxX,u.maxY)}return r},o.render=function render(r,o){for(var u,p=_createForOfIteratorHelperLoose(this.layers);!(u=p()).done;){var y=u.value,_=y.glyph,P=y.color;r.fillColor([P.red,P.green,P.blue],P.alpha/255*100),_.render(r,o)}},_createClass(COLRGlyph,[{key:"layers",get:function get(){for(var r=this._font.CPAL,o=this._font.COLR,u=0,p=o.baseGlyphRecord.length-1;u<=p;){var y=u+p>>1,_=o.baseGlyphRecord[y];if(this.id<_.gid)p=y-1;else{if(!(this.id>_.gid)){var P=_;break}u=y+1}}if(null==P){var k=this._font._getBaseGlyph(this.id);return[new Ua(k,B={red:0,green:0,blue:0,alpha:255})]}for(var I=[],O=P.firstLayerIndex;O<P.firstLayerIndex+P.numLayers;O++){_=o.layerRecords[O];var B=r.colorRecords[_.paletteIndex];k=this._font._getBaseGlyph(_.gid);I.push(new Ua(k,B))}return I}}])}(Oa),za=function(){function GlyphVariationProcessor(r,o){this.font=r,this.normalizedCoords=this.normalizeCoords(o),this.blendVectors=new Map}var r=GlyphVariationProcessor.prototype;return r.normalizeCoords=function normalizeCoords(r){for(var o=[],u=0;u<this.font.fvar.axis.length;u++){var p=this.font.fvar.axis[u];r[u]<p.defaultValue?o.push((r[u]-p.defaultValue+Number.EPSILON)/(p.defaultValue-p.minValue+Number.EPSILON)):o.push((r[u]-p.defaultValue+Number.EPSILON)/(p.maxValue-p.defaultValue+Number.EPSILON))}if(this.font.avar)for(u=0;u<this.font.avar.segment.length;u++)for(var y=this.font.avar.segment[u],_=0;_<y.correspondence.length;_++){var P=y.correspondence[_];if(_>=1&&o[u]<P.fromCoord){var k=y.correspondence[_-1];o[u]=((o[u]-k.fromCoord)*(P.toCoord-k.toCoord)+Number.EPSILON)/(P.fromCoord-k.fromCoord+Number.EPSILON)+k.toCoord;break}}return o},r.transformPoints=function transformPoints(r,o){if(this.font.fvar&&this.font.gvar){var u=this.font.gvar;if(!(r>=u.glyphCount)){var p=u.offsets[r];if(p!==u.offsets[r+1]){var y=this.font.stream;if(y.pos=p,!(y.pos>=y.length)){var _=y.readUInt16BE(),P=p+y.readUInt16BE();if(32768&_){var k=y.pos;y.pos=P;var I=this.decodePoints();P=y.pos,y.pos=k}var O=o.map((function(r){return r.copy()}));_&=4095;for(var B=0;B<_;B++){var D=y.readUInt16BE(),R=y.readUInt16BE();if(32768&R)for(var N=[],U=0;U<u.axisCount;U++)N.push(y.readInt16BE()/16384);else{if((4095&R)>=u.globalCoordCount)throw new Error("Invalid gvar table");N=u.globalCoords[4095&R]}if(16384&R){for(var W=[],j=0;j<u.axisCount;j++)W.push(y.readInt16BE()/16384);for(var G=[],X=0;X<u.axisCount;X++)G.push(y.readInt16BE()/16384)}var Y=this.tupleFactor(R,N,W,G);if(0!==Y){k=y.pos;if(y.pos=P,8192&R)var K=this.decodePoints();else K=I;var J=0===K.length?o.length:K.length,$=this.decodeDeltas(J),ee=this.decodeDeltas(J);if(0===K.length)for(var te=0;te<o.length;te++){var re=o[te];re.x+=Math.round($[te]*Y),re.y+=Math.round(ee[te]*Y)}else{for(var ne=O.map((function(r){return r.copy()})),ie=o.map((function(){return!1})),oe=0;oe<K.length;oe++){var ae=K[oe];if(ae<o.length){var se=ne[ae];ie[ae]=!0,se.x+=Math.round($[oe]*Y),se.y+=Math.round(ee[oe]*Y)}}this.interpolateMissingDeltas(ne,O,ie);for(var ce=0;ce<o.length;ce++){var le=ne[ce].x-O[ce].x,ue=ne[ce].y-O[ce].y;o[ce].x+=le,o[ce].y+=ue}}P+=D,y.pos=k}else P+=D}}}}}},r.decodePoints=function decodePoints(){var r=this.font.stream,o=r.readUInt8();128&o&&(o=(127&o)<<8|r.readUInt8());for(var u=new Uint16Array(o),p=0,y=0;p<o;)for(var _=r.readUInt8(),P=1+(127&_),k=128&_?r.readUInt16:r.readUInt8,I=0;I<P&&p<o;I++)y+=k.call(r),u[p++]=y;return u},r.decodeDeltas=function decodeDeltas(r){for(var o=this.font.stream,u=0,p=new Int16Array(r);u<r;){var y=o.readUInt8(),_=1+(63&y);if(128&y)u+=_;else for(var P=64&y?o.readInt16BE:o.readInt8,k=0;k<_&&u<r;k++)p[u++]=P.call(o)}return p},r.tupleFactor=function tupleFactor(r,o,u,p){for(var y=this.normalizedCoords,_=this.font.gvar,P=1,k=0;k<_.axisCount;k++)if(0!==o[k]){if(0===y[k])return 0;if(16384&r){if(y[k]<u[k]||y[k]>p[k])return 0;P=y[k]<o[k]?P*(y[k]-u[k]+Number.EPSILON)/(o[k]-u[k]+Number.EPSILON):P*(p[k]-y[k]+Number.EPSILON)/(p[k]-o[k]+Number.EPSILON)}else{if(y[k]<Math.min(0,o[k])||y[k]>Math.max(0,o[k]))return 0;P=(P*y[k]+Number.EPSILON)/(o[k]+Number.EPSILON)}}return P},r.interpolateMissingDeltas=function interpolateMissingDeltas(r,o,u){if(0!==r.length)for(var p=0;p<r.length;){for(var y=p,_=p,P=r[_];!P.endContour;)P=r[++_];for(;p<=_&&!u[p];)p++;if(!(p>_)){var k=p,I=p;for(p++;p<=_;)u[p]&&(this.deltaInterpolate(I+1,p-1,I,p,o,r),I=p),p++;I===k?this.deltaShift(y,_,I,o,r):(this.deltaInterpolate(I+1,_,I,k,o,r),k>0&&this.deltaInterpolate(y,k-1,I,k,o,r)),p=_+1}}},r.deltaInterpolate=function deltaInterpolate(r,o,u,p,y,_){if(!(r>o))for(var P=["x","y"],k=0;k<P.length;k++){var I=P[k];if(y[u][I]>y[p][I]){var O=u;u=p,p=O}var B=y[u][I],D=y[p][I],R=_[u][I],N=_[p][I];if(B!==D||R===N)for(var U=B===D?0:(N-R)/(D-B),W=r;W<=o;W++){var j=y[W][I];j<=B?j+=R-B:j>=D?j+=N-D:j=R+(j-B)*U,_[W][I]=j}}},r.deltaShift=function deltaShift(r,o,u,p,y){var _=y[u].x-p[u].x,P=y[u].y-p[u].y;if(0!==_||0!==P)for(var k=r;k<=o;k++)k!==u&&(y[k].x+=_,y[k].y+=P)},r.getAdvanceAdjustment=function getAdvanceAdjustment(r,o){var u,p;if(o.advanceWidthMapping){var y=r;y>=o.advanceWidthMapping.mapCount&&(y=o.advanceWidthMapping.mapCount-1);o.advanceWidthMapping.entryFormat;var _=o.advanceWidthMapping.mapData[y];u=_.outerIndex,p=_.innerIndex}else u=0,p=r;return this.getDelta(o.itemVariationStore,u,p)},r.getDelta=function getDelta(r,o,u){if(o>=r.itemVariationData.length)return 0;var p=r.itemVariationData[o];if(u>=p.deltaSets.length)return 0;for(var y=p.deltaSets[u],_=this.getBlendVector(r,o),P=0,k=0;k<p.regionIndexCount;k++)P+=y.deltas[k]*_[k];return P},r.getBlendVector=function getBlendVector(r,o){var u=r.itemVariationData[o];if(this.blendVectors.has(u))return this.blendVectors.get(u);for(var p=this.normalizedCoords,y=[],_=0;_<u.regionIndexCount;_++){for(var P=1,k=u.regionIndexes[_],I=r.variationRegionList.variationRegions[k],O=0;O<I.length;O++){var B=I[O];P*=B.startCoord>B.peakCoord||B.peakCoord>B.endCoord||B.startCoord<0&&B.endCoord>0&&0!==B.peakCoord||0===B.peakCoord?1:p[O]<B.startCoord||p[O]>B.endCoord?0:p[O]===B.peakCoord?1:p[O]<B.peakCoord?(p[O]-B.startCoord+Number.EPSILON)/(B.peakCoord-B.startCoord+Number.EPSILON):(B.endCoord-p[O]+Number.EPSILON)/(B.endCoord-B.peakCoord+Number.EPSILON)}y[_]=P}return this.blendVectors.set(u,y),y},GlyphVariationProcessor}(),ja=Promise.resolve(),Ga=function(){function Subset(r){this.font=r,this.glyphs=[],this.mapping={},this.includeGlyph(0)}var r=Subset.prototype;return r.includeGlyph=function includeGlyph(r){return"object"==typeof r&&(r=r.id),null==this.mapping[r]&&(this.glyphs.push(r),this.mapping[r]=this.glyphs.length-1),this.mapping[r]},r.encodeStream=function encodeStream(){var r=this,o=new _.EncodeStream;return ja.then((function(){return r.encode(o),o.end()})),o},Subset}(),Va=function(){function Point$1(){}return Point$1.size=function size(r){return r>=0&&r<=255?1:2},Point$1.encode=function encode(r,o){o>=0&&o<=255?r.writeUInt8(o):r.writeInt16BE(o)},Point$1}(),Ha=new _.Struct({numberOfContours:_.int16,xMin:_.int16,yMin:_.int16,xMax:_.int16,yMax:_.int16,endPtsOfContours:new _.Array(_.uint16,"numberOfContours"),instructions:new _.Array(_.uint8,_.uint16),flags:new _.Array(_.uint8,0),xPoints:new _.Array(Va,0),yPoints:new _.Array(Va,0)}),qa=function(){function TTFGlyphEncoder(){}var r=TTFGlyphEncoder.prototype;return r.encodeSimple=function encodeSimple(r,o){void 0===o&&(o=[]);for(var u=[],p=[],y=[],P=[],k=0,I=0,O=0,B=0,D=0,R=0;R<r.commands.length;R++){for(var N=r.commands[R],U=0;U<N.args.length;U+=2){var W=N.args[U],j=N.args[U+1],G=0;if("quadraticCurveTo"===N.command&&2===U){var X=r.commands[R+1];if(X&&"quadraticCurveTo"===X.command){var Y=(I+X.args[0])/2,K=(O+X.args[1])/2;if(W===Y&&j===K)continue}}"quadraticCurveTo"===N.command&&0===U||(G|=1),G=this._encodePoint(W,I,p,G,2,16),(G=this._encodePoint(j,O,y,G,4,32))===B&&k<255?(P[P.length-1]|=8,k++):(k>0&&(P.push(k),k=0),P.push(G),B=G),I=W,O=j,D++}"closePath"===N.command&&u.push(D-1)}r.commands.length>1&&"closePath"!==r.commands[r.commands.length-1].command&&u.push(D-1);var J=r.bbox,$={numberOfContours:u.length,xMin:J.minX,yMin:J.minY,xMax:J.maxX,yMax:J.maxY,endPtsOfContours:u,instructions:o,flags:P,xPoints:p,yPoints:y},ee=Ha.size($),te=4-ee%4,re=new _.EncodeStream(ee+te);return Ha.encode(re,$),0!==te&&re.fill(0,te),re.buffer},r._encodePoint=function _encodePoint(r,o,u,p,y,_){var P=r-o;return r===o?p|=_:(-255<=P&&P<=255&&(p|=y,P<0?P=-P:p|=_),u.push(P)),p},TTFGlyphEncoder}(),Xa=function(r){function TTFSubset(o){var u;return(u=r.call(this,o)||this).glyphEncoder=new qa,u}_inheritsLoose(TTFSubset,r);var o=TTFSubset.prototype;return o._addGlyph=function _addGlyph(r){var o=this.font.getGlyph(r),u=o._decode(),p=this.font.loca.offsets[r],_=this.font.loca.offsets[r+1],P=this.font._getTableStream("glyf");P.pos+=p;var k=P.readBuffer(_-p);if(u&&u.numberOfContours<0){k=y.from(k);for(var I,O=_createForOfIteratorHelperLoose(u.components);!(I=O()).done;){var B=I.value;r=this.includeGlyph(B.glyphID),k.writeUInt16BE(r,B.pos)}}else u&&this.font._variationProcessor&&(k=this.glyphEncoder.encodeSimple(o.path,u.instructions));return this.glyf.push(k),this.loca.offsets.push(this.offset),this.hmtx.metrics.push({advance:o.advanceWidth,bearing:o._getMetrics().leftBearing}),this.offset+=k.length,this.glyf.length-1},o.encode=function encode(r){this.glyf=[],this.offset=0,this.loca={offsets:[],version:this.font.loca.version},this.hmtx={metrics:[],bearings:[]};for(var o=0;o<this.glyphs.length;)this._addGlyph(this.glyphs[o++]);var u=D(this.font.maxp);u.numGlyphs=this.glyf.length,this.loca.offsets.push(this.offset);var p=D(this.font.head);p.indexToLocFormat=this.loca.version;var y=D(this.font.hhea);y.numberOfMetrics=this.hmtx.metrics.length,li.encode(r,{tables:{head:p,hhea:y,loca:this.loca,maxp:u,"cvt ":this.font["cvt "],prep:this.font.prep,glyf:this.glyf,hmtx:this.hmtx,fpgm:this.font.fpgm}})},TTFSubset}(Ga),Za=function(r){function CFFSubset(o){var u;if((u=r.call(this,o)||this).cff=u.font["CFF "],!u.cff)throw new Error("Not a CFF Font");return u}_inheritsLoose(CFFSubset,r);var o=CFFSubset.prototype;return o.subsetCharstrings=function subsetCharstrings(){this.charstrings=[];for(var r,o={},u=_createForOfIteratorHelperLoose(this.glyphs);!(r=u()).done;){var p=r.value;this.charstrings.push(this.cff.getCharString(p));var y=this.font.getGlyph(p);y.path;for(var _ in y._usedGsubrs)o[_]=!0}this.gsubrs=this.subsetSubrs(this.cff.globalSubrIndex,o)},o.subsetSubrs=function subsetSubrs(r,o){for(var u=[],p=0;p<r.length;p++){var _=r[p];o[p]?(this.cff.stream.pos=_.offset,u.push(this.cff.stream.readBuffer(_.length))):u.push(y.from([11]))}return u},o.subsetFontdict=function subsetFontdict(r){r.FDArray=[],r.FDSelect={version:0,fds:[]};for(var o,u={},p=[],y={},_=_createForOfIteratorHelperLoose(this.glyphs);!(o=_()).done;){var P=o.value,k=this.cff.fdForGlyph(P);if(null!=k){u[k]||(r.FDArray.push(Object.assign({},this.cff.topDict.FDArray[k])),p.push({}),y[k]=r.FDArray.length-1),u[k]=!0,r.FDSelect.fds.push(y[k]);var I=this.font.getGlyph(P);I.path;for(var O in I._usedSubrs)p[y[k]][O]=!0}}for(var B=0;B<r.FDArray.length;B++){var D=r.FDArray[B];delete D.FontName,D.Private&&D.Private.Subrs&&(D.Private=Object.assign({},D.Private),D.Private.Subrs=this.subsetSubrs(D.Private.Subrs,p[B]))}},o.createCIDFontdict=function createCIDFontdict(r){for(var o,u={},p=_createForOfIteratorHelperLoose(this.glyphs);!(o=p()).done;){var y=o.value,_=this.font.getGlyph(y);_.path;for(var P in _._usedSubrs)u[P]=!0}var k=Object.assign({},this.cff.topDict.Private);return this.cff.topDict.Private&&this.cff.topDict.Private.Subrs&&(k.Subrs=this.subsetSubrs(this.cff.topDict.Private.Subrs,u)),r.FDArray=[{Private:k}],r.FDSelect={version:3,nRanges:1,ranges:[{first:0,fd:0}],sentinel:this.charstrings.length}},o.addString=function addString(r){return r?(this.strings||(this.strings=[]),this.strings.push(r),Be.length+this.strings.length-1):null},o.encode=function encode(r){this.subsetCharstrings();var o={version:this.charstrings.length>255?2:1,ranges:[{first:1,nLeft:this.charstrings.length-2}]},u=Object.assign({},this.cff.topDict);u.Private=null,u.charset=o,u.Encoding=null,u.CharStrings=this.charstrings;for(var p=0,y=["version","Notice","Copyright","FullName","FamilyName","Weight","PostScript","BaseFontName","FontName"];p<y.length;p++){var _=y[p];u[_]=this.addString(this.cff.string(u[_]))}u.ROS=[this.addString("Adobe"),this.addString("Identity"),0],u.CIDCount=this.charstrings.length,this.cff.isCIDFont?this.subsetFontdict(u):this.createCIDFontdict(u);var P={version:1,hdrSize:this.cff.hdrSize,offSize:4,header:this.cff.header,nameIndex:[this.cff.postscriptName],topDictIndex:[u],stringIndex:this.strings,globalSubrIndex:this.gsubrs};Ft.encode(r,P)},CFFSubset}(Ga),Ya=(Ta=function(){function TTFFont(r,o){for(var u in void 0===o&&(o=null),this.defaultLanguage=null,this.stream=r,this.variationCoords=o,this._directoryPos=this.stream.pos,this._tables={},this._glyphs={},this._decodeDirectory(),this.directory.tables){var p=this.directory.tables[u];ai[u]&&p.length>0&&Object.defineProperty(this,u,{get:this._getTable.bind(this,p)})}}TTFFont.probe=function probe(r){var o=r.toString("ascii",0,4);return"true"===o||"OTTO"===o||o===String.fromCharCode(0,1,0,0)};var r=TTFFont.prototype;return r.setDefaultLanguage=function setDefaultLanguage(r){void 0===r&&(r=null),this.defaultLanguage=r},r._getTable=function _getTable(r){if(!(r.tag in this._tables))try{this._tables[r.tag]=this._decodeTable(r)}catch(r){W.logErrors}return this._tables[r.tag]},r._getTableStream=function _getTableStream(r){var o=this.directory.tables[r];return o?(this.stream.pos=o.offset,this.stream):null},r._decodeDirectory=function _decodeDirectory(){return this.directory=li.decode(this.stream,{_startOffset:0})},r._decodeTable=function _decodeTable(r){var o=this.stream.pos,u=this._getTableStream(r.tag),p=ai[r.tag].decode(u,this,r.length);return this.stream.pos=o,p},r.getName=function getName(r,o){void 0===o&&(o=this.defaultLanguage||W.defaultLanguage);var u=this.name&&this.name.records[r];return u&&(u[o]||u[this.defaultLanguage]||u[W.defaultLanguage]||u.en||u[Object.keys(u)[0]])||null},r.hasGlyphForCodePoint=function hasGlyphForCodePoint(r){return!!this._cmapProcessor.lookup(r)},r.glyphForCodePoint=function glyphForCodePoint(r){return this.getGlyph(this._cmapProcessor.lookup(r),[r])},r.glyphsForString=function glyphsForString(r){for(var o=[],u=r.length,p=0,y=-1,_=-1;p<=u;){var P=0,k=0;if(p<u){if(55296<=(P=r.charCodeAt(p++))&&P<=56319&&p<u){var I=r.charCodeAt(p);56320<=I&&I<=57343&&(p++,P=((1023&P)<<10)+(1023&I)+65536)}k=65024<=P&&P<=65039||917760<=P&&P<=917999?1:0}else p++;0===_&&1===k?o.push(this.getGlyph(this._cmapProcessor.lookup(y,P),[y,P])):0===_&&0===k&&o.push(this.glyphForCodePoint(y)),y=P,_=k}return o},r.layout=function layout(r,o,u,p,y){return this._layoutEngine.layout(r,o,u,p,y)},r.stringsForGlyph=function stringsForGlyph(r){return this._layoutEngine.stringsForGlyph(r)},r.getAvailableFeatures=function getAvailableFeatures(r,o){return this._layoutEngine.getAvailableFeatures(r,o)},r._getBaseGlyph=function _getBaseGlyph(r,o){return void 0===o&&(o=[]),this._glyphs[r]||(this.directory.tables.glyf?this._glyphs[r]=new Da(r,o,this):(this.directory.tables["CFF "]||this.directory.tables.CFF2)&&(this._glyphs[r]=new Ra(r,o,this))),this._glyphs[r]||null},r.getGlyph=function getGlyph(r,o){return void 0===o&&(o=[]),this._glyphs[r]||(this.directory.tables.sbix?this._glyphs[r]=new Na(r,o,this):this.directory.tables.COLR&&this.directory.tables.CPAL?this._glyphs[r]=new Wa(r,o,this):this._getBaseGlyph(r,o)),this._glyphs[r]||null},r.createSubset=function createSubset(){return this.directory.tables["CFF "]?new Za(this):new Xa(this)},r.getVariation=function getVariation(r){if(!this.directory.tables.fvar||!(this.directory.tables.gvar&&this.directory.tables.glyf||this.directory.tables.CFF2))throw new Error("Variations require a font with the fvar, gvar and glyf, or CFF2 tables.");if("string"==typeof r&&(r=this.namedVariations[r]),"object"!=typeof r)throw new Error("Variation settings must be either a variation name or settings object.");var o=this.fvar.axis.map((function(o,u){var p=o.axisTag.trim();return p in r?Math.max(o.minValue,Math.min(o.maxValue,r[p])):o.defaultValue})),u=new _.DecodeStream(this.stream.buffer);u.pos=this._directoryPos;var p=new TTFFont(u,o);return p._tables=this._tables,p},r.getFont=function getFont(r){return this.getVariation(r)},_createClass(TTFFont,[{key:"postscriptName",get:function get(){return this.getName("postscriptName")}},{key:"fullName",get:function get(){return this.getName("fullName")}},{key:"familyName",get:function get(){return this.getName("fontFamily")}},{key:"subfamilyName",get:function get(){return this.getName("fontSubfamily")}},{key:"copyright",get:function get(){return this.getName("copyright")}},{key:"version",get:function get(){return this.getName("version")}},{key:"ascent",get:function get(){return this.hhea.ascent}},{key:"descent",get:function get(){return this.hhea.descent}},{key:"lineGap",get:function get(){return this.hhea.lineGap}},{key:"underlinePosition",get:function get(){return this.post.underlinePosition}},{key:"underlineThickness",get:function get(){return this.post.underlineThickness}},{key:"italicAngle",get:function get(){return this.post.italicAngle}},{key:"capHeight",get:function get(){var r=this["OS/2"];return r?r.capHeight:this.ascent}},{key:"xHeight",get:function get(){var r=this["OS/2"];return r?r.xHeight:0}},{key:"numGlyphs",get:function get(){return this.maxp.numGlyphs}},{key:"unitsPerEm",get:function get(){return this.head.unitsPerEm}},{key:"bbox",get:function get(){return Object.freeze(new pi(this.head.xMin,this.head.yMin,this.head.xMax,this.head.yMax))}},{key:"_cmapProcessor",get:function get(){return new fi(this.cmap)}},{key:"characterSet",get:function get(){return this._cmapProcessor.getCharacterSet()}},{key:"_layoutEngine",get:function get(){return new Sa(this)}},{key:"availableFeatures",get:function get(){return this._layoutEngine.getAvailableFeatures()}},{key:"variationAxes",get:function get(){var r={};if(!this.fvar)return r;for(var o,u=_createForOfIteratorHelperLoose(this.fvar.axis);!(o=u()).done;){var p=o.value;r[p.axisTag.trim()]={name:p.name.en,min:p.minValue,default:p.defaultValue,max:p.maxValue}}return r}},{key:"namedVariations",get:function get(){var r={};if(!this.fvar)return r;for(var o,u=_createForOfIteratorHelperLoose(this.fvar.instance);!(o=u()).done;){for(var p=o.value,y={},_=0;_<this.fvar.axis.length;_++){y[this.fvar.axis[_].axisTag.trim()]=p.coord[_]}r[p.name.en]=y}return r}},{key:"_variationProcessor",get:function get(){if(!this.fvar)return null;var r=this.variationCoords;return r||this.CFF2?(r||(r=this.fvar.axis.map((function(r){return r.defaultValue}))),new za(this,r)):null}}])}(),_applyDecoratedDescriptor(Ta.prototype,"bbox",[cache],Object.getOwnPropertyDescriptor(Ta.prototype,"bbox"),Ta.prototype),_applyDecoratedDescriptor(Ta.prototype,"_cmapProcessor",[cache],Object.getOwnPropertyDescriptor(Ta.prototype,"_cmapProcessor"),Ta.prototype),_applyDecoratedDescriptor(Ta.prototype,"characterSet",[cache],Object.getOwnPropertyDescriptor(Ta.prototype,"characterSet"),Ta.prototype),_applyDecoratedDescriptor(Ta.prototype,"_layoutEngine",[cache],Object.getOwnPropertyDescriptor(Ta.prototype,"_layoutEngine"),Ta.prototype),_applyDecoratedDescriptor(Ta.prototype,"variationAxes",[cache],Object.getOwnPropertyDescriptor(Ta.prototype,"variationAxes"),Ta.prototype),_applyDecoratedDescriptor(Ta.prototype,"namedVariations",[cache],Object.getOwnPropertyDescriptor(Ta.prototype,"namedVariations"),Ta.prototype),_applyDecoratedDescriptor(Ta.prototype,"_variationProcessor",[cache],Object.getOwnPropertyDescriptor(Ta.prototype,"_variationProcessor"),Ta.prototype),Ta),Ka=new _.Struct({tag:new _.String(4),offset:new _.Pointer(_.uint32,"void",{type:"global"}),compLength:_.uint32,length:_.uint32,origChecksum:_.uint32}),Ja=new _.Struct({tag:new _.String(4),flavor:_.uint32,length:_.uint32,numTables:_.uint16,reserved:new _.Reserved(_.uint16),totalSfntSize:_.uint32,majorVersion:_.uint16,minorVersion:_.uint16,metaOffset:_.uint32,metaLength:_.uint32,metaOrigLength:_.uint32,privOffset:_.uint32,privLength:_.uint32,tables:new _.Array(Ka,"numTables")});Ja.process=function(){for(var r,o={},u=_createForOfIteratorHelperLoose(this.tables);!(r=u()).done;){var p=r.value;o[p.tag]=p}this.tables=o};var Qa=function(r){function WOFFFont(){return r.apply(this,arguments)||this}_inheritsLoose(WOFFFont,r),WOFFFont.probe=function probe(r){return"wOFF"===r.toString("ascii",0,4)};var o=WOFFFont.prototype;return o._decodeDirectory=function _decodeDirectory(){this.directory=Ja.decode(this.stream,{_startOffset:0})},o._getTableStream=function _getTableStream(r){var o=this.directory.tables[r];if(o){if(this.stream.pos=o.offset,o.compLength<o.length){this.stream.pos+=2;var u=y.alloc(o.length),p=R(this.stream.readBuffer(o.compLength-2),u);return new _.DecodeStream(p)}return this.stream}return null},WOFFFont}(Ya),$a=function(r){function WOFF2Glyph(){return r.apply(this,arguments)||this}_inheritsLoose(WOFF2Glyph,r);var o=WOFF2Glyph.prototype;return o._decode=function _decode(){return this._font._transformedGlyphs[this.id]},o._getCBox=function _getCBox(){return this.path.bbox},WOFF2Glyph}(Da),es={decode:function decode(r){for(var o=0,u=[0,1,2,3,4],p=0;p<u.length;p++){var y=r.readUInt8();if(3758096384&o)throw new Error("Overflow");if(o=o<<7|127&y,!(128&y))return o}throw new Error("Bad base 128 number")}},ts=["cmap","head","hhea","hmtx","maxp","name","OS/2","post","cvt ","fpgm","glyf","loca","prep","CFF ","VORG","EBDT","EBLC","gasp","hdmx","kern","LTSH","PCLT","VDMX","vhea","vmtx","BASE","GDEF","GPOS","GSUB","EBSC","JSTF","MATH","CBDT","CBLC","COLR","CPAL","SVG ","sbix","acnt","avar","bdat","bloc","bsln","cvar","fdsc","feat","fmtx","fvar","gvar","hsty","just","lcar","mort","morx","opbd","prop","trak","Zapf","Silf","Glat","Gloc","Feat","Sill"],rs=new _.Struct({flags:_.uint8,customTag:new _.Optional(new _.String(4),(function(r){return!(63&~r.flags)})),tag:function tag(r){return r.customTag||ts[63&r.flags]},length:es,transformVersion:function transformVersion(r){return r.flags>>>6&3},transformed:function transformed(r){return"glyf"===r.tag||"loca"===r.tag?0===r.transformVersion:0!==r.transformVersion},transformLength:new _.Optional(es,(function(r){return r.transformed}))}),ns=new _.Struct({tag:new _.String(4),flavor:_.uint32,length:_.uint32,numTables:_.uint16,reserved:new _.Reserved(_.uint16),totalSfntSize:_.uint32,totalCompressedSize:_.uint32,majorVersion:_.uint16,minorVersion:_.uint16,metaOffset:_.uint32,metaLength:_.uint32,metaOrigLength:_.uint32,privOffset:_.uint32,privLength:_.uint32,tables:new _.Array(rs,"numTables")});ns.process=function(){for(var r={},o=0;o<this.tables.length;o++){var u=this.tables[o];r[u.tag]=u}return this.tables=r};var os=function(r){function WOFF2Font(){return r.apply(this,arguments)||this}_inheritsLoose(WOFF2Font,r),WOFF2Font.probe=function probe(r){return"wOF2"===r.toString("ascii",0,4)};var o=WOFF2Font.prototype;return o._decodeDirectory=function _decodeDirectory(){this.directory=ns.decode(this.stream),this._dataPos=this.stream.pos},o._decompress=function _decompress(){if(!this._decompressed){this.stream.pos=this._dataPos;var r=this.stream.readBuffer(this.directory.totalCompressedSize),o=0;for(var u in this.directory.tables){var p=this.directory.tables[u];p.offset=o,o+=null!=p.transformLength?p.transformLength:p.length}var P=N(r,o);if(!P)throw new Error("Error decoding compressed data in WOFF2");this.stream=new _.DecodeStream(y.from(P)),this._decompressed=!0}},o._decodeTable=function _decodeTable(o){return this._decompress(),r.prototype._decodeTable.call(this,o)},o._getBaseGlyph=function _getBaseGlyph(o,u){if(void 0===u&&(u=[]),!this._glyphs[o])return this.directory.tables.glyf&&this.directory.tables.glyf.transformed?(this._transformedGlyphs||this._transformGlyfTable(),this._glyphs[o]=new $a(o,u,this)):r.prototype._getBaseGlyph.call(this,o,u)},o._transformGlyfTable=function _transformGlyfTable(){this._decompress(),this.stream.pos=this.directory.tables.glyf.offset;for(var r=ss.decode(this.stream),o=[],u=0;u<r.numGlyphs;u++){var p={},y=r.nContours.readInt16BE();if(p.numberOfContours=y,y>0){for(var _=[],P=0,k=0;k<y;k++){P+=read255UInt16(r.nPoints),_.push(P)}p.points=decodeTriplet(r.flags,r.glyphs,P);for(var I=0;I<y;I++)p.points[_[I]-1].endContour=!0;read255UInt16(r.glyphs)}else if(y<0){if(Da.prototype._decodeComposite.call({_font:this},p,r.composites))read255UInt16(r.glyphs)}o.push(p)}this._transformedGlyphs=o},WOFF2Font}(Ya),as=function(){function Substream(r){this.length=r,this._buf=new _.Buffer(r)}return Substream.prototype.decode=function decode(r,o){return new _.DecodeStream(this._buf.decode(r,o))},Substream}(),ss=new _.Struct({version:_.uint32,numGlyphs:_.uint16,indexFormat:_.uint16,nContourStreamSize:_.uint32,nPointsStreamSize:_.uint32,flagStreamSize:_.uint32,glyphStreamSize:_.uint32,compositeStreamSize:_.uint32,bboxStreamSize:_.uint32,instructionStreamSize:_.uint32,nContours:new as("nContourStreamSize"),nPoints:new as("nPointsStreamSize"),flags:new as("flagStreamSize"),glyphs:new as("glyphStreamSize"),composites:new as("compositeStreamSize"),bboxes:new as("bboxStreamSize"),instructions:new as("instructionStreamSize")}),cs=253,ls=254,us=255,fs=253;function read255UInt16(r){var o=r.readUInt8();return o===cs?r.readUInt16BE():o===us?r.readUInt8()+fs:o===ls?r.readUInt8()+2*fs:o}function withSign(r,o){return 1&r?o:-o}function decodeTriplet(r,o,u){for(var p,y=p=0,_=[],P=0;P<u;P++){var k=0,I=0,O=r.readUInt8(),B=!(O>>7);if((O&=127)<10)k=0,I=withSign(O,((14&O)<<7)+o.readUInt8());else if(O<20)k=withSign(O,((O-10&14)<<7)+o.readUInt8()),I=0;else if(O<84){k=withSign(O,1+(48&(D=O-20))+((R=o.readUInt8())>>4)),I=withSign(O>>1,1+((12&D)<<2)+(15&R))}else if(O<120){var D;k=withSign(O,1+((D=O-84)/12<<8)+o.readUInt8()),I=withSign(O>>1,1+(D%12>>2<<8)+o.readUInt8())}else if(O<124){var R=o.readUInt8(),N=o.readUInt8();k=withSign(O,(R<<4)+(N>>4)),I=withSign(O>>1,((15&N)<<8)+o.readUInt8())}else k=withSign(O,o.readUInt16BE()),I=withSign(O>>1,o.readUInt16BE());y+=k,p+=I,_.push(new Fa(B,!1,y,p))}return _}var hs=new _.VersionedStruct(_.uint32,{65536:{numFonts:_.uint32,offsets:new _.Array(_.uint32,"numFonts")},131072:{numFonts:_.uint32,offsets:new _.Array(_.uint32,"numFonts"),dsigTag:_.uint32,dsigLength:_.uint32,dsigOffset:_.uint32}}),ds=function(){function TrueTypeCollection(r){if(this.stream=r,"ttcf"!==r.readString(4))throw new Error("Not a TrueType collection");this.header=hs.decode(r)}return TrueTypeCollection.probe=function probe(r){return"ttcf"===r.toString("ascii",0,4)},TrueTypeCollection.prototype.getFont=function getFont(r){for(var o,u=_createForOfIteratorHelperLoose(this.header.offsets);!(o=u()).done;){var p=o.value,y=new _.DecodeStream(this.stream.buffer);y.pos=p;var P=new Ya(y);if(P.postscriptName===r)return P}return null},_createClass(TrueTypeCollection,[{key:"fonts",get:function get(){for(var r,o=[],u=_createForOfIteratorHelperLoose(this.header.offsets);!(r=u()).done;){var p=r.value,y=new _.DecodeStream(this.stream.buffer);y.pos=p,o.push(new Ya(y))}return o}}])}(),ps=new _.String(_.uint8),gs=(new _.Struct({len:_.uint32,buf:new _.Buffer("len")}),new _.Struct({id:_.uint16,nameOffset:_.int16,attr:_.uint8,dataOffset:_.uint24,handle:_.uint32})),ys=new _.Struct({name:new _.String(4),maxTypeIndex:_.uint16,refList:new _.Pointer(_.uint16,new _.Array(gs,(function(r){return r.maxTypeIndex+1})),{type:"parent"})}),vs=new _.Struct({length:_.uint16,types:new _.Array(ys,(function(r){return r.length+1}))}),ms=new _.Struct({reserved:new _.Reserved(_.uint8,24),typeList:new _.Pointer(_.uint16,vs),nameListOffset:new _.Pointer(_.uint16,"void")}),bs=new _.Struct({dataOffset:_.uint32,map:new _.Pointer(_.uint32,ms),dataLength:_.uint32,mapLength:_.uint32}),ws=function(){function DFont(r){this.stream=r,this.header=bs.decode(this.stream);for(var o,u=_createForOfIteratorHelperLoose(this.header.map.typeList.types);!(o=u()).done;){for(var p,y=o.value,_=_createForOfIteratorHelperLoose(y.refList);!(p=_()).done;){var P=p.value;P.nameOffset>=0?(this.stream.pos=P.nameOffset+this.header.map.nameListOffset,P.name=ps.decode(this.stream)):P.name=null}"sfnt"===y.name&&(this.sfnt=y)}}return DFont.probe=function probe(r){var o=new _.DecodeStream(r);try{var u=bs.decode(o)}catch(r){return!1}for(var p,y=_createForOfIteratorHelperLoose(u.map.typeList.types);!(p=y()).done;){if("sfnt"===p.value.name)return!0}return!1},DFont.prototype.getFont=function getFont(r){if(!this.sfnt)return null;for(var o,u=_createForOfIteratorHelperLoose(this.sfnt.refList);!(o=u()).done;){var p=o.value,y=this.header.dataOffset+p.dataOffset+4,P=new _.DecodeStream(this.stream.buffer.slice(y)),k=new Ya(P);if(k.postscriptName===r)return k}return null},_createClass(DFont,[{key:"fonts",get:function get(){for(var r,o=[],u=_createForOfIteratorHelperLoose(this.sfnt.refList);!(r=u()).done;){var p=r.value,y=this.header.dataOffset+p.dataOffset+4,P=new _.DecodeStream(this.stream.buffer.slice(y));o.push(new Ya(P))}return o}}])}();W.registerFormat(Ya),W.registerFormat(Qa),W.registerFormat(os),W.registerFormat(ds),W.registerFormat(ws),r.exports=W},10819:function(r,o,u){var p,y=u(34984),_=u(10196),P=u(2416),k=u(90682),I=u(520),O=u(12072),B=u(82194),D="prototype",R="script",N=B("IE_PROTO"),EmptyConstructor=function(){},scriptTag=function(r){return"<"+R+">"+r+"</"+R+">"},NullProtoObjectViaActiveX=function(r){r.write(scriptTag("")),r.close();var o=r.parentWindow.Object;return r=null,o},NullProtoObject=function(){try{p=new ActiveXObject("htmlfile")}catch(r){}var r,o,u;NullProtoObject="undefined"!=typeof document?document.domain&&p?NullProtoObjectViaActiveX(p):(o=O("iframe"),u="java"+R+":",o.style.display="none",I.appendChild(o),o.src=String(u),(r=o.contentWindow.document).open(),r.write(scriptTag("document.F=Object")),r.close(),r.F):NullProtoObjectViaActiveX(p);for(var y=P.length;y--;)delete NullProtoObject[D][P[y]];return NullProtoObject()};k[N]=!0,r.exports=Object.create||function create(r,o){var u;return null!==r?(EmptyConstructor[D]=y(r),u=new EmptyConstructor,EmptyConstructor[D]=null,u[N]=r):u=NullProtoObject(),void 0===o?u:_(u,o)}},10821:function(r,o,u){"use strict";var p=u(10884),y="undefined"==typeof globalThis?u.g:globalThis;r.exports=function availableTypedArrays(){for(var r=[],o=0;o<p.length;o++)"function"==typeof y[p[o]]&&(r[r.length]=p[o]);return r}},10849:function(r,o,u){"use strict";var p=u(56475),y=u(32010),_=u(38347),P=u(26882),k=u(16679),I=u(34858),O=u(47044),B=y.RangeError,D=y.String,R=Math.floor,N=_(I),U=_("".slice),W=_(1..toFixed),pow=function(r,o,u){return 0===o?u:o%2==1?pow(r,o-1,u*r):pow(r*r,o/2,u)},multiply=function(r,o,u){for(var p=-1,y=u;++p<6;)y+=o*r[p],r[p]=y%1e7,y=R(y/1e7)},divide=function(r,o){for(var u=6,p=0;--u>=0;)p+=r[u],r[u]=R(p/o),p=p%o*1e7},dataToString=function(r){for(var o=6,u="";--o>=0;)if(""!==u||0===o||0!==r[o]){var p=D(r[o]);u=""===u?p:u+N("0",7-p.length)+p}return u};p({target:"Number",proto:!0,forced:O((function(){return"0.000"!==W(8e-5,3)||"1"!==W(.9,0)||"1.25"!==W(1.255,2)||"1000000000000000128"!==W(0xde0b6b3a7640080,0)}))||!O((function(){W({})}))},{toFixed:function toFixed(r){var o,u,p,y,_=k(this),I=P(r),O=[0,0,0,0,0,0],R="",W="0";if(I<0||I>20)throw B("Incorrect fraction digits");if(_!=_)return"NaN";if(_<=-1e21||_>=1e21)return D(_);if(_<0&&(R="-",_=-_),_>1e-21)if(u=(o=function(r){for(var o=0,u=r;u>=4096;)o+=12,u/=4096;for(;u>=2;)o+=1,u/=2;return o}(_*pow(2,69,1))-69)<0?_*pow(2,-o,1):_/pow(2,o,1),u*=4503599627370496,(o=52-o)>0){for(multiply(O,0,u),p=I;p>=7;)multiply(O,1e7,0),p-=7;for(multiply(O,pow(10,p,1),0),p=o-1;p>=23;)divide(O,1<<23),p-=23;divide(O,1<<p),multiply(O,1,1),divide(O,2),W=dataToString(O)}else multiply(O,0,u),multiply(O,1<<-o,0),W=dataToString(O)+N("0",I);return W=I>0?R+((y=W.length)<=I?"0."+N("0",I-y)+W:U(W,0,y-I)+"."+U(W,y-I)):R+W}})},10884:function(r){"use strict";r.exports=["Float16Array","Float32Array","Float64Array","Int8Array","Int16Array","Int32Array","Uint8Array","Uint8ClampedArray","Uint16Array","Uint32Array","BigInt64Array","BigUint64Array"]},11206:function(r,o,u){var p=u(32010);r.exports=p},11220:function(r,o,u){"use strict";var p=u(91867).isUndefined,y=u(54861);function PageElementWriter(r,o){this.transactionLevel=0,this.repeatables=[],this.tracker=o,this.writer=new y(r,o)}function fitOnPage(r,o){var u=o(r);return u||(r.moveToNextPage(),u=o(r)),u}PageElementWriter.prototype.addLine=function(r,o,u){return fitOnPage(this,(function(p){return p.writer.addLine(r,o,u)}))},PageElementWriter.prototype.addImage=function(r,o){return fitOnPage(this,(function(u){return u.writer.addImage(r,o)}))},PageElementWriter.prototype.addSVG=function(r,o){return fitOnPage(this,(function(u){return u.writer.addSVG(r,o)}))},PageElementWriter.prototype.addQr=function(r,o){return fitOnPage(this,(function(u){return u.writer.addQr(r,o)}))},PageElementWriter.prototype.addVector=function(r,o,u,p,y){return this.writer.addVector(r,o,u,p,y)},PageElementWriter.prototype.beginClip=function(r,o){return this.writer.beginClip(r,o)},PageElementWriter.prototype.endClip=function(){return this.writer.endClip()},PageElementWriter.prototype.alignCanvas=function(r){this.writer.alignCanvas(r)},PageElementWriter.prototype.addFragment=function(r,o,u,p){this.writer.addFragment(r,o,u,p)||(this.moveToNextPage(),this.writer.addFragment(r,o,u,p))},PageElementWriter.prototype.moveToNextPage=function(r){var o=this.writer.context.moveToNextPage(r);this.repeatables.forEach((function(r){p(r.insertedOnPages[this.writer.context.page])?(r.insertedOnPages[this.writer.context.page]=!0,this.writer.addFragment(r,!0)):this.writer.context.moveDown(r.height)}),this),this.writer.tracker.emit("pageChanged",{prevPage:o.prevPage,prevY:o.prevY,y:this.writer.context.y})},PageElementWriter.prototype.beginUnbreakableBlock=function(r,o){0==this.transactionLevel++&&(this.originalX=this.writer.context.x,this.writer.pushContext(r,o))},PageElementWriter.prototype.commitUnbreakableBlock=function(r,o){if(0==--this.transactionLevel){var u=this.writer.context;this.writer.popContext();var p=u.pages.length;if(p>0){var y=u.pages[0];if(y.xOffset=r,y.yOffset=o,p>1)if(void 0!==r||void 0!==o)y.height=u.getCurrentPage().pageSize.height-u.pageMargins.top-u.pageMargins.bottom;else{y.height=this.writer.context.getCurrentPage().pageSize.height-this.writer.context.pageMargins.top-this.writer.context.pageMargins.bottom;for(var _=0,P=this.repeatables.length;_<P;_++)y.height-=this.repeatables[_].height}else y.height=u.y;void 0!==r||void 0!==o?this.writer.addFragment(y,!0,!0,!0):this.addFragment(y)}}},PageElementWriter.prototype.currentBlockToRepeatable=function(){var r=this.writer.context,o={items:[]};return r.pages[0].items.forEach((function(r){o.items.push(r)})),o.xOffset=this.originalX,o.height=r.y,o.insertedOnPages=[],o},PageElementWriter.prototype.pushToRepeatables=function(r){this.repeatables.push(r)},PageElementWriter.prototype.popFromRepeatables=function(){this.repeatables.pop()},PageElementWriter.prototype.context=function(){return this.writer.context},r.exports=PageElementWriter},11326:function(r,o,u){"use strict";var p=u(93143).Buffer;function Utf7Codec(r,o){this.iconv=o}o.utf7=Utf7Codec,o.unicode11utf7="utf7",Utf7Codec.prototype.encoder=Utf7Encoder,Utf7Codec.prototype.decoder=Utf7Decoder,Utf7Codec.prototype.bomAware=!0;var y=/[^A-Za-z0-9'\(\),-\.\/:\? \n\r\t]+/g;function Utf7Encoder(r,o){this.iconv=o.iconv}function Utf7Decoder(r,o){this.iconv=o.iconv,this.inBase64=!1,this.base64Accum=""}Utf7Encoder.prototype.write=function(r){return p.from(r.replace(y,function(r){return"+"+("+"===r?"":this.iconv.encode(r,"utf16-be").toString("base64").replace(/=+$/,""))+"-"}.bind(this)))},Utf7Encoder.prototype.end=function(){};for(var _=/[A-Za-z0-9\/+]/,P=[],k=0;k<256;k++)P[k]=_.test(String.fromCharCode(k));var I="+".charCodeAt(0),O="-".charCodeAt(0),B="&".charCodeAt(0);function Utf7IMAPCodec(r,o){this.iconv=o}function Utf7IMAPEncoder(r,o){this.iconv=o.iconv,this.inBase64=!1,this.base64Accum=p.alloc(6),this.base64AccumIdx=0}function Utf7IMAPDecoder(r,o){this.iconv=o.iconv,this.inBase64=!1,this.base64Accum=""}Utf7Decoder.prototype.write=function(r){for(var o="",u=0,y=this.inBase64,_=this.base64Accum,k=0;k<r.length;k++)if(y){if(!P[r[k]]){if(k==u&&r[k]==O)o+="+";else{var B=_+this.iconv.decode(r.slice(u,k),"ascii");o+=this.iconv.decode(p.from(B,"base64"),"utf16-be")}r[k]!=O&&k--,u=k+1,y=!1,_=""}}else r[k]==I&&(o+=this.iconv.decode(r.slice(u,k),"ascii"),u=k+1,y=!0);if(y){var D=(B=_+this.iconv.decode(r.slice(u),"ascii")).length-B.length%8;_=B.slice(D),B=B.slice(0,D),o+=this.iconv.decode(p.from(B,"base64"),"utf16-be")}else o+=this.iconv.decode(r.slice(u),"ascii");return this.inBase64=y,this.base64Accum=_,o},Utf7Decoder.prototype.end=function(){var r="";return this.inBase64&&this.base64Accum.length>0&&(r=this.iconv.decode(p.from(this.base64Accum,"base64"),"utf16-be")),this.inBase64=!1,this.base64Accum="",r},o.utf7imap=Utf7IMAPCodec,Utf7IMAPCodec.prototype.encoder=Utf7IMAPEncoder,Utf7IMAPCodec.prototype.decoder=Utf7IMAPDecoder,Utf7IMAPCodec.prototype.bomAware=!0,Utf7IMAPEncoder.prototype.write=function(r){for(var o=this.inBase64,u=this.base64Accum,y=this.base64AccumIdx,_=p.alloc(5*r.length+10),P=0,k=0;k<r.length;k++){var I=r.charCodeAt(k);32<=I&&I<=126?(o&&(y>0&&(P+=_.write(u.slice(0,y).toString("base64").replace(/\//g,",").replace(/=+$/,""),P),y=0),_[P++]=O,o=!1),o||(_[P++]=I,I===B&&(_[P++]=O))):(o||(_[P++]=B,o=!0),o&&(u[y++]=I>>8,u[y++]=255&I,y==u.length&&(P+=_.write(u.toString("base64").replace(/\//g,","),P),y=0)))}return this.inBase64=o,this.base64AccumIdx=y,_.slice(0,P)},Utf7IMAPEncoder.prototype.end=function(){var r=p.alloc(10),o=0;return this.inBase64&&(this.base64AccumIdx>0&&(o+=r.write(this.base64Accum.slice(0,this.base64AccumIdx).toString("base64").replace(/\//g,",").replace(/=+$/,""),o),this.base64AccumIdx=0),r[o++]=O,this.inBase64=!1),r.slice(0,o)};var D=P.slice();D[",".charCodeAt(0)]=!0,Utf7IMAPDecoder.prototype.write=function(r){for(var o="",u=0,y=this.inBase64,_=this.base64Accum,P=0;P<r.length;P++)if(y){if(!D[r[P]]){if(P==u&&r[P]==O)o+="&";else{var k=_+this.iconv.decode(r.slice(u,P),"ascii").replace(/,/g,"/");o+=this.iconv.decode(p.from(k,"base64"),"utf16-be")}r[P]!=O&&P--,u=P+1,y=!1,_=""}}else r[P]==B&&(o+=this.iconv.decode(r.slice(u,P),"ascii"),u=P+1,y=!0);if(y){var I=(k=_+this.iconv.decode(r.slice(u),"ascii").replace(/,/g,"/")).length-k.length%8;_=k.slice(I),k=k.slice(0,I),o+=this.iconv.decode(p.from(k,"base64"),"utf16-be")}else o+=this.iconv.decode(r.slice(u),"ascii");return this.inBase64=y,this.base64Accum=_,o},Utf7IMAPDecoder.prototype.end=function(){var r="";return this.inBase64&&this.base64Accum.length>0&&(r=this.iconv.decode(p.from(this.base64Accum,"base64"),"utf16-be")),this.inBase64=!1,this.base64Accum="",r}},11432:function(r,o,u){"use strict";u(41584);var p=u(14598),y=p.Buffer;function copyProps(r,o){for(var u in r)o[u]=r[u]}function SafeBuffer(r,o,u){return y(r,o,u)}y.from&&y.alloc&&y.allocUnsafe&&y.allocUnsafeSlow?r.exports=p:(copyProps(p,o),o.Buffer=SafeBuffer),copyProps(y,SafeBuffer),SafeBuffer.from=function(r,o,u){if("number"==typeof r)throw new TypeError("Argument must not be a number");return y(r,o,u)},SafeBuffer.alloc=function(r,o,u){if("number"!=typeof r)throw new TypeError("Argument must be a number");var p=y(r);return void 0!==o?"string"==typeof u?p.fill(o,u):p.fill(o):p.fill(0),p},SafeBuffer.allocUnsafe=function(r){if("number"!=typeof r)throw new TypeError("Argument must be a number");return y(r)},SafeBuffer.allocUnsafeSlow=function(r){if("number"!=typeof r)throw new TypeError("Argument must be a number");return p.SlowBuffer(r)}},11548:function(r,o,u){"use strict";var p=u(91867).isString,y=u(91867).isNumber,_=u(91867).isObject,P=u(91867).isArray,k=u(91867).isUndefined,I=u(5417),O=/^(\s)+/g,B=/(\s)+$/g;function TextTools(r){this.fontProvider=r}function splitWords(r,o){var u=[];if(r=r.replace(/\t/g,"    "),o)return u.push({text:r}),u;for(var p,y=new I(r),_=0;p=y.nextBreak();){var P=r.slice(_,p.position);p.required||P.match(/\r?\n$|\r$/)?(P=P.replace(/\r?\n$|\r$/,""),u.push({text:P,lineEnd:!0})):u.push({text:P}),_=p.position}return u}function copyStyle(r,o){for(var u in o=o||{},r=r||{})"text"!=u&&r.hasOwnProperty(u)&&(o[u]=r[u]);return o}function normalizeString(r){return null==r?"":y(r)?r.toString():p(r)?r:r.toString()}function getStyleProperty(r,o,u,p){var y;return void 0!==r[u]&&null!==r[u]?r[u]:o?(o.auto(r,(function(){y=o.getProperty(u)})),null!=y?y:p):p}function widthOfString(r,o,u,p,y){return o.widthOfString(r,u,y)+(p||0)*(r.length-1)}TextTools.prototype.buildInlines=function(r,o){var u,p=function measure(r,o,u){var p=function normalizeTextArray(r,o){function flatten(r){return r.reduce((function(r,o){var u=P(o.text)?flatten(o.text):o,p=[].concat(u).some(Array.isArray);return r.concat(p?flatten(u):u)}),[])}function getOneWord(r,o,u){if(k(o[r]))return null;if(o[r].lineEnd)return null;var p=o[r].text;if(u){var y=splitWords(normalizeString(p),!1);if(k(y[y.length-1]))return null;p=y[y.length-1].text}return p}var u=[];P(r)||(r=[r]);r=flatten(r);for(var p=null,y=0,I=r.length;y<I;y++){var O,B=r[y],D=null,R=getStyleProperty(B||{},o,"noWrap",!1);if(_(B)?(B._textRef&&B._textRef._textNodeRef.text&&(B.text=B._textRef._textNodeRef.text),O=splitWords(normalizeString(B.text),R),D=copyStyle(B)):O=splitWords(normalizeString(B),R),p&&O.length)1===splitWords(normalizeString(p+getOneWord(0,O,R)),!1).length&&(u[u.length-1].noNewLine=!0);for(var N=0,U=O.length;N<U;N++){var W={text:O[N].text};O[N].lineEnd&&(W.lineEnd=!0),copyStyle(D,W),u.push(W)}p=null,y+1<I&&(p=getOneWord(O.length-1,O,R))}return u}(o,u);if(p.length){var y=getStyleProperty(p[0],u,"leadingIndent",0);y&&(p[0].leadingCut=-y,p[0].leadingIndent=y)}return p.forEach((function(o){var p=getStyleProperty(o,u,"font","Roboto"),y=getStyleProperty(o,u,"fontSize",12),_=getStyleProperty(o,u,"fontFeatures",null),P=getStyleProperty(o,u,"bold",!1),k=getStyleProperty(o,u,"italics",!1),I=getStyleProperty(o,u,"color","black"),D=getStyleProperty(o,u,"decoration",null),R=getStyleProperty(o,u,"decorationColor",null),N=getStyleProperty(o,u,"decorationStyle",null),U=getStyleProperty(o,u,"background",null),W=getStyleProperty(o,u,"lineHeight",1),j=getStyleProperty(o,u,"characterSpacing",0),G=getStyleProperty(o,u,"link",null),X=getStyleProperty(o,u,"linkToPage",null),Y=getStyleProperty(o,u,"linkToDestination",null),K=getStyleProperty(o,u,"noWrap",null),J=getStyleProperty(o,u,"preserveLeadingSpaces",!1),$=getStyleProperty(o,u,"preserveTrailingSpaces",!1),ee=getStyleProperty(o,u,"opacity",1),te=getStyleProperty(o,u,"sup",!1),re=getStyleProperty(o,u,"sub",!1);(te||re)&&void 0===o.fontSize&&(y*=.58);var ne,ie,oe=r.provideFont(p,P,k);o.width=widthOfString(o.text,oe,y,j,_),o.height=oe.lineHeight(y)*W,o.leadingCut||(o.leadingCut=0),!J&&(ne=o.text.match(O))&&(o.leadingCut+=widthOfString(ne[0],oe,y,j,_)),!$&&(ie=o.text.match(B))?o.trailingCut=widthOfString(ie[0],oe,y,j,_):o.trailingCut=0,o.alignment=getStyleProperty(o,u,"alignment","left"),o.font=oe,o.fontSize=y,o.fontFeatures=_,o.characterSpacing=j,o.color=I,o.decoration=D,o.decorationColor=R,o.decorationStyle=N,o.background=U,o.link=G,o.linkToPage=X,o.linkToDestination=Y,o.noWrap=K,o.opacity=ee,o.sup=te,o.sub=re})),p}(this.fontProvider,r,o),y=0,I=0;return p.forEach((function(r){y=Math.max(y,r.width-r.leadingCut-r.trailingCut),u||(u={width:0,leadingCut:r.leadingCut,trailingCut:0}),u.width+=r.width,u.trailingCut=r.trailingCut,I=Math.max(I,function getTrimmedWidth(r){return Math.max(0,r.width-r.leadingCut-r.trailingCut)}(u)),r.lineEnd&&(u=null)})),getStyleProperty({},o,"noWrap",!1)&&(y=I),{items:p,minWidth:y,maxWidth:I}},TextTools.prototype.sizeOfString=function(r,o){r=r?r.toString().replace(/\t/g,"    "):"";var u=getStyleProperty({},o,"font","Roboto"),p=getStyleProperty({},o,"fontSize",12),y=getStyleProperty({},o,"fontFeatures",null),_=getStyleProperty({},o,"bold",!1),P=getStyleProperty({},o,"italics",!1),k=getStyleProperty({},o,"lineHeight",1),I=getStyleProperty({},o,"characterSpacing",0),O=this.fontProvider.provideFont(u,_,P);return{width:widthOfString(r,O,p,I,y),height:O.lineHeight(p)*k,fontSize:p,lineHeight:k,ascender:O.ascender/1e3*p,descender:O.descender/1e3*p}},TextTools.prototype.sizeOfRotatedText=function(r,o,u){var p=o*Math.PI/-180,y=this.sizeOfString(r,u);return{width:Math.abs(y.height*Math.sin(p))+Math.abs(y.width*Math.cos(p)),height:Math.abs(y.width*Math.sin(p))+Math.abs(y.height*Math.cos(p))}},TextTools.prototype.widthOfString=function(r,o,u,p,y){return widthOfString(r,o,u,p,y)},r.exports=TextTools},11610:function(r,o,u){"use strict";function _setPrototypeOf(r,o){return _setPrototypeOf=Object.setPrototypeOf?Object.setPrototypeOf.bind():function(r,o){return r.__proto__=o,r},_setPrototypeOf(r,o)}u(14032),u(68067);var p=u(68332),y=u(56128).Number,_=u(18128),P=u(7187).inspect,k=function(r){function LazyArrayT(){return r.apply(this,arguments)||this}!function _inheritsLoose(r,o){r.prototype=Object.create(o.prototype),r.prototype.constructor=r,_setPrototypeOf(r,o)}(LazyArrayT,r);var o=LazyArrayT.prototype;return o.decode=function decode(r,o){var u=r.pos,p=_.resolveLength(this.length,r,o);this.length instanceof y&&(o={parent:o,_startOffset:u,_currentOffset:0,_length:p});var P=new I(this.type,p,r,o);return r.pos+=p*this.type.size(null,o),P},o.size=function size(o,u){return o instanceof I&&(o=o.toArray()),r.prototype.size.call(this,o,u)},o.encode=function encode(o,u,p){return u instanceof I&&(u=u.toArray()),r.prototype.encode.call(this,o,u,p)},LazyArrayT}(p),I=function(){function LazyArray(r,o,u,p){this.type=r,this.length=o,this.stream=u,this.ctx=p,this.base=this.stream.pos,this.items=[]}var r=LazyArray.prototype;return r.get=function get(r){if(!(r<0||r>=this.length)){if(null==this.items[r]){var o=this.stream.pos;this.stream.pos=this.base+this.type.size(null,this.ctx)*r,this.items[r]=this.type.decode(this.stream,this.ctx),this.stream.pos=o}return this.items[r]}},r.toArray=function toArray(){for(var r=[],o=0,u=this.length;o<u;o++)r.push(this.get(o));return r},r.inspect=function inspect(){return P(this.toArray())},LazyArray}();r.exports=k},11765:function(r,o,u){var p=u(56475),y=u(95717);p({target:"Array",stat:!0,forced:!u(46769)((function(r){Array.from(r)}))},{from:y})},11813:function(r,o,u){"use strict";u(61726);var p=u(38347),y=u(13711),_=u(49820),P=u(47044),k=u(38688),I=u(48914),O=k("species"),B=RegExp.prototype;r.exports=function(r,o,u,D){var R=k(r),N=!P((function(){var o={};return o[R]=function(){return 7},7!=""[r](o)})),U=N&&!P((function(){var o=!1,u=/a/;return"split"===r&&((u={}).constructor={},u.constructor[O]=function(){return u},u.flags="",u[R]=/./[R]),u.exec=function(){return o=!0,null},u[R](""),!o}));if(!N||!U||u){var W=p(/./[R]),j=o(R,""[r],(function(r,o,u,y,P){var k=p(r),I=o.exec;return I===_||I===B.exec?N&&!P?{done:!0,value:W(o,u,y)}:{done:!0,value:k(u,o,y)}:{done:!1}}));y(String.prototype,r,j[0]),y(B,R,j[1])}D&&I(B[R],"sham",!0)}},11905:function(r,o,u){"use strict";var p,y,_,P,k,I;r.exports=(I=u(48352),u(66664),y=(p=I).lib.WordArray,_=p.algo,P=_.SHA256,k=_.SHA224=P.extend({_doReset:function _doReset(){this._hash=new y.init([3238371032,914150663,812702999,4144912697,4290775857,1750603025,1694076839,3204075428])},_doFinalize:function _doFinalize(){var r=P._doFinalize.call(this);return r.sigBytes-=4,r}}),p.SHA224=P._createHelper(k),p.HmacSHA224=P._createHmacHelper(k),I.SHA224)},12072:function(r,o,u){var p=u(32010),y=u(24517),_=p.document,P=y(_)&&y(_.createElement);r.exports=function(r){return P?_.createElement(r):{}}},12167:function(r,o,u){"use strict";var p=u(83797).F.ERR_STREAM_PREMATURE_CLOSE;function noop(){}r.exports=function eos(r,o,u){if("function"==typeof o)return eos(r,null,o);o||(o={}),u=function once(r){var o=!1;return function(){if(!o){o=!0;for(var u=arguments.length,p=new Array(u),y=0;y<u;y++)p[y]=arguments[y];r.apply(this,p)}}}(u||noop);var y=o.readable||!1!==o.readable&&r.readable,_=o.writable||!1!==o.writable&&r.writable,P=function onlegacyfinish(){r.writable||I()},k=r._writableState&&r._writableState.finished,I=function onfinish(){_=!1,k=!0,y||u.call(r)},O=r._readableState&&r._readableState.endEmitted,B=function onend(){y=!1,O=!0,_||u.call(r)},D=function onerror(o){u.call(r,o)},R=function onclose(){var o;return y&&!O?(r._readableState&&r._readableState.ended||(o=new p),u.call(r,o)):_&&!k?(r._writableState&&r._writableState.ended||(o=new p),u.call(r,o)):void 0},N=function onrequest(){r.req.on("finish",I)};return!function isRequest(r){return r.setHeader&&"function"==typeof r.abort}(r)?_&&!r._writableState&&(r.on("end",P),r.on("close",P)):(r.on("complete",I),r.on("abort",R),r.req?N():r.on("request",N)),r.on("end",B),r.on("finish",I),!1!==o.error&&r.on("error",D),r.on("close",R),function(){r.removeListener("complete",I),r.removeListener("abort",R),r.removeListener("request",N),r.req&&r.req.removeListener("finish",I),r.removeListener("end",P),r.removeListener("close",P),r.removeListener("finish",I),r.removeListener("end",B),r.removeListener("error",D),r.removeListener("close",R)}}},12636:function(r,o,u){var p=u(98086),y=u(74841),_=u(45495),createMethod=function(r){return function(o,u,P){var k,I=p(o),O=_(I),B=y(P,O);if(r&&u!=u){for(;O>B;)if((k=I[B++])!=k)return!0}else for(;O>B;B++)if((r||B in I)&&I[B]===u)return r||B||0;return!r&&-1}};r.exports={includes:createMethod(!0),indexOf:createMethod(!1)}},12719:function(r){"use strict";var o=Object.prototype.toString,u=Math.max,p=function concatty(r,o){for(var u=[],p=0;p<r.length;p+=1)u[p]=r[p];for(var y=0;y<o.length;y+=1)u[y+r.length]=o[y];return u};r.exports=function bind(r){var y=this;if("function"!=typeof y||"[object Function]"!==o.apply(y))throw new TypeError("Function.prototype.bind called on incompatible "+y);for(var _,P=function slicy(r,o){for(var u=[],p=o||0,y=0;p<r.length;p+=1,y+=1)u[y]=r[p];return u}(arguments,1),k=u(0,y.length-P.length),I=[],O=0;O<k;O++)I[O]="$"+O;if(_=Function("binder","return function ("+function(r,o){for(var u="",p=0;p<r.length;p+=1)u+=r[p],p+1<r.length&&(u+=o);return u}(I,",")+"){ return binder.apply(this,arguments); }")((function(){if(this instanceof _){var o=y.apply(this,p(P,arguments));return Object(o)===o?o:this}return y.apply(r,p(P,arguments))})),y.prototype){var B=function Empty(){};B.prototype=y.prototype,_.prototype=new B,B.prototype=null}return _}},12843:function(r){"use strict";r.exports=function hasSymbols(){if("function"!=typeof Symbol||"function"!=typeof Object.getOwnPropertySymbols)return!1;if("symbol"==typeof Symbol.iterator)return!0;var r={},o=Symbol("test"),u=Object(o);if("string"==typeof o)return!1;if("[object Symbol]"!==Object.prototype.toString.call(o))return!1;if("[object Symbol]"!==Object.prototype.toString.call(u))return!1;for(var p in r[o]=42,r)return!1;if("function"==typeof Object.keys&&0!==Object.keys(r).length)return!1;if("function"==typeof Object.getOwnPropertyNames&&0!==Object.getOwnPropertyNames(r).length)return!1;var y=Object.getOwnPropertySymbols(r);if(1!==y.length||y[0]!==o)return!1;if(!Object.prototype.propertyIsEnumerable.call(r,o))return!1;if("function"==typeof Object.getOwnPropertyDescriptor){var _=Object.getOwnPropertyDescriptor(r,o);if(42!==_.value||!0!==_.enumerable)return!1}return!0}},13250:function(r,o,u){"use strict";var p=u(59754),y=u(34815),_=u(47044),P=u(73163),k=p.aTypedArray;(0,p.exportTypedArrayMethod)("slice",(function slice(r,o){for(var u=P(k(this),r,o),p=y(this),_=0,I=u.length,O=new p(I);I>_;)O[_]=u[_++];return O}),_((function(){new Int8Array(1).slice()})))},13501:function(r){r.exports="W5/fcQLn5gKf2XUbAiQ1XULX+TZz6ADToDsgqk6qVfeC0e4m6OO2wcQ1J76ZBVRV1fRkEsdu//62zQsFEZWSTCnMhcsQKlS2qOhuVYYMGCkV0fXWEoMFbESXrKEZ9wdUEsyw9g4bJlEt1Y6oVMxMRTEVbCIwZzJzboK5j8m4YH02qgXYhv1V+PM435sLVxyHJihaJREEhZGqL03txGFQLm76caGO/ovxKvzCby/3vMTtX/459f0igi7WutnKiMQ6wODSoRh/8Lx1V3Q99MvKtwB6bHdERYRY0hStJoMjNeTsNX7bn+Y7e4EQ3bf8xBc7L0BsyfFPK43dGSXpL6clYC/I328h54/VYrQ5i0648FgbGtl837svJ35L3Mot/+nPlNpWgKx1gGXQYqX6n+bbZ7wuyCHKcUok12Xjqub7NXZGzqBx0SD+uziNf87t7ve42jxSKQoW3nyxVrWIGlFShhCKxjpZZ5MeGna0+lBkk+kaN8F9qFBAFgEogyMBdcX/T1W/WnMOi/7ycWUQloEBKGeC48MkiwqJkJO+12eQiOFHMmck6q/IjWW3RZlany23TBm+cNr/84/oi5GGmGBZWrZ6j+zykVozz5fT/QH/Da6WTbZYYPynVNO7kxzuNN2kxKKWche5WveitPKAecB8YcAHz/+zXLjcLzkdDSktNIDwZE9J9X+tto43oJy65wApM3mDzYtCwX9lM+N5VR3kXYo0Z3t0TtXfgBFg7gU8oN0Dgl7fZlUbhNll+0uuohRVKjrEd8egrSndy5/Tgd2gqjA4CAVuC7ESUmL3DZoGnfhQV8uwnpi8EGvAVVsowNRxPudck7+oqAUDkwZopWqFnW1riss0t1z6iCISVKreYGNvQcXv+1L9+jbP8cd/dPUiqBso2q+7ZyFBvENCkkVr44iyPbtOoOoCecWsiuqMSML5lv+vN5MzUr+Dnh73G7Q1YnRYJVYXHRJaNAOByiaK6CusgFdBPE40r0rvqXV7tksKO2DrHYXBTv8P5ysqxEx8VDXUDDqkPH6NNOV/a2WH8zlkXRELSa8P+heNyJBBP7PgsG1EtWtNef6/i+lcayzQwQCsduidpbKfhWUDgAEmyhGu/zVTacI6RS0zTABrOYueemnVa19u9fT23N/Ta6RvTpof5DWygqreCqrDAgM4LID1+1T/taU6yTFVLqXOv+/MuQOFnaF8vLMKD7tKWDoBdALgxF33zQccCcdHx8fKIVdW69O7qHtXpeGr9jbbpFA+qRMWr5hp0s67FPc7HAiLV0g0/peZlW7hJPYEhZyhpSwahnf93/tZgfqZWXFdmdXBzqxGHLrQKxoAY6fRoBhgCRPmmGueYZ5JexTVDKUIXzkG/fqp/0U3hAgQdJ9zumutK6nqWbaqvm1pgu03IYR+G+8s0jDBBz8cApZFSBeuWasyqo2OMDKAZCozS+GWSvL/HsE9rHxooe17U3s/lTE+VZAk4j3dp6uIGaC0JMiqR5CUsabPyM0dOYDR7Ea7ip4USZlya38YfPtvrX/tBlhHilj55nZ1nfN24AOAi9BVtz/Mbn8AEDJCqJgsVUa6nQnSxv2Fs7l/NlCzpfYEjmPrNyib/+t0ei2eEMjvNhLkHCZlci4WhBe7ePZTmzYqlY9+1pxtS4GB+5lM1BHT9tS270EWUDYFq1I0yY/fNiAk4bk9yBgmef/f2k6AlYQZHsNFnW8wBQxCd68iWv7/35bXfz3JZmfGligWAKRjIs3IpzxQ27vAglHSiOzCYzJ9L9A1CdiyFvyR66ucA4jKifu5ehwER26yV7HjKqn5Mfozo7Coxxt8LWWPT47BeMxX8p0Pjb7hZn+6bw7z3Lw+7653j5sI8CLu5kThpMlj1m4c2ch3jGcP1FsT13vuK3qjecKTZk2kHcOZY40UX+qdaxstZqsqQqgXz+QGF99ZJLqr3VYu4aecl1Ab5GmqS8k/GV5b95zxQ5d4EfXUJ6kTS/CXF/aiqKDOT1T7Jz5z0PwDUcwr9clLN1OJGCiKfqvah+h3XzrBOiLOW8wvn8gW6qE8vPxi+Efv+UH55T7PQFVMh6cZ1pZQlzJpKZ7P7uWvwPGJ6DTlR6wbyj3Iv2HyefnRo/dv7dNx+qaa0N38iBsR++Uil7Wd4afwDNsrzDAK4fXZwvEY/jdKuIKXlfrQd2C39dW7ntnRbIp9OtGy9pPBn/V2ASoi/2UJZfS+xuGLH8bnLuPlzdTNS6zdyk8Dt/h6sfOW5myxh1f+zf3zZ3MX/mO9cQPp5pOx967ZA6/pqHvclNfnUFF+rq+Vd7alKr6KWPcIDhpn6v2K6NlUu6LrKo8b/pYpU/Gazfvtwhn7tEOUuXht5rUJdSf6sLjYf0VTYDgwJ81yaqKTUYej/tbHckSRb/HZicwGJqh1mAHB/IuNs9dc9yuvF3D5Xocm3elWFdq5oEy70dYFit79yaLiNjPj5UUcVmZUVhQEhW5V2Z6Cm4HVH/R8qlamRYwBileuh07CbEce3TXa2JmXWBf+ozt319psboobeZhVnwhMZzOeQJzhpTDbP71Tv8HuZxxUI/+ma3XW6DFDDs4+qmpERwHGBd2edxwUKlODRdUWZ/g0GOezrbzOZauFMai4QU6GVHV6aPNBiBndHSsV4IzpvUiiYyg6OyyrL4Dj5q/Lw3N5kAwftEVl9rNd7Jk5PDij2hTH6wIXnsyXkKePxbmHYgC8A6an5Fob/KH5GtC0l4eFso+VpxedtJHdHpNm+Bvy4C79yVOkrZsLrQ3OHCeB0Ra+kBIRldUGlDCEmq2RwXnfyh6Dz+alk6eftI2n6sastRrGwbwszBeDRS/Fa/KwRJkCzTsLr/JCs5hOPE/MPLYdZ1F1fv7D+VmysX6NpOC8aU9F4Qs6HvDyUy9PvFGDKZ/P5101TYHFl8pjj6wm/qyS75etZhhfg0UEL4OYmHk6m6dO192AzoIyPSV9QedDA4Ml23rRbqxMPMxf7FJnDc5FTElVS/PyqgePzmwVZ26NWhRDQ+oaT7ly7ell4s3DypS1s0g+tOr7XHrrkZj9+x/mJBttrLx98lFIaRZzHz4aC7r52/JQ4VjHahY2/YVXZn/QC2ztQb/sY3uRlyc5vQS8nLPGT/n27495i8HPA152z7Fh5aFpyn1GPJKHuPL8Iw94DuW3KjkURAWZXn4EQy89xiKEHN1mk/tkM4gYDBxwNoYvRfE6LFqsxWJtPrDGbsnLMap3Ka3MUoytW0cvieozOmdERmhcqzG+3HmZv2yZeiIeQTKGdRT4HHNxekm1tY+/n06rGmFleqLscSERzctTKM6G9P0Pc1RmVvrascIxaO1CQCiYPE15bD7c3xSeW7gXxYjgxcrUlcbIvO0r+Yplhx0kTt3qafDOmFyMjgGxXu73rddMHpV1wMubyAGcf/v5dLr5P72Ta9lBF+fzMJrMycwv+9vnU3ANIl1cH9tfW7af8u0/HG0vV47jNFXzFTtaha1xvze/s8KMtCYucXc1nzfd/MQydUXn/b72RBt5wO/3jRcMH9BdhC/yctKBIveRYPrNpDWqBsO8VMmP+WvRaOcA4zRMR1PvSoO92rS7pYEv+fZfEfTMzEdM+6X5tLlyxExhqLRkms5EuLovLfx66de5fL2/yX02H52FPVwahrPqmN/E0oVXnsCKhbi/yRxX83nRbUKWhzYceXOntfuXn51NszJ6MO73pQf5Pl4in3ec4JU8hF7ppV34+mm9r1LY0ee/i1O1wpd8+zfLztE0cqBxggiBi5Bu95v9l3r9r/U5hweLn+TbfxowrWDqdJauKd8+q/dH8sbPkc9ttuyO94f7/XK/nHX46MPFLEb5qQlNPvhJ50/59t9ft3LXu7uVaWaO2bDrDCnRSzZyWvFKxO1+vT8MwwunR3bX0CkfPjqb4K9O19tn5X50PvmYpEwHtiW9WtzuV/s76B1zvLLNkViNd8ySxIl/3orfqP90TyTGaf7/rx8jQzeHJXdmh/N6YDvbvmTBwCdxfEQ1NcL6wNMdSIXNq7b1EUzRy1/Axsyk5p22GMG1b+GxFgbHErZh92wuvco0AuOLXct9hvw2nw/LqIcDRRmJmmZzcgUa7JpM/WV/S9IUfbF56TL2orzqwebdRD8nIYNJ41D/hz37Fo11p2Y21wzPcn713qVGhqtevStYfGH4n69OEJtPvbbLYWvscDqc3Hgnu166+tAyLnxrX0Y5zoYjV++1sI7t5kMr02KT/+uwtkc+rZLOf/qn/s3nYCf13Dg8/sB2diJgjGqjQ+TLhxbzyue2Ob7X6/9lUwW7a+lbznHzOYy8LKW1C/uRPbQY3KW/0gO9LXunHLvPL97afba9bFtc9hmz7GAttjVYlCvQAiOwAk/gC5+hkLEs6tr3AZKxLJtOEwk2dLxTYWsIB/j/ToWtIWzo906FrSG8iaqqqqqqiIiIiAgzMzMzNz+AyK+01/zi8n8S+Y1MjoRaQ80WU/G8MBlO+53VPXANrWm4wzGUVZUjjBJZVdhpcfkjsmcWaO+UEldXi1e+zq+HOsCpknYshuh8pOLISJun7TN0EIGW2xTnlOImeecnoGW4raxe2G1T3HEvfYUYMhG+gAFOAwh5nK8mZhwJMmN7r224QVsNFvZ87Z0qatvknklyPDK3Hy45PgVKXji52Wen4d4PlFVVYGnNap+fSpFbK90rYnhUc6n91Q3AY9E0tJOFrcfZtm/491XbcG/jsViUPPX76qmeuiz+qY1Hk7/1VPM405zWVuoheLUimpWYdVzCmUdKHebMdzgrYrb8mL2eeLSnRWHdonfZa8RsOU9F37w+591l5FLYHiOqWeHtE/lWrBHcRKp3uhtr8yXm8LU/5ms+NM6ZKsqu90cFZ4o58+k4rdrtB97NADFbwmEG7lXqvirhOTOqU14xuUF2myIjURcPHrPOQ4lmM3PeMg7bUuk0nnZi67bXsU6H8lhqIo8TaOrEafCO1ARK9PjC0QOoq2BxmMdgYB9G/lIb9++fqNJ2s7BHGFyBNmZAR8J3KCo012ikaSP8BCrf6VI0X5xdnbhHIO+B5rbOyB54zXkzfObyJ4ecwxfqBJMLFc7m59rNcw7hoHnFZ0b00zee+gTqvjm61Pb4xn0kcDX4jvHM0rBXZypG3DCKnD/Waa/ZtHmtFPgO5eETx+k7RrVg3aSwm2YoNXnCs3XPQDhNn+Fia6IlOOuIG6VJH7TP6ava26ehKHQa2T4N0tcZ9dPCGo3ZdnNltsHQbeYt5vPnJezV/cAeNypdml1vCHI8M81nSRP5Qi2+mI8v/sxiZru9187nRtp3f/42NemcONa+4eVC3PCZzc88aZh851CqSsshe70uPxeN/dmYwlwb3trwMrN1Gq8jbnApcVDx/yDPeYs5/7r62tsQ6lLg+DiFXTEhzR9dHqv0iT4tgj825W+H3XiRUNUZT2kR9Ri0+lp+UM3iQtS8uOE23Ly4KYtvqH13jghUntJRAewuzNLDXp8RxdcaA3cMY6TO2IeSFRXezeWIjCqyhsUdMYuCgYTZSKpBype1zRfq8FshvfBPc6BAQWl7/QxIDp3VGo1J3vn42OEs3qznws+YLRXbymyB19a9XBx6n/owcyxlEYyFWCi+kG9F+EyD/4yn80+agaZ9P7ay2Dny99aK2o91FkfEOY8hBwyfi5uwx2y5SaHmG+oq/zl1FX/8irOf8Y3vAcX/6uLP6A6nvMO24edSGPjQc827Rw2atX+z2bKq0CmW9mOtYnr5/AfDa1ZfPaXnKtlWborup7QYx+Or2uWb+N3N//2+yDcXMqIJdf55xl7/vsj4WoPPlxLxtVrkJ4w/tTe3mLdATOOYwxcq52w5Wxz5MbPdVs5O8/lhfE7dPj0bIiPQ3QV0iqm4m3YX8hRfc6jQ3fWepevMqUDJd86Z4vwM40CWHnn+WphsGHfieF02D3tmZvpWD+kBpNCFcLnZhcmmrhpGzzbdA+sQ1ar18OJD87IOKOFoRNznaHPNHUfUNhvY1iU+uhvEvpKHaUn3qK3exVVyX4joipp3um7FmYJWmA+WbIDshRpbVRx5/nqstCgy87FGbfVB8yDGCqS+2qCsnRwnSAN6zgzxfdB2nBT/vZ4/6uxb6oH8b4VBRxiIB93wLa47hG3w2SL/2Z27yOXJFwZpSJaBYyvajA7vRRYNKqljXKpt/CFD/tSMr18DKKbwB0xggBePatl1nki0yvqW5zchlyZmJ0OTxJ3D+fsYJs/mxYN5+Le5oagtcl+YsVvy8kSjI2YGvGjvmpkRS9W2dtXqWnVuxUhURm1lKtou/hdEq19VBp9OjGvHEQSmrpuf2R24mXGheil8KeiANY8fW1VERUfBImb64j12caBZmRViZHbeVMjCrPDg9A90IXrtnsYCuZtRQ0PyrKDjBNOsPfKsg1pA02gHlVr0OXiFhtp6nJqXVzcbfM0KnzC3ggOENPE9VBdmHKN6LYaijb4wXxJn5A0FSDF5j+h1ooZx885Jt3ZKzO5n7Z5WfNEOtyyPqQEnn7WLv5Fis3PdgMshjF1FRydbNyeBbyKI1oN1TRVrVK7kgsb/zjX4NDPIRMctVeaxVB38Vh1x5KbeJbU138AM5KzmZu3uny0ErygxiJF7GVXUrPzFxrlx1uFdAaZFDN9cvIb74qD9tzBMo7L7WIEYK+sla1DVMHpF0F7b3+Y6S+zjvLeDMCpapmJo1weBWuxKF3rOocih1gun4BoJh1kWnV/Jmiq6uOhK3VfKxEHEkafjLgK3oujaPzY6SXg8phhL4TNR1xvJd1Wa0aYFfPUMLrNBDCh4AuGRTbtKMc6Z1Udj8evY/ZpCuMAUefdo69DZUngoqE1P9A3PJfOf7WixCEj+Y6t7fYeHbbxUAoFV3M89cCKfma3fc1+jKRe7MFWEbQqEfyzO2x/wrO2VYH7iYdQ9BkPyI8/3kXBpLaCpU7eC0Yv/am/tEDu7HZpqg0EvHo0nf/R/gRzUWy33/HXMJQeu1GylKmOkXzlCfGFruAcPPhaGqZOtu19zsJ1SO2Jz4Ztth5cBX6mRQwWmDwryG9FUMlZzNckMdK+IoMJv1rOWnBamS2w2KHiaPMPLC15hCZm4KTpoZyj4E2TqC/P6r7/EhnDMhKicZZ1ZwxuC7DPzDGs53q8gXaI9kFTK+2LTq7bhwsTbrMV8Rsfua5lMS0FwbTitUVnVa1yTb5IX51mmYnUcP9wPr8Ji1tiYJeJV9GZTrQhF7vvdU2OTU42ogJ9FDwhmycI2LIg++03C6scYhUyUuMV5tkw6kGUoL+mjNC38+wMdWNljn6tGPpRES7veqrSn5TRuv+dh6JVL/iDHU1db4c9WK3++OrH3PqziF916UMUKn8G67nN60GfWiHrXYhUG3yVWmyYak59NHj8t1smG4UDiWz2rPHNrKnN4Zo1LBbr2/eF9YZ0n0blx2nG4X+EKFxvS3W28JESD+FWk61VCD3z/URGHiJl++7TdBwkCj6tGOH3qDb0QqcOF9Kzpj0HUb/KyFW3Yhj2VMKJqGZleFBH7vqvf7WqLC3XMuHV8q8a4sTFuxUtkD/6JIBvKaVjv96ndgruKZ1k/BHzqf2K9fLk7HGXANyLDd1vxkK/i055pnzl+zw6zLnwXlVYVtfmacJgEpRP1hbGgrYPVN6v2lG+idQNGmwcKXu/8xEj/P6qe/sB2WmwNp6pp8jaISMkwdleFXYK55NHWLTTbutSUqjBfDGWo/Yg918qQ+8BRZSAHZbfuNZz2O0sov1Ue4CWlVg3rFhM3Kljj9ksGd/NUhk4nH+a5UN2+1i8+NM3vRNp7uQ6sqexSCukEVlVZriHNqFi5rLm9TMWa4qm3idJqppQACol2l4VSuvWLfta4JcXy3bROPNbXOgdOhG47LC0CwW/dMlSx4Jf17aEU3yA1x9p+Yc0jupXgcMuYNku64iYOkGToVDuJvlbEKlJqsmiHbvNrIVZEH+yFdF8DbleZ6iNiWwMqvtMp/mSpwx5KxRrT9p3MAPTHGtMbfvdFhyj9vhaKcn3At8Lc16Ai+vBcSp1ztXi7rCJZx/ql7TXcclq6Q76UeKWDy9boS0WHIjUuWhPG8LBmW5y2rhuTpM5vsLt+HOLh1Yf0DqXa9tsfC+kaKt2htA0ai/L2i7RKoNjEwztkmRU0GfgW1TxUvPFhg0V7DdfWJk5gfrccpYv+MA9M0dkGTLECeYwUixRzjRFdmjG7zdZIl3XKB9YliNKI31lfa7i2JG5C8Ss+rHe0D7Z696/V3DEAOWHnQ9yNahMUl5kENWS6pHKKp2D1BaSrrHdE1w2qNxIztpXgUIrF0bm15YML4b6V1k+GpNysTahKMVrrS85lTVo9OGJ96I47eAy5rYWpRf/mIzeoYU1DKaQCTUVwrhHeyNoDqHel+lLxr9WKzhSYw7vrR6+V5q0pfi2k3L1zqkubY6rrd9ZLvSuWNf0uqnkY+FpTvFzSW9Fp0b9l8JA7THV9eCi/PY/SCZIUYx3BU2alj7Cm3VV6eYpios4b6WuNOJdYXUK3zTqj5CVG2FqYM4Z7CuIU0qO05XR0d71FHM0YhZmJmTRfLlXEumN82BGtzdX0S19t1e+bUieK8zRmqpa4Qc5TSjifmaQsY2ETLjhI36gMR1+7qpjdXXHiceUekfBaucHShAOiFXmv3sNmGQyU5iVgnoocuonQXEPTFwslHtS8R+A47StI9wj0iSrtbi5rMysczFiImsQ+bdFClnFjjpXXwMy6O7qfjOr8Fb0a7ODItisjnn3EQO16+ypd1cwyaAW5Yzxz5QknfMO7643fXW/I9y3U2xH27Oapqr56Z/tEzglj6IbT6HEHjopiXqeRbe5mQQvxtcbDOVverN0ZgMdzqRYRjaXtMRd56Q4cZSmdPvZJdSrhJ1D9zNXPqAEqPIavPdfubt5oke2kmv0dztIszSv2VYuoyf1UuopbsYb+uX9h6WpwjpgtZ6fNNawNJ4q8O3CFoSbioAaOSZMx2GYaPYB+rEb6qjQiNRFQ76TvwNFVKD+BhH9VhcKGsXzmMI7BptU/CNWolM7YzROvpFAntsiWJp6eR2d3GarcYShVYSUqhmYOWj5E96NK2WvmYNTeY7Zs4RUEdv9h9QT4EseKt6LzLrqEOs3hxAY1MaNWpSa6zZx8F3YOVeCYMS88W+CYHDuWe4yoc6YK+djDuEOrBR5lvh0r+Q9uM88lrjx9x9AtgpQVNE8r+3O6Gvw59D+kBF/UMXyhliYUtPjmvXGY6Dk3x+kEOW+GtdMVC4EZTqoS/jmR0P0LS75DOc/w2vnri97M4SdbZ8qeU7gg8DVbERkU5geaMQO3mYrSYyAngeUQqrN0C0/vsFmcgWNXNeidsTAj7/4MncJR0caaBUpbLK1yBCBNRjEv6KvuVSdpPnEMJdsRRtqJ+U8tN1gXA4ePHc6ZT0eviI73UOJF0fEZ8YaneAQqQdGphNvwM4nIqPnXxV0xA0fnCT+oAhJuyw/q8jO0y8CjSteZExwBpIN6SvNp6A5G/abi6egeND/1GTguhuNjaUbbnSbGd4L8937Ezm34Eyi6n1maeOBxh3PI0jzJDf5mh/BsLD7F2GOKvlA/5gtvxI3/eV4sLfKW5Wy+oio+es/u6T8UU+nsofy57Icb/JlZHPFtCgd/x+bwt3ZT+xXTtTtTrGAb4QehC6X9G+8YT+ozcLxDsdCjsuOqwPFnrdLYaFc92Ui0m4fr39lYmlCaqTit7G6O/3kWDkgtXjNH4BiEm/+jegQnihOtfffn33WxsFjhfMd48HT+f6o6X65j7XR8WLSHMFkxbvOYsrRsF1bowDuSQ18Mkxk4qz2zoGPL5fu9h2Hqmt1asl3Q3Yu3szOc+spiCmX4AETBM3pLoTYSp3sVxahyhL8eC4mPN9k2x3o0xkiixIzM3CZFzf5oR4mecQ5+ax2wCah3/crmnHoqR0+KMaOPxRif1oEFRFOO/kTPPmtww+NfMXxEK6gn6iU32U6fFruIz8Q4WgljtnaCVTBgWx7diUdshC9ZEa5yKpRBBeW12r/iNc/+EgNqmhswNB8SBoihHXeDF7rrWDLcmt3V8GYYN7pXRy4DZjj4DJuUBL5iC3DQAaoo4vkftqVTYRGLS3mHZ7gdmdTTqbgNN/PTdTCOTgXolc88MhXAEUMdX0iy1JMuk5wLsgeu0QUYlz2S4skTWwJz6pOm/8ihrmgGfFgri+ZWUK2gAPHgbWa8jaocdSuM4FJYoKicYX/ZSENkg9Q1ZzJfwScfVnR2DegOGwCvmogaWJCLQepv9WNlU6QgsmOwICquU28Mlk3d9W5E81lU/5Ez0LcX6lwKMWDNluNKfBDUy/phJgBcMnfkh9iRxrdOzgs08JdPB85Lwo+GUSb4t3nC+0byqMZtO2fQJ4U2zGIr49t/28qmmGv2RanDD7a3FEcdtutkW8twwwlUSpb8QalodddbBfNHKDQ828BdE7OBgFdiKYohLawFYqpybQoxATZrheLhdI7+0Zlu9Q1myRcd15r9UIm8K2LGJxqTegntqNVMKnf1a8zQiyUR1rxoqjiFxeHxqFcYUTHfDu7rhbWng6qOxOsI+5A1p9mRyEPdVkTlE24vY54W7bWc6jMgZvNXdfC9/9q7408KDsbdL7Utz7QFSDetz2picArzrdpL8OaCHC9V26RroemtDZ5yNM/KGkWMyTmfnInEvwtSD23UcFcjhaE3VKzkoaEMKGBft4XbIO6forTY1lmGQwVmKicBCiArDzE+1oIxE08fWeviIOD5TznqH+OoHadvoOP20drMPe5Irg3XBQziW2XDuHYzjqQQ4wySssjXUs5H+t3FWYMHppUnBHMx/nYIT5d7OmjDbgD9F6na3m4l7KdkeSO3kTEPXafiWinogag7b52taiZhL1TSvBFmEZafFq2H8khQaZXuitCewT5FBgVtPK0j4xUHPfUz3Q28eac1Z139DAP23dgki94EC8vbDPTQC97HPPSWjUNG5tWKMsaxAEMKC0665Xvo1Ntd07wCLNf8Q56mrEPVpCxlIMVlQlWRxM3oAfpgIc+8KC3rEXUog5g06vt7zgXY8grH7hhwVSaeuvC06YYRAwpbyk/Unzj9hLEZNs2oxPQB9yc+GnL6zTgq7rI++KDJwX2SP8Sd6YzTuw5lV/kU6eQxRD12omfQAW6caTR4LikYkBB1CMOrvgRr/VY75+NSB40Cni6bADAtaK+vyxVWpf9NeKJxN2KYQ8Q2xPB3K1s7fuhvWbr2XpgW044VD6DRs0qXoqKf1NFsaGvKJc47leUV3pppP/5VTKFhaGuol4Esfjf5zyCyUHmHthChcYh4hYLQF+AFWsuq4t0wJyWgdwQVOZiV0efRHPoK5+E1vjz9wTJmVkITC9oEstAsyZSgE/dbicwKr89YUxKZI+owD205Tm5lnnmDRuP/JnzxX3gMtlrcX0UesZdxyQqYQuEW4R51vmQ5xOZteUd8SJruMlTUzhtVw/Nq7eUBcqN2/HVotgfngif60yKEtoUx3WYOZlVJuJOh8u59fzSDPFYtQgqDUAGyGhQOAvKroXMcOYY0qjnStJR/G3aP+Jt1sLVlGV8POwr/6OGsqetnyF3TmTqZjENfnXh51oxe9qVUw2M78EzAJ+IM8lZ1MBPQ9ZWSVc4J3mWSrLKrMHReA5qdGoz0ODRsaA+vwxXA2cAM4qlfzBJA6581m4hzxItQw5dxrrBL3Y6kCbUcFxo1S8jyV44q//+7ASNNudZ6xeaNOSIUffqMn4A9lIjFctYn2gpEPAb3f7p3iIBN8H14FUGQ9ct2hPsL+cEsTgUrR47uJVN4n4wt/wgfwwHuOnLd4yobkofy8JvxSQTA7rMpDIc608SlZFJfZYcmbT0tAHpPE8MrtQ42siTUNWxqvWZOmvu9f0JPoQmg+6l7sZWwyfi6PXkxJnwBraUG0MYG4zYHQz3igy/XsFkx5tNQxw43qvI9dU3f0DdhOUlHKjmi1VAr2Kiy0HZwD8VeEbhh0OiDdMYspolQsYdSwjCcjeowIXNZVUPmL2wwIkYhmXKhGozdCJ4lRKbsf4NBh/XnQoS92NJEWOVOFs2YhN8c5QZFeK0pRdAG40hqvLbmoSA8xQmzOOEc7wLcme9JOsjPCEgpCwUs9E2DohMHRhUeyGIN6TFvrbny8nDuilsDpzrH5mS76APoIEJmItS67sQJ+nfwddzmjPxcBEBBCw0kWDwd0EZCkNeOD7NNQhtBm7KHL9mRxj6U1yWU2puzlIDtpYxdH4ZPeXBJkTGAJfUr/oTCz/iypY6uXaR2V1doPxJYlrw2ghH0D5gbrhFcIxzYwi4a/4hqVdf2DdxBp6vGYDjavxMAAoy+1+3aiO6S3W/QAKNVXagDtvsNtx7Ks+HKgo6U21B+QSZgIogV5Bt+BnXisdVfy9VyXV+2P5fMuvdpAjM1o/K9Z+XnE4EOCrue+kcdYHqAQ0/Y/OmNlQ6OI33jH/uD1RalPaHpJAm2av0/xtpqdXVKNDrc9F2izo23Wu7firgbURFDNX9eGGeYBhiypyXZft2j3hTvzE6PMWKsod//rEILDkzBXfi7xh0eFkfb3/1zzPK/PI5Nk3FbZyTl4mq5BfBoVoqiPHO4Q4QKZAlrQ3MdNfi3oxIjvsM3kAFv3fdufurqYR3PSwX/mpGy/GFI/B2MNPiNdOppWVbs/gjF3YH+QA9jMhlAbhvasAHstB0IJew09iAkmXHl1/TEj+jvHOpOGrPRQXbPADM+Ig2/OEcUcpgPTItMtW4DdqgfYVI/+4hAFWYjUGpOP/UwNuB7+BbKOcALbjobdgzeBQfjgNSp2GOpxzGLj70Vvq5cw2AoYENwKLUtJUX8sGRox4dVa/TN4xKwaKcl9XawQR/uNus700Hf17pyNnezrUgaY9e4MADhEDBpsJT6y1gDJs1q6wlwGhuUzGR7C8kgpjPyHWwsvrf3yn1zJEIRa5eSxoLAZOCR9xbuztxFRJW9ZmMYfCFJ0evm9F2fVnuje92Rc4Pl6A8bluN8MZyyJGZ0+sNSb//DvAFxC2BqlEsFwccWeAl6CyBcQV1bx4mQMBP1Jxqk1EUADNLeieS2dUFbQ/c/kvwItbZ7tx0st16viqd53WsRmPTKv2AD8CUnhtPWg5aUegNpsYgasaw2+EVooeNKmrW3MFtj76bYHJm5K9gpAXZXsE5U8DM8XmVOSJ1F1WnLy6nQup+jx52bAb+rCq6y9WXl2B2oZDhfDkW7H3oYfT/4xx5VncBuxMXP2lNfhUVQjSSzSRbuZFE4vFawlzveXxaYKVs8LpvAb8IRYF3ZHiRnm0ADeNPWocwxSzNseG7NrSEVZoHdKWqaGEBz1N8Pt7kFbqh3LYmAbm9i1IChIpLpM5AS6mr6OAPHMwwznVy61YpBYX8xZDN/a+lt7n+x5j4bNOVteZ8lj3hpAHSx1VR8vZHec4AHO9XFCdjZ9eRkSV65ljMmZVzaej2qFn/qt1lvWzNZEfHxK3qOJrHL6crr0CRzMox5f2e8ALBB4UGFZKA3tN6F6IXd32GTJXGQ7DTi9j/dNcLF9jCbDcWGKxoKTYblIwbLDReL00LRcDPMcQuXLMh5YzgtfjkFK1DP1iDzzYYVZz5M/kWYRlRpig1htVRjVCknm+h1M5LiEDXOyHREhvzCGpFZjHS0RsK27o2avgdilrJkalWqPW3D9gmwV37HKmfM3F8YZj2ar+vHFvf3B8CRoH4kDHIK9mrAg+owiEwNjjd9V+FsQKYR8czJrUkf7Qoi2YaW6EVDZp5zYlqiYtuXOTHk4fAcZ7qBbdLDiJq0WNV1l2+Hntk1mMWvxrYmc8kIx8G3rW36J6Ra4lLrTOCgiOihmow+YnzUT19jbV2B3RWqSHyxkhmgsBqMYWvOcUom1jDQ436+fcbu3xf2bbeqU/ca+C4DOKE+e3qvmeMqW3AxejfzBRFVcwVYPq4L0APSWWoJu+5UYX4qg5U6YTioqQGPG9XrnuZ/BkxuYpe6Li87+18EskyQW/uA+uk2rpHpr6hut2TlVbKgWkFpx+AZffweiw2+VittkEyf/ifinS/0ItRL2Jq3tQOcxPaWO2xrG68GdFoUpZgFXaP2wYVtRc6xYCfI1CaBqyWpg4bx8OHBQwsV4XWMibZZ0LYjWEy2IxQ1mZrf1/UNbYCJplWu3nZ4WpodIGVA05d+RWSS+ET9tH3RfGGmNI1cIY7evZZq7o+a0bjjygpmR3mVfalkT/SZGT27Q8QGalwGlDOS9VHCyFAIL0a1Q7JiW3saz9gqY8lqKynFrPCzxkU4SIfLc9VfCI5edgRhDXs0edO992nhTKHriREP1NJC6SROMgQ0xO5kNNZOhMOIT99AUElbxqeZF8A3xrfDJsWtDnUenAHdYWSwAbYjFqQZ+D5gi3hNK8CSxU9i6f6ClL9IGlj1OPMQAsr84YG6ijsJpCaGWj75c3yOZKBB9mNpQNPUKkK0D6wgLH8MGoyRxTX6Y05Q4AnYNXMZwXM4eij/9WpsM/9CoRnFQXGR6MEaY+FXvXEO3RO0JaStk6OXuHVATHJE+1W+TU3bSZ2ksMtqjO0zfSJCdBv7y2d8DMx6TfVme3q0ZpTKMMu4YL/t7ciTNtdDkwPogh3Cnjx7qk08SHwf+dksZ7M2vCOlfsF0hQ6J4ehPCaHTNrM/zBSOqD83dBEBCW/F/LEmeh0nOHd7oVl3/Qo/9GUDkkbj7yz+9cvvu+dDAtx8NzCDTP4iKdZvk9MWiizvtILLepysflSvTLFBZ37RLwiriqyRxYv/zrgFd/9XVHh/OmzBvDX4mitMR/lUavs2Vx6cR94lzAkplm3IRNy4TFfu47tuYs9EQPIPVta4P64tV+sZ7n3ued3cgEx2YK+QL5+xms6osk8qQbTyuKVGdaX9FQqk6qfDnT5ykxk0VK7KZ62b6DNDUfQlqGHxSMKv1P0XN5BqMeKG1P4Wp5QfZDUCEldppoX0U6ss2jIko2XpURKCIhfaOqLPfShdtS37ZrT+jFRSH2xYVV1rmT/MBtRQhxiO4MQ3iAGlaZi+9PWBEIXOVnu9jN1f921lWLZky9bqbM3J2MAAI9jmuAx3gyoEUa6P2ivs0EeNv/OR+AX6q5SW6l5HaoFuS6jr6yg9limu+P0KYKzfMXWcQSfTXzpOzKEKpwI3YGXZpSSy2LTlMgfmFA3CF6R5c9xWEtRuCg2ZPUQ2Nb6dRFTNd4TfGHrnEWSKHPuRyiJSDAZ+KX0VxmSHjGPbQTLVpqixia2uyhQ394gBMt7C3ZAmxn/DJS+l1fBsAo2Eir/C0jG9csd4+/tp12pPc/BVJGaK9mfvr7M/CeztrmCO5qY06Edi4xAGtiEhnWAbzLy2VEyazE1J5nPmgU4RpW4Sa0TnOT6w5lgt3/tMpROigHHmexBGAMY0mdcDbDxWIz41NgdD6oxgHsJRgr5RnT6wZAkTOcStU4NMOQNemSO7gxGahdEsC+NRVGxMUhQmmM0llWRbbmFGHzEqLM4Iw0H7577Kyo+Zf+2cUFIOw93gEY171vQaM0HLwpjpdRR6Jz7V0ckE7XzYJ0TmY9znLdzkva0vNrAGGT5SUZ5uaHDkcGvI0ySpwkasEgZPMseYcu85w8HPdSNi+4T6A83iAwDbxgeFcB1ZM2iGXzFcEOUlYVrEckaOyodfvaYSQ7GuB4ISE0nYJc15X/1ciDTPbPCgYJK55VkEor4LvzL9S2WDy4xj+6FOqVyTAC2ZNowheeeSI5hA/02l8UYkv4nk9iaVn+kCVEUstgk5Hyq+gJm6R9vG3rhuM904he/hFmNQaUIATB1y3vw+OmxP4X5Yi6A5I5jJufHCjF9+AGNwnEllZjUco6XhsO5T5+R3yxz5yLVOnAn0zuS+6zdj0nTJbEZCbXJdtpfYZfCeCOqJHoE2vPPFS6eRLjIJlG69X93nfR0mxSFXzp1Zc0lt/VafDaImhUMtbnqWVb9M4nGNQLN68BHP7AR8Il9dkcxzmBv8PCZlw9guY0lurbBsmNYlwJZsA/B15/HfkbjbwPddaVecls/elmDHNW2r4crAx43feNkfRwsaNq/yyJ0d/p5hZ6AZajz7DBfUok0ZU62gCzz7x8eVfJTKA8IWn45vINLSM1q+HF9CV9qF3zP6Ml21kPPL3CXzkuYUlnSqT+Ij4tI/od5KwIs+tDajDs64owN7tOAd6eucGz+KfO26iNcBFpbWA5732bBNWO4kHNpr9D955L61bvHCF/mwSrz6eQaDjfDEANqGMkFc+NGxpKZzCD2sj/JrHd+zlPQ8Iz7Q+2JVIiVCuCKoK/hlAEHzvk/Piq3mRL1rT/fEh9hoT5GJmeYswg1otiKydizJ/fS2SeKHVu6Z3JEHjiW8NaTQgP5xdBli8nC57XiN9hrquBu99hn9zqwo92+PM2JXtpeVZS0PdqR5mDyDreMMtEws+CpwaRyyzoYtfcvt9PJIW0fJVNNi/FFyRsea7peLvJrL+5b4GOXJ8tAr+ATk9f8KmiIsRhqRy0vFzwRV3Z5dZ3QqIU8JQ/uQpkJbjMUMFj2F9sCFeaBjI4+fL/oN3+LQgjI4zuAfQ+3IPIPFQBccf0clJpsfpnBxD84atwtupkGqKvrH7cGNl/QcWcSi6wcVDML6ljOgYbo+2BOAWNNjlUBPiyitUAwbnhFvLbnqw42kR3Yp2kv2dMeDdcGOX5kT4S6M44KHEB/SpCfl7xgsUvs+JNY9G3O2X/6FEt9FyAn57lrbiu+tl83sCymSvq9eZbe9mchL7MTf/Ta78e80zSf0hYY5eUU7+ff14jv7Xy8qjzfzzzvaJnrIdvFb5BLWKcWGy5/w7+vV2cvIfwHqdTB+RuJK5oj9mbt0Hy94AmjMjjwYNZlNS6uiyxNnwNyt3gdreLb64p/3+08nXkb92LTkkRgFOwk1oGEVllcOj5lv1hfAZywDows0944U8vUFw+A/nuVq/UCygsrmWIBnHyU01d0XJPwriEOvx/ISK6Pk4y2w0gmojZs7lU8TtakBAdne4v/aNxmMpK4VcGMp7si0yqsiolXRuOi1Z1P7SqD3Zmp0CWcyK4Ubmp2SXiXuI5nGLCieFHKHNRIlcY3Pys2dwMTYCaqlyWSITwr2oGXvyU3h1Pf8eQ3w1bnD7ilocVjYDkcXR3Oo1BXgMLTUjNw2xMVwjtp99NhSVc5aIWrDQT5DHPKtCtheBP4zHcw4dz2eRdTMamhlHhtfgqJJHI7NGDUw1XL8vsSeSHyKqDtqoAmrQqsYwvwi7HW3ojWyhIa5oz5xJTaq14NAzFLjVLR12rRNUQ6xohDnrWFb5bG9yf8aCD8d5phoackcNJp+Dw3Due3RM+5Rid7EuIgsnwgpX0rUWh/nqPtByMhMZZ69NpgvRTKZ62ViZ+Q7Dp5r4K0d7EfJuiy06KuIYauRh5Ecrhdt2QpTS1k1AscEHvapNbU3HL1F2TFyR33Wxb5MvH5iZsrn3SDcsxlnnshO8PLwmdGN+paWnQuORtZGX37uhFT64SeuPsx8UOokY6ON85WdQ1dki5zErsJGazcBOddWJEKqNPiJpsMD1GrVLrVY+AOdPWQneTyyP1hRX/lMM4ZogGGOhYuAdr7F/DOiAoc++cn5vlf0zkMUJ40Z1rlgv9BelPqVOpxKeOpzKdF8maK+1Vv23MO9k/8+qpLoxrIGH2EDQlnGmH8CD31G8QqlyQIcpmR5bwmSVw9/Ns6IHgulCRehvZ/+VrM60Cu/r3AontFfrljew74skYe2uyn7JKQtFQBQRJ9ryGic/zQOsbS4scUBctA8cPToQ3x6ZBQu6DPu5m1bnCtP8TllLYA0UTQNVqza5nfew3Mopy1GPUwG5jsl0OVXniPmAcmLqO5HG8Hv3nSLecE9oOjPDXcsTxoCBxYyzBdj4wmnyEV4kvFDunipS8SSkvdaMnTBN9brHUR8xdmmEAp/Pdqk9uextp1t+JrtXwpN/MG2w/qhRMpSNxQ1uhg/kKO30eQ/FyHUDkWHT8V6gGRU4DhDMxZu7xXij9Ui6jlpWmQCqJg3FkOTq3WKneCRYZxBXMNAVLQgHXSCGSqNdjebY94oyIpVjMYehAiFx/tqzBXFHZaL5PeeD74rW5OysFoUXY8sebUZleFTUa/+zBKVTFDopTReXNuZq47QjkWnxjirCommO4L/GrFtVV21EpMyw8wyThL5Y59d88xtlx1g1ttSICDwnof6lt/6zliPzgVUL8jWBjC0o2D6Kg+jNuThkAlaDJsq/AG2aKA//A76avw2KNqtv223P+Wq3StRDDNKFFgtsFukYt1GFDWooFVXitaNhb3RCyJi4cMeNjROiPEDb4k+G3+hD8tsg+5hhmSc/8t2JTSwYoCzAI75doq8QTHe+E/Tw0RQSUDlU+6uBeNN3h6jJGX/mH8oj0i3caCNsjvTnoh73BtyZpsflHLq6AfwJNCDX4S98h4+pCOhGKDhV3rtkKHMa3EG4J9y8zFWI4UsfNzC/Rl5midNn7gwoN9j23HGCQQ+OAZpTTPMdiVow740gIyuEtd0qVxMyNXhHcnuXRKdw5wDUSL358ktjMXmAkvIB73BLa1vfF9BAUZInPYJiwxqFWQQBVk7gQH4ojfUQ/KEjn+A/WR6EEe4CtbpoLe1mzHkajgTIoE0SLDHVauKhrq12zrAXBGbPPWKCt4DGedq3JyGRbmPFW32bE7T20+73BatV/qQhhBWfWBFHfhYWXjALts38FemnoT+9bn1jDBMcUMmYgSc0e7GQjv2MUBwLU8ionCpgV+Qrhg7iUIfUY6JFxR0Y+ZTCPM+rVuq0GNLyJXX6nrUTt8HzFBRY1E/FIm2EeVA9NcXrj7S6YYIChVQCWr/m2fYUjC4j0XLkzZ8GCSLfmkW3PB/xq+nlXsKVBOj7vTvqKCOMq7Ztqr3cQ+N8gBnPaAps+oGwWOkbuxnRYj/x/WjiDclVrs22xMK4qArE1Ztk1456kiJriw6abkNeRHogaPRBgbgF9Z8i/tbzWELN4CvbqtrqV9TtGSnmPS2F9kqOIBaazHYaJ9bi3AoDBvlZasMluxt0BDXfhp02Jn411aVt6S4TUB8ZgFDkI6TP6gwPY85w+oUQSsjIeXVminrwIdK2ZAawb8Se6XOJbOaliQxHSrnAeONDLuCnFejIbp4YDtBcQCwMsYiRZfHefuEJqJcwKTTJ8sx5hjHmJI1sPFHOr6W9AhZ2NAod38mnLQk1gOz2LCAohoQbgMbUK9RMEA3LkiF7Sr9tLZp6lkciIGhE2V546w3Mam53VtVkGbB9w0Yk2XiRnCmbpxmHr2k4eSC0RuNbjNsUfDIfc8DZvRvgUDe1IlKdZTzcT4ZGEb53dp8VtsoZlyXzLHOdAbsp1LPTVaHvLA0GYDFMbAW/WUBfUAdHwqLFAV+3uHvYWrCfhUOR2i89qvCBoOb48usAGdcF2M4aKn79k/43WzBZ+xR1L0uZfia70XP9soQReeuhZiUnXFDG1T8/OXNmssTSnYO+3kVLAgeiY719uDwL9FQycgLPessNihMZbAKG7qwPZyG11G1+ZA3jAX2yddpYfmaKBlmfcK/V0mwIRUDC0nJSOPUl2KB8h13F4dlVZiRhdGY5farwN+f9hEb1cRi41ZcGDn6Xe9MMSTOY81ULJyXIHSWFIQHstVYLiJEiUjktlHiGjntN5/btB8Fu+vp28zl2fZXN+dJDyN6EXhS+0yzqpl/LSJNEUVxmu7BsNdjAY0jVsAhkNuuY0E1G48ej25mSt+00yPbQ4SRCVkIwb6ISvYtmJRPz9Zt5dk76blf+lJwAPH5KDF+vHAmACLoCdG2Adii6dOHnNJnTmZtoOGO8Q1jy1veMw6gbLFToQmfJa7nT7Al89mRbRkZZQxJTKgK5Kc9INzmTJFp0tpAPzNmyL/F08bX3nhCumM/cR/2RPn9emZ3VljokttZD1zVWXlUIqEU7SLk5I0lFRU0AcENXBYazNaVzsVHA/sD3o9hm42wbHIRb/BBQTKzAi8s3+bMtpOOZgLdQzCYPfX3UUxKd1WYVkGH7lh/RBBgMZZwXzU9+GYxdBqlGs0LP+DZ5g2BWNh6FAcR944B+K/JTWI3t9YyVyRhlP4CCoUk/mmF7+r2pilVBjxXBHFaBfBtr9hbVn2zDuI0kEOG3kBx8CGdPOjX1ph1POOZJUO1JEGG0jzUy2tK4X0CgVNYhmkqqQysRNtKuPdCJqK3WW57kaV17vXgiyPrl4KEEWgiGF1euI4QkSFHFf0TDroQiLNKJiLbdhH0YBhriRNCHPxSqJmNNoketaioohqMglh6wLtEGWSM1EZbQg72h0UJAIPVFCAJOThpQGGdKfFovcwEeiBuZHN2Ob4uVM7+gwZLz1D9E7ta4RmMZ24OBBAg7Eh6dLXGofZ4U2TFOCQMKjwhVckjrydRS+YaqCw1kYt6UexuzbNEDyYLTZnrY1PzsHZJT4U+awO2xlqTSYu6n/U29O2wPXgGOEKDMSq+zTUtyc8+6iLp0ivav4FKx+xxVy4FxhIF/pucVDqpsVe2jFOfdZhTzLz2QjtzvsTCvDPU7bzDH2eXVKUV9TZ+qFtaSSxnYgYdXKwVreIgvWhT9eGDB2OvnWyPLfIIIfNnfIxU8nW7MbcH05nhlsYtaW9EZRsxWcKdEqInq1DiZPKCz7iGmAU9/ccnnQud2pNgIGFYOTAWjhIrd63aPDgfj8/sdlD4l+UTlcxTI9jbaMqqN0gQxSHs60IAcW3cH4p3V1aSciTKB29L1tz2eUQhRiTgTvmqc+sGtBNh4ky0mQJGsdycBREP+fAaSs1EREDVo5gvgi5+aCN7NECw30owbCc1mSpjiahyNVwJd1jiGgzSwfTpzf2c5XJvG/g1n0fH88KHNnf+u7ZiRMlXueSIsloJBUtW9ezvsx9grfsX/FNxnbxU1Lvg0hLxixypHKGFAaPu0xCD8oDTeFSyfRT6s8109GMUZL8m2xXp8X2dpPCWWdX84iga4BrTlOfqox4shqEgh/Ht4qRst52cA1xOIUuOxgfUivp6v5f8IVyaryEdpVk72ERAwdT4aoY1usBgmP+0m06Q216H/nubtNYxHaOIYjcach3A8Ez/zc0KcShhel0HCYjFsA0FjYqyJ5ZUH1aZw3+zWC0hLpM6GDfcAdn9fq2orPmZbW6XXrf+Krc9RtvII5jeD3dFoT1KwZJwxfUMvc5KLfn8rROW23Jw89sJ2a5dpB3qWDUBWF2iX8OCuKprHosJ2mflBR+Wqs86VvgI/XMnsqb97+VlKdPVysczPj8Jhzf+WCvGBHijAqYlavbF60soMWlHbvKT+ScvhprgeTln51xX0sF+Eadc/l2s2a5BgkVbHYyz0E85p0LstqH+gEGiR84nBRRFIn8hLSZrGwqjZ3E29cuGi+5Z5bp7EM8MWFa9ssS/vy4VrDfECSv7DSU84DaP0sXI3Ap4lWznQ65nQoTKRWU30gd7Nn8ZowUvGIx4aqyXGwmA/PB4qN8msJUODezUHEl0VP9uo+cZ8vPFodSIB4C7lQYjEFj8yu49C2KIV3qxMFYTevG8KqAr0TPlkbzHHnTpDpvpzziAiNFh8xiT7C/TiyH0EguUw4vxAgpnE27WIypV+uFN2zW7xniF/n75trs9IJ5amB1zXXZ1LFkJ6GbS/dFokzl4cc2mamVwhL4XU0Av5gDWAl+aEWhAP7t2VIwU+EpvfOPDcLASX7H7lZpXA2XQfbSlD4qU18NffNPoAKMNSccBfO9YVVgmlW4RydBqfHAV7+hrZ84WJGho6bNT0YMhxxLdOx/dwGj0oyak9aAkNJ8lRJzUuA8sR+fPyiyTgUHio5+Pp+YaKlHrhR41jY5NESPS3x+zTMe0S2HnLOKCOQPpdxKyviBvdHrCDRqO+l96HhhNBLXWv4yEMuEUYo8kXnYJM8oIgVM4XJ+xXOev4YbWeqsvgq0lmw4/PiYr9sYLt+W5EAuYSFnJEan8CwJwbtASBfLBBpJZiRPor/aCJBZsM+MhvS7ZepyHvU8m5WSmaZnxuLts8ojl6KkS8oSAHkq5GWlCB/NgJ5W3rO2Cj1MK7ahxsCrbTT3a0V/QQH+sErxV4XUWDHx0kkFy25bPmBMBQ6BU3HoHhhYcJB9JhP6NXUWKxnE0raXHB6U9KHpWdQCQI72qevp5fMzcm+AvC85rsynVQhruDA9fp9COe7N56cg1UKGSas89vrN+WlGLYTwi5W+0xYdKEGtGCeNJwXKDU0XqU5uQYnWsMwTENLGtbQMvoGjIFIEMzCRal4rnBAg7D/CSn8MsCvS+FDJJAzoiioJEhZJgAp9n2+1Yznr7H+6eT4YkJ9Mpj60ImcW4i4iHDLn9RydB8dx3QYm3rsX6n4VRrZDsYK6DCGwkwd5n3/INFEpk16fYpP6JtMQpqEMzcOfQGAHXBTEGzuLJ03GYQL9bmV2/7ExDlRf+Uvf1sM2frRtCWmal12pMgtonvSCtR4n1CLUZRdTHDHP1Otwqd+rcdlavnKjUB/OYXQHUJzpNyFoKpQK+2OgrEKpGyIgIBgn2y9QHnTJihZOpEvOKIoHAMGAXHmj21Lym39Mbiow4IF+77xNuewziNVBxr6KD5e+9HzZSBIlUa/AmsDFJFXeyrQakR3FwowTGcADJHcEfhGkXYNGSYo4dh4bxwLM+28xjiqkdn0/3R4UEkvcBrBfn/SzBc1XhKM2VPlJgKSorjDac96V2UnQYXl1/yZPT4DVelgO+soMjexXwYO58VLl5xInQUZI8jc3H2CPnCNb9X05nOxIy4MlecasTqGK6s2az4RjpF2cQP2G28R+7wDPsZDZC/kWtjdoHC7SpdPmqQrUAhMwKVuxCmYTiD9q/O7GHtZvPSN0CAUQN/rymXZNniYLlJDE70bsk6Xxsh4kDOdxe7A2wo7P9F5YvqqRDI6brf79yPCSp4I0jVoO4YnLYtX5nzspR5WB4AKOYtR1ujXbOQpPyYDvfRE3FN5zw0i7reehdi7yV0YDRKRllGCGRk5Yz+Uv1fYl2ZwrnGsqsjgAVo0xEUba8ohjaNMJNwTwZA/wBDWFSCpg1eUH8MYL2zdioxRTqgGQrDZxQyNzyBJPXZF0+oxITJAbj7oNC5JwgDMUJaM5GqlGCWc//KCIrI+aclEe4IA0uzv7cuj6GCdaJONpi13O544vbtIHBF+A+JeDFUQNy61Gki3rtyQ4aUywn6ru314/dkGiP8Iwjo0J/2Txs49ZkwEl4mx+iYUUO55I6pJzU4P+7RRs+DXZkyKUYZqVWrPF4I94m4Wx1tXeE74o9GuX977yvJ/jkdak8+AmoHVjI15V+WwBdARFV2IPirJgVMdsg1Pez2VNHqa7EHWdTkl3XTcyjG9BiueWFvQfXI8aWSkuuRmqi/HUuzqyvLJfNfs0txMqldYYflWB1BS31WkuPJGGwXUCpjiQSktkuBMWwHjSkQxeehqw1Kgz0Trzm7QbtgxiEPDVmWCNCAeCfROTphd1ZNOhzLy6XfJyG6Xgd5MCAZw4xie0Sj5AnY1/akDgNS9YFl3Y06vd6FAsg2gVQJtzG7LVq1OH2frbXNHWH/NY89NNZ4QUSJqL2yEcGADbT38X0bGdukqYlSoliKOcsSTuqhcaemUeYLLoI8+MZor2RxXTRThF1LrHfqf/5LcLAjdl4EERgUysYS2geE+yFdasU91UgUDsc2cSQ1ZoT9+uLOwdgAmifwQqF028INc2IQEDfTmUw3eZxvz7Ud1z3xc1PQfeCvfKsB9jOhRj7rFyb9XcDWLcYj0bByosychMezMLVkFiYcdBBQtvI6K0KRuOZQH2kBsYHJaXTkup8F0eIhO1/GcIwWKpr2mouB7g5TUDJNvORXPXa/mU8bh27TAZYBe2sKx4NSv5OjnHIWD2RuysCzBlUfeNXhDd2jxnHoUlheJ3jBApzURy0fwm2FwwsSU0caQGl0Kv8hopRQE211NnvtLRsmCNrhhpEDoNiZEzD2QdJWKbRRWnaFedXHAELSN0t0bfsCsMf0ktfBoXBoNA+nZN9+pSlmuzspFevmsqqcMllzzvkyXrzoA+Ryo1ePXpdGOoJvhyru+EBRsmOp7MXZ0vNUMUqHLUoKglg1p73sWeZmPc+KAw0pE2zIsFFE5H4192KwDvDxdxEYoDBDNZjbg2bmADTeUKK57IPD4fTYF4c6EnXx/teYMORBDtIhPJneiZny7Nv/zG+YmekIKCoxr6kauE2bZtBLufetNG0BtBY7f+/ImUypMBvdWu/Q7vTMRzw5aQGZWuc1V0HEsItFYMIBnoKGZ0xcarba/TYZq50kCaflFysYjA4EDKHqGdpYWdKYmm+a7TADmW35yfnOYpZYrkpVEtiqF0EujI00aeplNs2k+qyFZNeE3CDPL9P6b4PQ/kataHkVpLSEVGK7EX6rAa7IVNrvZtFvOA6okKvBgMtFDAGZOx88MeBcJ8AR3AgUUeIznAN6tjCUipGDZONm1FjWJp4A3QIzSaIOmZ7DvF/ysYYbM/fFDOV0jntAjRdapxJxL0eThpEhKOjCDDq2ks+3GrwxqIFKLe1WdOzII8XIOPGnwy6LKXVfpSDOTEfaRsGujhpS4hBIsMOqHbl16PJxc4EkaVu9wpEYlF/84NSv5Zum4drMfp9yXbzzAOJqqS4YkI4cBrFrC7bMPiCfgI3nNZAqkk3QOZqR+yyqx+nDQKBBBZ7QKrfGMCL+XpqFaBJU0wpkBdAhbR4hJsmT5aynlvkouoxm/NjD5oe6BzVIO9uktM+/5dEC5P7vZvarmuO/lKXz4sBabVPIATuKTrwbJP8XUkdM6uEctHKXICUJGjaZIWRbZp8czquQYfY6ynBUCfIU+gG6wqSIBmYIm9pZpXdaL121V7q0VjDjmQnXvMe7ysoEZnZL15B0SpxS1jjd83uNIOKZwu5MPzg2NhOx3xMOPYwEn2CUzbSrwAs5OAtrz3GAaUkJOU74XwjaYUmGJdZBS1NJVkGYrToINLKDjxcuIlyfVsKQSG/G4DyiO2SlQvJ0d0Ot1uOG5IFSAkq+PRVMgVMDvOIJMdqjeCFKUGRWBW9wigYvcbU7CQL/7meF2KZAaWl+4y9uhowAX7elogAvItAAxo2+SFxGRsHGEW9BnhlTuWigYxRcnVUBRQHV41LV+Fr5CJYV7sHfeywswx4XMtUx6EkBhR+q8AXXUA8uPJ73Pb49i9KG9fOljvXeyFj9ixgbo6CcbAJ7WHWqKHy/h+YjBwp6VcN7M89FGzQ04qbrQtgrOFybg3gQRTYG5xn73ArkfQWjCJROwy3J38Dx/D7jOa6BBNsitEw1wGq780EEioOeD+ZGp2J66ADiVGMayiHYucMk8nTK2zzT9CnEraAk95kQjy4k0GRElLL5YAKLQErJ5rp1eay9O4Fb6yJGm9U4FaMwPGxtKD6odIIHKoWnhKo1U8KIpFC+MVn59ZXmc7ZTBZfsg6FQ8W10YfTr4u0nYrpHZbZ1jXiLmooF0cOm0+mPnJBXQtepc7n0BqOipNCqI6yyloTeRShNKH04FIo0gcMk0H/xThyN4pPAWjDDkEp3lNNPRNVfpMI44CWRlRgViP64eK0JSRp0WUvCWYumlW/c58Vcz/yMwVcW5oYb9+26TEhwvbxiNg48hl1VI1UXTU//Eta+BMKnGUivctfL5wINDD0giQL1ipt6U7C9cd4+lgqY2lMUZ02Uv6Prs+ZEZer7ZfWBXVghlfOOrClwsoOFKzWEfz6RZu1eCs+K8fLvkts5+BX0gyrFYve0C3qHrn5U/Oh6D/CihmWIrY7HUZRhJaxde+tldu6adYJ+LeXupQw0XExC36RETdNFxcq9glMu4cNQSX9cqR/GQYp+IxUkIcNGWVU7ZtGa6P3XAyodRt0XeS3Tp01AnCh0ZbUh4VrSZeV9RWfSoWyxnY3hzcZ30G/InDq4wxRrEejreBxnhIQbkxenxkaxl+k7eLUQkUR6vKJ2iDFNGX3WmVA1yaOH+mvhBd+sE6vacQzFobwY5BqEAFmejwW5ne7HtVNolOUgJc8CsUxmc/LBi8N5mu9VsIA5HyErnS6zeCz7VLI9+n/hbT6hTokMXTVyXJRKSG2hd2labXTbtmK4fNH3IZBPreSA4FMeVouVN3zG5x9CiGpLw/3pceo4qGqp+rVp+z+7yQ98oEf+nyH4F3+J9IheDBa94Wi63zJbLBCIZm7P0asHGpIJt3PzE3m0S4YIWyXBCVXGikj8MudDPB/6Nm2v4IxJ5gU0ii0guy5SUHqGUYzTP0jIJU5E82RHUXtX4lDdrihBLdP1YaG1AGUC12rQKuIaGvCpMjZC9bWSCYnjDlvpWbkdXMTNeBHLKiuoozMGIvkczmP0aRJSJ8PYnLCVNhKHXBNckH79e8Z8Kc2wUej4sQZoH8qDRGkg86maW/ZQWGNnLcXmq3FlXM6ssR/3P6E/bHMvm6HLrv1yRixit25JsH3/IOr2UV4BWJhxXW5BJ6Xdr07n9kF3ZNAk6/Xpc5MSFmYJ2R7bdL8Kk7q1OU9Elg/tCxJ8giT27wSTySF0GOxg4PbYJdi/Nyia9Nn89CGDulfJemm1aiEr/eleGSN+5MRrVJ4K6lgyTTIW3i9cQ0dAi6FHt0YMbH3wDSAtGLSAccezzxHitt1QdhW36CQgPcA8vIIBh3/JNjf/Obmc2yzpk8edSlS4lVdwgW5vzbYEyFoF4GCBBby1keVNueHAH+evi+H7oOVfS3XuPQSNTXOONAbzJeSb5stwdQHl1ZjrGoE49I8+A9j3t+ahhQj74FCSWpZrj7wRSFJJnnwi1T9HL5qrCFW/JZq6P62XkMWTb+u4lGpKfmmwiJWx178GOG7KbrZGqyWwmuyKWPkNswkZ1q8uptUlviIi+AXh2bOOTOLsrtNkfqbQJeh24reebkINLkjut5r4d9GR/r8CBa9SU0UQhsnZp5cP+RqWCixRm7i4YRFbtZ4EAkhtNa6jHb6gPYQv7MKqkPLRmX3dFsK8XsRLVZ6IEVrCbmNDc8o5mqsogjAQfoC9Bc7R6gfw03m+lQpv6kTfhxscDIX6s0w+fBxtkhjXAXr10UouWCx3C/p/FYwJRS/AXRKkjOb5CLmK4XRe0+xeDDwVkJPZau52bzLEDHCqV0f44pPgKOkYKgTZJ33fmk3Tu8SdxJ02SHM8Fem5SMsWqRyi2F1ynfRJszcFKykdWlNqgDA/L9lKYBmc7Zu/q9ii1FPF47VJkqhirUob53zoiJtVVRVwMR34gV9iqcBaHbRu9kkvqk3yMpfRFG49pKKjIiq7h/VpRwPGTHoY4cg05X5028iHsLvUW/uz+kjPyIEhhcKUwCkJAwbR9pIEGOn8z6svAO8i89sJ3dL5qDWFYbS+HGPRMxYwJItFQN86YESeJQhn2urGiLRffQeLptDl8dAgb+Tp47UQPxWOw17OeChLN1WnzlkPL1T5O+O3Menpn4C3IY5LEepHpnPeZHbvuWfeVtPlkH4LZjPbBrkJT3NoRJzBt86CO0Xq59oQ+8dsm0ymRcmQyn8w71mhmcuEI5byuF+C88VPYly2sEzjlzAQ3vdn/1+Hzguw6qFNNbqenhZGbdiG6RwZaTG7jTA2X9RdXjDN9yj1uQpyO4Lx8KRAcZcbZMafp4wPOd5MdXoFY52V1A8M9hi3sso93+uprE0qYNMjkE22CvK4HuUxqN7oIz5pWuETq1lQAjqlSlqdD2Rnr/ggp/TVkQYjn9lMfYelk2sH5HPdopYo7MHwlV1or9Bxf+QCyLzm92vzG2wjiIjC/ZHEJzeroJl6bdFPTpZho5MV2U86fLQqxNlGIMqCGy+9WYhJ8ob1r0+Whxde9L2PdysETv97O+xVw+VNN1TZSQN5I6l9m5Ip6pLIqLm4a1B1ffH6gHyqT9p82NOjntRWGIofO3bJz5GhkvSWbsXueTAMaJDou99kGLqDlhwBZNEQ4mKPuDvVwSK4WmLluHyhA97pZiVe8g+JxmnJF8IkV/tCs4Jq/HgOoAEGR9tCDsDbDmi3OviUQpG5D8XmKcSAUaFLRXb2lmJTNYdhtYyfjBYZQmN5qT5CNuaD3BVnlkCk7bsMW3AtXkNMMTuW4HjUERSJnVQ0vsBGa1wo3Qh7115XGeTF3NTz8w0440AgU7c3bSXO/KMINaIWXd0oLpoq/0/QJxCQSJ9XnYy1W7TYLBJpHsVWD1ahsA7FjNvRd6mxCiHsm8g6Z0pnzqIpF1dHUtP2ITU5Z1hZHbu+L3BEEStBbL9XYvGfEakv1bmf+bOZGnoiuHEdlBnaChxYKNzB23b8sw8YyT7Ajxfk49eJIAvdbVkdFCe2J0gMefhQ0bIZxhx3fzMIysQNiN8PgOUKxOMur10LduigREDRMZyP4oGWrP1GFY4t6groASsZ421os48wAdnrbovNhLt7ScNULkwZ5AIZJTrbaKYTLjA1oJ3sIuN/aYocm/9uoQHEIlacF1s/TM1fLcPTL38O9fOsjMEIwoPKfvt7opuI9G2Hf/PR4aCLDQ7wNmIdEuXJ/QNL72k5q4NejAldPfe3UVVqzkys8YZ/jYOGOp6c+YzRCrCuq0M11y7TiN6qk7YXRMn/gukxrEimbMQjr3jwRM6dKVZ4RUfWQr8noPXLJq6yh5R3EH1IVOHESst/LItbG2D2vRsZRkAObzvQAAD3mb3/G4NzopI0FAiHfbpq0X72adg6SRj+8OHMShtFxxLZlf/nLgRLbClwl5WmaYSs+yEjkq48tY7Z2bE0N91mJwt+ua0NlRJIDh0HikF4UvSVorFj2YVu9YeS5tfvlVjPSoNu/Zu6dEUfBOT555hahBdN3Sa5Xuj2Rvau1lQNIaC944y0RWj9UiNDskAK1WoL+EfXcC6IbBXFRyVfX/WKXxPAwUyIAGW8ggZ08hcijKTt1YKnUO6QPvcrmDVAb0FCLIXn5id4fD/Jx4tw/gbXs7WF9b2RgXtPhLBG9vF5FEkdHAKrQHZAJC/HWvk7nvzzDzIXZlfFTJoC3JpGgLPBY7SQTjGlUvG577yNutZ1hTfs9/1nkSXK9zzKLRZ3VODeKUovJe0WCq1zVMYxCJMenmNzPIU2S8TA4E7wWmbNkxq9rI2dd6v0VpcAPVMxnDsvWTWFayyqvKZO7Z08a62i/oH2/jxf8rpmfO64in3FLiL1GX8IGtVE9M23yGsIqJbxDTy+LtaMWDaPqkymb5VrQdzOvqldeU0SUi6IirG8UZ3jcpRbwHa1C0Dww9G/SFX3gPvTJQE+kyz+g1BeMILKKO+olcHzctOWgzxYHnOD7dpCRtuZEXACjgqesZMasoPgnuDC4nUviAAxDc5pngjoAITIkvhKwg5d608pdrZcA+qn5TMT6Uo/QzBaOxBCLTJX3Mgk85rMfsnWx86oLxf7p2PX5ONqieTa/qM3tPw4ZXvlAp83NSD8F7+ZgctK1TpoYwtiU2h02HCGioH5tkVCqNVTMH5p00sRy2JU1qyDBP2CII/Dg4WDsIl+zgeX7589srx6YORRQMBfKbodbB743Tl4WLKOEnwWUVBsm94SOlCracU72MSyj068wdpYjyz1FwC2bjQnxnB6Mp/pZ+yyZXtguEaYB+kqhjQ6UUmwSFazOb+rhYjLaoiM+aN9/8KKn0zaCTFpN9eKwWy7/u4EHzO46TdFSNjMfn2iPSJwDPCFHc0I1+vjdAZw5ZjqR/uzi9Zn20oAa5JnLEk/EA3VRWE7J/XrupfFJPtCUuqHPpnlL7ISJtRpSVcB8qsZCm2QEkWoROtCKKxUh3yEcMbWYJwk6DlEBG0bZP6eg06FL3v6RPb7odGuwm7FN8fG4woqtB8e7M5klPpo97GoObNwt+ludTAmxyC5hmcFx+dIvEZKI6igFKHqLH01iY1o7903VzG9QGetyVx5RNmBYUU+zIuSva/yIcECUi4pRmE3VkF2avqulQEUY4yZ/wmNboBzPmAPey3+dSYtBZUjeWWT0pPwCz4Vozxp9xeClIU60qvEFMQCaPvPaA70WlOP9f/ey39macvpGCVa+zfa8gO44wbxpJUlC8GN/pRMTQtzY8Z8/hiNrU+Zq64ZfFGIkdj7m7abcK1EBtws1X4J/hnqvasPvvDSDYWN+QcQVGMqXalkDtTad5rYY0TIR1Eqox3czwPMjKPvF5sFv17Thujr1IZ1Ytl4VX1J0vjXKmLY4lmXipRAro0qVGEcXxEVMMEl54jQMd4J7RjgomU0j1ptjyxY+cLiSyXPfiEcIS2lWDK3ISAy6UZ3Hb5vnPncA94411jcy75ay6B6DSTzK6UTCZR9uDANtPBrvIDgjsfarMiwoax2OlLxaSoYn4iRgkpEGqEkwox5tyI8aKkLlfZ12lO11TxsqRMY89j5JaO55XfPJPDL1LGSnC88Re9Ai+Nu5bZjtwRrvFITUFHPR4ZmxGslQMecgbZO7nHk32qHxYkdvWpup07ojcMCaVrpFAyFZJJbNvBpZfdf39Hdo2kPtT7v0/f8R/B5Nz4f1t9/3zNM/7n6SUHfcWk5dfQFJvcJMgPolGCpOFb/WC0FGWU2asuQyT+rm88ZKZ78Cei/CAh939CH0JYbpZIPtxc2ufXqjS3pHH9lnWK4iJ7OjR/EESpCo2R3MYKyE7rHfhTvWho4cL1QdN4jFTyR6syMwFm124TVDDRXMNveI1Dp/ntwdz8k8kxw7iFSx6+Yx6O+1LzMVrN0BBzziZi9kneZSzgollBnVwBh6oSOPHXrglrOj+QmR/AESrhDpKrWT+8/AiMDxS/5wwRNuGQPLlJ9ovomhJWn8sMLVItQ8N/7IXvtD8kdOoHaw+vBSbFImQsv/OCAIui99E+YSIOMlMvBXkAt+NAZK8wB9Jf8CPtB+TOUOR+z71d/AFXpPBT6+A5FLjxMjLIEoJzrQfquvxEIi+WoUzGR1IzQFNvbYOnxb2PyQ0kGdyXKzW2axQL8lNAXPk6NEjqrRD1oZtKLlFoofrXw0dCNWASHzy+7PSzOUJ3XtaPZsxLDjr+o41fKuKWNmjiZtfkOzItvlV2MDGSheGF0ma04qE3TUEfqJMrXFm7DpK+27DSvCUVf7rbNoljPhha5W7KBqVq0ShUSTbRmuqPtQreVWH4JET5yMhuqMoSd4r/N8sDmeQiQQvi1tcZv7Moc7dT5X5AtCD6kNEGZOzVcNYlpX4AbTsLgSYYliiPyVoniuYYySxsBy5cgb3pD+EK0Gpb0wJg031dPgaL8JZt6sIvzNPEHfVPOjXmaXj4bd4voXzpZ5GApMhILgMbCEWZ2zwgdeQgjNHLbPIt+KqxRwWPLTN6HwZ0Ouijj4UF+Sg0Au8XuIKW0WxlexdrFrDcZJ8Shauat3X0XmHygqgL1nAu2hrJFb4wZXkcS+i36KMyU1yFvYv23bQUJi/3yQpqr/naUOoiEWOxckyq/gq43dFou1DVDaYMZK9tho7+IXXokBCs5GRfOcBK7g3A+jXQ39K4YA8PBRW4m5+yR0ZAxWJncjRVbITvIAPHYRt1EJ3YLiUbqIvoKHtzHKtUy1ddRUQ0AUO41vonZDUOW+mrszw+SW/6Q/IUgNpcXFjkM7F4CSSQ2ExZg85otsMs7kqsQD4OxYeBNDcSpifjMoLb7GEbGWTwasVObmB/bfPcUlq0wYhXCYEDWRW02TP5bBrYsKTGWjnWDDJ1F7zWai0zW/2XsCuvBQjPFcTYaQX3tSXRSm8hsAoDdjArK/OFp6vcWYOE7lizP0Yc+8p16i7/NiXIiiQTp7c7Xus925VEtlKAjUdFhyaiLT7VxDagprMFwix4wZ05u0qj7cDWFd0W9OYHIu3JbJKMXRJ1aYNovugg+QqRN7fNHSi26VSgBpn+JfMuPo3aeqPWik/wI5Rz3BWarPQX4i5+dM0npwVOsX+KsOhC7vDg+OJsz4Q5zlnIeflUWL6QYMbf9WDfLmosLF4Qev3mJiOuHjoor/dMeBpA9iKDkMjYBNbRo414HCxjsHrB4EXNbHzNMDHCLuNBG6Sf+J4MZ/ElVsDSLxjIiGsTPhw8BPjxbfQtskj+dyNMKOOcUYIRBEIqbazz3lmjlRQhplxq673VklMMY6597vu+d89ec/zq7Mi4gQvh87ehYbpOuZEXj5g/Q7S7BFDAAB9DzG35SC853xtWVcnZQoH54jeOqYLR9NDuwxsVthTV7V99n/B7HSbAytbEyVTz/5NhJ8gGIjG0E5j3griULUd5Rg7tQR+90hJgNQKQH2btbSfPcaTOfIexc1db1BxUOhM1vWCpLaYuKr3FdNTt/T3PWCpEUWDKEtzYrjpzlL/wri3MITKsFvtF8QVV/NhVo97aKIBgdliNc10dWdXVDpVtsNn+2UIolrgqdWA4EY8so0YvB4a+aLzMXiMAuOHQrXY0tr+CL10JbvZzgjJJuB1cRkdT7DUqTvnswVUp5kkUSFVtIIFYK05+tQxT6992HHNWVhWxUsD1PkceIrlXuUVRogwmfdhyrf6zzaL8+c0L7GXMZOteAhAVQVwdJh+7nrX7x4LaIIfz2F2v7Dg/uDfz2Fa+4gFm2zHAor8UqimJG3VTJtZEoFXhnDYXvxMJFc6ku2bhbCxzij2z5UNuK0jmp1mnvkVNUfR+SEmj1Lr94Lym75PO7Fs0MIr3GdsWXRXSfgLTVY0FLqba97u1In8NAcY7IC6TjWLigwKEIm43NxTdaVTv9mcKkzuzBkKd8x/xt1p/9BbP7Wyb4bpo1K1gnOpbLvKz58pWl3B55RJ/Z5mRDLPtNQg14jdOEs9+h/V5UVpwrAI8kGbX8KPVPDIMfIqKDjJD9UyDOPhjZ3vFAyecwyq4akUE9mDOtJEK1hpDyi6Ae87sWAClXGTiwPwN7PXWwjxaR79ArHRIPeYKTunVW24sPr/3HPz2IwH8oKH4OlWEmt4BLM6W5g4kMcYbLwj2usodD1088stZA7VOsUSpEVl4w7NMb1EUHMRxAxLF0CIV+0L3iZb+ekB1vSDSFjAZ3hfLJf7gFaXrOKn+mhR+rWw/eTXIcAgl4HvFuBg1LOmOAwJH3eoVEjjwheKA4icbrQCmvAtpQ0mXG0agYp5mj4Rb6mdQ+RV4QBPbxMqh9C7o8nP0Wko2ocnCHeRGhN1XVyT2b9ACsL+6ylUy+yC3QEnaKRIJK91YtaoSrcWZMMwxuM0E9J68Z+YyjA0g8p1PfHAAIROy6Sa04VXOuT6A351FOWhKfTGsFJ3RTJGWYPoLk5FVK4OaYR9hkJvezwF9vQN1126r6isMGXWTqFW+3HL3I/jurlIdDWIVvYY+s6yq7lrFSPAGRdnU7PVwY/SvWbZGpXzy3BQ2LmAJlrONUsZs4oGkly0V267xbD5KMY8woNNsmWG1VVgLCra8aQBBcI4DP2BlNwxhiCtHlaz6OWFoCW0vMR3ErrG7JyMjTSCnvRcsEHgmPnwA6iNpJ2DrFb4gLlhKJyZGaWkA97H6FFdwEcLT6DRQQL++fOkVC4cYGW1TG/3iK5dShRSuiBulmihqgjR45Vi03o2RbQbP3sxt90VxQ6vzdlGfkXmmKmjOi080JSHkLntjvsBJnv7gKscOaTOkEaRQqAnCA4HWtB4XnMtOhpRmH2FH8tTXrIjAGNWEmudQLCkcVlGTQ965Kh0H6ixXbgImQP6b42B49sO5C8pc7iRlgyvSYvcnH9FgQ3azLbQG2cUW96SDojTQStxkOJyOuDGTHAnnWkz29aEwN9FT8EJ4yhXOg+jLTrCPKeEoJ9a7lDXOjEr8AgX4BmnMQ668oW0zYPyQiVMPxKRHtpfnEEyaKhdzNVThlxxDQNdrHeZiUFb6NoY2KwvSb7BnRcpJy+/g/zAYx3fYSN5QEaVD2Y1VsNWxB0BSO12MRsRY8JLfAezRMz5lURuLUnG1ToKk6Q30FughqWN6gBNcFxP/nY/iv+iaUQOa+2Nuym46wtI/DvSfzSp1jEi4SdYBE7YhTiVV5cX9gwboVDMVgZp5YBQlHOQvaDNfcCoCJuYhf5kz5kwiIKPjzgpcRJHPbOhJajeoeRL53cuMahhV8Z7IRr6M4hW0JzT7mzaMUzQpm866zwM7Cs07fJYXuWvjAMkbe5O6V4bu71sOG6JQ4oL8zIeXHheFVavzxmlIyBkgc9IZlEDplMPr8xlcyss4pVUdwK1e7CK2kTsSdq7g5SHRAl3pYUB9Ko4fsh4qleOyJv1z3KFSTSvwEcRO/Ew8ozEDYZSqpfoVW9uhJfYrNAXR0Z3VmeoAD+rVWtwP/13sE/3ICX3HhDG3CMc476dEEC0K3umSAD4j+ZQLVdFOsWL2C1TH5+4KiSWH+lMibo+B55hR3Gq40G1n25sGcN0mEcoU2wN9FCVyQLBhYOu9aHVLWjEKx2JIUZi5ySoHUAI9b8hGzaLMxCZDMLhv8MkcpTqEwz9KFDpCpqQhVmsGQN8m24wyB82FAKNmjgfKRsXRmsSESovAwXjBIoMKSG51p6Um8b3i7GISs7kjTq/PZoioCfJzfKdJTN0Q45kQEQuh9H88M3yEs3DbtRTKALraM0YC8laiMiOOe6ADmTcCiREeAWZelBaEXRaSuj2lx0xHaRYqF65O0Lo5OCFU18A8cMDE4MLYm9w2QSr9NgQAIcRxZsNpA7UJR0e71JL+VU+ISWFk5I97lra8uGg7GlQYhGd4Gc6rxsLFRiIeGO4abP4S4ekQ1fiqDCy87GZHd52fn5aaDGuvOmIofrzpVwMvtbreZ/855OaXTRcNiNE0wzGZSxbjg26v8ko8L537v/XCCWP2MFaArJpvnkep0pA+O86MWjRAZPQRfznZiSIaTppy6m3p6HrNSsY7fDtz7Cl4V/DJAjQDoyiL2uwf1UHVd2AIrzBUSlJaTj4k6NL97a/GqhWKU9RUmjnYKpm2r+JYUcrkCuZKvcYvrg8pDoUKQywY9GDWg03DUFSirlUXBS5SWn/KAntnf0IdHGL/7mwXqDG+LZYjbEdQmqUqq4y54TNmWUP7IgcAw5816YBzwiNIJiE9M4lPCzeI/FGBeYy3p6IAmH4AjXXmvQ4Iy0Y82NTobcAggT2Cdqz6Mx4TdGoq9fn2etrWKUNFyatAHydQTVUQ2S5OWVUlugcNvoUrlA8cJJz9MqOa/W3iVno4zDHfE7zhoY5f5lRTVZDhrQbR8LS4eRLz8iPMyBL6o4PiLlp89FjdokQLaSBmKHUwWp0na5fE3v9zny2YcDXG/jfI9sctulHRbdkI5a4GOPJx4oAJQzVZ/yYAado8KNZUdEFs9ZPiBsausotXMNebEgr0dyopuqfScFJ3ODNPHgclACPdccwv0YJGQdsN2lhoV4HVGBxcEUeUX/alr4nqpcc1CCR3vR7g40zteQg/JvWmFlUE4mAiTpHlYGrB7w+U2KdSwQz2QJKBe/5eiixWipmfP15AFWrK8Sh1GBBYLgzki1wTMhGQmagXqJ2+FuqJ8f0XzXCVJFHQdMAw8xco11HhM347alrAu+wmX3pDFABOvkC+WPX0Uhg1Z5MVHKNROxaR84YV3s12UcM+70cJ460SzEaKLyh472vOMD3XnaK7zxZcXlWqenEvcjmgGNR2OKbI1s8U+iwiW+HotHalp3e1MGDy6BMVIvajnAzkFHbeVsgjmJUkrP9OAwnEHYXVBqYx3q7LvXjoVR0mY8h+ZaOnh053pdsGkmbqhyryN01eVHySr+CkDYkSMeZ1xjPNVM+gVLTDKu2VGsMUJqWO4TwPDP0VOg2/8ITbAUaMGb4LjL7L+Pi11lEVMXTYIlAZ/QHmTENjyx3kDkBdfcvvQt6tKk6jYFM4EG5UXDTaF5+1ZjRz6W7MdJPC+wTkbDUim4p5QQH3b9kGk2Bkilyeur8Bc20wm5uJSBO95GfYDI1EZipoRaH7uVveneqz43tlTZGRQ4a7CNmMHgXyOQQOL6WQkgMUTQDT8vh21aSdz7ERiZT1jK9F+v6wgFvuEmGngSvIUR2CJkc5tx1QygfZnAruONobB1idCLB1FCfO7N1ZdRocT8/Wye+EnDiO9pzqIpnLDl4bkaRKW+ekBVwHn46Shw1X0tclt/0ROijuUB4kIInrVJU4buWf4YITJtjOJ6iKdr1u+flgQeFH70GxKjhdgt/MrwfB4K/sXczQ+9zYcrD4dhY6qZhZ010rrxggWA8JaZyg2pYij8ieYEg1aZJkZK9O1Re7sB0iouf60rK0Gd+AYlp7soqCBCDGwfKeUQhCBn0E0o0GS6PdmjLi0TtCYZeqazqwN+yNINIA8Lk3iPDnWUiIPLGNcHmZDxfeK0iAdxm/T7LnN+gemRL61hHIc0NCAZaiYJR+OHnLWSe8sLrK905B5eEJHNlWq4RmEXIaFTmo49f8w61+NwfEUyuJAwVqZCLFcyHBKAcIVj3sNzfEOXzVKIndxHw+AR93owhbCxUZf6Gs8cz6/1VdrFEPrv330+9s6BtMVPJ3zl/Uf9rUi0Z/opexfdL3ykF76e999GPfVv8fJv/Y/+/5hEMon1tqNFyVRevV9y9/uIvsG3dbB8GRRrgaEXfhx+2xeOFt+cEn3RZanNxdEe2+B6MHpNbrRE53PlDifPvFcp4kO78ILR0T4xyW/WGPyBsqGdoA7zJJCu1TKbGfhnqgnRbxbB2B3UZoeQ2bz2sTVnUwokTcTU21RxN1PYPS3Sar7T0eRIsyCNowr9amwoMU/od9s2APtiKNL6ENOlyKADstAEWKA+sdKDhrJ6BOhRJmZ+QJbAaZ3/5Fq0/lumCgEzGEbu3yi0Y4I4EgVAjqxh4HbuQn0GrRhOWyAfsglQJAVL1y/6yezS2k8RE2MstJLh92NOB3GCYgFXznF4d25qiP4ZCyI4RYGesut6FXK6GwPpKK8WHEkhYui0AyEmr5Ml3uBFtPFdnioI8RiCooa7Z1G1WuyIi3nSNglutc+xY8BkeW3JJXPK6jd2VIMpaSxpVtFq+R+ySK9J6WG5Qvt+C+QH1hyYUOVK7857nFmyDBYgZ/o+AnibzNVqyYCJQvyDXDTK+iXdkA71bY7TL3bvuLxLBQ8kbTvTEY9aqkQ3+MiLWbEgjLzOH+lXgco1ERgzd80rDCymlpaRQbOYnKG/ODoFl46lzT0cjM5FYVvv0qLUbD5lyJtMUaC1pFlTkNONx6lliaX9o0i/1vws5bNKn5OuENQEKmLlcP4o2ZmJjD4zzd3Fk32uQ4uRWkPSUqb4LBe3EXHdORNB2BWsws5daRnMfNVX7isPSb1hMQdAJi1/qmDMfRUlCU74pmnzjbXfL8PVG8NsW6IQM2Ne23iCPIpryJjYbVnm5hCvKpMa7HLViNiNc+xTfDIaKm3jctViD8A1M9YPJNk003VVr4Zo2MuGW8vil8SLaGpPXqG7I4DLdtl8a4Rbx1Lt4w5Huqaa1XzZBtj208EJVGcmKYEuaeN27zT9EE6a09JerXdEbpaNgNqYJdhP1NdqiPKsbDRUi86XvvNC7rME5mrSQtrzAZVndtSjCMqd8BmaeGR4l4YFULGRBeXIV9Y4yxLFdyoUNpiy2IhePSWzBofYPP0eIa2q5JP4j9G8at/AqoSsLAUuRXtvgsqX/zYwsE+of6oSDbUOo4RMJw+DOUTJq+hnqwKim9Yy/napyZNTc2rCq6V9jHtJbxGPDwlzWj/Sk3zF/BHOlT/fSjSq7FqlPI1q6J+ru8Aku008SFINXZfOfnZNOvGPMtEmn2gLPt+H4QLA+/SYe4j398auzhKIp2Pok3mPC5q1IN1HgR+mnEfc4NeeHYwd2/kpszR3cBn7ni9NbIqhtSWFW8xbUJuUPVOeeXu3j0IGZmFNiwaNZ6rH4/zQ2ODz6tFxRLsUYZu1bfd1uIvfQDt4YD/efKYv8VF8bHGDgK22w2Wqwpi43vNCOXFJZCGMqWiPbL8mil6tsmOTXAWCyMCw73e2rADZj2IK6rqksM3EXF2cbLb4vjB14wa/yXK5vwU+05MzERJ5nXsXsW21o7M+gO0js2OyKciP5uF2iXyb2DiptwQeHeqygkrNsqVCSlldxBMpwHi1vfc8RKpP/4L3Lmpq6DZcvhDDfxTCE3splacTcOtXdK2g303dIWBVe2wD/Gvja1cClFQ67gw0t1ZUttsUgQ1Veky8oOpS6ksYEc4bqseCbZy766SvL3FodmnahlWJRgVCNjPxhL/fk2wyvlKhITH/VQCipOI0dNcRa5B1M5HmOBjTLeZQJy237e2mobwmDyJNHePhdDmiknvLKaDbShL+Is1XTCJuLQd2wmdJL7+mKvs294whXQD+vtd88KKk0DXP8B1Xu9J+xo69VOuFgexgTrcvI6SyltuLix9OPuE6/iRJYoBMEXxU4shQMf4Fjqwf1PtnJ/wWSZd29rhZjRmTGgiGTAUQqRz+nCdjeMfYhsBD5Lv60KILWEvNEHfmsDs2L0A252351eUoYxAysVaCJVLdH9QFWAmqJDCODUcdoo12+gd6bW2boY0pBVHWL6LQDK5bYWh1V8vFvi0cRpfwv7cJiMX3AZNJuTddHehTIdU0YQ/sQ1dLoF2xQPcCuHKiuCWOY30DHe1OwcClLAhqAKyqlnIbH/8u9ScJpcS4kgp6HKDUdiOgRaRGSiUCRBjzI5gSksMZKqy7Sd51aeg0tgJ+x0TH9YH2Mgsap9N7ENZdEB0bey2DMTrBA1hn56SErNHf3tKtqyL9b6yXEP97/rc+jgD2N1LNUH6RM9AzP3kSipr06RkKOolR7HO768jjWiH1X92jA7dkg7gcNcjqsZCgfqWw0tPXdLg20cF6vnQypg7gLtkazrHAodyYfENPQZsdfnjMZiNu4nJO97D1/sQE+3vNFzrSDOKw+keLECYf7RJwVHeP/j79833oZ0egonYB2FlFE5qj02B/LVOMJQlsB8uNg3Leg4qtZwntsOSNidR0abbZmAK4sCzvt8Yiuz2yrNCJoH5O8XvX/vLeR/BBYTWj0sOPYM/jyxRd5+/JziKAABaPcw/34UA3aj/gLZxZgRCWN6m4m3demanNgsx0P237/Q+Ew5VYnJPkyCY0cIVHoFn2Ay/e7U4P19APbPFXEHX94N6KhEMPG7iwB3+I+O1jd5n6VSgHegxgaSawO6iQCYFgDsPSMsNOcUj4q3sF6KzGaH/0u5PQoAj/8zq6Uc9MoNrGqhYeb2jQo0WlGlXjxtanZLS24/OIN5Gx/2g684BPDQpwlqnkFcxpmP/osnOXrFuu4PqifouQH0eF5qCkvITQbJw/Zvy5mAHWC9oU+cTiYhJmSfKsCyt1cGVxisKu+NymEQIAyaCgud/V09qT3nk/9s/SWsYtha7yNpzBIMM40rCSGaJ9u6lEkl00vXBiEt7p9P5IBCiavynEOv7FgLqPdeqxRiCwuFVMolSIUBcoyfUC2e2FJSAUgYdVGFf0b0Kn2EZlK97yyxrT2MVgvtRikfdaAW8RwEEfN+B7/eK8bBdp7URpbqn1xcrC6d2UjdsKbzCjBFqkKkoZt7Mrhg6YagE7spkqj0jOrWM+UGQ0MUlG2evP1uE1p2xSv4dMK0dna6ENcNUF+xkaJ7B764NdxLCpuvhblltVRAf7vK5qPttJ/9RYFUUSGcLdibnz6mf7WkPO3MkUUhR2mAOuGv8IWw5XG1ZvoVMnjSAZe6T7WYA99GENxoHkMiKxHlCuK5Gd0INrISImHQrQmv6F4mqU/TTQ8nHMDzCRivKySQ8dqkpQgnUMnwIkaAuc6/FGq1hw3b2Sba398BhUwUZSAIO8XZvnuLdY2n6hOXws+gq9BHUKcKFA6kz6FDnpxLPICa3qGhnc97bo1FT/XJk48LrkHJ2CAtBv0RtN97N21plfpXHvZ8gMJb7Zc4cfI6MbPwsW7AilCSXMFIEUEmir8XLEklA0ztYbGpTTGqttp5hpFTTIqUyaAIqvMT9A/x+Ji5ejA4Bhxb/cl1pUdOD6epd3yilIdO6j297xInoiBPuEDW2/UfslDyhGkQs7Wy253bVnlT+SWg89zYIK/9KXFl5fe+jow2rd5FXv8zDPrmfMXiUPt9QBO/iK4QGbX5j/7Rx1c1vzsY8ONbP3lVIaPrhL4+1QrECTN3nyKavGG0gBBtHvTKhGoBHgMXHStFowN+HKrPriYu+OZ05Frn8okQrPaaxoKP1ULCS/cmKFN3gcH7HQlVjraCeQmtjg1pSQxeuqXiSKgLpxc/1OiZsU4+n4lz4hpahGyWBURLi4642n1gn9qz9bIsaCeEPJ0uJmenMWp2tJmIwLQ6VSgDYErOeBCfSj9P4G/vI7oIF+l/n5fp956QgxGvur77ynawAu3G9MdFbJbu49NZnWnnFcQHjxRuhUYvg1U/e84N4JTecciDAKb/KYIFXzloyuE1eYXf54MmhjTq7B/yBToDzzpx3tJCTo3HCmVPYfmtBRe3mPYEE/6RlTIxbf4fSOcaKFGk4gbaUWe44hVk9SZzhW80yfW5QWBHxmtUzvMhfVQli4gZTktIOZd9mjJ5hsbmzttaHQB29Am3dZkmx3g/qvYocyhZ2PXAWsNQiIaf+Q8W/MWPIK7/TjvCx5q2XRp4lVWydMc2wIQkhadDB0xsnw/kSEyGjLKjI4coVIwtubTF3E7MJ6LS6UOsJKj82XVAVPJJcepfewbzE91ivXZvOvYfsmMevwtPpfMzGmC7WJlyW2j0jh7AF1JLmwEJSKYwIvu6DHc3YnyLH9ZdIBnQ+nOVDRiP+REpqv++typYHIvoJyICGA40d8bR7HR2k7do6UQTHF4oriYeIQbxKe4Th6+/l1BjUtS9hqORh3MbgvYrStXTfSwaBOmAVQZzpYNqsAmQyjY56MUqty3c/xH6GuhNvNaG9vGbG6cPtBM8UA3e8r51D0AR9kozKuGGSMgLz3nAHxDNnc7GTwpLj7/6HeWp1iksDeTjwCLpxejuMtpMnGJgsiku1sOACwQ9ukzESiDRN77YNESxR5LphOlcASXA5uIts1LnBIcn1J7BLWs49DMALSnuz95gdOrTZr0u1SeYHinno/pE58xYoXbVO/S+FEMMs5qyWkMnp8Q3ClyTlZP52Y9nq7b8fITPuVXUk9ohG5EFHw4gAEcjFxfKb3xuAsEjx2z1wxNbSZMcgS9GKyW3R6KwJONgtA64LTyxWm8Bvudp0M1FdJPEGopM4Fvg7G/hsptkhCfHFegv4ENwxPeXmYhxwZy7js+BeM27t9ODBMynVCLJ7RWcBMteZJtvjOYHb5lOnCLYWNEMKC59BA7covu1cANa2PXL05iGdufOzkgFqqHBOrgQVUmLEc+Mkz4Rq8O6WkNr7atNkH4M8d+SD1t/tSzt3oFql+neVs+AwEI5JaBJaxARtY2Z4mKoUqxds4UpZ0sv3zIbNoo0J4fihldQTX3XNcuNcZmcrB5LTWMdzeRuAtBk3cZHYQF6gTi3PNuDJ0nmR+4LPLoHvxQIxRgJ9iNNXqf2SYJhcvCtJiVWo85TsyFOuq7EyBPJrAdhEgE0cTq16FQXhYPJFqSfiVn0IQnPOy0LbU4BeG94QjdYNB0CiQ3QaxQqD2ebSMiNjaVaw8WaM4Z5WnzcVDsr4eGweSLa2DE3BWViaxhZFIcSTjgxNCAfelg+hznVOYoe5VqTYs1g7WtfTm3e4/WduC6p+qqAM8H4ZyrJCGpewThTDPe6H7CzX/zQ8Tm+r65HeZn+MsmxUciEWPlAVaK/VBaQBWfoG/aRL/jSZIQfep/89GjasWmbaWzeEZ2R1FOjvyJT37O9B8046SRSKVEnXWlBqbkb5XCS3qFeuE9xb9+frEknxWB5h1D/hruz2iVDEAS7+qkEz5Ot5agHJc7WCdY94Ws61sURcX5nG8UELGBAHZ3i+3VulAyT0nKNNz4K2LBHBWJcTBX1wzf+//u/j/9+//v87+9/l9Lbh/L/uyNYiTsWV2LwsjaA6MxTuzFMqmxW8Jw/+IppdX8t/Clgi1rI1SN0UC/r6tX/4lUc2VV1OQReSeCsjUpKZchw4XUcjHfw6ryCV3R8s6VXm67vp4n+lcPV9gJwmbKQEsmrJi9c2vkwrm8HFbVYNTaRGq8D91t9n5+U+aD/hNtN3HjC/nC/vUoGFSCkXP+NlRcmLUqLbiUBl4LYf1U/CCvwtd3ryCH8gUmGITAxiH1O5rnGTz7y1LuFjmnFGQ1UWuM7HwfXtWl2fPFKklYwNUpF2IL/TmaRETjQiM5SJacI+3Gv5MBU8lP5Io6gWkawpyzNEVGqOdx4YlO1dCvjbWFZWbCmeiFKPSlMKtKcMFLs/KQxtgAHi7NZNCQ32bBAW2mbHflVZ8wXKi1JKVHkW20bnYnl3dKWJeWJOiX3oKPBD6Zbi0ZvSIuWktUHB8qDR8DMMh1ZfkBL9FS9x5r0hBGLJ8pUCJv3NYH+Ae8p40mZWd5m5fhobFjQeQvqTT4VKWIYfRL0tfaXKiVl75hHReuTJEcqVlug+eOIIc4bdIydtn2K0iNZPsYWQvQio2qbO3OqAlPHDDOB7DfjGEfVF51FqqNacd6QmgFKJpMfLp5DHTv4wXlONKVXF9zTJpDV4m1sYZqJPhotcsliZM8yksKkCkzpiXt+EcRQvSQqmBS9WdWkxMTJXPSw94jqI3varCjQxTazjlMH8jTS8ilaW8014/vwA/LNa+YiFoyyx3s/KswP3O8QW1jtq45yTM/DX9a8M4voTVaO2ebvw1EooDw/yg6Y1faY+WwrdVs5Yt0hQ5EwRfYXSFxray1YvSM+kYmlpLG2/9mm1MfmbKHXr44Ih8nVKb1M537ZANUkCtdsPZ80JVKVKabVHCadaLXg+IV8i5GSwpZti0h6diTaKs9sdpUKEpd7jDUpYmHtiX33SKiO3tuydkaxA7pEc9XIQEOfWJlszj5YpL5bKeQyT7aZSBOamvSHl8xsWvgo26IP/bqk+0EJUz+gkkcvlUlyPp2kdKFtt7y5aCdks9ZJJcFp5ZWeaWKgtnXMN3ORwGLBE0PtkEIek5FY2aVssUZHtsWIvnljMVJtuVIjpZup/5VL1yPOHWWHkOMc6YySWMckczD5jUj2mlLVquFaMU8leGVaqeXis+aRRL8zm4WuBk6cyWfGMxgtr8useQEx7k/PvRoZyd9nde1GUCV84gMX8Ogu/BWezYPSR27llzQnA97oo0pYyxobYUJfsj+ysTm9zJ+S4pk0TGo9VTG0KjqYhTmALfoDZVKla2b5yhv241PxFaLJs3i05K0AAIdcGxCJZmT3ZdT7CliR7q+kur7WdQjygYtOWRL9B8E4s4LI8KpAj7bE0dg7DLOaX+MGeAi0hMMSSWZEz+RudXbZCsGYS0QqiXjH9XQbd8sCB+nIVTq7/T/FDS+zWY9q7Z2fdq1tdLb6v3hKKVDAw5gjj6o9r1wHFROdHc18MJp4SJ2Ucvu+iQ9EgkekW8VCM+psM6y+/2SBy8tNN4a3L1MzP+OLsyvESo5gS7IQOnIqMmviJBVc6zbVG1n8eXiA3j46kmvvtJlewwNDrxk4SbJOtP/TV/lIVK9ueShNbbMHfwnLTLLhbZuO79ec5XvfgRwLFK+w1r5ZWW15rVFZrE+wKqNRv5KqsLNfpGgnoUU6Y71NxEmN7MyqwqAQqoIULOw/LbuUB2+uE75gJt+kq1qY4LoxV+qR/zalupea3D5+WMeaRIn0sAI6DDWDh158fqUb4YhAxhREbUN0qyyJYkBU4V2KARXDT65gW3gRsiv7xSPYEKLwzgriWcWgPr0sbZnv7m1XHNFW6xPdGNZUdxFiUYlmXNjDVWuu7LCkX/nVkrXaJhiYktBISC2xgBXQnNEP+cptWl1eG62a7CPXrnrkTQ5BQASbEqUZWMDiZUisKyHDeLFOaJILUo5f6iDt4ZO8MlqaKLto0AmTHVVbkGuyPa1R/ywZsWRoRDoRdNMMHwYTsklMVnlAd2S0282bgMI8fiJpDh69OSL6K3qbo20KfpNMurnYGQSr/stFqZ7hYsxKlLnKAKhsmB8AIpEQ4bd/NrTLTXefsE6ChRmKWjXKVgpGoPs8GAicgKVw4K0qgDgy1A6hFq1WRat3fHF+FkU+b6H4NWpOU3KXTxrIb2qSHAb+qhm8hiSROi/9ofapjxhyKxxntPpge6KL5Z4+WBMYkAcE6+0Hd3Yh2zBsK2MV3iW0Y6cvOCroXlRb2MMJtdWx+3dkFzGh2Pe3DZ9QpSqpaR/rE1ImOrHqYYyccpiLC22amJIjRWVAherTfpQLmo6/K2pna85GrDuQPlH1Tsar8isAJbXLafSwOof4gg9RkAGm/oYpBQQiPUoyDk2BCQ1k+KILq48ErFo4WSRhHLq/y7mgw3+L85PpP6xWr6cgp9sOjYjKagOrxF148uhuaWtjet953fh1IQiEzgC+d2IgBCcUZqgTAICm2bR8oCjDLBsmg+ThyhfD+zBalsKBY1Ce54Y/t9cwfbLu9SFwEgphfopNA3yNxgyDafUM3mYTovZNgPGdd4ZFFOj1vtfFW3u7N+iHEN1HkeesDMXKPyoCDCGVMo4GCCD6PBhQ3dRZIHy0Y/3MaE5zU9mTCrwwnZojtE+qNpMSkJSpmGe0EzLyFelMJqhfFQ7a50uXxZ8pCc2wxtAKWgHoeamR2O7R+bq7IbPYItO0esdRgoTaY38hZLJ5y02oIVwoPokGIzxAMDuanQ1vn2WDQ00Rh6o5QOaCRu99fwDbQcN0XAuqkFpxT/cfz3slGRVokrNU0iqiMAJFEbKScZdmSkTUznC0U+MfwFOGdLgsewRyPKwBZYSmy6U325iUhBQNxbAC3FLKDV9VSOuQpOOukJ/GAmu/tyEbX9DgEp6dv1zoU0IqzpG6gssSjIYRVPGgU1QAQYRgIT8gEV0EXr1sqeh2I6rXjtmoCYyEDCe/PkFEi/Q48FuT29p557iN+LCwk5CK/CZ2WdAdfQZh2Z9QGrzPLSNRj5igUWzl9Vi0rCqH8G1Kp4QMLkuwMCAypdviDXyOIk0AHTM8HBYKh3b0/F+DxoNj4ZdoZfCpQVdnZarqoMaHWnMLNVcyevytGsrXQEoIbubqWYNo7NRHzdc0zvT21fWVirj7g36iy6pxogfvgHp1xH1Turbz8QyyHnXeBJicpYUctbzApwzZ1HT+FPEXMAgUZetgeGMwt4G+DHiDT2Lu+PT21fjJCAfV16a/Wu1PqOkUHSTKYhWW6PhhHUlNtWzFnA7MbY+r64vkwdpfNB2JfWgWXAvkzd42K4lN9x7Wrg4kIKgXCb4mcW595MCPJ/cTfPAMQMFWwnqwde4w8HZYJFpQwcSMhjVz4B8p6ncSCN1X4klxoIH4BN2J6taBMj6lHkAOs8JJAmXq5xsQtrPIPIIp/HG6i21xMGcFgqDXSRF0xQg14d2uy6HgKE13LSvQe52oShF5Jx1R6avyL4thhXQZHfC94oZzuPUBKFYf1VvDaxIrtV6dNGSx7DO0i1p6CzBkuAmEqyWceQY7F9+U0ObYDzoa1iKao/cOD/v6Q9gHrrr1uCeOk8fST9MG23Ul0KmM3r+Wn6Hi6WAcL7gEeaykicvgjzkjSwFsAXIR81Zx4QJ6oosVyJkCcT+4xAldCcihqvTf94HHUPXYp3REIaR4dhpQF6+FK1H0i9i7Pvh8owu3lO4PT1iuqu+DkL2Bj9+kdfGAg2TXw03iNHyobxofLE2ibjsYDPgeEQlRMR7afXbSGQcnPjI2D+sdtmuQ771dbASUsDndU7t58jrrNGRzISvwioAlHs5FA+cBE5Ccznkd8NMV6BR6ksnKLPZnMUawRDU1MZ/ib3xCdkTblHKu4blNiylH5n213yM0zubEie0o4JhzcfAy3H5qh2l17uLooBNLaO+gzonTH2uF8PQu9EyH+pjGsACTMy4cHzsPdymUSXYJOMP3yTkXqvO/lpvt0cX5ekDEu9PUfBeZODkFuAjXCaGdi6ew4qxJ8PmFfwmPpkgQjQlWqomFY6UkjmcnAtJG75EVR+NpzGpP1Ef5qUUbfowrC3zcSLX3BxgWEgEx/v9cP8H8u1Mvt9/rMDYf6sjwU1xSOPBgzFEeJLMRVFtKo5QHsUYT8ZRLCah27599EuqoC9PYjYO6aoAMHB8X1OHwEAYouHfHB3nyb2B+SnZxM/vw/bCtORjLMSy5aZoEpvgdGvlJfNPFUu/p7Z4VVK1hiI0/UTuB3ZPq4ohEbm7Mntgc1evEtknaosgZSwnDC2BdMmibpeg48X8Ixl+/8+xXdbshQXUPPvx8jT3fkELivHSmqbhblfNFShWAyQnJ3WBU6SMYSIpTDmHjdLVAdlADdz9gCplZw6mTiHqDwIsxbm9ErGusiVpg2w8Q3khKV/R9Oj8PFeF43hmW/nSd99nZzhyjCX3QOZkkB6BsH4H866WGyv9E0hVAzPYah2tkRfQZMmP2rinfOeQalge0ovhduBjJs9a1GBwReerceify49ctOh5/65ATYuMsAkVltmvTLBk4oHpdl6i+p8DoNj4Fb2vhdFYer2JSEilEwPd5n5zNoGBXEjreg/wh2NFnNRaIUHSOXa4eJRwygZoX6vnWnqVdCRT1ARxeFrNBJ+tsdooMwqnYhE7zIxnD8pZH+P0Nu1wWxCPTADfNWmqx626IBJJq6NeapcGeOmbtXvl0TeWG0Y7OGGV4+EHTtNBIT5Wd0Bujl7inXgZgfXTM5efD3qDTJ54O9v3Bkv+tdIRlq1kXcVD0BEMirmFxglNPt5pedb1AnxuCYMChUykwsTIWqT23XDpvTiKEru1cTcEMeniB+HQDehxPXNmkotFdwUPnilB/u4Nx5Xc6l8J9jH1EgKZUUt8t8cyoZleDBEt8oibDmJRAoMKJ5Oe9CSWS5ZMEJvacsGVdXDWjp/Ype5x0p9PXB2PAwt2LRD3d+ftNgpuyvxlP8pB84oB1i73vAVpwyrmXW72hfW6Dzn9Jkj4++0VQ4d0KSx1AsDA4OtXXDo63/w+GD+zC7w5SJaxsmnlYRQ4dgdjA7tTl2KNLnpJ+mvkoDxtt1a4oPaX3EVqj96o9sRKBQqU7ZOiupeAIyLMD+Y3YwHx30XWHB5CQiw7q3mj1EDlP2eBsZbz79ayUMbyHQ7s8gu4Lgip1LiGJj7NQj905/+rgUYKAA5qdrlHKIknWmqfuR+PB8RdBkDg/NgnlT89G72h2NvySnj7UyBwD+mi/IWs1xWbxuVwUIVXun5cMqBtFbrccI+DILjsVQg6eeq0itiRfedn89CvyFtpkxaauEvSANuZmB1p8FGPbU94J9medwsZ9HkUYjmI7OH5HuxendLbxTaYrPuIfE2ffXFKhoNBUp33HsFAXmCV/Vxpq5AYgFoRr5Ay93ZLRlgaIPjhZjXZZChT+aE5iWAXMX0oSFQEtwjiuhQQItTQX5IYrKfKB+queTNplR1Hoflo5/I6aPPmACwQCE2jTOYo5Dz1cs7Sod0KTG/3kEDGk3kUaUCON19xSJCab3kNpWZhSWkO8l+SpW70Wn3g0ciOIJO5JXma6dbos6jyisuxXwUUhj2+1uGhcvuliKtWwsUTw4gi1c/diEEpZHoKoxTBeMDmhPhKTx7TXWRakV8imJR355DcIHkR9IREHxohP4TbyR5LtFU24umRPRmEYHbpe1LghyxPx7YgUHjNbbQFRQhh4KeU1EabXx8FS3JAxp2rwRDoeWkJgWRUSKw6gGP5U2PuO9V4ZuiKXGGzFQuRuf+tkSSsbBtRJKhCi3ENuLlXhPbjTKD4djXVnfXFds6Zb+1XiUrRfyayGxJq1+SYBEfbKlgjiSmk0orgTqzSS+DZ5rTqsJbttiNtp+KMqGE2AHGFw6jQqM5vD6vMptmXV9OAjq49Uf/Lx9Opam+Hn5O9p8qoBBAQixzQZ4eNVkO9sPzJAMyR1y4/RCQQ1s0pV5KAU5sKLw3tkcFbI/JqrjCsK4Mw+W8aod4lioYuawUiCyVWBE/qPaFi5bnkgpfu/ae47174rI1fqQoTbW0HrU6FAejq7ByM0V4zkZTg02/YJK2N7hUQRCeZ4BIgSEqgD8XsjzG6LIsSbuHoIdz/LhFzbNn1clci1NHWJ0/6/O8HJMdIpEZbqi1RrrFfoo/rI/7ufm2MPG5lUI0IYJ4MAiHRTSOFJ2oTverFHYXThkYFIoyFx6rMYFgaOKM4xNWdlOnIcKb/suptptgTOTdVIf4YgdaAjJnIAm4qNNHNQqqAzvi53GkyRCEoseUBrHohZsjUbkR8gfKtc/+Oa72lwxJ8Mq6HDfDATbfbJhzeIuFQJSiw1uZprHlzUf90WgqG76zO0eCB1WdPv1IT6sNxxh91GEL2YpgC97ikFHyoaH92ndwduqZ6IYjkg20DX33MWdoZk7QkcKUCgisIYslOaaLyvIIqRKWQj16jE1DlQWJJaPopWTJjXfixEjRJJo8g4++wuQjbq+WVYjsqCuNIQW3YjnxKe2M5ZKEqq+cX7ZVgnkbsU3RWIyXA1rxv4kGersYJjD//auldXGmcEbcfTeF16Y1708FB1HIfmWv6dSFi6oD4E+RIjCsEZ+kY7dKnwReJJw3xCjKvi3kGN42rvyhUlIz0Bp+fNSV5xwFiuBzG296e5s/oHoFtUyUplmPulIPl+e1CQIQVtjlzLzzzbV+D/OVQtYzo5ixtMi5BmHuG4N/uKfJk5UIREp7+12oZlKtPBomXSzAY0KgtbPzzZoHQxujnREUgBU+O/jKKhgxVhRPtbqyHiUaRwRpHv7pgRPyUrnE7fYkVblGmfTY28tFCvlILC04Tz3ivkNWVazA+OsYrxvRM/hiNn8Fc4bQBeUZABGx5S/xFf9Lbbmk298X7iFg2yeimvsQqqJ+hYbt6uq+Zf9jC+Jcwiccd61NKQtFvGWrgJiHB5lwi6fR8KzYS7EaEHf/ka9EC7H8D+WEa3TEACHBkNSj/cXxFeq4RllC+fUFm2xtstYLL2nos1DfzsC9vqDDdRVcPA3Ho95aEQHvExVThXPqym65llkKlfRXbPTRiDepdylHjmV9YTWAEjlD9DdQnCem7Aj/ml58On366392214B5zrmQz/9ySG2mFqEwjq5sFl5tYJPw5hNz8lyZPUTsr5E0F2C9VMPnZckWP7+mbwp/BiN7f4kf7vtGnZF2JGvjK/sDX1RtcFY5oPQnE4lIAYV49U3C9SP0LCY/9i/WIFK9ORjzM9kG/KGrAuwFmgdEpdLaiqQNpCTGZVuAO65afkY1h33hrqyLjZy92JK3/twdj9pafFcwfXONmPQWldPlMe7jlP24Js0v9m8bIJ9TgS2IuRvE9ZVRaCwSJYOtAfL5H/YS4FfzKWKbek+GFulheyKtDNlBtrdmr+KU+ibHTdalzFUmMfxw3f36x+3cQbJLItSilW9cuvZEMjKw987jykZRlsH/UI+HlKfo2tLwemBEeBFtmxF2xmItA/dAIfQ+rXnm88dqvXa+GapOYVt/2waFimXFx3TC2MUiOi5/Ml+3rj/YU6Ihx2hXgiDXFsUeQkRAD6wF3SCPi2flk7XwKAA4zboqynuELD312EJ88lmDEVOMa1W/K/a8tGylZRMrMoILyoMQzzbDJHNZrhH77L9qSC42HVmKiZ5S0016UTp83gOhCwz9XItK9fgXfK3F5d7nZCBUekoLxrutQaPHa16Rjsa0gTrzyjqTnmcIcrxg6X6dkKiucudc0DD5W4pJPf0vuDW8r5/uw24YfMuxFRpD2ovT2mFX79xH6Jf+MVdv2TYqR6/955QgVPe3JCD/WjAYcLA9tpXgFiEjge2J5ljeI/iUzg91KQuHkII4mmHZxC3XQORLAC6G7uFn5LOmlnXkjFdoO976moNTxElS8HdxWoPAkjjocDR136m2l+f5t6xaaNgdodOvTu0rievnhNAB79WNrVs6EsPgkgfahF9gSFzzAd+rJSraw5Mllit7vUP5YxA843lUpu6/5jAR0RvH4rRXkSg3nE+O5GFyfe+L0s5r3k05FyghSFnKo4TTgs07qj4nTLqOYj6qaW9knJTDkF5OFMYbmCP+8H16Ty482OjvERV6OFyw043L9w3hoJi408sR+SGo1WviXUu8d7qS+ehKjpKwxeCthsm2LBFSFeetx0x4AaKPxtp3CxdWqCsLrB1s/j5TAhc1jNZsXWl6tjo/WDoewxzg8T8NnhZ1niUwL/nhfygLanCnRwaFGDyLw+sfZhyZ1UtYTp8TYB6dE7R3VsKKH95CUxJ8u8N+9u2/9HUNKHW3x3w5GQrfOPafk2w5qZq8MaHT0ebeY3wIsp3rN9lrpIsW9c1ws3VNV+JwNz0Lo9+V7zZr6GD56We6gWVIvtmam5GPPkVAbr74r6SwhuL+TRXtW/0pgyX16VNl4/EAD50TnUPuwrW6OcUO2VlWXS0inq872kk7GUlW6o/ozFKq+Sip6LcTtSDfDrPTcCHhx75H8BeRon+KG2wRwzfDgWhALmiWOMO6h3pm1UCZEPEjScyk7tdLx6WrdA2N1QTPENvNnhCQjW6kl057/qv7IwRryHrZBCwVSbLLnFRiHdTwk8mlYixFt1slEcPD7FVht13HyqVeyD55HOXrh2ElAxJyinGeoFzwKA91zfrdLvDxJSjzmImfvTisreI25EDcVfGsmxLVbfU8PGe/7NmWWKjXcdTJ11jAlVIY/Bv/mcxg/Q10vCHwKG1GW/XbJq5nxDhyLqiorn7Wd7VEVL8UgVzpHMjQ+Z8DUgSukiVwWAKkeTlVVeZ7t1DGnCgJVIdBPZAEK5f8CDyDNo7tK4/5DBjdD5MPV86TaEhGsLVFPQSI68KlBYy84FievdU9gWh6XZrugvtCZmi9vfd6db6V7FmoEcRHnG36VZH8N4aZaldq9zZawt1uBFgxYYx+Gs/qW1jwANeFy+LCoymyM6zgG7j8bGzUyLhvrbJkTYAEdICEb4kMKusKT9V3eIwMLsjdUdgijMc+7iKrr+TxrVWG0U+W95SGrxnxGrE4eaJFfgvAjUM4SAy8UaRwE9j6ZQH5qYAWGtXByvDiLSDfOD0yFA3UCMKSyQ30fyy1mIRg4ZcgZHLNHWl+c9SeijOvbOJxoQy7lTN2r3Y8p6ovxvUY74aOYbuVezryqXA6U+fcp6wSV9X5/OZKP18tB56Ua0gMyxJI7XyNT7IrqN8GsB9rL/kP5KMrjXxgqKLDa+V5OCH6a5hmOWemMUsea9vQl9t5Oce76PrTyTv50ExOqngE3PHPfSL//AItPdB7kGnyTRhVUUFNdJJ2z7RtktZwgmQzhBG/G7QsjZmJfCE7k75EmdIKH7xlnmDrNM/XbTT6FzldcH/rcRGxlPrv4qDScqE7JSmQABJWqRT/TUcJSwoQM+1jvDigvrjjH8oeK2in1S+/yO1j8xAws/T5u0VnIvAPqaE1atNuN0cuRliLcH2j0nTL4JpcR7w9Qya0JoaHgsOiALLCCzRkl1UUESz+ze/gIXHGtDwgYrK6pCFKJ1webSDog4zTlPkgXZqxlQDiYMjhDpwTtBW2WxthWbov9dt2X9XFLFmcF+eEc1UaQ74gqZiZsdj63pH1qcv3Vy8JYciogIVKsJ8Yy3J9w/GhjWVSQAmrS0BPOWK+RKV+0lWqXgYMnIFwpcZVD7zPSp547i9HlflB8gVnSTGmmq1ClO081OW/UH11pEQMfkEdDFzjLC1Cdo/BdL3s7cXb8J++Hzz1rhOUVZFIPehRiZ8VYu6+7Er7j5PSZu9g/GBdmNzJmyCD9wiswj9BZw+T3iBrg81re36ihMLjoVLoWc+62a1U/7qVX5CpvTVF7rocSAKwv4cBVqZm7lLDS/qoXs4fMs/VQi6BtVbNA3uSzKpQfjH1o3x4LrvkOn40zhm6hjduDglzJUwA0POabgdXIndp9fzhOo23Pe+Rk9GSLX0d71Poqry8NQDTzNlsa+JTNG9+UrEf+ngxCjGEsDCc0bz+udVRyHQI1jmEO3S+IOQycEq7XwB6z3wfMfa73m8PVRp+iOgtZfeSBl01xn03vMaQJkyj7vnhGCklsCWVRUl4y+5oNUzQ63B2dbjDF3vikd/3RUMifPYnX5Glfuk2FsV/7RqjI9yKTbE8wJY+74p7qXO8+dIYgjtLD/N8TJtRh04N9tXJA4H59IkMmLElgvr0Q5OCeVfdAt+5hkh4pQgfRMHpL74XatLQpPiOyHRs/OdmHtBf8nOZcxVKzdGclIN16lE7kJ+pVMjspOI+5+TqLRO6m0ZpNXJoZRv9MPDRcAfJUtNZHyig/s2wwReakFgPPJwCQmu1I30/tcBbji+Na53i1W1N+BqoY7Zxo+U/M9XyJ4Ok2SSkBtoOrwuhAY3a03Eu6l8wFdIG1cN+e8hopTkiKF093KuH/BcB39rMiGDLn6XVhGKEaaT/vqb/lufuAdpGExevF1+J9itkFhCfymWr9vGb3BTK4j598zRH7+e+MU9maruZqb0pkGxRDRE1CD4Z8LV4vhgPidk5w2Bq816g3nHw1//j3JStz7NR9HIWELO8TMn3QrP/zZp//+Dv9p429/ogv+GATR+n/UdF+ns9xNkXZQJXY4t9jMkJNUFygAtzndXwjss+yWH9HAnLQQfhAskdZS2l01HLWv7L7us5uTH409pqitvfSOQg/c+Zt7k879P3K9+WV68n7+3cZfuRd/dDPP/03rn+d+/nBvWfgDlt8+LzjqJ/vx3CnNOwiXhho778C96iD+1TBvRZYeP+EH81LE0vVwOOrmCLB3iKzI1x+vJEsrPH4uF0UB4TJ4X3uDfOCo3PYpYe0MF4bouh0DQ/l43fxUF7Y+dpWuvTSffB0yO2UQUETI/LwCZE3BvnevJ7c9zUlY3H58xzke6DNFDQG8n0WtDN4LAYN4nogKav1ezOfK/z+t6tsCTp+dhx4ymjWuCJk1dEUifDP+HyS4iP/Vg9B2jTo9L4NbiBuDS4nuuHW6H+JDQn2JtqRKGkEQPEYE7uzazXIkcxIAqUq1esasZBETlEZY7y7Jo+RoV/IsjY9eIMkUvr42Hc0xqtsavZvhz1OLwSxMOTuqzlhb0WbdOwBH9EYiyBjatz40bUxTHbiWxqJ0uma19qhPruvcWJlbiSSH48OLDDpaHPszvyct41ZfTu10+vjox6kOqK6v0K/gEPphEvMl/vwSv+A4Hhm36JSP9IXTyCZDm4kKsqD5ay8b1Sad/vaiyO5N/sDfEV6Z4q95E+yfjxpqBoBETW2C7xl4pIO2bDODDFurUPwE7EWC2Uplq+AHmBHvir2PSgkR12/Ry65O0aZtQPeXi9mTlF/Wj5GQ+vFkYyhXsLTjrBSP9hwk4GPqDP5rBn5/l8b0mLRAvRSzXHc293bs3s8EsdE3m2exxidWVB4joHR+S+dz5/W+v00K3TqN14CDBth8eWcsTbiwXPsygHdGid0PEdy6HHm2v/IUuV5RVapYmzGsX90mpnIdNGcOOq64Dbc5GUbYpD9M7S+6cLY//QmjxFLP5cuTFRm3vA5rkFZroFnO3bjHF35uU3s8mvL7Tp9nyTc4mymTJ5sLIp7umSnGkO23faehtz3mmTS7fbVx5rP7x3HXIjRNeq/A3xCs9JNB08c9S9BF2O3bOur0ItslFxXgRPdaapBIi4dRpKGxVz7ir69t/bc9qTxjvtOyGOfiLGDhR4fYywHv1WdOplxIV87TpLBy3Wc0QP0P9s4G7FBNOdITS/tep3o3h1TEa5XDDii7fWtqRzUEReP2fbxz7bHWWJdbIOxOUJZtItNZpTFRfj6vm9sYjRxQVO+WTdiOhdPeTJ+8YirPvoeL88l5iLYOHd3b/Imkq+1ZN1El3UikhftuteEYxf1Wujof8Pr4ICTu5ezZyZ4tHQMxlzUHLYO2VMOoNMGL/20S5i2o2obfk+8qqdR7xzbRDbgU0lnuIgz4LelQ5XS7xbLuSQtNS95v3ZUOdaUx/Qd8qxCt6xf2E62yb/HukLO6RyorV8KgYl5YNc75y+KvefrxY+lc/64y9kvWP0a0bDz/rojq+RWjO06WeruWqNFU7r3HPIcLWRql8ICZsz2Ls/qOm/CLn6++X+Qf7mGspYCrZod/lpl6Rw4xN/yuq8gqV4B6aHk1hVE1SfILxWu5gvXqbfARYQpspcxKp1F/c8XOPzkZvmoSw+vEqBLdrq1fr3wAPv5NnM9i8F+jdAuxkP5Z71c6uhK3enlnGymr7UsWZKC12qgUiG8XXGQ9mxnqz4GSIlybF9eXmbqj2sHX+a1jf0gRoONHRdRSrIq03Ty89eQ1GbV/Bk+du4+V15zls+vvERvZ4E7ZbnxWTVjDjb4o/k8jlw44pTIrUGxxuJvBeO+heuhOjpFsO6lVJ/aXnJDa/bM0Ql1cLbXE/Pbv3EZ3vj3iVrB5irjupZTzlnv677NrI9UNYNqbPgp/HZXS+lJmk87wec+7YOxTDo2aw2l3NfDr34VNlvqWJBknuK7oSlZ6/T10zuOoPZOeoIk81N+sL843WJ2Q4Z0fZ3scsqC/JV2fuhWi1jGURSKZV637lf53Xnnx16/vKEXY89aVJ0fv91jGdfG+G4+sniwHes4hS+udOr4RfhFhG/F5gUG35QaU+McuLmclb5ZWmR+sG5V6nf+PxYzlrnFGxpZaK8eqqVo0NfmAWoGfXDiT/FnUbWvzGDOTr8aktOZWg4BYvz5YH12ZbfCcGtNk+dDAZNGWvHov+PIOnY9Prjg8h/wLRrT69suaMVZ5bNuK00lSVpnqSX1NON/81FoP92rYndionwgOiA8WMf4vc8l15KqEEG4yAm2+WAN5Brfu1sq9suWYqgoajgOYt/JCk1gC8wPkK+XKCtRX6TAtgvrnuBgNRmn6I8lVDipOVB9kX6Oxkp4ZKyd1M6Gj8/v2U7k+YQBL95Kb9PQENucJb0JlW3b5tObN7m/Z1j1ev388d7o15zgXsI9CikAGAViR6lkJv7nb4Ak40M2G8TJ447kN+pvfHiOFjSUSP6PM+QfbAywKJCBaxSVxpizHseZUyUBhq59vFwrkyGoRiHbo0apweEZeSLuNiQ+HAekOnarFg00dZNXaPeoHPTRR0FmEyqYExOVaaaO8c0uFUh7U4e/UxdBmthlBDgg257Q33j1hA7HTxSeTTSuVnPZbgW1nodwmG16aKBDKxEetv7D9OjO0JhrbJTnoe+kcGoDJazFSO8/fUN9Jy/g4XK5PUkw2dgPDGpJqBfhe7GA+cjzfE/EGsMM+FV9nj9IAhrSfT/J3QE5TEIYyk5UjsI6ZZcCPr6A8FZUF4g9nnpVmjX90MLSQysIPD0nFzqwCcSJmIb5mYv2Cmk+C1MDFkZQyCBq4c/Yai9LJ6xYkGS/x2s5/frIW2vmG2Wrv0APpCdgCA9snFvfpe8uc0OwdRs4G9973PGEBnQB5qKrCQ6m6X/H7NInZ7y/1674/ZXOVp7OeuCRk8JFS516VHrnH1HkIUIlTIljjHaQtEtkJtosYul77cVwjk3gW1Ajaa6zWeyHGLlpk3VHE2VFzT2yI/EvlGUSz2H9zYE1s4nsKMtMqNyKNtL/59CpFJki5Fou6VXGm8vWATEPwrUVOLvoA8jLuwOzVBCgHB2Cr5V6OwEWtJEKokJkfc87h+sNHTvMb0KVTp5284QTPupoWvQVUwUeogZR3kBMESYo0mfukewRVPKh5+rzLQb7HKjFFIgWhj1w3yN/qCNoPI8XFiUgBNT1hCHBsAz8L7Oyt8wQWUFj92ONn/APyJFg8hzueqoJdNj57ROrFbffuS/XxrSXLTRgj5uxZjpgQYceeMc2wJrahReSKpm3QjHfqExTLAB2ipVumE8pqcZv8LYXQiPHHsgb5BMW8zM5pvQit+mQx8XGaVDcfVbLyMTlY8xcfmm/RSAT/H09UQol5gIz7rESDmnrQ4bURIB4iRXMDQwxgex1GgtDxKp2HayIkR+E/aDmCttNm2C6lytWdfOVzD6X2SpDWjQDlMRvAp1symWv4my1bPCD+E1EmGnMGWhNwmycJnDV2WrQNxO45ukEb08AAffizYKVULp15I4vbNK5DzWwCSUADfmKhfGSUqii1L2UsE8rB7mLuHuUJZOx4+WiizHBJ/hwboaBzhpNOVvgFTf5cJsHef7L1HCI9dOUUbb+YxUJWn6dYOLz+THi91kzY5dtO5c+grX7v0jEbsuoOGnoIreDIg/sFMyG+TyCLIcAWd1IZ1UNFxE8Uie13ucm40U2fcxC0u3WLvLOxwu+F7MWUsHsdtFQZ7W+nlfCASiAKyh8rnP3EyDByvtJb6Kax6/HkLzT9SyEyTMVM1zPtM0MJY14DmsWh4MgD15Ea9Hd00AdkTZ0EiG5NAGuIBzQJJ0JR0na+OB7lQA6UKxMfihIQ7GCCnVz694QvykWXTxpS2soDu+smru1UdIxSvAszBFD1c8c6ZOobA8bJiJIvuycgIXBQIXWwhyTgZDQxJTRXgEwRNAawGSXO0a1DKjdihLVNp/taE/xYhsgwe+VpKEEB4LlraQyE84gEihxCnbfoyOuJIEXy2FIYw+JjRusybKlU2g/vhTSGTydvCvXhYBdtAXtS2v7LkHtmXh/8fly1do8FI/D0f8UbzVb5h+KRhMGSAmR2mhi0YG/uj7wgxcfzCrMvdjitUIpXDX8ae2JcF/36qUWIMwN6JsjaRGNj+jEteGDcFyTUb8X/NHSucKMJp7pduxtD6KuxVlyxxwaeiC1FbGBESO84lbyrAugYxdl+2N8/6AgWpo/IeoAOcsG35IA/b3AuSyoa55L7llBLlaWlEWvuCFd8f8NfcTUgzJv6CbB+6ohWwodlk9nGWFpBAOaz5uEW5xBvmjnHFeDsb0mXwayj3mdYq5gxxNf3H3/tnCgHwjSrpSgVxLmiTtuszdRUFIsn6LiMPjL808vL1uQhDbM7aA43mISXReqjSskynIRcHCJ9qeFopJfx9tqyUoGbSwJex/0aDE3plBPGtNBYgWbdLom3+Q/bjdizR2/AS/c/dH/d3G7pyl1qDXgtOFtEqidwLqxPYtrNEveasWq3vPUUtqTeu8gpov4bdOQRI2kneFvRNMrShyVeEupK1PoLDPMSfWMIJcs267mGB8X9CehQCF0gIyhpP10mbyM7lwW1e6TGvHBV1sg/UyTghHPGRqMyaebC6pbB1WKNCQtlai1GGvmq9zUKaUzLaXsXEBYtHxmFbEZ2kJhR164LhWW2Tlp1dhsGE7ZgIWRBOx3Zcu2DxgH+G83WTPceKG0TgQKKiiNNOlWgvqNEbnrk6fVD+AqRam2OguZb0YWSTX88N+i/ELSxbaUUpPx4vJUzYg/WonSeA8xUK6u7DPHgpqWpEe6D4cXg5uK9FIYVba47V/nb+wyOtk+zG8RrS4EA0ouwa04iByRLSvoJA2FzaobbZtXnq8GdbfqEp5I2dpfpj59TCVif6+E75p665faiX8gS213RqBxTZqfHP46nF6NSenOneuT+vgbLUbdTH2/t0REFXZJOEB6DHvx6N6g9956CYrY/AYcm9gELJXYkrSi+0F0geKDZgOCIYkLU/+GOW5aGj8mvLFgtFH5+XC8hvAE3CvHRfl4ofM/Qwk4x2A+R+nyc9gNu/9Tem7XW4XRnyRymf52z09cTOdr+PG6+P/Vb4QiXlwauc5WB1z3o+IJjlbxI8MyWtSzT+k4sKVbhF3xa+vDts3NxXa87iiu+xRH9cAprnOL2h6vV54iQRXuOAj1s8nLFK8gZ70ThIQcWdF19/2xaJmT0efrkNDkWbpAQPdo92Z8+Hn/aLjbOzB9AI/k12fPs9HhUNDJ1u6ax2VxD3R6PywN7BrLJ26z6s3QoMp76qzzwetrDABKSGkfW5PwS1GvYNUbK6uRqxfyVGNyFB0E+OugMM8kKwmJmupuRWO8XkXXXQECyRVw9UyIrtCtcc4oNqXqr7AURBmKn6Khz3eBN96LwIJrAGP9mr/59uTOSx631suyT+QujDd4beUFpZ0kJEEnjlP+X/Kr2kCKhnENTg4BsMTOmMqlj2WMFLRUlVG0fzdCBgUta9odrJfpVdFomTi6ak0tFjXTcdqqvWBAzjY6hVrH9sbt3Z9gn+AVDpTcQImefbB4edirjzrsNievve4ZT4EUZWV3TxEsIW+9MT/RJoKfZZYSRGfC1CwPG/9rdMOM8qR/LUYvw5f/emUSoD7YSFuOoqchdUg2UePd1eCtFSKgxLSZ764oy4lvRCIH6bowPxZWwxNFctksLeil47pfevcBipkkBIc4ngZG+kxGZ71a72KQ7VaZ6MZOZkQJZXM6kb/Ac0/XkJx8dvyfJcWbI3zONEaEPIW8GbkYjsZcwy+eMoKrYjDmvEEixHzkCSCRPRzhOfJZuLdcbx19EL23MA8rnjTZZ787FGMnkqnpuzB5/90w1gtUSRaWcb0eta8198VEeZMUSfIhyuc4/nywFQ9uqn7jdqXh+5wwv+RK9XouNPbYdoEelNGo34KyySwigsrfCe0v/PlWPvQvQg8R0KgHO18mTVThhQrlbEQ0Kp/JxPdjHyR7E1QPw/ut0r+HDDG7BwZFm9IqEUZRpv2WpzlMkOemeLcAt5CsrzskLGaVOAxyySzZV/D2EY7ydNZMf8e8VhHcKGHAWNszf1EOq8fNstijMY4JXyATwTdncFFqcNDfDo+mWFvxJJpc4sEZtjXyBdoFcxbUmniCoKq5jydUHNjYJxMqN1KzYV62MugcELVhS3Bnd+TLLOh7dws/zSXWzxEb4Nj4aFun5x4kDWLK5TUF/yCXB/cZYvI9kPgVsG2jShtXkxfgT+xzjJofXqPEnIXIQ1lnIdmVzBOM90EXvJUW6a0nZ/7XjJGl8ToO3H/fdxnxmTNKBZxnkpXLVgLXCZywGT3YyS75w/PAH5I/jMuRspej8xZObU9kREbRA+kqjmKRFaKGWAmFQspC+QLbKPf0RaK3OXvBSWqo46p70ws/eZpu6jCtZUgQy6r4tHMPUdAgWGGUYNbuv/1a6K+MVFsd3T183+T8capSo6m0+Sh57fEeG/95dykGJBQMj09DSW2bY0mUonDy9a8trLnnL5B5LW3Nl8rJZNysO8Zb+80zXxqUGFpud3Qzwb7bf+8mq6x0TAnJU9pDQR9YQmZhlna2xuxJt0aCO/f1SU8gblOrbIyMsxTlVUW69VJPzYU2HlRXcqE2lLLxnObZuz2tT9CivfTAUYfmzJlt/lOPgsR6VN64/xQd4Jlk/RV7UKVv2Gx/AWsmTAuCWKhdwC+4HmKEKYZh2Xis4KsUR1BeObs1c13wqFRnocdmuheaTV30gvVXZcouzHKK5zwrN52jXJEuX6dGx3BCpV/++4f3hyaW/cQJLFKqasjsMuO3B3WlMq2gyYfdK1e7L2pO/tRye2mwzwZPfdUMrl5wdLqdd2Kv/wVtnpyWYhd49L6rsOV+8HXPrWH2Kup89l2tz6bf80iYSd+V4LROSOHeamvexR524q4r43rTmtFzQvArpvWfLYFZrbFspBsXNUqqenjxNNsFXatZvlIhk7teUPfK+YL32F8McTnjv0BZNppb+vshoCrtLXjIWq3EJXpVXIlG6ZNL0dh6qEm2WMwDjD3LfOfkGh1/czYc/0qhiD2ozNnH4882MVVt3JbVFkbwowNCO3KL5IoYW5wlVeGCViOuv1svZx7FbzxKzA4zGqBlRRaRWCobXaVq4yYCWbZf8eiJwt3OY+MFiSJengcFP2t0JMfzOiJ7cECvpx7neg1Rc5x+7myPJOXt2FohVRyXtD+/rDoTOyGYInJelZMjolecVHUhUNqvdZWg2J2t0jPmiLFeRD/8fOT4o+NGILb+TufCo9ceBBm3JLVn+MO2675n7qiEX/6W+188cYg3Zn5NSTjgOKfWFSAANa6raCxSoVU851oJLY11WIoYK0du0ec5E4tCnAPoKh71riTsjVIp3gKvBbEYQiNYrmH22oLQWA2AdwMnID6PX9b58dR2QKo4qag1D1Z+L/FwEKTR7osOZPWECPJIHQqPUsM5i/CH5YupVPfFA5pHUBcsesh8eO5YhyWnaVRPZn/BmdXVumZWPxMP5e28zm2uqHgFoT9CymHYNNrzrrjlXZM06HnzDxYNlI5b/QosxLmmrqDFqmogQdqk0WLkUceoAvQxHgkIyvWU69BPFr24VB6+lx75Rna6dGtrmOxDnvBojvi1/4dHjVeg8owofPe1cOnxU1ioh016s/Vudv9mhV9f35At+Sh28h1bpp8xhr09+vf47Elx3Ms6hyp6QvB3t0vnLbOhwo660cp7K0vvepabK7YJfxEWWfrC2YzJfYOjygPwfwd/1amTqa0hZ5ueebhWYVMubRTwIjj+0Oq0ohU3zfRfuL8gt59XsHdwKtxTQQ4Y2qz6gisxnm2UdlmpEkgOsZz7iEk6QOt8BuPwr+NR01LTqXmJo1C76o1N274twJvl+I069TiLpenK/miRxhyY8jvYV6W1WuSwhH9q7kuwnJMtm7IWcqs7HsnyHSqWXLSpYtZGaR1V3t0gauninFPZGtWskF65rtti48UV9uV9KM8kfDYs0pgB00S+TlzTXV6P8mxq15b9En8sz3jWSszcifZa/NuufPNnNTb031pptt0+sRSH/7UG8pzbsgtt3OG3ut7B9JzDMt2mTZuyRNIV8D54TuTrpNcHtgmMlYJeiY9XS83NYJicjRjtJSf9BZLsQv629QdDsKQhTK5CnXhpk7vMNkHzPhm0ExW/VCGApHfPyBagtZQTQmPHx7g5IXXsrQDPzIVhv2LB6Ih138iSDww1JNHrDvzUxvp73MsQBVhW8EbrReaVUcLB1R3PUXyaYG4HpJUcLVxMgDxcPkVRQpL7VTAGabDzbKcvg12t5P8TSGQkrj/gOrpnbiDHwluA73xbXts/L7u468cRWSWRtgTwlQnA47EKg0OiZDgFxAKQQUcsbGomITgeXUAAyKe03eA7Mp4gnyKQmm0LXJtEk6ddksMJCuxDmmHzmVhO+XaN2A54MIh3niw5CF7PwiXFZrnA8wOdeHLvvhdoqIDG9PDI7UnWWHq526T8y6ixJPhkuVKZnoUruOpUgOOp3iIKBjk+yi1vHo5cItHXb1PIKzGaZlRS0g5d3MV2pD8FQdGYLZ73aae/eEIUePMc4NFz8pIUfLCrrF4jVWH5gQneN3S8vANBmUXrEcKGn6hIUN95y1vpsvLwbGpzV9L0ZKTan6TDXM05236uLJcIEMKVAxKNT0K8WljuwNny3BNQRfzovA85beI9zr1AGNYnYCVkR1aGngWURUrgqR+gRrQhxW81l3CHevjvGEPzPMTxdsIfB9dfGRbZU0cg/1mcubtECX4tvaedmNAvTxCJtc2QaoUalGfENCGK7IS/O8CRpdOVca8EWCRwv2sSWE8CJPW5PCugjCXPd3h6U60cPD+bdhtXZuYB6stcoveE7Sm5MM2yvfUHXFSW7KzLmi7/EeEWL0wqcOH9MOSKjhCHHmw+JGLcYE/7SBZQCRggox0ZZTAxrlzNNXYXL5fNIjkdT4YMqVUz6p8YDt049v4OXGdg3qTrtLBUXOZf7ahPlZAY/O+7Sp0bvGSHdyQ8B1LOsplqMb9Se8VAE7gIdSZvxbRSrfl+Lk5Qaqi5QJceqjitdErcHXg/3MryljPSIAMaaloFm1cVwBJ8DNmkDqoGROSHFetrgjQ5CahuKkdH5pRPigMrgTtlFI8ufJPJSUlGgTjbBSvpRc0zypiUn6U5KZqcRoyrtzhmJ7/caeZkmVRwJQeLOG8LY6vP5ChpKhc8Js0El+n6FXqbx9ItdtLtYP92kKfaTLtCi8StLZdENJa9Ex1nOoz1kQ7qxoiZFKRyLf4O4CHRT0T/0W9F8epNKVoeyxUXhy3sQMMsJjQJEyMOjmOhMFgOmmlscV4eFi1CldU92yjwleirEKPW3bPAuEhRZV7JsKV3Lr5cETAiFuX5Nw5UlF7d2HZ96Bh0sgFIL5KGaKSoVYVlvdKpZJVP5+NZ7xDEkQhmDgsDKciazJCXJ6ZN2B3FY2f6VZyGl/t4aunGIAk/BHaS+i+SpdRfnB/OktOvyjinWNfM9Ksr6WwtCa1hCmeRI6icpFM4o8quCLsikU0tMoZI/9EqXRMpKGaWzofl4nQuVQm17d5fU5qXCQeCDqVaL9XJ9qJ08n3G3EFZS28SHEb3cdRBdtO0YcTzil3QknNKEe/smQ1fTb0XbpyNB5xAeuIlf+5KWlEY0DqJbsnzJlQxJPOVyHiKMx5Xu9FcEv1Fbg6Fhm4t+Jyy5JC1W3YO8dYLsO0PXPbxodBgttTbH3rt9Cp1lJIk2r3O1Zqu94eRbnIz2f50lWolYzuKsj4PMok4abHLO8NAC884hiXx5Fy5pWKO0bWL7uEGXaJCtznhP67SlQ4xjWIfgq6EpZ28QMtuZK7JC0RGbl9nA4XtFLug/NLMoH1pGt9IonAJqcEDLyH6TDROcbsmGPaGIxMo41IUAnQVPMPGByp4mOmh9ZQMkBAcksUK55LsZj7E5z5XuZoyWCKu6nHmDq22xI/9Z8YdxJy4kWpD16jLVrpwGLWfyOD0Wd+cBzFBxVaGv7S5k9qwh/5t/LQEXsRqI3Q9Rm3QIoaZW9GlsDaKOUyykyWuhNOprSEi0s1G4rgoiX1V743EELti+pJu5og6X0g6oTynUqlhH9k6ezyRi05NGZHz0nvp3HOJr7ebrAUFrDjbkFBObEvdQWkkUbL0pEvMU46X58vF9j9F3j6kpyetNUBItrEubW9ZvMPM4qNqLlsSBJqOH3XbNwv/cXDXNxN8iFLzUhteisYY+RlHYOuP29/Cb+L+xv+35Rv7xudnZ6ohK4cMPfCG8KI7dNmjNk/H4e84pOxn/sZHK9psfvj8ncA8qJz7O8xqbxESDivGJOZzF7o5PJLQ7g34qAWoyuA+x3btU98LT6ZyGyceIXjrqob2CAVql4VOTQPUQYvHV/g4zAuCZGvYQBtf0wmd5lilrvuEn1BXLny01B4h4SMDlYsnNpm9d7m9h578ufpef9Z4WplqWQvqo52fyUA7J24eZD5av6SyGIV9kpmHNqyvdfzcpEMw97BvknV2fq+MFHun9BT3Lsf8pbzvisWiIQvYkng+8Vxk1V+dli1u56kY50LRjaPdotvT5BwqtwyF+emo/z9J3yVUVGfKrxQtJMOAQWoQii/4dp9wgybSa5mkucmRLtEQZ/pz0tL/NVcgWAd95nEQ3Tg6tNbuyn3Iepz65L3huMUUBntllWuu4DbtOFSMSbpILV4fy6wlM0SOvi6CpLh81c1LreIvKd61uEWBcDw1lUBUW1I0Z+m/PaRlX+PQ/oxg0Ye6KUiIiTF4ADNk59Ydpt5/rkxmq9tV5Kcp/eQLUVVmBzQNVuytQCP6Ezd0G8eLxWyHpmZWJ3bAzkWTtg4lZlw42SQezEmiUPaJUuR/qklVA/87S4ArFCpALdY3QRdUw3G3XbWUp6aq9z0zUizcPa7351p9JXOZyfdZBFnqt90VzQndXB/mwf8LC9STj5kenVpNuqOQQP3mIRJj7eV21FxG8VAxKrEn3c+XfmZ800EPb9/5lIlijscUbB6da0RQaMook0zug1G0tKi/JBC4rw7/D3m4ARzAkzMcVrDcT2SyFtUdWAsFlsPDFqV3N+EjyXaoEePwroaZCiLqEzb8MW+PNE9TmTC01EzWli51PzZvUqkmyuROU+V6ik+Le/9qT6nwzUzf9tP68tYei0YaDGx6kAd7jn1cKqOCuYbiELH9zYqcc4MnRJjkeGiqaGwLImhyeKs+xKJMBlOJ05ow9gGCKZ1VpnMKoSCTbMS+X+23y042zOb5MtcY/6oBeAo1Vy89OTyhpavFP78jXCcFH0t7Gx24hMEOm2gsEfGabVpQgvFqbQKMsknFRRmuPHcZu0Su/WMFphZvB2r/EGbG72rpGGho3h+Msz0uGzJ7hNK2uqQiE1qmn0zgacKYYZBCqsxV+sjbpoVdSilW/b94n2xNb648VmNIoizqEWhBnsen+d0kbCPmRItfWqSBeOd9Wne3c6bcd6uvXOJ6WdiSsuXq0ndhqrQ4QoWUjCjYtZ0EAhnSOP1m44xkf0O7jXghrzSJWxP4a/t72jU29Vu2rvu4n7HfHkkmQOMGSS+NPeLGO5I73mC2B7+lMiBQQZRM9/9liLIfowupUFAbPBbR+lxDM6M8Ptgh1paJq5Rvs7yEuLQv/7d1oU2woFSb3FMPWQOKMuCuJ7pDDjpIclus5TeEoMBy2YdVB4fxmesaCeMNsEgTHKS5WDSGyNUOoEpcC2OFWtIRf0w27ck34/DjxRTVIcc9+kqZE6iMSiVDsiKdP/Xz5XfEhm/sBhO50p1rvJDlkyyxuJ9SPgs7YeUJBjXdeAkE+P9OQJm6SZnn1svcduI78dYmbkE2mtziPrcjVisXG78spLvbZaSFx/Rks9zP4LKn0Cdz/3JsetkT06A8f/yCgMO6Mb1Hme0JJ7b2wZz1qleqTuKBGokhPVUZ0dVu+tnQYNEY1fmkZSz6+EGZ5EzL7657mreZGR3jUfaEk458PDniBzsSmBKhDRzfXameryJv9/D5m6HIqZ0R+ouCE54Dzp4IJuuD1e4Dc5i+PpSORJfG23uVgqixAMDvchMR0nZdH5brclYwRoJRWv/rlxGRI5ffD5NPGmIDt7vDE1434pYdVZIFh89Bs94HGGJbTwrN8T6lh1HZFTOB4lWzWj6EVqxSMvC0/ljWBQ3F2kc/mO2b6tWonT2JEqEwFts8rz2h+oWNds9ceR2cb7zZvJTDppHaEhK5avWqsseWa2Dt5BBhabdWSktS80oMQrL4TvAM9b5HMmyDnO+OkkbMXfUJG7eXqTIG6lqSOEbqVR+qYdP7uWb57WEJqzyh411GAVsDinPs7KvUeXItlcMdOUWzXBH6zscymV1LLVCtc8IePojzXHF9m5b5zGwBRdzcyUJkiu938ApmAayRdJrX1PmVguWUvt2ThQ62czItTyWJMW2An/hdDfMK7SiFQlGIdAbltHz3ycoh7j9V7GxNWBpbtcSdqm4XxRwTawc3cbZ+xfSv9qQfEkDKfZTwCkqWGI/ur250ItXlMlh6vUNWEYIg9A3GzbgmbqvTN8js2YMo87CU5y6nZ4dbJLDQJj9fc7yM7tZzJDZFtqOcU8+mZjYlq4VmifI23iHb1ZoT9E+kT2dolnP1AfiOkt7PQCSykBiXy5mv637IegWSKj9IKrYZf4Lu9+I7ub+mkRdlvYzehh/jaJ9n7HUH5b2IbgeNdkY7wx1yVzxS7pbvky6+nmVUtRllEFfweUQ0/nG017WoUYSxs+j2B4FV/F62EtHlMWZXYrjGHpthnNb1x66LKZ0Qe92INWHdfR/vqp02wMS8r1G4dJqHok8KmQ7947G13a4YXbsGgHcBvRuVu1eAi4/A5+ZixmdSXM73LupB/LH7O9yxLTVXJTyBbI1S49TIROrfVCOb/czZ9pM4JsZx8kUz8dQGv7gUWKxXvTH7QM/3J2OuXXgciUhqY+cgtaOliQQVOYthBLV3xpESZT3rmfEYNZxmpBbb24CRao86prn+i9TNOh8VxRJGXJfXHATJHs1T5txgc/opYrY8XjlGQQbRcoxIBcnVsMjmU1ymmIUL4dviJXndMAJ0Yet+c7O52/p98ytlmAsGBaTAmMhimAnvp1TWNGM9BpuitGj+t810CU2UhorrjPKGtThVC8WaXw04WFnT5fTjqmPyrQ0tN3CkLsctVy2xr0ZWgiWVZ1OrlFjjxJYsOiZv2cAoOvE+7sY0I/TwWcZqMoyIKNOftwP7w++Rfg67ljfovKYa50if3fzE/8aPYVey/Nq35+nH2sLPh/fP5TsylSKGOZ4k69d2PnH43+kq++sRXHQqGArWdwhx+hpwQC6JgT2uxehYU4Zbw7oNb6/HLikPyJROGK2ouyr+vzseESp9G50T4AyFrSqOQ0rroCYP4sMDFBrHn342EyZTMlSyk47rHSq89Y9/nI3zG5lX16Z5lxphguLOcZUndL8wNcrkyjH82jqg8Bo8OYkynrxZvbFno5lUS3OPr8Ko3mX9NoRPdYOKKjD07bvgFgpZ/RF+YzkWvJ/Hs/tUbfeGzGWLxNAjfDzHHMVSDwB5SabQLsIZHiBp43FjGkaienYoDd18hu2BGwOK7U3o70K/WY/kuuKdmdrykIBUdG2mvE91L1JtTbh20mOLbk1vCAamu7utlXeGU2ooVikbU/actcgmsC1FKk2qmj3GWeIWbj4tGIxE7BLcBWUvvcnd/lYxsMV4F917fWeFB/XbINN3qGvIyTpCalz1lVewdIGqeAS/gB8Mi+sA+BqDiX3VGD2eUunTRbSY+AuDy4E3Qx3hAhwnSXX+B0zuj3eQ1miS8Vux2z/l6/BkWtjKGU72aJkOCWhGcSf3+kFkkB15vGOsQrSdFr6qTj0gBYiOlnBO41170gOWHSUoBVRU2JjwppYdhIFDfu7tIRHccSNM5KZOFDPz0TGMAjzzEpeLwTWp+kn201kU6NjbiMQJx83+LX1e1tZ10kuChJZ/XBUQ1dwaBHjTDJDqOympEk8X2M3VtVw21JksChA8w1tTefO3RJ1FMbqZ01bHHkudDB/OhLfe7P5GOHaI28ZXKTMuqo0hLWQ4HabBsGG7NbP1RiXtETz074er6w/OerJWEqjmkq2y51q1BVI+JUudnVa3ogBpzdhFE7fC7kybrAt2Z6RqDjATAUEYeYK45WMupBKQRtQlU+uNsjnzj6ZmGrezA+ASrWxQ6LMkHRXqXwNq7ftv28dUx/ZSJciDXP2SWJsWaN0FjPX9Yko6LobZ7aYW/IdUktI9apTLyHS8DyWPyuoZyxN1TK/vtfxk3HwWh6JczZC8Ftn0bIJay2g+n5wd7lm9rEsKO+svqVmi+c1j88hSCxbzrg4+HEP0Nt1/B6YW1XVm09T1CpAKjc9n18hjqsaFGdfyva1ZG0Xu3ip6N6JGpyTSqY5h4BOlpLPaOnyw45PdXTN+DtAKg7DLrLFTnWusoSBHk3s0d7YouJHq85/R09Tfc37ENXZF48eAYLnq9GLioNcwDZrC6FW6godB8JnqYUPvn0pWLfQz0lM0Yy8Mybgn84Ds3Q9bDP10bLyOV+qzxa4Rd9Dhu7cju8mMaONXK3UqmBQ9qIg7etIwEqM/kECk/Dzja4Bs1xR+Q/tCbc8IKrSGsTdJJ0vge7IG20W687uVmK6icWQ6cD3lwFzgNMGtFvO5qyJeKflGLAAcQZOrkxVwy3cWvqlGpvjmf9Qe6Ap20MPbV92DPV0OhFM4kz8Yr0ffC2zLWSQ1kqY6QdQrttR3kh1YLtQd1kCEv5hVoPIRWl5ERcUTttBIrWp6Xs5Ehh5OUUwI5aEBvuiDmUoENmnVw1FohCrbRp1A1E+XSlWVOTi7ADW+5Ohb9z1vK4qx5R5lPdGCPBJZ00mC+Ssp8VUbgpGAvXWMuWQQRbCqI6Rr2jtxZxtfP7W/8onz+yz0Gs76LaT5HX9ecyiZCB/ZR/gFtMxPsDwohoeCRtiuLxE1GM1vUEUgBv86+eehL58/P56QFGQ/MqOe/vC76L63jzmeax4exd/OKTUvkXg+fOJUHych9xt/9goJMrapSgvXrj8+8vk/N80f22Sewj6cyGqt1B6mztoeklVHHraouhvHJaG/OuBz6DHKMpFmQULU1bRWlyYE0RPXYYkUycIemN7TLtgNCJX6BqdyxDKkegO7nJK5xQ7OVYDZTMf9bVHidtk6DQX9Et+V9M7esgbsYBdEeUpsB0Xvw2kd9+rI7V+m47u+O/tq7mw7262HU1WlS9uFzsV6JxIHNmUCy0QS9e077JGRFbG65z3/dOKB/Zk+yDdKpUmdXjn/aS3N5nv4fK7bMHHmPlHd4E2+iTbV5rpzScRnxk6KARuDTJ8Q1LpK2mP8gj1EbuJ9RIyY+EWK4hCiIDBAS1Tm2IEXAFfgKPgdL9O6mAa06wjCcUAL6EsxPQWO9VNegBPm/0GgkZbDxCynxujX/92vmGcjZRMAY45puak2sFLCLSwXpEsyy5fnF0jGJBhm+fNSHKKUUfy+276A7/feLOFxxUuHRNJI2Osenxyvf8DAGObT60pfTTlhEg9u/KKkhJqm5U1/+BEcSkpFDA5XeCqxwXmPac1jcuZ3JWQ+p0NdWzb/5v1ZvF8GtMTFFEdQjpLO0bwPb0BHNWnip3liDXI2fXf05jjvfJ0NpjLCUgfTh9CMFYVFKEd4Z/OG/2C+N435mnK+9t1gvCiVcaaH7rK4+PjCvpVNiz+t2QyqH1O8x3JKZVl6Q+Lp/XK8wMjVMslOq9FdSw5FtUs/CptXH9PW+wbWHgrV17R5jTVOtGtKFu3nb80T+E0tv9QkzW3J2dbaw/8ddAKZ0pxIaEqLjlPrji3VgJ3GvdFvlqD8075woxh4fVt0JZE0KVFsAvqhe0dqN9b35jtSpnYMXkU+vZq+IAHad3IHc2s/LYrnD1anfG46IFiMIr9oNbZDWvwthqYNqOigaKd/XlLU4XHfk/PXIjPsLy/9/kAtQ+/wKH+hI/IROWj5FPvTZAT9f7j4ZXQyG4M0TujMAFXYkKvEHv1xhySekgXGGqNxWeWKlf8dDAlLuB1cb/qOD+rk7cmwt+1yKpk9cudqBanTi6zTbXRtV8qylNtjyOVKy1HTz0GW9rjt6sSjAZcT5R+KdtyYb0zyqG9pSLuCw5WBwAn7fjBjKLLoxLXMI+52L9cLwIR2B6OllJZLHJ8vDxmWdtF+QJnmt1rsHPIWY20lftk8fYePkAIg6Hgn532QoIpegMxiWgAOfe5/U44APR8Ac0NeZrVh3gEhs12W+tVSiWiUQekf/YBECUy5fdYbA08dd7VzPAP9aiVcIB9k6tY7WdJ1wNV+bHeydNtmC6G5ICtFC1ZwmJU/j8hf0I8TRVKSiz5oYIa93EpUI78X8GYIAZabx47/n8LDAAJ0nNtP1rpROprqKMBRecShca6qXuTSI3jZBLOB3Vp381B5rCGhjSvh/NSVkYp2qIdP/Bg="},13549:function(r,o,u){"use strict";u(41584);var p=u(18128),y=function(){function Reserved(r,o){void 0===o&&(o=1),this.type=r,this.count=o}var r=Reserved.prototype;return r.decode=function decode(r,o){r.pos+=this.size(null,o)},r.size=function size(r,o){var u=p.resolveLength(this.count,null,o);return this.type.size()*u},r.encode=function encode(r,o,u){return r.fill(0,this.size(o,u))},Reserved}();r.exports=y},13624:function(r,o,u){"use strict";var p=u(59754),y=u(12636).indexOf,_=p.aTypedArray;(0,p.exportTypedArrayMethod)("indexOf",(function indexOf(r){return y(_(this),r,arguments.length>1?arguments[1]:void 0)}))},13711:function(r,o,u){var p=u(32010),y=u(94578),_=u(20340),P=u(48914),k=u(7421),I=u(10447),O=u(70172),B=u(7081).CONFIGURABLE,D=O.get,R=O.enforce,N=String(String).split("String");(r.exports=function(r,o,u,I){var O,D=!!I&&!!I.unsafe,U=!!I&&!!I.enumerable,W=!!I&&!!I.noTargetGet,j=I&&void 0!==I.name?I.name:o;y(u)&&("Symbol("===String(j).slice(0,7)&&(j="["+String(j).replace(/^Symbol\(([^)]*)\)/,"$1")+"]"),(!_(u,"name")||B&&u.name!==j)&&P(u,"name",j),(O=R(u)).source||(O.source=N.join("string"==typeof j?j:""))),r!==p?(D?!W&&r[o]&&(U=!0):delete r[o],U?r[o]=u:P(r,o,u)):U?r[o]=u:k(o,u)})(Function.prototype,"toString",(function toString(){return y(this)&&D(this).source||I(this)}))},13872:function(r,o,u){var p=u(52564),y=u(51839),_=u(15366),P=u(38688)("iterator");r.exports=function(r){if(null!=r)return y(r,P)||y(r,"@@iterator")||_[p(r)]}},13945:function(r,o,u){"use strict";var p=u(5844).IteratorPrototype,y=u(10819),_=u(97841),P=u(15216),k=u(15366),returnThis=function(){return this};r.exports=function(r,o,u){var I=o+" Iterator";return r.prototype=y(p,{next:_(1,u)}),P(r,I,!1,!0),k[I]=returnThis,r}},14032:function(r,o,u){var p=u(24663),y=u(13711),_=u(52598);p||y(Object.prototype,"toString",_,{unsafe:!0})},14287:function(r,o,u){"use strict";var p;u(81755),u(8953),u(14032),u(3131),u(90868),u(65553),u(75626),u(95756),u(56912),u(58099),u(47969),u(59735),u(73663),u(29883),u(35471),u(21012),u(88997),u(97464),u(2857),u(94715),u(13624),u(91132),u(62514),u(24597),u(88042),u(4660),u(92451),u(44206),u(66288),u(13250),u(3858),u(84538),u(64793),u(74202),u(52529),r.exports=(p=u(48352),function(){if("function"==typeof ArrayBuffer){var r=p.lib.WordArray,o=r.init,u=r.init=function(r){if(r instanceof ArrayBuffer&&(r=new Uint8Array(r)),(r instanceof Int8Array||"undefined"!=typeof Uint8ClampedArray&&r instanceof Uint8ClampedArray||r instanceof Int16Array||r instanceof Uint16Array||r instanceof Int32Array||r instanceof Uint32Array||r instanceof Float32Array||r instanceof Float64Array)&&(r=new Uint8Array(r.buffer,r.byteOffset,r.byteLength)),r instanceof Uint8Array){for(var u=r.byteLength,p=[],y=0;y<u;y++)p[y>>>2]|=r[y]<<24-y%4*8;o.call(this,p,u)}else o.apply(this,arguments)};u.prototype=r}}(),p.lib.WordArray)},14598:function(r,o,u){"use strict";function _defineProperties(r,o){for(var u=0;u<o.length;u++){var p=o[u];p.enumerable=p.enumerable||!1,p.configurable=!0,"value"in p&&(p.writable=!0),Object.defineProperty(r,_toPropertyKey(p.key),p)}}function _toPropertyKey(r){var o=function _toPrimitive(r,o){if("object"!=typeof r||!r)return r;var u=r[Symbol.toPrimitive];if(void 0!==u){var p=u.call(r,o||"default");if("object"!=typeof p)return p;throw new TypeError("@@toPrimitive must return a primitive value.")}return("string"===o?String:Number)(r)}(r,"string");return"symbol"==typeof o?o:o+""}function _setPrototypeOf(r,o){return _setPrototypeOf=Object.setPrototypeOf?Object.setPrototypeOf.bind():function(r,o){return r.__proto__=o,r},_setPrototypeOf(r,o)}u(65292),u(73844),u(5597),u(39081),u(68626),u(41584),u(58028),u(81755),u(20731),u(8953),u(7585),u(24863),u(7283),u(2876),u(14032),u(61726),u(57114),u(47458),u(46467),u(72095),u(56912),u(73663),u(29883),u(35471),u(21012),u(88997),u(97464),u(2857),u(94715),u(13624),u(91132),u(62514),u(24597),u(88042),u(4660),u(92451),u(44206),u(66288),u(13250),u(3858),u(84538),u(64793),u(74202),u(52529);var p=u(32504),y=u(89029),_="function"==typeof Symbol&&"function"==typeof Symbol.for?Symbol.for("nodejs.util.inspect.custom"):null;o.Buffer=Buffer,o.SlowBuffer=function SlowBuffer(r){+r!=r&&(r=0);return Buffer.alloc(+r)},o.INSPECT_MAX_BYTES=50;var P=2147483647;function createBuffer(r){if(r>P)throw new RangeError('The value "'+r+'" is invalid for option "size"');var o=new Uint8Array(r);return Object.setPrototypeOf(o,Buffer.prototype),o}function Buffer(r,o,u){if("number"==typeof r){if("string"==typeof o)throw new TypeError('The "string" argument must be of type string. Received type number');return allocUnsafe(r)}return from(r,o,u)}function from(r,o,u){if("string"==typeof r)return function fromString(r,o){"string"==typeof o&&""!==o||(o="utf8");if(!Buffer.isEncoding(o))throw new TypeError("Unknown encoding: "+o);var u=0|byteLength(r,o),p=createBuffer(u),y=p.write(r,o);y!==u&&(p=p.slice(0,y));return p}(r,o);if(ArrayBuffer.isView(r))return function fromArrayView(r){if(isInstance(r,Uint8Array)){var o=new Uint8Array(r);return fromArrayBuffer(o.buffer,o.byteOffset,o.byteLength)}return fromArrayLike(r)}(r);if(null==r)throw new TypeError("The first argument must be one of type string, Buffer, ArrayBuffer, Array, or Array-like Object. Received type "+typeof r);if(isInstance(r,ArrayBuffer)||r&&isInstance(r.buffer,ArrayBuffer))return fromArrayBuffer(r,o,u);if("undefined"!=typeof SharedArrayBuffer&&(isInstance(r,SharedArrayBuffer)||r&&isInstance(r.buffer,SharedArrayBuffer)))return fromArrayBuffer(r,o,u);if("number"==typeof r)throw new TypeError('The "value" argument must not be of type number. Received type number');var p=r.valueOf&&r.valueOf();if(null!=p&&p!==r)return Buffer.from(p,o,u);var y=function fromObject(r){if(Buffer.isBuffer(r)){var o=0|checked(r.length),u=createBuffer(o);return 0===u.length||r.copy(u,0,0,o),u}if(void 0!==r.length)return"number"!=typeof r.length||numberIsNaN(r.length)?createBuffer(0):fromArrayLike(r);if("Buffer"===r.type&&Array.isArray(r.data))return fromArrayLike(r.data)}(r);if(y)return y;if("undefined"!=typeof Symbol&&null!=Symbol.toPrimitive&&"function"==typeof r[Symbol.toPrimitive])return Buffer.from(r[Symbol.toPrimitive]("string"),o,u);throw new TypeError("The first argument must be one of type string, Buffer, ArrayBuffer, Array, or Array-like Object. Received type "+typeof r)}function assertSize(r){if("number"!=typeof r)throw new TypeError('"size" argument must be of type number');if(r<0)throw new RangeError('The value "'+r+'" is invalid for option "size"')}function allocUnsafe(r){return assertSize(r),createBuffer(r<0?0:0|checked(r))}function fromArrayLike(r){for(var o=r.length<0?0:0|checked(r.length),u=createBuffer(o),p=0;p<o;p+=1)u[p]=255&r[p];return u}function fromArrayBuffer(r,o,u){if(o<0||r.byteLength<o)throw new RangeError('"offset" is outside of buffer bounds');if(r.byteLength<o+(u||0))throw new RangeError('"length" is outside of buffer bounds');var p;return p=void 0===o&&void 0===u?new Uint8Array(r):void 0===u?new Uint8Array(r,o):new Uint8Array(r,o,u),Object.setPrototypeOf(p,Buffer.prototype),p}function checked(r){if(r>=P)throw new RangeError("Attempt to allocate Buffer larger than maximum size: 0x"+P.toString(16)+" bytes");return 0|r}function byteLength(r,o){if(Buffer.isBuffer(r))return r.length;if(ArrayBuffer.isView(r)||isInstance(r,ArrayBuffer))return r.byteLength;if("string"!=typeof r)throw new TypeError('The "string" argument must be one of type string, Buffer, or ArrayBuffer. Received type '+typeof r);var u=r.length,p=arguments.length>2&&!0===arguments[2];if(!p&&0===u)return 0;for(var y=!1;;)switch(o){case"ascii":case"latin1":case"binary":return u;case"utf8":case"utf-8":return utf8ToBytes(r).length;case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return 2*u;case"hex":return u>>>1;case"base64":return base64ToBytes(r).length;default:if(y)return p?-1:utf8ToBytes(r).length;o=(""+o).toLowerCase(),y=!0}}function slowToString(r,o,u){var p=!1;if((void 0===o||o<0)&&(o=0),o>this.length)return"";if((void 0===u||u>this.length)&&(u=this.length),u<=0)return"";if((u>>>=0)<=(o>>>=0))return"";for(r||(r="utf8");;)switch(r){case"hex":return hexSlice(this,o,u);case"utf8":case"utf-8":return utf8Slice(this,o,u);case"ascii":return asciiSlice(this,o,u);case"latin1":case"binary":return latin1Slice(this,o,u);case"base64":return base64Slice(this,o,u);case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return utf16leSlice(this,o,u);default:if(p)throw new TypeError("Unknown encoding: "+r);r=(r+"").toLowerCase(),p=!0}}function swap(r,o,u){var p=r[o];r[o]=r[u],r[u]=p}function bidirectionalIndexOf(r,o,u,p,y){if(0===r.length)return-1;if("string"==typeof u?(p=u,u=0):u>2147483647?u=2147483647:u<-2147483648&&(u=-2147483648),numberIsNaN(u=+u)&&(u=y?0:r.length-1),u<0&&(u=r.length+u),u>=r.length){if(y)return-1;u=r.length-1}else if(u<0){if(!y)return-1;u=0}if("string"==typeof o&&(o=Buffer.from(o,p)),Buffer.isBuffer(o))return 0===o.length?-1:arrayIndexOf(r,o,u,p,y);if("number"==typeof o)return o&=255,"function"==typeof Uint8Array.prototype.indexOf?y?Uint8Array.prototype.indexOf.call(r,o,u):Uint8Array.prototype.lastIndexOf.call(r,o,u):arrayIndexOf(r,[o],u,p,y);throw new TypeError("val must be string, number or Buffer")}function arrayIndexOf(r,o,u,p,y){var _,P=1,k=r.length,I=o.length;if(void 0!==p&&("ucs2"===(p=String(p).toLowerCase())||"ucs-2"===p||"utf16le"===p||"utf-16le"===p)){if(r.length<2||o.length<2)return-1;P=2,k/=2,I/=2,u/=2}function read(r,o){return 1===P?r[o]:r.readUInt16BE(o*P)}if(y){var O=-1;for(_=u;_<k;_++)if(read(r,_)===read(o,-1===O?0:_-O)){if(-1===O&&(O=_),_-O+1===I)return O*P}else-1!==O&&(_-=_-O),O=-1}else for(u+I>k&&(u=k-I),_=u;_>=0;_--){for(var B=!0,D=0;D<I;D++)if(read(r,_+D)!==read(o,D)){B=!1;break}if(B)return _}return-1}function hexWrite(r,o,u,p){u=Number(u)||0;var y=r.length-u;p?(p=Number(p))>y&&(p=y):p=y;var _,P=o.length;for(p>P/2&&(p=P/2),_=0;_<p;++_){var k=parseInt(o.substr(2*_,2),16);if(numberIsNaN(k))return _;r[u+_]=k}return _}function utf8Write(r,o,u,p){return blitBuffer(utf8ToBytes(o,r.length-u),r,u,p)}function asciiWrite(r,o,u,p){return blitBuffer(function asciiToBytes(r){for(var o=[],u=0;u<r.length;++u)o.push(255&r.charCodeAt(u));return o}(o),r,u,p)}function base64Write(r,o,u,p){return blitBuffer(base64ToBytes(o),r,u,p)}function ucs2Write(r,o,u,p){return blitBuffer(function utf16leToBytes(r,o){for(var u,p,y,_=[],P=0;P<r.length&&!((o-=2)<0);++P)p=(u=r.charCodeAt(P))>>8,y=u%256,_.push(y),_.push(p);return _}(o,r.length-u),r,u,p)}function base64Slice(r,o,u){return 0===o&&u===r.length?p.fromByteArray(r):p.fromByteArray(r.slice(o,u))}function utf8Slice(r,o,u){u=Math.min(r.length,u);for(var p=[],y=o;y<u;){var _=r[y],P=null,I=_>239?4:_>223?3:_>191?2:1;if(y+I<=u){var O=void 0,B=void 0,D=void 0,R=void 0;switch(I){case 1:_<128&&(P=_);break;case 2:128==(192&(O=r[y+1]))&&(R=(31&_)<<6|63&O)>127&&(P=R);break;case 3:O=r[y+1],B=r[y+2],128==(192&O)&&128==(192&B)&&(R=(15&_)<<12|(63&O)<<6|63&B)>2047&&(R<55296||R>57343)&&(P=R);break;case 4:O=r[y+1],B=r[y+2],D=r[y+3],128==(192&O)&&128==(192&B)&&128==(192&D)&&(R=(15&_)<<18|(63&O)<<12|(63&B)<<6|63&D)>65535&&R<1114112&&(P=R)}}null===P?(P=65533,I=1):P>65535&&(P-=65536,p.push(P>>>10&1023|55296),P=56320|1023&P),p.push(P),y+=I}return function decodeCodePointsArray(r){var o=r.length;if(o<=k)return String.fromCharCode.apply(String,r);var u="",p=0;for(;p<o;)u+=String.fromCharCode.apply(String,r.slice(p,p+=k));return u}(p)}o.kMaxLength=P,Buffer.TYPED_ARRAY_SUPPORT=function typedArraySupport(){try{var r=new Uint8Array(1),o={foo:function foo(){return 42}};return Object.setPrototypeOf(o,Uint8Array.prototype),Object.setPrototypeOf(r,o),42===r.foo()}catch(r){return!1}}(),!Buffer.TYPED_ARRAY_SUPPORT&&"undefined"!=typeof console&&console.error,Object.defineProperty(Buffer.prototype,"parent",{enumerable:!0,get:function get(){if(Buffer.isBuffer(this))return this.buffer}}),Object.defineProperty(Buffer.prototype,"offset",{enumerable:!0,get:function get(){if(Buffer.isBuffer(this))return this.byteOffset}}),Buffer.poolSize=8192,Buffer.from=function(r,o,u){return from(r,o,u)},Object.setPrototypeOf(Buffer.prototype,Uint8Array.prototype),Object.setPrototypeOf(Buffer,Uint8Array),Buffer.alloc=function(r,o,u){return function alloc(r,o,u){return assertSize(r),r<=0?createBuffer(r):void 0!==o?"string"==typeof u?createBuffer(r).fill(o,u):createBuffer(r).fill(o):createBuffer(r)}(r,o,u)},Buffer.allocUnsafe=function(r){return allocUnsafe(r)},Buffer.allocUnsafeSlow=function(r){return allocUnsafe(r)},Buffer.isBuffer=function isBuffer(r){return null!=r&&!0===r._isBuffer&&r!==Buffer.prototype},Buffer.compare=function compare(r,o){if(isInstance(r,Uint8Array)&&(r=Buffer.from(r,r.offset,r.byteLength)),isInstance(o,Uint8Array)&&(o=Buffer.from(o,o.offset,o.byteLength)),!Buffer.isBuffer(r)||!Buffer.isBuffer(o))throw new TypeError('The "buf1", "buf2" arguments must be one of type Buffer or Uint8Array');if(r===o)return 0;for(var u=r.length,p=o.length,y=0,_=Math.min(u,p);y<_;++y)if(r[y]!==o[y]){u=r[y],p=o[y];break}return u<p?-1:p<u?1:0},Buffer.isEncoding=function isEncoding(r){switch(String(r).toLowerCase()){case"hex":case"utf8":case"utf-8":case"ascii":case"latin1":case"binary":case"base64":case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return!0;default:return!1}},Buffer.concat=function concat(r,o){if(!Array.isArray(r))throw new TypeError('"list" argument must be an Array of Buffers');if(0===r.length)return Buffer.alloc(0);var u;if(void 0===o)for(o=0,u=0;u<r.length;++u)o+=r[u].length;var p=Buffer.allocUnsafe(o),y=0;for(u=0;u<r.length;++u){var _=r[u];if(isInstance(_,Uint8Array))y+_.length>p.length?(Buffer.isBuffer(_)||(_=Buffer.from(_)),_.copy(p,y)):Uint8Array.prototype.set.call(p,_,y);else{if(!Buffer.isBuffer(_))throw new TypeError('"list" argument must be an Array of Buffers');_.copy(p,y)}y+=_.length}return p},Buffer.byteLength=byteLength,Buffer.prototype._isBuffer=!0,Buffer.prototype.swap16=function swap16(){var r=this.length;if(r%2!=0)throw new RangeError("Buffer size must be a multiple of 16-bits");for(var o=0;o<r;o+=2)swap(this,o,o+1);return this},Buffer.prototype.swap32=function swap32(){var r=this.length;if(r%4!=0)throw new RangeError("Buffer size must be a multiple of 32-bits");for(var o=0;o<r;o+=4)swap(this,o,o+3),swap(this,o+1,o+2);return this},Buffer.prototype.swap64=function swap64(){var r=this.length;if(r%8!=0)throw new RangeError("Buffer size must be a multiple of 64-bits");for(var o=0;o<r;o+=8)swap(this,o,o+7),swap(this,o+1,o+6),swap(this,o+2,o+5),swap(this,o+3,o+4);return this},Buffer.prototype.toString=function toString(){var r=this.length;return 0===r?"":0===arguments.length?utf8Slice(this,0,r):slowToString.apply(this,arguments)},Buffer.prototype.toLocaleString=Buffer.prototype.toString,Buffer.prototype.equals=function equals(r){if(!Buffer.isBuffer(r))throw new TypeError("Argument must be a Buffer");return this===r||0===Buffer.compare(this,r)},Buffer.prototype.inspect=function inspect(){var r="",u=o.INSPECT_MAX_BYTES;return r=this.toString("hex",0,u).replace(/(.{2})/g,"$1 ").trim(),this.length>u&&(r+=" ... "),"<Buffer "+r+">"},_&&(Buffer.prototype[_]=Buffer.prototype.inspect),Buffer.prototype.compare=function compare(r,o,u,p,y){if(isInstance(r,Uint8Array)&&(r=Buffer.from(r,r.offset,r.byteLength)),!Buffer.isBuffer(r))throw new TypeError('The "target" argument must be one of type Buffer or Uint8Array. Received type '+typeof r);if(void 0===o&&(o=0),void 0===u&&(u=r?r.length:0),void 0===p&&(p=0),void 0===y&&(y=this.length),o<0||u>r.length||p<0||y>this.length)throw new RangeError("out of range index");if(p>=y&&o>=u)return 0;if(p>=y)return-1;if(o>=u)return 1;if(this===r)return 0;for(var _=(y>>>=0)-(p>>>=0),P=(u>>>=0)-(o>>>=0),k=Math.min(_,P),I=this.slice(p,y),O=r.slice(o,u),B=0;B<k;++B)if(I[B]!==O[B]){_=I[B],P=O[B];break}return _<P?-1:P<_?1:0},Buffer.prototype.includes=function includes(r,o,u){return-1!==this.indexOf(r,o,u)},Buffer.prototype.indexOf=function indexOf(r,o,u){return bidirectionalIndexOf(this,r,o,u,!0)},Buffer.prototype.lastIndexOf=function lastIndexOf(r,o,u){return bidirectionalIndexOf(this,r,o,u,!1)},Buffer.prototype.write=function write(r,o,u,p){if(void 0===o)p="utf8",u=this.length,o=0;else if(void 0===u&&"string"==typeof o)p=o,u=this.length,o=0;else{if(!isFinite(o))throw new Error("Buffer.write(string, encoding, offset[, length]) is no longer supported");o>>>=0,isFinite(u)?(u>>>=0,void 0===p&&(p="utf8")):(p=u,u=void 0)}var y=this.length-o;if((void 0===u||u>y)&&(u=y),r.length>0&&(u<0||o<0)||o>this.length)throw new RangeError("Attempt to write outside buffer bounds");p||(p="utf8");for(var _=!1;;)switch(p){case"hex":return hexWrite(this,r,o,u);case"utf8":case"utf-8":return utf8Write(this,r,o,u);case"ascii":case"latin1":case"binary":return asciiWrite(this,r,o,u);case"base64":return base64Write(this,r,o,u);case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return ucs2Write(this,r,o,u);default:if(_)throw new TypeError("Unknown encoding: "+p);p=(""+p).toLowerCase(),_=!0}},Buffer.prototype.toJSON=function toJSON(){return{type:"Buffer",data:Array.prototype.slice.call(this._arr||this,0)}};var k=4096;function asciiSlice(r,o,u){var p="";u=Math.min(r.length,u);for(var y=o;y<u;++y)p+=String.fromCharCode(127&r[y]);return p}function latin1Slice(r,o,u){var p="";u=Math.min(r.length,u);for(var y=o;y<u;++y)p+=String.fromCharCode(r[y]);return p}function hexSlice(r,o,u){var p=r.length;(!o||o<0)&&(o=0),(!u||u<0||u>p)&&(u=p);for(var y="",_=o;_<u;++_)y+=B[r[_]];return y}function utf16leSlice(r,o,u){for(var p=r.slice(o,u),y="",_=0;_<p.length-1;_+=2)y+=String.fromCharCode(p[_]+256*p[_+1]);return y}function checkOffset(r,o,u){if(r%1!=0||r<0)throw new RangeError("offset is not uint");if(r+o>u)throw new RangeError("Trying to access beyond buffer length")}function checkInt(r,o,u,p,y,_){if(!Buffer.isBuffer(r))throw new TypeError('"buffer" argument must be a Buffer instance');if(o>y||o<_)throw new RangeError('"value" argument is out of bounds');if(u+p>r.length)throw new RangeError("Index out of range")}function wrtBigUInt64LE(r,o,u,p,y){checkIntBI(o,p,y,r,u,7);var _=Number(o&BigInt(4294967295));r[u++]=_,_>>=8,r[u++]=_,_>>=8,r[u++]=_,_>>=8,r[u++]=_;var P=Number(o>>BigInt(32)&BigInt(4294967295));return r[u++]=P,P>>=8,r[u++]=P,P>>=8,r[u++]=P,P>>=8,r[u++]=P,u}function wrtBigUInt64BE(r,o,u,p,y){checkIntBI(o,p,y,r,u,7);var _=Number(o&BigInt(4294967295));r[u+7]=_,_>>=8,r[u+6]=_,_>>=8,r[u+5]=_,_>>=8,r[u+4]=_;var P=Number(o>>BigInt(32)&BigInt(4294967295));return r[u+3]=P,P>>=8,r[u+2]=P,P>>=8,r[u+1]=P,P>>=8,r[u]=P,u+8}function checkIEEE754(r,o,u,p,y,_){if(u+p>r.length)throw new RangeError("Index out of range");if(u<0)throw new RangeError("Index out of range")}function writeFloat(r,o,u,p,_){return o=+o,u>>>=0,_||checkIEEE754(r,0,u,4),y.write(r,o,u,p,23,4),u+4}function writeDouble(r,o,u,p,_){return o=+o,u>>>=0,_||checkIEEE754(r,0,u,8),y.write(r,o,u,p,52,8),u+8}Buffer.prototype.slice=function slice(r,o){var u=this.length;(r=~~r)<0?(r+=u)<0&&(r=0):r>u&&(r=u),(o=void 0===o?u:~~o)<0?(o+=u)<0&&(o=0):o>u&&(o=u),o<r&&(o=r);var p=this.subarray(r,o);return Object.setPrototypeOf(p,Buffer.prototype),p},Buffer.prototype.readUintLE=Buffer.prototype.readUIntLE=function readUIntLE(r,o,u){r>>>=0,o>>>=0,u||checkOffset(r,o,this.length);for(var p=this[r],y=1,_=0;++_<o&&(y*=256);)p+=this[r+_]*y;return p},Buffer.prototype.readUintBE=Buffer.prototype.readUIntBE=function readUIntBE(r,o,u){r>>>=0,o>>>=0,u||checkOffset(r,o,this.length);for(var p=this[r+--o],y=1;o>0&&(y*=256);)p+=this[r+--o]*y;return p},Buffer.prototype.readUint8=Buffer.prototype.readUInt8=function readUInt8(r,o){return r>>>=0,o||checkOffset(r,1,this.length),this[r]},Buffer.prototype.readUint16LE=Buffer.prototype.readUInt16LE=function readUInt16LE(r,o){return r>>>=0,o||checkOffset(r,2,this.length),this[r]|this[r+1]<<8},Buffer.prototype.readUint16BE=Buffer.prototype.readUInt16BE=function readUInt16BE(r,o){return r>>>=0,o||checkOffset(r,2,this.length),this[r]<<8|this[r+1]},Buffer.prototype.readUint32LE=Buffer.prototype.readUInt32LE=function readUInt32LE(r,o){return r>>>=0,o||checkOffset(r,4,this.length),(this[r]|this[r+1]<<8|this[r+2]<<16)+16777216*this[r+3]},Buffer.prototype.readUint32BE=Buffer.prototype.readUInt32BE=function readUInt32BE(r,o){return r>>>=0,o||checkOffset(r,4,this.length),16777216*this[r]+(this[r+1]<<16|this[r+2]<<8|this[r+3])},Buffer.prototype.readBigUInt64LE=defineBigIntMethod((function readBigUInt64LE(r){validateNumber(r>>>=0,"offset");var o=this[r],u=this[r+7];void 0!==o&&void 0!==u||boundsError(r,this.length-8);var p=o+this[++r]*Math.pow(2,8)+this[++r]*Math.pow(2,16)+this[++r]*Math.pow(2,24),y=this[++r]+this[++r]*Math.pow(2,8)+this[++r]*Math.pow(2,16)+u*Math.pow(2,24);return BigInt(p)+(BigInt(y)<<BigInt(32))})),Buffer.prototype.readBigUInt64BE=defineBigIntMethod((function readBigUInt64BE(r){validateNumber(r>>>=0,"offset");var o=this[r],u=this[r+7];void 0!==o&&void 0!==u||boundsError(r,this.length-8);var p=o*Math.pow(2,24)+this[++r]*Math.pow(2,16)+this[++r]*Math.pow(2,8)+this[++r],y=this[++r]*Math.pow(2,24)+this[++r]*Math.pow(2,16)+this[++r]*Math.pow(2,8)+u;return(BigInt(p)<<BigInt(32))+BigInt(y)})),Buffer.prototype.readIntLE=function readIntLE(r,o,u){r>>>=0,o>>>=0,u||checkOffset(r,o,this.length);for(var p=this[r],y=1,_=0;++_<o&&(y*=256);)p+=this[r+_]*y;return p>=(y*=128)&&(p-=Math.pow(2,8*o)),p},Buffer.prototype.readIntBE=function readIntBE(r,o,u){r>>>=0,o>>>=0,u||checkOffset(r,o,this.length);for(var p=o,y=1,_=this[r+--p];p>0&&(y*=256);)_+=this[r+--p]*y;return _>=(y*=128)&&(_-=Math.pow(2,8*o)),_},Buffer.prototype.readInt8=function readInt8(r,o){return r>>>=0,o||checkOffset(r,1,this.length),128&this[r]?-1*(255-this[r]+1):this[r]},Buffer.prototype.readInt16LE=function readInt16LE(r,o){r>>>=0,o||checkOffset(r,2,this.length);var u=this[r]|this[r+1]<<8;return 32768&u?4294901760|u:u},Buffer.prototype.readInt16BE=function readInt16BE(r,o){r>>>=0,o||checkOffset(r,2,this.length);var u=this[r+1]|this[r]<<8;return 32768&u?4294901760|u:u},Buffer.prototype.readInt32LE=function readInt32LE(r,o){return r>>>=0,o||checkOffset(r,4,this.length),this[r]|this[r+1]<<8|this[r+2]<<16|this[r+3]<<24},Buffer.prototype.readInt32BE=function readInt32BE(r,o){return r>>>=0,o||checkOffset(r,4,this.length),this[r]<<24|this[r+1]<<16|this[r+2]<<8|this[r+3]},Buffer.prototype.readBigInt64LE=defineBigIntMethod((function readBigInt64LE(r){validateNumber(r>>>=0,"offset");var o=this[r],u=this[r+7];void 0!==o&&void 0!==u||boundsError(r,this.length-8);var p=this[r+4]+this[r+5]*Math.pow(2,8)+this[r+6]*Math.pow(2,16)+(u<<24);return(BigInt(p)<<BigInt(32))+BigInt(o+this[++r]*Math.pow(2,8)+this[++r]*Math.pow(2,16)+this[++r]*Math.pow(2,24))})),Buffer.prototype.readBigInt64BE=defineBigIntMethod((function readBigInt64BE(r){validateNumber(r>>>=0,"offset");var o=this[r],u=this[r+7];void 0!==o&&void 0!==u||boundsError(r,this.length-8);var p=(o<<24)+this[++r]*Math.pow(2,16)+this[++r]*Math.pow(2,8)+this[++r];return(BigInt(p)<<BigInt(32))+BigInt(this[++r]*Math.pow(2,24)+this[++r]*Math.pow(2,16)+this[++r]*Math.pow(2,8)+u)})),Buffer.prototype.readFloatLE=function readFloatLE(r,o){return r>>>=0,o||checkOffset(r,4,this.length),y.read(this,r,!0,23,4)},Buffer.prototype.readFloatBE=function readFloatBE(r,o){return r>>>=0,o||checkOffset(r,4,this.length),y.read(this,r,!1,23,4)},Buffer.prototype.readDoubleLE=function readDoubleLE(r,o){return r>>>=0,o||checkOffset(r,8,this.length),y.read(this,r,!0,52,8)},Buffer.prototype.readDoubleBE=function readDoubleBE(r,o){return r>>>=0,o||checkOffset(r,8,this.length),y.read(this,r,!1,52,8)},Buffer.prototype.writeUintLE=Buffer.prototype.writeUIntLE=function writeUIntLE(r,o,u,p){(r=+r,o>>>=0,u>>>=0,p)||checkInt(this,r,o,u,Math.pow(2,8*u)-1,0);var y=1,_=0;for(this[o]=255&r;++_<u&&(y*=256);)this[o+_]=r/y&255;return o+u},Buffer.prototype.writeUintBE=Buffer.prototype.writeUIntBE=function writeUIntBE(r,o,u,p){(r=+r,o>>>=0,u>>>=0,p)||checkInt(this,r,o,u,Math.pow(2,8*u)-1,0);var y=u-1,_=1;for(this[o+y]=255&r;--y>=0&&(_*=256);)this[o+y]=r/_&255;return o+u},Buffer.prototype.writeUint8=Buffer.prototype.writeUInt8=function writeUInt8(r,o,u){return r=+r,o>>>=0,u||checkInt(this,r,o,1,255,0),this[o]=255&r,o+1},Buffer.prototype.writeUint16LE=Buffer.prototype.writeUInt16LE=function writeUInt16LE(r,o,u){return r=+r,o>>>=0,u||checkInt(this,r,o,2,65535,0),this[o]=255&r,this[o+1]=r>>>8,o+2},Buffer.prototype.writeUint16BE=Buffer.prototype.writeUInt16BE=function writeUInt16BE(r,o,u){return r=+r,o>>>=0,u||checkInt(this,r,o,2,65535,0),this[o]=r>>>8,this[o+1]=255&r,o+2},Buffer.prototype.writeUint32LE=Buffer.prototype.writeUInt32LE=function writeUInt32LE(r,o,u){return r=+r,o>>>=0,u||checkInt(this,r,o,4,4294967295,0),this[o+3]=r>>>24,this[o+2]=r>>>16,this[o+1]=r>>>8,this[o]=255&r,o+4},Buffer.prototype.writeUint32BE=Buffer.prototype.writeUInt32BE=function writeUInt32BE(r,o,u){return r=+r,o>>>=0,u||checkInt(this,r,o,4,4294967295,0),this[o]=r>>>24,this[o+1]=r>>>16,this[o+2]=r>>>8,this[o+3]=255&r,o+4},Buffer.prototype.writeBigUInt64LE=defineBigIntMethod((function writeBigUInt64LE(r,o){return void 0===o&&(o=0),wrtBigUInt64LE(this,r,o,BigInt(0),BigInt("0xffffffffffffffff"))})),Buffer.prototype.writeBigUInt64BE=defineBigIntMethod((function writeBigUInt64BE(r,o){return void 0===o&&(o=0),wrtBigUInt64BE(this,r,o,BigInt(0),BigInt("0xffffffffffffffff"))})),Buffer.prototype.writeIntLE=function writeIntLE(r,o,u,p){if(r=+r,o>>>=0,!p){var y=Math.pow(2,8*u-1);checkInt(this,r,o,u,y-1,-y)}var _=0,P=1,k=0;for(this[o]=255&r;++_<u&&(P*=256);)r<0&&0===k&&0!==this[o+_-1]&&(k=1),this[o+_]=(r/P|0)-k&255;return o+u},Buffer.prototype.writeIntBE=function writeIntBE(r,o,u,p){if(r=+r,o>>>=0,!p){var y=Math.pow(2,8*u-1);checkInt(this,r,o,u,y-1,-y)}var _=u-1,P=1,k=0;for(this[o+_]=255&r;--_>=0&&(P*=256);)r<0&&0===k&&0!==this[o+_+1]&&(k=1),this[o+_]=(r/P|0)-k&255;return o+u},Buffer.prototype.writeInt8=function writeInt8(r,o,u){return r=+r,o>>>=0,u||checkInt(this,r,o,1,127,-128),r<0&&(r=255+r+1),this[o]=255&r,o+1},Buffer.prototype.writeInt16LE=function writeInt16LE(r,o,u){return r=+r,o>>>=0,u||checkInt(this,r,o,2,32767,-32768),this[o]=255&r,this[o+1]=r>>>8,o+2},Buffer.prototype.writeInt16BE=function writeInt16BE(r,o,u){return r=+r,o>>>=0,u||checkInt(this,r,o,2,32767,-32768),this[o]=r>>>8,this[o+1]=255&r,o+2},Buffer.prototype.writeInt32LE=function writeInt32LE(r,o,u){return r=+r,o>>>=0,u||checkInt(this,r,o,4,2147483647,-2147483648),this[o]=255&r,this[o+1]=r>>>8,this[o+2]=r>>>16,this[o+3]=r>>>24,o+4},Buffer.prototype.writeInt32BE=function writeInt32BE(r,o,u){return r=+r,o>>>=0,u||checkInt(this,r,o,4,2147483647,-2147483648),r<0&&(r=4294967295+r+1),this[o]=r>>>24,this[o+1]=r>>>16,this[o+2]=r>>>8,this[o+3]=255&r,o+4},Buffer.prototype.writeBigInt64LE=defineBigIntMethod((function writeBigInt64LE(r,o){return void 0===o&&(o=0),wrtBigUInt64LE(this,r,o,-BigInt("0x8000000000000000"),BigInt("0x7fffffffffffffff"))})),Buffer.prototype.writeBigInt64BE=defineBigIntMethod((function writeBigInt64BE(r,o){return void 0===o&&(o=0),wrtBigUInt64BE(this,r,o,-BigInt("0x8000000000000000"),BigInt("0x7fffffffffffffff"))})),Buffer.prototype.writeFloatLE=function writeFloatLE(r,o,u){return writeFloat(this,r,o,!0,u)},Buffer.prototype.writeFloatBE=function writeFloatBE(r,o,u){return writeFloat(this,r,o,!1,u)},Buffer.prototype.writeDoubleLE=function writeDoubleLE(r,o,u){return writeDouble(this,r,o,!0,u)},Buffer.prototype.writeDoubleBE=function writeDoubleBE(r,o,u){return writeDouble(this,r,o,!1,u)},Buffer.prototype.copy=function copy(r,o,u,p){if(!Buffer.isBuffer(r))throw new TypeError("argument should be a Buffer");if(u||(u=0),p||0===p||(p=this.length),o>=r.length&&(o=r.length),o||(o=0),p>0&&p<u&&(p=u),p===u)return 0;if(0===r.length||0===this.length)return 0;if(o<0)throw new RangeError("targetStart out of bounds");if(u<0||u>=this.length)throw new RangeError("Index out of range");if(p<0)throw new RangeError("sourceEnd out of bounds");p>this.length&&(p=this.length),r.length-o<p-u&&(p=r.length-o+u);var y=p-u;return this===r&&"function"==typeof Uint8Array.prototype.copyWithin?this.copyWithin(o,u,p):Uint8Array.prototype.set.call(r,this.subarray(u,p),o),y},Buffer.prototype.fill=function fill(r,o,u,p){if("string"==typeof r){if("string"==typeof o?(p=o,o=0,u=this.length):"string"==typeof u&&(p=u,u=this.length),void 0!==p&&"string"!=typeof p)throw new TypeError("encoding must be a string");if("string"==typeof p&&!Buffer.isEncoding(p))throw new TypeError("Unknown encoding: "+p);if(1===r.length){var y=r.charCodeAt(0);("utf8"===p&&y<128||"latin1"===p)&&(r=y)}}else"number"==typeof r?r&=255:"boolean"==typeof r&&(r=Number(r));if(o<0||this.length<o||this.length<u)throw new RangeError("Out of range index");if(u<=o)return this;var _;if(o>>>=0,u=void 0===u?this.length:u>>>0,r||(r=0),"number"==typeof r)for(_=o;_<u;++_)this[_]=r;else{var P=Buffer.isBuffer(r)?r:Buffer.from(r,p),k=P.length;if(0===k)throw new TypeError('The value "'+r+'" is invalid for argument "value"');for(_=0;_<u-o;++_)this[_+o]=P[_%k]}return this};var I={};function E(r,o,u){I[r]=function(u){function NodeError(){var p;return p=u.call(this)||this,Object.defineProperty(p,"message",{value:o.apply(p,arguments),writable:!0,configurable:!0}),p.name=p.name+" ["+r+"]",p.stack,delete p.name,p}return function _inheritsLoose(r,o){r.prototype=Object.create(o.prototype),r.prototype.constructor=r,_setPrototypeOf(r,o)}(NodeError,u),NodeError.prototype.toString=function toString(){return this.name+" ["+r+"]: "+this.message},function _createClass(r,o,u){return o&&_defineProperties(r.prototype,o),u&&_defineProperties(r,u),Object.defineProperty(r,"prototype",{writable:!1}),r}(NodeError,[{key:"code",get:function get(){return r},set:function set(r){Object.defineProperty(this,"code",{configurable:!0,enumerable:!0,value:r,writable:!0})}}])}(u)}function addNumericalSeparator(r){for(var o="",u=r.length,p="-"===r[0]?1:0;u>=p+4;u-=3)o="_"+r.slice(u-3,u)+o;return""+r.slice(0,u)+o}function checkIntBI(r,o,u,p,y,_){if(r>u||r<o){var P,k="bigint"==typeof o?"n":"";throw P=_>3?0===o||o===BigInt(0)?">= 0"+k+" and < 2"+k+" ** "+8*(_+1)+k:">= -(2"+k+" ** "+(8*(_+1)-1)+k+") and < 2 ** "+(8*(_+1)-1)+k:">= "+o+k+" and <= "+u+k,new I.ERR_OUT_OF_RANGE("value",P,r)}!function checkBounds(r,o,u){validateNumber(o,"offset"),void 0!==r[o]&&void 0!==r[o+u]||boundsError(o,r.length-(u+1))}(p,y,_)}function validateNumber(r,o){if("number"!=typeof r)throw new I.ERR_INVALID_ARG_TYPE(o,"number",r)}function boundsError(r,o,u){if(Math.floor(r)!==r)throw validateNumber(r,u),new I.ERR_OUT_OF_RANGE(u||"offset","an integer",r);if(o<0)throw new I.ERR_BUFFER_OUT_OF_BOUNDS;throw new I.ERR_OUT_OF_RANGE(u||"offset",">= "+(u?1:0)+" and <= "+o,r)}E("ERR_BUFFER_OUT_OF_BOUNDS",(function(r){return r?r+" is outside of buffer bounds":"Attempt to access memory outside buffer bounds"}),RangeError),E("ERR_INVALID_ARG_TYPE",(function(r,o){return'The "'+r+'" argument must be of type number. Received type '+typeof o}),TypeError),E("ERR_OUT_OF_RANGE",(function(r,o,u){var p='The value of "'+r+'" is out of range.',y=u;return Number.isInteger(u)&&Math.abs(u)>Math.pow(2,32)?y=addNumericalSeparator(String(u)):"bigint"==typeof u&&(y=String(u),(u>Math.pow(BigInt(2),BigInt(32))||u<-Math.pow(BigInt(2),BigInt(32)))&&(y=addNumericalSeparator(y)),y+="n"),p+=" It must be "+o+". Received "+y}),RangeError);var O=/[^+/0-9A-Za-z-_]/g;function utf8ToBytes(r,o){var u;o=o||1/0;for(var p=r.length,y=null,_=[],P=0;P<p;++P){if((u=r.charCodeAt(P))>55295&&u<57344){if(!y){if(u>56319){(o-=3)>-1&&_.push(239,191,189);continue}if(P+1===p){(o-=3)>-1&&_.push(239,191,189);continue}y=u;continue}if(u<56320){(o-=3)>-1&&_.push(239,191,189),y=u;continue}u=65536+(y-55296<<10|u-56320)}else y&&(o-=3)>-1&&_.push(239,191,189);if(y=null,u<128){if((o-=1)<0)break;_.push(u)}else if(u<2048){if((o-=2)<0)break;_.push(u>>6|192,63&u|128)}else if(u<65536){if((o-=3)<0)break;_.push(u>>12|224,u>>6&63|128,63&u|128)}else{if(!(u<1114112))throw new Error("Invalid code point");if((o-=4)<0)break;_.push(u>>18|240,u>>12&63|128,u>>6&63|128,63&u|128)}}return _}function base64ToBytes(r){return p.toByteArray(function base64clean(r){if((r=(r=r.split("=")[0]).trim().replace(O,"")).length<2)return"";for(;r.length%4!=0;)r+="=";return r}(r))}function blitBuffer(r,o,u,p){var y;for(y=0;y<p&&!(y+u>=o.length||y>=r.length);++y)o[y+u]=r[y];return y}function isInstance(r,o){return r instanceof o||null!=r&&null!=r.constructor&&null!=r.constructor.name&&r.constructor.name===o.name}function numberIsNaN(r){return r!=r}var B=function(){for(var r="0123456789abcdef",o=new Array(256),u=0;u<16;++u)for(var p=16*u,y=0;y<16;++y)o[p+y]=r[u]+r[y];return o}();function defineBigIntMethod(r){return"undefined"==typeof BigInt?BufferBigIntNotDefined:r}function BufferBigIntNotDefined(){throw new Error("BigInt not supported")}},14623:function(r,o,u){"use strict";var p;u(39081),r.exports=(p=u(48352),u(51270),p.pad.Iso97971={pad:function pad(r,o){r.concat(p.lib.WordArray.create([2147483648],1)),p.pad.ZeroPadding.pad(r,o)},unpad:function unpad(r){p.pad.ZeroPadding.unpad(r),r.sigBytes--}},p.pad.Iso97971)},14903:function(r,o,u){"use strict";var p=u(9964),y=Object.keys||function(r){var o=[];for(var u in r)o.push(u);return o};r.exports=Duplex;var _=u(88261),P=u(29781);u(89784)(Duplex,_);for(var k=y(P.prototype),I=0;I<k.length;I++){var O=k[I];Duplex.prototype[O]||(Duplex.prototype[O]=P.prototype[O])}function Duplex(r){if(!(this instanceof Duplex))return new Duplex(r);_.call(this,r),P.call(this,r),this.allowHalfOpen=!0,r&&(!1===r.readable&&(this.readable=!1),!1===r.writable&&(this.writable=!1),!1===r.allowHalfOpen&&(this.allowHalfOpen=!1,this.once("end",onend)))}function onend(){this._writableState.ended||p.nextTick(onEndNT,this)}function onEndNT(r){r.end()}Object.defineProperty(Duplex.prototype,"writableHighWaterMark",{enumerable:!1,get:function get(){return this._writableState.highWaterMark}}),Object.defineProperty(Duplex.prototype,"writableBuffer",{enumerable:!1,get:function get(){return this._writableState&&this._writableState.getBuffer()}}),Object.defineProperty(Duplex.prototype,"writableLength",{enumerable:!1,get:function get(){return this._writableState.length}}),Object.defineProperty(Duplex.prototype,"destroyed",{enumerable:!1,get:function get(){return void 0!==this._readableState&&void 0!==this._writableState&&(this._readableState.destroyed&&this._writableState.destroyed)},set:function set(r){void 0!==this._readableState&&void 0!==this._writableState&&(this._readableState.destroyed=r,this._writableState.destroyed=r)}})},15216:function(r,o,u){var p=u(95892).f,y=u(20340),_=u(38688)("toStringTag");r.exports=function(r,o,u){r&&!y(r=u?r:r.prototype,_)&&p(r,_,{configurable:!0,value:o})}},15293:function(r){"use strict";r.exports=Error},15341:function(r,o,u){var p=u(13711);r.exports=function(r,o,u){for(var y in o)p(r,y,o[y],u);return r}},15366:function(r){r.exports={}},15548:function(r,o,u){var p,y,_;y=[],void 0===(_="function"==typeof(p=function(){"use strict";function b(r,o){return void 0===o?o={autoBom:!1}:"object"!=typeof o&&(o={autoBom:!o}),o.autoBom&&/^\s*(?:text\/\S*|application\/xml|\S*\/\S*\+xml)\s*;.*charset\s*=\s*utf-8/i.test(r.type)?new Blob(["\ufeff",r],{type:r.type}):r}function c(r,o,u){var p=new XMLHttpRequest;p.open("GET",r),p.responseType="blob",p.onload=function(){y(p.response,o,u)},p.onerror=function(){},p.send()}function d(r){var o=new XMLHttpRequest;o.open("HEAD",r,!1);try{o.send()}catch(r){}return 200<=o.status&&299>=o.status}function e(r){try{r.dispatchEvent(new MouseEvent("click"))}catch(u){var o=document.createEvent("MouseEvents");o.initMouseEvent("click",!0,!0,window,0,0,0,80,20,!1,!1,!1,!1,0,null),r.dispatchEvent(o)}}var o="object"==typeof window&&window.window===window?window:"object"==typeof self&&self.self===self?self:"object"==typeof u.g&&u.g.global===u.g?u.g:void 0,p=o.navigator&&/Macintosh/.test(navigator.userAgent)&&/AppleWebKit/.test(navigator.userAgent)&&!/Safari/.test(navigator.userAgent),y=o.saveAs||("object"!=typeof window||window!==o?function(){}:"undefined"!=typeof HTMLAnchorElement&&"download"in HTMLAnchorElement.prototype&&!p?function(r,u,p){var y=o.URL||o.webkitURL,_=document.createElement("a");u=u||r.name||"download",_.download=u,_.rel="noopener","string"==typeof r?(_.href=r,_.origin===location.origin?e(_):d(_.href)?c(r,u,p):e(_,_.target="_blank")):(_.href=y.createObjectURL(r),setTimeout((function(){y.revokeObjectURL(_.href)}),4e4),setTimeout((function(){e(_)}),0))}:"msSaveOrOpenBlob"in navigator?function(r,o,u){if(o=o||r.name||"download","string"!=typeof r)navigator.msSaveOrOpenBlob(b(r,u),o);else if(d(r))c(r,o,u);else{var p=document.createElement("a");p.href=r,p.target="_blank",setTimeout((function(){e(p)}))}}:function(r,u,y,_){if((_=_||open("","_blank"))&&(_.document.title=_.document.body.innerText="downloading..."),"string"==typeof r)return c(r,u,y);var P="application/octet-stream"===r.type,k=/constructor/i.test(o.HTMLElement)||o.safari,I=/CriOS\/[\d]+/.test(navigator.userAgent);if((I||P&&k||p)&&"undefined"!=typeof FileReader){var O=new FileReader;O.onloadend=function(){var r=O.result;r=I?r:r.replace(/^data:[^;]*;/,"data:attachment/file;"),_?_.location.href=r:location=r,_=null},O.readAsDataURL(r)}else{var B=o.URL||o.webkitURL,D=B.createObjectURL(r);_?_.location=D:location.href=D,_=null,setTimeout((function(){B.revokeObjectURL(D)}),4e4)}});o.saveAs=y.saveAs=y,r.exports=y})?p.apply(o,y):p)||(r.exports=_)},15567:function(r,o,u){var p=u(47044);r.exports=!p((function(){return 7!=Object.defineProperty({},1,{get:function(){return 7}})[1]}))},15892:function(r,o,u){var p=u(32010),y=u(2834),_=u(32631),P=u(34984),k=u(68664),I=u(13872),O=p.TypeError;r.exports=function(r,o){var u=arguments.length<2?I(r):o;if(_(u))return P(y(u,r));throw O(k(r)+" is not iterable")}},16190:function(r,o,u){"use strict";var p;u(94845),r.exports=(p=u(48352),function(){var r=p,o=r.lib.WordArray;function parseLoop(r,u,p){for(var y=[],_=0,P=0;P<u;P++)if(P%4){var k=p[r.charCodeAt(P-1)]<<P%4*2|p[r.charCodeAt(P)]>>>6-P%4*2;y[_>>>2]|=k<<24-_%4*8,_++}return o.create(y,_)}r.enc.Base64url={stringify:function stringify(r,o){void 0===o&&(o=!0);var u=r.words,p=r.sigBytes,y=o?this._safe_map:this._map;r.clamp();for(var _=[],P=0;P<p;P+=3)for(var k=(u[P>>>2]>>>24-P%4*8&255)<<16|(u[P+1>>>2]>>>24-(P+1)%4*8&255)<<8|u[P+2>>>2]>>>24-(P+2)%4*8&255,I=0;I<4&&P+.75*I<p;I++)_.push(y.charAt(k>>>6*(3-I)&63));var O=y.charAt(64);if(O)for(;_.length%4;)_.push(O);return _.join("")},parse:function parse(r,o){void 0===o&&(o=!0);var u=r.length,p=o?this._safe_map:this._map,y=this._reverseMap;if(!y){y=this._reverseMap=[];for(var _=0;_<p.length;_++)y[p.charCodeAt(_)]=_}var P=p.charAt(64);if(P){var k=r.indexOf(P);-1!==k&&(u=k)}return parseLoop(r,u,y)},_map:"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=",_safe_map:"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_"}}(),p.enc.Base64url)},16403:function(){},16465:function(r,o,u){function config(r){try{if(!u.g.localStorage)return!1}catch(r){return!1}var o=u.g.localStorage[r];return null!=o&&"true"===String(o).toLowerCase()}r.exports=function deprecate(r,o){if(config("noDeprecation"))return r;var u=!1;return function deprecated(){if(!u){if(config("throwDeprecation"))throw new Error(o);config("traceDeprecation"),u=!0}return r.apply(this,arguments)}}},16679:function(r,o,u){var p=u(38347);r.exports=p(1..valueOf)},16720:function(r,o,u){"use strict";var p;u(20731),r.exports=(p=u(48352),u(18909),function(r){var o=p,u=o.lib,y=u.WordArray,_=u.Hasher,P=o.x64.Word,k=o.algo,I=[],O=[],B=[];!function(){for(var r=1,o=0,u=0;u<24;u++){I[r+5*o]=(u+1)*(u+2)/2%64;var p=(2*r+3*o)%5;r=o%5,o=p}for(r=0;r<5;r++)for(o=0;o<5;o++)O[r+5*o]=o+(2*r+3*o)%5*5;for(var y=1,_=0;_<24;_++){for(var k=0,D=0,R=0;R<7;R++){if(1&y){var N=(1<<R)-1;N<32?D^=1<<N:k^=1<<N-32}128&y?y=y<<1^113:y<<=1}B[_]=P.create(k,D)}}();var D=[];!function(){for(var r=0;r<25;r++)D[r]=P.create()}();var R=k.SHA3=_.extend({cfg:_.cfg.extend({outputLength:512}),_doReset:function _doReset(){for(var r=this._state=[],o=0;o<25;o++)r[o]=new P.init;this.blockSize=(1600-2*this.cfg.outputLength)/32},_doProcessBlock:function _doProcessBlock(r,o){for(var u=this._state,p=this.blockSize/2,y=0;y<p;y++){var _=r[o+2*y],P=r[o+2*y+1];_=16711935&(_<<8|_>>>24)|4278255360&(_<<24|_>>>8),P=16711935&(P<<8|P>>>24)|4278255360&(P<<24|P>>>8),(oe=u[y]).high^=P,oe.low^=_}for(var k=0;k<24;k++){for(var R=0;R<5;R++){for(var N=0,U=0,W=0;W<5;W++)N^=(oe=u[R+5*W]).high,U^=oe.low;var j=D[R];j.high=N,j.low=U}for(R=0;R<5;R++){var G=D[(R+4)%5],X=D[(R+1)%5],Y=X.high,K=X.low;for(N=G.high^(Y<<1|K>>>31),U=G.low^(K<<1|Y>>>31),W=0;W<5;W++)(oe=u[R+5*W]).high^=N,oe.low^=U}for(var J=1;J<25;J++){var $=(oe=u[J]).high,ee=oe.low,te=I[J];te<32?(N=$<<te|ee>>>32-te,U=ee<<te|$>>>32-te):(N=ee<<te-32|$>>>64-te,U=$<<te-32|ee>>>64-te);var re=D[O[J]];re.high=N,re.low=U}var ne=D[0],ie=u[0];for(ne.high=ie.high,ne.low=ie.low,R=0;R<5;R++)for(W=0;W<5;W++){var oe=u[J=R+5*W],ae=D[J],se=D[(R+1)%5+5*W],ce=D[(R+2)%5+5*W];oe.high=ae.high^~se.high&ce.high,oe.low=ae.low^~se.low&ce.low}oe=u[0];var le=B[k];oe.high^=le.high,oe.low^=le.low}},_doFinalize:function _doFinalize(){var o=this._data,u=o.words,p=(this._nDataBytes,8*o.sigBytes),_=32*this.blockSize;u[p>>>5]|=1<<24-p%32,u[(r.ceil((p+1)/_)*_>>>5)-1]|=128,o.sigBytes=4*u.length,this._process();for(var P=this._state,k=this.cfg.outputLength/8,I=k/8,O=[],B=0;B<I;B++){var D=P[B],R=D.high,N=D.low;R=16711935&(R<<8|R>>>24)|4278255360&(R<<24|R>>>8),N=16711935&(N<<8|N>>>24)|4278255360&(N<<24|N>>>8),O.push(N),O.push(R)}return new y.init(O,k)},clone:function clone(){for(var clone=_.clone.call(this),r=clone._state=this._state.slice(0),o=0;o<25;o++)r[o]=r[o].clone();return clone}});o.SHA3=_._createHelper(R),o.HmacSHA3=_._createHmacHelper(R)}(Math),p.SHA3)},16793:function(r,o,u){"use strict";var p=u(93143).Buffer;function InternalCodec(r,o){this.enc=r.encodingName,this.bomAware=r.bomAware,"base64"===this.enc?this.encoder=InternalEncoderBase64:"cesu8"===this.enc&&(this.enc="utf8",this.encoder=InternalEncoderCesu8,"💩"!==p.from("eda0bdedb2a9","hex").toString()&&(this.decoder=InternalDecoderCesu8,this.defaultCharUnicode=o.defaultCharUnicode))}r.exports={utf8:{type:"_internal",bomAware:!0},cesu8:{type:"_internal",bomAware:!0},unicode11utf8:"utf8",ucs2:{type:"_internal",bomAware:!0},utf16le:"ucs2",binary:{type:"_internal"},base64:{type:"_internal"},hex:{type:"_internal"},_internal:InternalCodec},InternalCodec.prototype.encoder=InternalEncoder,InternalCodec.prototype.decoder=InternalDecoder;var y=u(43143).I;function InternalDecoder(r,o){this.decoder=new y(o.enc)}function InternalEncoder(r,o){this.enc=o.enc}function InternalEncoderBase64(r,o){this.prevStr=""}function InternalEncoderCesu8(r,o){}function InternalDecoderCesu8(r,o){this.acc=0,this.contBytes=0,this.accBytes=0,this.defaultCharUnicode=o.defaultCharUnicode}y.prototype.end||(y.prototype.end=function(){}),InternalDecoder.prototype.write=function(r){return p.isBuffer(r)||(r=p.from(r)),this.decoder.write(r)},InternalDecoder.prototype.end=function(){return this.decoder.end()},InternalEncoder.prototype.write=function(r){return p.from(r,this.enc)},InternalEncoder.prototype.end=function(){},InternalEncoderBase64.prototype.write=function(r){var o=(r=this.prevStr+r).length-r.length%4;return this.prevStr=r.slice(o),r=r.slice(0,o),p.from(r,"base64")},InternalEncoderBase64.prototype.end=function(){return p.from(this.prevStr,"base64")},InternalEncoderCesu8.prototype.write=function(r){for(var o=p.alloc(3*r.length),u=0,y=0;y<r.length;y++){var _=r.charCodeAt(y);_<128?o[u++]=_:_<2048?(o[u++]=192+(_>>>6),o[u++]=128+(63&_)):(o[u++]=224+(_>>>12),o[u++]=128+(_>>>6&63),o[u++]=128+(63&_))}return o.slice(0,u)},InternalEncoderCesu8.prototype.end=function(){},InternalDecoderCesu8.prototype.write=function(r){for(var o=this.acc,u=this.contBytes,p=this.accBytes,y="",_=0;_<r.length;_++){var P=r[_];128!=(192&P)?(u>0&&(y+=this.defaultCharUnicode,u=0),P<128?y+=String.fromCharCode(P):P<224?(o=31&P,u=1,p=1):P<240?(o=15&P,u=2,p=1):y+=this.defaultCharUnicode):u>0?(o=o<<6|63&P,p++,0===--u&&(y+=2===p&&o<128&&o>0||3===p&&o<2048?this.defaultCharUnicode:String.fromCharCode(o))):y+=this.defaultCharUnicode}return this.acc=o,this.contBytes=u,this.accBytes=p,y},InternalDecoderCesu8.prototype.end=function(){var r=0;return this.contBytes>0&&(r+=this.defaultCharUnicode),r}},17100:function(r,o,u){"use strict";var p=u(93143).Buffer;function Utf16BECodec(){}function Utf16BEEncoder(){}function Utf16BEDecoder(){this.overflowByte=-1}function Utf16Codec(r,o){this.iconv=o}function Utf16Encoder(r,o){void 0===(r=r||{}).addBOM&&(r.addBOM=!0),this.encoder=o.iconv.getEncoder("utf-16le",r)}function Utf16Decoder(r,o){this.decoder=null,this.initialBufs=[],this.initialBufsLen=0,this.options=r||{},this.iconv=o.iconv}function detectEncoding(r,o){var u=[],p=0,y=0,_=0;e:for(var P=0;P<r.length;P++)for(var k=r[P],I=0;I<k.length;I++)if(u.push(k[I]),2===u.length){if(0===p){if(255===u[0]&&254===u[1])return"utf-16le";if(254===u[0]&&255===u[1])return"utf-16be"}if(0===u[0]&&0!==u[1]&&_++,0!==u[0]&&0===u[1]&&y++,u.length=0,++p>=100)break e}return _>y?"utf-16be":_<y?"utf-16le":o||"utf-16le"}o.utf16be=Utf16BECodec,Utf16BECodec.prototype.encoder=Utf16BEEncoder,Utf16BECodec.prototype.decoder=Utf16BEDecoder,Utf16BECodec.prototype.bomAware=!0,Utf16BEEncoder.prototype.write=function(r){for(var o=p.from(r,"ucs2"),u=0;u<o.length;u+=2){var y=o[u];o[u]=o[u+1],o[u+1]=y}return o},Utf16BEEncoder.prototype.end=function(){},Utf16BEDecoder.prototype.write=function(r){if(0==r.length)return"";var o=p.alloc(r.length+1),u=0,y=0;for(-1!==this.overflowByte&&(o[0]=r[0],o[1]=this.overflowByte,u=1,y=2);u<r.length-1;u+=2,y+=2)o[y]=r[u+1],o[y+1]=r[u];return this.overflowByte=u==r.length-1?r[r.length-1]:-1,o.slice(0,y).toString("ucs2")},Utf16BEDecoder.prototype.end=function(){this.overflowByte=-1},o.utf16=Utf16Codec,Utf16Codec.prototype.encoder=Utf16Encoder,Utf16Codec.prototype.decoder=Utf16Decoder,Utf16Encoder.prototype.write=function(r){return this.encoder.write(r)},Utf16Encoder.prototype.end=function(){return this.encoder.end()},Utf16Decoder.prototype.write=function(r){if(!this.decoder){if(this.initialBufs.push(r),this.initialBufsLen+=r.length,this.initialBufsLen<16)return"";var o=detectEncoding(this.initialBufs,this.options.defaultEncoding);this.decoder=this.iconv.getDecoder(o,this.options);for(var u="",p=0;p<this.initialBufs.length;p++)u+=this.decoder.write(this.initialBufs[p]);return this.initialBufs.length=this.initialBufsLen=0,u}return this.decoder.write(r)},Utf16Decoder.prototype.end=function(){if(!this.decoder){var r=detectEncoding(this.initialBufs,this.options.defaultEncoding);this.decoder=this.iconv.getDecoder(r,this.options);for(var o="",u=0;u<this.initialBufs.length;u++)o+=this.decoder.write(this.initialBufs[u]);var p=this.decoder.end();return p&&(o+=p),this.initialBufs.length=this.initialBufsLen=0,o}return this.decoder.end()}},17506:function(r,o,u){var p=u(24517),y=Math.floor;r.exports=Number.isInteger||function isInteger(r){return!p(r)&&isFinite(r)&&y(r)===r}},17716:function(r,o,u){var p=u(40715);r.exports=/(?:ipad|iphone|ipod).*applewebkit/i.test(p)},17723:function(r,o,u){"use strict";r.exports=PassThrough;var p=u(48569);function PassThrough(r){if(!(this instanceof PassThrough))return new PassThrough(r);p.call(this,r)}u(89784)(PassThrough,p),PassThrough.prototype._transform=function(r,o,u){u(null,r)}},17802:function(r,o,u){"use strict";var p=u(5049),y=u(73036),_=u(10078),P=u(61909);r.exports=P||p.call(_,y)},18128:function(r,o,u){"use strict";var p=u(56128).Number;o.resolveLength=function(r,o,u){var y;if("number"==typeof r?y=r:"function"==typeof r?y=r.call(u,u):u&&"string"==typeof r?y=u[r]:o&&r instanceof p&&(y=r.decode(o)),isNaN(y))throw new Error("Not a fixed size");return y};o.PropertyDescriptor=function PropertyDescriptor(r){for(var o in void 0===r&&(r={}),this.enumerable=!0,this.configurable=!0,r){var u=r[o];this[o]=u}}},18425:function(r,o,u){"use strict";var p=u(56475),y=u(91159);p({target:"String",proto:!0,forced:u(7452)("strike")},{strike:function strike(){return y(this,"strike","","")}})},18645:function(r,o,u){"use strict";function _createForOfIteratorHelperLoose(r,o){var u="undefined"!=typeof Symbol&&r[Symbol.iterator]||r["@@iterator"];if(u)return(u=u.call(r)).next.bind(u);if(Array.isArray(r)||(u=function _unsupportedIterableToArray(r,o){if(r){if("string"==typeof r)return _arrayLikeToArray(r,o);var u={}.toString.call(r).slice(8,-1);return"Object"===u&&r.constructor&&(u=r.constructor.name),"Map"===u||"Set"===u?Array.from(r):"Arguments"===u||/^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(u)?_arrayLikeToArray(r,o):void 0}}(r))||o&&r&&"number"==typeof r.length){u&&(r=u);var p=0;return function(){return p>=r.length?{done:!0}:{done:!1,value:r[p++]}}}throw new TypeError("Invalid attempt to iterate non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.")}function _arrayLikeToArray(r,o){(null==o||o>r.length)&&(o=r.length);for(var u=0,p=Array(o);u<o;u++)p[u]=r[u];return p}function _regeneratorRuntime(){_regeneratorRuntime=function _regeneratorRuntime(){return o};var r,o={},u=Object.prototype,p=u.hasOwnProperty,y="function"==typeof Symbol?Symbol:{},_=y.iterator||"@@iterator",P=y.asyncIterator||"@@asyncIterator",k=y.toStringTag||"@@toStringTag";function c(r,o,u,p){return Object.defineProperty(r,o,{value:u,enumerable:!p,configurable:!p,writable:!p})}try{c({},"")}catch(r){c=function c(r,o,u){return r[o]=u}}function h(o,u,p,y){var _=u&&u.prototype instanceof Generator?u:Generator,P=Object.create(_.prototype);return c(P,"_invoke",function(o,u,p){var y=1;return function(_,P){if(3===y)throw Error("Generator is already running");if(4===y){if("throw"===_)throw P;return{value:r,done:!0}}for(p.method=_,p.arg=P;;){var k=p.delegate;if(k){var O=d(k,p);if(O){if(O===I)continue;return O}}if("next"===p.method)p.sent=p._sent=p.arg;else if("throw"===p.method){if(1===y)throw y=4,p.arg;p.dispatchException(p.arg)}else"return"===p.method&&p.abrupt("return",p.arg);y=3;var B=s(o,u,p);if("normal"===B.type){if(y=p.done?4:2,B.arg===I)continue;return{value:B.arg,done:p.done}}"throw"===B.type&&(y=4,p.method="throw",p.arg=B.arg)}}}(o,p,new Context(y||[])),!0),P}function s(r,o,u){try{return{type:"normal",arg:r.call(o,u)}}catch(r){return{type:"throw",arg:r}}}o.wrap=h;var I={};function Generator(){}function GeneratorFunction(){}function GeneratorFunctionPrototype(){}var O={};c(O,_,(function(){return this}));var B=Object.getPrototypeOf,D=B&&B(B(x([])));D&&D!==u&&p.call(D,_)&&(O=D);var R=GeneratorFunctionPrototype.prototype=Generator.prototype=Object.create(O);function g(r){["next","throw","return"].forEach((function(o){c(r,o,(function(r){return this._invoke(o,r)}))}))}function AsyncIterator(r,o){function e(u,y,_,P){var k=s(r[u],r,y);if("throw"!==k.type){var I=k.arg,O=I.value;return O&&"object"==typeof O&&p.call(O,"__await")?o.resolve(O.__await).then((function(r){e("next",r,_,P)}),(function(r){e("throw",r,_,P)})):o.resolve(O).then((function(r){I.value=r,_(I)}),(function(r){return e("throw",r,_,P)}))}P(k.arg)}var u;c(this,"_invoke",(function(r,p){function i(){return new o((function(o,u){e(r,p,o,u)}))}return u=u?u.then(i,i):i()}),!0)}function d(o,u){var p=u.method,y=o.i[p];if(y===r)return u.delegate=null,"throw"===p&&o.i.return&&(u.method="return",u.arg=r,d(o,u),"throw"===u.method)||"return"!==p&&(u.method="throw",u.arg=new TypeError("The iterator does not provide a '"+p+"' method")),I;var _=s(y,o.i,u.arg);if("throw"===_.type)return u.method="throw",u.arg=_.arg,u.delegate=null,I;var P=_.arg;return P?P.done?(u[o.r]=P.value,u.next=o.n,"return"!==u.method&&(u.method="next",u.arg=r),u.delegate=null,I):P:(u.method="throw",u.arg=new TypeError("iterator result is not an object"),u.delegate=null,I)}function w(r){this.tryEntries.push(r)}function m(o){var u=o[4]||{};u.type="normal",u.arg=r,o[4]=u}function Context(r){this.tryEntries=[[-1]],r.forEach(w,this),this.reset(!0)}function x(o){if(null!=o){var u=o[_];if(u)return u.call(o);if("function"==typeof o.next)return o;if(!isNaN(o.length)){var y=-1,P=function e(){for(;++y<o.length;)if(p.call(o,y))return e.value=o[y],e.done=!1,e;return e.value=r,e.done=!0,e};return P.next=P}}throw new TypeError(typeof o+" is not iterable")}return GeneratorFunction.prototype=GeneratorFunctionPrototype,c(R,"constructor",GeneratorFunctionPrototype),c(GeneratorFunctionPrototype,"constructor",GeneratorFunction),GeneratorFunction.displayName=c(GeneratorFunctionPrototype,k,"GeneratorFunction"),o.isGeneratorFunction=function(r){var o="function"==typeof r&&r.constructor;return!!o&&(o===GeneratorFunction||"GeneratorFunction"===(o.displayName||o.name))},o.mark=function(r){return Object.setPrototypeOf?Object.setPrototypeOf(r,GeneratorFunctionPrototype):(r.__proto__=GeneratorFunctionPrototype,c(r,k,"GeneratorFunction")),r.prototype=Object.create(R),r},o.awrap=function(r){return{__await:r}},g(AsyncIterator.prototype),c(AsyncIterator.prototype,P,(function(){return this})),o.AsyncIterator=AsyncIterator,o.async=function(r,u,p,y,_){void 0===_&&(_=Promise);var P=new AsyncIterator(h(r,u,p,y),_);return o.isGeneratorFunction(u)?P:P.next().then((function(r){return r.done?r.value:P.next()}))},g(R),c(R,k,"Generator"),c(R,_,(function(){return this})),c(R,"toString",(function(){return"[object Generator]"})),o.keys=function(r){var o=Object(r),u=[];for(var p in o)u.unshift(p);return function t(){for(;u.length;)if((p=u.pop())in o)return t.value=p,t.done=!1,t;return t.done=!0,t}},o.values=x,Context.prototype={constructor:Context,reset:function reset(o){if(this.prev=this.next=0,this.sent=this._sent=r,this.done=!1,this.delegate=null,this.method="next",this.arg=r,this.tryEntries.forEach(m),!o)for(var u in this)"t"===u.charAt(0)&&p.call(this,u)&&!isNaN(+u.slice(1))&&(this[u]=r)},stop:function stop(){this.done=!0;var r=this.tryEntries[0][4];if("throw"===r.type)throw r.arg;return this.rval},dispatchException:function dispatchException(o){if(this.done)throw o;var u=this;function n(r){_.type="throw",_.arg=o,u.next=r}for(var p=u.tryEntries.length-1;p>=0;--p){var y=this.tryEntries[p],_=y[4],P=this.prev,k=y[1],I=y[2];if(-1===y[0])return n("end"),!1;if(!k&&!I)throw Error("try statement without catch or finally");if(null!=y[0]&&y[0]<=P){if(P<k)return this.method="next",this.arg=r,n(k),!0;if(P<I)return n(I),!1}}},abrupt:function abrupt(r,o){for(var u=this.tryEntries.length-1;u>=0;--u){var p=this.tryEntries[u];if(p[0]>-1&&p[0]<=this.prev&&this.prev<p[2]){var y=p;break}}y&&("break"===r||"continue"===r)&&y[0]<=o&&o<=y[2]&&(y=null);var _=y?y[4]:{};return _.type=r,_.arg=o,y?(this.method="next",this.next=y[2],I):this.complete(_)},complete:function complete(r,o){if("throw"===r.type)throw r.arg;return"break"===r.type||"continue"===r.type?this.next=r.arg:"return"===r.type?(this.rval=this.arg=r.arg,this.method="return",this.next="end"):"normal"===r.type&&o&&(this.next=o),I},finish:function finish(r){for(var o=this.tryEntries.length-1;o>=0;--o){var u=this.tryEntries[o];if(u[2]===r)return this.complete(u[4],u[3]),m(u),I}},catch:function _catch(r){for(var o=this.tryEntries.length-1;o>=0;--o){var u=this.tryEntries[o];if(u[0]===r){var p=u[4];if("throw"===p.type){var y=p.arg;m(u)}return y}}throw Error("illegal catch attempt")},delegateYield:function delegateYield(o,u,p){return this.delegate={i:x(o),r:u,n:p},"next"===this.method&&(this.arg=r),I}},o}u(65292),u(73844),u(11765),u(24863),u(71950),u(68067),u(57114),u(42437),u(69330),u(81755),u(20731),u(14032),u(61726),u(58281),u(6422),u(94712);var p=function(){function StateMachine(r){this.stateTable=r.stateTable,this.accepting=r.accepting,this.tags=r.tags}var r=StateMachine.prototype;return r.match=function match(r){var o,u=this;return(o={})[Symbol.iterator]=_regeneratorRuntime().mark((function _callee(){var o,p,y,_,P,k;return _regeneratorRuntime().wrap((function _callee$(I){for(;;)switch(I.prev=I.next){case 0:o=1,p=null,y=null,_=null,P=0;case 5:if(!(P<r.length)){I.next=21;break}if(k=r[P],_=o,0!==(o=u.stateTable[o][k])){I.next=15;break}if(!(null!=p&&null!=y&&y>=p)){I.next=13;break}return I.next=13,[p,y,u.tags[_]];case 13:o=u.stateTable[1][k],p=null;case 15:0!==o&&null==p&&(p=P),u.accepting[o]&&(y=P),0===o&&(o=1);case 18:P++,I.next=5;break;case 21:if(!(null!=p&&null!=y&&y>=p)){I.next=24;break}return I.next=24,[p,y,u.tags[o]];case 24:case"end":return I.stop()}}),_callee)})),o},r.apply=function apply(r,o){for(var u,p=_createForOfIteratorHelperLoose(this.match(r));!(u=p()).done;)for(var y,_=u.value,P=_[0],k=_[1],I=_createForOfIteratorHelperLoose(_[2]);!(y=I()).done;){var O=y.value;"function"==typeof o[O]&&o[O](P,k,r.slice(P,k+1))}},StateMachine}();r.exports=p},18756:function(r,o,u){var p=u(56475),y=u(87146);p({target:"Object",stat:!0,forced:Object.assign!==y},{assign:y})},18828:function(r,o,u){"use strict";u(36673)("Map",(function(r){return function Map(){return r(this,arguments.length?arguments[0]:void 0)}}),u(9649))},18888:function(r){"use strict";r.exports=RangeError},18890:function(r,o,u){"use strict";var p=u(56649),y=function hasPropertyDescriptors(){return!!p};y.hasArrayLengthDefineBug=function hasArrayLengthDefineBug(){if(!p)return null;try{return 1!==p([],"length",{value:1}).length}catch(r){return!0}},r.exports=y},18904:function(r,o,u){var p=u(15567),y=u(47044),_=u(12072);r.exports=!p&&!y((function(){return 7!=Object.defineProperty(_("div"),"a",{get:function(){return 7}}).a}))},18909:function(r,o,u){"use strict";var p;u(20731),r.exports=(p=u(48352),function(r){var o=p,u=o.lib,y=u.Base,_=u.WordArray,P=o.x64={};P.Word=y.extend({init:function init(r,o){this.high=r,this.low=o}}),P.WordArray=y.extend({init:function init(o,u){o=this.words=o||[],this.sigBytes=u!=r?u:8*o.length},toX32:function toX32(){for(var r=this.words,o=r.length,u=[],p=0;p<o;p++){var y=r[p];u.push(y.high),u.push(y.low)}return _.create(u,this.sigBytes)},clone:function clone(){for(var clone=y.clone.call(this),r=clone.words=this.words.slice(0),o=r.length,u=0;u<o;u++)r[u]=r[u].clone();return clone}})}(),p)},19292:function(r){"use strict";r.exports=Math.round},20340:function(r,o,u){var p=u(38347),y=u(43162),_=p({}.hasOwnProperty);r.exports=Object.hasOwn||function hasOwn(r,o){return _(y(r),o)}},20731:function(r,o,u){"use strict";var p=u(56475),y=u(32010),_=u(59113),P=u(20884),k=u(24517),I=u(74841),O=u(45495),B=u(98086),D=u(38639),R=u(38688),N=u(56280),U=u(73163),W=N("slice"),j=R("species"),G=y.Array,X=Math.max;p({target:"Array",proto:!0,forced:!W},{slice:function slice(r,o){var u,p,y,R=B(this),N=O(R),W=I(r,N),Y=I(void 0===o?N:o,N);if(_(R)&&(u=R.constructor,(P(u)&&(u===G||_(u.prototype))||k(u)&&null===(u=u[j]))&&(u=void 0),u===G||void 0===u))return U(R,W,Y);for(p=new(void 0===u?G:u)(X(Y-W,0)),y=0;W<Y;W++,y++)W in R&&D(p,y,R[W]);return p.length=y,p}})},20884:function(r,o,u){var p=u(38347),y=u(47044),_=u(94578),P=u(52564),k=u(38486),I=u(10447),noop=function(){},O=[],B=k("Reflect","construct"),D=/^\s*(?:class|function)\b/,R=p(D.exec),N=!D.exec(noop),isConstructorModern=function(r){if(!_(r))return!1;try{return B(noop,O,r),!0}catch(r){return!1}};r.exports=!B||y((function(){var r;return isConstructorModern(isConstructorModern.call)||!isConstructorModern(Object)||!isConstructorModern((function(){r=!0}))||r}))?function(r){if(!_(r))return!1;switch(P(r)){case"AsyncFunction":case"GeneratorFunction":case"AsyncGeneratorFunction":return!1}return N||!!R(D,I(r))}:isConstructorModern},20980:function(r,o,u){var p=u(98197).z,y=u(98197).y,_=u(34097),P=u(80614),k=u(81561).z,I=u(81561).u,O=u(7043),B=u(42210),D=u(87984),R=1080,N=new Uint8Array([1,2,3,4,0,5,17,6,16,7,8,9,10,11,12,13,14,15]),U=new Uint8Array([3,2,1,0,3,3,3,3,3,3,2,2,2,2,2,2]),W=new Int8Array([0,0,0,0,-1,1,-2,2,-3,3,-1,1,-2,2,-3,3]),j=new Uint16Array([256,402,436,468,500,534,566,598,630,662,694,726,758,790,822,854,886,920,952,984,1016,1048,1080]);function DecodeWindowBits(r){var o;return 0===r.readBits(1)?16:(o=r.readBits(3))>0?17+o:(o=r.readBits(3))>0?8+o:17}function DecodeVarLenUint8(r){if(r.readBits(1)){var o=r.readBits(3);return 0===o?1:r.readBits(o)+(1<<o)}return 0}function MetaBlockLength(){this.meta_block_length=0,this.input_end=0,this.is_uncompressed=0,this.is_metadata=!1}function DecodeMetaBlockLength(r){var o,u,p,y=new MetaBlockLength;if(y.input_end=r.readBits(1),y.input_end&&r.readBits(1))return y;if(7===(o=r.readBits(2)+4)){if(y.is_metadata=!0,0!==r.readBits(1))throw new Error("Invalid reserved bit");if(0===(u=r.readBits(2)))return y;for(p=0;p<u;p++){var _=r.readBits(8);if(p+1===u&&u>1&&0===_)throw new Error("Invalid size byte");y.meta_block_length|=_<<8*p}}else for(p=0;p<o;++p){var P=r.readBits(4);if(p+1===o&&o>4&&0===P)throw new Error("Invalid size nibble");y.meta_block_length|=P<<4*p}return++y.meta_block_length,y.input_end||y.is_metadata||(y.is_uncompressed=r.readBits(1)),y}function ReadSymbol(r,o,u){var p;return u.fillBitWindow(),(p=r[o+=u.val_>>>u.bit_pos_&255].bits-8)>0&&(u.bit_pos_+=8,o+=r[o].value,o+=u.val_>>>u.bit_pos_&(1<<p)-1),u.bit_pos_+=r[o].bits,r[o].value}function ReadHuffmanCode(r,o,u,p){var y,_,P=new Uint8Array(r);if(p.readMoreInput(),1===(_=p.readBits(2))){for(var O=r-1,B=0,D=new Int32Array(4),R=p.readBits(2)+1;O;)O>>=1,++B;for(U=0;U<R;++U)D[U]=p.readBits(B)%r,P[D[U]]=2;switch(P[D[0]]=1,R){case 1:break;case 3:if(D[0]===D[1]||D[0]===D[2]||D[1]===D[2])throw new Error("[ReadHuffmanCode] invalid symbols");break;case 2:if(D[0]===D[1])throw new Error("[ReadHuffmanCode] invalid symbols");P[D[1]]=1;break;case 4:if(D[0]===D[1]||D[0]===D[2]||D[0]===D[3]||D[1]===D[2]||D[1]===D[3]||D[2]===D[3])throw new Error("[ReadHuffmanCode] invalid symbols");p.readBits(1)?(P[D[2]]=3,P[D[3]]=3):P[D[0]]=2}}else{var U,W=new Uint8Array(18),j=32,G=0,X=[new k(2,0),new k(2,4),new k(2,3),new k(3,2),new k(2,0),new k(2,4),new k(2,3),new k(4,1),new k(2,0),new k(2,4),new k(2,3),new k(3,2),new k(2,0),new k(2,4),new k(2,3),new k(4,5)];for(U=_;U<18&&j>0;++U){var Y,K=N[U],J=0;p.fillBitWindow(),J+=p.val_>>>p.bit_pos_&15,p.bit_pos_+=X[J].bits,Y=X[J].value,W[K]=Y,0!==Y&&(j-=32>>Y,++G)}if(1!==G&&0!==j)throw new Error("[ReadHuffmanCode] invalid num_codes or space");!function ReadHuffmanCodeLengths(r,o,u,p){for(var y=0,_=8,P=0,O=0,B=32768,D=[],R=0;R<32;R++)D.push(new k(0,0));for(I(D,0,5,r,18);y<o&&B>0;){var N,U=0;if(p.readMoreInput(),p.fillBitWindow(),U+=p.val_>>>p.bit_pos_&31,p.bit_pos_+=D[U].bits,(N=255&D[U].value)<16)P=0,u[y++]=N,0!==N&&(_=N,B-=32768>>N);else{var W,j,G=N-14,X=0;if(16===N&&(X=_),O!==X&&(P=0,O=X),W=P,P>0&&(P-=2,P<<=G),y+(j=(P+=p.readBits(G)+3)-W)>o)throw new Error("[ReadHuffmanCodeLengths] symbol + repeat_delta > num_symbols");for(var Y=0;Y<j;Y++)u[y+Y]=O;y+=j,0!==O&&(B-=j<<15-O)}}if(0!==B)throw new Error("[ReadHuffmanCodeLengths] space = "+B);for(;y<o;y++)u[y]=0}(W,r,P,p)}if(0===(y=I(o,u,8,P,r)))throw new Error("[ReadHuffmanCode] BuildHuffmanTable failed: ");return y}function ReadBlockLength(r,o,u){var p,y;return p=ReadSymbol(r,o,u),y=B.kBlockLengthPrefixCode[p].nbits,B.kBlockLengthPrefixCode[p].offset+u.readBits(y)}function TranslateShortCodes(r,o,u){var p;return r<16?(u+=U[r],p=o[u&=3]+W[r]):p=r-16+1,p}function MoveToFront(r,o){for(var u=r[o],p=o;p;--p)r[p]=r[p-1];r[0]=u}function HuffmanTreeGroup(r,o){this.alphabet_size=r,this.num_htrees=o,this.codes=new Array(o+o*j[r+31>>>5]),this.htrees=new Uint32Array(o)}function DecodeContextMap(r,o){var u,p,y={num_htrees:null,context_map:null},_=0;o.readMoreInput();var P=y.num_htrees=DecodeVarLenUint8(o)+1,I=y.context_map=new Uint8Array(r);if(P<=1)return y;for(o.readBits(1)&&(_=o.readBits(4)+1),u=[],p=0;p<R;p++)u[p]=new k(0,0);for(ReadHuffmanCode(P+_,u,0,o),p=0;p<r;){var O;if(o.readMoreInput(),0===(O=ReadSymbol(u,0,o)))I[p]=0,++p;else if(O<=_)for(var B=1+(1<<O)+o.readBits(O);--B;){if(p>=r)throw new Error("[DecodeContextMap] i >= context_map_size");I[p]=0,++p}else I[p]=O-_,++p}return o.readBits(1)&&function InverseMoveToFrontTransform(r,o){var u,p=new Uint8Array(256);for(u=0;u<256;++u)p[u]=u;for(u=0;u<o;++u){var y=r[u];r[u]=p[y],y&&MoveToFront(p,y)}}(I,r),y}function DecodeBlockType(r,o,u,p,y,_,P){var k,I=2*u,O=u,B=ReadSymbol(o,u*R,P);(k=0===B?y[I+(1&_[O])]:1===B?y[I+(_[O]-1&1)]+1:B-2)>=r&&(k-=r),p[u]=k,y[I+(1&_[O])]=k,++_[O]}function CopyUncompressedBlockToOutput(r,o,u,p,y,P){var k,I=y+1,O=u&y,B=P.pos_&_.IBUF_MASK;if(o<8||P.bit_pos_+(o<<3)<P.bit_end_pos_)for(;o-- >0;)P.readMoreInput(),p[O++]=P.readBits(8),O===I&&(r.write(p,I),O=0);else{if(P.bit_end_pos_<32)throw new Error("[CopyUncompressedBlockToOutput] br.bit_end_pos_ < 32");for(;P.bit_pos_<32;)p[O]=P.val_>>>P.bit_pos_,P.bit_pos_+=8,++O,--o;if(B+(k=P.bit_end_pos_-P.bit_pos_>>3)>_.IBUF_MASK){for(var D=_.IBUF_MASK+1-B,R=0;R<D;R++)p[O+R]=P.buf_[B+R];k-=D,O+=D,o-=D,B=0}for(R=0;R<k;R++)p[O+R]=P.buf_[B+R];if(o-=k,(O+=k)>=I){r.write(p,I),O-=I;for(R=0;R<O;R++)p[R]=p[I+R]}for(;O+o>=I;){if(k=I-O,P.input_.read(p,O,k)<k)throw new Error("[CopyUncompressedBlockToOutput] not enough bytes");r.write(p,I),o-=k,O=0}if(P.input_.read(p,O,o)<o)throw new Error("[CopyUncompressedBlockToOutput] not enough bytes");P.reset()}}function JumpToByteBoundary(r){var o=r.bit_pos_+7&-8;return 0==r.readBits(o-r.bit_pos_)}function BrotliDecompressedSize(r){var o=new p(r),u=new _(o);return DecodeWindowBits(u),DecodeMetaBlockLength(u).meta_block_length}function BrotliDecompress(r,o){var u,p,y,I,N,U,W,j,G,X,Y=0,K=0,J=0,$=[16,15,11,4],ee=0,te=0,re=0,ne=[new HuffmanTreeGroup(0,0),new HuffmanTreeGroup(0,0),new HuffmanTreeGroup(0,0)],ie=128+_.READ_SIZE;y=(1<<(p=DecodeWindowBits(X=new _(r))))-16,N=(I=1<<p)-1,U=new Uint8Array(I+ie+P.maxDictionaryWordLength),W=I,j=[],G=[];for(var oe=0;oe<3240;oe++)j[oe]=new k(0,0),G[oe]=new k(0,0);for(;!K;){var ae,se,ce,le,ue,fe,he,de,pe,ge,ye,ve=0,me=[1<<28,1<<28,1<<28],be=[0],we=[1,1,1],_e=[0,1,0,1,0,1],Se=[0],xe=null,Ae=null,Ce=0,Pe=null,ke=0,Ee=0,Te=0;for(u=0;u<3;++u)ne[u].codes=null,ne[u].htrees=null;X.readMoreInput();var Ie=DecodeMetaBlockLength(X);if(Y+(ve=Ie.meta_block_length)>o.buffer.length){var Oe=new Uint8Array(Y+ve);Oe.set(o.buffer),o.buffer=Oe}if(K=Ie.input_end,ae=Ie.is_uncompressed,Ie.is_metadata)for(JumpToByteBoundary(X);ve>0;--ve)X.readMoreInput(),X.readBits(8);else if(0!==ve)if(ae)X.bit_pos_=X.bit_pos_+7&-8,CopyUncompressedBlockToOutput(o,ve,Y,U,N,X),Y+=ve;else{for(u=0;u<3;++u)we[u]=DecodeVarLenUint8(X)+1,we[u]>=2&&(ReadHuffmanCode(we[u]+2,j,u*R,X),ReadHuffmanCode(26,G,u*R,X),me[u]=ReadBlockLength(G,u*R,X),Se[u]=1);for(X.readMoreInput(),le=(1<<(se=X.readBits(2)))-1,ue=(ce=16+(X.readBits(4)<<se))+(48<<se),xe=new Uint8Array(we[0]),u=0;u<we[0];++u)X.readMoreInput(),xe[u]=X.readBits(2)<<1;var Be=DecodeContextMap(we[0]<<6,X);he=Be.num_htrees,fe=Be.context_map;var Fe=DecodeContextMap(we[2]<<2,X);for(pe=Fe.num_htrees,de=Fe.context_map,ne[0]=new HuffmanTreeGroup(256,he),ne[1]=new HuffmanTreeGroup(704,we[1]),ne[2]=new HuffmanTreeGroup(ue,pe),u=0;u<3;++u)ne[u].decode(X);for(Ae=0,Pe=0,ge=xe[be[0]],Ee=O.lookupOffsets[ge],Te=O.lookupOffsets[ge+1],ye=ne[1].htrees[0];ve>0;){var Le,De,Re,Me,Ne,Ue,We,ze,je,Ge,Ve,He;for(X.readMoreInput(),0===me[1]&&(DecodeBlockType(we[1],j,1,be,_e,Se,X),me[1]=ReadBlockLength(G,R,X),ye=ne[1].htrees[be[1]]),--me[1],(De=(Le=ReadSymbol(ne[1].codes,ye,X))>>6)>=2?(De-=2,We=-1):We=0,Re=B.kInsertRangeLut[De]+(Le>>3&7),Me=B.kCopyRangeLut[De]+(7&Le),Ne=B.kInsertLengthPrefixCode[Re].offset+X.readBits(B.kInsertLengthPrefixCode[Re].nbits),Ue=B.kCopyLengthPrefixCode[Me].offset+X.readBits(B.kCopyLengthPrefixCode[Me].nbits),te=U[Y-1&N],re=U[Y-2&N],je=0;je<Ne;++je)X.readMoreInput(),0===me[0]&&(DecodeBlockType(we[0],j,0,be,_e,Se,X),me[0]=ReadBlockLengt